#!/usr/bin/env python
# -*- coding: utf-8 -*-
###############################################################################
# $Id$
#
# Project:  GDAL/OGR Test Suite
# Purpose:  test librarified gdal_translate
# Author:   Faza Mahamood <fazamhd at gmail dot com>
#
###############################################################################
# Copyright (c) 2015, Faza Mahamood <fazamhd at gmail dot com>
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
###############################################################################

import sys
import os

sys.path.append('../pymod')

from osgeo import gdal
import gdaltest

###############################################################################
# Simple test


def test_gdal_translate_lib_1():

    ds = gdal.Open('../gcore/data/byte.tif')

    ds = gdal.Translate('tmp/test1.tif', ds)
    if ds is None:
        gdaltest.post_reason('got error/warning')
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    ds = gdal.Open('tmp/test1.tif')
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test format option and callback


def mycallback(pct, msg, user_data):
    # pylint: disable=unused-argument
    user_data[0] = pct
    return 1


def test_gdal_translate_lib_2():

    src_ds = gdal.Open('../gcore/data/byte.tif')
    tab = [0]
    ds = gdal.Translate('tmp/test2.tif', src_ds, format='GTiff', callback=mycallback, callback_data=tab)
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if tab[0] != 1.0:
        gdaltest.post_reason('Bad percentage')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test outputType option


def test_gdal_translate_lib_3():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test3.tif', ds, outputType=gdal.GDT_Int16)
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).DataType != gdal.GDT_Int16:
        gdaltest.post_reason('Bad data type')
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test bandList option


def test_gdal_translate_lib_4():

    ds = gdal.Open('../gcore/data/rgbsmall.tif')

    ds = gdal.Translate('tmp/test4.tif', ds, bandList=[3, 2, 1])
    if ds is None:
        gdaltest.post_reason('got error/warning')
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 21349:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if ds.GetRasterBand(2).Checksum() != 21053:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if ds.GetRasterBand(3).Checksum() != 21212:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test rgbExpand option


def test_gdal_translate_lib_5():

    ds = gdal.Open('../gdrivers/data/bug407.gif')
    ds = gdal.Translate('tmp/test5.tif', ds, rgbExpand='rgb')
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).GetRasterColorInterpretation() != gdal.GCI_RedBand:
        gdaltest.post_reason('Bad color interpretation')
        return 'fail'

    if ds.GetRasterBand(2).GetRasterColorInterpretation() != gdal.GCI_GreenBand:
        gdaltest.post_reason('Bad color interpretation')
        return 'fail'

    if ds.GetRasterBand(3).GetRasterColorInterpretation() != gdal.GCI_BlueBand:
        gdaltest.post_reason('Bad color interpretation')
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 20615:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if ds.GetRasterBand(2).Checksum() != 59147:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if ds.GetRasterBand(3).Checksum() != 63052:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test oXSizePixel and oYSizePixel option


def test_gdal_translate_lib_6():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test6.tif', ds, width=40, height=40)
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 18784:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test oXSizePct and oYSizePct option


def test_gdal_translate_lib_7():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test7.tif', ds, widthPct=200.0, heightPct=200.0)
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 18784:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test outputSRS and GCPs options


def test_gdal_translate_lib_8():

    gcpList = [gdal.GCP(440720.000, 3751320.000, 0, 0, 0), gdal.GCP(441920.000, 3751320.000, 0, 20, 0), gdal.GCP(441920.000, 3750120.000, 0, 20, 20), gdal.GCP(440720.000, 3750120.000, 0, 0, 20)]
    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test8.tif', ds, outputSRS='EPSG:26711', GCPs=gcpList)
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    gcps = ds.GetGCPs()
    if len(gcps) != 4:
        gdaltest.post_reason('GCP count wrong.')
        return 'fail'

    if ds.GetGCPProjection().find('26711') == -1:
        gdaltest.post_reason('Bad GCP projection.')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test nodata option


def test_gdal_translate_lib_9():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test9.tif', ds, noData=1)
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).GetNoDataValue() != 1:
        gdaltest.post_reason('Bad nodata value')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test srcWin option


def test_gdal_translate_lib_10():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test10.tif', ds, srcWin=[0, 0, 1, 1])
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 2:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test projWin option


def test_gdal_translate_lib_11():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test11.tif', ds, projWin=[440720.000, 3751320.000, 441920.000, 3750120.000])
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if not gdaltest.geotransform_equals(gdal.Open('../gcore/data/byte.tif').GetGeoTransform(), ds.GetGeoTransform(), 1e-9):
        gdaltest.post_reason('Bad geotransform')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test outputBounds option


def test_gdal_translate_lib_12():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test12.tif', ds, outputBounds=[440720.000, 3751320.000, 441920.000, 3750120.000])
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if not gdaltest.geotransform_equals(gdal.Open('../gcore/data/byte.tif').GetGeoTransform(), ds.GetGeoTransform(), 1e-9):
        gdaltest.post_reason('Bad geotransform')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test metadataOptions


def test_gdal_translate_lib_13():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test13.tif', ds, metadataOptions=['TIFFTAG_DOCUMENTNAME=test13'])
    if ds is None:
        return 'fail'

    md = ds.GetMetadata()
    if 'TIFFTAG_DOCUMENTNAME' not in md:
        gdaltest.post_reason('Did not get TIFFTAG_DOCUMENTNAME')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test creationOptions


def test_gdal_translate_lib_14():

    ds = gdal.Open('../gcore/data/byte.tif')
    ds = gdal.Translate('tmp/test14.tif', ds, creationOptions=['COMPRESS=LZW'])
    if ds is None:
        return 'fail'

    md = ds.GetMetadata('IMAGE_STRUCTURE')
    if 'COMPRESSION' not in md or md['COMPRESSION'] != 'LZW':
        gdaltest.post_reason('Did not get COMPRESSION')
        return 'fail'

    ds = None

    return 'success'

###############################################################################
# Test internal wrappers


def test_gdal_translate_lib_100():

    # No option
    with gdaltest.error_handler():
        gdal.TranslateInternal('', gdal.Open('../gcore/data/byte.tif'), None)

    # Will create an implicit options structure
    with gdaltest.error_handler():
        gdal.TranslateInternal('', gdal.Open('../gcore/data/byte.tif'), None, gdal.TermProgress)

    # Null dest name
    try:
        gdal.TranslateInternal(None, gdal.Open('../gcore/data/byte.tif'), None)
    except:
        pass

    return 'success'

###############################################################################
# Test behaviour with SIGNEDBYTE


def test_gdal_translate_lib_101():

    ds = gdal.Translate('/vsimem/test_gdal_translate_lib_101.tif', gdal.Open('../gcore/data/byte.tif'), creationOptions=['PIXELTYPE=SIGNEDBYTE'], noData='-128')
    if ds.GetRasterBand(1).GetMetadataItem('PIXELTYPE', 'IMAGE_STRUCTURE') != 'SIGNEDBYTE':
        gdaltest.post_reason('Did not get SIGNEDBYTE')
        return 'fail'
    if ds.GetRasterBand(1).GetNoDataValue() != -128:
        gdaltest.post_reason('Did not get -128')
        return 'fail'
    ds2 = gdal.Translate('/vsimem/test_gdal_translate_lib_101_2.tif', ds, noData=-127)
    if ds2.GetRasterBand(1).GetNoDataValue() != -127:
        gdaltest.post_reason('Did not get -127')
        return 'fail'
    ds = None
    ds2 = None
    gdal.Unlink('/vsimem/test_gdal_translate_lib_101.tif')
    gdal.Unlink('/vsimem/test_gdal_translate_lib_101_2.tif')
    return 'success'

###############################################################################
# Test -scale


def test_gdal_translate_lib_102():

    ds = gdal.Translate('', gdal.Open('../gcore/data/byte.tif'), format='MEM', scaleParams=[[0, 255, 0, 65535]], outputType=gdal.GDT_UInt16)
    result = ds.GetRasterBand(1).ComputeRasterMinMax(False)
    if result != (19018.0, 65535.0):
        gdaltest.post_reason('failure')
        print(result)
        return 'fail'

    (approx_min, approx_max) = ds.GetRasterBand(1).ComputeRasterMinMax(True)
    ds2 = gdal.Translate('', ds, format='MEM', scaleParams=[[approx_min, approx_max]], outputType=gdal.GDT_Byte)
    expected_stats = ds2.GetRasterBand(1).ComputeStatistics(False)

    # Implicit source statistics use approximate source min/max
    ds2 = gdal.Translate('', ds, format='MEM', scaleParams=[[]], outputType=gdal.GDT_Byte)
    stats = ds2.GetRasterBand(1).ComputeStatistics(False)
    for i in range(4):
        if abs(stats[i] - expected_stats[i]) > 1e-3:
            gdaltest.post_reason('failure')
            print(stats)
            return 'fail'

    return 'success'

###############################################################################
# Test that -projwin with nearest neighbor resampling uses integer source
# pixel boundaries (#6610)


def test_gdal_translate_lib_103():

    ds = gdal.Translate('', '../gcore/data/byte.tif', format='MEM', projWin=[440730, 3751310, 441910, 3750140])
    if ds is None:
        return 'fail'

    if ds.GetRasterBand(1).Checksum() != 4672:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    if not gdaltest.geotransform_equals(gdal.Open('../gcore/data/byte.tif').GetGeoTransform(), ds.GetGeoTransform(), 1e-9):
        gdaltest.post_reason('Bad geotransform')
        return 'fail'

    return 'success'

###############################################################################
# Test translate with a MEM source to a anonymous VRT


def test_gdal_translate_lib_104():

    src_ds = gdal.GetDriverByName('MEM').Create('', 2, 2)
    src_ds.GetRasterBand(1).Fill(255)
    ds = gdal.Translate('', '../gcore/data/byte.tif', format='VRT', width=1, height=1)
    if ds.GetRasterBand(1).Checksum() != 3:
        gdaltest.post_reason('Bad checksum')
        return 'fail'

    return 'success'

###############################################################################
# Test GCPs propagation in "VRT path"


def test_gdal_translate_lib_gcp_vrt_path():

    src_ds = gdal.Open('../gcore/data/gcps.vrt')
    ds = gdal.Translate('', src_ds, format='MEM', metadataOptions=['FOO=BAR'])
    if len(ds.GetGCPs()) != len(src_ds.GetGCPs()):
        return 'fail'
    for i in range(len(src_ds.GetGCPs())):
        if ds.GetGCPs()[i].GCPX != src_ds.GetGCPs()[i].GCPX:
            return 'fail'
        if ds.GetGCPs()[i].GCPY != src_ds.GetGCPs()[i].GCPY:
            return 'fail'
        if ds.GetGCPs()[i].GCPPixel != src_ds.GetGCPs()[i].GCPPixel:
            return 'fail'
        if ds.GetGCPs()[i].GCPLine != src_ds.GetGCPs()[i].GCPLine:
            return 'fail'

    return 'success'

###############################################################################
# Test RPC propagation in "VRT path"


def test_gdal_translate_lib_rcp_vrt_path():

    src_ds = gdal.Open('../gcore/data/rpc.vrt')
    ds = gdal.Translate('', src_ds, format='MEM', metadataOptions=['FOO=BAR'])
    if ds.GetMetadata('RPC') != src_ds.GetMetadata('RPC'):
        return 'fail'

    return 'success'

###############################################################################
# Test GeoLocation propagation in "VRT path"


def test_gdal_translate_lib_geolocation_vrt_path():

    src_ds = gdal.Open('../gcore/data/sstgeo.vrt')
    ds = gdal.Translate('/vsimem/temp.vrt', src_ds, format='VRT', metadataOptions=['FOO=BAR'])
    if ds.GetMetadata('GEOLOCATION') != src_ds.GetMetadata('GEOLOCATION'):
        return 'fail'
    gdal.Unlink('/vsimem/temp.vrt')

    return 'success'

###############################################################################
# Test -colorinterp and -colorinterp_X


def test_gdal_translate_lib_colorinterp():

    src_ds = gdal.Open('../gcore/data/rgbsmall.tif')

    # Less bands specified than available
    ds = gdal.Translate('', src_ds, options='-f MEM -colorinterp blue,gray')
    if ds.GetRasterBand(1).GetColorInterpretation() != gdal.GCI_BlueBand:
        gdaltest.post_reason('fail')
        return 'fail'
    if ds.GetRasterBand(2).GetColorInterpretation() != gdal.GCI_GrayIndex:
        gdaltest.post_reason('fail')
        return 'fail'
    if ds.GetRasterBand(3).GetColorInterpretation() != gdal.GCI_BlueBand:
        gdaltest.post_reason('fail')
        return 'fail'

    # More bands specified than available and a unknown color interpretation
    with gdaltest.error_handler():
        ds = gdal.Translate('', src_ds, options='-f MEM -colorinterp alpha,red,undefined,foo')
    if ds.GetRasterBand(1).GetColorInterpretation() != gdal.GCI_AlphaBand:
        gdaltest.post_reason('fail')
        return 'fail'
    if ds.GetRasterBand(2).GetColorInterpretation() != gdal.GCI_RedBand:
        gdaltest.post_reason('fail')
        return 'fail'
    if ds.GetRasterBand(3).GetColorInterpretation() != gdal.GCI_Undefined:
        gdaltest.post_reason('fail')
        return 'fail'

    # Test colorinterp_
    ds = gdal.Translate('', src_ds, options='-f MEM -colorinterp_2 alpha')
    if ds.GetRasterBand(1).GetColorInterpretation() != gdal.GCI_RedBand:
        gdaltest.post_reason('fail')
        return 'fail'
    if ds.GetRasterBand(2).GetColorInterpretation() != gdal.GCI_AlphaBand:
        gdaltest.post_reason('fail')
        return 'fail'
    if ds.GetRasterBand(3).GetColorInterpretation() != gdal.GCI_BlueBand:
        gdaltest.post_reason('fail')
        return 'fail'

    # Test invalid colorinterp_
    try:
        with gdaltest.error_handler():
            gdal.Translate('', src_ds, options='-f MEM -colorinterp_0 alpha')
        gdaltest.post_reason('fail')
        return 'fail'
    except:
        pass

    return 'success'

###############################################################################
# Cleanup


def test_gdal_translate_lib_cleanup():
    for i in range(14):
        try:
            os.remove('tmp/test' + str(i + 1) + '.tif')
        except OSError:
            pass
        try:
            os.remove('tmp/test' + str(i + 1) + '.tif.aux.xml')
        except OSError:
            pass

    return 'success'


gdaltest_list = [
    test_gdal_translate_lib_1,
    test_gdal_translate_lib_2,
    test_gdal_translate_lib_3,
    test_gdal_translate_lib_4,
    test_gdal_translate_lib_5,
    test_gdal_translate_lib_6,
    test_gdal_translate_lib_7,
    test_gdal_translate_lib_8,
    test_gdal_translate_lib_9,
    test_gdal_translate_lib_10,
    test_gdal_translate_lib_11,
    test_gdal_translate_lib_12,
    test_gdal_translate_lib_13,
    test_gdal_translate_lib_14,
    test_gdal_translate_lib_100,
    test_gdal_translate_lib_101,
    test_gdal_translate_lib_102,
    test_gdal_translate_lib_103,
    test_gdal_translate_lib_104,
    test_gdal_translate_lib_gcp_vrt_path,
    test_gdal_translate_lib_rcp_vrt_path,
    test_gdal_translate_lib_geolocation_vrt_path,
    test_gdal_translate_lib_colorinterp,
    test_gdal_translate_lib_cleanup
]


if __name__ == '__main__':

    gdaltest.setup_run('test_gdal_translate_lib')

    gdaltest.run_tests(gdaltest_list)

    gdaltest.summarize()
