#!/usr/bin/env python
###############################################################################
# $Id$
#
# Project:  GDAL/OGR Test Suite
# Purpose:  Test OGR Memory driver functionality.
# Author:   Frank Warmerdam <warmerdam@pobox.com>
#
###############################################################################
# Copyright (c) 2003, Frank Warmerdam <warmerdam@pobox.com>
# Copyright (c) 2008-2011, Even Rouault <even dot rouault at mines-paris dot org>
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
###############################################################################

import sys

sys.path.append('../pymod')

import gdaltest
import ogrtest
from osgeo import ogr
from osgeo import gdal

###############################################################################
# Open Memory datasource.


def ogr_mem_1():

    mem_drv = ogr.GetDriverByName('Memory')
    gdaltest.mem_ds = mem_drv.CreateDataSource('wrk_in_memory')

    if gdaltest.mem_ds is None:
        return 'fail'

    return 'success'

###############################################################################
# Create table from data/poly.shp


def ogr_mem_2():

    if gdaltest.mem_ds is None:
        return 'skip'

    if gdaltest.mem_ds.TestCapability(ogr.ODsCCreateLayer) == 0:
        gdaltest.post_reason('ODsCCreateLayer TestCapability failed.')
        return 'fail'

    #######################################################
    # Create memory Layer
    gdaltest.mem_lyr = gdaltest.mem_ds.CreateLayer('tpoly')

    #######################################################
    # Setup Schema
    ogrtest.quick_create_layer_def(gdaltest.mem_lyr,
                                   [('AREA', ogr.OFTReal),
                                    ('EAS_ID', ogr.OFTInteger),
                                    ('PRFEDEA', ogr.OFTString),
                                    ('WHEN', ogr.OFTDateTime)])

    #######################################################
    # Copy in poly.shp

    dst_feat = ogr.Feature(feature_def=gdaltest.mem_lyr.GetLayerDefn())

    shp_ds = ogr.Open('data/poly.shp')
    gdaltest.shp_ds = shp_ds
    shp_lyr = shp_ds.GetLayer(0)

    feat = shp_lyr.GetNextFeature()
    gdaltest.poly_feat = []

    while feat is not None:

        gdaltest.poly_feat.append(feat)

        dst_feat.SetFrom(feat)
        ret = gdaltest.mem_lyr.CreateFeature(dst_feat)
        if ret != 0:
            gdaltest.post_reason('CreateFeature() failed.')
            return 'fail'

        feat = shp_lyr.GetNextFeature()

    return 'success'

###############################################################################
# Verify that stuff we just wrote is still OK.


def ogr_mem_3():
    if gdaltest.mem_ds is None:
        return 'skip'

    expect = [168, 169, 166, 158, 165]

    gdaltest.mem_lyr.SetAttributeFilter('eas_id < 170')
    tr = ogrtest.check_features_against_list(gdaltest.mem_lyr,
                                             'eas_id', expect)
    gdaltest.mem_lyr.SetAttributeFilter(None)

    for i in range(len(gdaltest.poly_feat)):
        orig_feat = gdaltest.poly_feat[i]
        read_feat = gdaltest.mem_lyr.GetNextFeature()

        if ogrtest.check_feature_geometry(read_feat, orig_feat.GetGeometryRef(),
                                          max_error=0.000000001) != 0:
            return 'fail'

        for fld in range(3):
            if orig_feat.GetField(fld) != read_feat.GetField(fld):
                gdaltest.post_reason('Attribute %d does not match' % fld)
                return 'fail'

    gdaltest.poly_feat = None
    gdaltest.shp_ds = None

    return 'success' if tr else 'fail'

###############################################################################
# Write more features with a bunch of different geometries, and verify the
# geometries are still OK.


def ogr_mem_4():

    if gdaltest.mem_ds is None:
        return 'skip'

    dst_feat = ogr.Feature(feature_def=gdaltest.mem_lyr.GetLayerDefn())
    wkt_list = ['10', '2', '1', '3d_1', '4', '5', '6']

    for item in wkt_list:

        wkt = open('data/wkb_wkt/' + item + '.wkt').read()
        geom = ogr.CreateGeometryFromWkt(wkt)

        ######################################################################
        # Write geometry as a new memory feature.

        dst_feat.SetGeometryDirectly(geom)
        dst_feat.SetField('PRFEDEA', item)
        gdaltest.mem_lyr.CreateFeature(dst_feat)

        ######################################################################
        # Read back the feature and get the geometry.

        gdaltest.mem_lyr.SetAttributeFilter("PRFEDEA = '%s'" % item)
        feat_read = gdaltest.mem_lyr.GetNextFeature()

        if ogrtest.check_feature_geometry(feat_read, geom) != 0:
            return 'fail'

    return 'success'

###############################################################################
# Test ExecuteSQL() results layers without geometry.


def ogr_mem_5():

    if gdaltest.mem_ds is None:
        return 'skip'

    expect = [179, 173, 172, 171, 170, 169, 168, 166, 165, 158, None]

    sql_lyr = gdaltest.mem_ds.ExecuteSQL('select distinct eas_id from tpoly order by eas_id desc')

    tr = ogrtest.check_features_against_list(sql_lyr, 'eas_id', expect)

    gdaltest.mem_ds.ReleaseResultSet(sql_lyr)

    return 'success' if tr else 'fail'

###############################################################################
# Test ExecuteSQL() results layers with geometry.


def ogr_mem_6():

    if gdaltest.mem_ds is None:
        return 'skip'

    sql_lyr = gdaltest.mem_ds.ExecuteSQL(
        "select * from tpoly where prfedea = '2'")

    tr = ogrtest.check_features_against_list(sql_lyr, 'prfedea', ['2'])
    if tr:
        sql_lyr.ResetReading()
        feat_read = sql_lyr.GetNextFeature()
        if ogrtest.check_feature_geometry(feat_read, 'MULTILINESTRING ((5.00121349 2.99853132,5.00121349 1.99853133),(5.00121349 1.99853133,5.00121349 0.99853133),(3.00121351 1.99853127,5.00121349 1.99853133),(5.00121349 1.99853133,6.00121348 1.99853135))') != 0:
            tr = 0

    gdaltest.mem_ds.ReleaseResultSet(sql_lyr)

    return 'success' if tr else 'fail'

###############################################################################
# Test spatial filtering.


def ogr_mem_7():

    if gdaltest.mem_ds is None:
        return 'skip'

    gdaltest.mem_lyr.SetAttributeFilter(None)

    geom = ogr.CreateGeometryFromWkt(
        'LINESTRING(479505 4763195,480526 4762819)')
    gdaltest.mem_lyr.SetSpatialFilter(geom)
    geom.Destroy()

    if gdaltest.mem_lyr.TestCapability(ogr.OLCFastSpatialFilter):
        gdaltest.post_reason('OLCFastSpatialFilter capability test should have failed.')
        return 'fail'

    tr = ogrtest.check_features_against_list(gdaltest.mem_lyr, 'eas_id',
                                             [158])

    gdaltest.mem_lyr.SetSpatialFilter(None)

    return 'success' if tr else 'fail'

###############################################################################
# Test adding a new field.


def ogr_mem_8():

    ####################################################################
    # Add new string field.
    field_defn = ogr.FieldDefn('new_string', ogr.OFTString)
    gdaltest.mem_lyr.CreateField(field_defn)

    ####################################################################
    # Apply a value to this field in one feature.

    gdaltest.mem_lyr.SetAttributeFilter("PRFEDEA = '2'")
    feat_read = gdaltest.mem_lyr.GetNextFeature()
    feat_read.SetField('new_string', 'test1')
    gdaltest.mem_lyr.SetFeature(feat_read)

    # Test expected failed case of SetFeature()
    new_feat = ogr.Feature(gdaltest.mem_lyr.GetLayerDefn())
    new_feat.SetFID(-2)
    gdal.PushErrorHandler('CPLQuietErrorHandler')
    ret = gdaltest.mem_lyr.SetFeature(new_feat)
    gdal.PopErrorHandler()
    if ret == 0:
        return 'fail'
    new_feat = None

    ####################################################################
    # Now fetch two features and verify the new column works OK.

    gdaltest.mem_lyr.SetAttributeFilter("PRFEDEA IN ( '2', '1' )")

    tr = ogrtest.check_features_against_list(gdaltest.mem_lyr, 'new_string',
                                             ['test1', None])

    gdaltest.mem_lyr.SetAttributeFilter(None)

    return 'success' if tr else 'fail'

###############################################################################
# Test deleting a feature.


def ogr_mem_9():

    if not gdaltest.mem_lyr.TestCapability(ogr.OLCDeleteFeature):
        gdaltest.post_reason('OLCDeleteFeature capability test failed.')
        return 'fail'

    if not gdaltest.mem_lyr.TestCapability(ogr.OLCFastFeatureCount):
        gdaltest.post_reason('OLCFastFeatureCount capability test failed.')
        return 'fail'

    old_count = gdaltest.mem_lyr.GetFeatureCount()

    ####################################################################
    # Delete target feature.

    target_fid = 2
    if gdaltest.mem_lyr.DeleteFeature(target_fid) != 0:
        gdaltest.post_reason('DeleteFeature returned error code.')
        return 'fail'

    if gdaltest.mem_lyr.DeleteFeature(target_fid) == 0:
        gdaltest.post_reason('DeleteFeature should have returned error code.')
        return 'fail'

    ####################################################################
    # Verify that count has dropped by one, and that the feature in question
    # can't be fetched.
    new_count = gdaltest.mem_lyr.GetFeatureCount()
    if new_count != old_count - 1:
        gdaltest.post_reason('got feature count of %d, not expected %d.'
                             % (new_count, old_count - 1))

    if not gdaltest.mem_lyr.TestCapability(ogr.OLCRandomRead):
        gdaltest.post_reason('OLCRandomRead capability test failed.')
        return 'fail'

    if gdaltest.mem_lyr.GetFeature(target_fid) is not None:
        gdaltest.post_reason('Got deleted feature!')
        return 'fail'

    if gdaltest.mem_lyr.GetFeature(-1) is not None:
        gdaltest.post_reason('GetFeature() should have failed')
        return 'fail'

    if gdaltest.mem_lyr.GetFeature(1000) is not None:
        gdaltest.post_reason('GetFeature() should have failed')
        return 'fail'

    return 'success'

###############################################################################
# Test GetDriver() / name bug (#1674)
#
# Mostly we are verifying that this doesn't still cause a crash.


def ogr_mem_10():

    d = ogr.GetDriverByName('Memory')
    ds = d.CreateDataSource('xxxxxx')

    d2 = ds.GetDriver()
    if d2 is None or d2.GetName() != 'Memory':
        gdaltest.post_reason('Did not get expected driver name.')
        return 'fail'

    return 'success'

###############################################################################
# Verify that we can delete layers properly


def ogr_mem_11():

    if gdaltest.mem_ds.TestCapability('DeleteLayer') == 0:
        gdaltest.post_reason('Deletelayer TestCapability failed.')
        return 'fail'

    gdaltest.mem_ds.CreateLayer('extra')
    gdaltest.mem_ds.CreateLayer('extra2')
    layer_count = gdaltest.mem_ds.GetLayerCount()

    gdaltest.mem_lyr = None
    # Delete extra layer
    if gdaltest.mem_ds.DeleteLayer(layer_count - 2) != 0:
        gdaltest.post_reason('DeleteLayer() failed')
        return 'fail'

    if gdaltest.mem_ds.DeleteLayer(-1) == 0:
        gdaltest.post_reason('DeleteLayer() should have failed')
        return 'fail'

    if gdaltest.mem_ds.DeleteLayer(gdaltest.mem_ds.GetLayerCount()) == 0:
        gdaltest.post_reason('DeleteLayer() should have failed')
        return 'fail'

    if gdaltest.mem_ds.GetLayer(-1) is not None:
        gdaltest.post_reason('GetLayer() should have failed')
        return 'fail'

    if gdaltest.mem_ds.GetLayer(gdaltest.mem_ds.GetLayerCount()) is not None:
        gdaltest.post_reason('GetLayer() should have failed')
        return 'fail'

    lyr = gdaltest.mem_ds.GetLayer(gdaltest.mem_ds.GetLayerCount() - 1)

    if lyr.GetName() != 'extra2':
        gdaltest.post_reason('delete layer seems iffy')
        return 'fail'

    return 'success'

###############################################################################
# Test some date handling


def ogr_mem_12():

    if gdaltest.mem_ds is None:
        return 'skip'

    #######################################################
    # Create memory Layer
    lyr = gdaltest.mem_ds.GetLayerByName('tpoly')
    if lyr is None:
        return 'fail'

    # Set the date of the first feature
    f = lyr.GetFeature(1)
    f.SetField("WHEN", 2008, 3, 19, 16, 15, 00, 0)
    lyr.SetFeature(f)
    f = lyr.GetFeature(1)
    idx = f.GetFieldIndex('WHEN')
    expected = [2008, 3, 19, 16, 15, 0.0, 0]
    result = f.GetFieldAsDateTime(idx)
    for i, value in enumerate(result):
        if value != expected[i]:
            gdaltest.post_reason('%s != %s' % (result, expected))
            return 'fail'
    return 'success'

###############################################################################
# Test Get/Set on StringList, IntegerList, RealList


def ogr_mem_13():

    if gdaltest.mem_ds is None:
        return 'skip'

    lyr = gdaltest.mem_ds.CreateLayer('listlayer')
    field_defn = ogr.FieldDefn('stringlist', ogr.OFTStringList)
    lyr.CreateField(field_defn)
    field_defn = ogr.FieldDefn('intlist', ogr.OFTIntegerList)
    lyr.CreateField(field_defn)
    field_defn = ogr.FieldDefn('reallist', ogr.OFTRealList)
    lyr.CreateField(field_defn)
    feat = ogr.Feature(feature_def=lyr.GetLayerDefn())

    try:
        feat.SetFieldStringList
    except AttributeError:
        # OG python bindings
        return 'skip'

    feat.SetFieldStringList(0, ['a', 'b'])
    if feat.GetFieldAsStringList(0) != ['a', 'b']:
        print(feat.GetFieldAsStringList(0))
        return 'fail'

    feat.SetFieldIntegerList(1, [2, 3])
    if feat.GetFieldAsIntegerList(1) != [2, 3]:
        print(feat.GetFieldAsIntegerList(1))
        return 'fail'

    feat.SetFieldDoubleList(2, [4., 5.])
    if feat.GetFieldAsDoubleList(2) != [4., 5.]:
        print(feat.GetFieldAsDoubleList(2))
        return 'fail'

    return 'success'

###############################################################################
# Test SetNextByIndex


def ogr_mem_14():

    if gdaltest.mem_ds is None:
        return 'skip'

    lyr = gdaltest.mem_ds.CreateLayer('SetNextByIndex')
    field_defn = ogr.FieldDefn('foo', ogr.OFTString)
    lyr.CreateField(field_defn)
    feat = ogr.Feature(feature_def=lyr.GetLayerDefn())
    feat.SetField(0, 'first feature')
    lyr.CreateFeature(feat)
    feat = ogr.Feature(feature_def=lyr.GetLayerDefn())
    feat.SetField(0, 'second feature')
    lyr.CreateFeature(feat)
    feat = ogr.Feature(feature_def=lyr.GetLayerDefn())
    feat.SetField(0, 'third feature')
    lyr.CreateFeature(feat)

    if not lyr.TestCapability(ogr.OLCFastSetNextByIndex):
        gdaltest.post_reason('OLCFastSetNextByIndex capability test failed.')
        return 'fail'

    if lyr.SetNextByIndex(1) != 0:
        gdaltest.post_reason('SetNextByIndex() failed')
        return 'fail'
    feat = lyr.GetNextFeature()
    if feat.GetFieldAsString(0) != 'second feature':
        gdaltest.post_reason('did not get expected feature')
        return 'fail'

    if lyr.SetNextByIndex(-1) == 0:
        gdaltest.post_reason('SetNextByIndex() should have failed')
        return 'fail'

    if lyr.SetNextByIndex(100) == 0:
        gdaltest.post_reason('SetNextByIndex() should have failed')
        return 'fail'

    lyr.SetAttributeFilter("foo != 'second feature'")

    if lyr.TestCapability(ogr.OLCFastSetNextByIndex):
        gdaltest.post_reason('OLCFastSetNextByIndex capability test should have failed.')
        return 'fail'

    if lyr.SetNextByIndex(1) != 0:
        gdaltest.post_reason('SetNextByIndex() failed')
        return 'fail'
    feat = lyr.GetNextFeature()
    if feat.GetFieldAsString(0) != 'third feature':
        gdaltest.post_reason('did not get expected feature')
        return 'fail'

    return 'success'

###############################################################################
# Test non-linear geometries


def ogr_mem_15():

    lyr = gdaltest.mem_ds.CreateLayer('wkbCircularString', geom_type=ogr.wkbCircularString)
    f = ogr.Feature(lyr.GetLayerDefn())
    f.SetGeometry(ogr.CreateGeometryFromWkt('CIRCULARSTRING(0 0,1 0,0 0)'))
    lyr.CreateFeature(f)
    f = None

    if lyr.GetGeomType() != ogr.wkbCircularString:
        gdaltest.post_reason('fail')
        return 'fail'
    if lyr.GetLayerDefn().GetGeomType() != ogr.wkbCircularString:
        gdaltest.post_reason('fail')
        return 'fail'
    if lyr.GetLayerDefn().GetGeomFieldDefn(0).GetType() != ogr.wkbCircularString:
        gdaltest.post_reason('fail')
        return 'fail'
    f = lyr.GetNextFeature()
    g = f.GetGeometryRef()
    if g.GetGeometryType() != ogr.wkbCircularString:
        gdaltest.post_reason('fail')
        return 'fail'

    # Test SetNonLinearGeometriesEnabledFlag(False)
    old_val = ogr.GetNonLinearGeometriesEnabledFlag()
    ogr.SetNonLinearGeometriesEnabledFlag(False)
    if lyr.GetGeomType() != ogr.wkbLineString:
        gdaltest.post_reason('fail')
        return 'fail'
    if lyr.GetLayerDefn().GetGeomType() != ogr.wkbLineString:
        gdaltest.post_reason('fail')
        return 'fail'
    if lyr.GetLayerDefn().GetGeomFieldDefn(0).GetType() != ogr.wkbLineString:
        gdaltest.post_reason('fail')
        return 'fail'
    lyr.ResetReading()
    f = lyr.GetNextFeature()
    g = f.GetGeometryRef()
    if g.GetGeometryType() != ogr.wkbLineString:
        gdaltest.post_reason('fail')
        return 'fail'

    lyr.ResetReading()
    f = lyr.GetNextFeature()
    g = f.GetGeomFieldRef(0)
    if g.GetGeometryType() != ogr.wkbLineString:
        gdaltest.post_reason('fail')
        return 'fail'
    ogr.SetNonLinearGeometriesEnabledFlag(old_val)

    return 'success'

###############################################################################
# Test map implementation


def ogr_mem_16():

    lyr = gdaltest.mem_ds.CreateLayer('ogr_mem_16')
    f = ogr.Feature(lyr.GetLayerDefn())
    ret = lyr.CreateFeature(f)
    if ret != 0:
        gdaltest.post_reason('fail')
        return 'fail'
    if f.GetFID() != 0:
        gdaltest.post_reason('fail')
        return 'fail'

    f = ogr.Feature(lyr.GetLayerDefn())
    ret = lyr.CreateFeature(f)
    if ret != 0:
        gdaltest.post_reason('fail')
        return 'fail'
    if f.GetFID() != 1:
        gdaltest.post_reason('fail')
        return 'fail'

    f = ogr.Feature(lyr.GetLayerDefn())
    f.SetFID(100000000)
    ret = lyr.CreateFeature(f)
    if ret != 0:
        gdaltest.post_reason('fail')
        return 'fail'

    f = ogr.Feature(lyr.GetLayerDefn())
    f.SetFID(100000000)
    ret = lyr.SetFeature(f)
    if ret != 0:
        gdaltest.post_reason('fail')
        return 'fail'

    if lyr.GetFeatureCount() != 3:
        gdaltest.post_reason('fail')
        print(lyr.GetFeatureCount())
        return 'fail'

    if lyr.GetFeature(0) is None:
        gdaltest.post_reason('fail')
        return 'fail'
    if lyr.GetFeature(1) is None:
        gdaltest.post_reason('fail')
        return 'fail'
    if lyr.GetFeature(2) is not None:
        gdaltest.post_reason('fail')
        return 'fail'
    if lyr.GetFeature(100000000) is None:
        gdaltest.post_reason('fail')
        return 'fail'

    lyr.ResetReading()
    f = lyr.GetNextFeature()
    if f.GetFID() != 0:
        gdaltest.post_reason('fail')
        return 'fail'
    f = lyr.GetNextFeature()
    if f.GetFID() != 1:
        gdaltest.post_reason('fail')
        return 'fail'
    f = lyr.GetNextFeature()
    if f.GetFID() != 100000000:
        gdaltest.post_reason('fail')
        return 'fail'
    f = lyr.GetNextFeature()
    if f is not None:
        gdaltest.post_reason('fail')
        return 'fail'

    f = ogr.Feature(lyr.GetLayerDefn())
    f.SetFID(100000000)
    ret = lyr.CreateFeature(f)
    if ret != 0:
        gdaltest.post_reason('fail')
        return 'fail'
    if f.GetFID() != 2:
        gdaltest.post_reason('fail')
        print(f.GetFID())
        return 'fail'

    if lyr.GetFeatureCount() != 4:
        gdaltest.post_reason('fail')
        print(lyr.GetFeatureCount())
        return 'fail'

    ret = lyr.DeleteFeature(1)
    if ret != 0:
        gdaltest.post_reason('fail')
        return 'fail'

    if lyr.GetFeatureCount() != 3:
        gdaltest.post_reason('fail')
        print(lyr.GetFeatureCount())
        return 'fail'

    ret = lyr.DeleteFeature(1)
    if ret == 0:
        gdaltest.post_reason('fail')
        return 'fail'

    if lyr.GetFeatureCount() != 3:
        gdaltest.post_reason('fail')
        print(lyr.GetFeatureCount())
        return 'fail'

    # Test first feature with huge ID
    lyr = gdaltest.mem_ds.CreateLayer('ogr_mem_16_bis')
    f = ogr.Feature(lyr.GetLayerDefn())
    f.SetFID(1234567890123)
    ret = lyr.CreateFeature(f)
    if ret != 0:
        gdaltest.post_reason('fail')
        return 'fail'
    if f.GetFID() != 1234567890123:
        gdaltest.post_reason('fail')
        return 'fail'
    f = None  # Important we must not have dangling references before modifying the schema !

    # Create a field so as to test OGRMemLayerIteratorMap
    lyr.CreateField(ogr.FieldDefn('foo'))

    return 'success'

###############################################################################
# Test Dataset.GetNextFeature() implementation


def ogr_mem_17():

    ds = gdal.GetDriverByName('Memory').Create('', 0, 0, 0, gdal.GDT_Unknown)
    lyr = ds.CreateLayer('ogr_mem_1')
    f = ogr.Feature(lyr.GetLayerDefn())
    lyr.CreateFeature(f)
    f = ogr.Feature(lyr.GetLayerDefn())
    lyr.CreateFeature(f)

    lyr = ds.CreateLayer('ogr_mem_2')
    f = ogr.Feature(lyr.GetLayerDefn())
    lyr.CreateFeature(f)

    lyr = ds.CreateLayer('ogr_mem_3')
    # Empty layer

    lyr = ds.CreateLayer('ogr_mem_4')
    f = ogr.Feature(lyr.GetLayerDefn())
    lyr.CreateFeature(f)

    f, lyr = ds.GetNextFeature()
    if f is None or lyr.GetName() != 'ogr_mem_1':
        gdaltest.post_reason('fail')
        print(f)
        print(lyr.GetName())
        return 'fail'

    f, lyr = ds.GetNextFeature()
    if f is None or lyr.GetName() != 'ogr_mem_1':
        gdaltest.post_reason('fail')
        print(f)
        print(lyr.GetName())
        return 'fail'

    f, lyr = ds.GetNextFeature()
    if f is None or lyr.GetName() != 'ogr_mem_2':
        gdaltest.post_reason('fail')
        print(f)
        print(lyr.GetName())
        return 'fail'

    f, lyr = ds.GetNextFeature()
    if f is None or lyr.GetName() != 'ogr_mem_4':
        gdaltest.post_reason('fail')
        print(f)
        print(lyr.GetName())
        return 'fail'

    f, lyr = ds.GetNextFeature()
    if f is not None or lyr is not None:
        gdaltest.post_reason('fail')
        return 'fail'

    f, lyr = ds.GetNextFeature()
    if f is not None or lyr is not None:
        gdaltest.post_reason('fail')
        return 'fail'

    ds.ResetReading()

    f, lyr = ds.GetNextFeature()
    if f is None or lyr.GetName() != 'ogr_mem_1':
        gdaltest.post_reason('fail')
        print(f)
        print(lyr.GetName())
        return 'fail'

    ds.ResetReading()

    f, lyr, pct = ds.GetNextFeature(include_pct=True)
    if f is None or lyr.GetName() != 'ogr_mem_1' or pct != 0.25:
        gdaltest.post_reason('fail')
        print(f)
        print(lyr.GetName())
        print(pct)
        return 'fail'

    f, pct = ds.GetNextFeature(include_layer=False, include_pct=True)
    if f is None or pct != 0.50:
        gdaltest.post_reason('fail')
        print(f)
        print(pct)
        return 'fail'

    f, pct = ds.GetNextFeature(include_layer=False, include_pct=True)
    if f is None or pct != 0.75:
        gdaltest.post_reason('fail')
        print(f)
        print(pct)
        return 'fail'

    f, pct = ds.GetNextFeature(include_layer=False, include_pct=True)
    if f is None or pct != 1.0:
        gdaltest.post_reason('fail')
        print(f)
        print(pct)
        return 'fail'

    f, pct = ds.GetNextFeature(include_layer=False, include_pct=True)
    if f is not None or pct != 1.0:
        gdaltest.post_reason('fail')
        print(f)
        print(pct)
        return 'fail'

    f, pct = ds.GetNextFeature(include_layer=False, include_pct=True)
    if f is not None or pct != 1.0:
        gdaltest.post_reason('fail')
        print(f)
        print(pct)
        return 'fail'

    ds.ResetReading()

    f = ds.GetNextFeature(include_layer=False)
    if f is None:
        gdaltest.post_reason('fail')
        print(f)
        return 'fail'

    return 'success'


def ogr_mem_cleanup():

    if gdaltest.mem_ds is None:
        return 'skip'

    ogr.SetNonLinearGeometriesEnabledFlag(True)
    gdaltest.mem_ds = None

    return 'success'


gdaltest_list = [
    ogr_mem_1,
    ogr_mem_2,
    ogr_mem_3,
    ogr_mem_4,
    ogr_mem_5,
    ogr_mem_6,
    ogr_mem_7,
    ogr_mem_8,
    ogr_mem_9,
    ogr_mem_10,
    ogr_mem_11,
    ogr_mem_12,
    ogr_mem_13,
    ogr_mem_14,
    ogr_mem_15,
    ogr_mem_16,
    ogr_mem_17,
    ogr_mem_cleanup]

if __name__ == '__main__':

    gdaltest.setup_run('ogr_mem')

    gdaltest.run_tests(gdaltest_list)

    gdaltest.summarize()
