------------------------------------------------------------------------------
--                                                                          --
--                 GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                 --
--                                                                          --
--                         A D A . I N T E R R U P T S                      --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--             Copyright (C) 1991-2017, Florida State University            --
--                     Copyright (C) 1995-2021, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- As a special exception under Section 7 of GPL version 3, you are granted --
-- additional permissions described in the GCC Runtime Library Exception,   --
-- version 3.1, as published by the Free Software Foundation.               --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;

package body Ada.Interrupts is

   package SI renames System.Interrupts;

   function To_System is new Ada.Unchecked_Conversion
     (Parameterless_Handler, SI.Parameterless_Handler);

   function To_Ada is new Ada.Unchecked_Conversion
     (SI.Parameterless_Handler, Parameterless_Handler);

   --------------------
   -- Attach_Handler --
   --------------------

   procedure Attach_Handler
     (New_Handler : Parameterless_Handler;
      Interrupt   : Interrupt_ID)
   is
   begin
      SI.Attach_Handler
        (To_System (New_Handler), SI.Interrupt_ID (Interrupt), False);
   end Attach_Handler;

   ---------------------
   -- Current_Handler --
   ---------------------

   function Current_Handler
     (Interrupt : Interrupt_ID) return Parameterless_Handler
   is
   begin
      return To_Ada (SI.Current_Handler (SI.Interrupt_ID (Interrupt)));
   end Current_Handler;

   --------------------
   -- Detach_Handler --
   --------------------

   procedure Detach_Handler (Interrupt : Interrupt_ID) is
   begin
      SI.Detach_Handler (SI.Interrupt_ID (Interrupt), False);
   end Detach_Handler;

   ----------------------
   -- Exchange_Handler --
   ----------------------

   procedure Exchange_Handler
     (Old_Handler : out Parameterless_Handler;
      New_Handler : Parameterless_Handler;
      Interrupt   : Interrupt_ID)
   is
      H : SI.Parameterless_Handler;

   begin
      SI.Exchange_Handler
        (H, To_System (New_Handler),
         SI.Interrupt_ID (Interrupt), False);
      Old_Handler := To_Ada (H);
   end Exchange_Handler;

   -------------
   -- Get_CPU --
   -------------

   function Get_CPU
     (Interrupt : Interrupt_ID) return System.Multiprocessors.CPU_Range
   is
      pragma Unreferenced (Interrupt);

   begin
      --  The underlying operating system does not indicate the processor on
      --  which the handler for Interrupt is executed.

      return System.Multiprocessors.Not_A_Specific_CPU;
   end Get_CPU;

   -----------------
   -- Is_Attached --
   -----------------

   function Is_Attached (Interrupt : Interrupt_ID) return Boolean is
   begin
      return SI.Is_Handler_Attached (SI.Interrupt_ID (Interrupt));
   end Is_Attached;

   -----------------
   -- Is_Reserved --
   -----------------

   function Is_Reserved (Interrupt : Interrupt_ID) return Boolean is
   begin
      return SI.Is_Reserved (SI.Interrupt_ID (Interrupt));
   end Is_Reserved;

   ---------------
   -- Reference --
   ---------------

   function Reference (Interrupt : Interrupt_ID) return System.Address is
   begin
      return SI.Reference (SI.Interrupt_ID (Interrupt));
   end Reference;

end Ada.Interrupts;
