/*
 * gawcolor.c - color functions
 *   GTK interface to Gaw
 * 
 * include LICENSE
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <sys/stat.h>

#include <gtk/gtk.h>

#include <duprintf.h>
#include <gaw.h>
 
#ifdef TRACE_MEM
#include <tracemem.h>
#endif

static char *ac_str_to_gdk_color(char *str )
{
   char *p;
   char *acColor = NULL;
   unsigned int val;
   
   if ( ( p = app_strcasestr(str, "0x" )) ) {
      p += 2;
   } else {
      p = str;
   }
   val = (unsigned int) g_ascii_strtoull( p, NULL, 16);
   acColor = app_strdup_printf( "#%06x", val);
   return acColor ;
}

void ac_rgb_color_str_set(char **valp, GdkColor *color )
{
   char colorspec[32];
   
   sprintf(colorspec , "0x%02x%02x%02x", (color->red >> 8), (color->green >> 8),
	   (color->blue >> 8)  ) ;
   msg_dbg ("%s :pixel 0x%x red %d, green %d, blue %d",
	    colorspec, color->pixel, color->red >> 8, color->green >> 8, color->blue >> 8);
   app_dup_str( valp, colorspec );
}
			  
static void ac_rgb_color_ary_set(ArrayStr *ary, GdkColor *color, int index )
{
   char colorspec[32];
   
   sprintf(colorspec , "0x%02x%02x%02x", (color->red >> 8), (color->green >> 8),
	   (color->blue >> 8)  ) ;
   msg_dbg ("%s :pixel 0x%x red %d, green %d, blue %d",
	    colorspec, color->pixel, color->red >> 8, color->green >> 8, color->blue >> 8);
   array_strPtr_replace_kill( ary, colorspec, index );
}


void
ac_gdk_colormap_alloc_color(GdkColormap *colormap, GdkColor *color, char *text)
{
   if ( ! gdk_colormap_alloc_color(colormap, color, FALSE, TRUE)) {
      msg_fatal (_("gdk_color_alloc failed for %s"), text );
   }
   msg_dbg ("%s: pixel 0x%x red %d, green %d, blue %d",
	    text, color->pixel, color->red, color->green, color->blue);
}

static void
ac_color_find_style_color( GtkWidget *widget, gchar *name, GdkColor *color)
{
   if ( ! name ) {
      name = (gchar *) gtk_widget_get_name (widget);
   }
   GtkStyle *style;
   style = gtk_widget_get_style (widget);
   if ( ! gtk_style_lookup_color ( style, name, color) ) {
      msg_warning (_("Can't find color for '%s'"), name) ;
   }
}

/* 
 * color init from style or from rc
 */ 

void ac_color_init(GdkColor *destColor, GdkColor *styleColor, char *rcColor )
{
   char *acColor = NULL;
   int ret = 0;
   
   if ( rcColor && *rcColor ){
      acColor = ac_str_to_gdk_color(rcColor);
      ret = gdk_color_parse( acColor, destColor);
   } 
   
   if ( acColor == NULL || ret == 0 ) {
      if ( acColor == NULL) {
         msg_dbg("color == NULL using style color for (%s)", rcColor );
      } else {
         msg_dbg("color parse error for (%s)", rcColor );
      }
      if (styleColor) {
         memcpy( destColor, styleColor, sizeof(GdkColor) );
      }
   }
   app_free(acColor);
}

void ac_color_background_set ( UserData  *ud)
{
   ac_gdk_colormap_alloc_color(ud->colormap, ud->bg_color, "background" );
   ac_rgb_color_str_set( &ud->up->panelBgColor, ud->bg_color );
}

void ac_color_grid_set (UserData  *ud)
{
   ac_gdk_colormap_alloc_color(ud->colormap, ud->pg_color, "graticule" );
   ac_rgb_color_str_set( &ud->up->gridColor, ud->pg_color );
}

void ac_color_panel_grid_set (WavePanel *wp, gpointer data)
{
   GdkColor *color = (GdkColor *) data;
 
   if ( color ) {
      memcpy( wp->grid_color, color, sizeof(GdkColor) );
   }
   ac_gdk_colormap_alloc_color(wp->ud->colormap, wp->grid_color, "grid" );
}

/*
 * called in drawing configure once for each panel
 */
void ac_color_panel_grid_init (WavePanel *wp)
{
   wp->grid_color = g_new0 (GdkColor, 1);
   ac_color_panel_grid_set (wp, wp->ud->pg_color);
}


void ac_color_cursor_set ( UserData  *ud, int i )
{
   AWCursor *csp = ud->cursors[i];

   ac_gdk_colormap_alloc_color(ud->colormap, csp->color, "dacursor*" );
   ac_rgb_color_ary_set( ud->up->cursorsColor, csp->color, i );
//   csp->color->pixel = csp->color->pixel ^ ud->bg_color->pixel;
}

void ac_color_hl_set ( UserData  *ud )
{
   ac_gdk_colormap_alloc_color(ud->colormap, ud->hl_color, "highlight"  );
   ac_rgb_color_str_set( &ud->up->hlColor, ud->hl_color );
}

void ac_color_srange_set ( UserData  *ud )
{
   ac_gdk_colormap_alloc_color(ud->colormap, ud->srange->color, "srange"  );
   ac_rgb_color_str_set( &ud->up->srangeColor, ud->srange->color );
}
void ac_color_bg_wave_button_set (VisibleWave *vw )
{
   GdkColor *color = vw->wp->ud->bg_color;
return;   
   gtk_widget_modify_bg (vw->button, GTK_STATE_NORMAL, color);
   gtk_widget_modify_bg (vw->button, GTK_STATE_PRELIGHT, color);
   gtk_widget_modify_bg (vw->button, GTK_STATE_ACTIVE, color);
}

void ac_color_wave_set (VisibleWave *vw)
{
   ac_gdk_colormap_alloc_color(vw->wp->ud->colormap, vw->color, "wave" );
   gtk_widget_modify_fg (vw->label, GTK_STATE_NORMAL, vw->color);
   gtk_widget_modify_fg (vw->label, GTK_STATE_PRELIGHT, vw->color);
   gtk_widget_modify_fg (vw->label, GTK_STATE_ACTIVE, vw->color);
   ac_color_bg_wave_button_set (vw );
}

void ac_color_wave_init (VisibleWave *vw, GdkColor *color)
{
   vw->color = g_new0 (GdkColor, 1);
   if ( color ) {
      memcpy( vw->color, color, sizeof(GdkColor) );
   } else {
      ac_color_find_style_color( vw->label, NULL, vw->color);
   }
   ac_color_wave_set (vw);
}

/*
 * initialize some vars from font settings once at startup
 */
void ac_color_font_init (WavePanel *wp)
{
   UserData *ud = wp->ud;
   PangoContext *context;
   PangoFontDescription *desc;
   PangoFont *font;
   PangoFontMetrics *metrics;

   context = gtk_widget_get_pango_context (wp->drawing);
   desc = pango_context_get_font_description ( context );

  /* FIXME - Should get the locale language here */
   pango_context_set_language (context, gtk_get_default_language());
   font = pango_context_load_font (context, desc);
   metrics = pango_font_get_metrics (font, pango_context_get_language (context));

   ud->char_width = 1 +
      PANGO_PIXELS (pango_font_metrics_get_approximate_char_width (metrics));
   ud->char_height =
      PANGO_PIXELS (pango_font_metrics_get_ascent (metrics)) +
      PANGO_PIXELS (pango_font_metrics_get_descent (metrics)) ;

   msg_dbg("char_width %d, char_height %d", ud->char_width, ud->char_height );
}

/*
 * initialize colors once at startup
 */
void ac_color_initialize (WavePanel *wp)
{
   UserData *ud = wp->ud;
   UserPrefs *up = ud->up;
   GdkColor styleColor;
   AWCursor *csp;
   int i;

   GdkWindow *window = gtk_widget_get_window (wp->drawing);
   ud->colormap = gdk_drawable_get_colormap(window);
   ud->bg_color = g_new0 (GdkColor, 1);
   
   /* background color */
   GtkStyle *style = gtk_widget_get_style (wp->drawing);

   ac_color_init( ud->bg_color,
		  &style->bg[gtk_widget_get_state(wp->drawing)],
		  up->panelBgColor );
   ac_color_background_set (ud);
   /* init bg for all created drawing */
   g_list_foreach(ud->panelList, (GFunc) pa_panel_drawing_background_set, NULL );
   
   /* graticule */
   ud->pg_color = g_new0 (GdkColor, 1);
   ac_color_find_style_color( wp->drawing, "graticule", &styleColor);
   ac_color_init( ud->pg_color, &styleColor, up->gridColor );
   ac_color_grid_set (ud);
   
   /* vertical bar cursors */
   for (i = 0 ; i < 2 ; i++) {
      char rcname[64];
      char *cursorColor = array_strPtr_get( up->cursorsColor, i);
      csp = ud->cursors[i];
      sprintf(rcname, "dacursor%d", i);

      csp->color = g_new0 (GdkColor, 1);
      ac_color_find_style_color( wp->drawing, rcname, &styleColor);
      ac_color_init( csp->color , &styleColor, cursorColor );
      ac_color_cursor_set (ud, i);
   }

   /* highlight */
   ud->hl_color = g_new0 (GdkColor, 1);
   ac_color_find_style_color( wp->drawing, "highlight", &styleColor);
   ac_color_init( ud->hl_color, &styleColor, up->hlColor );
   ac_color_hl_set (ud);

   /* select range */
   ud->srange->color = g_new0 (GdkColor, 1);
   ac_color_find_style_color( wp->drawing, "srange", &styleColor);
   ac_color_init( ud->srange->color, &styleColor, up->srangeColor );
   ac_color_srange_set (ud);

   ac_color_font_init (wp);
}


static gint
ac_color_dialog (GtkWidget *window, GdkColor *color, gchar *title )
{
   GtkWidget *dialog;
   GtkColorSelection *colorsel;
   gint response;

   dialog = gtk_color_selection_dialog_new (title);
   
   gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (window));
  
   colorsel = (GtkColorSelection *) gtk_color_selection_dialog_get_color_selection(
                     GTK_COLOR_SELECTION_DIALOG (dialog) );

   gtk_color_selection_set_previous_color (colorsel, color);
   gtk_color_selection_set_current_color (colorsel, color);
  
   response = gtk_dialog_run (GTK_DIALOG (dialog));
   gtk_color_selection_get_current_color (colorsel, color);

   gtk_widget_destroy (dialog);

   return response;
}

void
ac_color_wave_action (GtkAction *action, UserData *ud )
{
   gint response;
   VisibleWave *vw =  (VisibleWave *) aw_get_menu_data (ud->ui, "/ButtonPopup",
						    "ButtonPopup-action" );

   response = ac_color_dialog( ud->window, vw->color, _("Changing wave color"));
  
   if (response == GTK_RESPONSE_OK) {
      ac_color_wave_set(vw);
  
      pa_panel_full_redraw(vw->wp);
   }
}

void ac_pop_color_grid_action (GtkAction *action, WavePanel *wp )
{
   gint response;
   
   if ( ! wp ) {
      return;
   }
   response = ac_color_dialog( wp->ud->window, wp->grid_color, _("Changing grid color"));
   
   if (response == GTK_RESPONSE_OK) {
      ac_color_panel_grid_set (wp, NULL);

      da_drawing_redraw(wp->drawing);
   }
}


int ac_color_background_cmd (UserData *ud, char *colorspec )
{
   if ( ! gdk_color_parse(colorspec, ud->bg_color) ) {
      return -1;
   }
   ac_color_background_set (ud);
   g_list_foreach(ud->panelList, (GFunc) pa_panel_drawing_background_set, NULL );
   ap_all_redraw(ud);
   return 0;
}

int ac_color_grid_cmd (UserData *ud, char *colorspec )
{
   if ( ! gdk_color_parse(colorspec, ud->pg_color) ) {
      return -1;
   }
   ac_color_grid_set (ud);
   g_list_foreach(ud->panelList, (GFunc) ac_color_panel_grid_set, ud->pg_color );
   ap_all_redraw(ud);
   return 0;
}

void ac_color_set_cb (GtkColorButton *widget, gpointer user_data)
{
   msg_dbg("dialog color cb");
   GdkColor *color = (GdkColor *) user_data;
   gtk_color_button_get_color(widget, color);
}

void ac_color_panel_colors_action (GtkAction *action, UserData *ud )
{
   GtkWidget *dialog;
   GtkWidget *vbox;
   GtkWidget *frame;
   GtkWidget *table;
   GtkWidget *label;
   GtkWidget *picker;
   GdkColor *color;
   gint response;
   
   dialog = gtk_dialog_new_with_buttons (_("Gaw panel colors settings"),
                                         GTK_WINDOW (ud->window),
                                         GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_STOCK_CANCEL,
                                         GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK,
                                         GTK_RESPONSE_OK,
                                         NULL);

   vbox = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);
   
   /* frame panel colors */
   frame = gtk_frame_new ( _("Panel Colors Settings"));
   gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (frame), 8);

   table = gtk_table_new (6, 2, FALSE);    
   gtk_table_set_col_spacing (GTK_TABLE (table), 0, 10);
   gtk_table_set_row_spacings (GTK_TABLE (table), 3);
   gtk_container_add (GTK_CONTAINER (frame), table);
   gtk_container_set_border_width (GTK_CONTAINER (table), 10);

   label = gtk_label_new (_("Background Color:"));
   gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
   color = ud->bg_color;
   picker = gtk_color_button_new_with_color (color);
   gtk_table_attach_defaults (GTK_TABLE (table), label,  0, 1, 0, 1);
   gtk_table_attach_defaults (GTK_TABLE (table), picker, 1, 2, 0, 1);
   g_signal_connect (picker, "color-set",
                     G_CALLBACK (ac_color_set_cb), (gpointer) color );
      
   label = gtk_label_new (_("Grid Color:") );
   gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
   color = ud->pg_color;
   picker = gtk_color_button_new_with_color (color);
   gtk_table_attach_defaults (GTK_TABLE (table), label,  0, 1, 1, 2);
   gtk_table_attach_defaults (GTK_TABLE (table), picker, 1, 2, 1, 2);
   g_signal_connect (picker, "color-set",
                     G_CALLBACK (ac_color_set_cb), (gpointer) color );

   label = gtk_label_new (_("Highlight Color:") );
   gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
   color = ud->hl_color;
   picker = gtk_color_button_new_with_color (color);
   gtk_table_attach_defaults (GTK_TABLE (table), label,  0, 1, 2, 3);
   gtk_table_attach_defaults (GTK_TABLE (table), picker, 1, 2, 2, 3);
   g_signal_connect (picker, "color-set",
                     G_CALLBACK (ac_color_set_cb), (gpointer) color );

   label = gtk_label_new (_("Select Range Color:") );
   gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
   color = ud->srange->color;
   picker = gtk_color_button_new_with_color (color);
   gtk_table_attach_defaults (GTK_TABLE (table), label,  0, 1, 3, 4);
   gtk_table_attach_defaults (GTK_TABLE (table), picker, 1, 2, 3, 4);
   g_signal_connect (picker, "color-set",
                     G_CALLBACK (ac_color_set_cb), (gpointer) color );

   label = gtk_label_new (_("Cursor 0 Color:") );
   gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
   color = ud->cursors[0]->color;
   picker = gtk_color_button_new_with_color (color);
   gtk_table_attach_defaults (GTK_TABLE (table), label,  0, 1, 4, 5);
   gtk_table_attach_defaults (GTK_TABLE (table), picker, 1, 2, 4, 5);
   g_signal_connect (picker, "color-set",
                     G_CALLBACK (ac_color_set_cb), (gpointer) color );

   label = gtk_label_new (_("Cursor 1 Color:") );
   gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
   color = ud->cursors[1]->color;
   picker = gtk_color_button_new_with_color (color);
   gtk_table_attach_defaults (GTK_TABLE (table), label,  0, 1, 5, 6);
   gtk_table_attach_defaults (GTK_TABLE (table), picker, 1, 2, 5, 6);
   g_signal_connect (picker, "color-set",
                     G_CALLBACK (ac_color_set_cb), (gpointer) color );

   gtk_widget_show_all (vbox);
   response = gtk_dialog_run (GTK_DIALOG (dialog));

  if (response == GTK_RESPONSE_OK) {
     msg_dbg("dialog OK");
     ac_color_background_set (ud);
     g_list_foreach(ud->panelList, (GFunc) pa_panel_background_set, NULL );
    /* init grid for all created drawing */
     g_list_foreach(ud->panelList, (GFunc) ac_color_panel_grid_set, ud->pg_color );
     ac_color_grid_set (ud);
     ac_color_cursor_set ( ud, 0 );
     ac_color_cursor_set ( ud, 1 );
     ac_color_hl_set ( ud );     
     ac_color_srange_set ( ud );     
     ap_all_redraw(ud);
  }

  gtk_widget_destroy (dialog);
}
