
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Paul Brook

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

#include <string.h>
#include "safe-ctype.h"
#include "runtime.h"


#define null_string prefix(null_string);
char null_string;



/* copy_string()-- Copy a string */

void copy_string(char *dest, G95_DINT dest_len, char *src, G95_DINT src_len) {

    if (dest == src) { /* Some memmove()'s segfault if src == dest */
	if (src_len < dest_len)
	    memset(dest + src_len, ' ', dest_len-src_len);

    } else if (src_len >= dest_len)
	memmove(dest, src, dest_len);

    else {      /* Copy and pad with spaces.  */
	memmove(dest, src, src_len);
	memset(dest + src_len, ' ', dest_len-src_len);
    }
}



/* compare_string()-- Strings of unequal length are extended with blanks. */

G95_DINT compare_string(char *s1, G95_DINT s1_len, char *s2, G95_DINT s2_len) {
const char *s;
G95_DINT len;
int r;

    r = memcmp(s1, s2, (size_t) ((s1_len < s2_len) ? s1_len : s2_len));

    if (r != 0)
	return r;

    if (s1_len == s2_len)
	return 0;

    if (s1_len < s2_len) {
	len = s2_len - s1_len;
	s = s2 + s1_len;
	r = -1;

    } else{
	len = s1_len - s2_len;
	s = s1 + s2_len;
	r = 1;
    }

    for(; len>0; len--, s++) {
	if (*s == ' ')
	    continue;

	return (*s > ' ') ? r : -r;
    }

    return 0;
}



/* string_minmax()-- Subroutine to support min() and max() of
 * character variables. */

#define string_minmax prefix(string_minmax)

void string_minmax(G95_DINT max_flag, char **acc, G95_DINT *acc_len,
		   char *string, G95_DINT string_len) {

    if (compare_string(*acc, *acc_len, string, string_len)
	*(max_flag ? 1 : -1) < 0) {
	*acc = string;
	*acc_len = string_len;
    }
}


/* concat_string()-- Concatenate two strings.  The size of the
 * destination is always the sum of the sizes of the operands. */

#define concat_string prefix(concat_string)

void concat_string(char *dest, char *s1, G95_DINT s1_len,
		   char *s2, G95_DINT s2_len) {

    memcpy(dest, s1, s1_len);
    memcpy(dest + s1_len, s2, s2_len);
}



/* compare_fc()-- Compare a C-style string with a fortran style string
 * in a case-insensitive manner.  Used for decoding string options to
 * various statements.  Returns zero if not equal, zero if equal */

static int compare_fc(char *s1, G95_DINT s1_len, char *s2) {
size_t n;
int i;

    n = strlen(s2); 
    if (strncasecmp(s1, s2, n) != 0)
	return 0;

    /* The rest of s1 needs to be blanks for equality */

    for(i=n; i<s1_len; i++)
	if (s1[i] != ' ')
	    return 0;

    return 1;
}



/* fstrlen()-- Given a fortran string, return its length exclusive of
 * the trailing spaces */

int fstrlen(char *string, int len) {

    for(len--; len>=0; len--)
	if (string[len] != ' ')
	    break;

    return len+1;
}



char *string_copy_in(char *p, int len) {
char *q;

    q = get_mem(len+1);
    memmove(q, p, len);

    while(len > 0 && q[len-1] == ' ')
	len--;

    q[len] = '\0';
    return q;
}



void string_copy_out(char *source, char *dest, int dest_len) {

    copy_string(dest, dest_len, source, strlen(source));
}



/* fix_string()-- Convert a C string to a fortran string. */

void fix_string(char *result, int len) {
char *p;
int n;

    p = strchr(result, '\0');

    n = len - (p - result);
    if (n != 0)
	memset(p, ' ', n);
}



/* find_option()-- Given a fortran string and an array of st_option
 * structures, search through the array to find a match.  If the
 * option is not found, we generate an error if no default is
 * provided. */

int find_option(char *s1, int s1_len, st_option *opts, char *error_message) {

    for(; opts->name; opts++)
	if (compare_fc(s1, s1_len, opts->name))
	    return opts->value;

    generate_error(ERROR_BAD_OPTION, error_message);

    return -1;
}



/* expanded_string_length()-- Calculate the length of a string after
 * expanding any non-printable characters and delimiters in list
 * formatted mode.  The list of \-sequences must agree with what is in
 * copy_string_expand(). */

G95_DINT expanded_string_length(char *string, G95_DINT len, int delim) {
unsigned char c;
int m;

    m = (delim == '\'' || delim == '"') ? 2 : 0;

    while(len > 0) {
	c = *string++;
	len--;

	if (c == delim) {
	    m += 2;
	    continue;
	}

	if (c == '\a') {
	    if (!options.quiet)
		m++;

	    continue;
	}

	if (!options.expand_unprintable) {
	    m++;
	    continue;
	}

	if (isprint(c)) {
	    m++;
	    continue;
	}

	m += (c == '\n' || c == '\v' || c == '\r' || c == '\\') ? 2 : 4;
    }

    return m;
}



/* copy_string_expand()-- Copy a string, expanding unprintable
 * characters into multicharacter sequences if necessary.  The list of
 * \-sequences must agree with what is in expanded_string_len(). */

void copy_string_expand(char *dest, G95_DINT dest_len, char *src,
			G95_DINT src_len, int delim) {
unsigned char c, d;

    if ((delim == '\'' || delim == '"') && dest_len > 0) {
	*dest++ = delim;
	dest_len--;
    }

    while(src_len > 0) {
	c = *src++;
	src_len--;

	if (c == delim) {
	    if (dest_len == 0)
		break;

	    dest_len--;
	    *dest++ = c;

	    if (dest_len == 0)
		break;

	    dest_len--;

	    *dest++ = c;
	    continue;
	}

	if (c == '\t' || c == '\b' || c == '\f') {
	    if (dest_len == 0)
		break;

	    dest_len--;
	    *dest++ = c;
	    continue;
	}

	if (c == '\a') {
	    if (!options.quiet) {
		if (dest_len == 0)
		    break;

		dest_len--;
		*dest++ = c;
	    }

	    continue;
	}

	if (isprint(c) || !options.expand_unprintable) {
	    if (dest_len == 0)
		break;
	    dest_len--;

	    *dest++ = c;
	    continue;
	}

	if (dest_len == 0)
	    break;

	dest_len--;
	*dest++ = '\\';

	switch(c) {
	case '\a': c = 'a'; break;
	case '\n': c = 'n'; break;
	case '\v': c = 'v'; break;
	case '\r': c = 'r'; break;
	case '\\':          break;
	default:
	    if (dest_len == 0)
		break;

	    dest_len--;
	    *dest++ = 'x';

	    d = c >> 4;
	    d += (d > 9)
		? 'A' - 10
		: '0';

	    if (dest_len == 0)
		break;

	    *dest++ = d;
	    dest_len--;

	    d = c & 0x0F;
	    d += (d > 9)
		? 'A' - 10
		: '0';

	    c = d;
	    break;
	}

	if (dest_len == 0)
	    break;

	*dest++ = c;
	dest_len--;
    }

    if ((delim == '\'' || delim == '"') && dest_len > 0) {
	*dest++ = delim;
	dest_len--;
    }

    if (dest_len > 0)
	memset(dest, ' ', dest_len);
}



#define substring_oob prefix(substring_oob)

void substring_oob(G95_DINT length, G95_DINT start, G95_DINT end) {
char message[100];

    st_sprintf(message, "Substring reference out of bounds: (%d:%d) in string "
	       "of length %d", start+1, end, length);

    runtime_error(message);
}




#define selected_char_kind prefix(selected_char_kind)

G95_DINT selected_char_kind(char *name, G95_DINT name_len) {

    if (compare_fc(name, name_len, "default") ||
	compare_fc(name, name_len, "ascii"))
	return 1;

    return -1;
}



#define sys_len prefix(len)

G95_DINT sys_len(char *str, G95_DINT len) {

    return len;
}

