
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

#include <stdlib.h>
#include <string.h>
#include "runtime.h"

/* There are a several classes of memory in a g95 program, all of
 * which use the underlying malloc()/free() interface.  There are four
 * other classes of memory:
 *
 * First, there is library memory, which is allocated and managed by
 * the library independently of the user program.
 *
 * Then there is temporary memory, which is used to hold temporary
 * value, particularly as return values.  Third, memory associated
 * with a procedure that is freed automatically when the procedure
 * exits and fourth, memory which is allocated and deallocated under
 * control of the user program.
 *
 * Library memory is given to the called zeroed.  Temporary and
 * procedure memory is given to the caller uninitialized with the
 * assumption that it will soon be filled and not used uninitialized.
 * User memory can be optionally initialized or not under the control
 * of environment variables.  Library memory is not kept track of
 * outside of malloc().
 *
 * For debugging, temporary memory is allocated in a doubly linked
 * list.  All we are interested in is making sure that all this memory
 * is deallocated at the end of the program.
 *
 * User memory is kept track of in a balanced binary tree.  This is
 * because a DEALLOCATE statement is not always allowed to terminate
 * the program when it is given a bad pointer.  There are also
 * additional fields in user memory like line number and file for
 * debugging purposes.
 *
 * When the program exits normally, the user list is examined for
 * memory leaks if the appropriate option is set.  It is possible for
 * temporary and procedure memory to remain allocated if a STOP is
 * executed.
 */


typedef struct user_mem {
    struct user_mem *left, *right;
    char *filename;
    int priority, line;
    size_t size;
} user_mem;


typedef struct temp_mem {
    struct temp_mem *prev, *next;
    size_t size, pad;
} temp_mem;


static int stack_depth = 0, array_full;
static user_mem *user_root = NULL;
static temp_mem temp_root;
static size_t section_size_value;

static int memory_lock = 0;


#define total_alloc prefix(total_alloc)
#define max_alloc prefix(max_alloc)

G95_INT8 total_alloc = 0, max_alloc = 0;


typedef struct {
    g95_array_descriptor *array;
    G95_AINT dynamic, full, real_size, element_size, index[G95_MAX_DIMENSIONS];
} ac_info;




#define INITIAL_SIZE 2

#define DATA_ADDRESS(p, type) ((void *) ((char *) p + sizeof(type)))
#define DESC_ADDRESS(p, type) ((void *) ((char *) p - sizeof(type)))



/* init_memory()-- Initialize memory. */

void init_memory(void) {

    temp_root.next = &temp_root;
    temp_root.prev = &temp_root;
}



static void traverse_allocated(user_mem *u) {

    if (u == NULL)
	return;

    if (options.mem_segments != 0) {
	stack_depth++;

	if (stack_depth == options.mem_segments+1)
	    st_printf(" ... More segments remain\n");

	if (stack_depth > options.mem_segments &&
	    options.mem_segments > 0)
	    return;
    }

    st_printf("Remaining memory: %l bytes at %p allocated at line %d of %s\n",
	      (G95_MINT) u->size, DATA_ADDRESS(u, user_mem),
	      u->line, u->filename);

    traverse_allocated(u->left);
    traverse_allocated(u->right);
}



/* memory_done()-- This doesn't actually do any cleaning up, just
 * prints requested error messages. */

void memory_done(void) {

    if (options.mem_max_alloc)
	st_printf("Maximum user memory allocated: %l\n", max_alloc);

    if (options.mem_segments == 0)
	return;

    stack_depth = 0;
    traverse_allocated(user_root);

    if (temp_root.next != &temp_root)
	internal_error("Temporary memory remains allocated");
}



/* free_mem()-- Free library memory. */

void free_mem(void *p) {

    aquire_lock(&memory_lock);

    free(p);

    memory_lock = 0;
}



/* no_memory()-- Come here when memory allocation fails and there is
 * no other way to handle the error. */

static void no_memory(void) {

    os_error("Memory allocation failed");
}



/* get_mem()-- This gets a block of zeroed library memory. */

void *get_mem(size_t n) {
void *p;

    aquire_lock(&memory_lock);
    p = (void *) calloc(n, 1);
    memory_lock = 0;

    if (p == NULL)
	no_memory();

    return p;
}



/* temp_alloc()-- Temporary memory.  The same as library memory except
 * not initialized. */

void *temp_alloc(G95_DINT size) {
temp_mem *t;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    aquire_lock(&memory_lock);
    t = (temp_mem *) malloc(sizeof(temp_mem) + size);

    if (t == NULL) {
	memory_lock = 0;
	no_memory();
    }

    t->size = size;
    t->next = temp_root.next;
    t->prev = &temp_root;

    temp_root.next->prev = t;
    temp_root.next = t;

    memory_lock = 0;

    return DATA_ADDRESS(t, temp_mem);
}



void temp_free(void **p) {
temp_mem *t;
char *q;

    aquire_lock(&memory_lock);
   
    q = *p;
    *p = NULL;

    t = DESC_ADDRESS(q, temp_mem);

    t->prev->next = t->next;
    t->next->prev = t->prev;

    free(t);

    memory_lock = 0;
}



/* rotate_left()-- Rotate the treap left */

static user_mem *rotate_left(user_mem *t) {
user_mem *temp;

    temp = t->right;
    t->right = t->right->left;
    temp->left = t;

    return temp;
}



/* rotate_right()-- Rotate the treap right */

static user_mem *rotate_right(user_mem *t) {
user_mem *temp;

    temp = t->left;
    t->left = t->left->right;
    temp->right = t;

    return temp;
}



/* compare()-- Compare two user_mem nodes */

static int compare(user_mem *a, user_mem *b) {

    if (a < b) return -1;
    if (a > b) return 1;

    return 0;
}



/* insert_mem()-- Recursive insertion function.  Returns the updated
 * treap. */

static user_mem *insert_mem(user_mem *new, user_mem *t) {
int c;

    if (t == NULL)
	return new;

    c = compare(new, t);

    if (c < 0) {
	t->left = insert_mem(new, t->left);
	if (t->priority < t->left->priority)
	    t = rotate_right(t);
    }

    if (c > 0) {
	t->right = insert_mem(new, t->right);
	if (t->priority < t->right->priority)
	    t = rotate_left(t);
    }

    if (c == 0)
	internal_error("insert_mem(): Duplicate key found!");

    return t;
}



/* delete_root()-- Delete the root node */

static user_mem *delete_root(user_mem *t) {
user_mem *temp;

    if (t->left  == NULL)
	return t->right;

    if (t->right == NULL)
	return t->left;

    if (t->left->priority > t->right->priority) {
	temp = rotate_right(t);
	temp->right = delete_root(t);

    } else {
	temp = rotate_left(t);
	temp->left = delete_root(t);
    }

    return temp;
}



/* delete_treap()-- Delete an element from a tree.  The 'old' value
 * does not necessarily have to point to the element to be deleted, it
 * must just point to a treap structure with the key to be deleted.
 * Returns the new root node of the tree. */

static user_mem *delete_treap(user_mem *old, user_mem *t) {
int c;

    if (t == NULL)
	return NULL;

    c = compare(old, t);

    if (c < 0)
	t->left  = delete_treap(old, t->left);

    else if (c > 0)
	t->right = delete_treap(old, t->right);

    else
	t = delete_root(t);

    return t;
}



/* initialize_memory()-- Initialize memory to a fixed, hopefully
 * nonsense state before the user can mess with it. */

static void initialize_memory(char *memory, int bytes) {
int *ip, v, i;

    if (!options.allocate_init_flag)
	return;

    ip = (int *) memory;

    v = options.allocate_init_value;
    i = bytes / sizeof(int);

    while(i > 0) {
	*ip++ = v;
	i--;
    }
}



/* free_user_mem()-- Given a pointer to a user_memory node, remove the
 * node from the tree and deallocate the memory.  Returns nonzero if
 * the node was not in the tree. */

static int free_user_mem(user_mem *u) {
user_mem *c;

    aquire_lock(&memory_lock);

    total_alloc -= u->size;
    c = user_root;

    for(;;) {
	if (c == u)
	    break;

	if (c == NULL) {
	    memory_lock = 0;
	    return 1;
	}

	c = (c < u) ? c->right : c->left;
    }

    user_root = delete_treap(u, user_root);
    free(u);

    memory_lock = 0;
    return 0;
}



/* get_user_mem()-- Given a size, allocate memory for the user_mem
 * structure and the data behind it, and insert the new node into the
 * treap.  Returns a NULL pointer if we were unable to allocate
 * memory. */

static user_mem *get_user_mem(size_t size) {
user_mem *u;

    aquire_lock(&memory_lock);

    u = malloc(sizeof(user_mem) + size);
    if (u == NULL) {
	memory_lock = 0;
	return NULL;
    }

    total_alloc += size;
    if (total_alloc > max_alloc)
	max_alloc = total_alloc;

    u->priority = xorshift128();

    u->size = size;
    u->filename = filename;
    u->line = line;
    u->left = u->right = NULL;

    user_root = insert_mem(u, user_root);
    memory_lock = 0;

    /* Initialize allocated memory if necessary */

    initialize_memory(DATA_ADDRESS(u, user_mem), size);
    return u;
}



/* allocate()-- Process one element of a scalar ALLOCATE statement. */

#define allocate_scalar prefix(allocate_scalar)

void allocate_scalar(void **mem, G95_DINT size, void *init, G95_DINT flag) {
user_mem *u;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    if (flag && junk_stat != 0)
	return;

    u = get_user_mem(size);

    if (u == NULL) {
	if (flag)
	    junk_stat = ERROR_NOMEM;

	else
	    generate_error(ERROR_NOMEM, NULL);

	return;
    }

    *mem = DATA_ADDRESS(u, user_mem);

    if (init != NULL)
	memcpy(*mem, init, size);
}



/* allocate_string()-- Allocate a string with the ALLOCATE statement.  */

#define allocate_string prefix(allocate_string)

void allocate_string(void **mem, G95_DINT size, G95_DINT flag) {
user_mem *u;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    if (flag && junk_stat != 0)
	return;

    u = get_user_mem(size);
    if (u == NULL) {
	if (flag)
	    junk_stat = ERROR_NOMEM;

	else
	    generate_error(ERROR_NOMEM, NULL);

	return;
    }

    *mem = DATA_ADDRESS(u, user_mem);
}



/* section_size()-- Given the section_info[] array initialized for
 * pointer array allocation or allocatable array allocation, calculate
 * the size of the array.  Returns nonzero if the array is too large
 * for a size_t.  If OK, section_size_value is set. */

static int section_size(void) {
G95_AINT extent;
int i, rank;
size_t max;

    rank = section_info[0] & 0xFF;
    section_size_value = section_info[1];

    max = 0;
    max--;

    for(i=0; i<rank; i++) {
	extent = 1 + section_info[2*i+3] - section_info[2*i+2];
	if (extent <= 0) {
	    section_info[2*i+2] = 1;
	    section_info[2*i+3] = 0;
	    extent = 0;
	}

	if (extent != 0 && section_size_value != 0 &&
	    max / extent < section_size_value)
	    return 1;

	section_size_value *= extent;
    }

    return 0;
}



/* array_from_section()-- Allocate a temporary array used for a
 * function that is about to return an array.  This function uses the
 * section_info[] to compute bounds and such.  A single block of
 * memory is allocated consisting of the array storage followed by the
 * descriptor.  The base of the descriptor points to the memory, so
 * deallocating the base gets rid of the descriptor as well.  The
 * caller of an array-returning function must always dispose of the
 * array. */

g95_array_descriptor *array_from_section(char *init) {
g95_array_descriptor *desc;
size_t j, size_round;
int i, rank;
user_mem *u;
char *q;

    rank = section_info[0];

    if (section_size())
	no_memory();

    size_round = (section_size_value + 7) & (-8L);

    u = get_user_mem(size_round + sizeof(g95_array_descriptor));
    if (u == NULL)
	no_memory();

    desc = (void *) (((char *) DATA_ADDRESS(u, user_mem)) + size_round);
    desc->rank = rank;
    desc->base = (char *) DATA_ADDRESS(u, user_mem);

    desc->element_size = section_info[1];

    for(i=0; i<rank; i++) {
	desc->info[i].lbound = section_info[2*i+2];
	desc->info[i].ubound = section_info[2*i+3];
    }

    init_multipliers(desc);

    if (init != NULL) {
	section_size_value = section_size_value / desc->element_size;
	q = desc->base;

	for(j=0; j<section_size_value; j++) {
	    memcpy(q, init, desc->element_size);
	    q += desc->element_size;
	}
    }

    return desc;
}



/* allocate_array()-- In this case, we are given a pointer to a
 * descriptor and just fill in the block. */

void allocate_array(g95_array_descriptor *desc, G95_DINT pointer, void *init,
		    G95_DINT flag) {
user_mem *u;
int rank;
size_t i;
char *p;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    if (flag && junk_stat != 0)
	return;

    if (!pointer && desc->base != NULL) {
	if (flag)
	    junk_stat = ERROR_ALREADY_ALLOC;

	else
	    generate_error(ERROR_ALREADY_ALLOC, NULL);

	return;
    }

    rank = section_info[0] & 0xFF;

    u = NULL;
    if (!section_size())
	u = get_user_mem(section_size_value);

    if (u == NULL) {
	if (flag)
	    junk_stat = ERROR_NOMEM;

	else
	    no_memory();

	return;
    }

    if (init == NULL)
	initialize_memory(DATA_ADDRESS(u, user_mem), section_size_value);

    desc->base = DATA_ADDRESS(u, user_mem);
    desc->rank = rank;
    desc->element_size = section_info[1];

    for(i=0; i<rank; i++) {
	desc->info[i].lbound = section_info[2*i+2];
	desc->info[i].ubound = section_info[2*i+3];
    }

    init_multipliers(desc);

    if (init != NULL) {
	section_size_value = section_size_value / section_info[1];
	p = desc->base;

	for(i=0; i<section_size_value; i++) {
	    memcpy(p, init, section_info[1]);
	    p += section_info[1];
	}
    }
}



/* deallocate_array()-- Deallocate an allocatable array */

void deallocate_array(g95_array_descriptor *desc, alloc_struct *a,
		      G95_DINT flag) {
G95_AINT index[G95_MAX_DIMENSIONS];
user_mem *u;
char *p;
int i;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    if (desc->base == NULL) {
	if (flag)
	    junk_stat = ERROR_BADFREE;

	else
	    generate_error(ERROR_BADFREE, NULL);

	return;
    }

    if (a != NULL) {
	for(i=0; i<desc->rank; i++) {
	    index[i] = desc->info[i].lbound;
	    if (desc->info[i].lbound > desc->info[i].ubound)
		goto done;
	}

	do {
	    p = desc->offset;
	    for(i=0; i<desc->rank; i++)
		p += index[i] * desc->info[i].mult;

	    deep_dealloc(p, a);
	} while(!bump_element(desc, index));
    }

done:
    u = DESC_ADDRESS(desc->base, user_mem);
    free_user_mem(u);

    desc->base = NULL;
}



/* deallocate_pointer()-- Process a scalar element of a DEALLOCATE
 * statement. */

#define deallocate_pointer prefix(deallocate_pointer)

void deallocate_pointer(void **mem, G95_DINT flag) {
user_mem *u;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    if (*mem == NULL)
	goto badfree;

    u = DESC_ADDRESS(*mem, user_mem);
    if (free_user_mem(u))
	goto badfree;

    *mem = NULL;
    return;

badfree:
    if (flag)
	junk_stat = ERROR_BADFREE;

    else
	generate_error(ERROR_BADFREE, NULL);
}



/* start_ac_assign()-- Subroutine called at the start of assignment
 * from an array constructor.  Dynamic arrays are always one
 * dimensional.  We grow the array as elements are assigned.  If the
 * element_size is -1, then we are building an string array without
 * knowing the size yet. */

#define start_ac_assign prefix(start_ac_assign)

void start_ac_assign(ac_info *info, g95_array_descriptor *array,
		     G95_DINT dynamic, G95_DINT element_size) {
int i;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    info->array        = array;
    info->dynamic      = dynamic;
    info->element_size = element_size;
    info->full         = 0;

    if (dynamic) {
	section_info[0] = 1;
	section_info[1] = (element_size == -1) ? 0 : element_size;
	section_info[2] = 1;
	section_info[3] = INITIAL_SIZE;

	info->real_size = INITIAL_SIZE;
    }

    allocate_array(array, 1, NULL, 0);

    if (dynamic)
	array->info[0].ubound = 0;

    else {
	for(i=0; i<array->rank; i++)
	    info->index[i] = array->info[i].lbound;

	array_full = 0;
    }
}



/* ac_assign()-- Assign a value from an array constructor to an array
 * that grows as large as necessary. */

#define ac_assign prefix(ac_assign)

void ac_assign(ac_info *info, char *element, G95_DINT string_length) {
g95_array_descriptor *array;
size_t new_size;
int i, rank;
user_mem *u;
char *dest;

    array = info->array;

    if (info->dynamic) { 
	if (string_length != -1 && string_length != array->element_size) {
	    if (array->info[0].ubound != 0)
		runtime_error("Inconsistent string size in array constructor");

	    /* Reallocate a character array in which the string length was
	     * unknown until now. */

	    deallocate_array(array, NULL, 0);

	    section_info[0] = 1;
	    section_info[1] = string_length;
	    section_info[2] = 1;
	    section_info[3] = INITIAL_SIZE;

	    info->real_size = INITIAL_SIZE;
	    allocate_array(array, 1, NULL, 0);
	    array->info[0].ubound = 0;
	}

	array->info[0].ubound++;

	if (array->info[0].ubound > info->real_size) { /* Need more memory */
	    new_size = 2*info->real_size;
	    u = get_user_mem(new_size * array->element_size);

	    memcpy(DATA_ADDRESS(u, user_mem), array->base,
		   info->real_size * array->element_size);

	    free_user_mem(DESC_ADDRESS(array->base, user_mem));

	    array->base = DATA_ADDRESS(u, user_mem);

	    /* Not true in general, but lbound=1 for this case */
	    array->offset = array->base - array->element_size;
	    info->real_size = new_size;
	}

	dest = array->offset + array->info[0].ubound * array->info[0].mult;

    } else {   /* Static array */
	if (info->full)
	    runtime_error("Array constructor larger than array");

	rank = array->rank;
	dest = array->offset;

	for(i=0; i<rank; i++)
	    dest += info->index[i] * array->info[i].mult;

	info->full = bump_element(array, info->index);
    }

    if (string_length == -1)
	memmove(dest, element, array->element_size);

    else {
	copy_string(dest, array->element_size, element, string_length);
	array->element_size = string_length;
    }
}



/* ac_array()-- When a function returning an array is inside a
 * constructor, the descriptor is passed here.  From there we loop
 * over all elements. */

#define ac_array prefix(ac_array)

void ac_array(ac_info *info, g95_array_descriptor *desc,
	      G95_DINT string_length) {
G95_AINT index[G95_MAX_DIMENSIONS];
int i, rank;
char *p;

    rank = desc->rank;

    for(i=0; i<rank; i++) {
	if (desc->info[i].lbound > desc->info[i].ubound)
	    return;

	index[i] = desc->info[i].lbound;
    }

    do {
	p = desc->offset;
	for(i=0; i<rank; i++)
	    p += index[i] * desc->info[i].mult;

	ac_assign(info, p, string_length);
    } while(!bump_element(desc, index));
}



/* deep_dealloc()-- Do a deep deallocation */

#define deep_dealloc prefix(deep_dealloc)

void deep_dealloc(char *p, alloc_struct *a) {
G95_AINT index[G95_MAX_DIMENSIONS];
g95_array_descriptor *desc;
char *q;
int i;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    if (a == NULL)
	return;

    for(; a->offset!=-1; a++) {
	if (a->count != -1) {  /* Array of allocatable types */
	    q = p + a->offset;

	    for(i=0; i<a->count; i++) {
		deep_dealloc(q, a->pointer);
		q += a->size;
	    }

	    continue;
	}

	desc = (g95_array_descriptor *) (p + a->offset);
	if (desc->base == NULL)
	    continue;

	if (a->pointer == NULL)
	    goto done;

	/* Allocatable array containing other allocatables. */

	if (a->coarray)
	    deep_dealloc(desc->base, a->pointer);

	else {
	    for(i=0; i<desc->rank; i++) {
		index[i] = desc->info[i].lbound;

		if (desc->info[i].lbound > desc->info[i].ubound)
		    goto done;
	    }

	    do {
		q = desc->offset;
		for(i=0; i<desc->rank; i++)
		    q = q + index[i]*desc->info[i].mult;

		deep_dealloc(q, a->pointer);
	    } while(!bump_element(desc, index));
	}

    done:
	if (a->coarray)
	    deallocate_coarray(desc, 1);

	else
	    deallocate_array(desc, NULL, 1);
    }
}



/* deep_dealloc1()-- Do a deep deallocation of an array. */

#define deep_dealloc1 prefix(deep_dealloc1)

void deep_dealloc1(g95_array_descriptor *desc, alloc_struct *a) {
G95_AINT index[G95_MAX_DIMENSIONS];
char *q;
int i;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    if (desc->base == NULL)
	return;

    /* Allocatable array containing other allocatables. */

    for(i=0; i<desc->rank; i++) {
	index[i] = desc->info[i].lbound;

	if (desc->info[i].lbound > desc->info[i].ubound)
	    return;
    }

    do {
	q = desc->offset;
	for(i=0; i<desc->rank; i++)
	    q = q + index[i]*desc->info[i].mult;

	deep_dealloc(q, a);
    } while(!bump_element(desc, index));
}



/* deep_copy()-- Deep copy structure with allocated parts.
 * deep_dealloc() has already been called on the destination. */

void deep_copy(char *dest, char *src, G95_DINT size, alloc_struct *a) {
G95_AINT i, j, esize, elements, extent;
g95_array_descriptor *ddest, *dsrc;
char *p, *q;
user_mem *u;

    memmove(dest, src, size);
    if (a == NULL)
	return;

    for(; a->offset!=-1; a++) {
	if (a->count != -1) {  /* Array of alloctable types */
	    for(i=0; i<a->count; i++) {
		j = a->offset + i*a->size;
		deep_copy(dest+j, src+j, a->size, a->pointer);
	    }

	    continue;
	}

	dsrc = (g95_array_descriptor *) (src + a->offset);
	if (dsrc->base == NULL)
	    continue;

	ddest = (g95_array_descriptor *) (dest + a->offset);

	esize = dsrc->element_size;
	elements = 1;

	for(i=0; i<dsrc->rank; i++) {
	    extent = dsrc->info[i].ubound - dsrc->info[i].lbound + 1;
	    if (extent < 0)
		extent = 0;

	    elements *= extent;
	}

	u = get_user_mem(((size_t) elements)*esize);
	if (u == NULL)
	    no_memory();

	ddest->base = DATA_ADDRESS(u, user_mem);
	ddest->offset = dsrc->offset + (ddest->base - dsrc->base);

	p = ddest->base;
	q = dsrc->base;

	for(i=0; i<elements; i++) {
	    deep_copy(p, q, esize, a->pointer);

	    p += esize;
	    q += esize;
	}
    }
}



/* array_init()-- Initialize a derived type array with a default
 * initialization.  The array is contiguous.  */

#define array_init prefix(array_init)

void array_init(g95_array_descriptor *desc, char *init) {
G95_AINT size, extent;
int i, rank, esize;
char *p;

    if (!init_flag)
	g95_runtime_start(0, NULL);

    rank = desc->rank;
    esize = desc->element_size;
    size = 1;

    for(i=0; i<rank; i++) {
	extent = desc->info[i].ubound - desc->info[i].lbound + 1;
	if (extent < 0)
	    extent = 0;

	size *= extent;
    }

    p = desc->base;

    while(size-- > 0) {
	memcpy(p, init, esize);
	p += esize;
    }
}



#define move_alloc prefix(move_alloc)

void move_alloc(g95_array_descriptor *from, g95_array_descriptor *to,
		alloc_struct *a) {
int i;

    deallocate_array(to, a, 1);

    to->base   = from->base;
    to->offset = from->offset;
    to->rank   = from->rank;
    to->corank = from->corank;

    for(i=0; i<from->rank; i++)
	to->info[i] = from->info[i];

    from->base = NULL;
    from->offset = NULL;
}

