
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

#include <string.h>
#include "runtime.h"


/* write_a()-- Write with the A format */

void write_a(fnode *f, char *source, int len) {
G95_DINT wlen, xlen;
char *p;

    xlen = expanded_string_length(source, len, -1);
    wlen = (f->u.string.length < 0) ? xlen : f->u.string.length;

    p = write_block(wlen);
    if (p == NULL)
	return;

    if (wlen < xlen)
	copy_string_expand(p, wlen, source, xlen, -1);
    else {
	memset(p, ' ', wlen - xlen);
	copy_string_expand(p + wlen - xlen, xlen, source, xlen, -1);
    }
}


/* extract_logical()-- Return a logical truth value in a machine integer */

int extract_logical(void *p, int len) {
int m;

    switch(len) {
    case 1:  m = (int) (!! *((G95_INT1 *) p));  break;
    case 2:  m = (int) (!! *((G95_INT2 *) p));  break;
    case 4:  m = (int) (!! *((G95_INT4 *) p));  break;

#ifdef G95_INT8
    case 8:  m = (int) (!! *((G95_INT8 *) p));  break;
#endif

#ifdef G95_INT16
    case 16: m = (int) (!! *((G95_INT16 *) p)); break;
#endif

    default:
	internal_error("extract_logical(): Bad kind");
	m = 0;
    }

    return m;
}


/*write_l()-- Write with the L format */

void write_l(fnode *f, char *p, int len) {
char *q;

    q = write_block(f->u.w);
    if (q == NULL)
	return;

    memset(q, ' ', f->u.w - 1);
    q[f->u.w - 1] = extract_logical(p, len) ? 'T' : 'F';
}


/* calculate_sign()-- Given a flag that indicate if a value is
 * negative or not, return a sign_t that gives the sign that we need
 * to produce. */

sign_t calculate_sign(int negative_flag) {
sign_t s;

    if (negative_flag)  
	s = SIGN_MINUS;

    else
	switch(ioparm->sign_status) {
	case SIGN_SP: s = SIGN_PLUS;  break;
	case SIGN_SS: s = SIGN_NONE;  break;
	case SIGN_S:  s = options.optional_plus ? SIGN_PLUS : SIGN_NONE; break;
	default:
	    internal_error("calculate_sign(): Bad sign status");
	    s = SIGN_NONE;
	}

    return s;
}


/* write_x() */
/* write_x()-- Take care of the X/TR descriptor */

void write_x(fnode *f) {
char *p;

    p = write_block(f->u.n); 
    if (p == NULL)
	return;

    memset(p, ' ', f->u.n);
}


/* write_char()-- Write a single character to the output.  Returns
 * nonzero if something goes wrong. */

static int write_char(char c) {
char *p;

    p = write_block(1);
    if (p == NULL)
	return 1;

    *p = c;
    return 0;
}



/* write_logical()-- Write a list-directed logical value */

static void write_logical(char *source, int length) {

    write_char(extract_logical(source, length) ? 'T' : 'F');
}



/* write_character()-- Write a list-directed string.  We have to worry
 * about delimiting the strings if the file has been opened in that
 * mode. */

static void write_character(char *source, int length) {
int xlen, d;
char *p;

    switch(current_unit->flags.delim) {
    case DELIM_APOSTROPHE:    d = '\'';   break;
    case DELIM_QUOTE:         d = '"';    break;
    default:                  d = -1;     break;
    }

    xlen = expanded_string_length(source, length, d);
    p = write_block(xlen);

    copy_string_expand(p, xlen, source, length, d);
}


/* write_complex()-- Write a complex number */

static void write_complex(char *source, int len) {

    if (write_char('('))
	return;

    write_real(source, len);

    if (write_char(COMMA_SEP()))
	return;

    write_real(source + REAL_SIZE(len), len);

    write_char(')');
}


/* write_separator()-- Write the separator between items. */

static void write_separator(void) {
char *p;

    if (ioparm->namelist != NULL) { 
	p = write_block(2);
	if (p != NULL) {
	    p[0] = COMMA_SEP();
	    p[1] = ' ';
	}

	if (current_unit->offset > 55) {
	    next_record();
	    if (ioparm->namelist != NULL) {
		p = write_block(1);
		p[0] = ' ';
	    }
	}

    } else {
	p = write_block(options.separator_len);
	if (p != NULL)
	    memcpy(p, options.separator, options.separator_len);
    }
}



/* list_formatted_write()-- Write an item with list formatting.
 * TODO: handle skipping to the next record correctly, particularly
 * with strings. */

void list_formatted_write(bt type, void *p, int len) {
static int char_flag;
char *q;

    if (ioparm->namelist == NULL) {
	if (ioparm->first_item) {
	    ioparm->first_item = 0;
	    char_flag = 0;

	    if (current_unit->offset == 0 && current_unit->max_offset == 0) {
		q = write_block(1);
		*q = ' ';
	    }

	} else if (type != BT_CHARACTER || !char_flag ||
		   current_unit->flags.delim != DELIM_NONE)
	    write_separator();
    }

    switch(type) {
    case BT_INTEGER:     write_integer(p, len);     break;
    case BT_LOGICAL:     write_logical(p, len);     break;
    case BT_CHARACTER:   write_character(p, len);   break;
    case BT_REAL:        write_real(p, len);        break;
    case BT_COMPLEX:     write_complex(p, len);     break;
    default:
	internal_error("list_formatted_write(): Bad type");
    }

    char_flag = (type == BT_CHARACTER);

    if (ioparm->namelist != NULL)
	write_separator();
}


/* write_derived()-- Namelist write of a derived type.  We just
 * traverse the info structure, possibly recursively. */

static void write_derived(char *pointer, derived_info *info) {
G95_AINT index[G95_MAX_DIMENSIONS];
g95_array_descriptor *dp, desc;
int rank, i, t;
char *a, *p;

    t = 0;
    while(info->name != NULL) {
	a = pointer + info->offset;

	if (info->rank == 0)
	    switch(info->type) {
	    case 'i':
		list_formatted_write(BT_INTEGER, a, info->kind);
		break;

	    case 'r':
		list_formatted_write(BT_REAL, a, info->kind);
		break;

	    case 'z':
		list_formatted_write(BT_COMPLEX, a, info->kind);
		break;

	    case 'l':
		list_formatted_write(BT_LOGICAL, a, info->kind);
		break;

	    case 'c':
		list_formatted_write(BT_CHARACTER, a, info->kind);
		break;

	    case 'd':
		write_derived(a, info->info);
		break;
	    }

	else {   /* Array transfer */
	    rank = info->rank;

	    if (info->shape == NULL)
		dp = (g95_array_descriptor *) a;

	    else {  /* Build a descriptor */
		for(i=0; i<rank; i++) {
		    desc.info[i].lbound = info->shape[2*i];
		    desc.info[i].ubound = info->shape[2*i+1];
		}

		desc.rank = rank;
		desc.element_size = info->kind;
		desc.base = a;

		init_multipliers(&desc);
	    }

	    switch(info->type) {
	    case 'i':
		t = BT_INTEGER;
		goto array;

	    case 'r':
		t = BT_REAL;
		goto array;

	    case 'z':
		t = BT_COMPLEX;
		goto array;

	    case 'l':
		t = BT_LOGICAL;
		goto array;

	    case 'c':
		t = BT_CHARACTER;
		goto array;

	    case 'd':
		t = BT_DERIVED;
		goto array;
	    }

	array:
	    for(i=0; i<rank; i++) {
		if (desc.info[i].lbound > desc.info[i].ubound)
		    goto zero_size;

		index[i] = desc.info[i].lbound;
	    }

	    for(;;) {
		p = desc.offset;
		for(i=0; i<rank; i++)
		    p += desc.info[i].mult * index[i];

		if (t == BT_DERIVED)
		    write_derived(p, info->info);
		else
		    list_formatted_write(t, p, info->kind);

		if (bump_element(&desc, index))
		    break;
	    }
	}

    zero_size:
	info++;
    }
}



/* str_upper()-- Convert a string to uppercase */

static void str_upper(char *p, int len) {

    while(len > 0) {
	if ('a' <= *p && *p <= 'z')
	    *p += 'A' - 'a';

	len--;
	p++;
    }
}


/* write_rep()-- Write a repeat count */

static void write_rep(int rep) {

    if (rep > 1) {
	write_integer(&rep, sizeof(rep));
	write_char('*');
    }
}



/* write_name()-- Write the name of the variable plus a trailing
 * equals sign to the namelist. */

static void write_name(namelist_info *n) {
char *p;
int i;

    i = strlen(n->name);
    p = write_block(i+3);
    if (p == NULL)
	return;

    memcpy(p, n->name, i);
    str_upper(p, i);

    memcpy(p+i, " = ", 3);
}



/* namelist_write0()-- Write a single namelist element */

static void namelist_write0(namelist_info *n) {
G95_AINT index[G95_MAX_DIMENSIONS];
g95_array_descriptor *desc;
int i, t, rank, rep;
char *p, *q;

    t = -1;

    switch(n->type) {
    case 'r':
	t = BT_REAL;
	goto scalar;

    case 'i':
	t = BT_INTEGER;
	goto scalar;

    case 'l':
	t = BT_LOGICAL;
	goto scalar;

    case 'c':
	t = BT_CHARACTER;
	goto scalar;

    case 'z':
	t = BT_COMPLEX;
	goto scalar;

    case 'd':
	t = BT_DERIVED;
	goto scalar;

    case 'R':
	t = BT_REAL;
	goto array;

    case 'I':
	t = BT_INTEGER;
	goto array;

    case 'L':
	t = BT_LOGICAL;
	goto array;

    case 'C':
	t = BT_CHARACTER;
	goto array;

    case 'Z':
	t = BT_COMPLEX;
	goto array;

    case 'D':
	t = BT_DERIVED;
	goto array;

    default:
	internal_error("namelist_write(): Corrupt namelist");
    }

scalar:
    write_name(n);
    p = n->pointer;

    if (t == BT_DERIVED)
	write_derived(p, n->dt_info);
    else
	list_formatted_write(t, p, n->kind);

    return;

array:
    desc = n->pointer;
    rank = desc->rank;

    for(i=0; i<rank; i++) {
	if (desc->info[i].lbound > desc->info[i].ubound)
	    return;

	index[i] = desc->info[i].lbound;
    }

    write_name(n);

    rep = 0;
    q = NULL;

    do {
	p = desc->offset;
	for(i=0; i<rank; i++)
	    p += desc->info[i].mult * index[i];

	if (t == BT_DERIVED)
	    write_derived(p, n->dt_info);

	else {
	    if (rep == 0) {
		rep = 1;
		q = p;
	    } else if (memcmp(p, q, n->kind) == 0)
		rep++;

	    else {
		write_rep(rep);
		list_formatted_write(t, q, n->kind);

		rep = 1;
		q = p;
	    }
	}

    } while(!bump_element(desc, index));

    if (rep > 0) {
	write_rep(rep);
	list_formatted_write(t, q, n->kind);
    }
}


/* namelist_write()-- Do a namelist write */

void namelist_write(void) {
namelist_info *n;
int d, m;
char *p;

    d = current_unit->flags.delim;  
    if (d == DELIM_NONE)
	current_unit->flags.delim = DELIM_APOSTROPHE;

    m = strlen(ioparm->namelist);
    p = write_block(m + 3); 
    if (p == NULL)
	goto cleanup;

    p[0] = ' ';
    p[1] = '&';
    memcpy(p+2, ioparm->namelist, m);
    str_upper(p+2, m);
    p[m+2] = ' ';

    for(n=first_namelist(); n; n=n->next)
	namelist_write0(n);

    if (!is_internal_unit())
	next_record();

    p = write_block(2);
    if (p == NULL)
	goto cleanup;

    p[0] = ' ';
    p[1] = '/';

cleanup:
    namelist_done();
    current_unit->flags.delim = d;
}
