
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


/* Implement the non-IOLENGTH variant of the INQUIRY statement */

#include <string.h>
#include "runtime.h"

static char undefined[] = "UNDEFINED";


/* cf_strcpy()-- Copy a C string to a fortran string. */

static void cf_strcpy(char *dest, int dest_len, char *src) {
int src_len;

    src_len = strlen(src); 

    if (src_len >= dest_len)       /* This will truncate if too long.  */
	memcpy(dest, src, dest_len);

    else {
	memcpy(dest, src, src_len);
	memset(dest+src_len, ' ', dest_len-src_len);    /* Pad with spaces. */
    }
}



/* inquire_via_unit()-- Inquiry via unit number.  The unit might not
 * exist. */

static void inquire_via_unit(iounit_t *u) {
char *fname, *p;
int m, fname_len;

    if (u == NULL) {
	fname = NULL;
	fname_len = 0;

    } else {
	fname = u->file;
	fname_len = u->file_len;
    }

    if (ioparm->exist != NULL) {
	if (u != NULL)
	    m = 1;
	else if (ioparm->unit != NULL)
	    m = (extract_mint(ioparm->unit, ioparm->unit_kind) >= 0);
	else
	    m = 0;

	set_integer(m, ioparm->exist, ioparm->exist_kind);
    }

    if (ioparm->opened != NULL)
	set_integer((u != NULL), ioparm->opened, ioparm->opened_kind);

    if (ioparm->named != NULL) {
	m = (u != NULL && u->flags.status != STATUS_SCRATCH && fname_len > 0);
	set_integer(m, ioparm->named, ioparm->named_kind);
    }

    if (ioparm->number != NULL)
	set_mint((u != NULL) ? u->unit_number : -1,
		 ioparm->number, ioparm->number_kind);

    if (ioparm->name != NULL && u != NULL && u->flags.status != STATUS_SCRATCH)
	copy_string(ioparm->name, ioparm->name_len, fname, fname_len);

    p = NULL;

    if (ioparm->access != NULL) {
	if (u == NULL)
	    p = undefined;
	else
	    switch(u->flags.access) {
	    case ACCESS_SEQUENTIAL:   p = "SEQUENTIAL";  break;
	    case ACCESS_DIRECT:       p = "DIRECT";      break;
	    case ACCESS_STREAM:       p = "STREAM";      break;
	    default:
		internal_error("inquire_via_unit(): Bad access");
	    }

	cf_strcpy(ioparm->access, ioparm->access_len, p);
    }

    if (ioparm->sequential != NULL)
	cf_strcpy(ioparm->sequential, ioparm->sequential_len,
		  inquire_sequential_fd(u));

    if (ioparm->direct != NULL)
	cf_strcpy(ioparm->direct, ioparm->direct_len, inquire_direct_fd(u));

    if (ioparm->form != NULL) {
	if (u == NULL)
	    p = undefined;

	else
	    switch(u->flags.form) {
	    case FORM_FORMATTED:     p = "FORMATTED";     break;
	    case FORM_UNFORMATTED:   p = "UNFORMATTED";   break;
	    default:
		internal_error("inquire_via_unit(): Bad form");
	    }

	cf_strcpy(ioparm->form, ioparm->form_len, p);
    }

    if (ioparm->formatted != NULL)
	cf_strcpy(ioparm->formatted, ioparm->formatted_len,
		  inquire_formatted_fd(u));

    if (ioparm->unformatted != NULL)
	cf_strcpy(ioparm->unformatted, ioparm->unformatted_len,
		  inquire_unformatted_fd(u));

    if (ioparm->recl_out != NULL)
	*ioparm->recl_out = (u != NULL) ? u->recl : 0;

    if (ioparm->nextrec != NULL)
	set_mint((u != NULL) ? u->last_record : 0,
		 ioparm->nextrec, ioparm->nextrec_kind);

    if (ioparm->blank != NULL) {
	if (u == NULL)
	    p = undefined;
	else
	    switch(u->flags.blank) {
	    case BLANK_NULL:   p = "NULL";   break;
	    case BLANK_ZERO:   p = "ZERO";    break;
	    default:
		internal_error("inquire_via_unit(): Bad blank");
	    }

	cf_strcpy(ioparm->blank, ioparm->blank_len, p);
    }

    if (ioparm->position != NULL) {
	if (u == NULL || u->flags.access == ACCESS_DIRECT)
	    p = undefined;

	else if (u->repos || u->last_record != 0) {
	    if (u->endfile == AT_ENDFILE || u->endfile == AFTER_ENDFILE)
		p = "APPEND";
	    else if (u->last_record == 0)
		p = "REWIND";
	    else
		p = "ASIS";

	} else
	    switch(u->flags.position) {
	    case POSITION_ASIS:   p = "ASIS";      break;
	    case POSITION_REWIND: p = "REWIND";    break;
	    case POSITION_APPEND: p = "APPEND";    break;
	    default:              p = "UNDEFINED"; break;
	    }

	cf_strcpy(ioparm->position, ioparm->position_len, p);
    }

    if (ioparm->action != NULL) {
	if (u == NULL)
	    p = undefined;
	
	else
	    switch(u->flags.action) {
	    case ACTION_READ:       p = "READ";       break;
	    case ACTION_WRITE:      p = "WRITE";      break;
	    case ACTION_READWRITE:  p = "READWRITE";  break;
	    default:
		internal_error("inquire_via_unit(): Bad action");
	    }

	cf_strcpy(ioparm->action, ioparm->action_len, p);
    }

    if (ioparm->read != NULL)
	cf_strcpy(ioparm->read, ioparm->read_len, inquire_read_fd(u));

    if (ioparm->write != NULL)
	cf_strcpy(ioparm->write, ioparm->write_len, inquire_write_fd(u));

    if (ioparm->readwrite != NULL)
	cf_strcpy(ioparm->readwrite, ioparm->readwrite_len,
		  inquire_readwrite_fd(u));

    if (ioparm->delim != NULL) {
	if (u == NULL || u->flags.form != FORM_FORMATTED)
	    p = undefined;

	else
	    switch(u->flags.delim) {
	    case DELIM_NONE:         p = "NONE";         break;
	    case DELIM_QUOTE:        p = "QUOTE";        break;
	    case DELIM_APOSTROPHE:   p = "APOSTROPHE";   break;
	    default:
		internal_error("inquire_via_unit(): Bad delim");
	    }

	cf_strcpy(ioparm->delim, ioparm->delim_len, p);
    }

    if (ioparm->pad != NULL) {
	if (u == NULL || u->flags.form != FORM_FORMATTED)
	    p = "UNDEFINED";  /* F2003 */

	else
	    switch(u->flags.pad) {
	    case PAD_NO:     p = "NO";   break;
	    case PAD_YES:    p = "YES";  break;
	    default:
		internal_error("inquire_via_unit(): Bad pad");
	    }

	cf_strcpy(ioparm->pad, ioparm->pad_len, p);
    }

    if (ioparm->pos != NULL)
	set_mint((u == NULL) ? 0 : u->s->logical_offset+1,
		 ioparm->pos, ioparm->pos_kind);

    if (ioparm->size != NULL)
	set_mint((u == NULL) ? 0 : fd_length(u->s->fd),
		 ioparm->size, ioparm->size_kind);
}



/* inquire_via_filename()-- Inquiry via filename.  This subroutine is
 * only used if the filename is *not* connected to a unit number. */
 
static void inquire_via_filename(void) {
char *p;

    if (ioparm->exist != NULL)
	set_integer(file_exists(), ioparm->exist, ioparm->exist_kind);

    if (ioparm->opened != NULL)
	set_integer(0, ioparm->opened, ioparm->opened_kind);

    if (ioparm->named != NULL)
	set_integer(1, ioparm->named, ioparm->named_kind);

    if (ioparm->number != NULL)
	set_mint(-1, ioparm->number, ioparm->number_kind);

    if (ioparm->name != NULL)
	copy_string(ioparm->name, ioparm->name_len, ioparm->file,
		    ioparm->file_len);

    if (ioparm->access != NULL)
	cf_strcpy(ioparm->access, ioparm->access_len, undefined);

    if (ioparm->sequential != NULL) {
	p = inquire_sequential(ioparm->file, ioparm->file_len);
	cf_strcpy(ioparm->sequential, ioparm->sequential_len, p);
    }

    if (ioparm->direct != NULL) {
	p = inquire_direct(ioparm->file, ioparm->file_len);
	cf_strcpy(ioparm->direct, ioparm->direct_len, p);
    }

    if (ioparm->form != NULL)
	cf_strcpy(ioparm->form, ioparm->form_len, undefined);

    if (ioparm->formatted != NULL) {
	p = inquire_formatted(ioparm->file, ioparm->file_len);
	cf_strcpy(ioparm->formatted, ioparm->formatted_len, p);
    }

    if (ioparm->unformatted != NULL) {
	p = inquire_unformatted(ioparm->file, ioparm->file_len);
	cf_strcpy(ioparm->unformatted, ioparm->unformatted_len, p);
    }

    if (ioparm->recl_out != NULL)
	*ioparm->recl_out = 0;

    if (ioparm->nextrec != NULL)
	set_mint(0, ioparm->nextrec, ioparm->nextrec_kind);

    if (ioparm->blank != NULL)
	cf_strcpy(ioparm->blank, ioparm->blank_len, undefined);

    if (ioparm->position != NULL)
	cf_strcpy(ioparm->position, ioparm->position_len, undefined);

    if (ioparm->access != NULL)
	cf_strcpy(ioparm->access, ioparm->access_len, undefined);

    if (ioparm->read != NULL) {
	p = inquire_read(ioparm->file, ioparm->file_len);
	cf_strcpy(ioparm->read, ioparm->read_len, p);
    }

    if (ioparm->write != NULL) {
	p = inquire_write(ioparm->file, ioparm->file_len);
	cf_strcpy(ioparm->write, ioparm->write_len, p);
    }

    if (ioparm->readwrite != NULL) {
	p = inquire_read(ioparm->file, ioparm->file_len);
	cf_strcpy(ioparm->readwrite, ioparm->readwrite_len, p);
    }

    if (ioparm->delim != NULL)
	cf_strcpy(ioparm->delim, ioparm->delim_len, undefined);

    if (ioparm->pad != NULL)
	cf_strcpy(ioparm->pad, ioparm->pad_len, undefined);

    if (ioparm->pos != NULL)
	set_mint(0, ioparm->pos, ioparm->pos_kind);

    if (ioparm->size != NULL)
	set_mint(filename_length(), ioparm->size, ioparm->size_kind);
}




void st_inquire(void) {
iounit_t *u;

    library_start(); 

    if (ioparm->file == NULL)
	inquire_via_unit(find_unit(ioparm->unit, ioparm->unit_kind));

    else {
	u = find_file();
	if (u == NULL)
	    inquire_via_filename();
	else
	    inquire_via_unit(u);
    }

    library_end();
}

