
/* Copyright (C) 2003-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught and George Marsaglia

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

/* This implementation uses Marsaglia's xorshift-96 random number
 * generator.  It is very fast and has a large period with the minimum
 * possible number of state variables.  It passes all the tests in his
 * DIEHARD random test suite.  The only caveat on the seed is that the
 * values cannot all be zero. */

#include <string.h>
#include "runtime.h"

#define m16Mask 0xFFFF           /* mask for lower 16 bits */
#define m31Mask 0x7FFFFFFF       /* mask for 31 bits */
#define m32Double  4294967296.0  /* 2^32 */
#define m24Double  16777216.0    /* 2^24 */

#define M1 0xf95f9500
#define M2 0x0f95f950
#define M3 0x00f95f95
#define M4 0x22222222

static unsigned int seed = 0x0f95f950;

static unsigned int legacy_x = 887341857, legacy_y = 32967915,
                    legacy_z = 890712030;


/* The mask array is for scrambling and unscrambling the user supplied
 * seed.  The numbers that people are apt to provide lead to some
 * initially nonrandom-looking numbers. */


#define STATE_SIZE 4

static unsigned int rand_state[STATE_SIZE],
       rand_mask[STATE_SIZE] = { M1, M2, M3, M4 };

#define random_4     prefix(random_4)
#define random_8     prefix(random_8)
#define random_10    prefix(random_10)
#define random_seed  prefix(random_seed)

void random_4(float *);
void random_8(double *);


/* xorshf96()-- Generate a uniform integer */

static unsigned int xorshf96(void) {    /* period 2^96-1 */
unsigned int t;

    rand_state[0] ^= rand_state[0] << 16;
    rand_state[0] ^= rand_state[0] >> 5;
    rand_state[0] ^= rand_state[0] << 1;

    t = rand_state[0];
    rand_state[0] = rand_state[1];
    rand_state[1] = rand_state[2];
    rand_state[2] = t ^ rand_state[0] ^ rand_state[1];

    rand_state[3] = (16807 * rand_state[3]) % 2147483647;  /* LCG */

    return rand_state[2] ^ rand_state[3];
}



/* time_seed()-- Get a seed involving the current time. */

#if HAVE_WINDOWS
#include <windows.h>

static int time_seed(void) {
LARGE_INTEGER count;

    QueryPerformanceCounter(&count);
    return (int) count.LowPart;
}

#else

#include <sys/time.h>
#include <time.h>

static int time_seed(void) {
struct timeval tv;

    gettimeofday(&tv, NULL);
    return (tv.tv_usec << 16) ^ tv.tv_sec;
}

#endif



/* init_random_seed()-- Use a small multiply with carry generator to
 * generate a new seed for the main generator.  This is called on
 * initialization and when RANDOM_SEED() is called with no
 * arguments */

void init_random_seed(int starting) {
unsigned int number, *p;
unsigned short sNumber;
int n;

    if (starting && options.seed_rng)
	seed = time_seed();

    sNumber = seed & m16Mask;    /* The low 16 bits */
    number  = seed & m31Mask;    /* Only want 31 bits */

    p = rand_state;
    for(n=0; n<STATE_SIZE; n++) {
	number = 30903*sNumber + (number>>16); /*One line multiply-with-carry*/
	*p = sNumber = number & m16Mask;

	number = 30903*sNumber + (number>>16); /*One line multiply-with-carry*/
	sNumber = number & m16Mask;
	*p++ |= sNumber << 16;
    }

    seed = number;
}



/* check_seed()-- Check a seed set by a user to make sure that it is
 * OK.  For the xorshf96 generator, a vector of all zeros is bad
 * because it causes the generation of nothing but zeros.  If the seed
 * is all zeros then we map it to something else.  The important thing
 * is that the same seed being input must cause the same sequence of
 * random numbers to be produced.  */

static void check_seed(void) {
unsigned int i;

    if (rand_state[0] != 0 || rand_state[1] == 0 || rand_state[2] == 0 ||
	rand_state[3] != 0)
	return;

    /* Replace the current seed with a "good" constant */

    i = seed;
    seed = 0xf95f9500;

    init_random_seed(0);
    seed = i;
}



/* random_4()-- Kind 4 RANDOM_NUMBER intrinsic.  Because the random
 * integer is 32 bits and the mantissa of a float is 24 bits, the
 * number can get rounded up to 1.0, which is illegal. */

void random_4(float *harvest) {

    *harvest = (xorshf96() >> 8) * (1.0 / m24Double);
}



/* random_8()-- Kind 8 RANDOM_NUMBER intrinsic.  TODO: Build random
 * number from a random mantissa.  A kind 8 real has 52 bits in the
 * mantissa, 32 of which are actually random. */

void random_8(double *harvest) {

    *harvest = xorshf96() * (1.0 / m32Double);
}



#if HAVE_REAL_10

void random_10(unsigned *harvest) {
int scale;

    scale = -63;
    harvest[0] = xorshf96();
    harvest[1] = xorshf96() & 0x7FFFFFFF;

    asm("fild %0\n"
	"mov %1, %" EAX "\n"
	"fildll (%" EAX ")\n"
	"fscale\n"
	"fstpt (%" EAX ")\n"
	"fstp %%st\n" : : "m" (scale), "m" (harvest) : EAX);
}

#endif


/* random_seed()-- Implement the RANDOM_SEED() intrinsic subroutine. */

void random_seed(G95_DINT *size, g95_array_descriptor *put,
		 g95_array_descriptor *get) {
G95_DINT *m, *p, *q;
int i, n;

    if (size == NULL && put == NULL && get == NULL) {
	seed ^= time_seed();

	for(i=0; i<10; i++)
	    init_random_seed(0);  /* Churn it around a little */

	check_seed();
	return;
    }

    /* n is the number of default integers in the rand_state[] array */

    n = STATE_SIZE * sizeof(int) / sizeof(G95_DINT);

    if (size != NULL)
	*size = n;

    /* Do the GET before the PUT */

    if (get != NULL) {
	if (get->info[0].ubound - get->info[0].lbound + 1 < n)
	    runtime_error("RANDOM_SEED(): GET array too small");

	p = (G95_DINT *) &rand_state;
	m = (G95_DINT *) &rand_mask;

	for(i=0; i<n; i++) {
	    q = (G95_DINT *)
		(get->offset+(i+get->info[0].lbound)*get->info[0].mult);
	    *q++ = (*p++) ^ (*m++);
	}
    }

    if (put != NULL) {
	if (put->info[0].ubound - put->info[0].lbound + 1 < n)
	    runtime_error("RANDOM_SEED(): PUT array too small");

	p = (G95_DINT *) &rand_state;
	m = (G95_DINT *) &rand_mask;

	for(i=0; i<n; i++) {
	    q = (G95_DINT *)
		(put->offset+(i+put->info[0].lbound)*put->info[0].mult);
	    *p++ = (*q++) ^ (*m++);
	}

	check_seed();
    }
}



#define srand prefix(srand)

/* srand()-- Legacy function srand() */

void srand(G95_DINT *seed) {

    legacy_x = (*seed<<1) | 1;
    legacy_x = legacy_x * 3 * legacy_x;

    legacy_y = 1221199477;
    legacy_z = 1953104437;
}



/* rand()-- Legacy function rand() */

#define rand prefix(rand)

float rand(G95_DINT *flag) {
unsigned int v;

    if (flag != NULL && *flag != 0)
	srand(flag);

    v = legacy_x * legacy_y;
    legacy_x = legacy_y;
    legacy_y = v;
    legacy_z = (legacy_z & 65535) * 30903 + (legacy_z >> 16);

    v = (legacy_y + (legacy_z & 65535));

    return (float) v / 4294967296.0;
}
