#if !defined  HAVE_CATALAN_PATH_LEX_H__
#define       HAVE_CATALAN_PATH_LEX_H__
// This file is part of the FXT library.
// Copyright (C) 2012, 2013, 2014, 2019, 2023, 2024 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "comb/is-catalan-path.h"
#include "comb/print-catalan-path-aa.h"
#include "comb/comb-print.h"

#include "fxttypes.h"


#define CATALAN_PATH_LEX_OPT  // default on (modest speedup)

class catalan_path_lex
// Catalan paths in lexicographic order, CAT algorithm.
// Steps are +1, -1 (up, down),
// the first and last elements are 0, all elements are non-negative,
// and even/odd positions respectively have even/odd entries only.
{
public:
    ulong *a_;
    ulong n_;

private:  // have pointer data
    catalan_path_lex(const catalan_path_lex&) = delete;
    catalan_path_lex & operator = (const catalan_path_lex&) = delete;

public:
    explicit catalan_path_lex(ulong n)
    // Must have n>=1.
    {
        n_ = 2 * n;  // length = 2 * n

        if ( n==0 )  n_ = 1;  // make things work for n==0


        a_ = new ulong[n_+2];
        a_[0] = 0;  // lower sentinel
        ++a_;  // nota bene
        a_[n_] = 0;  // used in print() as final element

        first();
    }

    ~catalan_path_lex()
    {
        --a_;
        delete [] a_;
    }

    void first()
    {
        for (ulong j=0; j<n_; ++j)  a_[j] = j & 1UL;
    }

    const ulong * data()  const  { return a_; }

    ulong next()
    {
#if defined CATALAN_PATH_LEX_OPT
        // easy case optimization
        ulong j = n_ - 1;
        if ( a_[j] == 0 )  // easy case: trailing zero becomes one
        {
            a_[j] = 1;
            return j;
        }

        a_[j] = j & 1;
        --j;
        ulong m = 2;

#else  // CATALAN_PATH_LEX_OPT

        ulong j = n_ - 1;
        ulong m = 1;
#endif  // CATALAN_PATH_LEX_OPT


        // Scan falling part:
        while ( a_[j] == m )
        {
            a_[j] = j & 1;
            ++m;
            --j;
        }
        // here m - 1 == t

        // Scan rising part:
        while ( a_[j] > a_[j-1] )  // can read lower sentinel
        {
            a_[j] = j & 1;
            --j;
        }

        if ( j==0 )   return 0;  // current is last


        if ( a_[j-1] == 0 )
        {
            a_[j-1] = 1;
            a_[j] = 0;
            return  j - 1;  // leftmost position changed
        }
        else
        {
            const ulong i = j;
            ulong v = a_[j] + 1;
            a_[j] = v + 1;
            ++j;

            do
            {
                a_[j] = v;
                --v;
                ++j;
            }
            while ( v );

            return i;  // leftmost position changed
        }
    }


    ulong area()  const
    {
        ulong s = 0;
        for (ulong j=1; j<n_; ++j)  s += a_[j];
        return s;
    }

    void print(const char *bla, bool dfz=false)  const
    { print_vec(bla, a_, n_+1, dfz); }

    void print_vert_aa()  const  // ASCII art
    {
        print_catalan_path_vert_aa(data(), n_, true);
    }

    void print_horiz_aa()  const  // ASCII art
    { print_catalan_path_horiz_aa(data(), n_); }

    bool OK()  const
    { return is_catalan_path( data(), n_ ); }
};
// -------------------------


#endif  // !defined HAVE_CATALAN_PATH_LEX_H__
