
#include "comb/mixedradix-fixed-content-lex.h"

#include "fxttypes.h"
#include "fxtalloca.h"
#include "fxtio.h"
#include "nextarg.h"  // NXARG()
#include "jjassert.h"

#include <cstdlib>  // strtoul()

//% Mixed radix counting, fixed content.
//% Equivalently, subsets of a multiset with a fixed number of elements.
//% Lexicographic order.

// Cf. comb/mixedradix-fixed-content-demo.cc
// Cf. comb/mixedradix-rev-demo.cc

//#define TIMING  // uncomment to disable printing

int
main(int argc, char **argv)
{
    ulong n = 5;
    NXARG(n, "Number of digits");
    ulong c = 4;
    NXARG(c, "Content");
    ulong rr = 3;
    NXARG(rr, "Base (radix) of digits (0==>falling factorial, 1==>rising factorial)");

    ulong ct = 0;

#if defined TIMING
    mixedradix_fixed_content_lex M(n, c, rr);

    const bool q = M.first();
    jjassert( q );
    do  { ++ct; }  while ( M.next() );

#else  // TIMING

    ALLOCA(ulong, r, n);
    for (ulong j=0; j<n; ++j)  r[j] = rr;
    RESTARGS("Optionally supply radix for all digits (rr ignored)");
    if ( argc>4 )  jjassert( argc == (int)n+4 );
    for (ulong k=4;  k<(ulong)argc; ++k)  r[k-4] = strtoul(argv[k], nullptr, 10);

    mixedradix_fixed_content_lex M(n, c, rr, ( argc > 4 ? r : nullptr ) );
    const bool q = M.first();
    jjassert( q );
    M.print_nines("Nines: ");  cout << endl;

    do
    {
        cout << " " << setw(4) << ct << ":";
        M.print("  ", true );
        cout << "  " << M.pos();
        print_multi_deltaset_as_set("    ", M.data(), n, 1 );
        cout << endl;

        ++ct;
    }
    while ( M.next() );
#endif  // TIMING

    cout << " ct=" << ct << endl;

    return 0;
}
// -------------------------

/*
Timing: Intel(R) Core(TM) i7-8700K CPU @ 3.70GHz
GCC 12.2.0

time ./bin 32 16 2 # combinations
arg 1: 32 == n  [Number of digits]  default=5
arg 2: 16 == c  [Content (c >= 1)]  default=5
arg 3: 2 == rr  [Base (radix) of digits (0==>falling factorial, 1==>rising factorial)]  default=3
 ct=601080390
2.88user 0.00system 0:02.89elapsed 99%CPU
 ==> 601080390 / 2.88 == 208,708,468 per second

time ./bin 16 16 8
arg 1: 16 == n  [Number of digits]  default=5
arg 2: 16 == c  [Content (c >= 1)]  default=5
arg 3: 8 == rr  [Base (radix) of digits (0==>falling factorial, 1==>rising factorial)]  default=3
 ct=292695291
1.20user 0.00system 0:01.21elapsed 99%CPU
 ==> 292695291 / 1.20 == 243,912,742 per second

 time ./bin 32 10 3
arg 1: 32 == n  [Number of digits]  default=5
arg 2: 10 == c  [Content (c >= 1)]  default=5
arg 3: 3 == rr  [Base (radix) of digits (0==>falling factorial, 1==>rising factorial)]  default=3
 ct=743063056
4.80user 0.00system 0:04.80elapsed 100%CPU
 ==> 743063056 / 4.80 == 154,804,803 per second

*/

/// Emacs:
/// Local Variables:
/// MyRelDir: "demo/comb"
/// makefile-dir: "../../"
/// make-target: "1demo DSRC=demo/comb/mixedradix-fixed-content-lex-demo.cc"
/// make-target2: "1demo DSRC=demo/comb/mixedradix-fixed-content-lex-demo.cc DEMOFLAGS=-DTIMING"
/// End:

