
#include "comb/combination-revdoor.h"

#include "aux0/binomial.h"

#include "fxttypes.h"
#include "jjassert.h"
#include "nextarg.h"

#include "fxtio.h"

//% Combinations in a minimal-change order.
//% Algorithm R, "revolving-door combinations", TAOCP 4A/1, pp.363.
//%  W. H. Payne, F. M. Ives: "Combination Generators",
//%  ACM Transactions on Mathematical Software (TOMS),
//%  vol.5, no.2, pp.163-172, (June-1979).

//#define TIMING  // uncomment to disable printing


int
main(int argc, char **argv)
{
    ulong n = 7;
    NXARG(n, "Combination (n choose k) in minimal-change order: n>=1");
    ulong k = 4;
    NXARG(k, "(n choose k):  1<=k<=n");
    jjassert( k<=n );
    jjassert( k>=1 );
    jjassert( n>=1 );

    combination_revdoor C(n, k);

    ulong ct = 0;
    do
    {
#if !defined TIMING
        cout << setw(4) << ct << ":  ";
        C.print_set("  ");
        C.print_deltaset("    ");
        cout << endl;
#endif  // TIMING
        ++ct;
    }
    while ( C.next() );

    ulong bnk =  binomial(n, k);
    cout << "binomial(" << n << ", " << k << ")=" << bnk;
    jjassert( bnk==ct );
    cout << endl;

    return 0;
}
// -------------------------

// for n in $(seq 1 10); do for k in $(seq 1 $n); do ./bin $n $k || break 2; done; done

/*
Timing: Intel(R) Core(TM) i7-8700K CPU @ 3.70GHz
GCC 12.2.0

 time ./bin 32 20
arg 1: 32 == n  [Combination (n choose k) in minimal-change order: n>=1]  default=7
arg 2: 20 == k  [(n choose k):  1<=k<=n]  default=4
binomial(32, 20)=225792840
0.39user 0.00system 0:00.39elapsed 100%CPU
 ==> 225792840/0.39 == 578,956,000 per second

 time ./bin 32 12
arg 1: 32 == n  [Combination (n choose k) in minimal-change order: n>=1]  default=7
arg 2: 12 == k  [(n choose k):  1<=k<=n]  default=4
binomial(32, 12)=225792840
0.23user 0.00system 0:00.23elapsed 100%CPU
 ==> 225792840/0.23 == 981,708,000 per second


 time ./bin 64 7
arg 1: 64 == n  [Combination (n choose k) in minimal-change order: n>=1]  default=7
arg 2: 7 == k  [(n choose k):  1<=k<=n]  default=4
binomial(64, 7)=621216192
0.50user 0.00system 0:00.50elapsed 100%CPU
 ==> 621216192/0.50 == 1,242,432,384 per second

 time ./bin 64 57
arg 1: 64 == n  [Combination (n choose k) in minimal-change order: n>=1]  default=7
arg 2: 57 == k  [(n choose k):  1<=k<=n]  default=4
binomial(64, 57)=621216192
2.70user 0.00system 0:02.70elapsed 100%CPU
 ==> 621216192/2.70 == 230,080,071 per second

*/


/*
Timing: (AMD Phenom II X4 945 3000MHz)

  time ./bin 32 20
binomial(32, 20)=225792840
./bin 32 20  0.96s user 0.00s system 99% cpu 0.958 total
 ==> 225792840/0.96 == 235,200,875 per second

 time ./bin 32 12
binomial(32, 12)=225792840
./bin 32 12  0.66s user 0.00s system 99% cpu 0.661 total
 ==> 225792840/0.66 == 342,110,363 per second

*/


/// Emacs:
/// Local Variables:
/// MyRelDir: "demo/comb"
/// makefile-dir: "../../"
/// make-target: "1demo DSRC=demo/comb/combination-revdoor-demo.cc"
/// make-target2: "1demo DSRC=demo/comb/combination-revdoor-demo.cc DEMOFLAGS=-DTIMING"
/// End:

