// This file is part of the FXT library.
// Copyright (C) 2010, 2012 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "fxttypes.h"
#include "perm/graypermute.h"
#include "perm/ziprev.h"
#include "perm/revbinpermute.h"
#include "walsh/walshwak.h"

#include <cmath>  // sqrt()

void
slant(double *f, ulong ldn)
// Slant transform.
{
    walsh_wak(f, ldn);

    ulong n = 1UL<<ldn;
    for (ulong ldm=0; ldm<ldn-1; ++ldm)
    {
        ulong m = 1UL<<ldm;  // m = 1, 2, 4, 8, ..., n/4
        double N = (double)m*2.0,  N2 = N*N;
        double a = sqrt(3.0*N2/(4.0*N2-1.0));
        double b = sqrt(1.0-a*a);  // == sqrt((N2-1)/(4*N2-1));
        for (ulong j=m; j<n-1; j+=4*m)
        {
            ulong t1 = j;
            ulong t2 = j + m;
            double f1 = f[t1],  f2 = f[t2];
            f[t1] = a * f1 - b * f2;
            f[t2] = b * f1 + a * f2;
        }
    }
}
// -------------------------

void
inverse_slant(double *f, ulong ldn)
// Inverse of slant()
{
    ulong n = 1UL<<ldn;
    ulong ldm=ldn-2;
    do
    {
        ulong m = 1UL<<ldm;  // m = n/4, n/2, ..., 4, 2, 1
        double N = (double)m*2.0,  N2 = N*N;
        double a = sqrt(3.0*N2/(4.0*N2-1.0));
        double b = sqrt(1.0-a*a);  // == sqrt((N2-1)/(4*N2-1));
        for (ulong j=m; j<n-1; j+=4*m)
        {
            ulong t1 = j;
            ulong t2 = j + m;
            double f1 = f[t1],  f2 = f[t2];
            f[t1] = b * f2 + a * f1;
            f[t2] = a * f2 - b * f1;
        }
    }
    while ( ldm-- );

    walsh_wak(f, ldn);
}
// -------------------------

void
slant_seq(double *f, ulong ldn)
// Sequency-ordered slant transform.
{
    slant(f, ldn);

    ulong n = 1UL<<ldn;
    inverse_gray_permute(f, n);
    unzip_rev(f, n);
    revbin_permute(f, n);
}
// -------------------------

void
inverse_slant_seq(double *f, ulong ldn)
// Inverse of slant_seq().
{
    ulong n = 1UL<<ldn;
    revbin_permute(f, n);
    zip_rev(f, n);
    gray_permute(f, n);

    inverse_slant(f, ldn);
}
// -------------------------

// cf. Zhong-De Wang:  "New Algorithm for the Slant Transform"
// IEEE Transactions on Pattern Analysis and Machine Intelligence,
// Vol.PAMI-4, No.5, September 1982

//  m=1   N=2  a=0.894427=2/sqrt(5)  b=0.447214=1/sqrt(5)
//  m=2   N=4  a=0.872872  b=0.48795
//  m=4   N=8  a=0.867722  b=0.49705
//  m=8   N=16  a=0.866449  b=0.499266
//  m=16   N=32  a=0.866131  b=0.499817
//  m=32   N=64  a=0.866052  b=0.499954
//  m=64   N=128  a=0.866032  b=0.499989
//  m=128   N=256  a=0.866027  b=0.499997
//  m=256   N=512  a=0.866026  b=0.499999
//  m=512   N=1024  a=0.866026  b=0.5
//  m=1024   N=2048  a=0.866025  b=0.5
//
//  m=\infty         a=sqrt(3)/2 b=1/2


// basis functions of slant_seq() produced by the inverse transform of
// delta peaks:
//
//  ++++++++++++++++++++++++++++++++++++ 0  seq=0
//   norm = 4
//                             |---------------------------*                 1   0
//                             |---------------------------*                 1   1
//                             |---------------------------*                 1   2
//                             |---------------------------*                 1   3
//                             |---------------------------*                 1   4
//                             |---------------------------*                 1   5
//                             |---------------------------*                 1   6
//                             |---------------------------*                 1   7
//                             |---------------------------*                 1   8
//                             |---------------------------*                 1   9
//                             |---------------------------*                 1   10
//                             |---------------------------*                 1   11
//                             |---------------------------*                 1   12
//                             |---------------------------*                 1   13
//                             |---------------------------*                 1   14
//                             |---------------------------*                 1   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 1  seq=1
//   norm = 4
//                              |--------------------------*           1.62698   0
//                              |----------------------*               1.41005   1
//                              |-------------------*                  1.19312   2
//                              |---------------*                     0.976187   3
//                              |-----------*                         0.759257   4
//                              |--------*                            0.542326   5
//                              |----*                                0.325396   6
//                              |*                                    0.108465   7
//                            *-|                                    -0.108465   8
//                        *-----|                                    -0.325396   9
//                    *---------|                                    -0.542326   10
//                 *------------|                                    -0.759257   11
//             *----------------|                                    -0.976187   12
//          *-------------------|                                     -1.19312   13
//      *-----------------------|                                     -1.41005   14
//  *---------------------------|                                     -1.62698   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 2  seq=2
//   norm = 4
//                              |--------------------------*           1.52753   0
//                              |------------------*                   1.09109   1
//                              |----------*                          0.654654   2
//                              |--*                                  0.218218   3
//                          *---|                                    -0.218218   4
//                  *-----------|                                    -0.654654   5
//          *-------------------|                                     -1.09109   6
//  *---------------------------|                                     -1.52753   7
//  *---------------------------|                                     -1.52753   8
//          *-------------------|                                     -1.09109   9
//                  *-----------|                                    -0.654654   10
//                          *---|                                    -0.218218   11
//                              |--*                                  0.218218   12
//                              |----------*                          0.654654   13
//                              |------------------*                   1.09109   14
//                              |--------------------------*           1.52753   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 3  seq=3
//   norm = 4
//                             |------------*                         0.828417   0
//                             |------*                               0.449712   1
//                             |-*                                   0.0710072   2
//                         *---|                                     -0.307698   3
//                   *---------|                                     -0.686403   4
//              *--------------|                                      -1.06511   5
//        *--------------------|                                      -1.44381   6
//  *--------------------------|                                      -1.82252   7
//                             |---------------------------*           1.82252   8
//                             |---------------------*                 1.44381   9
//                             |---------------*                       1.06511   10
//                             |----------*                           0.686403   11
//                             |----*                                 0.307698   12
//                            *|                                    -0.0710072   13
//                       *-----|                                     -0.449712   14
//                 *-----------|                                     -0.828417   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 4  seq=4
//   norm = 4
//                              |--------------------------*           1.34164   0
//                              |--------*                            0.447214   1
//                    *---------|                                    -0.447214   2
//  *---------------------------|                                     -1.34164   3
//  *---------------------------|                                     -1.34164   4
//                    *---------|                                    -0.447214   5
//                              |--------*                            0.447214   6
//                              |--------------------------*           1.34164   7
//                              |--------------------------*           1.34164   8
//                              |--------*                            0.447214   9
//                    *---------|                                    -0.447214   10
//  *---------------------------|                                     -1.34164   11
//  *---------------------------|                                     -1.34164   12
//                    *---------|                                    -0.447214   13
//                              |--------*                            0.447214   14
//                              |--------------------------*           1.34164   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 5  seq=5
//   norm = 4
//                              |--------------------------*           1.34164   0
//                              |--------*                            0.447214   1
//                    *---------|                                    -0.447214   2
//  *---------------------------|                                     -1.34164   3
//  *---------------------------|                                     -1.34164   4
//                    *---------|                                    -0.447214   5
//                              |--------*                            0.447214   6
//                              |--------------------------*           1.34164   7
//  *---------------------------|                                     -1.34164   8
//                    *---------|                                    -0.447214   9
//                              |--------*                            0.447214   10
//                              |--------------------------*           1.34164   11
//                              |--------------------------*           1.34164   12
//                              |--------*                            0.447214   13
//                    *---------|                                    -0.447214   14
//  *---------------------------|                                     -1.34164   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 6  seq=6
//   norm = 4
//                             |-----------*                           0.68313   0
//                            *|                                      -0.09759   1
//               *-------------|                                      -0.87831   2
//  *--------------------------|                                      -1.65903   3
//                             |---------------------------*           1.65903   4
//                             |--------------*                        0.87831   5
//                             |-*                                     0.09759   6
//                  *----------|                                      -0.68313   7
//                  *----------|                                      -0.68313   8
//                             |-*                                     0.09759   9
//                             |--------------*                        0.87831   10
//                             |---------------------------*           1.65903   11
//  *--------------------------|                                      -1.65903   12
//               *-------------|                                      -0.87831   13
//                            *|                                      -0.09759   14
//                             |-----------*                           0.68313   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 7  seq=7
//   norm = 4
//                             |-----------*                           0.68313   0
//                            *|                                      -0.09759   1
//               *-------------|                                      -0.87831   2
//  *--------------------------|                                      -1.65903   3
//                             |---------------------------*           1.65903   4
//                             |--------------*                        0.87831   5
//                             |-*                                     0.09759   6
//                  *----------|                                      -0.68313   7
//                             |-----------*                           0.68313   8
//                            *|                                      -0.09759   9
//               *-------------|                                      -0.87831   10
//  *--------------------------|                                      -1.65903   11
//                             |---------------------------*           1.65903   12
//                             |--------------*                        0.87831   13
//                             |-*                                     0.09759   14
//                  *----------|                                      -0.68313   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 8  seq=8
//   norm = 4
//                             |---------------------------*                 1   0
//  *--------------------------|                                            -1   1
//  *--------------------------|                                            -1   2
//                             |---------------------------*                 1   3
//                             |---------------------------*                 1   4
//  *--------------------------|                                            -1   5
//  *--------------------------|                                            -1   6
//                             |---------------------------*                 1   7
//                             |---------------------------*                 1   8
//  *--------------------------|                                            -1   9
//  *--------------------------|                                            -1   10
//                             |---------------------------*                 1   11
//                             |---------------------------*                 1   12
//  *--------------------------|                                            -1   13
//  *--------------------------|                                            -1   14
//                             |---------------------------*                 1   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 9  seq=9
//   norm = 4
//                             |---------------------------*                 1   0
//  *--------------------------|                                            -1   1
//  *--------------------------|                                            -1   2
//                             |---------------------------*                 1   3
//                             |---------------------------*                 1   4
//  *--------------------------|                                            -1   5
//  *--------------------------|                                            -1   6
//                             |---------------------------*                 1   7
//  *--------------------------|                                            -1   8
//                             |---------------------------*                 1   9
//                             |---------------------------*                 1   10
//  *--------------------------|                                            -1   11
//  *--------------------------|                                            -1   12
//                             |---------------------------*                 1   13
//                             |---------------------------*                 1   14
//  *--------------------------|                                            -1   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 10  seq=10
//   norm = 4
//                             |---------------------------*                 1   0
//  *--------------------------|                                            -1   1
//  *--------------------------|                                            -1   2
//                             |---------------------------*                 1   3
//  *--------------------------|                                            -1   4
//                             |---------------------------*                 1   5
//                             |---------------------------*                 1   6
//  *--------------------------|                                            -1   7
//  *--------------------------|                                            -1   8
//                             |---------------------------*                 1   9
//                             |---------------------------*                 1   10
//  *--------------------------|                                            -1   11
//                             |---------------------------*                 1   12
//  *--------------------------|                                            -1   13
//  *--------------------------|                                            -1   14
//                             |---------------------------*                 1   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 11  seq=11
//   norm = 4
//                             |---------------------------*                 1   0
//  *--------------------------|                                            -1   1
//  *--------------------------|                                            -1   2
//                             |---------------------------*                 1   3
//  *--------------------------|                                            -1   4
//                             |---------------------------*                 1   5
//                             |---------------------------*                 1   6
//  *--------------------------|                                            -1   7
//                             |---------------------------*                 1   8
//  *--------------------------|                                            -1   9
//  *--------------------------|                                            -1   10
//                             |---------------------------*                 1   11
//  *--------------------------|                                            -1   12
//                             |---------------------------*                 1   13
//                             |---------------------------*                 1   14
//  *--------------------------|                                            -1   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 12  seq=12
//   norm = 4
//                              |--------*                            0.447214   0
//  *---------------------------|                                     -1.34164   1
//                              |--------------------------*           1.34164   2
//                    *---------|                                    -0.447214   3
//                    *---------|                                    -0.447214   4
//                              |--------------------------*           1.34164   5
//  *---------------------------|                                     -1.34164   6
//                              |--------*                            0.447214   7
//                              |--------*                            0.447214   8
//  *---------------------------|                                     -1.34164   9
//                              |--------------------------*           1.34164   10
//                    *---------|                                    -0.447214   11
//                    *---------|                                    -0.447214   12
//                              |--------------------------*           1.34164   13
//  *---------------------------|                                     -1.34164   14
//                              |--------*                            0.447214   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 13  seq=13
//   norm = 4
//                              |--------*                            0.447214   0
//  *---------------------------|                                     -1.34164   1
//                              |--------------------------*           1.34164   2
//                    *---------|                                    -0.447214   3
//                    *---------|                                    -0.447214   4
//                              |--------------------------*           1.34164   5
//  *---------------------------|                                     -1.34164   6
//                              |--------*                            0.447214   7
//                    *---------|                                    -0.447214   8
//                              |--------------------------*           1.34164   9
//  *---------------------------|                                     -1.34164   10
//                              |--------*                            0.447214   11
//                              |--------*                            0.447214   12
//  *---------------------------|                                     -1.34164   13
//                              |--------------------------*           1.34164   14
//                    *---------|                                    -0.447214   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 14  seq=14
//   norm = 4
//                              |--------*                            0.447214   0
//  *---------------------------|                                     -1.34164   1
//                              |--------------------------*           1.34164   2
//                    *---------|                                    -0.447214   3
//                              |--------*                            0.447214   4
//  *---------------------------|                                     -1.34164   5
//                              |--------------------------*           1.34164   6
//                    *---------|                                    -0.447214   7
//                    *---------|                                    -0.447214   8
//                              |--------------------------*           1.34164   9
//  *---------------------------|                                     -1.34164   10
//                              |--------*                            0.447214   11
//                    *---------|                                    -0.447214   12
//                              |--------------------------*           1.34164   13
//  *---------------------------|                                     -1.34164   14
//                              |--------*                            0.447214   15
//
//
//  ++++++++++++++++++++++++++++++++++++ 15  seq=15
//   norm = 4
//                              |--------*                            0.447214   0
//  *---------------------------|                                     -1.34164   1
//                              |--------------------------*           1.34164   2
//                    *---------|                                    -0.447214   3
//                              |--------*                            0.447214   4
//  *---------------------------|                                     -1.34164   5
//                              |--------------------------*           1.34164   6
//                    *---------|                                    -0.447214   7
//                              |--------*                            0.447214   8
//  *---------------------------|                                     -1.34164   9
//                              |--------------------------*           1.34164   10
//                    *---------|                                    -0.447214   11
//                              |--------*                            0.447214   12
//  *---------------------------|                                     -1.34164   13
//                              |--------------------------*           1.34164   14
//                    *---------|                                    -0.447214   15
//
