"""
Linux syscall fuzzer: generate random syscalls

Project based on "sysfuzz.c" fuzzer by Digital Dwarf Society
   http://www.digitaldwarf.be/
"""

from fusil.c_tools import FuzzyFunctionC, CodeC, MIN_INT32, MAX_INT32
from fusil.process.create import CreateProcess
from fusil.process.watch import WatchProcess
from random import choice, randint
from fusil.project_agent import ProjectAgent
from fusil.linux.syslog import Syslog

USERLAND_ADDRESS = "0x0804fd00"
UNMAPPED_ADDRESS = "0x0000a000"
# kernel addr, this is a guess ... should actually get a real one ...
KERNEL_ADDRESS = "0xc01fa0b6"

def setupProject(project):
    IGNORE_SYSCALLS = set((
        2, 120, 190, # fork, clone, vfork
        29, 72, # pause, sigsuspend (suspend until signal send)
        88, # reboot
        91, # munmap
        113, 166, # vm86old, vm86: enter VM86 mode (virtual-8086 in Intel literature)
        119, 173, # sigreturn, rt_sigreturn
        162, # nanosleep
        252, # epoll_wait
    ))

    syscall = GenerateSyscall(project)

    syscall.fixed_arguments[SYS_EXIT] = {1: "0"}
    syscall.fixed_arguments[SYS_OLD_SELECT] = {5: "0"}

    syscall.syscalls = list(set(SYSCALL_NAMES.keys()) - IGNORE_SYSCALLS)

    process = SyscallProcess(project, name="syscall")
    WatchProcess(process)
    syslog = Syslog(project)
    syslog.syslog.patterns['syscall'] = 1.0
    syslog.messages.patterns['syscall'] = 1.0

SYS_EXIT = 1
SYS_OLD_SELECT = 82
SYS_EPOLL_WAIT = 252

class SyscallProcess(CreateProcess):
    def on_syscall_program(self, program):
        self.cmdline.arguments = [program]
        self.createProcess()

class Main(FuzzyFunctionC):
    def __init__(self, syscall):
        FuzzyFunctionC.__init__(self, "main", type="int", random_bytes=400)
        self.footer.append('return 0;')
        self.syscall = syscall

    def getarg(self, syscall, arg_index):
        try:
            return self.syscall.fixed_arguments[syscall][arg_index]
        except KeyError:
            pass
        state = randint(0,4)
        if state == 0:
            return USERLAND_ADDRESS
        elif state == 1:
            return UNMAPPED_ADDRESS
        elif state == 2:
            return KERNEL_ADDRESS
        elif state == 3:
            return "%sU" % self.createInt32()
        else:
            return "&%s" % self.createRandomBytes()[0]

class GenerateSyscall(ProjectAgent):
    def __init__(self, project):
        ProjectAgent.__init__(self, project, "syscall")

        # Syscall parameters
        self.syscalls = xrange(0, 255+1)
        self.fixed_arguments = {}

    def on_session_start(self):
        # Intialize some parameters
        self.buffer_count = 0

        # Create program using C compiler
        code = CodeC()
        code.includes = [
            "<unistd.h>",
            "<sys/syscall.h>",
            "<stdlib.h>",
        ]
        main = Main(self)
        code.addFunction(main)
        main.footer = ['exit(0);']

        syscallnr = choice(self.syscalls)
        if syscallnr in SYSCALL_NAMES:
           syscall = "/* %s */ %s" % (SYSCALL_NAMES[syscallnr], syscallnr)
        else:
           syscall = str(syscallnr)

        arguments = [syscall]
        for index in xrange(1, 8+1):
            value = main.getarg(syscallnr, index)
            arguments.append("/* argument %s */ %s" % (index, value))
        main.callFunction("syscall", arguments)

        directory = self.project().session.directory
        self.c_filename = directory.uniqueFilename("syscall.c")
        self.program_filename = directory.uniqueFilename("syscall")

        code.compile(self, self.c_filename, self.program_filename)
        self.send('syscall_program', self.program_filename)

SYSCALL_NAMES = {
      0: "sys_restart_syscall",
      1: "sys_exit",
      2: "sys_fork",
      3: "sys_read",
      4: "sys_write",
      5: "sys_open",
      6: "sys_close",
      7: "sys_waitpid",
      8: "sys_creat",
      9: "sys_link",
     10: "sys_unlink",
     11: "sys_execve",
     12: "sys_chdir",
     13: "sys_time",
     14: "sys_mknod",
     15: "sys_chmod",
     16: "sys_lchown",
     18: "sys_stat",
     19: "sys_lseek",
     20: "sys_getpid",
     21: "sys_mount",
     22: "sys_oldumount",
     23: "sys_setuid",
     24: "sys_getuid",
     25: "sys_stime",
     26: "sys_ptrace",
     27: "sys_alarm",
     28: "sys_fstat",
     29: "sys_pause",
     30: "sys_utime",
     33: "sys_access",
     34: "sys_nice",
     36: "sys_sync",
     37: "sys_kill",
     38: "sys_rename",
     39: "sys_mkdir",
     40: "sys_rmdir",
     41: "sys_dup",
     42: "sys_pipe",
     43: "sys_times",
     45: "sys_brk",
     46: "sys_setgid",
     47: "sys_getgid",
     48: "sys_signal",
     49: "sys_geteuid",
     50: "sys_getegid",
     51: "sys_acct",
     52: "sys_umount",
     54: "sys_ioctl",
     55: "sys_fcntl",
     57: "sys_setpgid",
     59: "sys_olduname",
     60: "sys_umask",
     61: "sys_chroot",
     62: "sys_ustat",
     63: "sys_dup2",
     64: "sys_getppid",
     65: "sys_getpgrp",
     66: "sys_setsid",
     67: "sys_sigaction",
     68: "sys_sgetmask",
     69: "sys_ssetmask",
     70: "sys_setreuid",
     71: "sys_setregid",
     72: "sys_sigsuspend",
     73: "sys_sigpending",
     74: "sys_sethostname",
     75: "sys_setrlimit",
     76: "sys_getrlimit",
     77: "sys_getrusage",
     78: "sys_gettimeofday",
     79: "sys_settimeofday",
     80: "sys_getgroups",
     81: "sys_setgroups",
     82: "old_select",
     83: "sys_symlink",
     84: "sys_lstat",
     85: "sys_readlink",
     86: "sys_uselib",
     87: "sys_swapon",
     88: "sys_reboot",
     89: "old_readdir",
     90: "old_mmap",
     91: "sys_munmap",
     92: "sys_truncate",
     93: "sys_ftruncate",
     94: "sys_fchmod",
     95: "sys_fchown",
     96: "sys_getpriority",
     97: "sys_setpriority",
     99: "sys_statfs",
    100: "sys_fstatfs",
    101: "sys_ioperm",
    102: "sys_socketcall",
    103: "sys_syslog",
    104: "sys_setitimer",
    105: "sys_getitimer",
    106: "sys_newstat",
    107: "sys_newlstat",
    108: "sys_newfstat",
    109: "sys_uname",
    110: "sys_iopl",
    111: "sys_vhangup",
    112: "sys_idle",
    113: "sys_vm86old",
    114: "sys_wait4",
    115: "sys_swapoff",
    116: "sys_sysinfo",
    117: "sys_ipc",
    118: "sys_fsync",
    119: "sys_sigreturn",
    120: "sys_clone",
    121: "sys_setdomainname",
    122: "sys_newuname",
    123: "sys_modify_ldt",
    124: "sys_adjtimex",
    125: "sys_mprotect",
    126: "sys_sigprocmask",
    127: "sys_create_module",
    128: "sys_init_module",
    129: "sys_delete_module",
    130: "sys_get_kernel_syms",
    131: "sys_quotactl",
    132: "sys_getpgid",
    133: "sys_fchdir",
    134: "sys_bdflush",
    135: "sys_sysfs",
    136: "sys_personality",
    138: "sys_setfsuid",
    139: "sys_setfsgid",
    140: "sys_llseek",
    141: "sys_getdents",
    142: "sys_select",
    143: "sys_flock",
    144: "sys_msync",
    145: "sys_readv",
    146: "sys_writev",
    147: "sys_getsid",
    148: "sys_fdatasync",
    149: "sys_sysctl",
    150: "sys_mlock",
    151: "sys_munlock",
    152: "sys_mlockall",
    153: "sys_munlockall",
    154: "sys_sched_setparam",
    155: "sys_sched_getparam",
    156: "sys_sched_setscheduler",
    157: "sys_sched_getscheduler",
    158: "sys_sched_yield",
    159: "sys_sched_get_priority_max",
    160: "sys_sched_get_priority_min",
    161: "sys_sched_rr_get_interval",
    162: "sys_nanosleep",
    163: "sys_mremap",
    164: "sys_setresuid",
    165: "sys_getresuid",
    166: "sys_vm86",
    167: "sys_query_module",
    168: "sys_poll",
    169: "sys_nfsservctl",
    170: "sys_setresgid",
    171: "sys_getresgid",
    172: "sys_prctl",
    173: "sys_rt_sigreturn",
    174: "sys_rt_sigaction",
    175: "sys_rt_sigprocmask",
    176: "sys_rt_sigpending",
    177: "sys_rt_sigtimedwait",
    178: "sys_rt_sigqueueinfo",
    179: "sys_rt_sigsuspend",
    180: "sys_pread",
    181: "sys_pwrite",
    182: "sys_chown",
    183: "sys_getcwd",
    184: "sys_capget",
    185: "sys_capset",
    186: "sys_sigaltstack",
    187: "sys_sendfile",
#   188: (reserved)
#   189: (reserved)
    190: "sys_vfork",
    191: "sys_ugetrlimit",
    192: "sys_mmap2",
    193: "sys_truncate64",
    194: "sys_ftruncate64",
    195: "sys_stat64",
    196: "sys_lstat64",
    197: "sys_fstat64",
    198: "sys_lchown32",
    199: "sys_getuid32",
    200: "sys_getgid32",
    201: "sys_geteuid32",
    202: "sys_getegid32",
    203: "sys_setreuid32",
    204: "sys_setregid32",
    205: "sys_getgroups32",
    206: "sys_setgroups32",
    207: "sys_fchown32",
    208: "sys_setresuid32",
    209: "sys_getresuid32",
    210: "sys_setresgid32",
    211: "sys_getresgid32",
    212: "sys_chown32",
    213: "sys_setuid32",
    214: "sys_setgid32",
    215: "sys_setfsuid32",
    216: "sys_setfsgid32",
    217: "sys_getdents64",
    218: "sys_pivot_root",
    219: "sys_mincore",
    220: "sys_madvise",
    221: "sys_fcntl64",
#   222: sys_tux (reserved)
#   223: (unused)
    224: "sys_gettid",
    225: "sys_readahead",
    226: "sys_setxattr",
    227: "sys_lsetxattr",
    228: "sys_fsetxattr",
    229: "sys_getxattr",
    230: "sys_lgetxattr",
    231: "sys_fgetxattr",
    232: "sys_listxattr",
    233: "sys_llistxattr",
    234: "sys_flistxattr",
    235: "sys_removexattr",
    236: "sys_lremovexattr",
    237: "sys_fremovexattr",
    238: "sys_tkill",
    239: "sys_sendfile64",
    240: "sys_futex",
    241: "sys_sched_setaffinity",
    242: "sys_sched_getaffinity",
    243: "sys_io_setup",
    244: "sys_io_destroy",
    245: "sys_io_getevents",
    246: "sys_io_submit",
    247: "sys_io_cancel",
    248: "sys_exit_group",
    249: "sys_lookup_dcookie",
    250: "sys_epoll_create",
    251: "sys_epoll_ctl",
    252: "sys_epoll_wait",
    253: "sys_remap_file_pages",
#   254: sys_set_thread_area (reserved)
#   255: sys_get_thread_area (reserved)
    256: "sys_set_tid_address",
    257: "sys_timer_create",
    258: "sys_timer_settime",
    259: "sys_timer_gettime",
    260: "sys_timer_getoverrun",
    261: "sys_timer_delete",
    262: "sys_clock_settime",
    263: "sys_clock_gettime",
    264: "sys_clock_getres",
    265: "sys_clock_nanosleep",
    266: "sys_statfs64",
    267: "sys_fstatfs64",
    268: "sys_tgkill",
    269: "sys_utimes",
    270: "sys_arm_fadvise64_64",
    271: "sys_pciconfig_iobase",
    272: "sys_pciconfig_read",
    273: "sys_pciconfig_write",
    274: "sys_mq_open",
    275: "sys_mq_unlink",
    276: "sys_mq_timedsend",
    277: "sys_mq_timedreceive",
    278: "sys_mq_notify",
    279: "sys_mq_getsetattr",
    280: "sys_waitid",
}

