from fusil.project_agent import ProjectAgent
from random import randint, choice
from array import array
from os import fstat
from os.path import basename, join as path_join
from stat import ST_SIZE
from fusil.session_agent import SessionAgent
from shutil import rmtree

def generateSpecialValues():
    values = (
        # Special values in big endian
        # SPECIAL_VALUES will contains value in big endian and little endian
        "\x00",
        "\x00\x00",
        "\x7f",
        "\x7f\xff",
        "\x7f\xff\xff\xff",
        "\x80",
        "\x80\x00",
        "\x80\x00\x00\x00",
        "\xfe",
        "\xfe\xff",
        "\xfe\xff\xff\xff",
        "\xff",
        "\xff\xff",
        "\xff\xff\xff\xff",
    )
    result = []
    for item in values:
        result.append(item)
        itemb = item[::-1]
        if item != itemb:
            result.append(itemb)
    return result

SPECIAL_VALUES = generateSpecialValues()

class MangleConfig:
    def __init__(self, min_op=1, max_op=100, operations=None):
        """
        Number of operations: min_op..max_op
        Operations: list of function names (eg. ["replace", "bit"])
        """
        self.min_op = min_op
        self.max_op = max_op
        self.max_insert_bytes = 4
        self.max_delete_bytes = 4
        self.first_offset = 0
        self.change_size = False
        if operations:
            self.operations = operations
        else:
            self.operations = None

class Mangle:
    def __init__(self, config, data):
        self.config = config
        self.data = data

    def generateByte(self):
        return randint(0, 255)

    def offset(self, last=1):
        first = self.config.first_offset
        last = len(self.data)-last
        if last < first:
            raise ValueError(
                "Invalid first_offset value (first=%s > last=%s)"
                % (first, last))
        return randint(first, last)

    def mangle_replace(self):
        self.data[self.offset()] = self.generateByte()

    def mangle_bit(self):
        offset = self.offset()
        bit = randint(0, 7)
        if randint(0, 1) == 1:
            value = self.data[offset] | (1 << bit)
        else:
            value = self.data[offset] & (~(1 << bit) & 0xFF)
        self.data[offset] = value

    def mangle_special_value(self):
        text = choice(SPECIAL_VALUES)
        offset = self.offset(len(text))
        self.data[offset:offset+len(text)] = array("B", text)

    def mangle_insert_bytes(self):
        offset = self.offset()
        count = randint(1, self.config.max_insert_bytes)
        for index in xrange(count):
            self.data.insert(offset, self.generateByte())

    def mangle_delete_bytes(self):
        offset = self.offset(2)
        count = randint(1, self.config.max_delete_bytes)
        count = min(count, len(self.data)-offset)
        del self.data[offset:offset+count]

    def run(self):
        """
        Mangle data and return number of applied operations
        """

        operation_names = self.config.operations
        if not operation_names:
            operation_names = ["replace", "bit", "special_value"]
            if self.config.change_size:
                operation_names.extend(("insert_bytes", "delete_bytes"))

        operations = []
        for name in operation_names:
            operation = getattr(self, "mangle_" + name)
            operations.append(operation)

        if self.config.max_op <= 0:
            return 0
        count = randint(self.config.min_op, self.config.max_op)
        for index in xrange(count):
            operation = choice(operations)
            operation()
        return count

class MangleFile(ProjectAgent):
    def __init__(self, project, source, nb_file=1):
        ProjectAgent.__init__(self, project, "mangle:%s" % basename(source))
        self.source_filename = source
        self.max_size = None # 10*1024*1024
        self.config = MangleConfig()
        self.nb_file = nb_file

    def readData(self, filename):
        # Open file and read file size
        self.warning("Load input file: %s" % filename)
        data = open(filename, 'rb')
        orig_filesize = fstat(data.fileno())[ST_SIZE]
        if not orig_filesize:
            raise ValueError("Input file (%s) is empty!" % filename)

        # Read bytes
        if self.max_size:
            data = data.read(self.max_size)
        else:
            data = data.read()

        # Display message if input is truncated
        if len(data) < orig_filesize:
            percent = len(data) * 100.0 / orig_filesize
            self.warning("Truncate file to %s bytes (%.2f%% of %s bytes)" \
                % (len(data), percent, orig_filesize))

        # Convert to Python array object
        return array('B', data)

    def mangleData(self, data):
        # Mangle bytes
        count = Mangle(self.config, data).run()
        self.warning("Mangle operation: %s" % count)
        return data

    def writeData(self, data):
        filename = self.createFilename()
        self.warning("Generate file: %s" % filename)
        output = open(filename, 'w')
        data.tofile(output)
        return filename

    def createFilename(self):
        if 1 < self.nb_file:
            count = 1
        else:
            count = None
        directory = self.project().session.directory
        return directory.uniqueFilename(self.source_filename, count=count)

    def mangle(self):
        filenames = []
        for index in xrange(self.nb_file):
            data = self.readData(self.source_filename)
            data = self.mangleData(data)
            filename = self.writeData(data)
            filenames.append(filename)
        self.send('mangle_filenames', filenames)

    def on_session_start(self):
        self.mangle()

