(*
   Copyright 2008-2014 Nikhil Swamy and Microsoft Research

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*)
#light "off"
// (c) Microsoft Corporation. All rights reserved

module FStar.TypeChecker.Util
open FStar.ST
open FStar.Exn
open FStar.All
open FStar
open FStar.Util
open FStar.Errors
open FStar.TypeChecker
open FStar.Syntax
open FStar.TypeChecker.Common
open FStar.TypeChecker.Env
open FStar.TypeChecker.Rel
open FStar.Syntax.Syntax
open FStar.Ident
open FStar.Syntax.Subst
open FStar.TypeChecker.Common
open FStar.Syntax
open FStar.Dyn

type lcomp_with_binder = option<bv> * lcomp

module SS = FStar.Syntax.Subst
module S = FStar.Syntax.Syntax
module BU = FStar.Util
module U = FStar.Syntax.Util
module N = FStar.TypeChecker.Normalize
module TcComm = FStar.TypeChecker.Common
module P = FStar.Syntax.Print
module C = FStar.Parser.Const

//Reporting errors
let report env errs =
    Errors.log_issue (Env.get_range env)
               (Err.failed_to_prove_specification errs)

(************************************************************************)
(* Unification variables *)
(************************************************************************)
let new_implicit_var reason r env k =
    new_implicit_var_aux reason r env k Strict None

let close_guard_implicits env solve_deferred (xs:binders) (g:guard_t) : guard_t =
  if Options.eager_subtyping () || solve_deferred then
    let solve_now, defer =
      g.deferred |> List.partition (fun (_, p) -> Rel.flex_prob_closing env xs p)
    in
    if Env.debug env <| Options.Other "Rel"
    then begin
      BU.print_string "SOLVE BEFORE CLOSING:\n";
      List.iter (fun (s, p) -> BU.print2 "%s: %s\n" s (Rel.prob_to_string env p)) solve_now;
      BU.print_string " ...DEFERRED THE REST:\n";
      List.iter (fun (s, p) -> BU.print2 "%s: %s\n" s (Rel.prob_to_string env p)) defer;
      BU.print_string "END\n"
    end;
    let g = Rel.solve_deferred_constraints env ({g with deferred=solve_now}) in
    let g = {g with deferred=defer} in
    g
  else g

let check_uvars r t =
  let uvs = Free.uvars t in
  if not (BU.set_is_empty uvs)
  then
    let us = List.map (fun u -> Print.uvar_to_string u.ctx_uvar_head) (BU.set_elements uvs) |> String.concat ", " in
    (* ignoring the hide_uvar_nums and print_implicits flags here *)
    Options.push();
    Options.set_option "hide_uvar_nums" (Options.Bool false);
    Options.set_option "print_implicits" (Options.Bool true);
    Errors.log_issue r
      (Errors.Error_UncontrainedUnificationVar, (BU.format2 "Unconstrained unification variables %s in type signature %s; \
       please add an annotation" us (Print.term_to_string t)));
    Options.pop()

(************************************************************************)
(* Extracting annotations from a term *)
(************************************************************************)
let extract_let_rec_annotation env {lbname=lbname; lbunivs=univ_vars; lbtyp=t; lbdef=e} :
    list<univ_name>
   * typ
   * bool //true indicates that the type needs to be checked; false indicates that it is already checked
   =
  let rng = S.range_of_lbname lbname in
  let t = SS.compress t in
  match t.n with
   | Tm_unknown ->
     //if univ_vars <> [] then failwith "Impossible: non-empty universe variables but the type is unknown"; //AR: not necessarily for universe annotated let recs
     let univ_vars, e = SS.open_univ_vars univ_vars e in
     let env = Env.push_univ_vars env univ_vars in
     let r = Env.get_range env in
     let rec aux e : either<typ,comp> =
        let e = SS.compress e in
        match e.n with
        | Tm_meta(e, _) ->
          aux e
        | Tm_ascribed(e, t, _) ->
          fst t
        | Tm_abs(bs, body, _) ->
          let res = aux body in
          let c =
              match res with
              | Inl t ->
                if Options.ml_ish()
                then U.ml_comp t r
                else S.mk_Total t //let rec without annotations default to Tot, except if --MLish
              | Inr c -> c in
          let t = S.mk (Tm_arrow(bs, c)) None c.pos in
          if debug env Options.High
          then BU.print2 "(%s) Using type %s\n"
                    (Range.string_of_range r) (Print.term_to_string t);
          Inl t
        | _ ->
          Inl S.tun
    in
    let t =
       match aux e with
       | Inr c ->
         if U.is_tot_or_gtot_comp c
         then U.comp_result c
         else raise_error (Errors.Fatal_UnexpectedComputationTypeForLetRec,
                           BU.format1 "Expected a 'let rec' to be annotated with a value type; got a computation type %s"
                                       (Print.comp_to_string c))
                           rng
       | Inl t -> t
    in
    univ_vars, t, true

  | _ ->
    let univ_vars, t = open_univ_vars univ_vars t in
    univ_vars, t, false

(************************************************************************)
(* Utilities on patterns  *)
(************************************************************************)

//let decorate_pattern env p exp =
//    let qq = p in
//    let rec aux p e : pat  =
//        let pkg q = withinfo q p.p in
//        let e = U.unmeta e in
//        match p.v, e.n with
//            | _, Tm_uinst(e, _) -> aux p e

//            | Pat_constant _, _ ->
//              pkg p.v

//            | Pat_var x, Tm_name y ->
//              if not (bv_eq x y)
//              then failwith (BU.format2 "Expected pattern variable %s; got %s" (Print.bv_to_string x) (Print.bv_to_string y));
//              if Env.debug env <| Options.Other "Pat"
//              then BU.print2 "Pattern variable %s introduced at type %s\n" (Print.bv_to_string x) (Normalize.term_to_string env y.sort);
//              let s = Normalize.normalize [Env.Beta] env y.sort in
//              let x = {x with sort=s} in
//              pkg (Pat_var x)

//            | Pat_wild x, Tm_name y ->
//              if bv_eq x y |> not
//              then failwith (BU.format2 "Expected pattern variable %s; got %s" (Print.bv_to_string x) (Print.bv_to_string y));
//              let s = Normalize.normalize [Env.Beta] env y.sort in
//              let x = {x with sort=s} in
//              pkg (Pat_wild x)

//            | Pat_dot_term(x, _), _ ->
//              pkg (Pat_dot_term(x, e))

//            | Pat_cons(fv, []), Tm_fvar fv' ->
//              if not (Syntax.fv_eq fv fv')
//              then failwith (BU.format2 "Expected pattern constructor %s; got %s" fv.fv_name.v.str fv'.fv_name.v.str);
//              pkg (Pat_cons(fv', []))

//            | Pat_cons(fv, argpats), Tm_app({n=Tm_fvar(fv')}, args)
//            | Pat_cons(fv, argpats), Tm_app({n=Tm_uinst({n=Tm_fvar(fv')}, _)}, args) ->

//              if fv_eq fv fv' |> not
//              then failwith (BU.format2 "Expected pattern constructor %s; got %s" fv.fv_name.v.str fv'.fv_name.v.str);

//              let fv = fv' in
//              let rec match_args matched_pats args argpats = match args, argpats with
//                | [], [] -> pkg (Pat_cons(fv, List.rev matched_pats))
//                | arg::args, (argpat, _)::argpats ->
//                  begin match arg, argpat.v with
//                        | (e, Some (Implicit true)), Pat_dot_term _ ->
//                          let x = Syntax.new_bv (Some p.p) S.tun in
//                          let q = withinfo (Pat_dot_term(x, e)) p.p in
//                          match_args ((q, true)::matched_pats) args argpats

//                        | (e, imp), _ ->
//                          let pat = aux argpat e, S.is_implicit imp in
//                          match_args (pat::matched_pats) args argpats
//                 end

//                | _ -> failwith (BU.format2 "Unexpected number of pattern arguments: \n\t%s\n\t%s\n" (Print.pat_to_string p) (Print.term_to_string e)) in

//              match_args [] args argpats

//           | _ ->
//            failwith (BU.format3
//                            "(%s) Impossible: pattern to decorate is %s; expression is %s\n"
//                            (Range.string_of_range qq.p)
//                            (Print.pat_to_string qq)
//                            (Print.term_to_string exp))
//    in
//    aux p exp

 let rec decorated_pattern_as_term (pat:pat) : list<bv> * term =
    let mk f : term = mk f None pat.p in

    let pat_as_arg (p, i) =
        let vars, te = decorated_pattern_as_term p in
        vars, (te, as_implicit i) in
    match pat.v with
    | Pat_constant c ->
        [], mk (Tm_constant c)

    | Pat_wild x
    | Pat_var x  ->
        [x], mk (Tm_name x)

    | Pat_cons(fv, pats) ->
        let vars, args = pats |> List.map pat_as_arg |> List.unzip in
        let vars = List.flatten vars in
        vars,  mk (Tm_app(Syntax.fv_to_tm fv, args))

    | Pat_dot_term(x, e) ->
        [], e


(*********************************************************************************************)
(* Utils related to monadic computations *)
(*********************************************************************************************)
let comp_univ_opt c =
    match c.n with
    | Total (_, uopt)
    | GTotal (_, uopt) -> uopt
    | Comp c ->
      match c.comp_univs with
      | [] -> None
      | hd::_ -> Some hd

let lcomp_univ_opt lc = lc |> TcComm.lcomp_comp |> (fun (c, g) -> comp_univ_opt c, g)

let destruct_wp_comp c : (universe * typ * typ) = U.destruct_comp c

let mk_comp_l mname u_result result wp flags =
  mk_Comp ({ comp_univs=[u_result];
             effect_name=mname;
             result_typ=result;
             effect_args=[S.as_arg wp];
             flags=flags})

let mk_comp md = mk_comp_l md.mname

let effect_args_from_repr (repr:term) (is_layered:bool) (r:Range.range) : list<term> =
  let err () =
    raise_error (Errors.Fatal_UnexpectedEffect,
      BU.format2 "Could not get effect args from repr %s with is_layered %s"
        (Print.term_to_string repr) (string_of_bool is_layered)) r in
  let repr = SS.compress repr in
  if is_layered
  then match repr.n with
       | Tm_app (_, _::is) -> is |> List.map fst
       | _ -> err ()
  else match repr.n with 
       | Tm_arrow (_, c) -> c |> U.comp_to_comp_typ |> (fun ct -> ct.effect_args |> List.map fst)
       | _ -> err ()


(*
 * Build the M.return comp for a wp effect
 *
 * Caller must ensure that ed is a wp-based effect
 *)
let mk_wp_return env (ed:S.eff_decl) (u_a:universe) (a:typ) (e:term) (r:Range.range)
: comp
= let ret_wp = ed |> U.get_return_vc_combinator in
  let wp = mk_Tm_app
    (inst_effect_fun_with [u_a] env ed ret_wp)
    [S.as_arg a; S.as_arg e]
    None
    r in
  mk_comp ed u_a a wp [RETURN]


(*
 * Build the M.return comp for an indexed effect
 *
 * Caller must ensure that ed is an indexed effect
 *)
let mk_indexed_return env (ed:S.eff_decl) (u_a:universe) (a:typ) (e:term) (r:Range.range)
: comp * guard_t
= if Env.debug env <| Options.Other "LayeredEffects"
  then BU.print4 "Computing %s.return for u_a:%s, a:%s, and e:%s{\n"
         (Ident.string_of_lid ed.mname) (Print.univ_to_string u_a)
         (Print.term_to_string a) (Print.term_to_string e);

  let _, return_t = Env.inst_tscheme_with
    (ed |> U.get_return_vc_combinator)
    [u_a] in

  let return_t_shape_error (s:string) =
    (Errors.Fatal_UnexpectedEffect, BU.format3
      "%s.return %s does not have proper shape (reason:%s)"
      (Ident.string_of_lid ed.mname) (Print.term_to_string return_t) s) in

  let a_b, x_b, rest_bs, return_ct =
    match (SS.compress return_t).n with
    | Tm_arrow (bs, c) when List.length bs >= 2 ->
      let ((a_b::x_b::bs, c)) = SS.open_comp bs c in
      a_b, x_b, bs, U.comp_to_comp_typ c
    | _ -> raise_error (return_t_shape_error "Either not an arrow or not enough binders") r in

  let rest_bs_uvars, g_uvars = Env.uvars_for_binders
    env rest_bs [NT (a_b |> fst, a); NT (x_b |> fst, e)]
    (fun b -> BU.format3 "implicit var for binder %s of %s at %s"
             (Print.binder_to_string b)
             (BU.format1 "%s.return" (Ident.string_of_lid ed.mname))
             (Range.string_of_range r)) r in

  let subst = List.map2
    (fun b t -> NT (b |> fst, t))
    (a_b::x_b::rest_bs) (a::e::rest_bs_uvars) in

  let is =
    effect_args_from_repr (SS.compress return_ct.result_typ) (U.is_layered ed) r
    |> List.map (SS.subst subst) in

  let c = mk_Comp ({
    comp_univs = [u_a];
    effect_name = ed.mname;
    result_typ = a;
    effect_args = is |> List.map S.as_arg;
    flags = []
  }) in

  if Env.debug env <| Options.Other "LayeredEffects"
  then BU.print1 "} c after return %s\n" (Print.comp_to_string c);

  c, g_uvars


(*
 * Wrapper over mk_wp_return and mk_indexed_return
 *)
let mk_return env (ed:S.eff_decl) (u_a:universe) (a:typ) (e:term) (r:Range.range)
: comp * guard_t
= if ed |> U.is_layered
  then mk_indexed_return env ed u_a a e r
  else mk_wp_return env ed u_a a e r, Env.trivial_guard


let lift_comp env (c:comp_typ) lift : comp * guard_t =
  ({ c with flags = [] }) |> S.mk_Comp |> lift.mlift_wp env

let join_effects env l1_in l2_in =
  let l1, l2 = Env.norm_eff_name env l1_in, Env.norm_eff_name env l2_in in
  match Env.join_opt env l1 l2 with
  | Some (m, _, _) -> m
  | None ->
    match Env.exists_polymonadic_bind env l1 l2 with
    | Some (m, _) -> m
    | None ->
      raise_error (Errors.Fatal_EffectsCannotBeComposed,
        (BU.format2 "Effects %s and %s cannot be composed" 
          (Print.lid_to_string l1_in) (Print.lid_to_string l2_in))) env.range

let join_lcomp env c1 c2 =
  if TcComm.is_total_lcomp c1
  && TcComm.is_total_lcomp c2
  then C.effect_Tot_lid
  else join_effects env c1.eff_name c2.eff_name


let lift_comps env c1 c2 (b:option<bv>) (for_bind:bool) : lident * comp * comp * guard_t =
  let c1 = Env.unfold_effect_abbrev env c1 in
  let c2 = Env.unfold_effect_abbrev env c2 in
  match Env.join_opt env c1.effect_name c2.effect_name with
  | Some (m, lift1, lift2) ->
    let c1, g1 = lift_comp env c1 lift1 in
    let c2, g2 =
      if not for_bind then lift_comp env c2 lift2
      else
        let x_a =
          match b with
          | None -> S.null_binder (U.comp_result c1)
          | Some x -> S.mk_binder x in
        let env_x = Env.push_binders env [x_a] in
        let c2, g2 = lift_comp env_x c2 lift2 in
        c2, Env.close_guard env [x_a] g2 in
    m, c1, c2, Env.conj_guard g1 g2
  | None ->

    (*
     * AR: we could not lift the comps using the wp lattice
     *     try using the polymonadic binds
     *     if there exists a polymonadic bind (M1, M2) |> M2,
     *       then lift using, lift_M1_M2 e = bind_M1_M2 e (fun x -> M2.return x)
     *
     *     (and similarly if exists (M2, M1) |> M1
     *
     *     we could also try to find a P such that (M1, P) |> P and (M2, P) |> P exist
     *       but leaving that for later
     *)

    let rng = env.range in
    let err () =
      raise_error (Errors.Fatal_EffectsCannotBeComposed,
        (BU.format2 "Effects %s and %s cannot be composed" 
          (Print.lid_to_string c1.effect_name) (Print.lid_to_string c2.effect_name))) rng in

    if Env.debug env <| Options.Other "LayeredEffects"
    then BU.print3 "Lifting comps %s and %s with for_bind %s{\n"
           (c1 |> S.mk_Comp |> Print.comp_to_string)
           (c2 |> S.mk_Comp |> Print.comp_to_string)
           (string_of_bool for_bind);

    if for_bind then err ()
    else

      let bind_with_return (ct:comp_typ) (ret_eff:lident) (f_bind:Env.polymonadic_bind_t)
      : comp * guard_t =
        let x_bv = S.gen_bv "x" None ct.result_typ in
        let c_ret, g_ret = mk_return
          (Env.push_bv env x_bv)
          (Env.get_effect_decl env ret_eff)
          (List.hd ct.comp_univs)
          ct.result_typ (S.bv_to_name x_bv) rng in
        let c, g_bind = f_bind env ct (Some x_bv) (U.comp_to_comp_typ c_ret) [] rng in
        c, Env.conj_guard g_ret g_bind in

      let try_lift c1 c2 : option<(lident * comp * comp * guard_t)> =
        let p_bind_opt = Env.exists_polymonadic_bind env c1.effect_name c2.effect_name in
        if p_bind_opt |> is_some
        then let p, f_bind = p_bind_opt |> must in
             if lid_equals p c2.effect_name
             then (let c1, g = bind_with_return c1 p f_bind in
                   Some (c2.effect_name, c1, S.mk_Comp c2, g))
             else None
        else None in
  
      let p, c1, c2, g =
        match try_lift c1 c2 with
        | Some (p, c1, c2, g) -> p, c1, c2, g
        | None ->
          match try_lift c2 c1 with
          | Some (p, c2, c1, g) -> p, c1, c2, g
          | None -> err () in

      if Env.debug env <| Options.Other "LayeredEffects"
      then BU.print3 "} Returning p %s, c1 %s, and c2 %s\n"
             (Ident.string_of_lid p) (Print.comp_to_string c1) (Print.comp_to_string c2);

      p, c1, c2, g

let is_pure_effect env l =
  let l = norm_eff_name env l in
  lid_equals l C.effect_PURE_lid

let is_pure_or_ghost_effect env l =
  let l = norm_eff_name env l in
  lid_equals l C.effect_PURE_lid
  || lid_equals l C.effect_GHOST_lid

let lax_mk_tot_or_comp_l mname u_result result flags =
    if Ident.lid_equals mname C.effect_Tot_lid
    then S.mk_Total' result (Some u_result)
    else mk_comp_l mname u_result result S.tun flags

let is_function t = match (compress t).n with
    | Tm_arrow _ -> true
    | _ -> false

let label reason r f : term =
    mk (Tm_meta(f, Meta_labeled(reason, r, false))) None f.pos

let label_opt env reason r f = match reason with
    | None -> f
    | Some reason ->
        if not <| Env.should_verify env
        then f
        else label (reason()) r f

let label_guard r reason (g:guard_t) = match g.guard_f with
    | Trivial -> g
    | NonTrivial f -> {g with guard_f=NonTrivial (label reason r f)}

let close_wp_comp env bvs (c:comp) =
    if U.is_ml_comp c then c
    else if env.lax
    && Options.ml_ish() //NS: disabling this optimization temporarily
    then c
    else begin
            let close_wp u_res md res_t bvs wp0 =
              let close = md |> U.get_wp_close_combinator |> must in
              List.fold_right (fun x wp ->
                  let bs = [mk_binder x] in
                  let us = u_res::[env.universe_of env x.sort] in
                  let wp = U.abs bs wp (Some (U.mk_residual_comp C.effect_Tot_lid None [TOTAL])) in
                  mk_Tm_app (inst_effect_fun_with us env md close) [S.as_arg res_t; S.as_arg x.sort; S.as_arg wp] None wp0.pos)
              bvs wp0 in
            let c = Env.unfold_effect_abbrev env c in
            let u_res_t, res_t, wp = destruct_wp_comp c in
            let md = Env.get_effect_decl env c.effect_name in
            let wp = close_wp u_res_t md res_t bvs wp in
            mk_comp md u_res_t c.result_typ wp c.flags
        end

let close_wp_lcomp env bvs (lc:lcomp) =
  let bs = bvs |> List.map S.mk_binder in
  lc |>
  TcComm.apply_lcomp
    (close_wp_comp env bvs)
    (fun g -> g |> Env.close_guard env bs |> close_guard_implicits env false bs)

(*
 * Closing of layered computations happens via substitution
 *)
let close_layered_lcomp env bvs tms (lc:lcomp) =
  let bs = bvs |> List.map S.mk_binder in
  let substs = List.map2 (fun bv tm ->
    NT (bv, tm)
  ) bvs tms in
  lc |>
  TcComm.apply_lcomp
    (SS.subst_comp substs)
    (fun g -> g |> Env.close_guard env bs |> close_guard_implicits env false bs)

let should_not_inline_lc (lc:lcomp) =
    lc.cflags |> BU.for_some (function SHOULD_NOT_INLINE -> true | _ -> false)

(* should_return env (Some e) lc:
 * We will "return" e, adding an equality to the VC, if all of the following conditions hold
 * (a) e is a pure or ghost term
 * (b) Its return type, lc.res_typ, is not a sub-singleton (unit, squash, etc), if lc.res_typ is an arrow, then we check the comp type of the arrow
 *     An exception is made for reifiable effects -- they are useful even if they return unit
 * (c) Its head symbol is not marked irreducible (in this case inlining is not going to help, it is equivalent to having a bound variable)
 * (d) It's not a let rec, as determined by the absence of the SHOULD_NOT_INLINE flag---see issue #1362. Would be better to just encode inner let recs to the SMT solver properly
 *)
let should_return env eopt lc =
    //if lc.res_typ is not an arrow, arrow_formals_comp returns Tot lc.res_typ
    let lc_is_unit_or_effectful =
      lc.res_typ |> U.arrow_formals_comp |> snd |> (fun c ->
        not (Env.is_reifiable_comp env c) &&
        (U.comp_result c |> U.is_unit || not (U.is_pure_or_ghost_comp c)))
    in
    match eopt with
    | None -> false //no term to return
    | Some e ->
      TcComm.is_pure_or_ghost_lcomp lc                &&  //condition (a), (see above)
      not lc_is_unit_or_effectful                &&  //condition (b)
      (let head, _ = U.head_and_args' e in
       match (U.un_uinst head).n with
       | Tm_fvar fv ->  not (Env.is_irreducible env (lid_of_fv fv)) //condition (c)
       | _ -> true)                              &&
     not (should_not_inline_lc lc)                   //condition (d)

let return_value env u_t_opt t v =
  let c =
    if not <| Env.lid_exists env C.effect_GTot_lid //we're still in prims, not yet having fully defined the primitive effects
    then mk_Total t
    else if U.is_unit t
    then S.mk_Total' t (Some U_zero)
    else let m = Env.get_effect_decl env C.effect_PURE_lid in //if Tot isn't fully defined in prims yet, then just return (Total t)
         let u_t =
             match u_t_opt with
             | None -> env.universe_of env t
             | Some u_t -> u_t
         in
         let wp =
            if env.lax
            && Options.ml_ish() //NS: Disabling this optimization temporarily
            then S.tun
            else let ret_wp = m |> U.get_return_vc_combinator in
                 N.normalize [Env.Beta; Env.NoFullNorm]
                            env
                            (mk_Tm_app (inst_effect_fun_with [u_t] env m ret_wp)
                                       [S.as_arg t; S.as_arg v]
                                       None
                                       v.pos) in
         mk_comp m u_t t wp [RETURN]
  in
  if debug env <| Options.Other "Return"
  then BU.print3 "(%s) returning %s at comp type %s\n"
                    (Range.string_of_range v.pos)
                    (P.term_to_string v)
                    (N.comp_to_string env c);
  c

(* private *)

(*
 * Bind for indexed effects
 *
 * This covers both the binds of an effect M,
 *   and polymonadic binds (M, N) |> P (this former is just (M, M) |> M)
 * 
 * Let c1 = M c1_a (t1 ... tm)
 *     c2 = N c2_a (s1 ... sn) - where b is free in (s1 ... sn)
 *
 *     bind_t = ((u_a, u_b), a:Type -> b:Type -> <some binders> ->
 *                           f:M.repr a i_1 ... i_n ->
 *                           g:(x:a -> N.repr b j_1 ... j_n) ->
 *                           P.repr b k_1 ... k_p)
 *
 * First we instantiate bind_t with [u_c1_a, u_c2_a]
 *
 * Then we substitute [a/c1_a; b/c2_a] in <some binders>
 *
 * Next we create ?u1 ... ?un for each of the binders in <some binders>
 *   while substituting [bi/?ui] in subsequent binders (so that their sorts are well-typed)
 *
 * Let substs = [a/c1_a; b/c2_a; bi/?ui]
 *
 * let i_i = i_i[substs]  //i_i are the indices of f in the bind_wp
 * let j_i = j_i[x/b; substs]  //j_i are the indices of g in the bind_wp and x/x is replacing x with the binder b
 * let k_i = k_i[substs]  //k_i are the indices of the return type in bind
 *
 * We now unify i_i with t_i (where t_i are the indices of c1)
 *        unify j_i with s_i (where s_i are the indices of c2,
 *                            these are done in an env with b, and the returned guard is closed over b)
 * and return k_i as the output indices
 *)
let mk_indexed_bind env
  (m:lident) (n:lident) (p:lident) (bind_t:tscheme)
  (ct1:comp_typ) (b:option<bv>) (ct2:comp_typ)
  (flags:list<cflag>) (r1:Range.range) : comp * guard_t =

  if Env.debug env <| Options.Other "LayeredEffects" then
    BU.print2 "Binding c1:%s and c2:%s {\n"
      (Print.comp_to_string (S.mk_Comp ct1)) (Print.comp_to_string (S.mk_Comp ct2));

  let m_ed, n_ed, p_ed = Env.get_effect_decl env m, Env.get_effect_decl env n, Env.get_effect_decl env p in

  let u1, t1, is1 = List.hd ct1.comp_univs, ct1.result_typ, List.map fst ct1.effect_args in
  let u2, t2, is2 = List.hd ct2.comp_univs, ct2.result_typ, List.map fst ct2.effect_args in

  let _, bind_t = Env.inst_tscheme_with bind_t [u1; u2] in

  let bind_t_shape_error (s:string) =
    (Errors.Fatal_UnexpectedEffect, BU.format2
       "bind %s does not have proper shape (reason:%s)"
       (Print.term_to_string bind_t) s) in

  let a_b, b_b, rest_bs, f_b, g_b, bind_ct =
    match (SS.compress bind_t).n with
    | Tm_arrow (bs, c) when List.length bs >= 4 ->
      let ((a_b::b_b::bs), c) = SS.open_comp bs c in
      let rest_bs, f_b, g_b =
        List.splitAt (List.length bs - 2) bs |> (fun (l1, l2) -> l1, List.hd l2, List.hd (List.tl l2)) in
      a_b, b_b, rest_bs, f_b, g_b, U.comp_to_comp_typ c
    | _ -> raise_error (bind_t_shape_error "Either not an arrow or not enough binders") r1 in

  //create uvars for rest_bs, with proper substitutions of a_b, b_b, and b_i with t1, t2, and ?ui
  let rest_bs_uvars, g_uvars = Env.uvars_for_binders
    env rest_bs [NT (a_b |> fst, t1); NT (b_b |> fst, t2)]
    (fun b -> BU.format3
      "implicit var for binder %s of %s at %s"
      (Print.binder_to_string b)
      (BU.format3 "(%s, %s) |> %s" (Ident.string_of_lid m) (Ident.string_of_lid n) (Ident.string_of_lid p))
      (Range.string_of_range r1)) r1 in

  let subst = List.map2
    (fun b t -> NT (b |> fst, t))
    (a_b::b_b::rest_bs) (t1::t2::rest_bs_uvars) in

  let f_guard =  //unify c1's indices with f's indices in the bind_wp
    let f_sort_is = effect_args_from_repr
      (SS.compress (f_b |> fst).sort)
      (U.is_layered m_ed) r1 |> List.map (SS.subst subst) in
    List.fold_left2
      (fun g i1 f_i1 -> Env.conj_guard g (Rel.teq env i1 f_i1))
      Env.trivial_guard is1 f_sort_is in 

  let g_guard =  //unify c2's indices with g's indices in the bind_wp
    let x_a =
      match b with
      | None -> S.null_binder ct1.result_typ
      | Some x -> S.mk_binder x in

    let g_sort_is : list<term> =
      match (SS.compress (g_b |> fst).sort).n with
      | Tm_arrow (bs, c) ->
        let bs, c = SS.open_comp bs c in
        let bs_subst = NT (List.hd bs |> fst, x_a |> fst |> S.bv_to_name) in
        let c = SS.subst_comp [bs_subst] c in
        effect_args_from_repr (SS.compress (U.comp_result c)) (U.is_layered n_ed) r1
        |> List.map (SS.subst subst) in

    let env_g = Env.push_binders env [x_a] in
    List.fold_left2
      (fun g i1 g_i1 -> Env.conj_guard g (Rel.teq env_g i1 g_i1))
      Env.trivial_guard is2 g_sort_is
    |> Env.close_guard env [x_a] in

  let is : list<term> =  //indices of the resultant computation
    effect_args_from_repr (SS.compress bind_ct.result_typ) (U.is_layered p_ed) r1
    |> List.map (SS.subst subst) in

  let c = mk_Comp ({
    comp_univs = ct2.comp_univs;
    effect_name = p_ed.mname;
    result_typ = t2;
    effect_args = List.map S.as_arg is;
    flags = flags
  }) in

  if Env.debug env <| Options.Other "LayeredEffects" then
    BU.print1 "} c after bind: %s\n" (Print.comp_to_string c);

  c, Env.conj_guards [ g_uvars; f_guard; g_guard ]

let mk_wp_bind env (m:lident) (ct1:comp_typ) (b:option<bv>) (ct2:comp_typ) (flags:list<cflag>) (r1:Range.range)
  : comp =

  let (md, a, kwp), (u_t1, t1, wp1), (u_t2, t2, wp2) =
    let md = Env.get_effect_decl env m in
    let a, kwp = Env.wp_signature env m in
    (md, a, kwp), destruct_wp_comp ct1, destruct_wp_comp ct2 in
    
  let bs =
    match b with
    | None -> [null_binder t1]
    | Some x -> [S.mk_binder x]
  in
  let mk_lam wp =
    //we know it's total; indicate for the normalizer reduce it by adding  the TOTAL flag
    U.abs bs wp (Some (U.mk_residual_comp C.effect_Tot_lid None [TOTAL]))
  in
  let r1 = S.mk (S.Tm_constant (FStar.Const.Const_range r1)) None r1 in
  let wp_args = [
    S.as_arg r1;
    S.as_arg t1;
    S.as_arg t2;
    S.as_arg wp1;
    S.as_arg (mk_lam wp2)]
  in
  let bind_wp = md |> U.get_bind_vc_combinator in
  let wp = mk_Tm_app  (inst_effect_fun_with [u_t1;u_t2] env md bind_wp) wp_args None t2.pos in
  mk_comp md u_t2 t2 wp flags

let mk_bind env (c1:comp) (b:option<bv>) (c2:comp) (flags:list<cflag>) (r1:Range.range) : comp * guard_t =
  let ct1, ct2 = Env.unfold_effect_abbrev env c1, Env.unfold_effect_abbrev env c2 in

  match Env.exists_polymonadic_bind env ct1.effect_name ct2.effect_name with
  | Some (p, f_bind) -> f_bind env ct1 b ct2 flags r1
  | None ->    
    let m, c1, c2, g_lift = lift_comps env c1 c2 b true in
    let ct1, ct2 = U.comp_to_comp_typ c1, U.comp_to_comp_typ c2 in

    let c, g_bind =
      if Env.is_layered_effect env m
      then
        let bind_t = m |> Env.get_effect_decl env |> U.get_bind_vc_combinator in
        mk_indexed_bind env m m m bind_t ct1 b ct2 flags r1
      else mk_wp_bind env m ct1 b ct2 flags r1, Env.trivial_guard in
    c, Env.conj_guard g_lift g_bind

let bind_pure_wp_with env (wp1:typ) (c:comp) (flags:list<cflag>) : comp * guard_t =
  let r = Env.get_range env in

  let pure_c = S.mk_Comp ({
    comp_univs = [S.U_zero];
    effect_name = C.effect_PURE_lid;
    result_typ = S.t_unit;
    effect_args = [wp1 |> S.as_arg];
    flags = []
  }) in

  mk_bind env pure_c None c flags r

let weaken_flags flags =
    if flags |> BU.for_some (function SHOULD_NOT_INLINE -> true | _ -> false)
    then [SHOULD_NOT_INLINE]
    else flags |> List.collect (function
         | TOTAL -> [TRIVIAL_POSTCONDITION]
         | RETURN -> [PARTIAL_RETURN; TRIVIAL_POSTCONDITION]
         | f -> [f])

let weaken_comp env (c:comp) (formula:term) : comp * guard_t =
  if U.is_ml_comp c
  then c, Env.trivial_guard
  else let ct = Env.unfold_effect_abbrev env c in

       (*
        * The following code does:
        *   M.bind_wp (lift_pure_M (Prims.pure_assume_wp f)) (fun _ -> wp)
        *)

       (*
        * lookup the pure_assume_wp from prims
        * its type is p:Type -> pure_wp unit
        *  and it is not universe polymorphic
        *)
       let pure_assume_wp = S.fv_to_tm (S.lid_as_fv C.pure_assume_wp_lid (Delta_constant_at_level 1) None) in

       (* apply it to f, after decorating f with the reason *)
       let pure_assume_wp = mk_Tm_app
         pure_assume_wp
         [ S.as_arg <| formula ]
         None
         (Env.get_range env)
       in
         
       bind_pure_wp_with env pure_assume_wp c (weaken_flags ct.flags)

let weaken_precondition env lc (f:guard_formula) : lcomp =
  let weaken () =
      let c, g_c = TcComm.lcomp_comp lc in
      if env.lax
      && Options.ml_ish() //NS: Disabling this optimization temporarily
      then c, g_c
      else match f with
           | Trivial -> c, g_c
           | NonTrivial f ->
             let c, g_w = weaken_comp env c f in
             c, Env.conj_guard g_c g_w
  in
  TcComm.mk_lcomp lc.eff_name lc.res_typ (weaken_flags lc.cflags) weaken


let strengthen_comp env (reason:option<(unit -> string)>) (c:comp) (f:formula) flags : comp * guard_t =
    if env.lax
    then c, Env.trivial_guard
    else let r = Env.get_range env in
         (*
          * The following code does:
          *   M.bind_wp (lift_pure_M (Prims.pure_assert_wp f)) (fun _ -> wp)
          *)

         (*
          * lookup the pure_assert_wp from prims
          * its type is p:Type -> pure_wp unit
          *  and it is not universe polymorphic
          *)
         let pure_assert_wp = S.fv_to_tm (S.lid_as_fv C.pure_assert_wp_lid (Delta_constant_at_level 1) None) in

         (* apply it to f, after decorating f with the reason *)
         let pure_assert_wp = mk_Tm_app
           pure_assert_wp
           [ S.as_arg <| label_opt env reason r f ]
           None
           r
         in

         bind_pure_wp_with env pure_assert_wp c flags

let record_simplify = 
  let x = BU.mk_ref 0 in
  fun env guard ->
    let n = !x in
    x := n + 1;
    let start = BU.now() in
    let g = Rel.simplify_guard env guard in
    let fin = BU.now () in
    if Options.debug_any()
    then
      BU.print2 "Simplify_guard %s in %s ms\n"
        (BU.string_of_int n)
        (BU.string_of_int (snd (BU.time_diff start fin)));
    g

  
let strengthen_precondition
            (reason:option<(unit -> string)>)
            env
            (e_for_debugging_only:term)
            (lc:lcomp)
            (g0:guard_t)
    : lcomp * guard_t =
    if Env.is_trivial_guard_formula g0
    then lc, g0
    else let flags =
            let maybe_trivial_post, flags =
              if TcComm.is_tot_or_gtot_lcomp lc then true, [TRIVIAL_POSTCONDITION] else false, []
            in
            flags @ (
            lc.cflags
            |> List.collect (function
                 | RETURN
                 | PARTIAL_RETURN -> [PARTIAL_RETURN]
                 | SOMETRIVIAL
                 | TRIVIAL_POSTCONDITION
                    when not maybe_trivial_post ->
                   [TRIVIAL_POSTCONDITION]
                 | SHOULD_NOT_INLINE -> [SHOULD_NOT_INLINE]
                 | _ -> []))
         in
         let strengthen () =
            let c, g_c = TcComm.lcomp_comp lc in
            if env.lax
            then c, g_c
            else let g0 = Rel.simplify_guard env g0 in
                 match guard_form g0 with
                 | Trivial -> c, g_c
                 | NonTrivial f ->
                   if Env.debug env <| Options.Extreme
                   then BU.print2 "-------------Strengthening pre-condition of term %s with guard %s\n"
                     (N.term_to_string env e_for_debugging_only)
                     (N.term_to_string env f);
                   let c, g_s = strengthen_comp env reason c f flags in
                   c, Env.conj_guard g_c g_s
         in
       TcComm.mk_lcomp (norm_eff_name env lc.eff_name)
                       lc.res_typ
                       flags
                       strengthen,
       {g0 with guard_f=Trivial}


let lcomp_has_trivial_postcondition lc =
    TcComm.is_tot_or_gtot_lcomp lc
    || BU.for_some (function SOMETRIVIAL | TRIVIAL_POSTCONDITION -> true | _ -> false)
                   lc.cflags

let maybe_add_with_type env uopt lc e =
    if TcComm.is_lcomp_partial_return lc
    || env.lax
    then e
    else if lcomp_has_trivial_postcondition lc
         && Option.isSome (Env.try_lookup_lid env C.with_type_lid) //and we're not very early in prims
    then let u = match uopt with
                 | Some u -> u
                 | None -> env.universe_of env lc.res_typ
         in
         U.mk_with_type u lc.res_typ e
    else e


(*
 * This is used in bind, when c1 is a Tot (x:unit{phi})
 * In such cases, e1 is inlined in c2, but we still want to capture inhabitance of phi
 *
 * For wp-effects, we do forall (x:unit{phi}). c2
 * For layered effects, we do: weaken_comp (phi[x/()]) c2
 *
 * We should make wp-effects also same as the layered effects
 *)
let maybe_capture_unit_refinement (env:env) (t:term) (x:bv) (c:comp) : comp * guard_t =
  let t = N.normalize_refinement N.whnf_steps env t in
  match t.n with
  | Tm_refine (b, phi) ->
    let is_unit =
      match b.sort.n with
      | Tm_fvar fv -> S.fv_eq_lid fv C.unit_lid
      | _ -> false in
    if is_unit then      
      if c |> U.comp_effect_name |> Env.norm_eff_name env |> Env.is_layered_effect env
      then
        let [b], phi = SS.open_term [b, None] phi in
        let phi = SS.subst [NT (b |> fst, S.unit_const)] phi in
        weaken_comp env c phi
      else close_wp_comp env [x] c, Env.trivial_guard
    else c, Env.trivial_guard
  | _ -> c, Env.trivial_guard

let bind r1 env e1opt (lc1:lcomp) ((b, lc2):lcomp_with_binder) : lcomp =
  let debug f =
      if debug env Options.Extreme
      || debug env <| Options.Other "bind"
      then f ()
  in
  let lc1 = N.ghost_to_pure_lcomp env lc1 in //downgrade from ghost to pure, if possible
  let lc2 = N.ghost_to_pure_lcomp env lc2 in
  let joined_eff = join_lcomp env lc1 lc2 in
  let bind_flags =
      if should_not_inline_lc lc1
      || should_not_inline_lc lc2
      then [SHOULD_NOT_INLINE]
      else let flags =
              if TcComm.is_total_lcomp lc1
              then if TcComm.is_total_lcomp lc2
                   then [TOTAL]
                   else if TcComm.is_tot_or_gtot_lcomp lc2
                   then [SOMETRIVIAL]
                   else []
              else if TcComm.is_tot_or_gtot_lcomp lc1
                   && TcComm.is_tot_or_gtot_lcomp lc2
              then [SOMETRIVIAL]
              else []
          in
          if lcomp_has_trivial_postcondition lc2
          then TRIVIAL_POSTCONDITION::flags
          else flags
  in
  let bind_it () =
      if env.lax
      && Options.ml_ish() //NS: disabling this optimization temporarily
      then
         let u_t = env.universe_of env lc2.res_typ in
         lax_mk_tot_or_comp_l joined_eff u_t lc2.res_typ [], Env.trivial_guard  //AR: TODO: FIXME: fix for layered effects
      else begin
          let c1, g_c1 = TcComm.lcomp_comp lc1 in
          let c2, g_c2 = TcComm.lcomp_comp lc2 in
          debug (fun () ->
            BU.print3 "(1) bind: \n\tc1=%s\n\tx=%s\n\tc2=%s\n(1. end bind)\n"
            (Print.comp_to_string c1)
            (match b with
                | None -> "none"
                | Some x -> Print.bv_to_string x)
            (Print.comp_to_string c2));
          let aux () =
            if U.is_trivial_wp c1
            then match b with
                 | None ->
                   Inl (c2, "trivial no binder")
                 | Some _ ->
                   if U.is_ml_comp c2 //|| not (U.is_free [Inr x] (U.freevars_comp c2))
                   then Inl (c2, "trivial ml")
                   else Inr "c1 trivial; but c2 is not ML"
            else if U.is_ml_comp c1 && U.is_ml_comp c2
            then Inl (c2, "both ml")
            else Inr "c1 not trivial, and both are not ML"
          in
          let try_simplify () : either<(comp * guard_t * string), string> =
            let aux_with_trivial_guard () =
              match aux () with
              | Inl (c, reason) -> Inl (c, Env.trivial_guard, reason)
              | Inr reason -> Inr reason in
            if Option.isNone (Env.try_lookup_effect_lid env C.effect_GTot_lid) //if we're very early in prims
            then if U.is_tot_or_gtot_comp c1
                 && U.is_tot_or_gtot_comp c2
                 then Inl (c2, Env.trivial_guard, "Early in prims; we don't have bind yet")
                 else raise_error (Errors.Fatal_NonTrivialPreConditionInPrims,
                                   "Non-trivial pre-conditions very early in prims, even before we have defined the PURE monad")
                                   (Env.get_range env)
            else if U.is_total_comp c1
            then (*
                  * Helper routine to close the compuation c with c1's return type
                  * When c1's return type is of the form _:t{phi}, is is useful to know
                  *   that t{phi} is inhabited, even if c1 is inlined etc.
                  *)
                 let close (x:bv) (reason:string) (c:comp) =
                   let x = { x with sort = U.comp_result c1 } in
                   let c, g_c = maybe_capture_unit_refinement env x.sort x c in
                   Inl (c, g_c, reason) in
                 match e1opt, b with
                 | Some e, Some x ->
                   c2 |> SS.subst_comp [NT (x, e)] |> close x "c1 Tot"
                 | _, Some x -> c2 |> close x "c1 Tot only close"
                 | _, _ -> aux_with_trivial_guard ()
            else if U.is_tot_or_gtot_comp c1
                 && U.is_tot_or_gtot_comp c2
            then Inl (S.mk_GTotal (U.comp_result c2), Env.trivial_guard, "both GTot")
            else aux_with_trivial_guard ()
          in
          match try_simplify () with
          | Inl (c, g_c, reason) ->
            debug (fun () ->
                BU.print2 "(2) bind: Simplified (because %s) to\n\t%s\n"
                            reason
                            (Print.comp_to_string c));
            c, Env.conj_guard g_c (Env.conj_guard g_c1 g_c2)
          | Inr reason ->
            debug (fun () ->
                BU.print1 "(2) bind: Not simplified because %s\n" reason);
            
            let mk_bind c1 b c2 =  (* AR: end code for inlining pure and ghost terms *)
              let c, g_bind = mk_bind env c1 b c2 bind_flags r1 in
              c, Env.conj_guard (Env.conj_guard g_c1 g_c2) g_bind in

            let mk_seq c1 b c2 =
                //c1 is PURE or GHOST
                let c1 = Env.unfold_effect_abbrev env c1 in
                let c2 = Env.unfold_effect_abbrev env c2 in
                let m, _, lift2 = Env.join env c1.effect_name c2.effect_name in
                let c2, g2 = lift_comp env c2 lift2 in
                let u1, t1, wp1 = destruct_wp_comp c1 in
                let md_pure_or_ghost = Env.get_effect_decl env c1.effect_name in
                let trivial = md_pure_or_ghost |> U.get_wp_trivial_combinator |> must in
                let vc1 = mk_Tm_app (inst_effect_fun_with [u1] env md_pure_or_ghost trivial)
                                    [S.as_arg t1; S.as_arg wp1]
                                    None
                                    r1
                in
                let c, g_s = strengthen_comp env None c2 vc1 bind_flags in
                c, Env.conj_guards [g_c1; g_c2; g2; g_s]
            in
            (* AR: we have let the previously applied bind optimizations take effect, below is the code to do more inlining for pure and ghost terms *)
            let u_res_t1, res_t1 =
              let t = U.comp_result c1 in
              match comp_univ_opt c1 with
              | None -> env.universe_of env t, t
              | Some u -> u, t in
            //c1 and c2 are bound to the input comps
            if Option.isSome b
            && should_return env e1opt lc1
            then let e1 = Option.get e1opt in
                 let x = Option.get b in
                 //we will inline e1 in the WP of c2
                 //Aiming to build a VC of the form
                 //
                 //     M.bind (lift_(Pure/Ghost)_M wp1)
                 //            (x == e1 ==> lift_M2_M (wp2[e1/x]))
                 //
                 //
                 //The additional equality hypothesis may seem
                 //redundant, but c1's post-condition or type may carry
                 //some meaningful information Then, it's important to
                 //weaken wp2 to with the equality, So that whatever
                 //property is proven about the result of wp1 (i.e., x)
                 //is still available in the proof of wp2 However, we
                 //apply two optimizations:

                 //   a. if c1 is already a return or a partial return,
                 //      then it already provides this equality, so no
                 //      need to add it again and instead generate
                 //
                 //         M.bind (lift_(Pure/Ghost)_M wp1)
                 //                (lift_M2_M (wp2[e1/x]))

                 //   b. if c1 is marked with TRIVIAL_POSTCONDITION,
                 //      then the post-condition does not carry any
                 //      useful information. We have two sub-cases:

                 //      (i) In case the user option
                 //          `vcgen.optimize_bind_as_seq = without_type`
                 //          rather than generating
                 //          M.bind wp1 (\x. wp2), we generate:
                 //
                 //           M.assert_wp (wp1 (\x. True))
                 //                       (lift_M2_M  (wp2[e1/x]))
                 //
                 //      Note, although the post-condition of c1 does
                 //      not carry useful information, its result type
                 //      might. When applying the optimization above,
                 //      the SMT solver is faced with reconstructing
                 //      the type of e1. Usually, it can do this, but
                 //      in some cases (e.g., if the result type has a
                 //      complex refinement), then this optimization
                 //      can actually cause a VC to fail. So, we add an
                 //      option to recover from this, at the cost of
                 //      some VC bloat:
                 //
                 //      (ii). In case the user option
                 //            `vcgen.optimize_bind_as_seq = with_type`,
                 //            we build
                 //
                 //             M.assert_wp (wp1 (\x. True))
                 //                        (lift_M2_M (wp2[with_type e1 t1/x]))
                 //
                 //      Where `with_type e1 t1`, decorates `e1` with
                 //      its type before substituting. This allows the
                 //      SMT solver to recover the type of `e1` (using
                 //      a primitive axiom about with_type), without
                 //      polluting the VC with an additional equality.
                 //      Note, specific occurrences of `with_type e t`
                 //      can be normalized away to `e` if requested
                 //      explicitly by a user tactic.
                 //
                 //   c. If neither of the optimizations above apply,
                 //   then we generate the WP mentioned at the top,
                 //   i.e.
                 //
                 //      M.bind (lift_(Pure/Ghost)_M wp1)
                 //             (x == e1 ==> lift_M2_M (wp2[e1/x]))

                 if U.is_partial_return c1
                 then // case (a)
                      let _ = debug (fun () ->
                        BU.print2 "(3) bind (case a): Substituting %s for %s" (N.term_to_string env e1) (Print.bv_to_string x)) in
                      let c2 = SS.subst_comp [NT(x,e1)] c2 in
                      mk_bind c1 b c2
                 else if Options.vcgen_optimize_bind_as_seq()
                      && lcomp_has_trivial_postcondition lc1
                      && Option.isSome (Env.try_lookup_lid env C.with_type_lid) //and we're not very early in prims
                 then // case (b)
                      let e1' =
                        if Options.vcgen_decorate_with_type()
                        then U.mk_with_type u_res_t1 res_t1 e1 // case (b) (ii)
                        else e1                                // case (b) (i)
                      in
                      let _ = debug (fun () ->
                        BU.print2 "(3) bind (case b): Substituting %s for %s" (N.term_to_string env e1') (Print.bv_to_string x)) in
                      let c2 = SS.subst_comp [NT(x, e1')] c2 in
                      mk_seq c1 b c2
                 else // case (c)
                      let _ = debug (fun () ->
                        BU.print2 "(3) bind (case c): Adding equality %s = %s" (N.term_to_string env e1) (Print.bv_to_string x)) in
                      let c2 = SS.subst_comp [NT(x,e1)] c2 in
                      let x_eq_e = U.mk_eq2 u_res_t1 res_t1 e1 (bv_to_name x) in
                      let c2, g_w = weaken_comp (Env.push_binders env [S.mk_binder x]) c2 x_eq_e in
                      let c, g_bind = mk_bind c1 b c2 in
                      c, Env.conj_guard g_w g_bind
                //Caution: here we keep the flags for c2 as is, these flags will be overwritten later when we do md.bind below
                //If we decide to return c2 as is (after inlining), we should reset these flags else bad things will happen
            else mk_bind c1 b c2
      end
  in TcComm.mk_lcomp joined_eff
                     lc2.res_typ
      (* TODO : these cflags might be inconsistent with the one returned by bind_it  !!! *)
                     bind_flags
                     bind_it

let weaken_guard g1 g2 = match g1, g2 with
    | NonTrivial f1, NonTrivial f2 ->
      let g = (U.mk_imp f1 f2) in
      NonTrivial g
    | _ -> g2

let maybe_assume_result_eq_pure_term env (e:term) (lc:lcomp) : lcomp =
  let should_return =
       not (env.lax)
    && Env.lid_exists env C.effect_GTot_lid //we're not too early in prims
    && should_return env (Some e) lc
    && not (TcComm.is_lcomp_partial_return lc)
  in
  let flags =
    if should_return
    then if TcComm.is_total_lcomp lc
         then RETURN::lc.cflags
         else PARTIAL_RETURN::lc.cflags
    else lc.cflags
  in
  let refine () =
      let c, g_c = TcComm.lcomp_comp lc in
      let u_t =
          match comp_univ_opt c with
          | Some u_t -> u_t
          | None -> env.universe_of env (U.comp_result c)
      in
      if U.is_tot_or_gtot_comp c
      then //insert a return
           let retc = return_value env (Some u_t) (U.comp_result c) e in
           if not (U.is_pure_comp c) //it started in GTot, so it should end up in Ghost
           then let retc = U.comp_to_comp_typ retc in
                let retc = {retc with effect_name=C.effect_GHOST_lid; flags=flags} in
                S.mk_Comp retc, g_c
           else U.comp_set_flags retc flags, g_c
       else //augment c's post-condition with a return
            let c = Env.unfold_effect_abbrev env c in
            let t = c.result_typ in
            let c = mk_Comp c in
            let x = S.new_bv (Some t.pos) t in
            let xexp = S.bv_to_name x in
            let ret =
                TcComm.lcomp_of_comp
                <| U.comp_set_flags (return_value env (Some u_t) t xexp) [PARTIAL_RETURN] in
            let eq = U.mk_eq2 u_t t xexp e in
            let eq_ret = weaken_precondition env ret (NonTrivial eq) in
            let bind_c, g_bind = TcComm.lcomp_comp (bind e.pos env None (TcComm.lcomp_of_comp c) (Some x, eq_ret)) in
            U.comp_set_flags bind_c flags, Env.conj_guard g_c g_bind
  in
  if not should_return then lc
  else TcComm.mk_lcomp lc.eff_name lc.res_typ flags refine

let maybe_return_e2_and_bind
        (r:Range.range)
        (env:env)
        (e1opt:option<term>)
        (lc1:lcomp)
        (e2:term)
        (x, lc2)
   : lcomp =
   let lc2 =
        let eff1 = Env.norm_eff_name env lc1.eff_name in
        let eff2 = Env.norm_eff_name env lc2.eff_name in
        if (not (is_pure_or_ghost_effect env eff1)
            || should_not_inline_lc lc1)
        && is_pure_or_ghost_effect env eff2
        then maybe_assume_result_eq_pure_term env e2 lc2
        else lc2 in //the resulting computation is still pure/ghost and inlineable; no need to insert a return
   bind r env e1opt lc1 (x, lc2)

let fvar_const env lid =  S.fvar (Ident.set_lid_range lid (Env.get_range env)) delta_constant None


(*
 * Conjunction combinator for layered effects
 *
 * let ct1 = M a (t1...tn)
 *     ct2 = M a (s1...sn)
 *
 *     M.conjunction = fun (a_b:Type) ..<some binders>.. (f:repr a i1...in) (g:repr a j1...jn) (p_b:Type0) -> repr a k1...n
 *
 * First we instantiate M.conjunction with [u_a]
 *
 * Then we create uvars ?u1..?un for each of the binders in <some binder>
 *   while substituting [a_b/a] and [bi/?ui] in subsequent binders (handled by Env.uvars_for_binders)
 *
 * let substs = [a_b/a; bi/?ui; p_b/p]
 *
 * let i_i = i_i[substs]
 * let j_i = i_i[substs]
 * let k_i = i_i[substs]
 *
 * Unify i_i with t_i (where t_i are the indices of ct1)
 * Unify j_i with s_i (where t_i are the indices of ct2)
 * 
 * And return k_i
 *)
let mk_layered_conjunction env (ed:S.eff_decl) (u_a:universe) (a:term) (p:typ) (ct1:comp_typ) (ct2:comp_typ) (r:Range.range)
: comp * guard_t =

  let _, conjunction =
    Env.inst_tscheme_with (ed |> U.get_layered_if_then_else_combinator |> must) [u_a] in
  let is1, is2 = List.map fst ct1.effect_args, List.map fst ct2.effect_args in

  let conjunction_t_error (s:string) =
    (Errors.Fatal_UnexpectedEffect, BU.format2
      "conjunction %s does not have proper shape (reason:%s)"
      (Print.term_to_string conjunction) s) in

  let a_b, rest_bs, f_b, g_b, p_b, body =
    match (SS.compress conjunction).n with
    | Tm_abs (bs, body, _) when List.length bs >= 4 ->
      let (a_b::bs), body = SS.open_term bs body in
      let rest_bs, (f_b::g_b::p_b::[]) = List.splitAt (List.length bs - 3) bs in
      a_b, rest_bs, f_b, g_b, p_b, body |> U.unascribe
    | _ -> raise_error (conjunction_t_error "Either not an abstraction or not enough binders") r in

  let rest_bs_uvars, g_uvars = Env.uvars_for_binders
    env rest_bs [NT (a_b |> fst, a)]
    (fun b -> BU.format3
      "implicit var for binder %s of %s:conjunction at %s"
      (Print.binder_to_string b) (Ident.string_of_lid ed.mname)
      (r |> Range.string_of_range)) r in

  let substs = List.map2
    (fun b t -> NT (b |> fst, t))
    (a_b::(rest_bs@[p_b])) (a::(rest_bs_uvars@[p])) in

  let f_guard =
    let f_sort_is =
      match (SS.compress (f_b |> fst).sort).n with
      | Tm_app (_, _::is) ->
        is |> List.map fst |> List.map (SS.subst substs)
      | _ -> raise_error (conjunction_t_error "f's type is not a repr type") r in
    List.fold_left2
      (fun g i1 f_i -> Env.conj_guard g (Rel.teq env i1 f_i))
      Env.trivial_guard is1 f_sort_is in

  let g_guard =
    let g_sort_is =
      match (SS.compress (g_b |> fst).sort).n with
      | Tm_app (_, _::is) ->
        is |> List.map fst |> List.map (SS.subst substs)
      | _ -> raise_error (conjunction_t_error "g's type is not a repr type") r in
    List.fold_left2
      (fun g i2 g_i -> Env.conj_guard g (Rel.teq env i2 g_i))
      Env.trivial_guard is2 g_sort_is in

  let body = SS.subst substs body in

  let is =
    match (SS.compress body).n with
    | Tm_app (_, a::args) -> List.map fst args
    | _ -> raise_error (conjunction_t_error "body is not a repr type") r in

  mk_Comp ({
    comp_univs = [u_a];
    effect_name = ed.mname;
    result_typ = a;
    effect_args = is |> List.map S.as_arg;
    flags = []
  }), Env.conj_guard (Env.conj_guard g_uvars f_guard) g_guard


(*
 * For non-layered effects, just apply the if_then_else combinator
 *)
let mk_non_layered_conjunction env (ed:S.eff_decl) (u_a:universe) (a:term) (p:typ) (ct1:comp_typ) (ct2:comp_typ) (_:Range.range)
: comp * guard_t =
  let if_then_else = ed |> U.get_wp_if_then_else_combinator |> must in
  let _, _, wp_t = destruct_wp_comp ct1 in
  let _, _, wp_e = destruct_wp_comp ct2 in
  let wp = mk_Tm_app (inst_effect_fun_with [u_a] env ed if_then_else)
    [S.as_arg a; S.as_arg p; S.as_arg wp_t; S.as_arg wp_e]
    None (Range.union_ranges wp_t.pos wp_e.pos) in
  mk_comp ed u_a a wp [], Env.trivial_guard

let bind_cases env (res_t:typ) (lcases:list<(formula * lident * list<cflag> * (bool -> lcomp))>) : lcomp =
    let eff = List.fold_left (fun eff (_, eff_label, _, _) -> join_effects env eff eff_label)
                             C.effect_PURE_lid
                             lcases
    in
    let should_not_inline_whole_match, bind_cases_flags =
        if lcases |> BU.for_some (fun (_, _, flags, _) ->
           flags |> BU.for_some (function SHOULD_NOT_INLINE -> true | _ -> false))
        then true, [SHOULD_NOT_INLINE]
        else false, []
    in
    let bind_cases () =
        let u_res_t = env.universe_of env res_t in
        if env.lax
        && Options.ml_ish() //NS: Disabling this optimization temporarily
        then
             lax_mk_tot_or_comp_l eff u_res_t res_t [], Env.trivial_guard
        else begin
            let default_case =
                let post_k = U.arrow [null_binder res_t] (S.mk_Total U.ktype0) in
                let kwp    = U.arrow [null_binder post_k] (S.mk_Total U.ktype0) in
                let post   = S.new_bv None post_k in
                let wp     = U.abs [mk_binder post]
                                   (label Err.exhaustiveness_check (Env.get_range env) <| fvar_const env C.false_lid)
                                   (Some (U.mk_residual_comp C.effect_Tot_lid None [TOTAL])) in
                let md     = Env.get_effect_decl env C.effect_PURE_lid in
                mk_comp md u_res_t res_t wp [] in
            let maybe_return eff_label_then cthen =
               if should_not_inline_whole_match
               || not (is_pure_or_ghost_effect env eff)
               then cthen true //inline each the branch, if eligible
               else cthen false //the entire match is pure and inlineable, so no need to inline each branch
            in
            let md, comp, g_comp = List.fold_right (fun (g, eff_label, _, cthen) (_, celse, g_comp) ->
                let cthen, gthen = TcComm.lcomp_comp (maybe_return eff_label cthen) in
                let md, ct_then, ct_else, g_lift =
                  let m, cthen, celse, g_lift = lift_comps env cthen celse None false in
                  let md = Env.get_effect_decl env m in
                  md, cthen |> U.comp_to_comp_typ, celse |> U.comp_to_comp_typ, g_lift in
                let fn =
                  if md |> U.is_layered then mk_layered_conjunction
                  else mk_non_layered_conjunction in
                let c, g_conjunction = fn env md u_res_t res_t g ct_then ct_else (Env.get_range env) in
                Some md,
                c,
                Env.conj_guard (Env.conj_guard (Env.conj_guard g_comp gthen) g_lift) g_conjunction
            ) lcases (None, default_case, Env.trivial_guard) in
            match lcases with
            | []
            | [_] -> comp, g_comp
            | _ ->
              if md |> must |> U.is_layered then comp, g_comp
              else
                let comp = Env.comp_to_comp_typ env comp in
                let md = Env.get_effect_decl env comp.effect_name in
                let _, _, wp = destruct_wp_comp comp in
                let ite_wp = md |> U.get_wp_ite_combinator |> must in
                let wp = mk_Tm_app (inst_effect_fun_with [u_res_t] env md ite_wp)
                                   [S.as_arg res_t; S.as_arg wp]
                                   None
                                   wp.pos in
                mk_comp md u_res_t res_t wp bind_cases_flags, g_comp
        end
    in
    TcComm.mk_lcomp eff res_t bind_cases_flags bind_cases

let check_comp env (e:term) (c:comp) (c':comp) : term * comp * guard_t =
  if false then
    BU.print3 "Checking sub_comp:\n%s has type %s\n\t<:\n%s\n"
            (Print.term_to_string e)
            (Print.comp_to_string c)
            (Print.comp_to_string c');
  match Rel.sub_comp env c c' with
    | None ->
        if env.use_eq
        then raise_error (Err.computed_computation_type_does_not_match_annotation_eq env e c c') (Env.get_range env)
        else raise_error (Err.computed_computation_type_does_not_match_annotation env e c c') (Env.get_range env)
    | Some g -> e, c', g

let universe_of_comp env u_res c =
  (*
   * Universe computation for M t wp:
   *   if M is pure or ghost, then return universe of t
   *   else if M is not marked Total, then return u0
   *        else if M has no additional binders, then return universe of t
   *        else delegate the computation to repr of M, error out of no repr
   *)
  let c_lid = c |> U.comp_effect_name |> Env.norm_eff_name env in
  if U.is_pure_or_ghost_effect c_lid then u_res  //if pure or ghost, return the universe of the return type
  else
    let is_total = Env.lookup_effect_quals env c_lid |> List.existsb (fun q -> q = S.TotalEffect) in
    if not is_total then S.U_zero  //if it is a non-total effect then u0
    else match Env.effect_repr env c u_res with
         | None ->
           raise_error (Errors.Fatal_EffectCannotBeReified,
                        (BU.format1 "Effect %s is marked total but does not have a repr" (Print.lid_to_string c_lid)))
                        c.pos
         | Some tm -> env.universe_of env tm

let check_trivial_precondition env c =
  let ct = c |> Env.unfold_effect_abbrev env in
  let md = Env.get_effect_decl env ct.effect_name in
  let u_t, t, wp = destruct_wp_comp ct in
  let vc = mk_Tm_app
    (inst_effect_fun_with [u_t] env md (md |> U.get_wp_trivial_combinator |> must))
    [S.as_arg t; S.as_arg wp]
    None
    (Env.get_range env)
  in

  ct, vc, Env.guard_of_guard_formula <| NonTrivial vc

let coerce_with (env:Env.env)
                (e : term) (lc : lcomp) // original term and its computation type
                (ty : typ) // new result typ
                (f : lident) // coercion
                (us : universes) (eargs : args) // extra arguments to coertion
                (mkcomp : term -> comp)
                : term * lcomp =
    match Env.try_lookup_lid env f with
    | Some _ ->
        if Env.debug env (Options.Other "Coercions") then
            BU.print1 "Coercing with %s!\n" (Ident.string_of_lid f);
        let coercion = S.fvar (Ident.set_lid_range f e.pos) (Delta_constant_at_level 1) None in
        let coercion = S.mk_Tm_uinst coercion us in
        let coercion = U.mk_app coercion eargs in
        let lc = bind e.pos env (Some e) lc (None, TcComm.lcomp_of_comp <| mkcomp ty) in
        let e = mk_Tm_app coercion [S.as_arg e] None e.pos in
        e, lc
    | None ->
        Errors.log_issue e.pos (Errors.Warning_CoercionNotFound,
                                (BU.format1 "Coercion %s was not found in the environment, not coercing."
                                            (string_of_lid f)));
        e, lc

type isErased =
    | Yes of term
    | Maybe
    | No

let rec check_erased (env:Env.env) (t:term) : isErased =
  let norm' = N.normalize [Env.Beta; Env.Eager_unfolding;
                           Env.UnfoldUntil delta_constant;
                           Env.Exclude Env.Zeta; Env.Primops;
                           Env.Weak; Env.HNF; Env.Iota]
  in
  let t = norm' env t in
  let t = U.unrefine t in
  let h, args = U.head_and_args t in
  let h = U.un_uinst h in
  let r =
    match (SS.compress h).n, args with
    | Tm_fvar fv, [(a, None)] when S.fv_eq_lid fv C.erased_lid ->
      Yes a

    (* In these two cases, we cannot guarantee that `t` is not
     * an erased, so we're conservatively returning `false` *)
    | Tm_uvar _, _
    | Tm_unknown, _ -> Maybe

    (*
     * AR: For Tm_match:
     *     We are only interested in returning a No or Maybe
     *     Since even if all the branched are erased types,
     *       we need to find their join to return to the caller
     *     That's messy
     *     We can't always return Maybe, since that breaks simple
     *       cases like the int types in FStar.Integers
     *     So we iterate over all the branches and return a No if possible
     *)
    | Tm_match (_, branches), _ ->
      branches |> List.fold_left (fun acc br ->
        match acc with
        | Yes _ | Maybe -> Maybe
        | No ->
          let _, _, br_body = Subst.open_branch br in
          match
            br_body
            |> check_erased
                (br_body
                 |> Free.names
                 |> BU.set_elements
                 |> Env.push_bvs env) with
          | No -> No
          | _ -> Maybe) No


    (* Anything else cannot be `erased` *)
    | _ ->
      No
  in
  (* if Options.debug_any () then *)
  (*   BU.print2 "check_erased (%s) = %s\n" *)
  (*     (Print.term_to_string t) *)
  (*     (match r with *)
  (*      | Yes a -> "Yes " ^ Print.term_to_string a *)
  (*      | Maybe -> "Maybe" *)
  (*      | No -> "No"); *)
  r

let maybe_coerce_lc env (e:term) (lc:lcomp) (exp_t:term) : term * lcomp * guard_t =
    let should_coerce =
         not (Options.use_two_phase_tc ()) // always coerce without 2 phase TC
      || env.phase1 // otherwise only on phase1
      || env.lax
      || Options.lax ()
    in
    if not should_coerce
    then (e, lc, Env.trivial_guard)
    else
    let is_t_term t =
        let t = N.unfold_whnf env t in
        match (SS.compress t).n with
        | Tm_fvar fv -> S.fv_eq_lid fv C.term_lid
        | _ -> false
    in
    let is_t_term_view t =
        let t = N.unfold_whnf env t in
        match (SS.compress t).n with
        | Tm_fvar fv -> S.fv_eq_lid fv C.term_view_lid
        | _ -> false
    in
    let is_type t =
        let t = N.unfold_whnf env t in
        match (SS.compress t).n with
        | Tm_type _ -> true
        | _ -> false
    in
    let res_typ = U.unrefine lc.res_typ in
    let head, args = U.head_and_args res_typ in
    if Env.debug env (Options.Other "Coercions") then
            BU.print4 "(%s) Trying to coerce %s from type (%s) to type (%s)\n"
                    (Range.string_of_range e.pos)
                    (Print.term_to_string e)
                    (Print.term_to_string res_typ)
                    (Print.term_to_string exp_t);

    let mk_erased u t =
      U.mk_app
        (S.mk_Tm_uinst (fvar_const env C.erased_lid) [u])
        [S.as_arg t]
    in
    match (U.un_uinst head).n, args with
    | Tm_fvar fv, [] when S.fv_eq_lid fv C.bool_lid && is_type exp_t ->
        let e, lc = coerce_with env e lc U.ktype0 C.b2t_lid [] [] S.mk_Total in
        e, lc, Env.trivial_guard


    | Tm_fvar fv, [] when S.fv_eq_lid fv C.term_lid && is_t_term_view exp_t ->
        let e, lc = coerce_with env e lc S.t_term_view C.inspect [] [] S.mk_Tac in
        e, lc, Env.trivial_guard

    | Tm_fvar fv, [] when S.fv_eq_lid fv C.term_view_lid && is_t_term exp_t ->
        let e, lc = coerce_with env e lc S.t_term C.pack [] [] S.mk_Tac in
        e, lc, Env.trivial_guard

    | Tm_fvar fv, [] when S.fv_eq_lid fv C.binder_lid && is_t_term exp_t ->
        let e, lc = coerce_with env e lc S.t_term C.binder_to_term [] [] S.mk_Tac in
        e, lc, Env.trivial_guard

    | _ ->
    match check_erased env res_typ, check_erased env exp_t with
    | No, Yes ty ->
        begin
        let u = env.universe_of env ty in
        match Rel.get_subtyping_predicate env res_typ ty with
        | None ->
          e, lc, Env.trivial_guard
        | Some g ->
          let g = Env.apply_guard g e in
          let e, lc = coerce_with env e lc exp_t C.hide [u] [S.iarg ty] S.mk_Total in
          e, lc, g
        end

    | Yes ty, No ->
        let u = env.universe_of env ty in
        let e, lc = coerce_with env e lc ty C.reveal [u] [S.iarg ty] S.mk_GTotal in
        e, lc, Env.trivial_guard

    | _ ->
      e, lc, Env.trivial_guard

(* Coerces regardless of expected type if a view exists, useful for matches *)
(* Returns `None` if no coercion was applied. *)
let coerce_views (env:Env.env) (e:term) (lc:lcomp) : option<(term * lcomp)> =
    let rt = lc.res_typ in
    let rt = U.unrefine rt in
    let hd, args = U.head_and_args rt in
    match (SS.compress hd).n, args with
    | Tm_fvar fv, [] when S.fv_eq_lid fv C.term_lid ->
        Some <| coerce_with env e lc S.t_term_view C.inspect [] [] S.mk_Tac
    | _ ->
        None

let weaken_result_typ env (e:term) (lc:lcomp) (t:typ) : term * lcomp * guard_t =
  if Env.debug env Options.High then
    BU.print3 "weaken_result_typ e=(%s) lc=(%s) t=(%s)\n"
            (Print.term_to_string e)
            (TcComm.lcomp_to_string lc)
            (Print.term_to_string t);
  let use_eq =
    env.use_eq ||
    (match Env.effect_decl_opt env lc.eff_name with
     // See issue #881 for why weakening result type of a reifiable computation is problematic
     | Some (ed, qualifiers) -> qualifiers |> List.contains Reifiable
     | _ -> false) in
  let gopt = if use_eq
             then Rel.try_teq true env lc.res_typ t, false
             else Rel.get_subtyping_predicate env lc.res_typ t, true in
  match gopt with
    | None, _ ->
        if env.failhard
        then raise_error (Err.basic_type_error env (Some e) t lc.res_typ) e.pos
        else (
            subtype_fail env e lc.res_typ t; //log a sub-typing error
            e, {lc with res_typ=t}, Env.trivial_guard //and keep going to type-check the result of the program
        )
    | Some g, apply_guard ->
      match guard_form g with
        | Trivial ->
          (*
           * AR: when the guard is trivial, simply setting the result type to t might lose some precision
           *     e.g. when input lc has return type x:int{phi} and we are weakening it to int
           *     so we should capture the precision before setting the comp type to t (see e.g. #1500, #1470)
           *)
          let strengthen_trivial () =
            let c, g_c = TcComm.lcomp_comp lc in
            let res_t = Util.comp_result c in

            let set_result_typ (c:comp) :comp = Util.set_result_typ c t in

            if Util.eq_tm t res_t = Util.Equal then begin  //if the two types res_t and t are same, then just set the result type
              if Env.debug env <| Options.Extreme
              then BU.print2 "weaken_result_type::strengthen_trivial: res_t:%s is same as t:%s\n"
                             (Print.term_to_string res_t) (Print.term_to_string t);
              set_result_typ c, g_c
            end
            else
              let is_res_t_refinement =
                let res_t = N.normalize_refinement N.whnf_steps env res_t in
                match res_t.n with
                | Tm_refine _ -> true
                | _ -> false
              in
              //if t is a refinement, insert a return to capture the return type res_t
              //we are not inlining e, rather just adding (fun (x:res_t) -> p x) at the end
              if is_res_t_refinement then
                let x = S.new_bv (Some res_t.pos) res_t in
                let cret = return_value env (comp_univ_opt c) res_t (S.bv_to_name x) in
                let lc = bind e.pos env (Some e) (TcComm.lcomp_of_comp c) (Some x, TcComm.lcomp_of_comp cret) in
                if Env.debug env <| Options.Extreme
                then BU.print4 "weaken_result_type::strengthen_trivial: inserting a return for e: %s, c: %s, t: %s, and then post return lc: %s\n"
                               (Print.term_to_string e) (Print.comp_to_string c) (Print.term_to_string t) (TcComm.lcomp_to_string lc);
                let c, g_lc = TcComm.lcomp_comp lc in
                set_result_typ c, Env.conj_guard g_c g_lc
              else begin
                if Env.debug env <| Options.Extreme
                then BU.print2 "weaken_result_type::strengthen_trivial: res_t:%s is not a refinement, leaving c:%s as is\n"
                               (Print.term_to_string res_t) (Print.comp_to_string c);
                set_result_typ c, g_c
              end
          in
          let lc = TcComm.mk_lcomp lc.eff_name t lc.cflags strengthen_trivial in
          e, lc, g

        | NonTrivial f ->
          let g = {g with guard_f=Trivial} in
          let strengthen () =
              if env.lax
              && Options.ml_ish() //NS: disabling this optimization temporarily
              then
                TcComm.lcomp_comp lc
              else begin
                  //try to normalize one more time, since more unification variables may be resolved now
                  let f = N.normalize [Env.Beta; Env.Eager_unfolding; Env.Simplify; Env.Primops] env f in
                  match (SS.compress f).n with
                      | Tm_abs(_, {n=Tm_fvar fv}, _) when S.fv_eq_lid fv C.true_lid ->
                        //it's trivial
                        let lc = {lc with res_typ=t} in //NS: what's the point of this?
                        TcComm.lcomp_comp lc

                      | _ ->
                          let c, g_c = TcComm.lcomp_comp lc in
                          if Env.debug env <| Options.Extreme
                          then BU.print4 "Weakened from %s to %s\nStrengthening %s with guard %s\n"
                                  (N.term_to_string env lc.res_typ)
                                  (N.term_to_string env t)
                                  (N.comp_to_string env c)
                                  (N.term_to_string env f);

                          let u_t_opt = comp_univ_opt c in
                          let x = S.new_bv (Some t.pos) t in
                          let xexp = S.bv_to_name x in
                          let cret = return_value env u_t_opt t xexp in
                          let guard = if apply_guard
                                      then mk_Tm_app f [S.as_arg xexp] None f.pos
                                      else f
                          in
                          let eq_ret, _trivial_so_ok_to_discard =
                              strengthen_precondition (Some <| Err.subtyping_failed env lc.res_typ t)
                                                      (Env.set_range env e.pos)
                                                      e  //use e for debugging only
                                                      (TcComm.lcomp_of_comp cret)
                                                      (guard_of_guard_formula <| NonTrivial guard)
                          in
                          let x = {x with sort=lc.res_typ} in
                          let c = bind e.pos env (Some e) (TcComm.lcomp_of_comp c) (Some x, eq_ret) in
                          let c, g_lc = TcComm.lcomp_comp c in
                          if Env.debug env <| Options.Extreme
                          then BU.print1 "Strengthened to %s\n" (Normalize.comp_to_string env c);
                          c, Env.conj_guard g_c g_lc
                end
          in
          let flags = lc.cflags |> List.collect (function
                                                 | RETURN | PARTIAL_RETURN -> [PARTIAL_RETURN]
                                                 | CPS -> [CPS] // KM : Not exactly sure if it is necessary
                                                 | _ -> [])
          in
          let lc = TcComm.mk_lcomp (norm_eff_name env lc.eff_name) t flags strengthen in
          let g = {g with guard_f=Trivial} in
          (e, lc, g)

let pure_or_ghost_pre_and_post env comp =
    let mk_post_type res_t ens =
        let x = S.new_bv None res_t in
        U.refine x (S.mk_Tm_app ens [S.as_arg (S.bv_to_name x)] None res_t.pos) in
    let norm t = Normalize.normalize [Env.Beta;Env.Eager_unfolding;Env.EraseUniverses] env t in
    if U.is_tot_or_gtot_comp comp
    then None, U.comp_result comp
    else begin match comp.n with
            | GTotal _
            | Total _ -> failwith "Impossible"
            | Comp ct ->
              if lid_equals ct.effect_name C.effect_Pure_lid
              || lid_equals ct.effect_name C.effect_Ghost_lid
              then begin match ct.effect_args with
                      | (req, _)::(ens, _)::_ ->
                         Some (norm req), (norm <| mk_post_type ct.result_typ ens)
                      | _ ->
                        raise_error (Errors.Fatal_EffectConstructorNotFullyApplied, (BU.format1 "Effect constructor is not fully applied; got %s" (Print.comp_to_string comp))) comp.pos
                   end
              else let ct = Env.unfold_effect_abbrev env comp in
                   begin match ct.effect_args with
                            | (wp, _)::_ ->
                              let us_r, _ = fst <| Env.lookup_lid env C.as_requires in
                              let us_e, _ = fst <| Env.lookup_lid env C.as_ensures in
                              let r = ct.result_typ.pos in
                              let as_req = S.mk_Tm_uinst (S.fvar (Ident.set_lid_range C.as_requires r) delta_equational None) us_r in
                              let as_ens = S.mk_Tm_uinst (S.fvar (Ident.set_lid_range C.as_ensures r) delta_equational None) us_e in
                              let req = mk_Tm_app as_req [(ct.result_typ, Some S.imp_tag); S.as_arg wp] None ct.result_typ.pos in
                              let ens = mk_Tm_app as_ens [(ct.result_typ, Some S.imp_tag); S.as_arg wp] None ct.result_typ.pos in
                              Some (norm req), norm (mk_post_type ct.result_typ ens)
                            | _ -> failwith "Impossible"
                  end

         end

(* [reify_body env t] assumes that [t] has a reifiable computation type *)
(* that is env |- t : M t' for some effect M and type t' where M is reifiable *)
(* and returns the result of reifying t *)
let reify_body (env:Env.env) (steps:Env.steps) (t:S.term) : S.term =
    let tm = U.mk_reify t in
    let tm' = N.normalize
      ([Env.Beta; Env.Reify; Env.Eager_unfolding; Env.EraseUniverses; Env.AllowUnboundUniverses; Env.Exclude Env.Zeta]@steps)
      env tm in
    if Env.debug env <| Options.Other "SMTEncodingReify"
    then BU.print2 "Reified body %s \nto %s\n"
        (Print.term_to_string tm)
        (Print.term_to_string tm') ;
    tm'

let reify_body_with_arg (env:Env.env) (steps:Env.steps) (head:S.term) (arg:S.arg): S.term =
    let tm = S.mk (S.Tm_app(head, [arg])) None head.pos in
    let tm' = N.normalize
      ([Env.Beta; Env.Reify; Env.Eager_unfolding; Env.EraseUniverses; Env.AllowUnboundUniverses; Env.Exclude Env.Zeta]@steps)
      env tm in
    if Env.debug env <| Options.Other "SMTEncodingReify"
    then BU.print2 "Reified body %s \nto %s\n"
        (Print.term_to_string tm)
        (Print.term_to_string tm') ;
    tm'

let remove_reify (t: S.term): S.term =
  if (match (SS.compress t).n with | Tm_app _ -> false | _ -> true)
  then t
  else
    let head, args = U.head_and_args t in
    if (match (SS.compress head).n with Tm_constant FStar.Const.Const_reify -> true | _ -> false)
    then begin match args with
        | [x] -> fst x
        | _ -> failwith "Impossible : Reify applied to multiple arguments after normalization."
    end
    else t


(*********************************************************************************************)
(* Instantiation and generalization *)
(*********************************************************************************************)
let maybe_instantiate (env:Env.env) e t =
  let torig = SS.compress t in
  if not env.instantiate_imp
  then e, torig, Env.trivial_guard
  else begin
       if Env.debug env Options.High then
         BU.print3 "maybe_instantiate: starting check for (%s) of type (%s), expected type is %s\n"
                 (Print.term_to_string e) (Print.term_to_string t) (FStar.Common.string_of_option Print.term_to_string (Env.expected_typ env));
       let number_of_implicits t =
            let t = N.unfold_whnf env t in
            let formals, _ = U.arrow_formals t in
            let n_implicits =
            match formals |> BU.prefix_until (fun (_, imp) -> Option.isNone imp || U.eq_aqual imp (Some Equality) = U.Equal) with
                | None -> List.length formals
                | Some (implicits, _first_explicit, _rest) -> List.length implicits in
            n_implicits
       in
       let inst_n_binders t =
           match Env.expected_typ env with
           | None -> None
           | Some expected_t ->
             let n_expected = number_of_implicits expected_t in
             let n_available = number_of_implicits t in
             if n_available < n_expected
             then raise_error (Errors.Fatal_MissingImplicitArguments, (BU.format3 "Expected a term with %s implicit arguments, but %s has only %s"
                                        (BU.string_of_int n_expected)
                                        (Print.term_to_string e)
                                        (BU.string_of_int n_available))) (Env.get_range env)
             else Some (n_available - n_expected)
        in
        let decr_inst = function
                | None -> None
                | Some i -> Some (i - 1)
        in
        let t = N.unfold_whnf env t in
        begin match t.n with
            | Tm_arrow(bs, c) ->
              let bs, c = SS.open_comp bs c in
              //instantiate at most inst_n implicit binders, when inst_n = Some n
              //otherwise, instantate all implicits
              //See issue #807 for why this is important
              let rec aux subst inst_n bs =
                  match inst_n, bs with
                  | Some 0, _ -> [], bs, subst, Env.trivial_guard //no more instantiations to do
                  | _, (x, Some (Implicit _))::rest ->
                      let t = SS.subst subst x.sort in
                      let v, _, g = new_implicit_var "Instantiation of implicit argument" e.pos env t in
                      if Env.debug env Options.High then
                        BU.print1 "maybe_instantiate: Instantiating implicit with %s\n"
                                (Print.term_to_string v);
                      let subst = NT(x, v)::subst in
                      let args, bs, subst, g' = aux subst (decr_inst inst_n) rest in
                      (v, Some S.imp_tag)::args, bs, subst, Env.conj_guard g g'

                  | _, (x, Some (Meta tau))::rest ->
                      let t = SS.subst subst x.sort in
                      let v, _, g = new_implicit_var_aux "Instantiation of meta argument"
                                                         e.pos env t Strict
                                                         (Some (mkdyn env, tau)) in
                      if Env.debug env Options.High then
                        BU.print1 "maybe_instantiate: Instantiating meta argument with %s\n"
                                (Print.term_to_string v);
                      let subst = NT(x, v)::subst in
                      let args, bs, subst, g' = aux subst (decr_inst inst_n) rest in
                      (v, Some S.imp_tag)::args, bs, subst, Env.conj_guard g g'

                 | _, bs -> [], bs, subst, Env.trivial_guard
              in
              let args, bs, subst, guard = aux [] (inst_n_binders t) bs in
              begin match args, bs with
                | [], _ -> //no implicits were instantiated
                  e, torig, guard

                | _, [] when not (U.is_total_comp c) ->
                  //don't instantiate implicitly, if it has an effect
                  e, torig, Env.trivial_guard

                | _ ->

                  let t = match bs with
                    | [] -> U.comp_result c
                    | _ -> U.arrow bs c in
                  let t = SS.subst subst t in
                  let e = S.mk_Tm_app e args None e.pos in
                  e, t, guard
              end

            | _ -> e, torig, Env.trivial_guard
       end
  end

(**************************************************************************************)
(* Generalizing types *)
(**************************************************************************************)
let string_of_univs univs =
  BU.set_elements univs
  |> List.map (fun u -> Unionfind.univ_uvar_id u |> string_of_int) |> String.concat ", "

let gen_univs env (x:BU.set<universe_uvar>) : list<univ_name> =
    if BU.set_is_empty x then []
    else let s = BU.set_difference x (Env.univ_vars env) |> BU.set_elements in
         if Env.debug env <| Options.Other "Gen" then
         BU.print1 "univ_vars in env: %s\n" (string_of_univs (Env.univ_vars env));
         let r = Some (Env.get_range env) in
         let u_names = s |> List.map (fun u ->
            let u_name = Syntax.new_univ_name r in
            if Env.debug env <| Options.Other "Gen"
            then BU.print3 "Setting ?%s (%s) to %s\n"
                            (string_of_int <| Unionfind.univ_uvar_id u)
                            (Print.univ_to_string (U_unif u))
                            (Print.univ_to_string (U_name u_name));
            Unionfind.univ_change u (U_name u_name);
            u_name) in
         u_names

let gather_free_univnames env t : list<univ_name> =
    let ctx_univnames = Env.univnames env in
    let tm_univnames = Free.univnames t in
    let univnames = BU.set_difference tm_univnames ctx_univnames |> BU.set_elements in
    // BU.print4 "Closing universe variables in term %s : %s in ctx, %s in tm, %s globally\n"
    //     (Print.term_to_string t)
    //     (Print.set_to_string Ident.text_of_id ctx_univnames)
    //     (Print.set_to_string Ident.text_of_id tm_univnames)
    //     (Print.list_to_string Ident.text_of_id univnames);
    univnames

let check_universe_generalization
  (explicit_univ_names : list<univ_name>)
  (generalized_univ_names : list<univ_name>)
  (t : term)
  : list<univ_name>
=
  match explicit_univ_names, generalized_univ_names with
  | [], _ -> generalized_univ_names
  | _, [] -> explicit_univ_names
  | _ -> raise_error (Errors.Fatal_UnexpectedGeneralizedUniverse, ("Generalized universe in a term containing explicit universe annotation : "
                      ^ Print.term_to_string t)) t.pos

let generalize_universes (env:env) (t0:term) : tscheme =
    let t = N.normalize [Env.NoFullNorm; Env.Beta; Env.DoNotUnfoldPureLets] env t0 in
    let univnames = gather_free_univnames env t in
    if Env.debug env <| Options.Other "Gen"
    then BU.print2 "generalizing universes in the term (post norm): %s with univnames: %s\n" (Print.term_to_string t) (Print.univ_names_to_string univnames);
    let univs = Free.univs t in
    if Env.debug env <| Options.Other "Gen"
    then BU.print1 "univs to gen : %s\n" (string_of_univs univs);
    let gen = gen_univs env univs in
    if Env.debug env <| Options.Other "Gen"
    then BU.print2 "After generalization, t: %s and univs: %s\n"  (Print.term_to_string t) (Print.univ_names_to_string gen);
    let univs = check_universe_generalization univnames gen t0 in
    let t = N.reduce_uvar_solutions env t in
    let ts = SS.close_univ_vars univs t in
    univs, ts

let gen env (is_rec:bool) (lecs:list<(lbname * term * comp)>) : option<list<(lbname * list<univ_name> * term * comp * list<binder>)>> =
  if not <| (BU.for_all (fun (_, _, c) -> U.is_pure_or_ghost_comp c) lecs) //No value restriction in F*---generalize the types of pure computations
  then None
  else
     let norm c =
        if debug env Options.Medium
        then BU.print1 "Normalizing before generalizing:\n\t %s\n" (Print.comp_to_string c);
         let c = Normalize.normalize_comp [Env.Beta; Env.Exclude Env.Zeta; Env.NoFullNorm; Env.DoNotUnfoldPureLets] env c in
         if debug env Options.Medium then
            BU.print1 "Normalized to:\n\t %s\n" (Print.comp_to_string c);
         c in
     let env_uvars = Env.uvars_in_env env in
     let gen_uvars uvs = BU.set_difference uvs env_uvars |> BU.set_elements in
     let univs_and_uvars_of_lec (lbname, e, c) =
          let c = norm c in
          let t = U.comp_result c in
          let univs = Free.univs t in
          let uvt = Free.uvars t in
          if Env.debug env <| Options.Other "Gen"
          then BU.print2 "^^^^\n\tFree univs = %s\n\tFree uvt=%s\n"
                (BU.set_elements univs |> List.map (fun u -> Print.univ_to_string (U_unif u)) |> String.concat ", ")
                (BU.set_elements uvt |> List.map (fun u -> BU.format2 "(%s : %s)"
                                                                    (Print.uvar_to_string u.ctx_uvar_head)
                                                                    (Print.term_to_string u.ctx_uvar_typ)) |> String.concat ", ");
          let univs =
            List.fold_left
              (fun univs uv -> BU.set_union univs (Free.univs uv.ctx_uvar_typ))
              univs
             (BU.set_elements uvt) in
          let uvs = gen_uvars uvt in
          if Env.debug env <| Options.Other "Gen"
          then BU.print2 "^^^^\n\tFree univs = %s\n\tgen_uvars =%s"
                (BU.set_elements univs |> List.map (fun u -> Print.univ_to_string (U_unif u)) |> String.concat ", ")
                (uvs |> List.map (fun u -> BU.format2 "(%s : %s)"
                                                        (Print.uvar_to_string u.ctx_uvar_head)
                                                        (N.term_to_string env u.ctx_uvar_typ)) |> String.concat ", ");

         univs, uvs, (lbname, e, c)
     in
     let univs, uvs, lec_hd = univs_and_uvars_of_lec (List.hd lecs) in
     let force_univs_eq lec2 u1 u2 =
        if BU.set_is_subset_of u1 u2
        && BU.set_is_subset_of u2 u1
        then ()
        else let lb1, _, _ = lec_hd in
             let lb2, _, _ = lec2 in
             let msg = BU.format2 "Generalizing the types of these mutually recursive definitions \
                                   requires an incompatible set of universes for %s and %s"
                            (Print.lbname_to_string lb1)
                            (Print.lbname_to_string lb2) in
             raise_error (Errors.Fatal_IncompatibleSetOfUniverse, msg) (Env.get_range env)
     in
     let force_uvars_eq lec2 (u1:list<ctx_uvar>) (u2:list<ctx_uvar>) =
        let uvars_subseteq u1 u2 =
            u1 |> BU.for_all (fun u ->
            u2 |> BU.for_some (fun u' -> Unionfind.equiv u.ctx_uvar_head u'.ctx_uvar_head))
        in
        if uvars_subseteq u1 u2
        && uvars_subseteq u2 u1
        then ()
        else let lb1, _, _ = lec_hd in
             let lb2, _, _ = lec2 in
             let msg = BU.format2 "Generalizing the types of these mutually recursive definitions \
                                   requires an incompatible number of types for %s and %s"
                            (Print.lbname_to_string lb1)
                            (Print.lbname_to_string lb2) in
             raise_error (Errors.Fatal_IncompatibleNumberOfTypes, msg) (Env.get_range env)
     in

     let lecs =
        List.fold_right (fun this_lec lecs ->
           let this_univs, this_uvs, this_lec = univs_and_uvars_of_lec this_lec in
           force_univs_eq this_lec univs this_univs;
           force_uvars_eq this_lec uvs this_uvs;
           this_lec::lecs)
        (List.tl lecs)
        []
     in

     let lecs = lec_hd :: lecs in

     let gen_types (uvs:list<ctx_uvar>) : list<(bv * aqual)> =
         let fail k : unit =
             let lbname, e, c = lec_hd in
               raise_error (Errors.Fatal_FailToResolveImplicitArgument,
                            BU.format3 "Failed to resolve implicit argument of type '%s' in the type of %s (%s)"
                                       (Print.term_to_string k)
                                       (Print.lbname_to_string lbname)
                                       (Print.term_to_string (U.comp_result c)))
                            (Env.get_range env)
         in
         uvs |> List.map (fun u ->
         match Unionfind.find u.ctx_uvar_head with
         | Some _ -> failwith "Unexpected instantiation of mutually recursive uvar"
         | _ ->
           let k = N.normalize [Env.Beta; Env.Exclude Env.Zeta] env u.ctx_uvar_typ in
           let bs, kres = U.arrow_formals k in
           let _ =
             //we only generalize variables at type k = a:Type{phi}
             //where k is closed
             //this is in support of ML-style polymorphism, while also allowing generalizing
             //over things like eqtype, which is a common case
             //Otherwise, things go badly wrong: see #1091
             match (U.unrefine (N.unfold_whnf env kres)).n with
             | Tm_type _ ->
                let free = FStar.Syntax.Free.names kres in
                if not (BU.set_is_empty free) then fail kres

             | _ ->
               fail kres
           in
           let a = S.new_bv (Some <| Env.get_range env) kres in
           let t =
               match bs with
               | [] -> S.bv_to_name a
               | _ -> U.abs bs (S.bv_to_name a) (Some (U.residual_tot kres))
           in
           U.set_uvar u.ctx_uvar_head t;
            //t clearly has a free variable; this is the one place we break the
            //invariant of a uvar always being resolved to a term well-typed in its given context
           a, Some S.imp_tag)
     in

     let gen_univs = gen_univs env univs in
     let gen_tvars = gen_types uvs in

     let ecs = lecs |> List.map (fun (lbname, e, c) ->
         let e, c, gvs =
            match gen_tvars, gen_univs with
            | [], [] ->
              //nothing generalized
              e, c, []

            | _ ->
              //before we manipulate the term further, we must normalize it to get rid of the invariant-broken uvars
              let e0, c0 = e, c in
              let c = N.normalize_comp [Env.Beta; Env.DoNotUnfoldPureLets; Env.CompressUvars; Env.NoFullNorm; Env.Exclude Env.Zeta] env c in
              let e = N.reduce_uvar_solutions env e in
              let e =
                if is_rec
                then let tvar_args = List.map (fun (x, _) -> S.iarg (S.bv_to_name x)) gen_tvars in
                     let instantiate_lbname_with_app tm fv =
                        if S.fv_eq fv (right lbname)
                        then S.mk_Tm_app tm tvar_args None tm.pos
                        else tm
                    in FStar.Syntax.InstFV.inst instantiate_lbname_with_app e
                else e
              in
              //now, with the uvars gone, we can close over the newly introduced type names
              let t = match (SS.compress (U.comp_result c)).n with
                    | Tm_arrow(bs, cod) ->
                      let bs, cod = SS.open_comp bs cod in
                      U.arrow (gen_tvars@bs) cod

                    | _ ->
                      U.arrow gen_tvars c in
              let e' = U.abs gen_tvars e (Some (U.residual_comp_of_comp c)) in
              e', S.mk_Total t, gen_tvars in
          (lbname, gen_univs, e, c, gvs)) in
     Some ecs

let generalize' env (is_rec:bool) (lecs:list<(lbname*term*comp)>) : (list<(lbname*univ_names*term*comp*list<binder>)>) =
  assert (List.for_all (fun (l, _, _) -> is_right l) lecs); //only generalize top-level lets
  if debug env Options.Low
  then BU.print1 "Generalizing: %s\n"
       (List.map (fun (lb, _, _) -> Print.lbname_to_string lb) lecs |> String.concat ", ");
  let univnames_lecs = List.map (fun (l, t, c) -> gather_free_univnames env t) lecs in
  let generalized_lecs =
      match gen env is_rec lecs with
          | None -> lecs |> List.map (fun (l,t,c) -> l,[],t,c,[])
          | Some luecs ->
            if debug env Options.Medium
            then luecs |> List.iter
                    (fun (l, us, e, c, gvs) ->
                         BU.print5 "(%s) Generalized %s at type %s\n%s\nVars = (%s)\n"
                                          (Range.string_of_range e.pos)
                                          (Print.lbname_to_string l)
                                          (Print.term_to_string (U.comp_result c))
                                          (Print.term_to_string e)
                                          (Print.binders_to_string ", " gvs));
            luecs
   in
   List.map2 (fun univnames (l,generalized_univs, t, c, gvs) ->
              (l, check_universe_generalization univnames generalized_univs t, t, c, gvs))
             univnames_lecs
             generalized_lecs

let generalize env is_rec lecs = 
  Profiling.profile (fun () -> generalize' env is_rec lecs)
                    (Some (Ident.string_of_lid (Env.current_module env)))
                    "FStar.TypeChecker.Util.generalize"
                    
(************************************************************************)
(* Convertibility *)
(************************************************************************)
//check_has_type env e t1 t2
//checks is e:t1 has type t2, subject to some guard.
let check_has_type env (e:term) (lc:lcomp) (t2:typ) : term * lcomp * guard_t =
  let env = Env.set_range env e.pos in
  let check env t1 t2 =
    if env.use_eq
    then Rel.try_teq true env t1 t2
    else match Rel.get_subtyping_predicate env t1 t2 with
            | None -> None
            | Some f -> Some <| apply_guard f e
  in
  let e, lc, g_c = maybe_coerce_lc env e lc t2 in
  match check env lc.res_typ t2 with
  | None ->
    raise_error (Err.expected_expression_of_type env t2 e lc.res_typ) (Env.get_range env)
  | Some g ->
    if debug env <| Options.Other "Rel" then
      BU.print1 "Applied guard is %s\n" <| guard_to_string env g;
    e, lc, (Env.conj_guard g g_c)

/////////////////////////////////////////////////////////////////////////////////
let check_top_level env g lc : (bool * comp) =
  if debug env Options.Medium then
    BU.print1 "check_top_level, lc = %s\n" (TcComm.lcomp_to_string lc);
  let discharge g =
    force_trivial_guard env g;
    TcComm.is_pure_lcomp lc in
  let g = Rel.solve_deferred_constraints env g in
  let c, g_c = TcComm.lcomp_comp lc in
  if TcComm.is_total_lcomp lc
  then discharge (Env.conj_guard g g_c), c
  else let steps = [Env.Beta; Env.NoFullNorm; Env.DoNotUnfoldPureLets] in
       let c = Env.unfold_effect_abbrev env c
              |> S.mk_Comp
              |> Normalize.normalize_comp steps env in
       let ct, vc, g_pre = check_trivial_precondition env c in
       if Env.debug env <| Options.Other "Simplification"
       then BU.print1 "top-level VC: %s\n" (Print.term_to_string vc);
       discharge (Env.conj_guard g (Env.conj_guard g_c g_pre)), ct |> mk_Comp

(* Having already seen_args to head (from right to left),
   compute the guard, if any, for the next argument,
   if head is a short-circuiting operator *)
let short_circuit (head:term) (seen_args:args) : guard_formula =
    let short_bin_op f : args -> guard_formula = function
        | [] -> (* no args seen yet *) Trivial
        | [(fst, _)] -> f fst
        | _ -> failwith "Unexpexted args to binary operator" in

    let op_and_e e = U.b2t e   |> NonTrivial in
    let op_or_e e  = U.mk_neg (U.b2t e) |> NonTrivial in
    let op_and_t t = t |> NonTrivial in
    let op_or_t t  = t |> U.mk_neg |> NonTrivial in
    let op_imp_t t = t |> NonTrivial in

    let short_op_ite : args -> guard_formula = function
        | [] -> Trivial
        | [(guard, _)] -> NonTrivial guard
        | [_then;(guard, _)] -> U.mk_neg guard |> NonTrivial
        | _ -> failwith "Unexpected args to ITE" in
    let table =
        [(C.op_And,  short_bin_op op_and_e);
         (C.op_Or,   short_bin_op op_or_e);
         (C.and_lid, short_bin_op op_and_t);
         (C.or_lid,  short_bin_op op_or_t);
         (C.imp_lid, short_bin_op op_imp_t);
         (C.ite_lid, short_op_ite);] in

     match head.n with
        | Tm_fvar fv ->
          let lid = fv.fv_name.v in
          begin match BU.find_map table (fun (x, mk) -> if lid_equals x lid then Some (mk seen_args) else None) with
            | None ->   Trivial
            | Some g -> g
          end
        | _ -> Trivial

let short_circuit_head l =
    match (U.un_uinst l).n with
        | Tm_fvar fv ->
           BU.for_some (S.fv_eq_lid fv)
                   [C.op_And;
                    C.op_Or;
                    C.and_lid;
                    C.or_lid;
                    C.imp_lid;
                    C.ite_lid]
        | _ -> false



(************************************************************************)
(* maybe_add_implicit_binders (env:env) (bs:binders)                    *)
(* Adding implicit binders for ticked variables                         *)
(* in case the expected type is of the form #'a1 -> ... -> #'an -> t    *)
(* and bs does not begin with any implicit binders                      *)
(* add #'a1 ... #'an to bs                                              *)
(************************************************************************)
let maybe_add_implicit_binders (env:env) (bs:binders)  : binders =
    let pos bs = match bs with
        | (hd, _)::_ -> S.range_of_bv hd
        | _ -> Env.get_range env in
    match bs with
        | (_, Some (Implicit _))::_ -> bs //bs begins with an implicit binder; don't add any
        | _ ->
          match Env.expected_typ env with
            | None -> bs
            | Some t ->
                match (SS.compress t).n with
                    | Tm_arrow(bs', _) ->
                      begin match BU.prefix_until (function (_, Some (Implicit _)) -> false | _ -> true) bs' with
                        | None -> bs
                        | Some ([], _, _) -> bs //no implicits
                        | Some (imps, _,  _) ->
                          if imps |> BU.for_all (fun (x, _) -> BU.starts_with x.ppname.idText "'")
                          then let r = pos bs in
                               let imps = imps |> List.map (fun (x, i) -> (S.set_range_of_bv x r, i)) in
                               imps@bs //we have a prefix of ticked variables
                          else bs
                      end

                    | _ -> bs


//Decorating terms with monadic operators
let maybe_lift env e c1 c2 t =
    let m1 = Env.norm_eff_name env c1 in
    let m2 = Env.norm_eff_name env c2 in
    if Ident.lid_equals m1 m2
    || (U.is_pure_effect c1 && U.is_ghost_effect c2)
    || (U.is_pure_effect c2 && U.is_ghost_effect c1)
    then e
    else mk (Tm_meta(e, Meta_monadic_lift(m1, m2, t))) None e.pos

let maybe_monadic env e c t =
    let m = Env.norm_eff_name env c in
    if is_pure_or_ghost_effect env m
    || Ident.lid_equals m C.effect_Tot_lid
    || Ident.lid_equals m C.effect_GTot_lid //for the cases in prims where Pure is not yet defined
    then e
    else mk (Tm_meta(e, Meta_monadic (m, t))) None e.pos

let d s = BU.print1 "\x1b[01;36m%s\x1b[00m\n" s

// Takes care of creating the [fv], generating the top-level let-binding, and
// return a term that's a suitable reference (a [Tm_fv]) to the definition
let mk_toplevel_definition (env: env_t) lident (def: term): sigelt * term =
  // Debug
  if Env.debug env (Options.Other "ED") then begin
    d (text_of_lid lident);
    BU.print2 "Registering top-level definition: %s\n%s\n" (text_of_lid lident) (Print.term_to_string def)
  end;
  // Allocate a new top-level name.
  let fv = S.lid_as_fv lident (U.incr_delta_qualifier def) None in
  let lbname: lbname = Inr fv in
  let lb: letbindings =
    // the effect label will be recomputed correctly
    false, [U.mk_letbinding lbname [] S.tun C.effect_Tot_lid def [] Range.dummyRange]
  in
  // [Inline] triggers a "Impossible: locally nameless" error // FIXME: Doc?
  let sig_ctx = mk_sigelt (Sig_let (lb, [ lident ])) in
  {sig_ctx with sigquals=[ Unfold_for_unification_and_vcgen ]},
  mk (Tm_fvar fv) None Range.dummyRange


/////////////////////////////////////////////////////////////////////////////
//Checks that the qualifiers on this sigelt are legal for it
/////////////////////////////////////////////////////////////////////////////
let check_sigelt_quals (env:FStar.TypeChecker.Env.env) se =
    let visibility = function Private -> true | _ -> false in
    let reducibility = function
        | Abstract | Irreducible
        | Unfold_for_unification_and_vcgen | Visible_default
        | Inline_for_extraction -> true
        | _ -> false in
    let assumption = function Assumption | New -> true | _ -> false in
    let reification = function Reifiable | Reflectable _ -> true | _ -> false in
    let inferred = function
      | Discriminator _
      | Projector _
      | RecordType _
      | RecordConstructor _
      | ExceptionConstructor
      | HasMaskedEffect
      | Effect -> true
      | _ -> false in
    let has_eq = function Noeq | Unopteq -> true | _ -> false in
    let quals_combo_ok quals q =
        match q with
        | Assumption ->
          quals
          |> List.for_all (fun x -> x=q
                              || x=Logic
                              || inferred x
                              || visibility x
                              || assumption x
                              || (env.is_iface && x=Inline_for_extraction)
                              || x=NoExtract)

        | New -> //no definition provided
          quals
          |> List.for_all (fun x -> x=q || inferred x || visibility x || assumption x)

        | Inline_for_extraction ->
          quals |> List.for_all (fun x -> x=q || x=Logic || visibility x || reducibility x
                                              || reification x || inferred x || has_eq x
                                              || (env.is_iface && x=Assumption)
                                              || x=NoExtract)

        | Unfold_for_unification_and_vcgen
        | Visible_default
        | Irreducible
        | Abstract
        | Noeq
        | Unopteq ->
          quals
          |> List.for_all (fun x -> x=q || x=Logic || x=Abstract || x=Inline_for_extraction || x=NoExtract || has_eq x || inferred x || visibility x || reification x)

        | TotalEffect ->
          quals
          |> List.for_all (fun x -> x=q || inferred x || visibility x || reification x)

        | Logic ->
          quals
          |> List.for_all (fun x -> x=q || x=Assumption || inferred x || visibility x || reducibility x)

        | Reifiable
        | Reflectable _ ->
          quals
          |> List.for_all (fun x -> reification x || inferred x || visibility x || x=TotalEffect || x=Visible_default)

        | Private ->
          true //only about visibility; always legal in combination with others

        | _ -> //inferred
          true
    in
    let check_erasable quals se r =
        let lids = U.lids_of_sigelt se in
        let val_exists =
          lids |> BU.for_some (fun l -> Option.isSome (Env.try_lookup_val_decl env l))
        in
        let val_has_erasable_attr =
          lids |> BU.for_some (fun l ->
            let attrs_opt = Env.lookup_attrs_of_lid env l in
            Option.isSome attrs_opt
            && U.has_attribute (Option.get attrs_opt) FStar.Parser.Const.erasable_attr)
        in
        let se_has_erasable_attr = U.has_attribute se.sigattrs FStar.Parser.Const.erasable_attr in
        if ((val_exists && val_has_erasable_attr) && not se_has_erasable_attr)
        then raise_error
             (Errors.Fatal_QulifierListNotPermitted,
              "Mismatch of attributes between declaration and definition: \
               Declaration is marked `erasable` but the definition is not")
              r;
        if ((val_exists && not val_has_erasable_attr) && se_has_erasable_attr)
        then raise_error
             (Errors.Fatal_QulifierListNotPermitted,
              "Mismatch of attributed between declaration and definition: \
               Definition is marked `erasable` but the declaration is not")
              r;
        if se_has_erasable_attr
        then begin
          match se.sigel with
          | Sig_bundle _ ->
            if not (quals |> BU.for_some (function Noeq -> true | _ -> false))
            then raise_error
                   (Errors.Fatal_QulifierListNotPermitted,
                    "Incompatible attributes and qualifiers: \
                     erasable types do not support decidable equality and must be marked `noeq`")
                    r
          | Sig_declare_typ _ ->
            ()
          | _ ->
            raise_error
              (Errors.Fatal_QulifierListNotPermitted,
               "Illegal attribute: \
                the `erasable` attribute is only permitted on inductive type definitions")
               r
        end
    in
    let quals = U.quals_of_sigelt se |> List.filter (fun x -> not (x = Logic)) in  //drop logic since it is deprecated
    if quals |> BU.for_some (function OnlyName -> true | _ -> false) |> not
    then
      let r = U.range_of_sigelt se in
      let no_dup_quals = BU.remove_dups (fun x y -> x=y) quals in
      let err' msg =
          raise_error (Errors.Fatal_QulifierListNotPermitted, (BU.format2
                          "The qualifier list \"[%s]\" is not permissible for this element%s"
                          (Print.quals_to_string quals) msg)) r in
      let err msg = err' (": " ^ msg) in
      let err' () = err' "" in
      if List.length quals <> List.length no_dup_quals
      then err "duplicate qualifiers";
      if not (quals |> List.for_all (quals_combo_ok quals))
      then err "ill-formed combination";
      check_erasable quals se r;
      match se.sigel with
      | Sig_let((is_rec, _), _) -> //let rec
        if is_rec && quals |> List.contains Unfold_for_unification_and_vcgen
        then err "recursive definitions cannot be marked inline";
        if quals |> BU.for_some (fun x -> assumption x || has_eq x)
        then err "definitions cannot be assumed or marked with equality qualifiers"
      | Sig_bundle _ ->
        if not (quals |> BU.for_all (fun x ->
              x=Abstract
              || x=Inline_for_extraction
              || x=NoExtract
              || inferred x
              || visibility x
              || has_eq x))
        then err' ();
        if quals |> List.existsb (function Unopteq -> true | _ -> false) &&
           U.has_attribute se.sigattrs FStar.Parser.Const.erasable_attr
        then err "unopteq is not allowed on an erasable inductives since they don't have decidable equality"
      | Sig_declare_typ _ ->
        if quals |> BU.for_some has_eq
        then err' ()
      | Sig_assume _ ->
        if not (quals |> BU.for_all (fun x -> visibility x || x=Assumption))
        then err' ()
      | Sig_new_effect _ ->
        if not (quals |> BU.for_all (fun x ->
              x=TotalEffect
              || inferred x
              || visibility x
              || reification x))
        then err' ()
      | Sig_effect_abbrev _ ->
        if not (quals |> BU.for_all (fun x -> inferred x || visibility x))
        then err' ()
      | _ -> ()

let must_erase_for_extraction (g:env) (t:typ) =
    let rec descend env t = //t is expected to b in WHNF
      match (SS.compress t).n with
      | Tm_arrow _ ->
           let bs, c = U.arrow_formals_comp t in
           let env = FStar.TypeChecker.Env.push_binders env bs in
           (U.is_ghost_effect (U.comp_effect_name c))
           || (U.is_pure_or_ghost_comp c && aux env (U.comp_result c))
      | Tm_refine({sort=t}, _) ->
           aux env t
      | Tm_app (head, _)
      | Tm_uinst (head, _) ->
           descend env head
      | Tm_fvar fv ->
           //special treatment for must_erase_for_extraction here
           //See Env.type_is_erasable for more explanations
           Env.fv_has_attr env fv C.must_erase_for_extraction_attr
      | _ -> false
    and aux env t =
        let t = N.normalize [Env.Primops;
                             Env.Weak;
                             Env.HNF;
                             Env.UnfoldUntil delta_constant;
                             Env.Beta;
                             Env.AllowUnboundUniverses;
                             Env.Zeta;
                             Env.Iota;
                             Env.Unascribe] env t in
//        debug g (fun () -> BU.print1 "aux %s\n" (Print.term_to_string t));
        let res = Env.non_informative env t || descend env t in
        if Env.debug env <| Options.Other "Extraction"
        then BU.print2 "must_erase=%s: %s\n" (if res then "true" else "false") (Print.term_to_string t);
        res
    in
    aux g t

let fresh_effect_repr env r eff_name signature_ts repr_ts_opt u a_tm =
  let fail t = raise_error (Err.unexpected_signature_for_monad env eff_name t) r in
  
  let _, signature = Env.inst_tscheme signature_ts in

  (*
   * We go through the binders in the signature a -> bs
   * For each binder in bs, create a fresh uvar
   * But keep substituting [a/a_tm, b_i/?ui] in the sorts of the subsequent binders
   *)
  match (SS.compress signature).n with
  | Tm_arrow (bs, _) ->
    let bs = SS.open_binders bs in
    (match bs with
     | a::bs ->
       //is is all the uvars, and g is their collective guard
       let is, g = Env.uvars_for_binders env bs [NT (fst a, a_tm)]
         (fun b -> BU.format3
           "uvar for binder %s when creating a fresh repr for %s at %s"
           (Print.binder_to_string b) eff_name.str (Range.string_of_range r)) r in
       (match repr_ts_opt with
        | None ->  //no repr, return thunked computation type
          let eff_c = mk_Comp ({
            comp_univs = [u];
            effect_name = eff_name;
            result_typ = a_tm;
            effect_args = List.map S.as_arg is;
            flags = [] }) in
          S.mk (Tm_arrow ([S.null_binder S.t_unit], eff_c)) None r
        | Some repr_ts ->
          let repr = Env.inst_tscheme_with repr_ts [u] |> snd in
          S.mk_Tm_app
            repr
            (List.map S.as_arg (a_tm::is))
            None r), g
     | _ -> fail signature)
  | _ -> fail signature

let fresh_effect_repr_en env r eff_name u a_tm =
  eff_name
  |> Env.get_effect_decl env
  |> (fun ed -> fresh_effect_repr env r eff_name ed.signature (ed |> U.get_eff_repr)  u a_tm)

let layered_effect_indices_as_binders env r eff_name sig_ts u a_tm =
  let _, sig_tm = Env.inst_tscheme_with sig_ts [u] in

  let fail t = raise_error (Err.unexpected_signature_for_monad env eff_name t) r in

  match (SS.compress sig_tm).n with
  | Tm_arrow (bs, _) ->
    let bs = SS.open_binders bs in
    (match bs with
     | (a', _)::bs -> bs |> SS.subst_binders [NT (a', a_tm)]
     | _ -> fail sig_tm)
  | _ -> fail sig_tm

(*
 * Lifting a comp c to the layered effect eff_name
 *
 * let c = M<u_c> a_c wp_c
 *
 * let lift_M_eff_name = (u, lift_t) where
 *   lift_t = a:Type u -> wp:M_wp a -> (x_i:t_i) -> f:(unit -> M a wp) -> repr<u> a i_1 ... i_n)
 *
 * We first instantiate lift_t with u_c
 *
 * Then we create uvars (?u_i:t_i), while subtituting [a/a_c; wp/wp_c; x_j/?u_j] (forall j < i)
 *
 * let substs = [a/a_c; wp/wp_c; x_i/?u_i]
 *
 * We return M'<u_c> a_c i_i[substs]
 *)
let lift_tf_layered_effect (tgt:lident) (lift_ts:tscheme) env (c:comp) : comp * guard_t =
  if Env.debug env <| Options.Other "LayeredEffects" then
    BU.print2 "Lifting comp %s to layered effect %s {\n"
      (Print.comp_to_string c) (Print.lid_to_string tgt);

  let r = Env.get_range env in

  let ct = U.comp_to_comp_typ c in

  let u, a, c_is = List.hd ct.comp_univs, ct.result_typ, ct.effect_args |> List.map fst in

  //lift_ts has the arrow type: <u>a:Type -> ..bs.. -> f -> repr a is

  let _, lift_t = Env.inst_tscheme_with lift_ts [u] in

  let lift_t_shape_error s =
    BU.format4 "Lift from %s to %s has unexpected shape, reason: %s (lift:%s)"
      (Ident.string_of_lid ct.effect_name) (Ident.string_of_lid tgt)
      s (Print.term_to_string lift_t) in

  let a_b, (rest_bs, [f_b]), lift_ct =
    match (SS.compress lift_t).n with
    | Tm_arrow (bs, c) when List.length bs >= 2 ->
      let ((a_b::bs), c) = SS.open_comp bs c in
      a_b, bs |> List.splitAt (List.length bs - 1), U.comp_to_comp_typ c
    | _ ->
      raise_error (Errors.Fatal_UnexpectedEffect, lift_t_shape_error
        "either not an arrow or not enough binders") r in

  let rest_bs_uvars, g = Env.uvars_for_binders env rest_bs
    [NT (a_b |> fst, a)]
    (fun b -> BU.format4
      "implicit var for binder %s of %s~>%s at %s"
      (Print.binder_to_string b) (Ident.string_of_lid ct.effect_name)
      (Ident.string_of_lid tgt) (Range.string_of_range r)) r in

  if debug env <| Options.Other "LayeredEffects" then
    BU.print1 "Introduced uvars: %s\n"
      (List.fold_left (fun s u -> s ^ ";;;;" ^ (Print.term_to_string u)) "" rest_bs_uvars);

  let substs = List.map2
    (fun b t -> NT (b |> fst, t))
    (a_b::rest_bs) (a::rest_bs_uvars) in

  let guard_f =
    let f_sort = (fst f_b).sort |> SS.subst substs |> SS.compress in
    let f_sort_is = effect_args_from_repr f_sort (Env.is_layered_effect env ct.effect_name) r in
    List.fold_left2
      (fun g i1 i2 -> Env.conj_guard g (Rel.teq env i1 i2))
      Env.trivial_guard c_is f_sort_is in

  let is = effect_args_from_repr lift_ct.result_typ (Env.is_layered_effect env tgt) r in

  let c = mk_Comp ({
    comp_univs = lift_ct.comp_univs;  //AR: TODO: not too sure about this
    effect_name = tgt;
    result_typ = a;
    effect_args = is |> List.map (SS.subst substs) |> List.map S.as_arg;
    flags = []  //AR: setting the flags to empty
  }) in

  if debug env <| Options.Other "LayeredEffects" then
    BU.print1 "} Lifted comp: %s\n" (Print.comp_to_string c);

  c, Env.conj_guard g guard_f

(*
 * Creating the Env.mlift.mlift_term function for layered effects
 * Quite simple, just apply the lift term, passing units for the
 * binders that are meant to compute indices
 *)
let lift_tf_layered_effect_term env (sub:sub_eff)
  (u:universe) (a:typ) (e:term) : term =

  let lift = sub.lift |> must |> (fun ts -> inst_tscheme_with ts [u]) |> snd in

  let rest_bs =
    let lift_t = sub.lift_wp |> must in
    match (lift_t |> snd |> SS.compress).n with
    | Tm_arrow (_::bs, _) when List.length bs >= 1 ->
      bs |> List.splitAt (List.length bs - 1) |> fst
    | _ ->
      raise_error (Errors.Fatal_UnexpectedEffect,
        BU.format1 "lift_t tscheme %s is not an arrow with enough binders"
          (Print.tscheme_to_string lift_t)) (snd lift_t).pos in

  let args = (S.as_arg a)::((rest_bs |> List.map (fun _ -> S.as_arg S.unit_const))@[S.as_arg e]) in
  mk (Tm_app (lift, args)) None e.pos

let get_field_projector_name env datacon index =
  let _, t = Env.lookup_datacon env datacon in
  let err n =
    raise_error (Errors.Fatal_UnexpectedDataConstructor,
      BU.format3 "Data constructor %s does not have enough binders (has %s, tried %s)"
        (Ident.string_of_lid datacon) (string_of_int n) (string_of_int index)) (Env.get_range env) in
  match (SS.compress t).n with
  | Tm_arrow (bs, _) ->
    let bs = bs |> List.filter (fun (_, q) -> match q with | Some (Implicit true) -> false | _ -> true) in
    if List.length bs <= index then err (List.length bs)
    else
      let b = List.nth bs index in
      U.mk_field_projector_name datacon (fst b) index |> fst
  | _ -> err 0


let get_mlift_for_subeff env (sub:S.sub_eff) : Env.mlift =
  if Env.is_layered_effect env sub.source || Env.is_layered_effect env sub.target

  then
    ({ mlift_wp = lift_tf_layered_effect sub.target (sub.lift_wp |> must);
       mlift_term = Some (lift_tf_layered_effect_term env sub) })

  else
    let mk_mlift_wp ts env c =
      let ct = U.comp_to_comp_typ c in
      let _, lift_t = inst_tscheme_with ts ct.comp_univs in
      let wp = List.hd ct.effect_args in
      S.mk_Comp ({ ct with
        effect_name = sub.target;
        effect_args =
          [mk (Tm_app (lift_t, [as_arg ct.result_typ; wp])) None (fst wp).pos |> S.as_arg]
      }), TcComm.trivial_guard
    in

    let mk_mlift_term ts u r e =
      let _, lift_t = inst_tscheme_with ts [u] in
      mk (Tm_app (lift_t, [as_arg r; as_arg S.tun; as_arg e])) None e.pos
    in

    ({ mlift_wp = sub.lift_wp |> must |> mk_mlift_wp;
       //AR: this is funky
       //it is saying, if you don't give us a lift term (a function that lifts terms),
       //we are assuming that the function is an identity
       //so for example, primitive effects just specify lift wps, and not terms
       //for them we assume that the terms are identity functions
       //why do we need it?
       //suppose programmer writes a layered effect M and defines a lift from DIV to M
       //now a PURE computation in the VC gets lifted via: PURE ~> DIV ~> M
       //when extracting (and reifying the monadic lifts), we go the same route
       //but if there is no lift term from PURE ~> DIV, we get an error
       //is this ok to do for DM4F? not sure in general
       //but currently PURE and DIV are lifted to DM4F effects using M.return
       //and not using the lift term (I don't think the lift term is even supported for DM4F, is it?)
       mlift_term =
         match sub.lift with
         | None -> Some (fun _ _ e -> return_all e)
         | Some ts -> Some (mk_mlift_term ts) })


let update_env_sub_eff env sub =
  Env.update_effect_lattice env sub.source sub.target (get_mlift_for_subeff env sub)

let update_env_polymonadic_bind env m n p ty =
  Env.add_polymonadic_bind env m n p
    (fun env c1 bv_opt c2 flags r -> mk_indexed_bind env m n p ty c1 bv_opt c2 flags r)
