(*
   Copyright 2008-2018 Microsoft Research

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*)
module Add

open FStar.Tactics.Typeclasses

(* A class for (additive, whatever that means) monoids *)
class additive a = {
  zero       : a;
  plus       : a -> a -> a;
  zero_l     : ((x : a) -> Lemma (plus zero x == x));
  zero_r     : ((x : a) -> Lemma (plus x zero == x));
  plus_assoc : ((x : a) -> (y : a) -> (z : a)
                  -> Lemma (plus (plus x y) z == plus x (plus y z)));
}

(*
 * A smart constructor, would be nice to autogen too.
 * But how? Mark some methods as `irrel`?
 * Note there's a nontrivial translation. Should we do forall's? Lemmas? Squashes?
 *)
val mkadd : #a:Type -> zero:a -> plus:(a -> a -> a)
             -> Pure (additive a)
                    (requires (forall (x : a). plus zero x == x)
                            /\ (forall (x : a). plus x zero == x)
                            /\ (forall (x y z : a).plus (plus x y) z == plus x (plus y z)))
                    (ensures (fun d -> Mkadditive?.zero d == zero /\ Mkadditive?.plus d == plus))
let mkadd #a zero plus = Mkadditive zero plus (fun x -> ()) (fun x -> ()) (fun x y z -> ())

(* These methods are generated by the splice *)
(* [@@tcnorm] let zero       (#a:Type) [|d : additive a|] = d.zero *)
(* [@@tcnorm] let plus       (#a:Type) [|d : additive a|] = d.plus *)
(* [@@tcnorm] let zero_l     (#a:Type) [|d : additive a|] = d.zero_l *)
(* [@@tcnorm] let zero_r     (#a:Type) [|d : additive a|] = d.zero_r *)
(* [@@tcnorm] let plus_assoc (#a:Type) [|d : additive a|] = d.plus_assoc *)

(* Instances *)
instance add_int : additive int = mkadd 0 (+)

instance add_bool : additive bool = mkadd false ( || )

instance add_list #a : additive (list a) =
  (* Couldn't use the smart mkadd here, oh well *)
  let open FStar.List.Tot in
  Mkadditive [] (@) append_nil_l append_l_nil append_assoc

(* Tests *)
let _ = assert (plus 1 2 = 3)
let _ = assert (plus false false = false)
let _ = assert (plus true false = true)
let _ = assert (plus [1] [2] = [1;2])
