(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
module Microsoft.FSharp.Compiler.Tlr 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
F#*)

(*F# module Ildiag = Microsoft.Research.AbstractIL.Diagnostics F#*) open Ildiag
open List
open Ast
open Tast
open Tastops
open Env
open Layout
open Detuple.GlobalUsageAnalysis
open Lib

let verboseTLR = false
let liftTLR    = ref false
  (* NOTE: liftTLR incomplete.
   *       Approach is to filter Top* let binds whilst "under lambdas",
   *       and wrap them around that expr ASAP (when get to TopLevel position).
   *       However, for arity assigned public vals (not TLR at moment),
   *       assumptions that their RHS are lambdas get broken since the
   *       lambda can be wrapped with bindings...
   * To resolve before activating liftTLR.
   *)
  (* Explicitlt lift TLR to be toplevel? otherwise, just mark as Top* repr.
   * e.g. use false to check terms produced by transform,
   *      without reordering everything to be hard to follow.
   * e.g. use true if want the TLR constants to be initialised once.
   *)


(*-------------------------------------------------------------------------
 *INDEX: rebinds
 *-------------------------------------------------------------------------*)

let internalError str = dprintf1 "Error: %s\n" str;raise (Failure str)  

let forceM' k   mp (str,soK) = try forceM k mp with e -> dprintf2 "forceM: %s %s\n" str (soK k); raise e


(*-------------------------------------------------------------------------
 *INDEX: misc
 *-------------------------------------------------------------------------*)


(* tree, used to store dec sequence *)
type 'a tree =
    Nd of 'a tree list
  | Lf of 'a

let fringeTR tr =
  let rec collect tr acc =
    match tr with
      Nd subts -> List.fold_right collect subts acc
    | Lf x     -> x::acc
  in
  collect tr []

let listTR ts = Nd ts
let unitTR x = Lf x
let emptyTR = Nd[]


(*-------------------------------------------------------------------------
 *INDEX: misc
 *-------------------------------------------------------------------------*)

(* CLEANUP NOTE: mk_appl ensures applications are kept in a collapsed *)
(* and combined form, so this function should not be needed *)
let destApp (f,fty,tys,args,m) =
   (* collapse reclinks on app and combine apps if possible *)
   (* recursive ids are inside reclinks and maybe be type instanced with a TExpr_app *)
   match strip_expr f with
   | TExpr_app (f2,fty2,tys2,[]     ,m2) -> (f2,fty2,tys2 @ tys,args,m)
   | TExpr_app (f2,fty2,tys2,argtys2,m2) -> (f,fty,tys,args,m) (* has args, so not combine ty args *)
   | f                                     -> (f,fty,tys,args,m)

let soTyparSet tps = showL (commaListL (map typarL (Zset.elements tps)))    
let soTyp t = (DebugPrint.showType t)    
let soLength xs = string_of_int (length xs)

(* CLEANUP NOTE: don't like the look of this function - this distinction *)
(* should never be needed *)
let isDelayedRepr f e = 
    let tps,vss,b,rty = dest_top_lambda (e,type_of_val f) in
    length vss>0

(*-------------------------------------------------------------------------
 *INDEX: mkLocalVal
 *-------------------------------------------------------------------------*)

(* REVIEW: these should just be replaced by direct calls to mk_local, mk_compgen_local etc. *)
(* REVIEW: However these set an arity whereas the others don't *)
let mkLocalNameTypeArity compgen m name ty arity_info =
    let name = nng.nngApply name m in
    let pub        = None       in
    let special_repr  = None  in
    let thisv      = new_vspec(ident(name,m),ty,Immutable,compgen,arity_info,pub,taccessPublic,ValNotInRecScope,special_repr,NormalVal,[],OptionalInline,emptyXMLDoc,false,false,false,false,None,ParentNone) in
    thisv

let mkLocalNameType compgen m name ty = mkLocalNameTypeArity compgen m name ty None
let mkLocalTmp      compgen m ty = mkLocalNameType compgen m "tmp" ty


(*-------------------------------------------------------------------------
 *INDEX: definitions: TLR, arity, arity-met, arity-short
 *-------------------------------------------------------------------------*)

(* DEFN: An f is TLR with arity wf if
         (a) it's repr is "LAM tps. lam x1...xN. body" and have N<=wf (i.e. have enough args)
         (b) it has no free tps
         (c) for g:freevars(repr), both
             (1) g is TLR with arity wg, and
             (2) g occurs in arity-met occurance.
         (d) if N=0, then further require that body be a TLR-constant.

   Conditions (a-c) are required if f is to have a static method/field represenation.
   Condition (d) chooses which constants can be lifted. (no effects, non-trivial).

   DEFN: An arity-met occurance of g is a g application with enough args supplied,
         ie. (g tps args) where wg <= |args|.

   DEFN: An arity-short occurance does not have enough args.

   DEFN: A TLR-constant:
         - can have constructors (tuples, datatype, records, exn).
         - should be non-trivial (says, causes allocation).
         - if calls are allowed, they must be effect free (since eval point is moving).
*)


(*-------------------------------------------------------------------------
 *INDEX: OVERVIEW
 *-------------------------------------------------------------------------*)

(* Overview of passes (over term) and steps (not over term):

   pass1 - decide which f will be TLR and determine their arity.
   pass2 - what closures are needed? Finds etps(f) and envReq(f) for TLR f.
           Depends on the arity choice, so must follow pass1.
   step3 - choose env packing, create fHats.
   pass4 - rewrite term fixing up definitions and callsites.
           Depends on closure and env packing, so must follow pass2 (and step 3).
   pass5 - copy_expr call to topexpr to ensure all bound ids are unique.
           For complexity reasons, better to re-recurse over expr once.
   pass6 - sanity check, confirm that all TLR marked bindings meet DEFN.

*)   

(*-------------------------------------------------------------------------
 *INDEX: pass1: valsBoundUnderMustInline (see comment further below)
 *-------------------------------------------------------------------------*)

let valsBoundUnderMustInline ccu xinfo =
    let accRejectFrom v repr rejectS =
      if inlineFlag_of_val v = PseudoValue then
        Zset.union (vals_bound_in_expr repr) rejectS
      else rejectS in
    let rejectS = Zset.empty val_spec_order in
    let rejectS = Zmap.fold accRejectFrom xinfo.xinfo_eqns rejectS in
    rejectS

(*-------------------------------------------------------------------------
 *INDEX: pass1: isTLRConstant
 *-------------------------------------------------------------------------*)

(*
let rec trivialExpr x =
  match x with
  | TExpr_val _                       -> true
  | TExpr_op(TOp_uconstr (_),tyargs,[],_)    -> true
  | TExpr_const _                     -> true
  | TExpr_app((f0,f0ty),tyargsl,[],m) -> not (is_tyfunc_vref_expr f0) && trivialExpr f0
  | _                                 -> false

let rec const_expr x =
  (* For now: constructions around constants
   * Later, can also refer to *PRIOR* TLR constants (i.e. CSE) - need declaration order.
   *)
  match x with
  | TExpr_const _                    -> true
  | TExpr_op(TOp_tuple, tys,args,m)        -> const_exprs args
  | TExpr_op(TOp_uconstr cr,tinst,args,m)   -> const_exprs args && not (tcref_alloc_observable (fst cr))
  | TExpr_op(TOp_recd   (ctor,tcr),tinst,args,m)  -> ctor = None && const_exprs args && not (tcref_alloc_observable tcr)
(*| TExpr_lambda _ | TExpr_tlambda _ -> true -- these should be ok *)
(*| could also allow calls to functions which have no known effect *)
  | _                                -> false
and const_exprs es = for_all const_expr es

let isTLRConstant x = const_expr x && not (trivialExpr x)
*)

(*-------------------------------------------------------------------------
 *INDEX: pass1: refusedTLR
 *-------------------------------------------------------------------------*)
       
let refusedTLR g f =  
    let mutableVal = mutability_of_val f <> Immutable in
    (* things marked NeverInline are special *)
    let dllImportStubOrOtherNeverInline = (inlineFlag_of_val f = NeverInline) in 
    (* Cannot have static fields of byref type *)
    let byrefVal = is_byref_ty g (type_of_val f) in
    (* Special values are instance methods etc. on .NET types.  For now leave these alone *)
    let specialVal = isSome(member_info_of_val f) in
    let alreadyChosen = isSome(arity_of_val f) in
    let refuseTest = alreadyChosen || mutableVal || byrefVal || specialVal || dllImportStubOrOtherNeverInline in
    refuseTest

let mandatoryTopLevel f = 
    let specialVal = isSome(member_info_of_val f) in
    let isModulBinding = modbind_of_val f in
    specialVal || isModulBinding

let mandatoryNonTopLevel g f =
    let byrefVal = is_byref_ty g (type_of_val f) in
    byrefVal

(*-------------------------------------------------------------------------
 *INDEX: pass1: decide which f are to be TLR? and if so, arity(f)
 *-------------------------------------------------------------------------*)

let maxNArgsAtUses xinfo f =
   match Zmap.tryfind f xinfo.xinfo_uses with
   | None       -> 0 (* no call sites *)
   | Some sites -> let nArgs (accessors,tinst,args) = length args in
                   maxOnL nArgs sites

(*
let refusedUsedOnce g f = refusedTLR g f

let selectUsedOnce g xinfo f e =
  if refusedUsedOnce g f then None else
  match e with 
  | TExpr_lambda _ | TExpr_tlambda _ -> 
    begin match Zmap.tryfind f xinfo.xinfo_uses with 
    | Some [site] -> Some(f ) 
    | _ -> None
    end
  | _ -> None
*)

(* REVIEW: This may be undoing good choices made in detuple *)
let selectTLR g xinfo f e =
  if refusedTLR g f then None 
  else if Zset.mem f xinfo.xinfo_dtree then None
  else
      (* Could the binding be TLR? with what arity? *)
      (* REVIEW: this is allowing bindings of the form; 
             let x = 
                  let y = <big-term>
                  let f z = ... y ...
         to have 'f' as a TLR value without any free variables (i.e. by representing 'y' as top-level).  
         Fine, except that this can cause a space leak *)
         
      let atTopLevel = Zset.mem f xinfo.xinfo_toplevel in 
      let tps,vss,b,rty = dest_top_lambda (e,type_of_val f) in
      let nFormals    = length vss in
      let nMaxApplied = maxNArgsAtUses xinfo f   in
      let arity       = min nFormals nMaxApplied in
      if atTopLevel or arity<>0 or nonNil tps then Some (f,arity)
      else None

(*
  DISABLED: TLR constants: this breaks the normal form where things labelled with "[]" arity (i.e. static fields)
     are only intializedin the cctor. 
  
    (* only choose arity=0 if it is a constant *)
    if (* atTopLevel || - no, since lifting will move non-constant expressions *)
           isTLRConstant e
    then
      (if verboseTLR then dprintf1 "selectTLR: TLR constant %s\n" (name_of_val f);
        Some (f,arity))
    else
        None
  else
*)

(*let isInRecursiveBinding xinfo f =
   let recursive,mudefs = forceM f xinfo.xinfo_mubinds in
   recursive 
*)
let isValueRecursionFree xinfo f =
   (* Check if f involves any value recursion (so can skip those).
    * ValRec considered: recursive && some f in mutual binding is not bound to a lambda
    *)
   let hasDelayedRepr f = isDelayedRepr f (forceM' f xinfo.xinfo_eqns ("isValueRecursionFree - hasDelayedRepr",name_of_val)) in
   let recursive,mudefs = forceM' f xinfo.xinfo_mubinds ("isValueRecursionFree",name_of_val) in
   not recursive || for_all hasDelayedRepr mudefs

let dumpArity arityM =
  let dump f n = dprintf2 "tlr: arity %50s = %d\n" (showL (vspecL f)) n in
  Zmap.iter dump arityM

let determineTLRAndArities ccu g (expr : typedAssembly) =
   if verboseTLR then dprintf0 "determineTLRAndArities------\n";
   let xinfo = xinfo_of_assembly g expr in
   let fArities = Zmap.chooseL (selectTLR g xinfo) xinfo.xinfo_eqns in
   let fArities = filter (fst >> isValueRecursionFree xinfo) fArities in
   (* Do not TLR v if it is bound under a mustinline defn *)
   (* There is simply no point - the original value will be duplicated and TLR'd anyway *)
   (* However we could report warnings for such values as they lead to duplication *)
   (* which could be avoided by making inlineable versions of the TLR'd values *)
   (* also available. *)
   let rejectS = valsBoundUnderMustInline ccu xinfo in
   let fArities = filter (fun (v,_) -> not (Zset.mem v rejectS)) fArities in
   (*-*)
   let tlrS      = listS val_spec_order (map fst fArities) in
   let topValS   = xinfo.xinfo_toplevel in                    (* genuinely top level *)
   let topValS   = Zset.filter (mandatoryNonTopLevel g >> not) topValS in    (* restrict *)
(*
   let fUsedOnce = Zmap.chooseL (selectUsedOnce g xinfo) xinfo.xinfo_eqns in 
   let fUsedOnce = filter (isInRecursiveBinding xinfo >> not) fUsedOnce in
   let fUsedOnce = filter (fun v -> not (Zset.mem v rejectS)) fUsedOnce in
   let fUsedOnce = filter (fun v -> not (Zset.mem v tlrS)) fUsedOnce in
   let fUsedOnce = filter (fun v -> not (Zset.mem v topValS)) fUsedOnce in
   dprintf1 "#fUsedOnce = %d\n" (length fUsedOnce);
   fUsedOnce |> List.iter (fun v -> (data_of_val v).val_mustinline <- PseudoValue);
*)
   (* REPORT MISSED CASES *)
   begin 
     if verboseTLR then 
       let missed = Zset.diff  xinfo.xinfo_toplevel tlrS in
       Zset.iter (fun v -> dprintf1 "TopLevel but not TLR = %s\n" (name_of_val v)) missed
   end;
   (* REPORT OVER *)   
   let arityM = listM val_spec_order fArities in
   if verboseTLR then dumpArity arityM;
   tlrS,topValS, arityM

(* NOTES:
   For constants,
     Want to fold in a declaration order,
     so can make decisions about TLR given TLR-knowledge about prior constants.
     Assuming ilxgen will fix up initialisations.
   So,
     xinfo to be extended to include some scoping representation.
     Maybe a telescope tree which can be walked over.
 *)


(*-------------------------------------------------------------------------
 *INDEX: pass2: determine etps(f) and envreq(f) - notes
 *-------------------------------------------------------------------------*)

(* What are the closing types/values for {f1,f2...} mutally defined?

   Note: arity-met g-applications (g TLR) will translated as:
           [[g @ tps ` args]] -> gHAT @ etps(g) tps ` env(g) args
         so they require availability of closing types/values for g.

   If g is free wrt f1,f2... then g's closure must be included.

   Note: mutual definitions have a common closure.

   For f1,f2,... = fBody1,fbody2... mutual bindings:

   DEFN: The generators are the free-values of fBody1,fBody2...
  
   What are the closure equations?

   etps(f1,f2..)    includes free-tps(f)
   etps(f1,f2..)    includes etps(g) if fBody has arity-met g-occurance (g TLR).

   envReq(f1,f2...) includes ReqSubEnv(g) if fBody has arity-met   g-occurance (g TLR)
   envReq(f1,f2...) includes ReqVal(g)    if fBody has arity-short g-occurance (g TLR)
   envReq(f1,f2...) includes ReqVal(g)    if fBody has g-occurance (g not TLR)

   and only collect requirements if g is a generator (see next notes).

   Note: "env-availability"
     In the translated code, env(h) will be defined at the h definition point.
     So, where-ever h could be called (recursive or not),
     the env(h) will be available (in scope).
   
   Note (subtle): "sub-env-requirement-only-for-generators"
     If have an arity-met call to h inside fBody, but h is not a freevar for f,
     then h does not contribute env(h) to env(f), the closure for f.
     It is true that env(h) will be required at the h call-site,
     but the env(h) will be available there (by "env-availability"),
     since h must be bound inside the fBody since h was not a freevar for f.
     .
     [note, f and h may mutally recurse and formals of f may be in env(h),
      so env(f) may be properly inside env(h),
      so better not have env(h) in env(f)!!!].
*)


(*-------------------------------------------------------------------------
 *INDEX: pass2: determine etps(f) and envreq(f) - plan
 *-------------------------------------------------------------------------*)

(* IMPLEMENTATION PLAN:

- fold over expr.

  - at an instance g,
     - (a) g arity-met,   logReqFrom g - ReqSubEnv(g) -- direct call will require env(g) and etps(g)
     - (b) g arity-short, logReqFrom g - ReqVal(g)    -- remains g call
     - (c) g non-TLR,     logReqFrom g - ReqVal(g)    -- remains g
    where
     logReqFrom g ... = logs info into (generators,env) if g in generators.

  - at some mu-bindings, f1,f2... = fBody1,fBody2,...
    "note generators, push (generators,env), fold-over bodies, pop, fold rest"
     
    - let fclass = ff1,... be the fi which are being made TLR.
    - required to find an env for these.
    - start a new envCollector:
        freetps = freetypars of (fBody1,fBody2,...)
        freevs  = freevars   of ..
        initialise:
          etps       = freetps
          envReq     = []      -- info collected from generator occurances in bindings
          generators = freevs
    - fold bodies, collecting info for generators.
    - pop and save env.
      - note: - etps(fclass) are only the freetps
              - they need to include etps(g) for each direct call to g (g a generator for fclass)
              - the etps(g) may not yet be known,
                e.g. if we are inside the definition of g and had recursively called it.
              - so need to FIX up the etps(-) function when collected info for all fclass.
    - fold rest (after binding)

- fix up etps(-) according to direct call dependencies.

*)


(*-------------------------------------------------------------------------
 *INDEX: pass2: fclass
 *-------------------------------------------------------------------------*)

type fclass =
   (* The subset of ids from a mutal binding that are chosen to be TLR.
    * They share a common env.
    * [Each fclass has an env, the fclass are the handles to envs.]
    *)
    FC of val_spec list

let mkFClass fs = FC fs
let fclass_order ccu = orderOn (fun (FC fs) -> fs) (listOrder val_spec_order)
let fclassPairs ((FC fs) as fc) = map (fun f -> (f,fc)) fs
let memFC f (FC fs) = exists (equalOn stamp_of_val f) fs
let isEmptyFC = function (FC []) -> true | _ -> false

let soFC = function (FC fs) -> "+" ^ String.concat "+" (map name_of_val fs)


(*-------------------------------------------------------------------------
 *INDEX: pass2: envItem, env
 *-------------------------------------------------------------------------*)

type envItem =
    (* It is required to make the TLR closed wrt it's freevars (the env generators).
     * For g a generator,
     *   An arity-met g occurance contributes the env required for that g call.
     *   Other occurances contribute the value g.
     *)
    ReqSubEnv of val_spec
  | ReqVal    of val_spec

let envItem_order ccu =
  let rep = function
      ReqSubEnv v -> true ,v
    | ReqVal    v -> false,v
  in
  orderOn rep (tup2Order (bool_order,val_spec_order))

type env =
   (* An env says what is needed to close the corresponding defn(s).
    * The etps   are the free etps of the defns, and those required by any direct TLR arity-met calls.
    * The envReq are the ids/subEnvs required from calls to freeVars.
    *)
   { etps   : typar_spec Zset.set;
     envReq : envItem    Zset.set;
     m      : Range.range;
   }
let env0 ccu m =
  {etps   = Zset.empty typar_spec_order;
   envReq = Zset.empty (envItem_order ccu);
   m      = m;
  }

let extendEnv (typars,items) env = {env with
                                    etps   = Zset.addL typars env.etps;
                                    envReq = Zset.addL items  env.envReq}

let envSubEnvs env =
  let select = function ReqSubEnv f -> Some f | ReqVal _ -> None in
  chooseL select (Zset.elements env.envReq)

let envVals env =
  let select = function ReqSubEnv f -> None | ReqVal f -> Some f in
  chooseL select (Zset.elements env.envReq)
   
(*--debug-stuff--*)

let soEnvItem = function
    ReqSubEnv f -> "&" ^ name_of_val f
  | ReqVal    f -> name_of_val f

let soEnv env =
  (showL (commaListL (map typarL (Zset.elements (env.etps))))) ^ "--" ^ 
  (String.concat "," (map soEnvItem (Zset.elements env.envReq)))


(*-------------------------------------------------------------------------
 *INDEX: pass2: collector - state
 *-------------------------------------------------------------------------*)

type generators = val_spec Zset.set
type state =
   (* Collects:
    *   envM          - fclass -> env
    *   fclassM       - f      -> fclass
    *   declist       - fclass list
    *   recShortCallS - the f which are "recursively-called" in arity short instance.
    *
    * When walking expr, at each mutual binding site,
    * push a (generator,env) collector frame on stack.
    * If occurances in body are relevant (for a generator) then it's contribution is logged.
    *
    * recShortCalls to f will require a binding for f in terms of fHat within the fHatBody.
    *)
    { stack         : (fclass * generators * env) list;
      envM          : (fclass,env) Zmap.map;
      fclassM       : (val_spec,fclass) Zmap.map;
      revDeclist    : fclass list;
      recShortCallS : val_spec Zset.set;
    }

let state0 ccu =
  {stack         = [];
   envM          = Zmap.empty (fclass_order ccu);
   fclassM       = Zmap.empty val_spec_order;
   revDeclist    = [];
   recShortCallS = Zset.empty val_spec_order;
  }

let soVSet fs = (showL (commaListL (map vspecL (Zset.elements fs))))

let pushFrame ccu fclass (etps0,generators,m) state =
  if isEmptyFC fclass then state else
    (* PUSH = start collecting for fclass *)
    ( (if verboseTLR then dprintf2 "pushFrame: %s\n - generators = %s\n" (soFC fclass) (soVSet generators));
      {state with
         revDeclist = fclass :: state.revDeclist;
         stack = let env = extendEnv (etps0,[]) (env0 ccu m) in
           (fclass,generators,env)::state.stack;
      })

let saveFrame     fclass state = 
   if isEmptyFC fclass then state else
     (* POP & SAVE = end collecting for fclass and store *)
     ( (if verboseTLR then dprintf1 "saveFrame: %s\n" (soFC fclass));
       match state.stack with
           []                             -> internalError "trl: popFrame has empty stack"
         | (fclass,generators,env)::stack -> (* ASSERT: same fclass *)
                                             {state with
                                              stack      = stack;
                                              envM       = Zmap.add  fclass env   state.envM;
                                              fclassM    = setsM (fclassPairs fclass) state.fclassM;
                                             })

let logReqFrom g items state =
  (* Log requirements for g in the relevant stack frames *)
  (* dprintf1 "logReqFrom: %s\n" (name_of_val g); *)
  let logIntoFrame (fclass,generators,env) =
    let env = if Zset.mem g generators then
                ((* dprintf1 "          : logging for generators=%s\n" (soVSet generators); *)
                 let typars = [] in
                 extendEnv (typars,items) env)
              else env
    in
    fclass,generators,env
  in
  {state with stack = map logIntoFrame state.stack}

let logShortCall g state =
  let frameFor g (fclass,generators,env) = memFC g fclass in
  if exists (frameFor g) state.stack then
    ((if verboseTLR then dprintf1 "shortCall:     rec: %s\n" (name_of_val g));
     (* Have short call to g within it's (mutual) definition(s) *)
     {state with
      recShortCallS = Zset.add g state.recShortCallS})
  else
    ((if verboseTLR then dprintf1 "shortCall: not-rec: %s\n" (name_of_val g));
     state)

let getEnv f state =
  match Zmap.tryfind f state.fclassM with
    Some fclass -> (* env(f) is known, f prior *)
                   ((if verboseTLR then dprintf1 "getEnv: fclass=%s\n" (soFC fclass));
                    let env = forceM' fclass state.envM ("getEnv",soFC) in
                    Some env)
  | None        -> (* env(f) unknown, perhaps in body of it's defn *)
                   None


(*-------------------------------------------------------------------------
 *INDEX: pass2: collector - exprFold intercepts
 *-------------------------------------------------------------------------*)

(* check a named function value applied to sufficient arguments *)
let arityMet vref  wf tys args = 
  (length tys = length(fst(try_dest_forall_typ (type_of_vref vref)))) && (wf <= length args) 

let freesOfBind  b  = free_in_expr (rhs_of_bind b)
let freesOfBinds bs = fold_left (foldOn freesOfBind union_freevars) empty_freevars bs

let exprEnvIntercept ccu (tlrS,arityM) exprF z expr = 
   (* Intercepts selected exprs.
    *   "letrec f1,f2,... = fBody1,fBody2,... in rest" - 
    *   "val v"                                        - free occurance
    *   "app (f,tps,args)"                             - occurance
    *---
    * On intercepted nodes, must exprF fold to collect from subexpressions.
    *)
   let accInstance z (fvref,tps,args) (* f known local *) = 
     let f = deref_val fvref in
     match Zmap.tryfind f arityM with
     | Some wf -> (* f is TLR with arity wf *)
                  if arityMet fvref wf tps args then
                    logReqFrom f [ReqSubEnv f] z                 (* arity-met call to a TLR g *)
                  else
                    let z = logReqFrom f [ReqVal f] z in         (* arity-short instance *)
                    let z = logShortCall f z in                  (* logShortCall - logs recursive short calls *)
                    z
     | None    -> (* f is non-TLR *)
                    logReqFrom f [ReqVal f] z                    (* non-TLR instance *)
   in
   let accBinds m z binds =
     let tlrBs,nonTlrBs = partition (fun b -> Zset.mem (var_of_bind b) tlrS) binds in
     (* for bindings marked TLR, collect implied env *)
     let fclass = mkFClass (map var_of_bind tlrBs) in
     (* what determines env? *)
     let frees      = freesOfBinds tlrBs in
     let etps0      = frees.free_tyvars.free_loctypars   |> Zset.elements |> map deref_local_typar in      (* put in env *)
     let generators = (frees.free_locvals |> Zset.elements |> map deref_local_val)  in      (* occurances contribute to env *)
     let generators = filter (fun g -> not (memFC g fclass)) generators in (* tlrBs not generators for themselves *)
     let generators = listS val_spec_order generators in
     (* collect into env over bodies *)
     let z          = pushFrame ccu fclass (etps0,generators,m) z in
     let z          = fold_left (foldOn rhs_of_bind exprF) z tlrBs in
     let z          = saveFrame     fclass z in
     (* for bindings not marked TRL, collect *)
     let z          = fold_left (foldOn rhs_of_bind exprF) z nonTlrBs in
     z
   in
   let rec recognise context expr = 
     match expr with
     | TExpr_val (v,_,m) -> let z = accInstance z (v,[],[]) in
                            Some z
     | TExpr_app (f,fty,tys,args,m)      -> let f,fty,tys,args,m = destApp (f,fty,tys,args,m) in
                                            (match f with
                                               TExpr_val (f,_,_) ->
                                                      (* YES: APP vspec tps args - log *)
                                                      let z = accInstance z (f,tys,args) in
                                                      let z = fold_left exprF z args in
                                                 Some z
                                            | _ ->
                                                (* NO: app, but function is not val - no log *)
                                                None)
     | TExpr_letrec (binds,body,m,_)       -> let z = accBinds m z binds in
                                              let z = exprF z body in
                                              Some z
     | TExpr_let    (bind,body,m,_)        -> let z = accBinds m z [bind] in
                                              let z = exprF z body in
                                              Some z
     | _                                   -> None (* NO: no intercept *)
   in
   let context = [] in
   recognise context expr


(*-------------------------------------------------------------------------
 *INDEX: pass2: closeEnvETps
 *-------------------------------------------------------------------------*)

let closeEnvETps fclassM envM =
  (* Initially, etps(fclass) = freetps(bodies).
   * For each direct call to a g, a generator for fclass,
   * Required to include the etps(g) in etps(fclass).
   *)
  if verboseTLR then dprintf0 "closeEnvETps------\n";
  let etpsFor envM f =
    let fc  = forceM' f  fclassM ("etpsFor",name_of_val) in
    let env = forceM' fc envM    ("etpsFor",soFC)        in
    env.etps
  in
  let closeStep envM changed fc env =
    let directCallFs   = envSubEnvs env in
    let directCallETps = map (etpsFor envM) directCallFs in
    let etps0 = env.etps in
    let etps  = List.fold_left Zset.union etps0 directCallETps in
    let changed = changed || (not (Zset.equal etps0 etps)) in
    let env   = {env with etps = etps} in
    if verboseTLR then (
      dprintf4 "closeStep: fc=%30s nSubs=%d etps0=%s etps=%s\n" (soFC fc) (length directCallFs) (soTyparSet etps0) (soTyparSet etps);
      List.iter (fun f    -> dprintf1 "closeStep: dcall    f=%s\n" (name_of_val f))           directCallFs;
      List.iter (fun f    -> dprintf1 "closeStep: dcall   fc=%s\n" (soFC (forceM f fclassM))) directCallFs;
      List.iter (fun etps -> dprintf1 "closeStep: dcall etps=%s\n" (soTyparSet etps0)) directCallETps;
    );
    changed,env
  in
  let rec fixpoint envM =
    let changed = false in
    let changed,envM = Zmap.fmap (closeStep envM) changed envM in
    if changed then
      fixpoint envM
    else
      envM
  in
  fixpoint envM


(*-------------------------------------------------------------------------
 *INDEX: pass2: determineTLREnvs
 *-------------------------------------------------------------------------*)

let dumpEnvM envM =
  let dump fc env = dprintf2 "CLASS=%s\n env=%s\n" (soFC fc) (soEnv env) in
  Zmap.iter dump envM

let determineTLREnvs ccu (tlrS,arityM) (expr: typedAssembly) =
   if verboseTLR then dprintf0 "determineTLREnvs------\n";
   let folder = {exprFolder0 with exprIntercept = exprEnvIntercept ccu (tlrS,arityM)} in
   let z = state0 ccu in
   let z = foldAssembly folder z expr in
   (* project *)
   let envM          = z.envM in
   let fclassM       = z.fclassM in
   let declist       = rev z.revDeclist in
   let recShortCallS = z.recShortCallS in
   (* diagnostic dump *)
   (if verboseTLR then dumpEnvM envM);
   (* close the etps under the subEnv reln *)
   let envM    = closeEnvETps fclassM envM in
   (* filter out trivial fclass - with no TLR defns *)
   let envM    = Zmap.remove (mkFClass[]) envM in
   (* restrict declist to those with envM bindings (the non-trivial ones) *)
   let declist = filter (memM envM) declist in
   (* diagnostic dump *)
   if verboseTLR then
     (dumpEnvM envM;
      List.iter (fun fc -> dprintf1 "Declist: %s\n" (soFC fc)) declist;
      Zset.iter (fun f -> dprintf1 "RecShortCall: %s\n" (name_of_val f)) recShortCallS
     );
   envM,fclassM,declist,recShortCallS


(*-------------------------------------------------------------------------
 *INDEX: step3: envPack
 *-------------------------------------------------------------------------*)

type envPack =
    (* Each env is represented by some carrier values, the aenvs.
     * An env packing defines these, and the pack/unpack bindings.
     * The bindings are in terms of the fvs directly.
     *------
     * When defining a new TLR f definition,
     *   the fvs   will become bound by the unpack bindings,
     *   the aenvs will become bound by the new lam, and
     *   the etps  will become bound by the new LAM.
     * For uniqueness of bound ids,
     *   all these ids (typar_spec/val_spec) will need to be freshened up.
     * It is OK to break the uniqueness-of-bound-ids rule during the rw,
     * provided it is fixed up via a copy_expr call on the final expr.
     *)
    { ep_etps   : typar_spec list; (* the actual typars             *)
      ep_aenvs  : val_spec   list; (* the actual env carrier values *)
      ep_pack   : bind list;       (* sequentially define the aenvs in terms of the fvs   *)
      ep_unpack : bind list;       (* sequentially define the fvs   in terms of the aenvs *)
    }


(*-------------------------------------------------------------------------
 *INDEX: step3: flatEnvPacks
 *-------------------------------------------------------------------------*)

exception AbortTLR of Range.range

let flatEnvPacks ccu fclassM topValS declist envM =
   (* A naive packing of environments.
    * Chooses to pass all env values as explicit args (no tupling).
    * Note, tupling would cause an allocation,
    * so, unless arg lists get very long, this flat packing will be preferable.
    * LATER: tuple some env if needed if arglists get too long.
    * GUIDE: maybe especially for letrecs, pack once, use many times.
    *------
    * Given (fclass,env).
    * Have env = ReqVal vj, ReqSubEnv subEnvk -- ranging over j,k
    * Define vals(env) = {vj}|j union vals(subEnvk)|k -- trans closure of vals of env.
    * Define <vi,aenvi> for each vi in vals(env).
    * This is the carrierMap for the env.
    *------
    * etps     = env.etps
    * carriers = aenvi|i
    * pack     = TBIND(aenvi = vi)            for each (aenvi,vi) in carrierMap
    * unpack   = TBIND(vj = aenvFor(vj))      for each vj in reqvals(env).
    *        and TBIND(asubEnvi = aenvFor(v)) for each (asubEnvi,v) in carrierMap(subEnvk) ranging over required subEnvk.
    * where
    *   aenvFor(v) = aenvi where (v,aenvi) in carrierMap.
    *)
   let fclassOf f = forceM' f fclassM ("fclassM",name_of_val) in
   let packEnv carrierMaps (fc:fclass) =
     if verboseTLR then dprintf1 "\ntlr: packEnv fc=%s\n" (soFC fc);
     let env = forceM' fc envM ("packEnv",soFC) in
     (* carrierMaps = (fclass,(v,aenv)map)map *)
     let carrierMapFor f = forceM' (fclassOf f) carrierMaps ("carrierMapFor",soFC) in
     let valsSubEnvFor f = Zmap.keys (carrierMapFor f) in
     (* determine vals(env) - transclosure *)
     let vals = envVals env @ mapConcat valsSubEnvFor (envSubEnvs env) in (* list, with repeats *)
     let vals = noRepeats val_spec_order vals in                          (* noRepeats *)
     (* remove genuinely toplevel, need not close over *)
     let vals = filter (mandatoryTopLevel >> not) vals in
     let vals = filter (Zset.mem_of topValS >> not) vals in
     
     (* Carrier sets cannot include constrained polymorphic values. We can't just take such a value out, so for the moment *)
     (* we'll just abandon TLR altogether and give a warning about this condition. *)
     (match vals |> tryfind is_poly_constrained_val with None -> () | Some v -> raise (AbortTLR (range_of_val v)));
     (* build carrierMap for env *)
     let cmapPairs = list_mapi (fun i v -> (v,mkLocalNameType true env.m (name_of_val v) (type_of_val v))) vals in
     let cmap      = listM val_spec_order cmapPairs in
     let aenvFor     v = forceM' v cmap ("aenvFor",name_of_val) in
     let aenvExprFor v = expr_for_val env.m (aenvFor v) in
     (* build envPack *)
     let etps   = env.etps in
     let aenvs  = Zmap.values cmap in
     let pack   = let mkPack (v,aenv) = TBind(aenv,(expr_for_val env.m v)) in
                  map mkPack cmapPairs
     in
     let unpack = let unpackCarrier (v,aenv) = TBind(set_val_has_no_arity v,expr_for_val env.m aenv) in
                  let unpackSubenv f = let subCMap  = carrierMapFor f     in
                                       let vaenvs   = Zmap.to_list subCMap in
                                       let unpack (subv,subaenv) = TBind(subaenv,aenvExprFor subv) in
                                       map unpack vaenvs
                  in
                  map unpackCarrier (Zmap.to_list cmap) @
                  mapConcat unpackSubenv (envSubEnvs env)
     in
     (* extend carrierMaps *)
     let carrierMaps = Zmap.add fc cmap carrierMaps in
     (* dump *)
     if verboseTLR then (
       dprintf1 "tlr: packEnv envVals =%s\n" (showL (listL vspecL  (envVals env)));
       dprintf1 "tlr: packEnv envSubs =%s\n" (showL (listL vspecL  (envSubEnvs env)));
       dprintf1 "tlr: packEnv vals    =%s\n" (showL (listL vspecL vals));
       dprintf1 "tlr: packEnv aenvs   =%s\n" (showL (listL vspecL aenvs));
       dprintf1 "tlr: packEnv pack    =%s\n" (showL (listL bindL  pack));
       dprintf1 "tlr: packEnv unpack  =%s\n" (showL (listL bindL  unpack))
     );
     (* result *)
     carrierMaps,
     (fc, { ep_etps   = Zset.elements etps;
            ep_aenvs  = aenvs;
            ep_pack   = pack;
            ep_unpack = unpack})
   in
   let carriedMaps = Zmap.empty (fclass_order ccu) in
   let carriedMaps,envPacks = fmap packEnv carriedMaps declist in  (* fmap in dec order *)
   let envPacks = listM (fclass_order ccu) envPacks in
   envPacks


(*-------------------------------------------------------------------------
 *INDEX: step3: chooseEnvPacks
 *-------------------------------------------------------------------------*)

let dumpEnvPackM envPackM =
   let dump fc envPack =
     dprintf1 "envPack: fc     = %s\n" (soFC fc);
     dprintf1 "         etps   = %s\n" (showL (commaListL (map typarL envPack.ep_etps)));
     dprintf1 "         aenvs  = %s\n" (showL (commaListL (map vspecL envPack.ep_aenvs)));
     dprintf1 "         pack   = %s\n" (showL (semiListL (map bindL envPack.ep_pack)));
     dprintf1 "         unpack = %s\n" (showL (semiListL (map bindL envPack.ep_unpack)));
     dprintf0 "\n"
   in
   Zmap.iter dump envPackM

let chooseEnvPackings ccu fclassM topValS  declist envM =
  (* For each fclass, have an env.
   * Required to choose an envPack,
   * e.g. deciding whether to tuple up the environment or not.
   * e.g. deciding whether to use known values for required sub environments.
   *------
   * Scope for optimisating env packing here.
   * For now, pass all environments via arguments since aiming to eliminate allocations.
   * Later, package as tuples if arg lists get too long.
   *)
  if verboseTLR then dprintf0 "chooseEnvPackings------\n";
  let envPackM = flatEnvPacks ccu fclassM topValS  declist envM in
  let envPackM : (fclass,envPack) Zmap.map = envPackM in
  if verboseTLR then dumpEnvPackM envPackM;
  envPackM

(* REVIEW:
   tuple up env under some decision function:
   - letrec want tupled environments more than let.
   - if #args rises, tupling may be preferable.
   - downside is the allocation.
*)   


(*-------------------------------------------------------------------------
 *INDEX: step3: createFHatM
 *-------------------------------------------------------------------------*)

(* arity info where nothing is untupled *)
(* REVIEW: could do better here by preserving names *)
let mkSimpleArityInfo tps n = TopValInfo (length tps,replicate n TopValData.unnamedTopArg,TopValData.unnamedRetVal)

let createFHatM ccu tlrS arityM fclassM envPackM = 
  if verboseTLR then dprintf0 "createFHatM------\n";
  let createFHat f =
    let wf     = forceM' f arityM ("createFHat - wf",(fun v -> showL (vspecL v))) in
    let fc     = forceM' f fclassM ("createFHat - fc",name_of_val) in   
    let envp   = forceM' fc envPackM ("createFHatM - envp",soFC) in
    let name   = name_of_val f (* ^ "_TLR_" ^ string_of_int wf *) in
    let m      = range_of_val f in
    let tau        = type_of_val f in
    let tps,tau    = try_dest_forall_typ tau in
    let argtys,res = strip_fun_typ tau in
    let newTps    = envp.ep_etps @ tps in
    let fHatTy = 
        let newArgtys = map type_of_val envp.ep_aenvs @ argtys in
        mk_lambda_ty newTps newArgtys res in
    let fHatArity = mkSimpleArityInfo newTps (length envp.ep_aenvs + wf) in
    let fHat = mkLocalNameTypeArity (compgen_of_val f) m name fHatTy (Some fHatArity) in
    (if verboseTLR then dprintf2 "new %50s : %s\n" (name_of_val fHat) ((DebugPrint.showType (type_of_val fHat))));
    fHat
  in
  let fs     = Zset.elements tlrS in
  let ffHats = map (fun f -> f,createFHat f) fs in
  let fHatM  = listM val_spec_order ffHats in
  fHatM


(*-------------------------------------------------------------------------
 *INDEX: pass4: rewrite - penv
 *-------------------------------------------------------------------------*)

type penv =
   { ccu           : ccu;
     g             : Env.tcGlobals;
     tlrS          : (val_spec) Zset.set;
     topValS       : (val_spec) Zset.set;
     arityM        : (val_spec,int) Zmap.map;
     fclassM       : (val_spec,fclass) Zmap.map;
     recShortCallS : (val_spec) Zset.set;
     envPackM      : (fclass,envPack) Zmap.map;
     fHatM         : (val_spec,val_spec) Zmap.map;
   }


(*-------------------------------------------------------------------------
 *INDEX: pass4: rwstate (z state)
 *-------------------------------------------------------------------------*)

type isRec   = IsRec | NotRec
type preDec  = isRec * bind list (* where bool=true if letrec *)    
type rwstate =
    (* This state is related to lifting to top-level
     * This is to ensure the TLR constants get initialised once.
     *------
     * Top-level status ends when stepping inside a lambda, where a lambda is:
     *   TExpr_tlambda, TExpr_lambda, TExpr_obj (and tmethods).
     *   [... also, try_catch handlers, and switch targets...]
     *------
     * Top* repr bindings already at top-level do not need moving...
     *   [and should not be, since they may lift over unmoved defns on which they depend].
     * Any TLR repr bindings under lambdas can be filtered out (and collected),
     * giving pre-declarations to insert before the outermost lambda expr.
     *------
     *)
    { rws_mustinline: bool;
      rws_innerLevel : int;        (* counts level of enclosing "lambdas"  *)
      rws_preDecs    : preDec tree (* collected preDecs (fringe is in-order) *)
    }

let rws0 = {rws_mustinline=false;rws_innerLevel=0;rws_preDecs=emptyTR}

(* move in/out of lambdas (or lambda containing construct) *)
let enterInner z = {z with rws_innerLevel = z.rws_innerLevel + 1}
let exitInner  z = {z with rws_innerLevel = z.rws_innerLevel - 1}

let enterMustInline b z f = 
    let orig = z.rws_mustinline in
    let z',x = f (if b then {z with rws_mustinline = true } else z) in 
    {z' with rws_mustinline = orig },x

(* extract PreDecs (iff at top-level) *)
let extractPreDecs z =
    (* If level=0, so at top-level, then pop decs,
     * else keep until get back to a top-level point.
     *)
    if z.rws_innerLevel=0 then
      (* at top-level, extract preDecs *)
      let preDecs = fringeTR z.rws_preDecs in
      {z with rws_preDecs=emptyTR}, preDecs
    else 
      (* not yet top-level, keep decs *)
      z,[]

(* pop and set preDecs  as "preDec tree" *)
let popPreDecs z     = {z with rws_preDecs=emptyTR},z.rws_preDecs
let setPreDecs z pdt = {z with rws_preDecs=pdt}

(* collect Top* repr bindings - if needed... *)
let liftTopBinds isRec penv z binds =
    let isTopBind bind = isSome (chosen_arity_of_bind bind) in 
    let topBinds,otherBinds = partition isTopBind binds in
    let liftTheseBindings =
      !liftTLR &&             (* lifting enabled *)
      not z.rws_mustinline &&   (* can't lift bindings in a mustinline context - they would become private an not inlined *)
      z.rws_innerLevel>0 &&   (* only collect Top* bindings when at inner levels (else will drop them!) *)
      nonNil topBinds            (* only collect topBinds if there are some! *)
    in
    if liftTheseBindings then
      let preDec = isRec,topBinds in                                          (* preDec Top* decs *)
      let z = {z with rws_preDecs = listTR [z.rws_preDecs;unitTR preDec]} in  (* logged at end *)
        z,otherBinds
    else
      z,binds (* not "topBinds @ otherBinds" since that has changed order... *)
      
(* Wrap preDecs (in order) over an expr - use letrec/let as approp *)
let mk_predec  m (isRec,binds) expr = if isRec=IsRec then mk_letrec_binds m binds expr
                                                     else mk_lets_bind    m binds expr
let mk_predecs m preDecs expr = fold_right (mk_predec m) preDecs expr

(* *)
let recPreDecs pdsA pdsB =
    let pds = fringeTR (Nd[pdsA;pdsB]) in
    let decs = concat (map snd pds) in
    unitTR (IsRec,decs)


(*-------------------------------------------------------------------------
 *INDEX: pass4: lowertop - convert_vterm_bind on TopLevel binds
 *-------------------------------------------------------------------------*)

let convert_bind (TBind(v,repr) as bind)  =
    begin match arity_of_val v with 
    | None -> (data_of_val v).val_arity <- Some (infer_arity_of_expr_bind v repr )
    | Some _ -> ()
    end;
    bind

(*-------------------------------------------------------------------------
 *INDEX: pass4: transBind (translate)
 *-------------------------------------------------------------------------*)

let transTLRBinds penv binds = 
    if isNil binds then [],[] else
    let fc   = mkFClass (map var_of_bind binds) in
    let envp = forceM' fc penv.envPackM ("transTLRBinds",soFC) in   
    let fRebinding (TBind(f,b)) =
        let m = range_of_val f in
        let tps,vss,b,rty = dest_top_lambda (b,type_of_val f) in
        let aenvExprs = map (expr_for_val m) envp.ep_aenvs in
        let vsExprs   = map (mk_tupled_vars penv.g m) vss in
        let fHat      = forceM' f penv.fHatM ("fRebinding",name_of_val)  in
        let w         = 0     in
        (* REVIEW: is this mutation really, really necessary? *)
        (* Why are we applying TLR if the thing already has an arity? *)
        let f = set_val_has_no_arity f in
        let fBind = 
           mk_multi_lambda_bind f m tps vss (mk_appl (typed_expr_for_val m fHat,
                                              [map mk_typar_ty (envp.ep_etps @ tps)],
                                              aenvExprs @ vsExprs,
                                              m),
                                             rty) in 
       fBind                                 in
   let fHatNewBinding shortRecBinds (TBind(f,b)) =
       let wf   = forceM' f penv.arityM ("fHatNewBinding - arityM",name_of_val) in
       let fHat = forceM' f penv.fHatM  ("fHatNewBinding - fHatM",name_of_val) in
       let tps,vss,b,rty = dest_top_lambda (b,type_of_val f) in
       let vssTake,vssDrop = splitAt wf vss in
       (* QUERY QUERY Whoa! Where have the type parameters gone?? Why aren't they just like the term parameters?? *)
       let b,rty = mk_multi_lambdas_core (range_of_expr b) vssDrop (b,rty) in
       (* fHat, args *)
       let m = range_of_val fHat in
       let fHat_tps  = envp.ep_etps @ tps in
       let fHat_args = map singletonList envp.ep_aenvs @ vssTake in
       let fHat_body = mk_lets_bind m envp.ep_unpack b         in
       let fHat_body = mk_lets_bind m shortRecBinds  fHat_body in   (* bind "f" if have short recursive calls (somewhere) *)
       (* fHat binding, f rebinding *)
       let wfHat      = length envp.ep_aenvs + wf in 
       let fHatBind   = mk_multi_lambda_bind fHat m fHat_tps fHat_args (fHat_body,rty) in
       fHatBind in
   let rebinds = map fRebinding binds in
   let shortRecBinds = filter (fun b -> Zset.mem (var_of_bind b) penv.recShortCallS) rebinds in
   let newBinds      = map (fHatNewBinding shortRecBinds) binds in
   newBinds,rebinds

let aenvBindings penv fc =
   match Zmap.tryfind fc penv.envPackM with
     None      -> []           (* no env for this mutual binding *)
   | Some envp -> envp.ep_pack (* environment pack bindings *)

let transBinds xisRec penv binds =
   let tlrBs,nonTlrBs = partition (fun b -> Zset.mem (var_of_bind b) penv.tlrS) binds in
   let fclass = mkFClass (map var_of_bind tlrBs) in   
   (* translate each TLR f binding into fHat and f rebind *) 
   let newTlrBinds,tlrRebinds = transTLRBinds penv tlrBs in
   let aenvBinds = aenvBindings penv fclass in
   (* lower nonTlrBs if they are GTL *)
   (* QUERY: we repeat this logic in Lowertop.  Do we really need to do this here? *)
   (* QUERY: yes and no - if we don't, we have an unrealizable term, and many decisions must *)
   (* QUERY: correlate with Lowertop.  *)
   let forceTopBindToHaveArity bind = if Zset.mem (var_of_bind bind) penv.topValS then convert_bind bind else bind in
   let nonTlrBs = map forceTopBindToHaveArity nonTlrBs in
   let tlrRebinds = map forceTopBindToHaveArity tlrRebinds in
   (* assemble into replacement bindings *)
   let bindAs,rebinds = match xisRec with
       IsRec  -> newTlrBinds @ tlrRebinds @ nonTlrBs @ aenvBinds,[]    (* note: aenv last, order matters in letrec! *)
     | NotRec -> aenvBinds @ newTlrBinds, tlrRebinds @ nonTlrBs in (* note: aenv go first, they may be used *)
   bindAs,rebinds


(*-------------------------------------------------------------------------
 *INDEX: pass4: transApp (translate)
 *-------------------------------------------------------------------------*)

let transApp penv (fx,fty,tys,args,m) =
    (* Is it a val app, where the val f is TLR with arity wf? *)
  (* CLEANUP NOTE: should be using a mk_appl to make all applications *)
    let unchanged = if tys=[] && args=[] then fx else TExpr_app (fx,fty,tys,args,m) in
    match fx with
    | TExpr_val (fvref,_,m) when 
            (Zset.mem (deref_val fvref) penv.tlrS) &&
            (let wf = forceM' (deref_val fvref) penv.arityM ("transApp - wf",name_of_val) in
             arityMet fvref wf tys args) ->

               let f = deref_val fvref in
               (* replace by direct call to corresponding fHat (and additional closure args) *)
               let fc   = forceM' f  penv.fclassM ("transApp - fc",name_of_val)in   
               let envp = forceM' fc penv.envPackM ("transApp - envp",soFC) in   
               let fHat = forceM' f  penv.fHatM ("transApp - fHat",name_of_val)in
               let tys  = (map mk_typar_ty envp.ep_etps) @ tys in
               let aenvExprs = map (expr_for_val m) envp.ep_aenvs in   
               let args = aenvExprs @ args in
               mk_appl (typed_expr_for_val m fHat,[tys],args,m) (* change, direct fHat call with closure (etps,aenvs) *)
    | _ -> unchanged                                              (* no change, f is expr *)

(*-------------------------------------------------------------------------
 *INDEX: pass4: pass (over expr)
 *-------------------------------------------------------------------------*)

let wrapPreDecs m pds x =
    (* Must wrapPreDecs around every construct that could do enterInner (which filters TLR decs).
     * i.e. let,letrec (bind may...), ilobj, lambda, tlambda.
     *)
    mk_predecs m pds x

let rec pass_expr (penv:penv) z expr =
    (* At bindings, fixup any TLR bindings.
     * At applications, fixup calls  if they are arity-met instances of TLR.
     * At free vals,    fixup 0-call if it is an arity-met constant.
     * Other cases rewrite structurally.
     *)
    match expr with
     (* Use pass_linear with a rebuild-continuation for some forms to avoid stack overflows on large terms *)
     | TExpr_letrec _ | TExpr_let    _ -> pass_linear penv z expr (fun res -> res)

     (* app - call sites may require z.
      *     - match the app (collapsing reclinks and type instances).
      *     - patch it.
      *)
     | TExpr_app (f,fty,tys,args,m) ->
       (* pass over f,args subexprs *)
       let z,f      = pass_expr penv z f in
       let z,args = fmap (pass_expr penv) z args in
       (* match app, and fixup if needed *)
       let f,fty,tys,args,m = destApp (f,fty,tys,args,m) in
       let expr = transApp penv (f,fty,tys,args,m) in
       z,expr
    | TExpr_val (v,_,m) ->
       (* consider this a trivial app *)
       let fx,fty = expr,type_of_vref v in
       let expr = transApp penv (fx,fty,[],[],m) in
       z,expr

    (* reclink - suppress *)
    | TExpr_link r ->
        pass_expr penv z (!r)

    (* ilobj - has implicit lambda exprs and recursive/base references *)
    | TExpr_obj (_,ty,basev,basecall,overrides,iimpls,m,_) ->
        let z,basecall  = pass_expr penv                            z basecall  in
        let z,overrides = fmap (pass_tmethod penv)                  z overrides in
        let z,iimpls    = fmap (fmap2'2 (fmap (pass_tmethod penv))) z iimpls    in
        let expr = TExpr_obj(new_uniq(),ty,basev,basecall,overrides,iimpls,m,new_cache()) in
        let z,pds = extractPreDecs z in 
        z,wrapPreDecs m pds expr (* if TopLevel, lift preDecs over the ilobj expr *)

    (* lambda, tlambda - explicit lambda terms *)
    | TExpr_lambda(_,basevopt,argvs,body,m,rty,_) ->
        let z = enterInner z in
        let z,body = pass_expr penv z body in
        let z = exitInner z in
        let z,pds = extractPreDecs z in
        z,wrapPreDecs m pds (mk_basev_multi_lambda m basevopt argvs (body,rty))
    | TExpr_tlambda(_,argtyvs,body,m,rty,_) ->
        let z = enterInner z in
        let z,body = pass_expr penv z body in
        let z = exitInner z in
        let z,pds = extractPreDecs z in 
        z,wrapPreDecs m pds (mk_tlambda m argtyvs (body,rty))

    (* Lifting TLR out over constructs,
     * should lift minimally to ensure the defn is not lifted up and over defns on which it depends.
     *------
     * Lifting over for, while, try_catch, try_finally?, ilobj's (tmethods), switches (targets).
     *)
    | TExpr_match(exprm,dtree,targets,m,ty,_) ->
        let targets = Array.to_list targets in 
        let z,dtree   = pass_dtree penv z dtree in
        let z,targets = fmap (pass_target penv) z targets in
        (* pass_target wraps enterInner/exitInnter, so need to collect any top decs *)  
        let z,pds = extractPreDecs z in 
        z,wrapPreDecs m pds (mk_and_optimize_match exprm m ty dtree targets)

    (* all others - below - rewrite structurally - so boiler plate code after this point... *)
    | TExpr_const _ -> z,expr (* constant wrt val_spec *)
    | TExpr_quote (raw,a,m,ty) -> 
        z,TExpr_quote(raw,a,m,ty)  (* Do not collect at higher levels *)
    | TExpr_op (c,tyargs,args,m) -> 
        let z,args = fmap (pass_expr penv) z args in
        z,TExpr_op(c,tyargs,args,m)
    | TExpr_seq (e1,e2,dir,m) -> 
        let z,e1 = pass_expr penv z e1 in      
        let z,e2 = pass_expr penv z e2 in      
        z,TExpr_seq(e1,e2,dir,m)
    | TExpr_static_optimization (constraints,e2,e3,m) ->
        let z,e2 = pass_expr penv z e2 in      
        let z,e3 = pass_expr penv z e3 in      
        z,TExpr_static_optimization(constraints,e2,e3,m)
    | TExpr_tchoose (_,_,m) -> error(Error("unexpected TExpr_tchoose",m))
    | TExpr_hole(m,_) -> error(Error("unexpected TExpr_hole",m))

(* walk over linear structured terms in tail-recursive loop, using a continuation *)
(* to represent the rebuild-the-term stack *)
and pass_linear penv z expr contf =
    match expr with 
     (* letrec - pass_recbinds does the work *)
     | TExpr_letrec (binds,e,m,_) ->
       let z = enterInner z in
       (* For letrec, preDecs from RHS must mutually recurse with those from the bindings *)
       let z,pdsPrior    = popPreDecs z in
       let z,binds       = fmap (pass_bind_rhs penv) z binds in
       let z,pdsRhs      = popPreDecs z in
       let binds,rebinds = transBinds   IsRec penv binds in
       let z,binds       = liftTopBinds IsRec penv z   binds in  (* factor Top* repr binds *)
       let z,rebinds     = liftTopBinds IsRec penv z rebinds in
       let z,pdsBind     = popPreDecs z in
       let z             = setPreDecs z (listTR [pdsPrior;recPreDecs pdsBind pdsRhs]) in
       let z = exitInner z in
       let z,pds = extractPreDecs z in
       pass_linear penv z e (contf << (fun (z,e) -> 
           let e = mk_lets_bind m rebinds e in
           z,wrapPreDecs m pds (TExpr_letrec (binds,e,m,new_cache()))))
     (* let - can consider the mu-let bindings as mu-letrec bindings - so like as above *)
     | TExpr_let    (bind,e,m,_) ->
       (* For let, preDecs from RHS go before those of bindings, which is collection order *)
       let z,bind       = pass_bind_rhs penv z bind in
       let binds,rebinds = transBinds   NotRec penv [bind] in
       let z,binds       = liftTopBinds NotRec penv z   binds in  (* factor Top* repr binds *)
       let z,rebinds     = liftTopBinds NotRec penv z rebinds in
       (* any lifted PreDecs from binding, if so wrap them... *)
       let z,pds = extractPreDecs z in
       pass_linear penv z e (contf << (fun (z,e) -> 
           let e = mk_lets_bind m rebinds e in
           z,wrapPreDecs m pds (mk_lets_bind m binds e)))
    | _ -> contf (pass_expr penv z expr)
  
and pass_tmethod penv z (TMethod(slotsig,tps,vs,e,m)) =
    let z = enterInner z in     
    let z,e = pass_expr penv z e in
    let z = exitInner z in      
    z,TMethod(slotsig,tps,vs,e,m)

and pass_bind_rhs penv z (TBind(v,e)) = 
    let mustInline = mustinline(inlineFlag_of_val v) in
    let z,e = enterMustInline mustInline z (fun z -> pass_expr penv z e) in
    z,TBind (v,e)

and pass_dtree penv z x =
   match x with 
   | TDSuccess (es,n) -> 
       let z,es = fmap (pass_expr penv) z es in 
       z,TDSuccess(es,n)
   | TDBind (bind,rest) -> 
       let z,bind       = pass_bind_rhs penv z bind in
       let z,rest = pass_dtree penv z rest in 
       z,TDBind(bind,rest)
   | TDSwitch (e,cases,dflt,m) ->
       let z,e = pass_expr penv z e in
       let pass_dtree_case penv z (TCase (discrim,dtree)) =
         let z,dtree = pass_dtree penv z dtree in
         z,TCase(discrim,dtree)
       in
       let z,cases = fmap       (pass_dtree_case penv) z cases in 
       let z,dflt  = fmapOption (pass_dtree penv)      z dflt in
       z,TDSwitch (e,cases,dflt,m)

and pass_target penv z (TTarget(vs,e)) =
    let z = enterInner z in     
    let z,e = pass_expr penv z e in
    let z = exitInner z in    
    z,TTarget(vs,e)

and pass_veqn penv z bind = let z,b = (pass_bind_rhs penv) z bind in z,b
and pass_veqns penv z binds = fmap (pass_veqn penv) z  binds
and pass_mexpr penv z x = 
    match x with  
    | TMTyped(mty,def,m) ->  
        let z,def = pass_mdef penv z def in 
        z,TMTyped(mty,def,m)
    
and pass_mdefs penv z x = fmap (pass_mdef penv) z x
and pass_mdef penv z x = 
    match x with 
    | TMDefRec(tycons,binds,m) -> 
        let z,binds = pass_veqns penv z binds in 
        z,TMDefRec(tycons,binds,m)
    | TMDefLet(bind,m)            -> 
        let z,bind = pass_veqn penv z bind in 
        z,TMDefLet(bind,m)
    | TMDefs(defs)   -> 
        let z,defs = pass_mdefs penv z defs in 
        z,TMDefs(defs)
    | TMAbstract(mexpr) -> 
        let z,mexpr = pass_mexpr penv z mexpr in 
        z,TMAbstract(mexpr)
    | TMDefModul(TMBind(nm, rhs))   -> 
        let z,rhs = pass_mdef penv z rhs in 
        z,TMDefModul(TMBind(nm,rhs))

let pass_assembly penv z (TAssembly(mvs)) = 
    let z,mvs = fmap (fmapTImplFile (pass_mexpr penv)) z mvs  in
    TAssembly(mvs)

(*-------------------------------------------------------------------------
 *INDEX: pass5: copy_expr
 *-------------------------------------------------------------------------*)

let recreateUniqueBounds g expr = copy_assembly g false expr

(*-------------------------------------------------------------------------
 *INDEX: entry point
 *-------------------------------------------------------------------------*)

let makeTLRDecisions ccu g ((expr : typedAssembly) as expr0) : typedAssembly =
   try
      (* pass1: choose the f to be TLR with arity(f) *)
      let tlrS,topValS, arityM = determineTLRAndArities ccu g expr in
      (* pass2: determine the typar/freevar closures, f->fclass and fclass declist *)
      let envM,fclassM,declist,recShortCallS = determineTLREnvs ccu (tlrS,arityM) expr in
      (* pass3: *)
      let envPackM = chooseEnvPackings ccu fclassM topValS  declist envM in
      let fHatM    = createFHatM ccu tlrS arityM fclassM envPackM in
      (* pass4: rewrite *) 
      if verboseTLR then dprintf0 "pass_expr(rw)------\n";
      let penv = {ccu=ccu; g=g; tlrS=tlrS; topValS=topValS; arityM=arityM; fclassM=fclassM; recShortCallS=recShortCallS; envPackM=envPackM; fHatM=fHatM} in
      let z = rws0 in
      let expr = pass_assembly penv z expr in
      (* pass5: copy_expr to restore "each bound is unique" property *)
      if verboseTLR then dprintf0 "copy_expr------\n";
      let expr = recreateUniqueBounds g expr in (* aka, copy_expr *)
      if verboseTLR then dprintf0 "TLR-done------\n";
      (* Summary:
       *   GTL = genuine top-level
       *   TLR = TopLevelRep = identified by this pass
       *   Note, some GTL are skipped until sort out the initial env...
       *)
    (*
      if verboseTLR then dprintf3 "note: tlr = %d inner-TLR + %d GenuineTopLevel-TLR + %d GenuineTopLevel skipped TLR (public)\n"
        (lengthS (Zset.diff  tlrS topValS))
        (lengthS (Zset.inter topValS tlrS))
        (lengthS (Zset.diff  topValS tlrS))
      ;
    *)
      (* DONE *)
      expr
   with AbortTLR m -> 
       warning(Error("Note: Lambda-lifting optimizations have not been applied because of the use of this local constrained generic function as a first class value. Adding type constraints may resolve this condition",m));
       expr
