(* (c) Microsoft Corporation. All rights reserved *)

(*-------------------------------------------------------------------------
!* Name environment and name resolution 
 *------------------------------------------------------------------------- *)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Il = Microsoft.Research.AbstractIL.IL 
F#*) 
open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Env
open Il (* Abstract IL  *)
open Lib
open Layout
open Outcome
open Infos
open Printf

(*-------------------------------------------------------------------------
!* Helpers for unionconstrs and recdfields
 *------------------------------------------------------------------------- *)

let ucrefs_in_tycon modref tycon = 
    tycon |> uconstrs_of_tycon |> map (ucref_of_uconstr (mk_tcref_in_modref modref tycon)) 

let ucrefs_in_modref modref = 
   fold_right (ucrefs_in_tycon modref >> (@)) (Namemap.range (mtyp_of_modref modref).mtyp_tycons) []

let is_uconstr_in_tycon id tycon = isSome (uconstr_of_tycon_by_name tycon id.idText) 
let is_rfield_in_tycon id tycon = isSome (any_rfield_of_tycon_by_name tycon id.idText)
let is_rfield_in_uconstr id uconstr = isSome (rfield_of_uconstr_by_name uconstr id.idText)

let tryfind_uconstr_in_modref modref id = 
    Namemap.tryfind_in_range (is_uconstr_in_tycon id) (mtyp_of_modref modref).mtyp_tycons

let tryfind_rfield_in_mtyp mty id = 
    Namemap.tryfind_in_range (is_rfield_in_tycon id) mty.mtyp_tycons

let aprefs_in_vref vref = 
   match apinfo_of_vref vref with
   | Some (APInfo(_,nms) as apinfo) -> list_mapi (fun i _ -> APElemRef(apinfo,vref, i)) nms
   | None -> [] 
let aprefs_in_vspec modref vspec = aprefs_in_vref (mk_vref_in_modref modref vspec)


let aprefs_of_modref modref : apelem_ref namemap = 
    let mtyp = mtyp_of_modref modref in 
    cacheOptRef mtyp.mtyp_apref_cache (fun () ->
       let aprefs = fold_right (aprefs_in_vspec modref >> (@)) (Namemap.range mtyp.mtyp_vals) [] in
       List.fold_right (fun apref acc -> Map.add (name_of_apref apref) apref acc) aprefs Map.empty)


(*-------------------------------------------------------------------------
!* Handle mangled .NET generic type names
 *------------------------------------------------------------------------- *)
 
let mangledGenericTypeNameSym = '`'
let isMangledGenericName n = 
    String.contains n mangledGenericTypeNameSym &&
    (* check what comes after the symbol is a number *)
    let m = String.rindex n mangledGenericTypeNameSym in 
    let res = ref (m < String.length n - 1) in 
    for i = m + 1 to String.length n - 1 do
      res := !res && String.get n i >= '0' && String.get n i <= '9';
    done;
    !res

let demangleGenericTypeName n = 
    if isMangledGenericName n then 
        let pos = String.rindex n mangledGenericTypeNameSym in 
        let res = String.sub n 0 pos in 
        let num = String.sub n (pos+1) (String.length n - pos - 1) in 
        res, int_of_string num
    else n,0

(*-------------------------------------------------------------------------
!* Managed cached type name lookup tables
 *------------------------------------------------------------------------- *)
 
let add_tyconsByDemangledNameAndArity nmf typarsf x map = 
    let nm,_ = demangleGenericTypeName (nmf x) in 
    Map.add (nm,length (typarsf x)) x map
            
let add_tyconsByAccessNames nmf x map = 
    let nm = nmf x in 
    let dnm,_ = demangleGenericTypeName nm in 
    let res = Namemap.add_multi nm x map in
    if nm = dnm then res else Namemap.add_multi dnm x res

let add_exconsByDemangledName tycon acc = 
    if tycon_is_exnc tycon 
    then Namemap.add (demangled_name_of_exnc tycon) tycon acc
    else acc
  
let tyconsByDemangledNameAndArity_of_modtyp mtyp : ((string * int), tycon_spec) Map.t = 
    cacheOptRef mtyp.mtyp_tyconsByDemangledNameAndArity_cache (fun () -> 
       Namemap.fold_range (add_tyconsByDemangledNameAndArity name_of_tycon typars_of_tycon) mtyp.mtyp_tycons  Map.empty)

let tyconsByAccessNames_of_modtyp mtyp : tycon_spec namemmap = 
    cacheOptRef mtyp.mtyp_tyconsByAccessNames_cache (fun () -> 
       Namemap.fold_range (add_tyconsByAccessNames name_of_tycon) mtyp.mtyp_tycons  Map.empty)

let mtyp_exconsByDemangledName_of_modtyp mtyp : tycon_spec namemap = 
    cacheOptRef mtyp.mtyp_exconsByDemangledName_cache (fun () -> 
       Namemap.fold_range add_exconsByDemangledName mtyp.mtyp_tycons  Map.empty)
  
(*---------------------------------------------------------------------------
 * 
 *------------------------------------------------------------------------- *)

(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type namedItemInEnv = 
  (* These exist in the "eItems" map in the type environment. *)
  | Item_val of  val_ref
  | Item_ucref of unionconstr_ref
  | Item_apres of apinfo * int 
  | Item_apelem of apelem_ref 
  | Item_ecref of tycon_ref 
  | Item_recdfield of recdfield_info

  (* The following are never in the items table but are valid results of binding *)
  (* an identitifer in different circumstances. *)
  | Item_newdef of ident
  | Item_il_field of il_field_info
  | Item_il_event of il_event_info
  | Item_property of string * prop_info list
  | Item_meth_group of string * meth_info list
  | Item_ctor_group of string * meth_info list
  | Item_fake_intf_ctor of Tast.typ
  | Item_delegate_ctor of Tast.typ
  | Item_typs of string * Tast.typ list
  | Item_modrefs of Tast.modul_ref list

(*---------------------------------------------------------------------------
 * 
 *------------------------------------------------------------------------- *)

(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type nameResEnv =
    { (* Display environment information for output *)
      edenv: displayEnv;  

      (* Values and Data Tags available by unqualified name *)
      eItems: namedItemInEnv namemap;

      (* Data Tags and Active Pattern Tags available by unqualified name *)
      ePatItems: namedItemInEnv namemap;

      (* Modules accessible via "." notation. Note this is a multi-map. *)
      (* Adding a module abbreviation adds it a local entry to this map. *)
      (* Likewise adding a ccu or opening a path adds entries to this map. *)
      (* The boolean flag is means the namespace or module entry shouldn't 'really' be in the *)
      (* map, and if it is everr used to resolve a name then we give a warning. *)
      (* This is used to give warnings on unqualified namespace accesses, e.g. *)
      (*    open System *)
      (*    open Collections                            <--- gives a warning *)
      (*    let v = new Collections.Generic.List<int>() <--- gives a warning *)
      
      eModulesAndNamespaces:  (Tast.modul_ref list) namemap; 
      
      (* Fully qualified modules and namespaces. 'open' does not change this. *)
      eFullyQualifiedModulesAndNamespaces:  (Tast.modul_ref list) namemap; 
      
      (* Field labels in scope.  Field labels are those where type are inferred *)
      (* by label rather than by known type annotation. *)
      (* Bools indicate if from a record, where no warning is given on indeterminate lookup *)
      eFieldLabels: (Tast.recdfield_ref * bool) namemmap; 

      (* Tycons indexed by the various names that may be used to access them, e.g. *)
      (*     "List" --> multiple tycon_refs for the various tycons accessible by this name. *)
      (*     "List`1" --> tycon_ref *)
      eTyconsByAccessNames: tycon_ref namemmap; 

      (* Tycons available by unqualified, demangled names (i.e. (List,1) --> tycon_ref) *)
      eTyconsByDemangledNameAndArity: ((string*int),tycon_ref) Map.t; 

      (* Extension members by type and name *)
      eExtensionMembers: val_ref tcref_mmap; 

      (* Typars (always available by unqualified names). Further typars can be *)
      (* in the tpenv, a structure folded through each top-level definition. *)
      eTypars: local_typar_ref namemap; 

    } 

let empty_nameResEnv g =
    { edenv=empty_denv g;
      eModulesAndNamespaces=Map.empty;
      eFullyQualifiedModulesAndNamespaces = Map.empty;
      eFieldLabels=Map.empty;
      eItems=Map.empty;
      ePatItems=Map.empty;
      eTyconsByAccessNames=Map.empty;
      eTyconsByDemangledNameAndArity=Map.empty;
      eExtensionMembers=tcref_map_empty();      
      eTypars=Map.empty; }

let denv_of_nenv tenv = tenv.edenv
let items_of_nenv tenv = tenv.eItems


(*-------------------------------------------------------------------------
!* namedItemInEnv functions
 *------------------------------------------------------------------------- *)

let name_of_item d = 
    try
        match d with
        | Item_val v -> display_name_of_val (deref_val v)
        | Item_apelem apref -> name_of_apref apref
        | Item_ucref(ucr) -> decompileOpName (ucr |> uconstr_of_ucref |> name_of_uconstr)
        | Item_ecref(ecr) -> demangled_name_of_exnc (deref_exnc ecr)
        | Item_recdfield(rfinfo) -> decompileOpName (rfield_of_rfinfo rfinfo).rfield_id.idText
        | Item_newdef(id) -> id.idText
        | Item_il_field(finfo) -> name_of_il_finfo finfo
        | Item_il_event(einfo) -> name_of_il_einfo einfo
        | Item_property(nm,pinfos) -> nm
        | Item_meth_group(nm,_) -> nm
        | Item_ctor_group(nm,_) -> demangle_dotnet_generic_overloading nm
        | Item_fake_intf_ctor typ 
        | Item_delegate_ctor typ -> demangle_dotnet_generic_overloading (name_of_tcref (tcref_of_stripped_typ typ))
        | Item_typs(nm,tcref) -> demangle_dotnet_generic_overloading nm
        | Item_modrefs(modref :: _) -> demangle_dotnet_generic_overloading (name_of_modref modref)
        | _ ->  ""
    with e -> "-- inaccessible item"


let add_vref_to_items vref eItems =
    (* Object model members are not added to the name resolution environment *)
    (* because they use compiler-internal mangled names. *)
    (* *)
    (* tcaug_adhoc cleanup: we would add vrefs to eAugmentation table here? *)
    (match member_info_of_vref vref with 
     | Some _ -> eItems
     | None -> Namemap.add (name_of_vref vref) (Item_val vref) eItems)

let add_vref_to_extension_members vref eExtensionMembers =
    (match member_info_of_vref vref with 
     | Some vspr -> 
         if (isext_of_vref vref) 
         then tcref_mmap_add (apparent_parent_of_vspr_vref vref) vref eExtensionMembers 
         else eExtensionMembers
     | _ -> eExtensionMembers)

let add_apref_to_ePatItems apref tab = Namemap.add (name_of_apref apref) (Item_apelem apref) tab


(* Entrypoint used to add some extra items to the environment for Visual Studio, e.g. static members *)
let add_fake_named_vref_to_nenv nm vref nenv =
    {nenv with eItems= Namemap.add nm (Item_val vref) nenv.eItems }

let add_vref_to_nenv vref nenv =
    {nenv with eItems= add_vref_to_items vref nenv.eItems;
               eExtensionMembers = add_vref_to_extension_members vref nenv.eExtensionMembers;
               ePatItems = 
                   (let ePatItems = List.fold_right add_apref_to_ePatItems (aprefs_in_vref vref) nenv.ePatItems in

                    (* Add literal constants to the environment available for resolving items in patterns *)
                    let ePatItems = 
                        match literal_const_of_val (deref_val vref) with 
                        | None -> ePatItems 
                        | Some _ -> Namemap.add (name_of_vref vref) (Item_val vref) ePatItems in

                    ePatItems) }

let add_apctxt_to_nenv g apinfo nenv =
    let nms = names_of_apinfo apinfo in 
    let apresl = nms |> list_mapi (fun j nm -> nm, j) in 
    { nenv with  eItems= fold_right (fun (nm,j) acc -> Map.add nm (Item_apres (apinfo,j)) acc) apresl nenv.eItems; } 

let add_tcref_to_nenv nenv tcref = 
    let add_rfref_to_tab rfref tab = Namemap.add_multi (name_of_rfref rfref) (rfref,is_recd_tcref (tcref_of_rfref rfref)) tab in
    let add_ucref_to_tab tab ucref  = Map.add (name_of_ucref ucref) (Item_ucref ucref) tab in
    let add_ucrefs_to_tab tab ucrefs = fold_left add_ucref_to_tab tab ucrefs in
    let ucrefs = List.map (ucref_of_uconstr tcref) (uconstrs_of_tcref tcref) in 
    { nenv with 
        eFieldLabels= 
            tcref 
            |> rfields_array_of_tcref 
            |> Array.fold_left (fun acc f -> 
                   if (static_of_rfield f || secret_of_rfield f) then acc 
                   else add_rfref_to_tab (rfref_of_rfield tcref f) acc ) nenv.eFieldLabels;
        eItems    = add_ucrefs_to_tab nenv.eItems    ucrefs ;
        ePatItems = add_ucrefs_to_tab nenv.ePatItems ucrefs;
        eTyconsByDemangledNameAndArity= add_tyconsByDemangledNameAndArity name_of_tcref typars_of_tcref tcref nenv.eTyconsByDemangledNameAndArity; 
        eTyconsByAccessNames= add_tyconsByAccessNames name_of_tcref tcref nenv.eTyconsByAccessNames 
    } 
    
let add_tcrefs_to_nenv tcrefs nenv = 
    fold_left add_tcref_to_nenv nenv tcrefs

let add_ecref_to_nenv ecref nenv = 
    let add_ecref_to_tab ecref tab = 
        Namemap.add (demangled_name_of_ecref ecref) (Item_ecref ecref) tab in
    {nenv with 
       eItems=add_ecref_to_tab ecref nenv.eItems;
       ePatItems = add_ecref_to_tab ecref nenv.ePatItems }

let add_modul_abbrev_to_nenv id modrefs nenv = 
    {nenv with
       eModulesAndNamespaces=
         let add old nw = nw @ old in 
         Namemap.layer_additive add (Map.add id.idText modrefs Map.empty) nenv.eModulesAndNamespaces }

let add_modrefs_to_nenv topRooted modrefs nenv =
    let add_modrefs_to_table modrefs tab = 
         let add old nw = 
             if verbose then  dprintf2 "add_modrefs_to_table, nm = %s, #old = %d\n" (name_of_modref nw) (length old);
             nw :: old in 
         Namemap.layer_additive add modrefs tab in
    {nenv with
       eModulesAndNamespaces= add_modrefs_to_table modrefs nenv.eModulesAndNamespaces;
       eFullyQualifiedModulesAndNamespaces =
         (if topRooted  
          then add_modrefs_to_table modrefs nenv.eFullyQualifiedModulesAndNamespaces
          else nenv.eFullyQualifiedModulesAndNamespaces) } 

let add_modref_to_nenv topRooted modref nenv =  
    add_modrefs_to_nenv topRooted (Map.add (name_of_modref modref) modref Map.empty) nenv

(*-------------------------------------------------------------------------
!* Open a structure or an IL namespace 
 *------------------------------------------------------------------------- *)

let modrefs_of_mtyp modref mty = 
  submoduls_of_mtyp mty |> Namemap.map (mk_modref_in_modref modref)

let add_modref_contents_to_nenv modref nenv = 
    let mty = mtyp_of_modref modref in 
    let tycons = mty.mtyp_tycons |> Namemap.range in 
    let exncs = tycons |> List.filter tycon_is_exnc in 
    let nenv = { nenv with edenv= denv_add_open_modref modref nenv.edenv } in
    let nenv = List.fold_right (mk_ecref_in_modref modref >> add_ecref_to_nenv) exncs nenv in
    let nenv = Namemap.fold_range (mk_vref_in_modref modref >> add_vref_to_nenv) mty.mtyp_vals nenv in
    let nenv = add_tcrefs_to_nenv (tycons |> list_map (mk_tcref_in_modref modref)) nenv in
    let nenv = add_modrefs_to_nenv false (modrefs_of_mtyp modref mty) nenv in 
    nenv
  
type checkForDuplicateTyparFlag = 
    | CheckForDuplicateTypars 
    | NoCheckForDuplicateTypars

let add_declared_typars_to_nenv check typars nenv = 
    let typarmap = 
      fold_right 
        (fun tp sofar -> 
          begin match check with 
          | CheckForDuplicateTypars -> 
              if Map.mem (name_of_typar tp) sofar then errorR (Duplicate("type parameter",id_of_typar tp))
          | NoCheckForDuplicateTypars -> 
              ()
          end;
          Map.add (name_of_typar tp) tp sofar) typars Map.empty  in 
    {nenv with eTypars=Namemap.layer typarmap nenv.eTypars }

(*--------------------------------------------------------------------------
!* Lookup tables
 *-------------------------------------------------------------------------- *)

let tryname s t id = 
    try Map.find id.idText t 
    with Not_found -> error (UndefinedName(0,s,id,Namemap.domainL t))

(*-------------------------------------------------------------------------
!* freshen_tycon and ginstf.  ginstf is a function to help us create the
 * type parameters by copying them from type parameter specifications read
 * from IL code.  
 *
 * When looking up items in generic types we create a fresh instantiation 
 * of the type, i.e. instantiate the type with inference variables. 
 * This means the item is returned ready for use by the type inference engine 
 * without further freshening. However it does mean we end up plumbing 'ginstf' 
 * around a bit more than we would like to, which is a bit annoying. 
 *------------------------------------------------------------------------- *)

let freshen_tycon m ginstf tcref = 
    let tinst = ginstf m (typars_of_tycon (deref_tycon tcref)) in 
    TType_app(tcref,tinst)

(*-------------------------------------------------------------------------
!* Resolve module paths, value, field etc. lookups.  Doing this involves
 * searching through many possibilities and disambiguating.  Hence first
 * define some ways of combining multiple results and for carrying
 * error information.  Errors are generally undefined names and are
 * reported by returning the error that occurs at greatest depth in the
 * sequence of identifiers. 
 *------------------------------------------------------------------------- *)

(* Accumulate a set of possible results. *)
(* If neither operations succeed, return an approximate error. *)
(* If one succeeds, return that one. *)
(* Prefer the error associated with the first argument. *)
let one_result res = 
    match res with 
    | Success x -> Success [x]
    | Raze e -> Raze e

let add_results res1 res2 =
    match res1, res2 with 
    | Success [],_ -> res2
    | _,Success [] -> res1
    | Success x,Success l -> Success (x @ l)
    | Raze _,Success l -> Success l
    | Success x,Raze _ -> Success x
     (* This prefers error messages coming from deeper failing long identifier paths *)
    | Raze (UndefinedName(n1,_,_,_) as e1),Raze (UndefinedName(n2,_,_,_) as e2) -> 
        if n1 < n2 then Raze e2 else Raze e1
    (* Prefer more concrete errors about things being undefined *)
    | Raze (UndefinedName(n1,_,_,_) as e1),Raze (Error _) -> Raze e1
    | Raze (Error _),Raze (UndefinedName(n1,_,_,_) as e2) -> Raze e2
    | Raze e1,Raze _ -> Raze e1

let (+++) x y = add_results x y
let no_results_or_useful_errors = Success []

let rec map_some m f = function
    | [] -> no_results_or_useful_errors
    | [h] -> one_result (f h)
    | h :: t -> add_results (one_result (f h)) (map_some m f t)

let at_most_one_result m res = 
    match res with 
    | Raze err -> raze err
    | Success [] -> raze (Error("invalid module/expression/type",m))
    | Success [res] -> success res
    | Success (res :: _) -> success res (* raze (Error("this module/expression/type is ambiguous",m))*)

(*-------------------------------------------------------------------------
!* Resolve (possibly mangled) type names 
 *------------------------------------------------------------------------- *)
 
(* These are for qualified lookups where the number of generic arguments is known *)
(* from context, e.g. Module.Type<args>.  In theory the full names suh as ``List`1`` can *)
(* be used to qualify access if needed *)
let lookupTypeNameInModulHaveArity nm ntyargs mty = 
    if verbose then dprintf2 "lookupTypeNameInModulHaveArity: nm = '%s', ntyargs = %d\n" nm ntyargs;
    if isMangledGenericName nm then 
      Map.tryfind nm mty.mtyp_tycons
    else 
        let mangledLookup =
            if ntyargs = 0 then None 
            else
              let gn = (nm^"`"^string_of_int ntyargs) in
              if verbose then dprintf2 "lookupTypeNameInModulHaveArity: gn = '%s', ntyargs = %d\n" gn ntyargs;
              Map.tryfind gn mty.mtyp_tycons in
        mangledLookup 
        +?? (fun () -> Map.tryfind nm mty.mtyp_tycons)

(* These are for unqualified lookups where the number of generic arguments is known *)
(* from context, e.g. List<arg>.  Rebindings due to 'open' may have rebound identifiers. *)
let lookupTypeNameInEnvHaveArity nm ntyargs nenv = 
    if isMangledGenericName nm then 
      Map.tryfind (demangleGenericTypeName nm) nenv.eTyconsByDemangledNameAndArity
      +?? (fun () -> Map.tryfind nm nenv.eTyconsByAccessNames |> Option.map List.hd)
    else 
      Map.tryfind (nm,ntyargs) nenv.eTyconsByDemangledNameAndArity
      +?? (fun () -> Map.tryfind nm nenv.eTyconsByAccessNames |> Option.map List.hd)

(* These are for unqualified lookups where the number of generic arguments is NOT known *)
(* from context. This is used in five places: *)
(*     -  static member lookups, e.g. MyType.StaticMember(3) *)
(*     -                         e.g. MyModule.MyType.StaticMember(3) *)
(*     -  type-qualified field names, e.g. { RecordType.field = 3 } *)
(*     -  type-qualified constructor names, e.g. match x with UnionType.A -> 3 *)
(*     -  identifiers to constructors for better error messages, e.g. 'String(3)' after 'open System' *)
(*     -  the special single-constructor rule in tc_tycon_cores *)
(* *)
(* Because of the potential ambiguity multiple results can be returned. *)
(* Explicit type annotations can be added where needed to specify the generic arity. *)
(*  *)
(* In theory the full names such as ``RecordType`1`` can *)
(* also be used to qualify access if needed, though this is almost never needed.  *)

let lookupTypeNameNoArity nm byDemangledNameAndArity byAccessNames = 
    if isMangledGenericName nm then 
      match Map.tryfind (demangleGenericTypeName nm) byDemangledNameAndArity with 
      | Some res -> [res]
      | None -> 
          match Map.tryfind nm byAccessNames with
          | Some res -> res
          | None -> []
    else 
      Namemap.find_multi nm byAccessNames

let lookupTypeNameInEnvNoArity nm nenv = lookupTypeNameNoArity nm nenv.eTyconsByDemangledNameAndArity nenv.eTyconsByAccessNames 
let lookupTypeNameInModulNoArity nm mtyp = lookupTypeNameNoArity nm (tyconsByDemangledNameAndArity_of_modtyp mtyp) (tyconsByAccessNames_of_modtyp mtyp) 

type typeNameInExprOrPatFlag = ResolveTypeNamesToCtors | ResolveTypeNamesToTypeRefs
type typeNameResInfo = typeNameInExprOrPatFlag * int option
let defaultTypeNameResInfo = (ResolveTypeNamesToCtors,None) 


let lookupTypeNameInEnvMaybeHaveArity nm ((_,ntyargsOpt):typeNameResInfo)  nenv = 
    match ntyargsOpt with 
    | None -> lookupTypeNameInEnvNoArity nm nenv
    | Some ntyargs -> lookupTypeNameInEnvHaveArity nm ntyargs nenv |> Option.to_list
let lookupTypeNameInModulMaybeHaveArity nm ((_,ntyargsOpt):typeNameResInfo) mtyp = 
    if verbose then dprintf1 "lookupTypeNameInModulMaybeHaveArity: nm = '%s'\n" nm ;
    match ntyargsOpt with 
    | None -> lookupTypeNameInModulNoArity nm mtyp
    | Some ntyargs -> lookupTypeNameInModulHaveArity nm ntyargs mtyp |> Option.to_list

let nested_typs_of_typ (optFilter,typeNameResInfo) ginstf g amap m typ =
    fold_primary_hierarchy_of_typ 
        (fun typ acc -> 
            let typs = 
                if is_stripped_tyapp_typ typ then 
                    let tcref,tinst = dest_stripped_tyapp_typ typ in 
                    let tycon = deref_tycon tcref in 
                    let mty = nested_of_tycon tycon in 
                    let mk_nested_typ tyconNested = 
                        let tcrefNested = mk_tcref_in_tcref tcref tyconNested in 
                        let _,tps = chop_at (length tinst) (typars_of_tycon tyconNested) in
                        let tinstNested = ginstf m tps in 
                        mk_tyapp_ty tcrefNested (tinst @ tinstNested) in 
                    match optFilter with 
                    | Some nm -> lookupTypeNameInModulMaybeHaveArity nm typeNameResInfo mty |> List.map mk_nested_typ 
                    | None -> mty |> tyconsByAccessNames_of_modtyp |> Namemap.range_multi |> List.map mk_nested_typ 
                else [] in
            typs @ acc)
        g amap m 
        typ
        []

(*-------------------------------------------------------------------------
!* Report environments to visual studio. We stuff intermediary results 
 * into a global variable. A little unpleasant. 
 * REVIEW: We could at least put the global in cenv!!!
 *------------------------------------------------------------------------- *)

let envSink = ref None

let callEnvSink (scopem:range) (nenv:nameResEnv) = 
  (match !envSink with None -> () | Some (f,_) -> (f scopem nenv : unit))

let callQualSink m ((nenv:nameResEnv),(item:namedItemInEnv),(denv:displayEnv)) = 
  (match !envSink with None -> () | Some (_,f) -> (f (end_of_range m,(item,denv,nenv,m)) : unit))


(*-------------------------------------------------------------------------
!* Consume ids that refer to a namespace
 *------------------------------------------------------------------------- *)

let rec tc_namespace_in_namespace depth modref mty lid =
    if verbose then dprintf2 "--> tc_namespace_in_namespace(%s), modules = %s\n" (text_of_lid lid) (String.concat ";" (Namemap.domainL mty.mtyp_submoduls)); 
    match lid with 
    | [] -> success (depth,modref,mty)
    | id:: rest ->
        match mtyp_tryfind_submodul id.idText mty with
        | Some mspec -> tc_namespace_in_namespace (depth+1) (mk_modref_in_modref modref mspec) (mtyp_of_modul mspec) rest
        | None -> raze (UndefinedName(depth,"namespace",id,[]))

type fullyQualified = 
    (* Only resolve full paths - currently unused, but would be needed for a fully-qualified syntax *)
    | FullyQualified 
    | OpenQualified 

let tc_namespace_lid fullyQualified (nenv:nameResEnv) lid =
    match lid with 
    | [] -> no_results_or_useful_errors
    | id:: rest -> 
        let tab = 
            match fullyQualified with 
            | FullyQualified -> nenv.eFullyQualifiedModulesAndNamespaces 
            | OpenQualified -> nenv.eModulesAndNamespaces in

        match Map.tryfind id.idText tab with
        | Some modrefs -> map_some id.idRange (fun modref -> tc_namespace_in_namespace 1 modref (mtyp_of_modref modref) rest) modrefs
        | None -> raze (UndefinedName(0,"namespace or module",id,[]))

let tc_namespace_lid_then fullyQualified (nenv:nameResEnv) lid f =
    match lid with 
    | [] -> no_results_or_useful_errors
    | id :: rest -> 
        match tc_namespace_lid fullyQualified nenv [id] with
        |  Success modrefs -> 
              callQualSink id.idRange (nenv,Item_modrefs (map p23 modrefs), denv_of_nenv nenv);
              map_some id.idRange (fun (depth,modref,mty) ->  f (depth+1) id.idRange modref mty rest) modrefs
        |  Raze err -> Raze err 

(*-------------------------------------------------------------------------
!* Bind name used in "new Foo.Bar(...)" constructs
 *------------------------------------------------------------------------- *)

let tc_tdef_ctor edenv g amap m typ = 
  if verbose then   dprintf0 "--> tc_tdef_ctor\n"; 
    if is_delegate_typ typ then 
      success (Item_delegate_ctor typ,[])
    else 
      let cinfos =  intrinsic_cinfos_of_typ amap m typ in
      if is_interface_typ typ && isNil cinfos then 
        success (Item_fake_intf_ctor typ, [])
      else 
        let defaultStructCtorInfo = if (is_struct_typ typ && not(cinfos |> exists (minfo_is_nullary g))) then [DefaultStructCtor typ] else [] in 
        if verbose then   dprintf0 "--> tc_tdef_ctor (2)\n"; 
        if (isNil defaultStructCtorInfo && isNil cinfos) or not (is_stripped_tyapp_typ typ) then 
            raze (Error("No constructors are available for the type '"^NicePrint.pretty_string_of_typ edenv typ^"'",m))
        else 
            let cinfos = filter (minfo_accessible g amap m AccessibleFromSomewhere) cinfos  in 
            success (Item_ctor_group (name_of_tcref (tcref_of_stripped_typ typ), (defaultStructCtorInfo@cinfos)),[]) 

(*-------------------------------------------------------------------------
!* Bind IL "." notation (member lookup or lookup in a type)
 *------------------------------------------------------------------------- *)

let intrinsic_pinfos_of_typ_in_scope (optFilter, includePrivates) findFlag g amap m typ =
    let pinfos = intrinsic_pinfos_of_typ (optFilter, includePrivates) findFlag g amap m typ in 
    let pinfos = pinfos |> exclude_hidden_of_pinfos g amap m  in
    pinfos

let extension_pinfos_of_typ_in_scope eExtensionMembers (optFilter, includePrivates) findFlag g amap m typ =
    fold_entire_hierarchy_of_typ
        (fun typ acc -> 
            (if (is_stripped_tyapp_typ typ) then 
                let tcref = tcref_of_stripped_typ typ in 
                (* NOTE: multiple "open"'s push multiple duplicate values into eExtensionMembers *)
                (* TODO: this looks a little slow: gen_setify is quadratic. *)
                let ext_vrefs = gen_setify g.vref_eq (tcref_mmap_find tcref eExtensionMembers) in
                gather_vref_pinfos typ optFilter (fun gather ->
                     ext_vrefs |> List.iter (fun vref ->
                       match member_info_of_vref vref with 
                       | None -> ()
                       | Some(vspr) -> gather vspr vref)) 
             else []) @ acc)
        g amap m 
        typ
        [] 

let all_pinfos_of_typ_in_scope eExtensionMembers (optFilter, includePrivates) findFlag g amap m typ =
    intrinsic_pinfos_of_typ_in_scope (optFilter, includePrivates) findFlag g amap m typ
    @ extension_pinfos_of_typ_in_scope eExtensionMembers (optFilter, includePrivates) findFlag g amap m typ 

let intrinsic_minfos_of_typ_in_scope (optFilter,includePrivates) findFlag g amap m typ =
    let minfos = intrinsic_minfos_of_typ (optFilter,includePrivates) findFlag g amap m typ in 
    let minfos = minfos |> exclude_hidden_of_minfos g amap m in 
    minfos

let extension_minfos_of_typ_in_scope eExtensionMembers (optFilter,includePrivates) findFlag g amap m typ =
    fold_entire_hierarchy_of_typ
        (fun typ acc -> 
            (if (is_stripped_tyapp_typ typ) then 
                let tcref = tcref_of_stripped_typ typ in 
                (* NOTE: multiple "open"'s push multiple duplicate values into eExtensionMembers *)
                (* TODO: this looks a little slow: gen_setify is quadratic. *)
                let ext_vrefs = gen_setify g.vref_eq (tcref_mmap_find tcref eExtensionMembers) in
                ext_vrefs |> chooseList (fun vref -> 
                       match member_info_of_vref vref with 
                       | None -> None
                       | Some(vspr) -> select_vref_minfo g optFilter typ vspr vref) 
             else []) @ acc)
        g amap m 
        typ
        [] 

let all_minfos_of_typ_in_scope eExtensionMembers (optFilter,includePrivates) findFlag g amap m typ =
    intrinsic_minfos_of_typ_in_scope (optFilter,includePrivates) findFlag g amap m typ 
    @ extension_minfos_of_typ_in_scope eExtensionMembers (optFilter,includePrivates) findFlag g amap m typ          

exception IndeterminateType of range

let rec tc_lid_in_typ nenv depth ginstf g amap m lid findFlag typeNameResInfo typ =
    match lid with 
    | [] -> error(InternalError("tc_lid_in_typ edenv",m))
    | id :: rest -> 
        let nm = id.idText in (* used to filter the searaches of the tables *)
        let optFilter = Some(nm) in (* used to filter the searaches of the tables *)
        let tryLookup includePrivates =
           (* Lookup: datatype constructors take precedence *)
           match (if is_stripped_tyapp_typ typ then 
                     let tcref = tcref_of_stripped_typ typ in 
                     uconstr_of_tcref_by_name tcref nm |> Option.map (ucref_of_uconstr tcref)
                  else None) with 
           | Some ucref -> success(Item_ucref(ucref),rest)
           | None -> 
              let pinfos = intrinsic_pinfos_of_typ_in_scope (optFilter, includePrivates) findFlag g amap m typ in 
              if nonNil(pinfos) then success (Item_property (nm,pinfos),rest) else
              
              let minfos = intrinsic_minfos_of_typ_in_scope (optFilter,includePrivates) findFlag g amap m typ in 
              if nonNil(minfos) then success (Item_meth_group (nm,minfos),rest) else 
              
              let finfos = il_finfos_of_typ (optFilter,includePrivates) g amap m typ in
              match finfos with
              | finfo :: _ -> success (Item_il_field finfo,rest)
              | [] -> 
              
              let einfos = il_einfos_of_typ (optFilter,includePrivates) g amap m typ in 
              match einfos with
              | einfo :: _ -> success (Item_il_event einfo,rest)
              | [] -> 
              
              match rfinfo_of_typ_by_name g amap m typ  nm with
              | Some rfinfo -> success(Item_recdfield(rfinfo),rest)
              | None -> 

              let pinfos = extension_pinfos_of_typ_in_scope nenv.eExtensionMembers (optFilter, includePrivates) findFlag g amap m typ in 
              if nonNil(pinfos) then success (Item_property (nm,pinfos),rest) else
              
              let minfos = extension_minfos_of_typ_in_scope nenv.eExtensionMembers (optFilter,includePrivates) findFlag g amap m typ in 
              if nonNil(minfos) then success (Item_meth_group (nm,minfos),rest) else 
              
              
              if is_typar_ty typ then raze (IndeterminateType(union_ranges m id.idRange))
              else raze (UndefinedName (depth,"field, constructor or member", id,[])) in
              
        let contentsSearchAccessible, contentsSearchAll = (tryLookup DontIncludePrivate), (tryLookup IncludePrivate) in
        let nestedSearchAccessible = 
            let nestedTypes = nested_typs_of_typ (Some nm,typeNameResInfo) ginstf g amap m typ in 
            let typeNameResFlag,ntyargsOpt = typeNameResInfo in 
            if isNil rest then 
                if isNil nestedTypes then no_results_or_useful_errors
                else 
                  match typeNameResFlag with 
                  | ResolveTypeNamesToCtors -> map_some m (tc_tdef_ctor nenv.edenv g amap m) nestedTypes
                  | ResolveTypeNamesToTypeRefs -> one_result (success (Item_typs (nm,nestedTypes),rest)) 
            else tc_lid_in_typs nenv (depth+1) ginstf g amap m rest findFlag typeNameResInfo nestedTypes in 
        (one_result contentsSearchAccessible +++ nestedSearchAccessible),
        (one_result contentsSearchAll)
        
and tc_lid_in_typs nenv depth ginstf g amap m lid findFlag typeNameResInfo typs = 
    map_some m (tc_lid_in_typ nenv depth ginstf g amap m lid findFlag typeNameResInfo >> fst >> at_most_one_result m) typs 
                
(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
(* This would help make it the separation between name resolution and *)
(* type inference more obvious. Some things like caching lookups would *)
(* become quite a lot simpler. However this would mean each caller *)
(* would have to freshen. *)
let tc_lid_in_tcref nenv depth ginstf g amap m lid typeNameResInfo tcref =
    tc_lid_in_typ nenv depth ginstf g amap m lid IgnoreOverrides typeNameResInfo (freshen_tycon m ginstf tcref) 

let tc_lid_in_tcrefs nenv depth ginstf g amap m lid typeNameResInfo tcrefs = 
    map_some m (tc_lid_in_tcref nenv depth ginstf g amap m lid typeNameResInfo >> fst >> at_most_one_result m) tcrefs,
    map_some m (tc_lid_in_tcref nenv depth ginstf g amap m lid typeNameResInfo >> snd >> at_most_one_result m) tcrefs


(*-------------------------------------------------------------------------
!* tc_expr_lid_in_modul 
 *------------------------------------------------------------------------- *)

(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
let rec tc_expr_lid_in_modul nenv ginstf g amap typeNameResInfo depth m modref mty lid  =
    match lid with 
    | [] -> raze (InternalError("tc_expr_lid_in_modul",m))
    | id :: rest ->
        match Map.tryfind id.idText mty.mtyp_vals with
        | Some vspec -> 
            success(Item_val (mk_vref_in_modref modref vspec),rest)
        | None ->
        match  tryfind_uconstr_in_modref modref id with
        | Some tycon -> 
            success (Item_ucref (mk_ucref (mk_tcref_in_modref modref tycon) id.idText),rest)
        | None -> 
        match Map.tryfind id.idText (mtyp_exconsByDemangledName_of_modtyp mty) with
        | Some excon -> 
            success (Item_ecref (mk_ecref_in_modref modref excon),rest)
        | None ->

            (* Something in a type? *)
            let tyconSearchAccessible, tyconSearchAll = 
                if (nonNil rest) then 
                    let tycons = lookupTypeNameInModulNoArity id.idText mty in
                    let tcrefs = map (mk_tcref_in_modref modref) tycons in 
                    tc_lid_in_tcrefs nenv (depth+1) ginstf g amap m rest typeNameResInfo tcrefs 
                (* check if we've got some explicit type arguments *)
                else if fst typeNameResInfo = ResolveTypeNamesToTypeRefs && isSome (snd typeNameResInfo) then 
                    let tycons = lookupTypeNameInModulMaybeHaveArity id.idText typeNameResInfo mty in
                    let tcrefs = map (mk_tcref_in_modref modref) tycons in 
                    let typs = tcrefs |> map (freshen_tycon m ginstf) in 
                    success (map (fun typ -> (Item_typs(id.idText,[typ]),[])) typs), no_results_or_useful_errors
                else no_results_or_useful_errors, no_results_or_useful_errors in

            (* Object constructor of a type? *)
            let ctor_search = 
                if isNil rest then 
                    let tycons = lookupTypeNameInModulNoArity id.idText mty in
                    let tcrefs = map (mk_tcref_in_modref modref) tycons in 
                    let typs = map (freshen_tycon m ginstf) tcrefs in 
                    map_some id.idRange (tc_tdef_ctor nenv.edenv g amap  id.idRange) typs
                else no_results_or_useful_errors in 

            (* Something in a sub-namespace or sub-module *)
            let modul_search = 
                if (nonNil rest) then 
                    match mtyp_tryfind_submodul id.idText mty with
                    | Some mspec -> one_result (tc_expr_lid_in_modul nenv ginstf g amap typeNameResInfo (depth+1) m (mk_modref_in_modref modref mspec) (mtyp_of_modul mspec) rest)
                    | None -> no_results_or_useful_errors
                else no_results_or_useful_errors in 

            at_most_one_result id.idRange ( tyconSearchAccessible +++   ctor_search +++ modul_search +++ raze (UndefinedName(depth,"value, constructor, namespace or type",id,[])))


(*-------------------------------------------------------------------------
!* Resolve F#/IL "." syntax in expressions
 *------------------------------------------------------------------------- *)

(* We have a series of identifiers sep. by dots, e.g. A.B.c.D.e  *)
(* This is when being used an expression.  Not all of the sequence *)
(* will necessarily be swallowed, i.e. we return some identifiers *)
(* that may represent further actions, e.g. further lookups. *)
 
(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
let tc_expr_lid ginstf g amap m nenv typeNameResInfo lid =
    match lid with 
    | [] -> error (Error("invalid expression: "^text_of_lid lid, m))
    | [id] ->
        (* Single identifier.  This is the basic rule: lookup the environment! simple enough *)
        begin match Map.tryfind id.idText nenv.eItems with
        | Some res -> (res,[])
        | None -> 
            (* Check if it's a type name, e.g. a constructor call or a type instantiation *)
            let ctorSearch = 
                let tcrefs = lookupTypeNameInEnvMaybeHaveArity id.idText typeNameResInfo nenv in
                match fst typeNameResInfo with 
                | ResolveTypeNamesToCtors ->
                    let tcrefs = tcrefs |> filter (fun tcref -> is_il_tcref tcref or is_fsobjmodel_tcref tcref) in 
                    let typs = map (freshen_tycon m ginstf) tcrefs in 
                    map_some id.idRange (tc_tdef_ctor nenv.edenv g amap id.idRange) typs
                | ResolveTypeNamesToTypeRefs ->
                    let typs = tcrefs |> map (freshen_tycon m ginstf) in 
                    success (map (fun typ -> (Item_typs(id.idText,[typ]),[])) typs) in

            let failingCase = raze (UndefinedName(0,"value or constructor",id,[])) in
            let search = ctorSearch +++ failingCase in
            forceRaise (at_most_one_result m search)
        end
            
    (* A compound identifier. *)
    (* It still might be a value in the environment, or something in an F# module, namespace, typ, or nested type *)
    | id :: rest -> 
    
        (* Values in the environment take total priority, but contructors do NOT for compound lookups, e.g. if someone in some imported  *)
        (* module has defined a constructor "String" (common enough) then "String.foo" doesn't give an error saying 'constructors have no members' *)
        (* Instead we go lookup the String module or type. *)
        let val_exists_in_env nm = 
            (match Map.tryfind nm nenv.eItems with Some(Item_val _) -> true | _ -> false) in

        if val_exists_in_env id.idText &&
           (* Workaround for bug 908: adding "int", "float" etc. as functions has broken their use as types, and now System.Int32 etc. have *)
           (* to be used instead.  Here we are friendly and allow them to be used as types and just give a warning instead. *)
           (* Here we check that the thing being referenced is indeed a function value, and we know that we're doing a "int.Foo" lookup *)
           (* which doesn't make sense on a function value, so we give a warning and revert to the old interpretation. *)
           begin match Map.find id.idText nenv.eItems with 
           | Item_val vref when 
                let nm = name_of_vref vref in 
                (match nm with "string" | "int" | "float" | "float32" | "single" | "double" | "sbyte" | "byte" | "int16" | "uint16" | "int32" | "uint32"  -> true | _ -> false)
                && let _,tau = try_dest_forall_typ (type_of_vref vref) in
                   is_fun_ty tau
                 -> let nm = name_of_vref vref in 
                    warning(Error(sprintf "The standard definition of the identifier '%s' is now as function used to convert values to type '%s', or will be in a future release. Access static methods via the canonical uppercase name for the type, e.g. replace 'int.Parse(...)' with 'System.Int32.Parse(...)'" nm nm,id.idRange));
                    false
           | _ -> true
           end
        then
              Map.find id.idText nenv.eItems, rest
        else
          (* Otherwise modules are searched first. REVIEW: modules and types should be searched together. *)
          (* For each module referenced by 'id', search the module as if it were an F# module and/or a .NET namespace. *)
          let namedModuleSearch = 
              tc_namespace_lid_then OpenQualified nenv lid (tc_expr_lid_in_modul nenv ginstf g amap typeNameResInfo) in

          let tyconSearchAccessible,tyconSearchAll = 
              let tcrefs = lookupTypeNameInEnvMaybeHaveArity id.idText typeNameResInfo nenv in
              tc_lid_in_tcrefs nenv 1 ginstf g amap m rest typeNameResInfo tcrefs in

          let envSearch = 
              match Map.tryfind id.idText nenv.eItems with
              | Some res -> one_result (success (res,[]))
              | None -> no_results_or_useful_errors in

          let failingCase = raze (UndefinedName(0,"value, namespace, type or module",id,[])) in
               
          let search = namedModuleSearch +++ tyconSearchAccessible +++ envSearch +++ tyconSearchAll +++ failingCase in
          forceRaise (at_most_one_result m search)


(*-------------------------------------------------------------------------
!* Resolve F#/IL "." syntax in patterns
 *------------------------------------------------------------------------- *)

(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
let rec tc_pat_lid_in_modul nenv ginstf g amap typeNameResInfo depth m modref mty lid =
    if verbose then  dprintf3 "--> tc_pat_lid_in_modul edenv, lid = %s@%a\n" (text_of_lid lid) output_range m ;
    match lid with 
    | [] -> raze (Error("tc_pat_lid_in_modul edenv",m))
    | id :: rest ->
        match tryfind_uconstr_in_modref modref id with
        | Some tycon -> 
            success (Item_ucref (mk_ucref (mk_tcref_in_modref modref tycon) id.idText),rest)
        | None -> 
        match Map.tryfind id.idText (mtyp_exconsByDemangledName_of_modtyp mty) with
        | Some exnc -> 
            success (Item_ecref (mk_ecref_in_modref modref exnc),rest)
        | None ->
        (* An active pattern constructor in a module *)
        match Map.tryfind id.idText (aprefs_of_modref modref) with
        | Some apref -> 
            success (Item_apelem apref,rest)
        | None -> 
        match Map.tryfind id.idText mty.mtyp_vals with
        | Some vspec -> 
            success(Item_val (mk_vref_in_modref modref vspec),rest)
        | None ->
        (* Something in a type? e.g. a literal field *)
        let tyconSearchAccessible,_ = 
            if (nonNil rest) then
                let tycons = lookupTypeNameInModulNoArity id.idText mty in
                let tcrefs = map (mk_tcref_in_modref modref) tycons in 
                tc_lid_in_tcrefs nenv (depth+1) ginstf g amap m rest typeNameResInfo tcrefs
            else no_results_or_useful_errors,no_results_or_useful_errors in
        (* Constructor of a type? *)
        let ctor_search = 
            if isNil rest  then 
                let tycons = lookupTypeNameInModulNoArity id.idText mty in
                let tcrefs = map (mk_tcref_in_modref modref) tycons in 
                let typs = map (freshen_tycon m ginstf) tcrefs in 
                map_some id.idRange (tc_tdef_ctor nenv.edenv g amap  id.idRange) typs
            else no_results_or_useful_errors in 
        (* Something in a sub-namespace or sub-module or nested-type *)
        let modul_search = 
            if (nonNil rest) then 
                match mtyp_tryfind_submodul id.idText mty with
                | Some mspec -> one_result (tc_pat_lid_in_modul nenv ginstf g amap typeNameResInfo (depth+1) m (mk_modref_in_modref modref mspec) (mtyp_of_modul mspec) rest)
                | None -> no_results_or_useful_errors
             else no_results_or_useful_errors in 
        let res = at_most_one_result id.idRange ( tyconSearchAccessible +++   ctor_search +++ modul_search +++ raze (UndefinedName(depth,"constructor, module or namespace",id,[]))) in 
        res
        
exception UpperCaseIdentifierInPattern of range
type warnOnUpperFlag = WarnOnUpperCase | AllIdsOK


(* Long ID in a pattern *)
(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
let tc_pat_lid warnOnUpper ambiguousDataTagHasArgs ginstf g amap m nenv typeNameResInfo lid =
    match lid with 
    (* Single identifiers in patterns *)
    | [id] ->
        begin 
          (* Single identifiers in patterns - bind to constructors and active patterns *)
          (* For the special case of *)
          (*   let C = x *)
          (* we only bind the identifier C to constructors that have zero arguments *)
          (* For the special case of *)
          (*   let C args = x *)
          (* we only bind the identifier C to constructors that have non-zero arguments *)
          match Map.tryfind id.idText nenv.ePatItems with
          | Some res 
               when (match ambiguousDataTagHasArgs,res with 
                     | Some(hasArgs),Item_ucref ucref -> (hasArgs = nonNil(rfields_of_ucref ucref))
                     | Some(hasArgs),Item_ecref ecref -> (hasArgs = nonNil(typs_of_ecref_rfields ecref))
                     | Some(hasArgs),Item_apelem  apref -> true
                     | _ -> true) -> res
          | _ -> 
          (* Single identifiers in patterns - variable bindings *)
          if (match ambiguousDataTagHasArgs with None -> true | Some(v) -> v) && 
             (warnOnUpper = WarnOnUpperCase) && 
             String.length id.idText >= 3 && 
             Char.lowercase id.idText.[0] <> id.idText.[0] then 
            warning(UpperCaseIdentifierInPattern(m));
          Item_newdef id
        end
        
    (* Long identifiers in patterns *)
    | _ -> 
        if verbose then  dprintf3 "--> tc_pat_lid, lid = %s@%a\n" (text_of_lid lid) output_range m ;
        let tyconSearchAccessible,tyconSearchAll = 
            match lid with 
            | tn:: rest when nonNil rest ->
                let m = tn.idRange in 
                let tcrefs = lookupTypeNameInEnvNoArity tn.idText nenv in
                tc_lid_in_tcrefs nenv 1 ginstf g amap m rest typeNameResInfo tcrefs 
            | _ -> no_results_or_useful_errors,no_results_or_useful_errors in 
        let namedModuleSearch = tc_namespace_lid_then OpenQualified nenv lid (tc_pat_lid_in_modul nenv ginstf g amap typeNameResInfo) in 
        let search = tyconSearchAccessible +++  namedModuleSearch +++ tyconSearchAll in 
        let res,rest = forceRaise (at_most_one_result m search) in 
        if rest <> [] then error(Error("this is not a constructor or literal, or a constructor is being used incorrectly",(hd rest).idRange));
        res


(*-------------------------------------------------------------------------
!* Resolve F#/IL "." syntax in types
 *------------------------------------------------------------------------- *)

let rec tc_tycon_id_in_tcref g amap typeNameResInfo depth m tcref lid =
    let tycon = deref_tycon tcref in 
    let mty = nested_of_tycon tycon in 
    match lid with 
    | [] -> error(Error("unexpected empty long identifier",m))
    | [id] -> 
        begin match lookupTypeNameInModulMaybeHaveArity id.idText typeNameResInfo mty with 
        | res :: _ -> success (mk_tcref_in_tcref tcref res)
        | [] -> raze (UndefinedName(depth,"type",id,[]))
        end
    | id::rest ->
        (* No modules in tycons *)
        (* let modulSearch = ... *)
        let tyconSearchAccessible = 
            match lookupTypeNameInModulMaybeHaveArity id.idText typeNameResInfo mty with
            | [] -> no_results_or_useful_errors
            | tycon :: _ -> one_result (tc_tycon_id_in_tcref g amap typeNameResInfo depth m (mk_tcref_in_tcref tcref tycon) lid) in
        at_most_one_result m tyconSearchAccessible


let rec tc_tycon_id_in_modul g amap typeNameResInfo depth m modref mty lid =
    match lid with 
    | [] -> error(Error("unexpected empty long identifier",m))
    | [id] -> 
        (* On all paths except error reporting we have isSome(ntyargsOpt), hence get at most one result back *)
        begin match lookupTypeNameInModulMaybeHaveArity id.idText typeNameResInfo mty with 
        | tycon :: _ -> success (mk_tcref_in_modref modref tycon)
        | [] -> raze (UndefinedName(depth,"type",id,[]))
        end
    | id::rest ->
        let modulSearch = 
            match mtyp_tryfind_submodul id.idText mty with
            | Some mspec -> tc_tycon_id_in_modul g amap typeNameResInfo (depth+1) m (mk_modref_in_modref modref mspec) (mtyp_of_modul mspec) rest
            | None ->  raze (UndefinedName(depth,"module or namespace",id,[])) in
        let tyconSearchAccessible = 
            (* On all paths except error reporting we have isSome(ntyargsOpt), hence get at most one result back *)
            match lookupTypeNameInModulMaybeHaveArity id.idText typeNameResInfo mty with
            | [] -> no_results_or_useful_errors
            | tycon :: _ -> one_result (tc_tycon_id_in_tcref g amap typeNameResInfo depth m (mk_tcref_in_modref modref tycon) rest) in
        at_most_one_result m (tyconSearchAccessible +++ one_result modulSearch)

let tc_tycon_id fullyQualified g amap m nenv lid ntyargs =
    match lid with 
    | [] -> error(Error("unexpected empty long identifier",m))
    | [id]  ->  
        begin match lookupTypeNameInEnvHaveArity id.idText ntyargs nenv with
        | Some res -> success(res)
        | None -> 
            (* For Good Error Reporting! *)
            let tcrefs = lookupTypeNameInEnvNoArity id.idText nenv in
            match tcrefs with
            | tcref :: _ -> 
                success(tcref)
            | [] -> 
                raze (UndefinedName(0,"type",id,Namemap.domainL nenv.eTyconsByAccessNames))
        end 
    | id::rest ->
        let tyconSearch = 
            match lookupTypeNameInEnvHaveArity id.idText ntyargs nenv with
            | Some tcref -> one_result (tc_tycon_id_in_tcref g amap (ResolveTypeNamesToTypeRefs,Some(ntyargs)) 1 m tcref rest) 
            | None -> no_results_or_useful_errors in        
        let modulSearch = tc_namespace_lid_then fullyQualified nenv lid (tc_tycon_id_in_modul g amap (ResolveTypeNamesToTypeRefs,Some(ntyargs))) in
        let modulSearchFailed = tc_namespace_lid_then fullyQualified nenv lid (tc_tycon_id_in_modul g amap (ResolveTypeNamesToTypeRefs,None)) in
(*
        let tyconSearchFailed = 
            (* For Good Error Reporting! *)
            match lookupTypeNameInEnvNoArity id.idText nenv with 
            | [] -> no_results_or_useful_errors
            | tcrefs -> success(tcrefs) in
*)
        at_most_one_result m (tyconSearch +++ modulSearch +++ modulSearchFailed (* +++ tyconSearchFailed *) )


(*-------------------------------------------------------------------------
!* Resolve F#/IL "." syntax in records etc.
 *------------------------------------------------------------------------- *)

(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
let rec tc_field_id_in_modul nenv ginstf g amap typeNameResInfo depth m modref mty lid : (_ * _) Outcome.outcome = 
    if verbose then  dprintf3 "--> tc_field_id_in_modul edenv, lid = %s@%a\n" (text_of_lid lid) output_range m ;
    match lid with 
    | id::rest -> 
        let error = raze (UndefinedName(depth,"record label or namespace",id,[])) in
        (* search for module-qualified names, e.g. { Microsoft.FSharp.Core.contents = 1 } *)
        let modulScopedFieldNames = 
            match tryfind_rfield_in_mtyp mty id  with
            | Some tycon -> success(mk_rfref_in_modref modref tycon id, rest)
            | None -> error in
        (* search for type-qualified names, e.g. { Microsoft.FSharp.Core.Ref.contents = 1 } *)
        let tyconSearchAccessible = 
            if (nonNil rest) then
                let tycons = lookupTypeNameInModulNoArity id.idText mty in
                let tcrefs = map (mk_tcref_in_modref modref) tycons in 
                let tyconSearchAccessible,_ = tc_lid_in_tcrefs nenv (depth+1) ginstf g amap m rest typeNameResInfo tcrefs in
                (* choose only fields *)
                let tyconSearchAccessible = tyconSearchAccessible ||> chooseList (function (Item_recdfield(RecdFieldInfo(_,rfref)),rest) -> Some(rfref,rest) | _ -> None) in 
                tyconSearchAccessible
            else no_results_or_useful_errors in
        (* search for names in nested modules, e.g. { Microsoft.FSharp.Core.contents = 1 } *)
        let modulSearch = 
            if nonNil rest then 
                match mtyp_tryfind_submodul id.idText mty with
                | Some mspec -> tc_field_id_in_modul nenv ginstf g amap typeNameResInfo (depth+1) m (mk_modref_in_modref modref mspec) (mtyp_of_modul mspec)  rest 
                | None -> error
            else error in 
        at_most_one_result m (one_result modulScopedFieldNames +++ tyconSearchAccessible +++ one_result modulSearch)
    | [] -> failwith "tc_field_id_in_modul edenv"

exception DeprecatedClassFieldInference of range


(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
let tc_field_id ginstf g amap nenv typ (mp,id) =
    let typeNameResInfo = defaultTypeNameResInfo in 
    if verbose then  dprintf1 "--> tc_field, lid = %s\n" (text_of_lid (mp@[id])) ;
    let m = id.idRange in 
    match mp with 
    | [] -> 
        if is_stripped_tyapp_typ typ then 
            match rfinfo_of_typ_by_name g amap m typ id.idText with
            | Some (RecdFieldInfo(_,rfref)) -> [(rfref,true)]
            | None -> error(Error(sprintf "The type %s does not contain a field %s" (NicePrint.pretty_string_of_typ nenv.edenv typ) id.idText,m))
        else 
            let frefs = tryname "record label" nenv.eFieldLabels id  in
            (* Eliminate duplicates arising from multiple 'open' *)
            let frefs = frefs |> gen_setify (fun (fref1,_) (fref2,_) -> g.tcref_eq (tcref_of_rfref fref1) (tcref_of_rfref fref2)) in
            frefs
                        
    | _ -> 
        let lid = (mp@[id]) in 
        let tyconSearchAccessible = 
            match lid with 
            | tn:: (_ :: _ as rest) -> 
                let m = tn.idRange in 
                let tcrefs = lookupTypeNameInEnvNoArity tn.idText nenv in 
                let tyconSearchAccessible,_ = tc_lid_in_tcrefs nenv 1 ginstf g amap m rest typeNameResInfo tcrefs in
                (* choose only fields *)
                let tyconSearchAccessible = tyconSearchAccessible ||> chooseList (function (Item_recdfield(RecdFieldInfo(_,rfref)),rest) -> Some(rfref,rest) | _ -> None) in 
                tyconSearchAccessible
            | _ -> no_results_or_useful_errors in 
        let modulSearch = tc_namespace_lid_then OpenQualified nenv lid (tc_field_id_in_modul nenv ginstf g amap typeNameResInfo) in 
        let item,rest = forceRaise (at_most_one_result m (modulSearch +++ tyconSearchAccessible)) in 
        if rest <> [] then errorR(Error("invalid field label",(hd rest).idRange));
        [(item,true)]

let freshen_rfref m ginstf rfref =
    Item_recdfield(RecdFieldInfo(ginstf m (typars_of_tycon (tycon_of_rfref rfref)), rfref))


(*-------------------------------------------------------------------------
!* Resolve F#/IL "." syntax in expressions (2).
 *------------------------------------------------------------------------- *)

(* We have an expr. on the left, and we do an access, e.g. *)
(* (f obj).field or (f obj).meth.  The basic rule is that if l-r type *)
(* inference has determined the outer type then we can proceed in a simple fashion. The exception *)
(* to the rule is for field types, which applies if l-r was insufficient to *)
(* determine any valid members *)

(* QUERY (ginstf cleanup): it would be really nice not to flow ginstf to here. *)
let tc_expr_dot_lid ginstf g amap m nenv typ lid findFlag =
    let typeNameResInfo = defaultTypeNameResInfo in
    let adhoctDotSearchAccessible,adhoctDotSearchAll = tc_lid_in_typ nenv 1 ginstf g amap m lid findFlag typeNameResInfo typ  in
    let adhoctDotSearchAccessible= adhoctDotSearchAccessible |> at_most_one_result m in 
    let adhoctDotSearchAll = adhoctDotSearchAll |> at_most_one_result m in
    match adhoctDotSearchAccessible with 
    | Raze _ ->
        (* If the dot is not resolved by adhoc overloading then look for a record field *)
        (* that can resolve the name. *)
        let dot_field_id_search = 
            match lid with 
            (* A unique record label access, e.g  expr.field  *)
            | id::rest when Map.mem id.idText nenv.eFieldLabels -> 
                begin match Map.find id.idText nenv.eFieldLabels with
                | [] -> no_results_or_useful_errors
                | (rfref,isRecdField) :: _ ->
                    if not isRecdField then errorR(DeprecatedClassFieldInference(m));
                    (* NOTE (ginstf cleanup): we need to freshen here because we don't know the type. *)
                    (* But perhaps the caller should freshen?? *)
                    let item = freshen_rfref m ginstf rfref in
                    one_result (success (item,rest))
                end
            | _ -> no_results_or_useful_errors  in 
        
        (* A unique record label access qualified by a module, e.g  expr.Module.field*)
        (* Really for OCaml compat. only *)
        let modul_dot_field_id_search =
            match lid with 
            | id::(_ :: _ as rest) when Map.mem id.idText nenv.eModulesAndNamespaces ->
                (* QUERY (ginstf cleanup): surely ginstf should not be usable by this resolution. *)
                (* QUERY (ginstf cleanup): It could only be required for a Module+Type qualification. *)
                tc_namespace_lid_then OpenQualified nenv lid (tc_field_id_in_modul nenv ginstf g amap typeNameResInfo) 
                (* QUERY: should caller freshen? *)
                ||> map (map_fst (fun rfref ->
                       warning(OCamlCompatibility("This lookup resolves to a record field by using the syntax 'expr.Module.field'. Although this is allowed for OCaml compatibility, the style is considered deprecated for F#. Consider using a simple field lookup 'expr.field', perhaps with an annotation to constrain the type of 'expr'",m));
                       freshen_rfref m ginstf rfref))
            | _  -> no_results_or_useful_errors in 
        forceRaise (at_most_one_result m (dot_field_id_search +++ modul_dot_field_id_search +++  one_result adhoctDotSearchAll))
    | Success _ -> 
        forceRaise adhoctDotSearchAccessible

let compute_item_range wholem lid rest =
    let item_lid,_ = chop_at (max 0 (length lid - length rest)) lid in 
    match item_lid with 
    | [] -> wholem
    | h :: _ -> let _,last = frontAndBack item_lid in union_ranges h.idRange last.idRange 

let tc_expr_lid_and_compute_range ginstf g amap wholem nenv typeNameResInfo lid = 
    let item,rest = tc_expr_lid ginstf g amap wholem nenv typeNameResInfo lid in 
    let itemm = compute_item_range wholem lid rest in
    item, compute_item_range wholem lid rest, rest

let tc_expr_dot_lid_and_compute_range ginstf g amap wholem nenv typ lid findFlag = 
    let item,rest = tc_expr_dot_lid ginstf g amap wholem nenv typ lid findFlag in 
    item, compute_item_range wholem lid rest, rest
    
(*-------------------------------------------------------------------------
!* Given an nenv resolve partial paths to sets of names, used by interactive
 * environments (Visual Studio)
 *
 * ptc = partial type check
 * ptci = partial type check item
 *
 * There are some inefficiencies in this code - e.g. we often 
 * create potentially large lists of methods/fields/properties and then
 * immiediately filter them.  We also use lots of "map/concats".  Dosen't
 * seem to hit the interactive experience too badly though.
 *------------------------------------------------------------------------- *)

let fake_ginstf (m:range) gps = map mk_typar_ty gps 

let ptc_collect f l =  fold_right (f >> (@)) l []

(* note: making local refs is ok since it is only used by VS *)
let ptci_of_vref v = Item_val(v)
let ptci_of_ucref v = Item_ucref v
let ptci_of_ecref v = Item_ecref v
let ptci_of_submodul v = Item_modrefs [v]
let ptci_of_recdfield v = Item_recdfield v
let ptci_of_il_finfo finfo = Item_il_field finfo
let ptci_of_il_einfo x = Item_il_event x
let ptci_of_pinfo pinfo = Item_property (name_of_pinfo pinfo,[pinfo])
let ptci_of_minfos (nm,minfos) = Item_meth_group(nm,minfos)

let ptci_of_tycon m x = 
   let nm = display_name_of_tcref x in
   Item_typs (nm,[freshen_tycon m fake_ginstf x])

let ptci_of_typ x = 
   let nm = if is_stripped_tyapp_typ x then display_name_of_tcref (tcref_of_stripped_typ x) else "?" in
   Item_typs (nm,[x])

(* Filter out 'PrivateImplementationDetail' classes *)
let interesting_modul nm =
  String.length nm >= 1 &&
  String.sub nm 0 1 <> "<"

let rec ptc_namespace_in_namespace_then f plid modref =
    if verbose then  dprintf1 "ptc_namespace_in_namespace_then, plid = %s\n" (text_of_path plid); 
    let mty = mtyp_of_modref modref in 
    match plid with 
    | [] -> f modref
    | id:: rest -> 
        match mtyp_tryfind_submodul id mty with
        | Some mty -> ptc_namespace_in_namespace_then f rest (mk_modref_in_modref modref mty) 
        | None -> []

let ptc_namespace_lid_then (nenv:nameResEnv) plid f =
    if verbose then  dprintf1 "ptc_namespace_lid_then, plid = %s\n" (text_of_path plid); 
    match plid with 
    | id:: rest -> 
        begin match Map.tryfind id nenv.eModulesAndNamespaces with
        | Some(modrefs) -> 
            ptc_collect (ptc_namespace_in_namespace_then f rest) modrefs
        | None ->
            []
        end
    | [] -> []

let ptc_at_typ nenv g amap m statics typ =
    let rfinfos = 
      if is_stripped_tyapp_typ typ then 
        let tc,tinst = dest_stripped_tyapp_typ typ in 
        (all_rfrefs_of_tcref tc)
        |> filter (fun fref -> static_of_rfref fref = statics)
        |> map (fun fref -> RecdFieldInfo(tinst,fref)) 
      else [] in 

    let ucinfos = 
      if statics then 
        let tc,tinst = dest_stripped_tyapp_typ typ in 
        map (fun ucref ->  Item_ucref(ucref)) (ucrefs_of_tcref tc)
      else [] in 

    let finfo_filter x = 
      il_finfo_is_static x = statics &&
      il_finfo_accessible g amap m AccessibleFromSomewhere x in

    let einfo_filter x = 
      is_standard_il_einfo g amap m x &&
      il_einfo_is_static x = statics &&
      il_einfo_accessible g amap m AccessibleFromSomewhere x in
    let einfos = filter einfo_filter (il_einfos_of_typ (None,DontIncludePrivate) g amap m typ) in

    let pinfo_filter x = 
      pinfo_is_static x = statics && 
      pinfo_accessible g amap m AccessibleFromSomewhere x in

    let nestedTypes = nested_typs_of_typ (None,(ResolveTypeNamesToTypeRefs,None)) fake_ginstf g amap m typ in 

    let finfos = filter finfo_filter (il_finfos_of_typ (None,DontIncludePrivate) g amap m typ) in
    let pinfos = filter pinfo_filter (all_pinfos_of_typ_in_scope nenv.eExtensionMembers (None,DontIncludePrivate) IgnoreOverrides g amap m typ) in

  (* Exclude get_ and set_ methods accessed by properties *)
    let pinfoMethNames = 
      map 
        (fun pinfo -> name_of_minfo (getter_minfo_of_pinfo pinfo))
        (filter pinfo_has_getter pinfos)
      @
      map 
        (fun pinfo -> name_of_minfo (setter_minfo_of_pinfo pinfo))
        (filter pinfo_has_setter pinfos) in 
    
    let einfoMethNames = 
      (einfos |> filter (il_einfo_is_static >> not) |> map (add_minfo_of_il_einfo >> name_of_il_minfo)) @
      (einfos |> filter (il_einfo_is_static >> not) |> map (remove_minfo_of_il_einfo >> name_of_il_minfo)) in 
    let names = Zset.addL pinfoMethNames (Zset.addL einfoMethNames (Zset.empty string_ord)) in 

    let minfo_filter minfo = 
      (* Only show the Finalize, MemberwiseClose etc. methods on System.Object for values whose static type really is *)
      (* System.Object. Few of these are typically used from F#.  *)
      (type_equiv g typ g.obj_ty 
       or name_of_minfo minfo = "GetType" 
       or name_of_minfo minfo = "GetHashCode" 
       or name_of_minfo minfo = "ToString" 
       or name_of_minfo minfo = "Equals" 
       or not (type_equiv g (typ_of_minfo minfo) g.obj_ty)) &&
      not (minfo_is_instance minfo) = statics &&
      minfo_accessible g amap m AccessibleFromSomewhere minfo &&
      not (minfo_is_ctor minfo) &&
      not (minfo_is_cctor minfo) &&
      not (Zset.mem (name_of_minfo minfo) names) in
    
    (* REVIEW: add a name filter here in the common cases *) 
    let minfos = all_minfos_of_typ_in_scope nenv.eExtensionMembers (None,DontIncludePrivate) IgnoreOverrides g amap m typ in
    let minfos = minfos |> exclude_hidden_of_minfos g amap m in 
    let minfos = minfos |> filter minfo_filter in 

    (* partition methods into overload sets *)
    let rec partitionl l acc = 
      match l with
      | [] -> acc
      | h::t -> 
          let nm = (name_of_minfo h) in
          partitionl t (Namemap.add_multi nm h acc) in
    ucinfos @
    map ptci_of_recdfield rfinfos @
    map ptci_of_pinfo pinfos @
    map ptci_of_il_finfo finfos @
    map ptci_of_il_einfo einfos @
    map ptci_of_typ nestedTypes @
    map ptci_of_minfos (Namemap.to_list (partitionl minfos Map.empty))
      

let rec ptc_in_typ nenv g amap m statics plid typ =
    if verbose then   dprintf2 "ptc_in_typ , typ = '%s', plid = '%s'\n" (NicePrint.pretty_string_of_typ (empty_denv g) typ) (text_of_path plid);
    match plid with
    | [] -> ptc_at_typ nenv g amap m statics typ
    | id :: rest ->
  
      let rfinfos = 
        if is_stripped_tyapp_typ typ then 
          let tc,tinst = dest_stripped_tyapp_typ typ in 
          (all_rfrefs_of_tcref tc)
          |> filter (fun fref -> static_of_rfref fref = statics)
          |> map (fun fref -> RecdFieldInfo(tinst,fref)) 
        else 
          [] in
  
      let nestedTypes = 
        typ 
        |> nested_typs_of_typ (Some id,(ResolveTypeNamesToTypeRefs,None)) fake_ginstf g amap m  
        |> filter (typ_accessible AccessibleFromSomewhere) in 

      let ucinfos = 
        if statics && is_stripped_tyapp_typ typ then 
          let tc,tinst = dest_stripped_tyapp_typ typ in 
          map (fun ucref ->  Item_ucref(ucref)) (ucrefs_of_tcref tc)
        else [] in 
  
      let finfo_filter x = 
        il_finfo_is_static x = statics &&
        il_finfo_accessible g amap m AccessibleFromSomewhere x in

      let einfo_filter x = 
        is_standard_il_einfo g amap m x &&
        il_einfo_is_static x = statics &&
        il_einfo_accessible g amap m AccessibleFromSomewhere x in

      let pinfo_filter x = 
        pinfo_is_static x = statics && 
        pinfo_accessible g amap m AccessibleFromSomewhere x in
 
      (* e.g. <val-id>.<recdfield-id>.<more> *)
      (rfinfos |> filter (fun x -> (name_of_rfinfo x) = id)
               |> mapConcat (vtyp_of_rfinfo >> ptc_in_typ nenv g amap m false rest)) @

      (* e.g. <val-id>.<property-id>.<more> *)
      (typ
         |> intrinsic_pinfos_of_typ (Some(id),DontIncludePrivate) IgnoreOverrides g amap m 
         |> filter pinfo_filter 
         |> mapConcat (vtyp_of_pinfo g amap m >> ptc_in_typ nenv g amap m false rest)) @

      (* e.g. <val-id>.<event-id>.<more> *)
      (typ 
         |> il_einfos_of_typ (Some(id),DontIncludePrivate) g amap m
         |> mapConcat (prop_typ_of_il_einfo g amap m >> ptc_in_typ nenv g amap m false rest)) @

      (* nested types! *)
      (typ 
        |> nested_typs_of_typ (Some id,(ResolveTypeNamesToTypeRefs,None)) fake_ginstf g amap m  
        |> filter (typ_accessible AccessibleFromSomewhere) 
        |> mapConcat (ptc_in_typ nenv g amap m statics rest)) @

      (* e.g. <val-id>.<il-field-id>.<more> *)
      (typ
         |> il_finfos_of_typ (Some(id),DontIncludePrivate) g amap m
         |> mapConcat (vtyp_of_il_finfo amap m >> ptc_in_typ nenv g amap m false rest))
     
let tycon2_accessible tycon =
  match repr_of_tycon tycon with 
  | Some (TIlObjModelRepr (_,_,tdef)) -> tdef_accessible tdef
  | _ -> true

(*-------------------------------------------------------------------------
!* Some code paths (e.g. Visual Studio) that manipulate minfos don't want to 
 * copy them, hence the need for fake_ginstf.  ginstf is passes a prefix
 * of type parameters - this is empty except when reading the type paramters 
 * of a generic method.
 *------------------------------------------------------------------------- *)


let ptcis_of_tycon_ctors g amap m tcref = 
    let typ = freshen_tycon m fake_ginstf tcref in 
    match tc_tdef_ctor (empty_denv g) g amap m typ with 
    | Success (item,_) -> 
        begin match item with 
        | Item_ctor_group(nm,cinfos) -> 
            cinfos 
            |> List.filter (minfo_accessible g amap m AccessibleFromSomewhere)
            |> List.map (fun minfo -> Item_ctor_group(nm,[minfo])) 
        | item -> 
            [item]
        end
    | Raze _ -> []

(* import.ml creates somewhat fake modules for nested members of types (so that *)
(* types never contain other types) *)
let not_fake_container_modul tyconNames nm = not (Zset.mem nm tyconNames)


(** Check is a namesapce or module contains something accessible *)
let rec modref_contains_something_accessible g amap m modref = 
  let mty = mtyp_of_modref modref in 

   (* Search the values in the module for an accessible value *)
   (* BUG: we're not applying accessibility checks here, just looking for any value *)
   (mty.mtyp_vals 
    |> Namemap.exists (fun _ v -> 
         let vref = mk_vref_in_modref modref v in 
         not (compgen_of_vref vref) && isNone(member_info_of_vref vref))) ||

   (* Search the types in the namespace/module for an accessible tycon *)
   (* BUG: tycon2_accessible is only an approximation to rule out private things in .NET modules *)
   (mty.mtyp_tycons 
    |> Namemap.exists (fun _ tc ->  
          not (tycon_is_modul tc) && tycon2_accessible tc)) ||

   (* Search the sub-modules of the namespace/modulefor something accessible *)
   (mty 
    |> submoduls_of_mtyp 
    |> Namemap.exists (fun _ submod -> 
        let submodref = mk_modref_in_modref modref submod in
        modref_contains_something_accessible g amap m submodref)) 



let rec ptc_in_modul nenv g amap m modref plid  =
  if verbose then   dprintf1 "ptc_in_modul, plid = %s\n" (text_of_path plid); 
  let mty = mtyp_of_modref modref in 
  
  let tycons = 
      mty.mtyp_tycons
      |> Namemap.range 
      |> filter tycon2_accessible in 

  let tyconNames = 
      Namemap.domain mty.mtyp_tycons in
      
  match plid with 
  | [] -> 

       (* Collect up the accessible values in the module *)
       (mty.mtyp_vals 
        |> Namemap.range
        |> map (mk_vref_in_modref modref)
        |> filter (fun v -> not (compgen_of_vref v) && member_info_of_vref v = None)
        |> map ptci_of_vref)


       (* Collect up the accessible discriminated union cases in the module *)
     @ (ucrefs_in_modref modref |> map ptci_of_ucref)
       (* @ map ptci_of_recdfield (recdfields_in_modul modref) *)

       (* Collect up the accessible F# exception declarations in the module *)
     @ (mtyp_exconsByDemangledName_of_modtyp mty 
        |> Namemap.range 
        |> map (mk_ecref_in_modref modref >> ptci_of_ecref))

       (* Collect up the accessible sub-modules *)
     @ (submoduls_of_mtyp mty 
        |> Namemap.range 
        |> filter (name_of_modul >> not_fake_container_modul tyconNames)
        |> filter (name_of_modul >> interesting_modul)
        |> map (mk_modref_in_modref modref)
        |> filter (modref_contains_something_accessible g amap m)
        |> map ptci_of_submodul)

  (* Get all the types and .NET constructor groups accessible from here *)
     @ (tycons 
        |> map (mk_tcref_in_modref modref >> ptci_of_tycon m) )

     @ (tycons 
        |> map (mk_tcref_in_modref modref >> ptcis_of_tycon_ctors g amap m) |> concat)

  | id :: rest  -> 
      (match mtyp_tryfind_submodul id mty with 
       | Some mspec -> ptc_in_modul nenv g amap m (mk_modref_in_modref modref mspec) rest 
       | None -> [])

    @ (lookupTypeNameInModulNoArity id mty 
       |> mapConcat (mk_tcref_in_modref modref  >> generalize_tcref >> snd >> ptc_in_typ nenv g amap m true rest))

let ptc_lid g amap m nenv plid = 
  match  plid with
  |  [] -> 

     let items = 
        nenv.eItems
        |> Namemap.range
        |> filter (function Item_val v -> not (compgen_of_vref v) | _ -> true) in

     let apats = 
        nenv.ePatItems
        |> Namemap.range
        |> filter (function Item_apelem v -> true | _ -> false) in 

     let mods = 
        nenv.eModulesAndNamespaces 
        |> Namemap.range_multi 
        |> filter (name_of_modref >> interesting_modul  )
        |> filter (name_of_modref >> not_fake_container_modul (Namemap.domain nenv.eTyconsByAccessNames) )
        |> filter (modref_contains_something_accessible g amap m)
        |> map ptci_of_submodul in 

     let tycons = 
        nenv.eTyconsByDemangledNameAndArity
        |> Namemap.range
        |> filter (deref_tycon >> tycon2_accessible) 
        |> filter (deref_tycon >> tycon_is_exnc >> not) 
        |> map (ptci_of_tycon m) in 

     (* Get all the constructors accessible from here *)
     let constructors =  
         nenv.eTyconsByDemangledNameAndArity
         |> Namemap.range
         |> filter (deref_tycon >> tycon2_accessible) 
         |> mapConcat (ptcis_of_tycon_ctors g amap m) in 

     items @ apats @ mods @ tycons @ constructors 

  | id :: rest -> 
  
      (* Look in the namespaces 'id' *)
      ptc_namespace_lid_then  nenv [id] (fun modref -> 
        if modref_contains_something_accessible g amap m modref then 
          ptc_in_modul nenv g amap m modref rest 
        else 
          [])

      (* Look for values called 'id' that accept the dot-notation *)
    @ (if Map.mem id nenv.eItems then 
               (* v.lookup : member of a value *)
        let v = Map.find id nenv.eItems in 
        match v with 
        | Item_val x -> 
            if verbose then   dprintf1 "ptc_lid (through val), plid = %s\n" (text_of_path plid); 
            let typ = (type_of_vref x) in
            let typ = if base_of_vref x = CtorThisVal then dest_refcell_ty g typ else typ in 
            ptc_in_typ nenv g amap m false rest  typ
        | _ -> []
       else [])
    @ 
  (* type.lookup : lookup a static something in a type *)
      (lookupTypeNameInEnvNoArity id nenv |> mapConcat (freshen_tycon m fake_ginstf >> ptc_in_typ nenv g amap m true rest))


