// (c) Microsoft Corporation 2005-2007. 

#light
namespace Microsoft.FSharp.Math

    #nowarn "60"

    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.LanguagePrimitives
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Math
    open Microsoft.FSharp.Primitives.Basics
    open Microsoft.FSharp.Primitives.Basics.RangeOps
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Text.StructuredFormat
    open Microsoft.FSharp.Text.StructuredFormat.LayoutOps
    open Microsoft.FSharp.Compatibility
    open System
    open System.Globalization


//----------------------------------------------------------------------------
//README: overview and suffix documentation
//    _GU  = generic unspecialized (Matrix<T>, Vector<T> etc.) 
//    _GUA = generic unspecialized op on (underlying) array
//    _DS  = Double specialized (Matrix<float> = matrix, Vector<float> = vector etc.)
//
//    DM   = dense matrix
//    SM   = sparse matrix
//    V    = vector (dense)
//    RV   = row vector (dense)
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// type opsData
//--------------------------------------------------------------------------*)

    type 'a opsData = INumeric<'a> option


//----------------------------------------------------------------------------
// type DenseMatrix<'a>
//--------------------------------------------------------------------------*)

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("{DebugDisplay}")>]
#endif
    type DenseMatrix<'a> = 
     { opsDM : 'a opsData;
#if CLI_AT_LEAST_2_0
       arrDM : 'a[,]
#else
       arrDM : 'a array array
#endif
      } with
    //  new(opsDM,arrDM) = {opsDM=opsDM;arrDM=arrDM}
      //interface DMGUatrix<'a> 

      member m.ElementOps = 
        match m.opsDM with 
        | None -> raise (new System.NotSupportedException("The element type carried by this matrix does not support numeric operations"))
        | Some a -> a
     end


//----------------------------------------------------------------------------
// type SparseMatrix<'a>
//--------------------------------------------------------------------------*)

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("{DebugDisplay}")>]
#endif
    type SparseMatrix<'a> = 
        { opsSM : 'a opsData;
          valsSM : 'a array;
          offsetsSM : int array; (* nrows + 1 elements *)
          ncolsSM : int;
          colsSM : int array; } 

        member m.ElementOps = 
              match m.opsSM with 
              | None -> raise (new System.NotSupportedException("The element type carried by this matrix does not support numeric operations"))
              | Some a -> a
(*
        override x.GetHashCode() = hash(x.valsSM)
        override x.Equals(yobj) = 
            let y = unbox<SparseMatrix<'a>>(yobj) in
            (x.valsSM = y.valsSM) &&
            (x.offsetsSM = y.offsetsSM) &&
            (x.ncolsSM = y.ncolsSM) &&
            (x.colsSM = y.colsSM) 
*)


//----------------------------------------------------------------------------
// type Matrix<'a>
//--------------------------------------------------------------------------*)

#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("{DebugDisplay}")>]
#endif
    type Matrix<'a> = 
        | Dense of DenseMatrix<'a>
        | Sparse of SparseMatrix<'a>
        interface System.IComparable
        interface IStructuralHash
        interface Microsoft.FSharp.Text.StructuredFormat.IFormattable 

        member m.ElementOps = match m with Dense mr -> mr.ElementOps | Sparse mr -> mr.ElementOps


//----------------------------------------------------------------------------
// type RowVector<'a>
//--------------------------------------------------------------------------*)
    
#if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("{DebugDisplay}")>]
#endif
    type RowVector<'a> = 
        { opsRV : INumeric<'a> option;
          arrRV : 'a array }
        interface System.IComparable
        interface IStructuralHash
        interface Microsoft.FSharp.Text.StructuredFormat.IFormattable 
        override x.GetHashCode() = hash(x.arrRV)
        override x.Equals(yobj) = (x.arrRV = (unbox yobj).arrRV)

        member m.Length = m.arrRV.Length
        member m.ElementOps = 
            match m.opsRV with 
            | None -> raise (new System.NotSupportedException("The element type carried by this row vector does not support numeric operations"))
            | Some a -> a

//----------------------------------------------------------------------------
// type Vector<'a>
//--------------------------------------------------------------------------*)

    #if CLI_AT_LEAST_2_0
    [<System.Diagnostics.DebuggerDisplay("{DebugDisplay}")>]
    #endif
    type Vector<'a> = 
        { opsV : INumeric<'a> option;
          arrV : 'a array }
        interface System.IComparable
        interface IStructuralHash
        interface Microsoft.FSharp.Text.StructuredFormat.IFormattable 
        override x.GetHashCode() = hash(x.arrV)
        override x.Equals(yobj) = (x.arrV = (unbox yobj).arrV)

        member m.Length = m.arrV.Length
        member m.ElementOps = 
            match m.opsV with 
            | None -> raise (new System.NotSupportedException("The element type carried by this vector does not support numeric operations"))
            | Some a -> a


//----------------------------------------------------------------------------
// module RawMatrixOps
//--------------------------------------------------------------------------*)

    module RawMatrixOps = begin


//----------------------------------------------------------------------------
// module RawMatrixOps.Unspecialized
//--------------------------------------------------------------------------*)
      
        module Unspecialized = begin

          open GlobalAssociations

          type 'a opsData = INumeric<'a> option

          let opsOfOpsData (d : 'a opsData)  =
             match d with 
             | None -> raise (new System.NotSupportedException("The element type '"+(type 'a).ToString()+"' carried by this vector or matrix does not support numeric operations (i.e. does not have a registered numeric association)"))
             | Some a -> a

          let getNormOps (ops:INumeric<'a>) = 
            match box ops with
              | :? INormFloat<'a> as ops -> ops
              | _                   -> raise (new System.NotSupportedException("The element type '"+(type 'a).ToString()+"' carried by this vector or matrix does not support the INormFloat<_> operation (i.e. does not have a registered numeric association that supports this type)"))

          let mkDMGU ops arr = { opsDM=ops; arrDM=arr }
#if CLI_AT_LEAST_2_0
          let nrowsDMGU a = a.arrDM.GetLength(0)
          let ncolsDMGU a = a.arrDM.GetLength(1)
#else
          let nrowsDMGU a = a.arrDM.Length
          let ncolsDMGU a = a.arrDM.(0).Length
#endif
          let nrowsSMGU a = Array.length a.offsetsSM - 1
          let ncolsSMGU a = a.ncolsSM
          let inline dimDMGU a = nrowsDMGU a,ncolsDMGU a
          let inline dimSMGU a = nrowsSMGU a,ncolsSMGU a
          let eopsDMGU (a: DenseMatrix<_>) = a.ElementOps
          let eopsSMGU (a: SparseMatrix<_>) = a.ElementOps

          let mkRVGU ops arr = {opsRV = ops; arrRV=arr}
          let dimRVGU a = a.arrRV.Length
          let ncolsRVGU a = a.arrRV.Length
          let eopsRVGU (a: RowVector<_>) = a.ElementOps

          let mkVGU ops arr = {opsV = ops; arrV=arr}
          let dimVGU a = a.arrV.Length
          let nrowsVGU a = a.arrV.Length
          let eopsVGU (a: Vector<_>) = a.ElementOps

#if CLI_AT_LEAST_2_0
          let inline getDMGUA  arrDM i j   = arrDM.(i,j)
          let inline setDMGUA  arrDM i j x = arrDM.(i,j) <- x
#else
          let inline getDMGUA  arrDM i j   = arrDM.(i).(j)
          let inline setDMGUA  arrDM i j x = arrDM.(i).(j) <- x
#endif
          let inline getRVGUA arrRV i     = arrRV.(i)
          let inline setRVGUA arrRV i   x = arrRV.(i) <- x
          let inline getVGUA  arrV   j   = arrV.(j)
          let inline setVGUA  arrV   j x = arrV.(j) <- x

          let inline getDMGU  m i j   = getDMGUA m.arrDM i j
          let inline setDMGU  m i j x = setDMGUA m.arrDM i j x
          let inline getRVGU m i     = getRVGUA m.arrRV i
          let inline setRVGU m i   x = setRVGUA m.arrRV i x
          let inline getVGU  m   j   = getVGUA m.arrV j
          let inline setVGU  m   j x = setVGUA m.arrV j x

          let inline createArray m = 
            Array.zero_create m

          let inline createArray2 m n = 
#if CLI_AT_LEAST_2_0
            CompatMatrix.zero_create m n
#else
            Array.init m (fun i -> Array.zero_create n) 
#endif

          let inline assignArray2 m n f arr = 
            for i = 0 to m - 1 do 
              for j = 0 to n - 1 do 
#if CLI_AT_LEAST_2_0
                arr.(i,j) <- f i j
#else
                arr.(i).(j) <- f i j
#endif
              done;
            done

          let inline setArray2 arrDM i j x = 
#if CLI_AT_LEAST_2_0
            arrDM.(i,j) <- x
#else
            arrDM.(i).(j) <- x
#endif

          let inline assignDMGU f a = 
            assignArray2 (nrowsDMGU a) (ncolsDMGU a) f a.arrDM
          
          let inline assignArray m f arr = 
            for i = 0 to m - 1 do 
                arr.(i) <- f i
            done

          let inline assignVGU f a = 
            assignArray (nrowsVGU a) f a.arrV
          
          let inline assignRVGU f a = 
            assignArray (ncolsRVGU a) f a.arrRV
          
          let inline createDMGU ops m n f = (* inline eliminates unknown f call *)
            let arr = createArray2 m n 
            assignArray2 m n f arr;
            mkDMGU ops arr
          
          let createRVGU ops m f = 
            let arr = createArray m 
            assignArray m f arr;
            mkRVGU ops arr
          
          let inline createVGU ops m f = (* inline eliminates unknown f call *)
            let arr = createArray m 
            assignArray m f arr;
            mkVGU ops arr

          (* create a matrix from a sparse sequence *)
          let init_sparseSMGU maxi maxj ops s = 
#if CLI_AT_MOST_1_1
              failwith "sparse matrices not implemented on .NET 1.x"
#else

              (* nb. could use sorted dictionary but that is in System.dll *)
              let tab = Array.create maxi null
              let count = ref 0
              for (i,j,v) in s do
                  if i < 0 || i >= maxi || j <0 || j >= maxj then failwith "initial value out of range";
                  count := !count + 1;
                  let tab2 = 
                      match tab.[i] with 
                      | null -> 
                          let tab2 = new System.Collections.Generic.Dictionary<_,_>(3) 
                          tab.[i] <- tab2;
                          tab2
                      | tab2 -> tab2
                  tab2.[j] <- v
              // optimize this line....
              let offsAcc =  
                 let rowsAcc = Array.zero_create (maxi + 1)
                 let mutable acc = 0 
                 for i = 0 to maxi-1 do 
                    rowsAcc.[i] <- acc;
                    acc <- match tab.[i] with 
                            | null -> acc
                            | tab2 -> acc+tab.[i].Count
                 rowsAcc.[maxi] <- acc;
                 rowsAcc
                 
              let colsA,valsA = 
                 (* sort a dictionary's entries by key/value pair *)
                 let sortd (tab : System.Collections.Generic.Dictionary<_,_>) =
                   let tab = tab |> Seq.to_array 
                   tab |> Array.sort (fun kvp1 kvp2 -> compare kvp1.Key kvp2.Key)
                   tab
                 let colsAcc = new ResizeArray<_>(!count)
                 let valsAcc = new ResizeArray<_>(!count)
                 for i = 0 to maxi-1 do 
                    match tab.[i] with 
                    | null -> ()
                    | tab2 -> tab2 |> sortd |> Array.iter (fun kvp -> colsAcc.Add(kvp.Key); valsAcc.Add(kvp.Value));
                 colsAcc.ToArray(), valsAcc.ToArray()
              { opsSM = ops; 
                offsetsSM=offsAcc; 
                ncolsSM = maxj;
                colsSM=colsA;
                valsSM=valsA }
#endif
          
          let zeroizeDMGUA arr  m n : DenseMatrix<'a> = 
              let opsData = GetNumericAssociationOption<'a>() 
              let ops = opsOfOpsData opsData 
              let zero = ops.Zero 
              assignArray2 m n (fun i j -> zero) arr;
              mkDMGU opsData arr

          let zeroizeArray opsData arr m  = 
              let ops = opsOfOpsData opsData 
              let zero = ops.Zero 
              assignArray m (fun i -> zero) arr

          let zeroizeVGUA arr m  : Vector<'a> = 
              let opsData = GetNumericAssociationOption<'a>() 
              zeroizeArray opsData arr m;
              mkVGU opsData arr

          let zeroizeRVGUA arr m  : RowVector<'a> = 
              let opsData = GetNumericAssociationOption<'a>() 
              zeroizeArray opsData arr m;
              mkRVGU opsData arr

          let listDMGU ops xss =
            let m = List.length xss
            match xss with 
            | [] -> invalid_arg "listDMGU: given empty list"
            | h :: t -> 
              let n = List.length h
              if not (List.for_all (fun xs -> List.length xs=n) t) then invalid_arg "matrix: given jagged lists";
        #if CLI_AT_LEAST_2_0
              let arrDM = (CompatMatrix.zero_create m n) 
              List.iteri (fun i rw -> List.iteri (fun j x -> arrDM.(i,j) <- x) rw) xss;
        #else
              let arrDM = Array.init m (fun i -> Array.zero_create n) 
              List.iteri (fun i rw -> List.iteri (fun j x -> arrDM.(i).(j) <- x) rw) xss;
        #endif
              mkDMGU ops arrDM
          
          let listRVGU ops xs = mkRVGU ops (Array.of_list xs) 
          let listVGU ops xs = mkVGU ops (Array.of_list xs) 

          let seqDMGU ops xss = listDMGU ops (xss |> Seq.to_list |> List.map Seq.to_list)
          let seqVGU  ops xss = listVGU ops (xss |> Seq.to_list)
          let seqRVGU ops xss = listRVGU ops (xss |> Seq.to_list)

          let inline bopDMGU bop a b = (* pointwise binary operator *)
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let nB = ncolsDMGU b 
            let mB = nrowsDMGU b
            if nA<>nB || mA<>mB then invalid_arg "different dimensions at binary operator";
            let arrA = a.arrDM 
            let arrB = b.arrDM 
            createDMGU a.opsDM mA nA (fun i j -> bop (getDMGUA arrA i j) (getDMGUA arrB i j))

          let kminSMGU a i = a.offsetsSM.[i]
          let kmaxSMGU a i = a.offsetsSM.[i+1]
              
          let getSMGU  (a:SparseMatrix<_>) i j   = 
              let imax = nrowsSMGU a
              let jmax = ncolsSMGU a
              if j < 0 || j >= jmax || i < 0 || i >= imax then raise (new System.ArgumentOutOfRangeException()) else
              let kmin = kminSMGU a i
              let kmax = kmaxSMGU a i
              let rec loopRow k =
                (* note: could do a binary chop here *)
                if k >= kmax then a.ElementOps.Zero else
                let j2 = a.colsSM.[k]
                if j < j2 then a.ElementOps.Zero else
                if j = j2 then a.valsSM.[k] else 
                loopRow (k+1)
              loopRow kmin

          let nonzero_entriesSMGU  (a:SparseMatrix<_>) = 
#if CLI_AT_MOST_1_1
            failwith "sparse matrices not implemented on .NET 1.x"
#else
              // This is heavily used, and this version is much faster than
              // the sequence operators.
              let entries = new ResizeArray<_>(Array.length a.colsSM)
              let imax = nrowsSMGU a
              let ops = a.ElementOps 
              let zero = ops.Zero
              for i in 0 .. imax - 1 do
                let kmin = kminSMGU a i
                let kmax = kmaxSMGU a i
                for k in kmin .. kmax - 1 do
                    let j = a.colsSM.[k]
                    let v = a.valsSM.[k]
                    if v <> zero then
                      entries.Add((i,j,v))
              (entries :> seq<_>)
#endif

          let nonzero_entriesDMGU  (a:DenseMatrix<_>) = 
              let imax = nrowsDMGU a
              let jmax = ncolsDMGU a
              let ops = a.ElementOps 
              let zero = ops.Zero
              { for i in 0 .. imax - 1 do 
                  for j in 0 .. jmax - 1 do 
                      let v = getDMGU a i j 
                      if v <> zero then
                           yield (i,j,v) }

          let sparse_nyi() = failwith "this operation is invalid on sparse matrices"
          let sparse_not_mutable() = failwith "sparse matrices are not mutable"
          let setSMGU  (a:SparseMatrix<_>) i j x = sparse_not_mutable()

          // pointwise operation on two sparse matrices. bop must be zero-zero-preserving, i.e. (bop 0 0 = 0) 
          let bopSMGU bop (a:SparseMatrix<_>) (b:SparseMatrix<_>) = 
#if CLI_AT_MOST_1_1
            failwith "sparse matrices not implemented on .NET 1.x"
#else
            let ops = a.ElementOps 
            let zero = ops.Zero
            let imin1 = 0
            let imax1 = nrowsSMGU a  
            let imax2 = nrowsSMGU b
            let jmax1 = a.ncolsSM
            let jmax2 = b.ncolsSM
            if imax1 <> imax2 || jmax1 <> jmax2 then invalid_arg "different dimensions at binary operator";
            let imin = 0
            let imax = imax1
            let jmax = jmax1
            let rowsR = Array.zero_create (imax+1)
            let colsR = new ResizeArray<_>(max (Array.length a.colsSM) (Array.length b.colsSM))
            let valsR = new ResizeArray<_>(max (Array.length a.valsSM) (Array.length b.valsSM))
            let jMax = ref 0 
            let rec loopRows i  = 
              rowsR.[i] <- valsR.Count;            
              if i >= imax1 then () else
              let kmin1 = kminSMGU a i
              let kmax1 = kmaxSMGU a i 
              let kmin2 = kminSMGU b i
              let kmax2 = kmaxSMGU b i
              let rec loopRow k1 k2  =
                if k1 >= kmax1 && k2 >= kmax2 then () else
                let j1 = if k1 >= kmax1 then jmax else a.colsSM.[k1]
                let j2 = if k2 >= kmax2 then jmax else b.colsSM.[k2]
                let v1 = if j1 <= j2 then a.valsSM.[k1] else zero
                let v2 = if j2 <= j1 then b.valsSM.[k2] else zero
                let jR = min j1 j2
                let vR = bop v1 v2
                (* if vR <> zero then  *)
                colsR.Add(jR);
                valsR.Add(vR);
                loopRow (if j1 <= j2 then k1+1 else k1) (if j2 <= j1 then k2+1 else k2)
              loopRow kmin1 kmin2;
              loopRows (i+1) 
            loopRows imin;
            { opsSM = a.opsSM; 
              offsetsSM=rowsR; 
              ncolsSM = a.ncolsSM;
              colsSM=colsR.ToArray();
              valsSM=valsR.ToArray() }
#endif

          let inline bopRVGU bop a b = (* pointwise binary operator *)
            let mA = dimRVGU a
            let mB = dimRVGU b
            if mA<>mB then invalid_arg "different dimensions at binary operator";
            createRVGU a.opsRV mA (fun i -> bop (getRVGU a i) (getRVGU b i))

          let inline bopVGU bop a b = (* pointwise binary operator *)
            let mA = dimVGU a
            let mB = dimVGU b
            if mA<>mB then invalid_arg "different dimensions at binary operator";
            createVGU a.opsV mA (fun i -> bop (getVGU a i) (getVGU b i))

          let inline unopDMGU f a =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a 
            let arrA = a.arrDM 
            createDMGU a.opsDM mA nA (fun i j -> f (getDMGUA arrA i j))

          let inline unopRVGU f a =
            let mA = dimRVGU a
            let arrA = a.arrRV 
            createRVGU a.opsRV mA (fun j -> f (getRVGUA arrA j))

          let inline unopVGU f a =
            let mA = dimVGU a 
            let arrA = a.arrV 
            createVGU a.opsV mA (fun i -> f (getVGUA arrA i))

          let unopSMGU f (a:SparseMatrix<_>) = (* pointwise zero-zero-preserving binary operator (f 0 = 0) *)
            { a with 
                offsetsSM=Array.copy a.offsetsSM; 
                colsSM=Array.copy a.colsSM;
                valsSM=Array.map f a.valsSM }

          (* strictly speaking these are non mutable so no copy is ever needed.  But implementing it *)
          (* anyway in case we move to mutability *)
          let copySMGU (a:SparseMatrix<_>) = 
            { a with 
                offsetsSM=Array.copy a.offsetsSM; 
                colsSM=Array.copy a.colsSM;
                valsSM=Array.copy a.valsSM }

          let addDMGU a b = let ops = eopsDMGU a in bopDMGU (fun x y -> ops.Add(x, y)) a b
          let addSMGU a b = let ops = eopsSMGU a in bopSMGU (fun x y -> ops.Add(x, y)) a b
          let addRVGU a b = let ops = eopsRVGU a in bopRVGU (fun x y -> ops.Add(x, y)) a b
          let addVGU  a b = let ops = eopsVGU  a in bopVGU  (fun x y -> ops.Add(x, y)) a b 

          let subDMGU a b = let ops = eopsDMGU a in bopDMGU (fun x y -> ops.Subtract(x, y)) a b
          let subSMGU a b = let ops = eopsSMGU a in bopSMGU (fun x y -> ops.Subtract(x, y)) a b
          let subRVGU a b = let ops = eopsRVGU a in bopRVGU (fun x y -> ops.Subtract(x, y)) a b
          let subVGU  a b = let ops = eopsVGU  a in bopVGU  (fun x y -> ops.Subtract(x, y)) a b 

          ///Point-wise multiplication 
          let cptMulDMGU a b = let ops = eopsDMGU a in bopDMGU  (fun x y -> ops.Multiply(x, y)) a b
          let cptMulSMGU a b = let ops = eopsSMGU a in bopSMGU  (fun x y -> ops.Multiply(x, y)) a b
          let cptMulRVGU a b = let ops = eopsRVGU a in bopRVGU (fun x y -> ops.Multiply(x, y)) a b
          let cptMulVGU  a b = let ops = eopsVGU  a in bopVGU  (fun x y -> ops.Multiply(x, y)) a b

          let cptMaxDMGU  a b = let ops = eopsDMGU  a in bopDMGU  max a b
          let cptMinDMGU  a b = let ops = eopsDMGU  a in bopDMGU  min a b
          let cptMaxSMGU  a b = let ops = eopsSMGU  a in bopSMGU  max a b
          let cptMinSMGU  a b = let ops = eopsSMGU  a in bopSMGU  min a b

          let cptMaxVGU  a b = let ops = eopsVGU  a in bopVGU  max a b
          let cptMinVGU  a b = let ops = eopsVGU  a in bopVGU  min a b

          let add (ops : INumeric<'a>) x y = ops.Add(x,y) 
          let sub (ops : INumeric<'a>) x y = ops.Subtract(x,y) 
          let mul (ops : INumeric<'a>) x y = ops.Multiply(x,y) 

          let inline sumRGU (ops : INumeric<_>) f r = 
            let zero = ops.Zero 
            r |> RangeOps.foldR (fun z k -> add ops z (f k)) zero

          let mulDMGU a b =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let nB = ncolsDMGU b 
            let mB = nrowsDMGU b
            if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
            let ops = a.ElementOps 
            let arrA = a.arrDM 
            let arrB = b.arrDM 
            createDMGU a.opsDM mA nB
              (fun i j -> (0,nA - 1) |> sumRGU ops (fun k -> mul ops (getDMGUA arrA i k) (getDMGUA arrB k j)))

          let debug = false
          
          // SParse matrix multiplication algorithm. inline to get specialization at the 'double' type
          let inline generic_mulSMGU zero add mul a b =
#if CLI_AT_MOST_1_1
            failwith "sparse matrices not implemented on .NET 1.x"
#else
            let nA = ncolsSMGU a
            let mA = nrowsSMGU a
            let nB = ncolsSMGU b 
            let mB = nrowsSMGU b
            if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
            let C = new ResizeArray<_>()
            let jC = new ResizeArray<_>()
            let MA1 = mA + 1 
            let offsAcc = Array.zero_create MA1
            let index = Array.zero_create mA
            let temp = Array.create mA zero
            let ptr = new System.Collections.Generic.Dictionary<_,_>(11)
            if debug then Microsoft.FSharp.Text.Printf.printf "start, #items in result = %d, #offsAcc = %d, mA = %d\n" jC.Count offsAcc.Length mA;

            let mutable mlast = 0
            for i = 0 to mA-1 do
              if debug then Microsoft.FSharp.Text.Printf.printf "i = %d, mlast = %d\n" i mlast;
              offsAcc.[i] <- mlast
              
              let kmin1 = kminSMGU a i
              let kmax1 = kmaxSMGU a i
              if kmin1 < kmax1 then 
                  let mutable itemp = 0
                  let mutable ptrNeedsClear = true // clear the ptr table on demand.  
                  for j = kmin1 to kmax1 - 1 do
                    if debug then Microsoft.FSharp.Text.Printf.printf "  j = %d\n" j;
                    let ja_j = a.colsSM.[j]
                    let kmin2 = kminSMGU b ja_j
                    let kmax2 = kmaxSMGU b ja_j
                    for k = kmin2 to kmax2 - 1 do
                      let jb_k = b.colsSM.[k]
                      if debug then Microsoft.FSharp.Text.Printf.printf "    i = %d, j = %d, k = %d, ja_j = %d, jb_k = %d\n" i j k ja_j jb_k;
                      let va = a.valsSM.[j] 
                      let vb = b.valsSM.[k]
                      if debug then Microsoft.FSharp.Text.Printf.printf "    va = %O, vb = %O\n" va vb;
                      let summand = mul va vb
                      if debug then Microsoft.FSharp.Text.Printf.printf "    summand = %O\n" summand;
                      if ptrNeedsClear then (ptr.Clear();ptrNeedsClear <- false);

                      if not (ptr.ContainsKey(jb_k)) then
                          if debug then Microsoft.FSharp.Text.Printf.printf "    starting entry %d\n" jb_k;
                          ptr.[jb_k] <- itemp
                          let ptr_jb_k = itemp
                          temp.[ptr_jb_k] <- summand
                          index.[ptr_jb_k] <- jb_k
                          itemp <- itemp + 1
                      else
                          if debug then Microsoft.FSharp.Text.Printf.printf "    adding to entry %d\n" jb_k;
                          let ptr_jb_k = ptr.[jb_k]
                          temp.[ptr_jb_k] <- add temp.[ptr_jb_k] summand
                    done
                  done
                  for s = 0 to itemp-1 do
                      if debug then Microsoft.FSharp.Text.Printf.printf "  writing value %O at index %d to result matrix\n" temp.[s] index.[s];
                      C.Add(temp.[s])
                      jC.Add(index.[s])
                  done
                  if debug then Microsoft.FSharp.Text.Printf.printf " itemp = %d, mlast = %d\n" itemp mlast;
                  mlast <- mlast + itemp 
            done
            offsAcc.[mA] <- mlast;
            if debug then Microsoft.FSharp.Text.Printf.printf "done, #items in result = %d, #offsAcc = %d, mA = %d\n" jC.Count offsAcc.Length mA;
            { opsSM = a.opsSM; 
              offsetsSM=offsAcc; 
              ncolsSM = nB;
              colsSM=jC.ToArray();
              valsSM=C.ToArray() }
#endif

          let mulSMGU (a: SparseMatrix<_>) b =
            let ops = a.ElementOps 
            let zero = ops.Zero
            generic_mulSMGU zero (add ops) (mul ops) a b


          let mulRVVGU a b =
            let mA = dimRVGU a 
            let nB = dimVGU b 
            if mA<>nB then invalid_arg "incompatible dimensions for matrix multiplication";
            let ops = (eopsRVGU a) 
            (0,mA - 1) |> sumRGU ops (fun k -> mul ops (getRVGU a k) (getVGU b k))

          let rowvecDMGU x = createDMGU x.opsRV 1          (dimRVGU x) (fun _ j -> getRVGU x j) 
          let vectorDMGU x = createDMGU x.opsV  (dimVGU x) 1           (fun i _ -> getVGU x i) 

          let mulVRVGU a b = mulDMGU (vectorDMGU a) (rowvecDMGU b)

          let mulRVDMGU a b =
            let    nA = dimRVGU a 
            let nB = ncolsDMGU b
            let mB = nrowsDMGU b 
            if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
            let ops = (eopsRVGU a) 
            let arrA = a.arrRV 
            let arrB = b.arrDM 
            createRVGU a.opsRV nB
              (fun j -> (0,nA - 1) |> sumRGU ops (fun k -> mul ops (getRVGUA arrA k) (getDMGUA arrB k j)))

          let mulDMVGU a b =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a 
            let mB    = dimVGU b 
            if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
            let ops = b.ElementOps 
            let arrA = a.arrDM 
            let arrB = b.arrV 
            createVGU b.opsV mA
              (fun i -> (0,nA - 1) |> sumRGU ops (fun k -> mul ops (getDMGUA arrA i k) (getVGUA arrB k)))

          let mulSMVGU a b =
            let nA = ncolsSMGU a 
            let mA = nrowsSMGU a 
            let mB    = dimVGU b 
            if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
            let ops = b.ElementOps 
            let zero = ops.Zero
            createVGU b.opsV mA (fun i -> 
              let mutable acc = zero
              for k = kminSMGU a i to kmaxSMGU a i - 1 do
                let j = a.colsSM.[k]
                let v = a.valsSM.[k] 
                acc <- add ops acc (mul ops v (getVGU b j));
              acc)

          let mulRVSMGU a b =
            let nA = ncolsSMGU b
            let mA = nrowsSMGU b 
            let mB    = dimRVGU a 
            if mA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
            let ops = b.ElementOps 
            let arr = createArray nA 
            zeroizeArray a.opsRV arr nA;
            for i = 0 to mA - 1 do
              for k = kminSMGU b i to kmaxSMGU b i - 1 do
                let j = b.colsSM.[k]
                let v = b.valsSM.[k] 
                arr.[j] <- add ops arr.[j] (mul ops (getRVGU a i) v)
            mkRVGU a.opsRV arr


          let scaleDMGU k a = let ops = eopsDMGU a in unopDMGU (fun x -> ops.Multiply(k,x)) a
          let scaleRVGU k a = let ops = eopsRVGU a in unopRVGU (fun x -> ops.Multiply(k,x)) a
          let scaleVGU  k a = let ops = eopsVGU  a in unopVGU  (fun x -> ops.Multiply(k,x)) a
          let scaleSMGU k a = let ops = eopsSMGU a in unopSMGU (fun x -> ops.Multiply(k,x)) a
          let negDMGU     a = let ops = eopsDMGU a in unopDMGU (fun x -> ops.Negate(x)) a
          let negRVGU     a = let ops = eopsRVGU a in unopRVGU (fun x -> ops.Negate(x)) a
          let negVGU      a = let ops = eopsVGU  a in unopVGU  (fun x -> ops.Negate(x)) a
          let negSMGU     a = let ops = eopsSMGU a in unopSMGU (fun x -> ops.Negate(x)) a

          let mapDMGU f (a : DenseMatrix<'a>) : DenseMatrix<'a> = 
            let nA = ncolsDMGU a
            let mA = nrowsDMGU a
            let arrA = a.arrDM 
            createDMGU a.opsDM mA nA (fun i j -> f (getDMGUA arrA i j))

          let mapVGU f a = 
            let mA= dimVGU a
            createVGU a.opsV mA (fun i -> f (getVGU a i))

          let copyDMGU (a : DenseMatrix<'a>) : DenseMatrix<'a> = 
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let arrA = a.arrDM 
            createDMGU a.opsDM mA nA (fun i j -> getDMGUA arrA i j)

          let copyVGU a = 
            let mA= dimVGU a
            createVGU a.opsV mA (fun i -> getVGU a i)

          let copyRVGU a = 
            let mA= dimRVGU a
            createRVGU a.opsRV mA (fun i -> getRVGU a i)

          let to_denseSMGU (a:SparseMatrix<_>) = 
            let nA = ncolsSMGU a 
            let mA = nrowsSMGU a
            createDMGU a.opsSM mA nA (getSMGU a)
            
          let mapiDMGU f (a: DenseMatrix<'a>) : DenseMatrix<'a> = 
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let arrA = a.arrDM 
            createDMGU a.opsDM mA nA (fun i j -> f i j (getDMGUA arrA i j))

          let mapiRVGU f a = 
            let mA= dimRVGU a
            createRVGU a.opsRV mA (fun i -> f i (getRVGU a i))

          let mapiVGU f a = 
            let mA= dimVGU a
            createVGU a.opsV mA (fun i -> f i (getVGU a i))

          let permuteVGU (p:Permutation) a = 
            let mA= dimVGU a 
            createVGU a.opsV mA (fun i -> getVGU a p.[i])

          let permuteRVGU (p:Permutation) a = 
            let mA= dimRVGU a 
            createRVGU a.opsRV mA (fun i -> getRVGU a p.[i])

          let inline inplace_mapiDMGU f a = 
            let arrA = a.arrDM 
            assignDMGU (fun i j -> f i j (getDMGUA arrA i j)) a

          let inline inplace_mapiRVGU f a = 
            assignRVGU (fun i -> f i (getRVGU a i)) a

          let inline inplace_mapiVGU f a = 
            assignVGU (fun i -> f i (getVGU a i)) a

          let inline foldDMGU f z a =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let arrA = a.arrDM 
            let mutable acc = z
            for i = 0 to mA-1 do
                for j = 0 to nA-1 do 
                   acc <- f acc (getDMGUA arrA i j)
            acc
          
          let inline foldVGU f z a =
            let mA = dimVGU a
            let mutable acc = z
            for i = 0 to mA-1 do acc <- f acc (getVGU a i)
            acc
          
          let inline foldiDMGU f z a =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let arrA = a.arrDM 
            let mutable acc = z
            for i = 0 to mA-1 do
                for j = 0 to nA-1 do 
                   acc <- f i j acc (getDMGUA arrA i j)
            acc
          
          let inline foldiVGU f z a =
            let mA = dimVGU a
            let mutable acc = z
            for i = 0 to mA-1 do acc <- f i acc (getVGU a i)
            acc
          
          let rec forallR f (n,m) = (n > m) || (f n && forallR f (n+1,m))
          let rec existsR f (n,m) = (n <= m) && (f n || existsR f (n+1,m))
          
          let foralliDMGU pred a =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let arrA = a.arrDM 
            (0,mA-1) |> forallR  (fun i ->
            (0,nA-1) |> forallR  (fun j ->
            pred i j (getDMGUA arrA i j)))

          let foralliVGU pred a =
            let mA = dimVGU a
            (0,mA-1) |> forallR  (fun i ->
            pred i (getVGU a i))

          let existsiDMGU pred a =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let arrA = a.arrDM 
            (0,mA-1) |> existsR (fun i ->
            (0,nA-1) |> existsR (fun j ->
            pred i j (getDMGUA arrA i j)))

          let existsiVGU pred a =
            let mA = dimVGU a
            (0,mA-1) |> existsR (fun i ->
            pred i (getVGU a i))

          let sumDMGU  (a:DenseMatrix<_>) = 
            let ops = a.ElementOps 
            let zero = ops.Zero 
            foldDMGU (fun acc aij -> add ops acc aij) zero a

          let sumSMGU  (a:SparseMatrix<_>) = 
            let ops = a.ElementOps 
            let zero = ops.Zero 
            a |> nonzero_entriesSMGU |> Seq.fold (fun acc (_,_,aij) -> add ops acc aij) zero 

          let sumVGU  a = 
            let ops = (eopsVGU a) 
            let zero = ops.Zero 
            foldVGU (fun acc ai -> add ops acc ai) zero a

          let prodDMGU (a:DenseMatrix<_>) = 
            let ops = a.ElementOps 
            let one = ops.One 
            foldDMGU (fun acc aij -> mul ops acc aij) one a

          let prodSMGU  (a:SparseMatrix<_>) = a |> to_denseSMGU |> prodDMGU

          let inline fold2DMGU f z a b =
            let nA = ncolsDMGU a 
            let mA = nrowsDMGU a
            let nB = ncolsDMGU b 
            let mB = nrowsDMGU b
            if nA <> nB || mA <> mB then invalid_arg "different dimensions";
            let arrA = a.arrDM 
            let arrB = b.arrDM 
            let mutable acc = z
            for i = 0 to mA-1 do
                for j = 0 to nA-1 do 
                   acc <- f acc (getDMGUA arrA i j) (getDMGUA arrB i j)
            acc

          let inline fold2VGU f z a b =
            let mA = dimVGU a
            let mB = dimVGU b
            if  mA <> mB then invalid_arg "different dimensions";
            let mutable acc = z
            for i = 0 to mA-1 do acc <- f acc (getVGU a i) (getVGU b i)
            acc

          let dotDMGU (a:DenseMatrix<_>) b =
            let ops = a.ElementOps 
            fold2DMGU (fun z va vb -> add ops z (mul ops va vb)) ops.Zero a b

          let dotVGU a b =
            let ops = (eopsVGU a) 
            let zero = ops.Zero 
            fold2VGU  (fun z va vb -> add ops z (mul ops va vb)) zero a b 

          let normDMGU (a:DenseMatrix<_>) = 
            let normOps = getNormOps a.ElementOps
            foldDMGU (fun z aij -> z + ((normOps.Norm aij)**2.0)) 0.0 a |> sqrt

          let normSMGU (a:SparseMatrix<_>) = 
            let normOps = getNormOps a.ElementOps
            a |> nonzero_entriesSMGU |> Seq.fold (fun acc (_,_,aij) -> acc + ((normOps.Norm aij)**2.0)) 0.0 |> sqrt

          let inplace_addDMGU  (a:DenseMatrix<_>) b = 
            let ops = a.ElementOps 
            let arrB = b.arrDM 
            inplace_mapiDMGU  (fun i j x -> add ops x (getDMGUA arrB i j)) a
          
          let inplace_addVGU  a b = 
            let ops = eopsVGU a 
            inplace_mapiVGU  (fun i x   -> add ops x (getVGU  b i)) a
          let inplace_addRVGU a b = 
            let ops = eopsRVGU a 
            inplace_mapiRVGU (fun i x   -> add ops x (getRVGU b i)) a
          let inplace_subDMGU  (a:DenseMatrix<_>) b = 
            let ops = a.ElementOps 
            let arrB = b.arrDM 
            inplace_mapiDMGU  (fun i j x -> sub ops x (getDMGUA  arrB i j)) a
          let inplace_subVGU  a b = 
            let ops = eopsVGU a 
            inplace_mapiVGU  (fun i x   -> sub ops x (getVGU  b i)  ) a
          let inplace_subRVGU a b = 
            let ops = (eopsRVGU a) 
            inplace_mapiRVGU (fun i x   -> sub ops x (getRVGU b i)  ) a
          let inplace_cptMulDMGU  (a:DenseMatrix<_>) b = 
            let ops = a.ElementOps 
            let arrB = b.arrDM 
            inplace_mapiDMGU  (fun i j x -> mul ops x (getDMGUA  arrB i j)) a
          let inplace_cptMulVGU  a b = 
            let ops = (eopsVGU a) 
            inplace_mapiVGU  (fun i x   -> mul ops x (getVGU  b i ) ) a
          let inplace_cptMulRVGU a b = 
            let ops = (eopsRVGU a) 
            inplace_mapiRVGU (fun i x   -> mul ops x (getRVGU b i)  ) a
          let inplace_scaleDMGU  x (a:DenseMatrix<_>) = 
            let ops = a.ElementOps 
            inplace_mapiDMGU  (fun i j y -> ops.Multiply(x,y)) a
          let inplace_scaleVGU  x a = 
            let ops = (eopsVGU a) 
            inplace_mapiVGU  (fun i y   -> ops.Multiply(x,y)) a
          let inplace_scaleRVGU x a = 
            let ops = (eopsRVGU a) 
            inplace_mapiRVGU (fun i y   -> ops.Multiply(x,y)) a


          let wrapList (pre,mid,post) show = function
            | []    -> [pre;post]
            | [x]   -> [pre;show x;post]
            | x::xs -> [pre;show x] @ List.concat (List.map (fun x -> [mid;show x]) xs) @ [post]

          let showItem opsData  x = 
            try 
              let ops = opsOfOpsData opsData 
              ops.ToString(x,"g10",System.Globalization.CultureInfo.InvariantCulture) 
            with :? System.NotSupportedException -> (box x).ToString()
          
          let mapR f (n,m) = if m < n then [] else List.init (m-n+1) (fun i -> f (n+i))

          let rec upto a b = if a<= b then a::upto (a+1) b else []
          let primShowDMGU (sepX,sepR) (a : DenseMatrix<'e>) =
            let mA,nA = dimDMGU a 
            let ops = a.opsDM 
            let showLine i = wrapList ("[",";","]") (showItem ops) ((0,nA-1) |> mapR  (fun j -> getDMGU a i j)) |> String.concat ""
            wrapList ("matrix [",";"+sepX,"]"+sepR) showLine (upto 0 (mA-1)) |> String.concat ""

          let showDMGU     m = primShowDMGU ("\n","\n") m
          let debugShowDMGU m = primShowDMGU (""  ,""  ) m
          
          let showVGU s (a : Vector<_>) =
            let mA = dimVGU a
            let ops = a.opsV 
            wrapList (s+" [",";","]") (showItem ops) ((0,mA-1) |> mapR  (fun i -> getVGU a i)) |> String.concat "" 

          let showRVGU s (a : RowVector<_>) =
            let mA = dimRVGU a
            let ops = a.opsRV 
            wrapList (s+" [",";","]") (showItem ops) ((0,mA-1) |> mapR  (fun i -> getRVGU a i)) |> String.concat "" 
            
          let semiSuffixL itemL = itemL $$ rightL ";"
          let colsL itemLs = squareBracketL (aboveListL (List.map semiSuffixL itemLs))
          let rowL itemLs = squareBracketL (semiListL itemLs)
          let layoutOfVector (len,get) (env : IEnvironment) a = 
            let n = len a
            let project i = if i=n then None else Some (get a i,i+1)
            let objL x = env.GetLayout(box(x)) 
            let itemLs = unfoldL objL project 0 env.MaxColumns
            rowL itemLs
          
          let layoutOfVGU (env : IEnvironment) (a : Vector<_>) = 
            wordL "vector" -- layoutOfVector (nrowsVGU,getVGU) env a
          
          let layoutOfRVGU (env : IEnvironment) (a : RowVector<_>) = 
            wordL "rowvec" -- layoutOfVector (ncolsRVGU,getRVGU) env a

          let layoutOfDMGU (env : IEnvironment) (a : DenseMatrix<_>) = 
            let objL x = env.GetLayout(box(x)) 
            let n1,n2 = dimDMGU a 
            let project2 i j =
              if j>=n2 then None
              else Some (getDMGU a i j,j+1)
            let oneRowL i = unfoldL objL (project2 i) 0 env.MaxColumns |> rowL
            let project1 i = if i>=n1 then None else Some (i,i+1)
            let rowsL  = unfoldL oneRowL project1 0 env.MaxRows |> colsL
            wordL "matrix" -- rowsL
        end


//----------------------------------------------------------------------------
// module RawMatrixOps.DoubleSpecialized
//--------------------------------------------------------------------------*)
     
        module DoubleSpecialized = begin

          module GU = Unspecialized
          open Instances
          
          // Element type opsDM
          //type elem = float
          let zero = 0.0
          let one  = 1.0
          let inline sub (x:float) (y:float) = x - y
          let inline add (x:float) (y:float) = x + y
          let inline mul (x:float) (y:float) = x * y
          let inline neg (x:float) = -x

          // Private
          let inline ncolsDMDS a = GU.ncolsDMGU a
          let inline nrowsDMDS a = GU.nrowsDMGU a
          let inline dimRVDS a = GU.dimRVGU a
          let inline dimVDS a = GU.dimVGU a
          let inline getDMDSA a i j = GU.getDMGUA a i j
          let inline getRVDSA a i = GU.getRVGUA a i
          let inline getVDSA a i = GU.getVGUA a i
          let inline getRVDS a i = GU.getRVGU a i
          let inline getVDS a i = GU.getVGU a i
          
          // Specialized: these know the relevant set of 
          // ops without doing a table lookup based on runtime type
          let FloatOps = Some (FloatNumerics :> INumeric<float>)
          let inline initDMDS m n f = GU.createDMGU  FloatOps m n f
          let inline createRVDS m f  = GU.createRVGU FloatOps m f
          let inline createVDS m f   = GU.createVGU  FloatOps m f
          let inline mkDMDS  arr = GU.mkDMGU  FloatOps arr
          let inline mkRVDS arr = GU.mkRVGU FloatOps arr
          let inline mkVDS  arr = GU.mkVGU  FloatOps arr
          let inline listDMDS  ll = GU.listDMGU  FloatOps ll
          let inline listRVDS l  = GU.listRVGU  FloatOps l
          let inline listVDS  l  = GU.listVGU  FloatOps l
          let inline seqDMDS  ll = GU.seqDMGU  FloatOps ll
          let inline seqRVDS l  = GU.seqRVGU  FloatOps l
          let inline seqVDS  l  = GU.seqVGU  FloatOps l

          let constDMDS  m n x = GU.createDMGU  FloatOps m n (fun i j -> x)
          let constRVDS m x   = GU.createRVGU FloatOps m   (fun i -> x)
          let constVDS  m x   = GU.createVGU  FloatOps m   (fun i -> x)
          let scalarDMDS   x = constDMDS  1 1 x 
          let scalarRVDS  x = constRVDS 1   x 
          let scalarVDS   x = constVDS  1   x 

        #if CLI_AT_LEAST_2_0
          // Beware - when compiled with non-generic code createArray2 creates an array of null values,
          // not zero values.  Hence the optimized version can only be used when compiling with generics.
          let inline zeroDMDS m n = 
            let arr = GU.createArray2 m n 
            GU.mkDMGU FloatOps arr
        #endif    
          // Specialized: these inline down to the efficient loops we need
          let addDMDS     a b = GU.bopDMGU  add a b
          let addSMDS     a b = GU.bopSMGU  add a b
          let addRVDS    a b = GU.bopRVGU add a b
          let addVDS     a b = GU.bopVGU  add a b
          let subDMDS     a b = GU.bopDMGU  sub a b 
          let subSMDS     a b = GU.bopSMGU  sub a b 
          let mulSMDS     a b = GU.generic_mulSMGU zero add mul a b
          let subRVDS    a b = GU.bopRVGU sub a b 
          let subVDS     a b = GU.bopVGU  sub a b 
          let cptMulDMDS  a b = GU.bopDMGU  mul a b
          let cptMulSMDS  a b = GU.bopSMGU  mul a b
          let cptMulRVDS a b = GU.bopRVGU mul a b
          let cptMulVDS  a b = GU.bopVGU  mul a b
          type smatrix = SparseMatrix<float>
          type dmatrix = DenseMatrix<float>
          type vector = Vector<float>
          type rowvec = RowVector<float>
          let cptMaxDMDS  (a:dmatrix) (b:dmatrix) = GU.bopDMGU  max a b
          let cptMinDMDS  (a:dmatrix) (b:dmatrix) = GU.bopDMGU  min a b
          let cptMaxSMDS  (a:smatrix) (b:smatrix) = GU.bopSMGU  max a b
          let cptMinSMDS  (a:smatrix) (b:smatrix) = GU.bopSMGU  min a b
          let cptMaxVDS  (a:vector) (b:vector) = GU.bopVGU  max a b
          let cptMinVDS  (a:vector) (b:vector) = GU.bopVGU  min a b

          // Don't make any mistake about these ones re. performance.
          let mulDMDS a b =
              let nA = ncolsDMDS a 
              let mA = nrowsDMDS a
              let nB = ncolsDMDS b 
              let mB = nrowsDMDS b
              if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
              // Beware - when compiled with non-generic code createArray2 creates an array of null values,
              // not zero values.  However we fill in all the entries to the array, hence OK.
              let arr = GU.createArray2 mA nB 
              let arrA = a.arrDM 
              let arrB = b.arrDM 
              for i = 0 to mA - 1 do 
                  for j = 0 to nB - 1 do 
                      let mutable r = 0.0 
                      for k = 0 to mB - 1 do 
                          r <- r + mul (getDMDSA arrA i k) (getDMDSA arrB k j)
                      GU.setArray2 arr i j r
              mkDMDS arr

          let mulRVDMDS a b =
              let nA = dimRVDS a 
              let nB = ncolsDMDS b 
              let mB = nrowsDMDS b
              if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
              let arr = Array.zero_create nB 
              let arrA = a.arrRV 
              let arrB = b.arrDM 
              for j = 0 to nB - 1 do 
                  let mutable r = 0.0 
                  for k = 0 to mB - 1 do 
                      r <- r + mul (getRVDSA arrA k) (getDMDSA arrB k j)
                  arr.[j] <- r
              mkRVDS arr

          let mulMVDS a b =
              let nA = ncolsDMDS a 
              let mA = nrowsDMDS a
              let mB = dimVDS b 
              if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
              let arr = Array.zero_create mA 
              let arrA = a.arrDM 
              let arrB = b.arrV 
              for i = 0 to mA - 1 do 
                  let mutable r = 0.0 
                  for k = 0 to nA - 1 do 
                      r <- r + mul (getDMDSA arrA i k) (getVDSA arrB k)
                  arr.[i] <- r
              mkVDS arr

          let mulRVVDS a b =
              let nA = dimRVDS a 
              let mB = dimVDS b 
              if nA<>mB then invalid_arg "incompatible dimensions for matrix multiplication";
              let arrA = a.arrRV 
              let arrB = b.arrV 
              let mutable r = 0.0 
              for k = 0 to nA - 1 do 
                  r <- r + mul (getRVDSA arrA k) (getVDSA arrB k)
              r

          let rowvecDMDS x = initDMDS 1         (dimRVDS x) (fun _ j -> getRVDS x j) 
          let vectorDMDS x = initDMDS (dimVDS x)  1         (fun i _ -> getVDS x i) 
          let mulVRVDS a b = mulDMDS (vectorDMDS a) (rowvecDMDS b) 

          let scaleDMDS   k m = GU.unopDMGU  (fun x -> mul k x) m
          let scaleSMDS   k m = GU.unopSMGU  (fun x -> mul k x) m
          let scaleRVDS  k m = GU.unopRVGU (fun x -> mul k x) m
          let scaleVDS   k m = GU.unopVGU  (fun x -> mul k x) m
          let negDMDS     m   = GU.unopDMGU  (fun x -> neg x) m
          let negSMDS     m   = GU.unopSMGU  (fun x -> neg x) m
          let negRVDS    m   = GU.unopRVGU (fun x -> neg x) m
          let negVDS     m   = GU.unopVGU  (fun x -> neg x) m

          let traceDMDS a =
              let nA = ncolsDMDS a 
              let mA = nrowsDMDS a
              if nA<>mA then invalid_arg "traceM: expected a square matrix";
              let arrA = a.arrDM 
              (0,nA-1) |> sumfR (fun i -> getDMDSA arrA i i) 

          let sumDMDS  a = GU.foldDMGU add zero a
          let sumVDS   a = GU.foldVGU  add zero a
          let prodDMDS a = GU.foldDMGU mul one  a
          let prodVDS  a = GU.foldVGU  mul one  a

          let dotDMDS a b = GU.fold2DMGU (fun z va vb -> add z (mul va vb)) zero a b
          let dotVDS a b = GU.fold2VGU (fun z va vb -> add z (mul va vb)) zero a b
          let sumfDMDS  f m = GU.foldDMGU (fun acc aij -> add acc (f aij)) zero m
          let normDMDS m = sqrt (sumfDMDS (fun x -> x*x) m)

          let inplace_addDMDS  a b = let arrB = b.arrDM  in GU.inplace_mapiDMGU  (fun i j x -> x + getDMDSA arrB i j) a
          let inplace_addVDS  a b = let arrB = b.arrV  in GU.inplace_mapiVGU  (fun i x   -> x + getVDSA arrB i  ) a
          let inplace_addRVDS a b = let arrB = b.arrRV in GU.inplace_mapiRVGU (fun i x   -> x + getRVDSA arrB i  ) a
          let inplace_subDMDS  a b = let arrB = b.arrDM  in GU.inplace_mapiDMGU  (fun i j x -> x - getDMDSA  arrB i j) a
          let inplace_subVDS  a b = let arrB = b.arrV  in GU.inplace_mapiVGU  (fun i x   -> x - getVDSA arrB i  ) a
          let inplace_subRVDS a b = let arrB = b.arrRV in GU.inplace_mapiRVGU (fun i x   -> x - getRVDSA arrB i  ) a
          let inplace_cptMulDMDS  a b = let arrB = b.arrDM  in GU.inplace_mapiDMGU  (fun i j x -> x * getDMDSA arrB i j) a
          let inplace_cptMulVDS  a b = let arrB = b.arrV  in GU.inplace_mapiVGU  (fun i x   -> x * getVDSA arrB i  ) a
          let inplace_cptMulRVDS a b = let arrB = b.arrRV in GU.inplace_mapiRVGU (fun i x   -> x * getRVDSA arrB i  ) a
          let inplace_scaleDMDS  (a:float) b = GU.inplace_mapiDMGU  (fun i j x -> a * x) b
          let inplace_scaleVDS  (a:float) b = GU.inplace_mapiVGU  (fun i x   -> a * x) b
          let inplace_scaleRVDS (a:float) b = GU.inplace_mapiRVGU (fun i x   -> a * x) b

        end


//----------------------------------------------------------------------------
// module RawMatrixOps.Specialized
//--------------------------------------------------------------------------*)

        module Specialized = begin

          open Instances
          open GlobalAssociations
          module GU = Unspecialized
          module DS = DoubleSpecialized

            
          type smatrix = SparseMatrix<float>
          type dmatrix = DenseMatrix<float>
          type vector = Vector<float>
          type rowvec = RowVector<float>
          let inline dense x = Dense(x)
          let inline sparse x = Sparse(x)
          let inline createMx  ops m n f = GU.createDMGU ops m n f |> dense
          let inline createVx  ops m f   = GU.createVGU ops m f
          let inline createRVx ops m f   = GU.createRVGU ops m f

          // Accessors
          let inline getDM a i j   = GU.getDMGU a i j
          let getM a i j   = 
              match a with 
              | Dense a -> GU.getDMGU a i j
              | Sparse a -> GU.getSMGU a i j

          let nonzero_entriesM a   = 
              match a with 
              | Dense a -> GU.nonzero_entriesDMGU a 
              | Sparse a -> GU.nonzero_entriesSMGU a 

          // This could be in the library, though it's a bit brutal
          let interleave cf (s1: #seq<'a>) (s2: #seq<'b>) =
              Seq.generate (fun () -> let e1 = s1.GetEnumerator()
                                      let e2 = s2.GetEnumerator()
                                      let havee1 = ref (e1.MoveNext())
                                      let havee2 = ref (e2.MoveNext())
                                      (havee1,e1,havee2,e2))
                                      
                           (fun (havee1,e1,havee2,e2) -> 
                              if !havee1 && !havee2 then 
                                 let v1 = e1.Current
                                 let v2 = e2.Current
                                 let c = cf v1 v2 
                                 if c < 0 then 
                                     havee1 := e1.MoveNext()
                                     Some(Some(v1),None)
                                 elif c = 0 then
                                     havee1 := e1.MoveNext()
                                     havee2 := e2.MoveNext()
                                     Some(Some(v1),Some(v2))
                                 else 
                                     havee2 := e2.MoveNext()
                                     Some(None,Some(v2))
                              else if !havee1 then 
                                 let v1 = e1.Current
                                 havee1 := e1.MoveNext()
                                 Some(Some(v1),None)
                              else if !havee2 then 
                                 let v2 = e2.Current
                                 havee2 := e2.MoveNext()
                                 Some(None,Some(v2))
                              else None)
                           (fun (havee1,e1,havee2,e2) -> e1.Dispose(); e2.Dispose())

          let nonzero_entries2M  (a:Matrix<_>) (b:Matrix<_>) = 
              let ops = a.ElementOps 
              let zero = ops.Zero
              interleave (fun (i1,j1,_) (i2,j2,_) -> let c = compare i1 i2 in if c <> 0 then c else compare j1 j2) (nonzero_entriesM a) (nonzero_entriesM b)
              |> Seq.map (function | Some(i,j,v1),Some(_,_,v2) -> (i,j,v1,v2)
                                   | Some(i,j,v1),None         -> (i,j,v1,zero)
                                   | None,        Some(i,j,v2) -> (i,j,zero,v2)
                                   | None,        None          -> failwith "unreachable")


          
          let inline setDM a i j x = GU.setDMGU a i j x
          let inline dimDM a       = GU.dimDMGU a
          let inline nrowsDM a     = GU.nrowsDMGU a
          let inline ncolsDM a     = GU.ncolsDMGU a
          let inline eopsDM a     = GU.eopsDMGU a

          let setM a i j x = 
              match a with 
              | Dense a -> GU.setDMGU a i j x
              | Sparse a -> GU.setSMGU a i j x
          let nrowsM a     = 
              match a with 
              | Sparse a ->  GU.nrowsSMGU a
              | Dense a -> GU.nrowsDMGU a
          let ncolsM a     = 
              match a with 
              | Sparse a -> GU.ncolsSMGU a
              | Dense a -> GU.ncolsDMGU a
          let dimM a       = 
              match a with 
              | Sparse a -> GU.dimSMGU a 
              | Dense a -> GU.dimDMGU a
          let eopsM a     = 
              match a with 
              | Sparse a -> GU.eopsSMGU a
              | Dense a -> GU.eopsDMGU a
          
          let inline getRV a i   = GU.getRVGU a i
          let inline setRV a i x = GU.setRVGU a i x
          let inline dimRV a   = GU.dimRVGU a
          let inline ncolsRV a   = GU.dimRVGU a
          let inline eopsRV a     = GU.eopsRVGU a

          let inline getV a i   = GU.getVGU a i
          let inline setV a i x = GU.setVGU a i x
          let inline dimV a     = GU.dimVGU a
          let inline nrowsV a     = GU.dimVGU a
          let inline eopsV a     = GU.eopsVGU a

          // Creation
          let listM    xss : Matrix<'a>    = GU.listDMGU    (GetNumericAssociationOption<'a>()) xss |> dense
          let listV    xss : Vector<'a>    = GU.listVGU    (GetNumericAssociationOption<'a>()) xss
          let listRV   xss : RowVector<'a> = GU.listRVGU   (GetNumericAssociationOption<'a>()) xss

          let seqM    xss : Matrix<'a>    = GU.seqDMGU    (GetNumericAssociationOption<'a>()) xss |> dense
          let seqV    xss : Vector<'a>    = GU.seqVGU    (GetNumericAssociationOption<'a>()) xss
          let seqRV   xss : RowVector<'a> = GU.seqRVGU   (GetNumericAssociationOption<'a>()) xss

          let initM  m n f : Matrix<'a>    = GU.createDMGU  (GetNumericAssociationOption<'a>()) m n f |> dense
          let initRV m   f : RowVector<'a> = GU.createRVGU (GetNumericAssociationOption<'a>()) m   f
          let initV  m   f : Vector<'a>    = GU.createVGU  (GetNumericAssociationOption<'a>()) m   f

          let inline inplace_assignM  f a = 
              match a with 
              | Sparse a -> GU.sparse_not_mutable()
              | Dense a -> GU.assignDMGU  f a
          let inline assignV  f a = GU.assignVGU  f a

          let coerce2 x = unbox(box(x))
          let loosenDM (x: dmatrix) : DenseMatrix<_>  = coerce2 x
          let loosenSM (x: smatrix) : SparseMatrix<_> = coerce2 x
          let loosenV  (x: vector)  : Vector<_>       = coerce2 x
          let loosenRV (x: rowvec)  : RowVector<_>    = coerce2 x
          let loosenF  (x: float)   : 'a              = coerce2 x

          let tightenDM (x: DenseMatrix<_>)  : dmatrix = coerce2 x
          let tightenSM (x: SparseMatrix<_>) : smatrix = coerce2 x
          let tightenV  (x: Vector<_>)       : vector  = coerce2 x
          let tightenRV (x: RowVector<_>)    : rowvec  = coerce2 x
          let tightenF  (x: 'a)              : float   = coerce2 x

          let zeroM m n = 
            let arr = GU.createArray2 m n
            // This is quite performance critical
            // REVIEW: do this type test differently??
        #if CLI_AT_LEAST_2_0
            // Avoid assigining zeros into the array
            match box arr with 
            | :? (float[,])   as arr -> GU.mkDMGU DS.FloatOps arr |> loosenDM |> dense
            | _ -> 
        #endif
            GU.zeroizeDMGUA arr m n  |> dense

          let zeroV m  : Vector<'a> = 
            let arr = GU.createArray m 
        #if CLI_AT_LEAST_2_0
            // Avoid assigining zeros into the array
            match box (arr: 'a[]) with 
            | :? (float[])   as arr -> GU.mkVGU DS.FloatOps arr |> loosenV
            | _ -> 
        #endif
            GU.zeroizeVGUA arr m

          let zeroRV m  : RowVector<'a> = 
            let arr = GU.createArray m 
        #if CLI_AT_LEAST_2_0
            // Avoid assigining zeros into the array
            match box (arr: 'a[]) with 
            | :? (float[])   as arr -> GU.mkRVGU DS.FloatOps arr |> loosenRV
            | _ -> 
        #endif
            GU.zeroizeRVGUA arr m
              
          let initNumericM m n f   = 
              let arr = GU.createArray2 m n 
              let opsData = GetNumericAssociationOption<'a>() 
              let ops = GU.opsOfOpsData opsData 
              GU.assignArray2 m n (f ops) arr;
              GU.mkDMGU opsData arr |> dense

          let identityM m   = 
            let arr = GU.createArray2 m m 
            // This is quite performance critical
            // REVIEW: do this type test differently??
        #if CLI_AT_LEAST_2_0
            // Avoid assigining zeros into the array
            match box arr with 
            | :? (float[,])   as arr -> 
                for i = 0 to m - 1 do 
                   arr.(i,i) <- 1.0 
                done; 
                GU.mkDMGU DS.FloatOps arr |> loosenDM |> dense
            | _ -> 
        #endif
            let opsData = GetNumericAssociationOption<'a>() 
            let ops = GU.opsOfOpsData opsData 
            let zero = ops.Zero 
            let one = ops.One 
            GU.assignArray2 m m (fun i j -> if i = j then one else zero) arr;
            GU.mkDMGU opsData arr |> dense

          let createNumericV m f  : Vector<'a> = 
              let arr = GU.createArray m 
              let opsData = GetNumericAssociationOption<'a>() 
              let ops = GU.opsOfOpsData opsData 
              GU.assignArray m (f ops) arr;
              GU.mkVGU opsData arr
              
          let constM  m n x = initM  m n (fun i j -> x)
          let constRV m x   = initRV m   (fun i -> x)
          let constV  m x   = initV  m   (fun i -> x)
          let scalarM   x = constM 1 1 x 
          let scalarRV  x = constRV 1 x 
          let scalarV   x = constV 1 x 

          let diagnM v n = 
              let ops = eopsV v 
              let zero = ops.Zero 
              let nV = dimV v + (if n < 0 then -n else n) 
              createMx v.opsV nV nV (fun i j -> if i+n=j then getV v i else zero)

          let diagM v = diagnM v 0

          let constDiagM  n x : Matrix<'a> = 
              let opsData = GetNumericAssociationOption<'a>() 
              let ops = GU.opsOfOpsData opsData 
              let zero = ops.Zero 
              createMx opsData n n (fun i j -> if i=j then x else zero) 

          // REVIEW: we are dropping sparseness on pointwise multiplication of sparse and dense.
          let inline bopM opDMDS opDMGU opSMDS opSMGU a b = 
              match a,b with 
              | Dense a,Dense b -> 
                  match box a with 
                  | (:? dmatrix as a) -> opDMDS   a (tightenDM b) |> loosenDM |> dense
                  | _                 -> opDMGU a b                           |> dense
              | Sparse a,Sparse b ->
                  match box a with 
                  | (:? smatrix as a) -> opSMDS a (tightenSM b) |> loosenSM |> sparse
                  | _                 -> opSMGU a b                         |> sparse
              | Sparse a, Dense b     -> opDMGU (GU.to_denseSMGU a) b         |> dense
              | Dense  a, Sparse b    -> opDMGU a (GU.to_denseSMGU b)         |> dense

          let inline uopM opDMDS opDMGU opSMDS opSMGU  b = 
              match b with 
              | Dense b -> 
                  match box b with 
                  | (:? dmatrix as b)  -> opDMDS b |> loosenDM |> dense
                  | _                  -> opDMGU b             |> dense
              | Sparse b ->             
                  match box b with 
                  | (:? smatrix as b) -> opSMDS b |> loosenSM |> sparse
                  | _                 -> opSMGU b             |> sparse

          let inline fuopM opDMDS opDMGU opSMDS opSMGU  b = 
              match b with 
              | Dense b -> 
                  match box b with 
                  | (:? dmatrix as b)  -> opDMDS b |> loosenF
                  | _                  -> opDMGU b             
              | Sparse b ->             
                  match box b with 
                  | (:? smatrix as b) -> opSMDS b |> loosenF 
                  | _                 -> opSMGU b             

          let addM a b = bopM DS.addDMDS GU.addDMGU DS.addSMDS GU.addSMGU a b
          let subM a b = bopM DS.subDMDS GU.subDMGU DS.subSMDS GU.subSMGU a b
          let mulM a b = bopM DS.mulDMDS GU.mulDMGU DS.mulSMDS GU.mulSMGU a b
          let cptMulM a b = bopM DS.cptMulDMDS GU.cptMulDMGU DS.cptMulSMDS GU.cptMulSMGU a b
          let cptMaxM a b = bopM DS.cptMaxDMDS GU.cptMaxDMGU DS.cptMaxSMDS GU.cptMaxSMGU a b
          let cptMinM a b = bopM DS.cptMinDMDS GU.cptMinDMGU DS.cptMinSMDS GU.cptMinSMGU a b

          let addRV a b = 
              match box a with 
              | (:? rowvec as a) -> DS.addRVDS a (tightenRV b) |> loosenRV
              | _                -> GU.addRVGU a b

          let addV a b = 
              match box a with 
              | (:? vector as a) -> DS.addVDS a (tightenV b) |> loosenV
              | _                -> GU.addVGU a b

          let subRV a b = 
              match box a with 
              | (:? rowvec as a) -> DS.subRVDS   a (tightenRV b) |> loosenRV
              | _                -> GU.subRVGU a b

          let subV a b = 
              match box a with 
              | (:? vector as a) -> DS.subVDS   a (tightenV b) |> loosenV
              | _                -> GU.subVGU a b

          let mulRVM a b = 
              match b with 
              | Dense b -> 
                  match box a with 
                  | (:? rowvec as a) -> DS.mulRVDMDS   a (tightenDM b) |> loosenRV
                  | _                -> GU.mulRVDMGU a b
              | Sparse b -> GU.mulRVSMGU a b

          let mulMV a b = 
              match a with 
              | Dense a -> 
                  match box a with 
                  | (:? dmatrix as a) -> DS.mulMVDS   a (tightenV b) |> loosenV
                  | _                 ->          GU.mulDMVGU a b
              | Sparse a -> GU.mulSMVGU a b 

          let mulRVV a b = 
              match box a with 
              | (:? rowvec as a) -> DS.mulRVVDS   a (tightenV b) |> loosenF
              | _                ->          GU.mulRVVGU a b

          let mulVRV a b = 
            match box a with 
            | (:? vector as a) -> DS.mulVRVDS   a (tightenRV b) |> loosenDM |> dense
            | _                ->           GU.mulVRVGU a b |> dense

          let cptMulRV a b = 
            match box a with 
            | (:? rowvec as a) -> DS.cptMulRVDS   a (tightenRV b) |> loosenRV
            | _                -> GU.cptMulRVGU a b

          let cptMulV a b = 
            match box a with 
            | (:? vector as a) -> DS.cptMulVDS   a (tightenV b) |> loosenV
            | _                -> GU.cptMulVGU a b

          let cptMaxV a b = 
            match box a with 
            | (:? vector as a) -> DS.cptMaxVDS   a (tightenV b) |> loosenV
            | _                -> GU.cptMaxVGU a b

          let cptMinV a b = 
            match box a with 
            | (:? vector as a) -> DS.cptMinVDS   a (tightenV b) |> loosenV
            | _                -> GU.cptMinVGU a b

          let scaleM a b = uopM (fun b -> DS.scaleDMDS (tightenF a) b) (GU.scaleDMGU a)
                                (fun b -> DS.scaleSMDS (tightenF a) b) (GU.scaleSMGU a) b

          let scaleRV a b = 
            match box b with 
            | (:? rowvec as b)  -> DS.scaleRVDS (tightenF a) b |> loosenRV 
            | _                 -> GU.scaleRVGU a b

          let scaleV a b = 
            match box b with 
            | (:? vector as b)  -> DS.scaleVDS (tightenF a) b |> loosenV
            | _                 -> GU.scaleVGU a b

          let dotM a b = 
            match a,b with 
            | Dense a,Dense b -> 
                match box b with 
                | (:? dmatrix as b)  -> DS.dotDMDS   (tightenDM a) b |> loosenF
                | _                  -> GU.dotDMGU a b
            | _ ->  
                let ops = a.ElementOps 
                nonzero_entries2M a b |> Seq.fold (fun z (_,_,va,vb) -> GU.add ops z (GU.mul ops va vb)) ops.Zero 

          let dotV a b = 
            match box b with 
            | (:? vector as b)  -> DS.dotVDS   (tightenV a) b |> loosenF
            | _                 -> GU.dotVGU a b

          let negM a = uopM DS.negDMDS GU.negDMGU DS.negSMDS GU.negSMGU a

          let negRV a = 
            match box a with 
            | (:? rowvec as a) -> DS.negRVDS a |> loosenRV
            | _               ->  GU.negRVGU a

          let negV a = 
            match box a with 
            | (:? vector as a) -> DS.negVDS a |> loosenV
            | _               ->  GU.negVGU a

          let traceMGU a =
            let nA = ncolsM a  
            let mA = nrowsM a 
            if nA<>mA then invalid_arg "traceM: expected a square matrix";
            let ops = a.ElementOps 
            (0,nA-1) |> GU.sumRGU ops (fun i -> getM a i i) 

          let traceM a = fuopM DS.traceDMDS (dense >> traceMGU) (sparse >> traceMGU) (sparse >> traceMGU) a
          let sumM a = fuopM DS.sumDMDS GU.sumDMGU GU.sumSMGU GU.sumSMGU a
          let prodM a = fuopM DS.prodDMDS GU.prodDMGU GU.prodSMGU GU.prodSMGU a
          let normM a = fuopM DS.normDMDS GU.normDMGU GU.normSMGU GU.normSMGU a

          let opsM a = 
            match a with 
            | Dense a -> a.opsDM 
            | Sparse a -> a.opsSM 
          
          let transM a = 
            match a with 
            | Dense a -> 
                let nA = ncolsDM a 
                let mA = nrowsDM a
                createMx a.opsDM nA mA (fun i j -> getDM a j i)
            | Sparse a -> 
                a |> GU.nonzero_entriesSMGU  |> Seq.map (fun (i,j,v) -> (j,i,v)) |> GU.init_sparseSMGU (GU.ncolsSMGU a) (GU.nrowsSMGU a) a.opsSM |> sparse
          
          let permuteRows (p: Permutation) a =
            match a with
            | Dense a ->
                let nA = ncolsDM a
                let mA = nrowsDM a
                createMx a.opsDM nA mA (fun i j -> getDM a p.[i] j)
            | Sparse a ->
                a |> GU.nonzero_entriesSMGU  |> Seq.map (fun (i,j,v) -> (p.[i],j,v)) |> GU.init_sparseSMGU (GU.ncolsSMGU a) (GU.nrowsSMGU a) a.opsSM |> sparse

          let permuteColumns (p: Permutation) a =
            match a with
            | Dense a ->
                let nA = ncolsDM a
                let mA = nrowsDM a
                createMx a.opsDM nA mA (fun i j -> getDM a i p.[j])
            | Sparse a ->
                a |> GU.nonzero_entriesSMGU  |> Seq.map (fun (i,j,v) -> (i,p.[j],v)) |> GU.init_sparseSMGU (GU.ncolsSMGU a) (GU.nrowsSMGU a) a.opsSM |> sparse

          let transRV a = 
            let mA = dimRV a
            createVx a.opsRV  mA (fun i -> getRV a i)

          let transV a = 
            let mA = dimV a
            createRVx a.opsV  mA (fun i -> getV a i)

          let inplace_addM a b = 
            match a,b with 
            | Dense a,Dense b -> 
                match box a with 
                | (:? dmatrix as a) -> DS.inplace_addDMDS   a (tightenDM b)
                | _                 -> GU.inplace_addDMGU a b
            | _ -> GU.sparse_not_mutable()

          let inplace_addV a b = 
            match box a with 
            | (:? vector as a) -> DS.inplace_addVDS   a (tightenV b)
            | _                -> GU.inplace_addVGU a b

          let inplace_addRV a b = 
            match box a with 
            | (:? rowvec as a) -> DS.inplace_addRVDS   a (tightenRV b)
            | _                -> GU.inplace_addRVGU a b

          let inplace_subM a b = 
            match a,b with 
            | Dense a,Dense b -> 
                match box a with 
                | (:? dmatrix as a) -> DS.inplace_subDMDS   a (tightenDM b)
                | _                -> GU.inplace_subDMGU a b
            | _ -> GU.sparse_not_mutable()

          let inplace_subV a b = 
            match box a with 
            | (:? vector as a) -> DS.inplace_subVDS   a (tightenV b)
            | _                -> GU.inplace_subVGU a b

          let inplace_subRV a b = 
            match box a with 
            | (:? rowvec as a) -> DS.inplace_subRVDS   a (tightenRV b)
            | _                -> GU.inplace_subRVGU a b


          let inplace_cptMulM a b = 
            match a,b with 
            | Dense a,Dense b -> 
                match box a with 
                | (:? dmatrix as a) -> DS.inplace_cptMulDMDS   a (tightenDM b)
                | _                -> GU.inplace_cptMulDMGU a b
            | _ -> GU.sparse_not_mutable()

          let inplace_cptMulV a b = 
            match box a with 
            | (:? vector as a) -> DS.inplace_cptMulVDS   a (tightenV b)
            | _                -> GU.inplace_cptMulVGU a b

          let inplace_cptMulRV a b = 
            match box a with 
            | (:? rowvec as a) -> DS.inplace_cptMulRVDS   a (tightenRV b)
            | _                -> GU.inplace_cptMulRVGU a b

          let inplace_scaleM a b = 
            match b with 
            | Dense b -> 
                match box b with 
                | (:? dmatrix as b)  -> DS.inplace_scaleDMDS   (tightenF a) b
                | _                 -> GU.inplace_scaleDMGU a b
            | _ -> GU.sparse_not_mutable()

          let inplace_scaleV a b = 
            match box b with 
            | (:? vector as b)  -> DS.inplace_scaleVDS   (tightenF a) b
            | _                 -> GU.inplace_scaleVGU a b

          let existsM  f a = 
            match a with 
            | Sparse a -> GU.sparse_nyi() // note: martin says "run f on a token element if it's not full"
            | Dense a -> GU.existsiDMGU  (fun _ _ -> f) a
          let existsV  f a = GU.existsiVGU  (fun _ -> f) a
          let forallM  f a = 
            match a with 
            | Sparse a -> GU.sparse_nyi()
            | Dense a -> GU.foralliDMGU  (fun _ _ -> f) a
          let forallV  f a = GU.foralliVGU  (fun _ -> f) a
          let existsiM  f a = 
            match a with 
            | Sparse a -> GU.sparse_nyi()
            | Dense a -> GU.existsiDMGU  f a
          let existsiV  f a = GU.existsiVGU  f a
          let foralliM  f a = 
            match a with 
            | Sparse a -> GU.sparse_nyi()
            | Dense a -> GU.foralliDMGU  f a
          let foralliV  f a = GU.foralliVGU  f a

          let mapM  f a = 
            match a with 
            | Sparse a -> GU.sparse_nyi()
            | Dense a -> Dense(GU.mapDMGU f a)

          let mapV  f a = GU.mapVGU f a

          let copyM  a = 
            match a with 
            | Sparse a -> Sparse (GU.copySMGU a)
            | Dense a -> Dense (GU.copyDMGU a)

          let copyV  a = GU.copyVGU a

          let copyRV  a = GU.copyRVGU a

          let mapiM  f a = 
            match a with 
            | Sparse a -> GU.sparse_nyi()
            | Dense a -> Dense (GU.mapiDMGU f a)

          let mapiV  f a = GU.mapiVGU f a
          let permuteV p a = GU.permuteVGU p a
          let permuteRV p a = GU.permuteRVGU p a

          let mapiRV  f a = GU.mapiRVGU f a

          let to_denseM a = 
            match a with 
            | Sparse a -> GU.to_denseSMGU a |> dense
            | Dense _ -> a

          let init_sparseM i j x : Matrix<'a> = 
#if CLI_AT_MOST_1_1
              failwith "sparse matrices not implemented on .NET 1.x"
#else
            let opsData = GetNumericAssociationOption<'a>() 
            GU.init_sparseSMGU i j opsData x |> sparse
#endif
            
          let init_denseM i j x : Matrix<'a> = 
            let r = zeroM i j
            x |> Seq.iter (fun (i,j,v) -> setM r i j v);
            r

          let getDiagnM a n =
            let nA = ncolsM a 
            let mA = nrowsM a
            if nA<>mA then invalid_arg "getDiagnM expected a square matrix";
            let ops = a.ElementOps 
            let ni = if n < 0 then -n else 0 
            let nj = if n > 0 then  n else 0 
            GU.createVGU (opsM a) (max (nA-abs(n)) 0) (fun i -> getM a (i+ni) (i+nj)) 

          let getDiagM  a = getDiagnM a 0

          let inline inplace_mapiM  f a = 
            match a with 
            | Sparse a -> GU.sparse_not_mutable()
            | Dense a -> GU.inplace_mapiDMGU f a
          let inline inplace_mapiV  f a = GU.inplace_mapiVGU f a
          
          let inline foldM  f z a = 
            match a with 
            | Sparse a -> GU.sparse_nyi()
            | Dense a -> GU.foldDMGU f z a
          let inline foldV  f z a = GU.foldVGU f z a


          let inline foldiM  f z a = 
            match a with 
            | Sparse a -> GU.sparse_nyi()
            | Dense a -> GU.foldiDMGU f z a
          let inline foldiV  f z a = GU.foldiVGU f z a

          let compare a b = StructuralComparison a b
          let hash a = StructuralHash a
          let min a = StructuralMinimum a
          let compareM a b = 
            let nA = ncolsM a 
            let mA = nrowsM a 
            let nB = ncolsM b 
            let mB = nrowsM b 
            let c = compare mA mB 
            if c <> 0 then c else
            let c = compare nA nB 
            if c <> 0 then c else
            match a,b with 
            | Dense a, Dense b -> 
              let rec go2 i j = 
                 if j < nA then 
                   let c = compare (getDM a i j) (getDM b i j) 
                   if c <> 0 then c else 
                   go2 i (j+1) 
                 else 0 
              let rec go1 i = 
                 if i < mA then 
                   let c = go2 i 0 
                   if c <> 0 then c 
                   else go1 (i+1) 
                 else 0 
              go1 0
           | _ -> 
             match (nonzero_entries2M a b |> Seq.first (fun (i,j,v1,v2) -> let c = compare v1 v2 in if c = 0 then None else Some(c))) with
             | None -> 0
             | Some(c) -> c
             

          let compareV a b = 
           let mA = dimV a
           let mB = dimV b 
           let c = compare mA mB 
           if c <> 0 then c else
           let rec go2 j = 
              if j < mA then 
                let c = compare (getV a j) (getV b j) 
                if c <> 0 then c else go2 (j+1) 
              else 0 
           go2 0

          let compareRV a b = 
              let mA = dimRV a 
              let mB = dimRV b 
              let c = compare mA mB 
              if c <> 0 then c else
              let rec go2 j = 
                 if j < mA then 
                   let c = compare (getRV a j) (getRV b j) 
                   if c <> 0 then c else go2 (j+1) 
                 else 0 
              go2 0

          let inline combineHash x y = (x <<< 1) + y + 631 

          let hashM a = 
             let nA = ncolsM a 
             let mA = nrowsM a 
             let acc = hash mA + hash nA
             a |> nonzero_entriesM |> Seq.truncate 20 |> Seq.fold (fun z v -> combineHash z (hash v)) acc
            
          let hashV a = 
           let mA = dimV a 
           hash mA +
           (let mutable c = 0 
            for i = 0 to mA - 1 do
                c <- combineHash c (hash (getV a i))
            done; c)
            
          let hashRV a = 
              let mA = dimRV a 
              hash mA +
              (let mutable c = 0 
               for i = 0 to mA - 1 do
                   c <- combineHash c (hash (getRV a i))
               done; c)
            
          type range = int * int

          let startR ((a,b) : range)   = a
          let countR ((a,b) : range)   = (b-a)+1
          let idxR    ((a,b) : range) i = a+i
          let inR    ((a,b) : range) i = a <= i && i <= b
          
          let getRowM  a i         = createRVx (opsM a) (ncolsM a) (fun j -> getM a i j)
          let selColM  a j         = createVx (opsM a) (nrowsM a) (fun i -> getM a i j) 
          let getRegionV a r       = createVx a.opsV (countR r) (fun i -> getV a (idxR r i)) 
          let getRegionRV a r      = createRVx a.opsRV (countR r) (fun i -> getRV a (idxR r i)) 

          let getRegionM  a ri rj    = 
              match a with 
              | Dense a -> createMx a.opsDM (countR ri) (countR rj) (fun i j -> getDM a (idxR ri i)  (idxR rj j)) 
              | _ -> nonzero_entriesM a 
                     |> Seq.filter (fun (i,j,_) -> inR ri i && inR rj j) 
                     |> Seq.map (fun (i,j,v) -> (i-startR ri,j-startR rj,v)) 
                     |> init_sparseM (countR ri) (countR rj)

          let getColsM a rj         = getRegionM a (0,nrowsM a - 1) rj
          let getRowsM a ri         = getRegionM a ri (0,ncolsM a - 1)

          let rowvecM x = initM 1         (dimRV x) (fun _ j -> getRV x j) 
          let vectorM x = initM (dimV x)  1         (fun i _ -> getV x i)  
          let to_vectorM x = selColM x 0 
          let to_rowvecM x = getRowM x 0 
          let to_scalarM x = getM x 0 0 

        end

    end


//----------------------------------------------------------------------------
// type Matrix<'a> augmentation
//--------------------------------------------------------------------------*)

    open RawMatrixOps

    type Matrix<'a> with
        static member ( +  )((a: 'a Matrix),b) = Specialized.addM a b
        static member ( -  )((a: 'a Matrix),b) = Specialized.subM a b
        [<OverloadID("MultiplyMatrixMatrix")>]
        static member ( *  )((a: 'a Matrix),b) = Specialized.mulM a b
        [<OverloadID("MultiplyMatrixVector")>]
        static member ( *  )((a: 'a Matrix),(b : 'a Vector)) = Specialized.mulMV a b

        [<OverloadID("MultiplyMatrixScalar")>]
        static member ( * )((m: 'a Matrix),(k : 'a)) = Specialized.scaleM k m

        static member ( .* )((a: 'a Matrix),b) = Specialized.cptMulM a b
        static member ( +=  )((a: 'a Matrix),b) = Specialized.inplace_addM a b
        static member ( -=  )((a: 'a Matrix),b) = Specialized.inplace_subM a b
        static member ( .*= )((a: 'a Matrix),b) = Specialized.inplace_cptMulM a b
        static member ( $* )(k,(m: 'a Matrix)) = Specialized.scaleM k m
        [<OverloadID("MultiplyScalarMatrix")>]
        static member ( * )(k,(m: 'a Matrix)) = Specialized.scaleM k m
        static member ( *$ )((m: 'a Matrix),k) = Specialized.scaleM k m
        static member ( ~- )(m: 'a Matrix)     = Specialized.negM m
        static member ( ~+ )(m: 'a Matrix)     = m
        member m.Item
           with get (idx1,idx2) = Specialized.getM m idx1 idx2
           and  set (idx1,idx2) v = Specialized.setM m idx1 idx2 v

        member m.GetSlice2D (?start1,?finish1,?start2,?finish2) = 
            let start1 = match start1 with None -> 0 | Some v -> v 
            let finish1 = match finish1 with None -> m.NumRows - 1 | Some v -> v 
            let start2 = match start2 with None -> 0 | Some v -> v 
            let finish2 = match finish2 with None -> m.NumCols - 1 | Some v -> v 
            Specialized.getRegionM m (start1,finish1) (start2,finish2)


        override m.ToString() = 
           match m with 
           | Dense m -> Unspecialized.showDMGU m
           | Sparse m -> "<sparse>"
        member m.DebugDisplay = 
           match m with 
           | Dense m -> Unspecialized.debugShowDMGU m
           | Sparse m -> "<sparse>"


        member m.NumCols    = Specialized.ncolsM m
        member m.NumRows    = Specialized.nrowsM m
        member m.Dimensions = Specialized.dimM m

        member m.Transpose = Specialized.transM m
        member m.PermuteRows (p: Permutation) = Specialized.permuteRows p m
        member m.PermuteColumns (p: Permutation) = Specialized.permuteColumns p m
        
#if CLI_AT_LEAST_2_0
        member inline m.Pin (f: 'a nativeptr -> 'b) = 
            match m with 
            | Dense m -> NativeOps.pinAny (box m.arrDM) (fun _ -> f (Array2.geta m.arrDM 0 0))
            | Sparse m -> failwith "cannot pin sparse matrices"
        member inline m.PinHandle () : 'a nativeptr * _ = 
            match m with 
            | Dense m -> let gch = NativeOps.pinUnscoped (box m.arrDM) 
                         Array2.geta m.arrDM 0 0, gch
            | Sparse m -> failwith "cannot pin sparse matrices"
#endif
      
     (*  interface IMatrix<'a> with 
        member x.NumRows = Specialized.nrowsM x
        member x.NumCols = Specialized.ncolsM x
        member x.Item with get (i,j)  = Specialized.getM x i j
        member x.ElementOps = x.opsDM
        member x.Transpose  = ((x.Transpose) :> DMGUatrix<'a>)
      end *)
        
        interface System.IComparable with 
             member x.CompareTo(yobj:obj) = Specialized.compareM x (yobj :?> Matrix<'a>)
        interface IStructuralHash with 
             member x.GetStructuralHashCode(nodesRemaining) = Specialized.hashM(x) // ignore nodesRemaining

        interface Microsoft.FSharp.Text.StructuredFormat.IFormattable with
            member m.GetLayout(env) = 
                 match m with 
                 | Dense dm -> Unspecialized.layoutOfDMGU env dm
                 | Sparse sm -> env.GetLayout(if m.NumRows < 20 && m.NumCols < 20 then box (Specialized.to_denseM m) else box(Specialized.nonzero_entriesM m))

        override x.GetHashCode() = Specialized.hashM(x) 
        override x.Equals(yobj) = (Specialized.compareM x (yobj :?> Matrix<'a>) = 0)

//----------------------------------------------------------------------------
// type Vector<'a> augmentation
//--------------------------------------------------------------------------*)

    type Vector<'a> with
        static member ( +  )((a: Vector<'a>),b) = Specialized.addV a b
        static member ( -  )((a: Vector<'a>),b) = Specialized.subV a b
        static member ( .* )((a: Vector<'a>),b) = Specialized.cptMulV a b
        static member ( +=  )((a: Vector<'a>),b) = Specialized.inplace_addV a b
        static member ( -=  )((a: Vector<'a>),b) = Specialized.inplace_subV a b
        static member ( .*= )((a: Vector<'a>),b) = Specialized.inplace_cptMulV a b
        static member ( $* )(k,(m: Vector<'a>)) = Specialized.scaleV k m
        
        [<OverloadID("ScalarVectorMultiply")>]
        static member ( * )(k,(m: Vector<'a>)) = Specialized.scaleV k m
        
        [<OverloadID("VectorRowVectorMultiply")>]
        static member ( * )((a: Vector<'a>),b) = Specialized.mulVRV a b
        
        [<OverloadID("VectorScalarMultiply")>]
        static member ( * )((m: Vector<'a>),k) = Specialized.scaleV k m
        static member ( *$ )((m: Vector<'a>),k) = Specialized.scaleV k m
        
        static member ( ~- )(m: Vector<'a>)     = Specialized.negV m
        static member ( ~+ )(m: Vector<'a>)     = m
        member m.Item
           with get idx1 = Specialized.getV m idx1
           and  set idx1 v = Specialized.setV m idx1 v

        member m.GetSlice (?start,?finish) = 
            let start = match start with None -> 0 | Some v -> v 
            let finish = match finish with None -> m.NumRows - 1 | Some v -> v 
            Specialized.getRegionV m (start,finish)

        override m.ToString() = Unspecialized.showVGU "vector" m

        member m.DebugDisplay = Unspecialized.showVGU "vector" m

        member m.NumRows = Specialized.dimV m
        member m.Details = m.arrV

        member m.Transpose = Specialized.transV m
        
        member m.Permute (p:Permutation) = Specialized.permuteV p m
      
#if CLI_AT_LEAST_2_0
        member inline m.Pin (f: 'a nativeptr -> 'b) = NativeOps.pinAny (box m.arrV) (fun _ -> f (Array.geta m.arrV 0))
        member inline m.PinHandle () : 'a nativeptr * _ = let gch = NativeOps.pinUnscoped (box m.arrV) in Array.geta m.arrV 0, gch
#endif
        (* interface IVector<'a> with 
          member x.Length = Specialized.nrowsV x
          member x.Item with get(i) = Specialized.getV x i
          member x.ElementOps = x.opsV
        end *)
          
        interface System.IComparable with 
             member x.CompareTo(y:obj) = Specialized.compareV x (y :?> Vector<'a>)
        interface IStructuralHash with 
             member x.GetStructuralHashCode(nodesRemaining) = Specialized.hashV(x) // ignore nodesRemaining
        interface Microsoft.FSharp.Text.StructuredFormat.IFormattable with
            member x.GetLayout(env) = Unspecialized.layoutOfVGU env x


//----------------------------------------------------------------------------
// type RowVector<'a> augmentation
//--------------------------------------------------------------------------*)

    type RowVector<'a> with
        static member ( +  )((a: RowVector<'a>),b) = Specialized.addRV a b
        static member ( -  )((a: RowVector<'a>),b) = Specialized.subRV a b
        static member ( .* )((a: RowVector<'a>),b) = Specialized.cptMulRV a b
        static member ( +=  )((a: RowVector<'a>),b) = Specialized.inplace_addRV a b
        static member ( -=  )((a: RowVector<'a>),b) = Specialized.inplace_subRV a b
        static member ( .*= )((a: RowVector<'a>),b) = Specialized.inplace_cptMulRV a b
        static member ( $* )(k,m: RowVector<'a>) = Specialized.scaleRV k m
        [<OverloadID("ScalarRowVectorMultiply")>]
        static member ( * )(k,m: RowVector<'a>) = Specialized.scaleRV k m
        
        [<OverloadID("RowVectorMatrixMultiply")>]
        static member ( * )((a: RowVector<'a>),(b: Matrix<'a>)) = Specialized.mulRVM a b
        [<OverloadID("RowVectorVectorMultiply")>]
        static member ( * )(a: RowVector<'a>,b:Vector<'a>) = Specialized.mulRVV a b
        [<OverloadID("RowVectorScalarMultiply")>]
        static member ( * )((m: RowVector<'a>),(k:'a)) = Specialized.scaleRV k m
        
        static member ( ~- )(m: RowVector<'a>)     = Specialized.negRV m
        static member ( ~+ )(m: RowVector<'a>)     = m
        member m.Item
           with get idx1 = Specialized.getRV m idx1
           and  set idx1 v = Specialized.setRV m idx1 v

        member m.GetSlice (?start,?finish) = 
            let start = match start with None -> 0 | Some v -> v 
            let finish = match finish with None -> m.NumCols - 1 | Some v -> v 
            Specialized.getRegionRV m (start,finish)

        override m.ToString() = Unspecialized.showRVGU "rowvec" m

        member m.DebugDisplay = Unspecialized.showRVGU "rowvec" m

        member m.NumCols = Specialized.dimRV m
        member m.Details = m.arrRV

        member m.Transpose = Specialized.transRV m
        
        member m.Permute (p:Permutation) = Specialized.permuteRV p m
      
#if CLI_AT_LEAST_2_0
        member inline m.Pin (f: 'a nativeptr -> 'b) = NativeOps.pinAny (box m.arrRV) (fun _ -> f (Array.geta m.arrRV 0))
        member inline m.PinHandle () : 'a nativeptr * _ = let gch = NativeOps.pinUnscoped (box m.arrRV) in Array.geta m.arrRV 0, gch
#endif
        (* interface IVector<'a> with 
          member x.Length = Specialized.ncolsRV x
          member x.Item with get(i)  = Specialized.getRV x i
          member x.ElementOps = x.opsRV
        end *)
          
        interface System.IComparable with 
            member x.CompareTo(y:obj) = Specialized.compareRV x (y :?> RowVector<'a>)

        interface IStructuralHash with 
            member x.GetStructuralHashCode(nodesRemaining) = Specialized.hashRV(x) // ignore nodesRemaining

        interface Microsoft.FSharp.Text.StructuredFormat.IFormattable with
            member x.GetLayout(env) = Unspecialized.layoutOfRVGU env x


//----------------------------------------------------------------------------
// type aliases (lowercase)
//--------------------------------------------------------------------------*)

    type matrix = Matrix<float>
    type vector = Vector<float>
    type rowvec = RowVector<float>


//----------------------------------------------------------------------------
// module MRandom
//--------------------------------------------------------------------------*)

    module MRandom = begin
      let seed = 99
      let random_gen = new System.Random(seed)
      let float f = (random_gen).NextDouble() * f 
    end


//----------------------------------------------------------------------------
// module Matrix
//--------------------------------------------------------------------------*)
      
    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
    module Matrix = begin
      
      module Generic = begin

        module MS = RawMatrixOps.Specialized

        // Accessors
        let get a i j   = MS.getM a i j
        let set a i j x = MS.setM a i j x
        let dims a       = MS.dimM a
        let nrows a     = MS.nrowsM a
        let ncols a     = MS.ncolsM a
        let eops a      = MS.eopsM a
        
        // Creation
        let of_list    xss      = MS.listM  xss
        let of_seq     xss      = MS.seqM  xss
        let init  m n f       = MS.initM  m n f
#if CLI_AT_MOST_1_1
#else
        let of_array2 arr       = init (Array2.length1 arr) (Array2.length2 arr) (fun i j -> arr.[i,j])
        let to_array2 m         = Array2.init (nrows m) (ncols m) (fun i j -> get m i j)
#endif
        let init_numeric m n f = MS.initNumericM m n f
        let inplace_assign f a          = MS.inplace_assignM  f a
        let zero m n            = MS.zeroM m n
        let identity m            = MS.identityM m
        let create  m n x     = MS.constM m n x
        let of_scalar   x       = MS.scalarM x
        let diag v              = MS.diagM v
        let constDiag   n x     = MS.constDiagM n x
      
        // Operators
        let add a b = MS.addM a b
        let sub a b = MS.subM a b
        let mul a b = MS.mulM a b
        let mulRV a b = MS.mulRVM a b
        let mulV a b = MS.mulMV a b
        let cptMul a b = MS.cptMulM a b
        let cptMax a b = MS.cptMaxM a b
        let cptMin a b = MS.cptMinM a b
        let scale a b = MS.scaleM a b
        let dot a b = MS.dotM a b
        let neg a = MS.negM a 
        let trace a = MS.traceM a
        let sum a = MS.sumM a
        let prod a = MS.prodM a
        let norm a = MS.normM a
        let transpose a = MS.transM a
        let inplace_add a b = MS.inplace_addM a b
        let inplace_sub a b = MS.inplace_subM a b
        let inplace_cptMul a b = MS.inplace_cptMulM a b
        let inplace_scale a b = MS.inplace_scaleM a b
        let exists  f a = MS.existsM  f a
        let forall  f a = MS.forallM  f a
        let existsi  f a = MS.existsiM  f a
        let foralli  f a = MS.foralliM  f a
        let map  f a = MS.mapM f a
        let copy a = MS.copyM a
        let mapi  f a = MS.mapiM f a
        let getDiagN  a n = MS.getDiagnM a n
        let getDiag  a = MS.getDiagnM a 0
        let to_dense a = MS.to_denseM a 
#if CLI_AT_MOST_1_1
#else
        let init_dense i j a = MS.init_denseM i j a 
        let init_sparse i j a = MS.init_sparseM i j a 
#endif
        let nonzero_entries a = MS.nonzero_entriesM a 
        let inplace_mapi  f a = MS.inplace_mapiM f a
        let fold  f z a = MS.foldM f z a
        let foldi  f z a = MS.foldiM f z a
      
        let compare a b = MS.compareM a b
        let hash a = MS.hashM a
        let getRow  a i          = MS.getRowM a i
        let getCol  a j          = MS.selColM a j
        let getCols a i1 i2         = MS.getColsM a (i1,i1+i2-1)
        let getRows a j1 j2         = MS.getRowsM a (j1,j1+j2-1)
        let getRegion  a i1 j1 i2 j2      = MS.getRegionM a (i1,i1+i2-1) (j1,j1+j2-1)
        
        let of_rowvec x = MS.rowvecM x
        let of_vector x = MS.vectorM x
        let to_vector x = MS.to_vectorM x
        let to_rowvec x = MS.to_rowvecM x
        let to_scalar x = MS.to_scalarM x
     
      end

      module MG = Generic
      module DS = RawMatrixOps.DoubleSpecialized
      module GU = RawMatrixOps.Unspecialized
      module MS = RawMatrixOps.Specialized

      // Element type opsDM
      type elem = float

      // Accessors
      let get (a:matrix) i j   = MG.get a i j
      let set (a:matrix) i j x = MG.set a i j x
      let dims (a:matrix)       = MG.dims a
      let nrows (a:matrix)     = MG.nrows a
      let ncols (a:matrix)     = MG.ncols a
      
      // Creation
      let init  m n f = DS.initDMDS  m n f |> MS.dense  (* TODO: decide and check which functions should be inline *)
      let of_list    xss   = DS.listDMDS    xss |> MS.dense
      let of_seq     xss   = DS.seqDMDS    xss |> MS.dense
      let diag  (v:vector)   = MG.diag v 
      let constDiag  n x : matrix  = MG.constDiag n x 
      let create  m n x  = DS.constDMDS  m n x |> MS.dense
      let of_scalar x     = DS.scalarDMDS x |> MS.dense

#if CLI_AT_MOST_1_1
#else
      let of_array2 arr : matrix = MG.of_array2 arr
      let to_array2 (m : matrix) = MG.to_array2 m
#endif

      let getDiagN  (a:matrix) n = MG.getDiagN a n
      let getDiag  (a:matrix) = MG.getDiag a

      // Operators
      let add (a:matrix) (b:matrix) = MS.addM   a b
      let sub (a:matrix) (b:matrix) = MS.subM   a b
      let mul (a:matrix) (b:matrix) = MS.mulM   a b
      let mulV (a:matrix) (b:vector) = MS.mulMV   a b
      let mulRV (a:rowvec) (b:matrix) = MS.mulRVM   a b
      let cptMul (a:matrix) (b:matrix) = MS.cptMulM   a b
      let cptMax (a:matrix) (b:matrix) = MS.cptMaxM a b
      let cptMin (a:matrix) (b:matrix) = MS.cptMinM a b
      let scale a (b:matrix) = MS.scaleM   a b
      let neg (a:matrix)  = MS.negM a
      let trace (a:matrix)  = MS.traceM a
      let transpose  (a:matrix) = MG.transpose a
      let forall f (a:matrix) = MG.forall f a
      let exists  f (a:matrix) = MG.exists f a
      let foralli f (a:matrix) = MG.foralli f a
      let existsi  f (a:matrix) = MG.existsi f a
      let map  f (a:matrix) = MG.map f a
      let copy  (a:matrix) = MG.copy a
      let mapi  f (a:matrix) : matrix = MG.mapi f a
      let fold  f z (a:matrix) = MG.fold f z a
      let foldi  f z (a:matrix) = MG.foldi f z a

      let to_dense (a:matrix) = MG.to_dense a 
#if CLI_AT_MOST_1_1
#else
      let init_dense i j a : matrix = MG.init_dense i j a 
      let init_sparse i j a : matrix = MG.init_sparse i j a 
#endif
      let nonzero_entries (a:matrix) = MG.nonzero_entries a 

    #if CLI_AT_LEAST_2_0
      let zero m n  = DS.zeroDMDS m n |> MS.dense
    #else
      let zero m n  : matrix = MG.zero m n 
    #endif
      let identity m  : matrix = MG.identity m 
      
      let ones m n  = create m n 1.0
      
      let getRow (a:matrix) i      = MG.getRow a i
      let getCol (a:matrix) j      = MG.getCol a j
      let getCols (a:matrix) i1 i2    = MG.getCols a i1 i2
      let getRows (a:matrix) j1 j2    = MG.getRows a j1 j2
      let getRegion (a:matrix) i1 j1 i2 j2    = MG.getRegion a i1 j1 i2 j2

      let rowRange a = (0,nrows a - 1)
      let colRange a = (0,ncols a - 1)
      let wholeRegion a = (colRange a, rowRange a)
      
      let foldByRow f (z:Vector<'a>) (a:matrix) = 
        colRange a |> RangeOps.foldR (fun z j -> MS.mapiV (fun i z -> f z (get a i j)) z) z
      let foldByCol f (z:RowVector<'a>) (a:matrix) = 
        rowRange a |> RangeOps.foldR (fun z i -> MS.mapiRV (fun j z -> f z (get a i j)) z) z

      let foldRow f (z:'a) (a:matrix) i = 
        colRange a |> RangeOps.foldR (fun (z:'a) j -> f z (get a i j)) z
      let foldCol f (z:'a) (a:matrix) j = 
        rowRange a |> RangeOps.foldR (fun (z:'a) i -> f z (get a i j)) z

      let sum (a:matrix)  = MS.sumM a
      let prod (a:matrix)  = MS.prodM a
      let norm  (a:matrix) = MS.normM  a
      let dot (a:matrix) b = MS.dotM a b

      let cptPow  a y = map (fun x -> x ** y) a
      
      // Functions that only make sense on this type
      let randomize v = map (fun vij -> MRandom.float vij) v      (* res_ij = random [0,vij] values *)

      let of_rowvec x : matrix = MS.rowvecM x
      let of_vector x : matrix = MS.vectorM x
      let to_vector x : vector = MS.to_vectorM x
      let to_rowvec x : rowvec = MS.to_rowvecM x
      let to_scalar x : float = MS.to_scalarM x

      // Mutation
      let inplace_assign  f (a:matrix) = MG.inplace_assign f a
      let inplace_mapi  f (a:matrix) = MG.inplace_mapi f a
      let inplace_add  (a:matrix) b = MS.inplace_addM a b
      let inplace_sub  (a:matrix) b = MS.inplace_subM a b
      let inplace_cptMul (a:matrix) b = MS.inplace_cptMulM a b
      let inplace_scale  a (b:matrix) = MS.inplace_scaleM a b
      
    end


//----------------------------------------------------------------------------
// module Vector
//--------------------------------------------------------------------------*)
      
    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
    module Vector = begin

      module Generic = begin

        module VS = RawMatrixOps.Specialized

        let get a i   = VS.getV a i
        let set a i x = VS.setV a i x
        let length a     = VS.dimV a
        let nrows a     = VS.dimV a
        let eops a     = VS.eopsV a
        let of_list    xss   = VS.listV xss
        let of_seq    xss   = VS.seqV xss
        let init  m   f = VS.initV m f
        let init_numeric  m   f = VS.createNumericV m f
        let of_array arr       = init (Array.length arr) (Array.get arr)
        let to_array m         = Array.init (length m) (get m)

        let create  m x   = VS.constV m x
        let zero n = VS.zeroV n
        let ones n = VS.createNumericV n (fun ops i -> ops.One)
        let of_scalar   x = VS.scalarV x
        let add a b = VS.addV a b
        let sub a b = VS.subV a b
        let mulRVV a b = VS.mulRVV a b
        let mulVRV a b = VS.mulVRV a b
        let cptMul a b = VS.cptMulV a b
        let cptMax a b = VS.cptMaxV a b
        let cptMin a b = VS.cptMinV a b
        let scale a b = VS.scaleV a b
        let dot a b = VS.dotV a b
        let neg a = VS.negV a 
        let transpose a = VS.transV a 
        let inplace_add a b = VS.inplace_addV a b
        let inplace_sub a b = VS.inplace_subV a b
        let inplace_cptMul a b = VS.inplace_cptMulV a b
        let inplace_scale a b = VS.inplace_scaleV a b
        let exists  f a = VS.existsV  f a
        let forall  f a = VS.forallV  f a
        let existsi  f a = VS.existsiV  f a
        let foralli  f a = VS.foralliV  f a
        let map  f a = VS.mapV f a
        let mapi f a = VS.mapiV f a
        let copy a = VS.copyV a
        let inplace_mapi  f a = VS.inplace_mapiV f a
        let fold  f z a = VS.foldV f z a
        let foldi  f z a = VS.foldiV f z a
        let compare a b = VS.compareV a b
        let hash a = VS.hashV a
        let inplace_assign  f a = VS.assignV f a
        let sum     a = let ops = eops a in fold (fun x y -> ops.Add(x,y)) ops.Zero a
        let prod a = let ops = eops a in fold (fun x y -> ops.Multiply(x,y)) ops.One a
        let norm    a = 
            let normOps = RawMatrixOps.Unspecialized.getNormOps (eops a) 
            sqrt (fold (fun x y -> x + normOps.Norm(y)**2.0) 0.0 a)
      end

      module VG = Generic
      module VDS = RawMatrixOps.DoubleSpecialized
      module VGU = RawMatrixOps.Unspecialized

      let get (a:vector) j   = VG.get a j 
      let set (a:vector) j x = VG.set a j x
      let length (a:vector)     = VG.length a
      let nrows (a:vector)   = VG.length a
      let init  m   f = VDS.createVDS  m   f
      let of_array arr : vector = VG.of_array arr
      let to_array (m : vector) = VG.to_array m

      type range = int * int
      let countR ((a,b) : range)   = (b-a)+1
      let idxR    ((a,b) : range) i = a+i
      type rangef = float * float * float // start, skip, end
      let countRF ((a,d,b) : rangef)   = System.Convert.ToInt32((b-a)/d) + 1
      //let countRF ((a,d,b) : rangef)   = Float.to_int((b-a)/d) + 1
      let idxRF  ((a,d,b) : rangef) i = System.Math.Min (a + d * float(i),b)

      let range n1 n2    = let r = (n1,n2)   in init (countR  r) (fun i -> float(idxR r i)) 

      let rangef a b c  = let r = (a,b,c) in init (countRF r) (fun i -> idxRF r i)

      let of_list    xs    = VDS.listVDS    xs
      let of_seq    xs    = VDS.seqVDS    xs
      let create  m   x  = VDS.constVDS  m   x
      let of_scalar x     = VDS.scalarVDS x
      let add a b = VDS.addVDS   a b
      let sub a b = VDS.subVDS   a b
      let mulRVV a b = VDS.mulRVVDS   a b
      let mulVRV a b = VDS.mulVRVDS   a b 
      let cptMul a b = VDS.cptMulVDS   a b
      let cptMax a b = VDS.cptMaxVDS a b
      let cptMin a b = VDS.cptMinVDS a b
      let scale a b = VDS.scaleVDS   a b
      let neg a  = VDS.negVDS a
      let dot a b = VDS.dotVDS a b
      let transpose  (a:vector) = VG.transpose a
      let exists  f (a:vector) = VG.exists f a
      let forall  f (a:vector) = VG.forall f a
      let existsi  f (a:vector) = VG.existsi f a
      let foralli  f (a:vector) = VG.foralli f a
      let map  f (a:vector) = VG.map f a
      let copy (a:vector) = VG.copy a
      let mapi  f (a:vector) : vector = VG.mapi f a
      let fold  f z (a:vector) = VG.fold f z a
      let foldi  f z (a:vector) = VG.foldi f z a
      let zero n = create n 0.0
      let ones n = create n 1.0
      let sum a  = VDS.sumVDS a
      let prod a   = fold      (fun x y -> x * y) 1.0 a
      let norm  (a:vector) = sqrt (fold (fun x y -> x + y * y) 0.0 a) (* fixed *)
      let cptPow  a y = map  (fun x -> x ** y) a
      let inplace_assign  f (a:vector) = VG.inplace_assign f a
      let inplace_mapi f (a:vector) = VG.inplace_mapi f a
      let inplace_add a b = VDS.inplace_addVDS a b
      let inplace_sub a b = VDS.inplace_subVDS a b
      let inplace_cptMul a b = VDS.inplace_cptMulVDS a b
      let inplace_scale a b = VDS.inplace_scaleVDS a b  
    end


//----------------------------------------------------------------------------
// module RowVector
//--------------------------------------------------------------------------*)

    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
    module RowVector = begin

      module Generic = begin

        module RVS = RawMatrixOps.Specialized

        let get a i   = RVS.getRV a i
        let set a i x = RVS.setRV a i x
        let zero n = RVS.zeroRV n
        let length a  = RVS.dimRV a
        let ncols a   = RVS.dimRV a
        let init m   f       = RVS.initRV m   f
        let create  m x      = RVS.constRV m x
        let transpose a      = RVS.transRV a
        let copy a           = RVS.copyRV a
        let of_list a        = RVS.listRV a
        let of_seq a        = RVS.seqRV a

        let of_array arr       = init (Array.length arr) (Array.get arr)
        let to_array m         = Array.init (length m) (get m)
      end

      module RVG = Generic

      let get (a:rowvec) i   = RVG.get a i 
      let set (a:rowvec) i x = RVG.set a i x
      let length (a:rowvec)  = RVG.length a
      let ncols (a:rowvec)   = RVG.length a
      let of_array arr : rowvec = RVG.of_array arr
      let to_array (m : rowvec) = RVG.to_array m
      
      let init m   f : rowvec      = RVG.init m   f
      let create m   f : rowvec    = RVG.create m   f
      let zero n = create n 0.0
      let of_list x : rowvec       = RVG.of_list x
      let of_seq x : rowvec       = RVG.of_seq x
      let transpose x : vector     = RVG.transpose x
      let copy x : rowvec          = RVG.copy x
    end


//----------------------------------------------------------------------------
// namespace Microsoft.FSharp.{Core,Math.Types} type aliases
//--------------------------------------------------------------------------*)

namespace Microsoft.FSharp.Core

/// The type of floating-point matrices.  See Microsoft.FSharp.Math
type matrix = Microsoft.FSharp.Math.matrix
/// The type of floating-point vectors.  See Microsoft.FSharp.Math
type vector = Microsoft.FSharp.Math.vector
/// The type of floating-point row vectors.  See Microsoft.FSharp.Math
type rowvec = Microsoft.FSharp.Math.rowvec

namespace Microsoft.FSharp.Math.Types

type Matrix<'a> = Microsoft.FSharp.Math.Matrix<'a>
type Vector<'a> = Microsoft.FSharp.Math.Vector<'a>
type RowVector<'a> = Microsoft.FSharp.Math.RowVector<'a>
