(* (c) Microsoft Corporation. All rights reserved *)

(*----------------------------------------------------------------------------
 * Driver for F# compiler. 
 * 
 * Roughly divides into:
 *    - Parsing
 *    - Flags 
 *    - Importing IL assemblies
 *    - Compiling (including optimizing and managing ccu_thunks)
 *    - Linking (including ILX-IL transformation)
 *--------------------------------------------------------------------------*)

(*F# 
module Microsoft.FSharp.Compiler.Driver 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.Research.AbstractIL.Extensions.ILX
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilsupp = Microsoft.Research.AbstractIL.Internal.Support 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Ilread = Microsoft.Research.AbstractIL.BinaryReader 
module Ilwrite = Microsoft.Research.AbstractIL.BinaryWriter 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Il = Microsoft.Research.AbstractIL.IL 
module Check = Microsoft.FSharp.Compiler.PostTypecheckSemanticChecks

open System.Runtime.CompilerServices
F#*)

open Ildiag
open Il
open Ilxgen
open Ast
open Range
open Tc
open Tast
open Tastops
open List
open Opt
open Env
open Printf
open Build
open Lib
open Fscopts


(*----------------------------------------------------------------------------
!* Reporting - warnings, errors
 *--------------------------------------------------------------------------*)

type callExiter = { exit: 'a. int -> 'a }
let exiter_ref = ref { exit = (fun (_ : int) -> failwith "exit!") }
let callExiter i = (!exiter_ref).exit i
let errors = ref 0
let warningOptions = newWarningOptions()
let ignoreFailureOnMono1_1_16 f = try f() with _ -> ()
let ignoreAllFailures f = try f() with _ -> ()

let withErrorColor f =
  (*F# 
#if CLI_AT_MOST_1_1
#else
  let c = System.Console.ForegroundColor in 
  ignoreFailureOnMono1_1_16 (fun () -> System.Console.ForegroundColor <- System.ConsoleColor.Red); 
  try 
#endif
  F#*)
    f()
  (*F# 
#if CLI_AT_MOST_1_1
#else
  finally 
    ignoreFailureOnMono1_1_16 (fun () -> System.Console.ForegroundColor <- c) 
#endif
  F#*)

let reportError err =
  withErrorColor (fun () -> begin
      (buff stderr (output_err false) err;  prerr_newline ())
  end);
  incr errors; 
  if !errors >= !maxErrors then begin
    withErrorColor (fun () -> Printf.eprintf "Exiting - too many errors\n") ; 
    callExiter 1
  end
  
let reportWarning err =
  withErrorColor (fun () -> begin
    if reportWarningAsError warningOptions err then 
      reportError err 
    else if reportWarning warningOptions err then 
      (buff stderr (output_err true) err;  prerr_newline ()) 
  end)

let rec split c str =
  if String.contains str c then
    let i = String.index str c in
    String.sub str 0 i :: split c (String.sub str (i+1) (String.length str - (i+1)))
  else
    [str]

let frontLast xs =
  let rxs = List.rev xs in
  let last,rfront = List.hd rxs,List.tl rxs in
  List.rev rfront,last

(*----------------------------------------------------------------------------
!* writeInterfaceFile
 *--------------------------------------------------------------------------*)

let basic_denv tcConfig tcImports tcGlobals  = 
    let denv = empty_denv tcGlobals in 
    { denv with 
       showImperativeTyparAnnotations=true;
       showAttributes=true;
       openTopPaths=
          begin 
            [ lib_MF_path ] @
            [ lib_MFCore_path ] @
            [ lib_MFColl_path ] @
            [ lib_MFControl_path ] @
            [ (Il.split_namespace lib_FSLib_Pervasives_name); ]  @
             (if not tcConfig.compilingFslib && 
                 not tcConfig.compilingMllib && 
                 tcConfig.useMLLib then 
               [ (Il.split_namespace lib_MLLib_OCaml_name);
                 (Il.split_namespace lib_MLLib_FSharp_name);
                 (Il.split_namespace lib_MLLib_Pervasives_name); ] 
             else []) 
          end } 

let writeInterfaceFile (tcGlobals,tcConfig,tcImports,generatedCcu,TAssembly(declaredImpls)) =
  let os = if !printSignatureFile="" then stdout else open_out !printSignatureFile in
  begin
    declaredImpls |>  List.iter (fun (TImplFile(_,mexpr)) ->
        let denv = basic_denv tcConfig tcImports tcGlobals in 
        buff os (fun os s -> Printf.bprintf os "%s\n\n" s)
          (NicePrint.inferred_sig_of_structL true denv mexpr |> Layout.squashTo 80 |> Layout.showL))
  end;
  if !printSignatureFile="" then flush stdout else close_out os

(*----------------------------------------------------------------------------
!* writeXMLDoc
 *--------------------------------------------------------------------------*)

let getDoc xmlDoc = 
    match processXMLDoc xmlDoc with
    | XMLDoc []   -> ""
    | XMLDoc strs -> String.concat "\n" strs

let writeXMLDoc (assemblyName,fullGenericsSupported,tcGlobals) generatedCcu xmlfile =
  if not (hasSuffixCaseInsensitive xmlfile "xml") then failwith (Printf.sprintf "doc file has no .xml suffix (safety check), file=%s" xmlfile) else
  (* the xmlDocSigOf* functions encode type into string to be used in "id" *)
  let members = ref [] in 
  let addMember id xmlDoc = 
    let doc = getDoc xmlDoc in
    members := (id,doc) :: !members in
  let g = tcGlobals in                                
  let do_val ptext    v  = addMember (xmlDocSigOfVal g ptext v)   (xmldoc_of_val v) in
  let do_excon ptext  e  = addMember (xmlDocSigOfTycon g ptext e)   (xmldoc_of_exnc e) in
  let do_tycon ptext tc = 
    addMember (xmlDocSigOfTycon g ptext tc) (xmldoc_of_tycon tc);
    iter (deref_val >> do_val ptext) (adhoc_of_tycon tc) in 

  let modulMember path m = addMember (xmlDocSigOfSubModul g path) (xmldoc_of_modul m) in
  (* moduleSpec - recurses *)
  let rec do_modul path mspec = 
    let mtype = mtyp_of_modul mspec in 
    let path = 
      (* skip the first item in the path which is the assembly name *)
      match path with 
      | None -> Some ""
      | Some "" -> Some (name_of_modul mspec)
      | Some p -> Some (p^"."^name_of_modul mspec) in 
    let ptext = match path with None -> "" | Some t -> t in 
    if mkind_of_mtyp mtype <> Namespace then modulMember ptext mspec;
    let vals = 
      Namemap.range mtype.mtyp_vals
      |> filter (compgen_of_val >> not) 
      |> filter (fun x -> member_info_of_val x = None) in
    iter (do_modul  path)  (all_submoduls_of_mtyp mtype);
    let excons,tycons = mtype.mtyp_tycons |> Namemap.range |> List.partition tycon_is_exnc in 
    iter (do_excon  ptext) excons;
    iter (do_val    ptext) vals;
    iter (do_tycon  ptext) tycons
  in
  begin
    do_modul None (top_modul_of_ccu generatedCcu);
    let os = open_out xmlfile in
    Printf.fprintf os "\239\187\191"; (* 239=0xef,187=0xbb,191=0xbf - byte order mark*)
    Printf.fprintf os ("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
    Printf.fprintf os ("<doc>\n");
    Printf.fprintf os ("<assembly><name>%s</name></assembly>\n") assemblyName;
    Printf.fprintf os ("<members>\n");
    List.iter (fun (id,doc) -> Printf.fprintf os  "<member name=\"%s\">\n%s\n</member>\n" id doc) !members;
    Printf.fprintf os ("</members>\n"); 
    Printf.fprintf os ("</doc>\n");   
    flush os;
    close_out os
  end


(*----------------------------------------------------------------------------
!* writeHTMLDoc
 *--------------------------------------------------------------------------*)

let pseudo_case_insensitive a b = 
  let c1 = compare (String.lowercase a) (String.lowercase b) in 
  if c1 <> 0 then c1 else 
  (* put upper case second  - currently a hack to get real definitions like 'Code' to come after abbreviations 'code *)
  -(compare a b)

open Layout
let width_val  = 80
let width_type = 120
let width_exn  = 80
let outputL width layout =
  (* String for layout squashed to a given width.
   * HTML translation and markup for TYPE and VAL syntax.
   * Assumes in <pre> context (since no explicit <br> linebreaks).
   *)
  let render = htmlR stringR in
  let render = {render with
                  addTag = fun z (tag,attrs,start) -> match tag,start with
                      "TYPE",_     -> z
                    | "VAL" ,true  -> render.addText z "<B>"
                    | "VAL" ,false -> render.addText z "</B>"
                    | _     ,start -> render.addTag z (tag,attrs,start)} in
  renderL render (squashTo width layout)

type kind = PathK | TypeK

let writeHTMLDoc (tcConfig,tcImports,tcGlobals) generatedCcu outdir append css nsfile =
  let assemblyName = (name_of_ccu generatedCcu) in
  let outdir = match outdir with None -> "." | Some x -> x in 
  let rconcat a b = a^"/"^b in  
  let css = match css with None -> "msdn.css" | Some x -> x in 
  let nsfile = match nsfile with None -> "namespaces.html" | Some x -> x in
  let nsfullfile = Filename.concat outdir nsfile in 
  let nsbfilename = (rconcat ".." nsfile) in (* the name to use to link up to the namespace file *)    
  let wrap oc a b f = 
    output_string oc a;
    f ();
    output_string oc b in

  let newExplicitFile append fullname bfilename title css f = 
    let oc = open_out_gen ( if append && fileExists fullname 
                            then [ Open_wronly; Open_append; Open_text ] 
                            else [ Open_wronly; Open_creat; Open_trunc;  Open_text ]) 0x1FF (* 0o777 *) fullname in 
    fprintf oc "<HTML><HEAD><TITLE>%s</TITLE><link rel=\"stylesheet\" type=\"text/css\"href=\"%s\"></link></HEAD><BODY>\n" title css;
    let backlink = (bfilename,title) in
    f backlink oc;
    fprintf oc "<br /> <br/><p><i>Documentation for assembly %s, version %s, generated using <a href='http://research.microsoft.com/fsharp/'>F#</a> version %s</i></p>" assemblyName (match !Fscopts.version with None -> "0.0.0.0" | Some v -> version_to_string v) Ilxconfig.version;
    fprintf oc "</BODY></HTML>\n";
    close_out oc; in

  let newFile fdir fname title f =
    let fullname = (Filename.concat outdir (Filename.concat fdir fname)) in  
    newExplicitFile false fullname fname title (rconcat ".." css) f in

  let hlink url text = sprintf "<a href='%s'>%s</a>" url text in    
 
  (* Path *)
  let path0        = [] in
  let path1 x kind = [(x,kind)] in
  let pathExtend xs x kind = xs @ [x,kind] in (* short shallow lists *)
  let pathText     xs = String.concat "." (map fst xs) in        
  let pathFilename xs =
    let encode = function
      | x,PathK  -> x
      | x,TypeK  ->
          (* slight hack to mangle colliding upper/lower names, 'complex' and 'Complex', to different filenames *)
          "type_" ^ (underscore_lowercase x) (* see also tastops.ml which prints type hrefs *) in
    String.concat "." (map encode xs) ^ ".html" in

  let collapseStrings xs = String.concat "." xs in
  let pathWrt knownNamespaces x =
    let xs = split '.' x in    
    let rec collapse front back = 
      match back with 
      | [] -> (if front = [] then [] else [collapseStrings front]) @ back
      | mid::back -> 
          if List.mem (collapseStrings (front@[mid])) knownNamespaces 
          then [collapseStrings (front@[mid])] @ back
          else collapse (front@[mid]) back in
    map (fun x -> (x,PathK)) (collapse [] xs) in

  let nestBlock hFile f =
    wrap hFile (sprintf "<br><dl>\n")
               (sprintf "</dl>\n")
       (fun () -> f()) in
  let nestItem hFile f =
    wrap hFile (sprintf "<dt></dt><dd>\n")
               (sprintf "</dd>\n")
       (fun () -> f()) in
    
  (* TopNav - from paths *)
  let newPathTrail hFile kind ptext =
    let rec writer prior = function
        []         -> ()
      | [x,k]      -> fprintf hFile "%s " x
      | (x,k)::xks -> let prior = pathExtend prior x k in
                      let url   = pathFilename prior in
                      let sep   = if xks=[] then "" else "." in
                      let item  = hlink url x in
                      fprintf hFile "%s%s" item sep;
                      writer prior xks in
    let uplink = sprintf "[<a href='%s'>Home</a>] " nsbfilename in
    nestItem hFile (fun () ->
      fprintf hFile "<h1>%s%s " uplink kind;
      writer path0 ptext;
      fprintf hFile "</h1>";
      fprintf hFile "<br>\n") in

  let newPartitions hFile f =
     nestItem hFile (fun () ->
       wrap hFile (sprintf "<table>\n") (sprintf "</table>\n") (fun () -> 
         f())) in
  
  let newPartition hFile desc f =
    wrap hFile (sprintf "  <tr valign='top'><td>%s\n" desc) (sprintf "  </td></tr>\n") (fun () -> 
      f()) in 

  let newSectionInPartition hFile title f =
     wrap hFile (sprintf "  <dt><h3>%s</h3></dt><dd>\n" title) (sprintf "  </dd>\n") (fun () -> 
       f()) in 

  let newPartitionsWithSeeAlsoBacklink hFile (bfilename,btitle) f = 
    newPartitions hFile (fun () ->
      f();
      newPartition hFile "" (fun () -> 
        newSectionInPartition hFile "See Also" (fun () -> 
          fprintf hFile "<a href=\"%s\">%s</a>" bfilename btitle))) in

  let newTable0 hFile title f = 
    newSectionInPartition hFile title (fun () -> 
      wrap hFile "<table width=\"100%%\">\n" "</table>" (fun () -> 
        f())) in 

  let newTable2 hFile title width1 h1 h2 f = 
    newTable0 hFile title (fun () -> 
      wrap hFile (sprintf "<tr><th width=%d%%>%s</th><th>%s</th></tr>\n" width1 h1 h2) "" (fun () -> 
        f())) in 
  
  let newNamespaceEntry hFile fdir ptext allmods desc =
    let fname = pathFilename ptext in
    let title = pathText ptext in    
    let url = fdir ^ "/" ^ fname in
    fprintf hFile "<tr valign='top'><td width='50%%'><a href='%s'>%s</a>\n" url title;
    (* Sort sub-modules into alpha order *)
    let moduleIsModuleNotNS modul = (mtyp_of_modul modul).mtyp_kind <> Namespace in
    let allmods = filter moduleIsModuleNotNS allmods in      
    let allmods = allmods |> sort (orderOn name_of_modul pseudo_case_insensitive) in
    (* Make them hyperlink to fdir/<path>.html *)
    let links = map (fun modul ->
                       let ptext = pathExtend ptext (name_of_modul modul) PathK in
                       let url   = fdir ^ "/" ^ pathFilename ptext in
                       hlink url (sprintf "[%s]" (name_of_modul modul))) allmods in
    fprintf hFile "</td>\n" ;
    fprintf hFile "<td>%s<br>%s</td></tr>\n" desc (String.concat ", " links);    
    () in

  let newEntry     hFile0 title desc = 
    fprintf hFile0 "<tr valign=\"top\"><td>%s</td><td>%s</td></tr>\n" title desc in

  let denv = basic_denv tcConfig tcImports tcGlobals in 
  let denv = 
    { denv with 
      html = true;
      htmlAssemMap = 
       Namemap.of_list 
         (let fslib_default = [(fslib(),"http://research.microsoft.com/fsharp/manual/fslib");] in 
          let mllib_default = [(mllib(),"http://research.microsoft.com/fsharp/manual/mllib");] in 
          if tcConfig.compilingFslib or tcConfig.compilingMllib 
          then []
          else (fslib_default @ mllib_default));  } in 
  newExplicitFile append nsfullfile nsbfilename "Namespaces" css (fun blinkNamespacesFile hNamespacesFile -> 

    wrap hNamespacesFile (sprintf "<dl><dt><br/></dt><dd><table><tr><th>Namespaces in assembly %s</th><th>Description</th></tr>\n" assemblyName)
                         (        "</table></dl>\n") (fun () ->

      let obsoleteText attribs = 
          match (fsthing_tryfind_attrib tcGlobals tcGlobals.attrib_SystemObsolete attribs) with
          | Some(Attrib(_,(TExpr_const (TConst_string(bytes),_,_) :: _),_)) ->
            sprintf "<p><b>Note</b>: %s</p>" (Bytes.unicode_bytes_as_string bytes) 
          | _ -> "" in
      let isObsolete attribs = fsthing_tryfind_attrib tcGlobals tcGlobals.attrib_SystemObsolete attribs <> None in 
      let rec do_val denv fdir hFile ptext v = 
          let denv = { denv with htmlHideRedundantKeywords=true } in
          newEntry hFile ("<pre>"^outputL width_val (NicePrint.valSpecL denv v)^"</pre>") (obsoleteText (attribs_of_val v)^getDoc (xmldoc_of_val v)) in

      let rec do_vals denv fdir hFile ptext title item vals = 
          let vals = vals |> filter (compgen_of_val >> not) in 
          let vals = vals |> sort (orderOn display_name_of_val pseudo_case_insensitive) in 
          if vals <> [] then 
            newTable2 hFile title 60 item  "Description" (fun () -> 
              vals |> iter (do_val denv fdir hFile ptext)) in 

      let rec do_tycon denv fdir hFile ptext tycon = 
          newSectionInPartition hFile "Full Type Signature" (fun () ->
            fprintf hFile "<pre>%s</pre>" (outputL width_type (NicePrint.tyconSpecL true denv (wordL "type") tycon)));
          let vals = adhoc_of_tycon tycon |> map deref_val in 
          let val_is_instance v = 
            assert (member_info_of_val v <> None);
            (the (member_info_of_val v)).vspr_flags.memFlagsInstance in 
          let dvals,vals = partition (attribs_of_val >> isObsolete) vals in 
          let ivals,svals = partition val_is_instance vals in 
          do_vals denv fdir hFile ptext "Instance Members" "Member" ivals;
          do_vals denv fdir hFile ptext "Static Members" "Member"   svals;
          do_vals denv fdir hFile ptext "Deprecated Members" "Member" dvals; in

      let rec do_tycons denv fdir blinkFile hFile ptext title tycons = 
        if tycons <> [] then  
          newTable2 hFile title 30 "Type" "Description" (fun () -> 
            let tycons = tycons |> sort (orderOn display_name_of_tycon pseudo_case_insensitive) in 
            tycons |> iter (fun tycon ->
              let tyname = display_name_of_tycon tycon in
              let ptext  = pathExtend ptext tyname TypeK in 
              let fname  = pathFilename ptext in
              let title  = pathText ptext in  (* used as html page title *)
              let text = obsoleteText (attribs_of_tycon tycon) in
              let text = text^(getDoc (xmldoc_of_tycon tycon)) in 
              let text = 
                match abbrev_of_tycon tycon with 
                | None -> text
                | Some ty -> text ^ " Note: an abbreviation for "^("<tt>"^outputL width_type (NicePrint.typL denv ty)^"</tt>") in
              newEntry hFile ("type " ^ hlink fname tyname) text;
              newFile fdir fname title (fun blinkFile2 hFile2 ->
                nestBlock hFile2 (fun () ->
                  newPathTrail hFile2 "Type" ptext;
                  newPartitionsWithSeeAlsoBacklink hFile2 blinkFile  (fun () -> 
                    newPartition hFile2 text (fun () ->
                      do_tycon denv fdir hFile2 ptext tycon)))))) in
      
      let rec do_exnc denv fdir hFile ptext exnc = 
        newSectionInPartition hFile "Full Signature" (fun () ->
          fprintf hFile "<pre>%s</pre>" (outputL width_exn (NicePrint.exnSpecL denv exnc))) in

      let rec do_exncs denv fdir blinkFile hFile ptext exncs = 
        if exncs <> [] then  
          newTable2 hFile "Exceptions" 40 "Exception" "Description" (fun () -> 
            let exncs = exncs |> sort (orderOn demangled_name_of_exnc pseudo_case_insensitive) in 
            exncs |> iter (fun exnc ->
              let ptext  = pathExtend ptext (demangled_name_of_exnc exnc) TypeK in 
              let fname  = pathFilename ptext in
              let exname = demangled_name_of_exnc exnc in
              let title  = pathText ptext in  (* used as html page title *)
              let text = obsoleteText (attribs_of_exnc exnc) in
              let text = text^(getDoc (xmldoc_of_exnc exnc)) in 
              let text = 
                match exn_repr_of_tycon exnc with
                | TExnAbbrevRepr ecref as r -> text^" Note: an abbreviation for "^("<tt>"^outputL width_exn (NicePrint.exnSpecReprL denv r)^"</tt>")
                | _ -> text in
              newEntry hFile ("exception " ^ hlink fname exname) text;
              newFile fdir fname title (fun blinkFile2 hFile2 ->
                nestBlock hFile2 (fun () ->
                  newPathTrail hFile2 "Exception" ptext;
                  newPartitionsWithSeeAlsoBacklink hFile2 blinkFile  (fun () -> 
                    newPartition hFile2 text (fun () ->
                      do_exnc denv fdir hFile2 ptext exnc)))))) in
      
      let rec do_modul denv fdir ptext blinkFile hFile modul = 
        let denv = denv_add_open_modref (mk_local_modref modul) denv in 
        let mtyp = mtyp_of_modul modul in 
        let moduls = all_submoduls_of_mtyp mtyp 
                     |> sort (orderOn name_of_modul pseudo_case_insensitive) in
        
        do_moduls denv fdir blinkFile hFile ptext moduls; 
        let exncs,tycons = mtyp.mtyp_tycons |> Namemap.range |> List.partition tycon_is_exnc in 
        let dtycons,tycons= partition (attribs_of_tycon >> isObsolete) tycons in 
        do_tycons denv fdir blinkFile hFile ptext "Type Definitions" tycons;
        do_exncs denv fdir blinkFile hFile ptext exncs;
        let vals = Namemap.range mtyp.mtyp_vals |> filter (fun x -> member_info_of_val x = None) in
        let dvals,vals = partition (attribs_of_val >> isObsolete) vals in 
        do_vals denv fdir hFile ptext "Values" "Value" vals; 
        do_tycons denv fdir blinkFile hFile ptext "Deprecated/Unsafe Type Definitions" dtycons;
        do_vals denv fdir hFile ptext "Deprecated Values" "Value" dvals 

      and do_moduls denv fdir blinkFile hFile ptext moduls = 
        if moduls <> [] then  
          newTable2 hFile (sprintf "Modules (as contributed by assembly '%s')" assemblyName) 30 "Module" "Description" (fun () -> 
            let moduls = moduls |> sort (orderOn name_of_modul pseudo_case_insensitive) in 
            moduls |> iter (fun modul -> 
              let mtyp = mtyp_of_modul modul in 
              let ptext = pathExtend ptext (name_of_modul modul) PathK in 
              let fname = pathFilename ptext in
              let title = pathText ptext in
              (* let prefix = if modul.mtyp_kind then "module " else "namespace " in*)
              let text = obsoleteText (attribs_of_modul modul) in
              let text  = getDoc (xmldoc_of_modul modul) in
              newEntry hFile ((* prefix ^ *) hlink fname title) text;
              newFile fdir fname title (fun blinkFile2 hFile2 ->
                nestBlock hFile2 (fun () ->
                  newPathTrail hFile2 "Module" ptext;
                  newPartitionsWithSeeAlsoBacklink hFile2 blinkFile  (fun () -> 
                    newPartition hFile2 text (fun () ->
                      do_modul denv fdir ptext blinkFile2 hFile2 modul)))))) in

      let rec do_namespace denv fdir ptext blinkFile hFile moduls tycons exncs = 
        do_moduls denv fdir blinkFile hFile ptext moduls;
        let dtycons,tycons= partition (attribs_of_tycon >> isObsolete) tycons in 
        do_tycons denv fdir blinkFile hFile ptext "Type Definitions" tycons;
        do_exncs denv fdir blinkFile hFile ptext exncs;
        do_tycons denv fdir blinkFile hFile ptext "Deprecated/Unsafe Type Definitions" dtycons in
        
      let rec do_possible_namespace fdir path knownNamespaces mspec = 
        let mtype = mtyp_of_modul mspec in
        let path = 
          (* skip the first item in the path which is the assembly name *)
          match path with 
          | None    -> Some ""
          | Some "" -> Some (name_of_modul mspec)
          | Some p  -> Some (p^"."^name_of_modul mspec) in 
        let ptext = match path with
        | None   -> path0
        | Some t -> pathWrt knownNamespaces t in
        let allmods = all_submoduls_of_mtyp mtype 
                      |> sort (orderOn name_of_modul pseudo_case_insensitive) in 
        let moduls,nsps = partition (fun m -> (mtyp_of_modul m).mtyp_kind <> Namespace) allmods in
        let exncs,tycons = mtype.mtyp_tycons |> Namemap.range |> List.partition tycon_is_exnc in 
        let knownNamespaces =
          if mtype.mtyp_kind = Namespace && (moduls <> [] or tycons <> [] or exncs <> []) then begin
            let fname = pathFilename ptext in
            let title = pathText ptext in
            newNamespaceEntry hNamespacesFile fdir ptext allmods (getDoc (xmldoc_of_modul mspec));
            newFile fdir fname title (fun blinkFile2 hFile2 ->
              nestBlock hFile2 (fun () ->
                newPathTrail hFile2 "Namespace" ptext;
                newPartitionsWithSeeAlsoBacklink hFile2 blinkNamespacesFile  (fun () -> 
                  newPartition hFile2 "" (fun () ->
                    do_namespace denv fdir ptext blinkFile2 hFile2 moduls tycons exncs))));
            title :: knownNamespaces
          end else
            knownNamespaces
        in
        iter (do_possible_namespace fdir path knownNamespaces)  nsps; 
      in
      do_possible_namespace assemblyName None [] (top_modul_of_ccu generatedCcu)))

(*----------------------------------------------------------------------------
!* cmd line - option state
 *--------------------------------------------------------------------------*)

(*IF-OCAML*) external camlStub_GetModuleFileName:  unit -> string  = "GetModuleFileNameCaml" (*ENDIF-OCAML*)

(* Workaround a bug in OCaml 3.08.2 *)
let getModuleFileName() = 
  (*IF-OCAML*) try camlStub_GetModuleFileName () with _ -> (*ENDIF-OCAML*)
  Sys.executable_name

let fsharpBinariesDir = Filename.dirname (getModuleFileName())

(*----------------------------------------------------------------------------
!* decideNames - infer from the command line arguments (output assembly name etc.)
 *--------------------------------------------------------------------------*)

let outpath outfile extn =
  String.concat "." (["out"; Filename.chop_extension (Filename.basename outfile); extn])
  
let decideNames parseInputs =
    (* return: outfile,pdbfile,assemblyName *)
    if verbose then dprint_endline ("Computing assembly details... ");     
    try 
        if parseInputs = [] then failwith "no code to link: an empty binary would be produced";
        let ext() = match !target with Dll -> ".dll" | Module -> ".netmodule" | Exe | WinExe -> ".exe" in
        let outfile = 
          match !outputFileRef, List.rev (chooseList (function ImplFileInput x -> Some(x)| _ -> None) parseInputs) with 
          | None,[] -> "out" ^ ext()
          | None, (ImplFile(h,_,_) :: _ ) -> 
              let basic = Filename.basename h in 
              let modname = try Filename.chop_extension basic with _ -> basic in
              modname^(ext())
          | Some f,_ -> f in
        let assemblyName = 
          let base = Filename.basename outfile in 
          if not (Filename.check_suffix (String.lowercase base) (ext())) then
            failwith "The output name extension doesn't match the flags used. If -a is used the output file name must end with .dll, if --target-module is used the output extension must be .netmodule, otherwise .exe ";
          Filename.chop_suffix base (ext()) in

        let pdbfile = 
            if !debuginfo 
            then Some (match !debugSymbolFileRef with None -> (Filename.chop_extension outfile)^"."^(Ilsupp.pdb_suffix_for_configuration (Ilsupp.current_configuration())) | Some f -> f) 
            else None in
    

        outfile,pdbfile,assemblyName

    with e -> errorRecoveryPoint e; callExiter 1


(*----------------------------------------------------------------------------
!* typeCheck
 *--------------------------------------------------------------------------*)

let typeCheck initm (tcConfig,tcImports,tcGlobals,assemblyName,niceNameGen,tcEnv0) parseInputs =
    try 
      if parseInputs = [] then failwith "no implementation files specified";
      let ccuName = assemblyName in 
      let tcInitialState = typecheckInitialState initm ccuName tcConfig tcGlobals niceNameGen  tcEnv0 in
      typecheckClosedInputSet (fun () -> !errors = 0) tcConfig tcImports tcGlobals None tcInitialState parseInputs
    with e -> errorRecoveryPoint e; callExiter 1


(*----------------------------------------------------------------------------
!* encodeInterfaceData
 *--------------------------------------------------------------------------*)

let generateInterfaceData() = (!target = Dll or !target = Module) && not !standalone && not !noSignatureData 

let encodeInterfaceData tcConfig tcGlobals exportRemapping generatedCcu = 
  try 
    if generateInterfaceData() then begin
      if verbose then dprintf0 "Generating interface data attribute...\n";
      let resource = write_IntfData tcConfig tcGlobals exportRemapping generatedCcu in 
      if verbose then dprintf0 "Generated interface data attribute!\n";
      [mk_IntfDataVersionAttr tcGlobals (Il.parse_version Ilxconfig.version) ], [resource]
    end else begin 
      [],[]
    end
  with e -> errorRecoveryPoint e; callExiter 1


(*----------------------------------------------------------------------------
!* encodeOptimizationData
 *--------------------------------------------------------------------------*)

let generateOptimizationData() = (!target =Dll or !target = Module) && not !standalone 

let encodeOptimizationData tcGlobals outfile exportRemapping data = 
  if not (generateOptimizationData()) then [] else 
 
  let data = map2'2 (Opt.remapModulInfo tcGlobals exportRemapping) data in 
  if verbose then dprint_endline "Generating optimization data attribute...";
  if !useOptimizationDataFile then (
      let ccu,modulInfo = data in 
      let bytes = Pickle.pickle_obj_with_dangling_ccus ccu Opt.pmodul_info modulInfo in 
      let optDataFileName = (Filename.chop_extension outfile)^".optdata" in
      writeBinaryFile optDataFileName bytes);
  let data = 
      if !onlyEssentialOptimizationData or !useOptimizationDataFile 
      then map2'2 Opt.abstractModulInfoToEssentials data 
      else data in
  [ write_OptData data ]


(*----------------------------------------------------------------------------
!* Final linking: hack some stuff into the standard library. A little
 * gross but useful as it means we can write a fair bit of the
 * library in C#.
 *--------------------------------------------------------------------------*)

let createMainModule  
        (tcConfig,tcImports,mscorlibRefs,desiredCLIMetadataVersion) 
        (pdbfile,assemblyName,outfile) 
        ((iattrs,intfDataResources),optDataResources) 
        (codegenResults: Ilxgen.codegenResults) =


  let mainmodx = 
    if !progress then dprintf0 "Creating main module...\n";
    let ilTypeDefs = 
        let top_tdef = mk_toplevel_tdef mscorlibRefs (mk_mdefs [], mk_fdefs []) in
        mk_tdefs codegenResults.ilTypeDefs in

    let manifest_module = 
        mk_simple_mainmod assemblyName (fsharpModuleName assemblyName) (!target = Dll or !target = Module) ilTypeDefs in 
              
    let quotDataResources = 
            codegenResults.quotationResourceBytes |> List.map (fun bytes -> 
                { resourceName=Sreflect.Raw.pickledDefinitionsResourceNameBase^string_of_int(new_uniq());
                  resourceWhere = Resource_local (fun () -> bytes);
                  resourceAccess= Resource_public;
                  resourceCustomAttrs = mk_custom_attrs [] })  in

    let resources = 
      mk_resources 
        ((!embedResources |> List.map 
           (fun (bytes,name,pub) -> 
             (* Read the bytes and throw the length on the front *)
             { resourceName=name; 
               resourceWhere=Resource_local (fun () -> bytes); 
               resourceAccess=pub; 
               resourceCustomAttrs=mk_custom_attrs [] }))
           
         @ quotDataResources
         @ intfDataResources
         @ optDataResources
         @ List.map 
           (fun ri -> 
             let file,name,pub = splitResourceInfo ri in 
             { resourceName=name; 
               resourceWhere=Resource_file(  { modulRefName=file; modulRefNoMetadata=true; modulRefHash=Some (sha1_hash_bytes (readBinaryFile file)) }, Int32.of_int 0);
               resourceAccess=pub; 
               resourceCustomAttrs=mk_custom_attrs [] })
           !linkResources) in 

    let nativeResources = 
      match !win32res with 
      | [] -> None
      | [h] -> Some (Lazy.lazy_from_val (readBinaryFile h))
      | _ -> failwith "Cannot currently emit more than one native resource" in 

    (* Add attributes, version number, resources etc. *)
    {manifest_module with 
          modulName = (if !target = Module then Filename.basename outfile else manifest_module.modulName);
          modulSubSystem = (if !target = WinExe then Int32.of_int 2 else Int32.of_int 3) ;
          modulResources= resources;
          modulImageBase = (match !baseAddress with None -> 0x00400000l | Some b -> b);
          modulDLL=(!target = Dll or !target=Module);
          modul32bit=(match !platform with Some X86 -> true | None -> false);
          modulCustomAttrs= mk_custom_attrs ((if !target = Module then iattrs else []) @ codegenResults.ilNetModuleAttrs);
          modulNativeResources=nativeResources } in 
  let mainmodx = 
    {mainmodx 
     with modulManifest = 
       if !target = Module then None 
       else
         let disableJitOptimizations = not (jitopt()) in 
         let attrs = 
           begin 
             if !internConstantStrings then [] 
             else
               [ mk_custom_attribute mscorlibRefs
                   (mk_tref (mscorlibRefs.mscorlib_scoref, "System.Runtime.CompilerServices.CompilationRelaxationsAttribute"),
                    [mscorlibRefs.typ_Int32],[CustomElem_int32( Int32.of_int 8)], []) ]
           end 
           @ iattrs
           @ codegenResults.ilAssemAttrs
           @
           begin 
             let generate_pdb = pdbfile <> None in 
             if generate_pdb then 
               [
               (if Il.version_compare desiredCLIMetadataVersion (Il.parse_version "2.0.0") < 0 then 
                 mk_DebuggableAttribute mscorlibRefs (!jitTracking, disableJitOptimizations)
               else
                 mk_DebuggableAttribute_v2 mscorlibRefs (!jitTracking, false (* ignoreSymbolStoreSequencePoints *) , disableJitOptimizations, false (* enableEnC *) ))
             ] 
             else [] 
           end in 
         
         let man = manifest_of_mainmod mainmodx in 
         Some { man with manifestVersion= !version;
                         manifestCustomAttrs = mk_custom_attrs attrs;
                         manifestDisableJitOptimizations=disableJitOptimizations;
                         manifestJitTracking= !jitTracking } } in

  mainmodx


(*----------------------------------------------------------------------------
!* ILX --> IL
 *--------------------------------------------------------------------------*)

let eraseILX (mscorlibRefs,manager,ilxMainModule) =
  if !progress then dprint_endline "Generating IL code...";
  
  Ilxerase.conv_module mscorlibRefs manager ilxMainModule
  


(*----------------------------------------------------------------------------
!* OPTIONAL STATIC LINKING OF ALL DLLs THAT DEPEND ON THE F# LIBRARY
 *--------------------------------------------------------------------------*)

type node = 
    { name: Il.assembly_name;
      data: Il.modul; 
      refs: Il.refs;
      mutable edges: node list; 
      mutable visited: bool }


let staticLinkModules tcConfig mscorlibRefs ilxMainModule dep_moduls = 
  if dep_moduls=[] then ilxMainModule,(fun x -> x) else
  let assems = List.map name_of_module (List.filter module_is_mainmod dep_moduls) in

  (* Rewrite scope references to be local references *)
  let rewriteExternalRefsToLocalRefs x = 
    if List.mem (name_of_scoref x) assems then ScopeRef_local else x in 
  let saved_resources = 
    let all_resources = 
      dep_moduls 
      |>  List.map (fun m -> dest_resources m.modulResources)
      |> List.concat in 
    (* Save only the interface/optimization attributes of generated data *)
    let intfDataResources,others = 
      let intfDataResources,others = List.partition is_IntfDataResource all_resources in 
      let intfDataResources = if generateInterfaceData()  then intfDataResources else [] in 
      intfDataResources,others in 
    let optDataResources,others = 
      let optDataResources,others = List.partition is_OptDataResource others in 
      let optDataResources = if generateOptimizationData()  then optDataResources else [] in 
      optDataResources,others in 
    let rresources,others = 
      let rresources,others = List.partition is_PickledDefinitionsResource others in 
      let rresources = rresources |> list_mapi (fun i r -> {r with resourceName = Sreflect.Raw.pickledDefinitionsResourceNameBase^string_of_int (i+1)}) in 
      rresources,others in 
    if verbose then dprintf3 "#intfDataResources = %d, #optDataResources = %d, #rresources = %d\n" (length intfDataResources) (length optDataResources)(length rresources);
    intfDataResources@optDataResources@rresources@others in 
  let moduls = ilxMainModule :: dep_moduls in 
  let top_tdefsl,normal_tdefsl = List.split (List.map (fun m -> List.partition (fun td -> is_toplevel_tname td.tdName) (dest_tdefs m.modulTypeDefs)) moduls) in
  let top_tdef = 
    let top_tdefs = List.concat top_tdefsl in
    mk_toplevel_tdef mscorlibRefs
       (mk_mdefs (List.concat (List.map (methods_of_tdef >> dest_mdefs) top_tdefs)),
        mk_fdefs (List.concat (List.map (fields_of_tdef >> dest_fdefs) top_tdefs))) in 
  let ilxMainModule = 
   { ilxMainModule with 
       modulManifest = (let m = (manifest_of_mainmod ilxMainModule) in Some {m with manifestCustomAttrs = mk_custom_attrs (dest_custom_attrs m.manifestCustomAttrs)});
       modulCustomAttrs = mk_custom_attrs (List.concat (List.map (fun m -> dest_custom_attrs m.modulCustomAttrs) moduls));
       modulTypeDefs = mk_tdefs (top_tdef :: List.concat normal_tdefsl);
       modulResources = mk_resources (saved_resources @ dest_resources ilxMainModule.modulResources);
       modulNativeResources = 
          begin 
            let resl = concat (map (fun x -> Option.to_list x.modulNativeResources) moduls) in
            match resl with 
            | [] -> None
            | [h] -> Some h
            | _ -> failwith "standalone DLLs cannot currently incorporate more than one unmanaged resource";
          end;
      (* modulFixups: fixups; *) } in 
  ilxMainModule, rewriteExternalRefsToLocalRefs


let printModule = output_to_file Ilprint.output_module

let staticLink (tcConfig,tcImports,mscorlibRefs) ilxMainModule outfile = 
  if not !standalone && !extraStaticLinkRoots = [] then ilxMainModule, (fun x -> x)
  else 
      begin
        (* Recursively find all referenced modules and add them to a module graph *)    
        let dep_modul_tab = Hashtbl.create 0 in
        let get_node n = Hashtbl.find dep_modul_tab n in 
        if !progress then dprint_endline "Performing static link...";
        begin 
          let remaining = ref (refs_of_module ilxMainModule).refsAssembly in 
          while !remaining <> [] do
            let aref = hd !remaining in 
            remaining := tl !remaining;
            if not (Hashtbl.mem dep_modul_tab aref.assemRefName) then begin
              if verbose then dprint_endline ("Finding "^aref.assemRefName);
              let modul = (findDllInfo tcImports Range.rangeStartup aref.assemRefName).dllinfo_il_modul in 
              if !progress then dprint_endline ("Finding references for "^aref.assemRefName);
              let refs = 
                match aref.assemRefName with 
                  ("mscorlib" | "FSharp.Core") -> Il.empty_refs
                |  x when List.mem x extended_framework or List.mem x core_framework -> Il.empty_refs
                | _ -> 
                   try refs_of_module modul 
                   with e -> 
                     eprintf "** warning: could not determine dependencies of assembly %s\n** reason: %s\n" modul.modulName (Printexc.to_string e); 
                     Il.empty_refs  in 
              Hashtbl.add dep_modul_tab aref.assemRefName { refs = refs;
                                                            name=aref.assemRefName;
                                                            data=modul; 
                                                            edges = []; 
                                                            visited = false };
              remaining := refs.refsAssembly @ !remaining;
            end;
          done;
        end;
        reportPhase "Find assembly references";
        (* Add edges from modules to the modules that depend on them *)
        begin 
          Hashtbl.iter
            (fun _ n -> n.refs.refsAssembly |> List.iter(fun aref -> let n2 = get_node aref.assemRefName in n2.edges <- n :: n2.edges)  ) 
            dep_modul_tab;
        end;
        (* Find everything that depends on fslib *)
        let dep_modules = 
            let fslib_roots = if !standalone && Hashtbl.mem dep_modul_tab (fslib()) then [get_node(fslib())] else [] in 
            let other_roots = 
                !extraStaticLinkRoots 
                |> List.map (fun n -> try [get_node n] with Not_found -> errorR(Failure("Assembly "^n^" not found in dependency set of target application, ignoring")); [])
                |> List.concat in 
            let remaining = ref (fslib_roots@other_roots) in 
            let dep_modules = ref [] in 
            while !remaining <> [] do
                let n = hd !remaining in 
                remaining := tl !remaining;
                if not (n.visited) then begin
                    if verbose then dprint_endline ("Module "^n.name^" depends on "^fslib());
                    n.visited <- true;
                    dep_modules := n.data :: !dep_modules;
                    remaining := n.edges @ !remaining
                end;
            done;
            !dep_modules in
        reportPhase "Find dependencies";

        (* Glue all this stuff into ilxMainModule *)
        let ilxMainModule,rewriteExternalRefsToLocalRefs = staticLinkModules tcConfig mscorlibRefs ilxMainModule dep_modules in 

        (* if List.mem (fslib()) (map (fun aref -> aref.assemRefName) (refs_of_module ilxMainModule).refsAssembly) then 
          warning (Failure("Internal compiler error: the assembly produced still contains references to the F# library. Please report this as a compiler bug. The assembly produced may still be usable, or in theory you can disassemble the text of the assembly using ILDASM, edit out the references by hand and reassemble using ILASM.")); *)
        (* Change the name of anything that begins with "Microsoft.FSharp" to "FSharp" and make it private *)
    (*     let morph_name tname =           
          (* match split_type_name tname with 
            ("Microsoft" :: (("FSharp" :: rest) as ns), s) -> String.concat "." (ns @ [s])
          | _ -> *)  tname in 
    *)

        if (*F# debug && F#*) !ilfiles  then (let _ = printModule (outpath outfile "il-pre-name-fixup") ilxMainModule in ());
        if verbose then dprint_endline "Making everything under Microsoft.FSharp namespace private...";
        let ilxMainModule = 
          if not !standalone then ilxMainModule 
          else
            Ilmorph.module_tdef2tdef (fun encl td -> 
              let tn = (* morph_name  *) td.tdName in  
              let access = 
                if tn = td.tdName then td.tdAccess 
                else if encl = [] then TypeAccess_private 
                else TypeAccess_nested MemAccess_assembly in 
              {td with tdName=tn; tdAccess=access}) ilxMainModule in 
        reportPhase "Make Microsoft.FSharp private";
        
        if verbose then dprint_endline "Renaming references to types under Microsoft.FSharp namespace...";
    (*     let renameMicrosoftFSharpRefs tref = 
          if not !standalone then tref
          else
            { tref with trefName = morph_name tref.trefName; trefNested = map morph_name tref.trefNested} in
        *) 
        
         (* Print it out if requested *)
        if (*F# debug && F#*) !ilfiles then (let _ = printModule (outpath outfile "ilx.main") ilxMainModule in ());
        ilxMainModule, (Ilmorph.tref_scoref2scoref rewriteExternalRefsToLocalRefs (* >> renameMicrosoftFSharpRefs *) )
      end


(*----------------------------------------------------------------------------
!* EMIT IL
 *--------------------------------------------------------------------------*)

let emitIL (tcConfig,tcImports,mscorlibRefs) (outfile,pdbfile) ilxMainModule =
  if (*F# debug && F#*) !ilfiles then dprint_endline "Printing module...";
  if (*F# debug && F#*) !ilfiles  then printModule (outpath outfile "il.txt") ilxMainModule; 
  if !progress then dprint_endline "Writing assembly...";
  if !showTimes then Ilwrite.showTimes := true;
  Ilwrite.write_binary_ex 
    outfile
    { Ilwrite.defaults with 
         Ilwrite.mscorlib=mscorlibRefs.mscorlib_scoref;
         Ilwrite.manager=tcConfig.manager;
         Ilwrite.pdbfile=pdbfile;
         Ilwrite.signer = (match !signer with 
                           | None -> None
                           | Some(KeyFile s) -> (try Some (Ilwrite.signerOpenKeyPairFile s) with e -> dprint_endline (Printexc.to_string e); None)
                           | Some(PublicKeyFile s) -> (try Some (Ilwrite.signerOpenPublicKeyFile s) with e -> dprint_endline (Printexc.to_string e); None));
         Ilwrite.fixupOverlappingSequencePoints = true;
         Ilwrite.desiredMetadataVersionOpt = Option.map Il.parse_version !desiredCLIMetadataVersionOpt } 
    ilxMainModule


(*----------------------------------------------------------------------------
!* writeConfigFile
 *--------------------------------------------------------------------------*)
  
let writeConfigFile desiredCLIMetadataVersion outfile = 
  if !generateConfigFile & (match !target with | Exe | WinExe -> true | _ -> false) then begin
    let os = open_out (outfile^".config") in 
    let (v1,v2,v3,_) = desiredCLIMetadataVersion in
    Printf.fprintf os "\n\
<?xml version =\"1.0\"?>\n\
<configuration>\n\
    <startup>\n\
        <supportedRuntime version=\"v%d.%d.%d\" safemode=\"true\"/>\n\
        <requiredRuntime version=\"v%d.%d.%d\" safemode=\"true\"/>\n\
    </startup>\n\
</configuration>" 
      (Nums.u16_to_int v1) (Nums.u16_to_int v2) (Nums.u16_to_int v3) 
      (Nums.u16_to_int v1) (Nums.u16_to_int v2) (Nums.u16_to_int v3);
    close_out os
  end

(*----------------------------------------------------------------------------
!* writeStatsFile
 *--------------------------------------------------------------------------*)

let writeStatsFile outfile = 
  if !stats then begin
    try 
      let oc = open_out (outpath outfile "stats.txt") in
      Ilread.report oc;
      Ilxgen.report oc;
      close_out oc
    with _ -> ()
  end

let compact(inputFiles) = (*IF-OCAML*) (if length inputFiles > 5 && not !progress then (Gc.compact(); reportPhase "Compact")); (*ENDIF-OCAML*) ()    
let abortOnError () = if !errors > 0 then callExiter 1 else ()

let copyBinFile fIn fOut = 
  let is = open_in_bin fIn in 
  let os = open_out_bin fOut in
  let rec read() = 
    let bytes = Bytes.maybe_input is 4096 in 
    if Bytes.length bytes = 0 then () 
    else (Bytes.output os bytes; read()) in 
  try read(); close_out os
  with End_of_file -> close_out os

(*----------------------------------------------------------------------------
!* main - split up to make sure that both OCaml and .NET can GC the
 * dead data at the end of each phase.  We explicitly communicate arguments
 * from one phase to the next.
 *--------------------------------------------------------------------------*)
  
let main1() =
    let tcConfig = Build.newTcConfig(fsharpBinariesDir, warningOptions,false,Sys.getcwd()) in

    (* The command line compiler creates a single manager to maintaing sharing amongst AbsIL terms *)
    tcConfig.manager <- Some(Il.new_manager());

    (* process command line, flags and collect filenames *)  
    let inputFiles = 
      (* The parseArgs function calls imperative function to process "real" args *)
      (* Rather than start processing, just collect names, then process them. *)
      let inputFilesRef   = ref ([] : string list) in
      let collect name = inputFilesRef := !inputFilesRef @ [name] in (* O(n^2), but n small... *)

      try 
          parseArgs collect (specs tcConfig []) (List.tl (Array.to_list Sys.argv));
          !inputFilesRef
      with 
          e -> errorRecoveryPoint e; callExiter 1  in

      let desiredCLIMetadataVersion,fullGenericsSupported  = configureCLIVersion(true,tcConfig) in 

      (*  get dll references *)
      let dllFiles,allSourceFiles = List.partition isDll inputFiles in 
      tcConfig.referencedDLLs <- tcConfig.referencedDLLs @ (List.map(fun  f -> (rangeStartup,f)) dllFiles);

      (*  parse allSourceFiles *)
      let parseInputs =
  
            let result = 
                try  mapConcat (processInput tcConfig ["COMPILED"] n >> Option.to_list) allSourceFiles
                with e -> 
                    errorRecoveryPoint e; 
                    callExiter 1 in
            if !parseOnly then callExiter 0;
            abortOnError();
            result
  
      in
      reportPhase "Parsed parseInputs";

    (*F#
      if !printAst then (
        let show input =
          let l = any_to_layout FormatOptions.Default input in
          output_layout FormatOptions.Default stdout l
        in
        List.iter (fun input -> printf "AST:\n"; show input; printf "\n") parseInputs
      );
    F#*)      

      List.iter (processMetaCommandsFromInputAsCommandLineFlags tcConfig) parseInputs;

      (*  decideNames *)  
      let outfile,pdbfile,assemblyName = decideNames parseInputs in
      abortOnError();
        
    let tcGlobals,tcImports =  buildTcImports(tcConfig) in 
    abortOnError();

    let niceNameGen = newNiceNameGenerator() in
    let tcEnv0 = initTcEnv rangeStartup tcConfig tcImports tcGlobals in 

    (* typecheck *)
    let tcState,topAttrs,declaredImpls,tcEnvAtEnd = typeCheck rangeStartup (tcConfig,tcImports,tcGlobals,assemblyName,niceNameGen,tcEnv0) parseInputs in 
    let generatedCcu = tcState.tcsCcu in 
    abortOnError();
    reportPhase "Typechecked";
    compact(inputFiles);  

    let importMap = { Import.g = tcGlobals; Import.assemMap = findCcu(tcConfig,tcImports,tcGlobals) }  in 
    abortOnError();
      
     
    (* write interface, xmldoc, generateHtmlDocs *) 
    begin
      if !printSignature   then writeInterfaceFile (tcGlobals,tcConfig,tcImports,generatedCcu,declaredImpls);
      reportPhase ("Wrote Interface File ");
      !xmldoc |> Option.iter (writeXMLDoc (assemblyName,fullGenericsSupported,tcGlobals) generatedCcu);
      if !generateHtmlDocs then writeHTMLDoc (tcConfig,tcImports,tcGlobals) generatedCcu !htmlDocDirectory !htmldoc_append !htmlDocCssFile !htmlDocNamespaceFile;
      reportPhase ("Wrote HTML docs");
    end;
    compact(inputFiles);  

    if !typeCheckOnly then callExiter 0;
    tcConfig,tcImports,tcGlobals,generatedCcu,outfile,declaredImpls,topAttrs,inputFiles,desiredCLIMetadataVersion,pdbfile,assemblyName
  
let main2(tcConfig,tcImports,tcGlobals,generatedCcu,outfile,declaredImpls,topAttrs,inputFiles,desiredCLIMetadataVersion,pdbfile,assemblyName) = 
      
    let exportRemapping = mk_export_remapping generatedCcu (top_modul_of_ccu generatedCcu) in 
    let idata = encodeInterfaceData tcConfig tcGlobals exportRemapping generatedCcu in
    reportPhase ("Encoded Interface Data");
        
    if !progress && !Opt.jitopt_user = Some false then dprintf0 "Note, optimizations are off.\n";
    (* optimize *)
    let optEnv0 = optEnvInit tcImports in
   
    let importMap = { Import.g = tcGlobals; Import.assemMap = findCcu(tcConfig,tcImports,tcGlobals) }  in 
    let optimizedImpls,optimizationData,_ = applyAllOptimizations (tcGlobals,outfile,importMap,false) optEnv0  (generatedCcu,declaredImpls) in

    abortOnError();
        
    let generatedOptData = encodeOptimizationData tcGlobals outfile exportRemapping (generatedCcu,optimizationData)  in
    
    let ilxGenEnv = ilxgenEnvInit tcConfig tcImports tcGlobals generatedCcu in 
    let codegenResults = generateILX (false,false) tcGlobals importMap  topAttrs optimizedImpls generatedCcu (name_of_ccu generatedCcu) ilxGenEnv in
    reportPhase "TAST -> ILX";
    compact(inputFiles);  
        
    let mainmodx = createMainModule  (tcConfig,tcImports,tcGlobals.ilg,desiredCLIMetadataVersion) (pdbfile,assemblyName,outfile) (idata,generatedOptData) codegenResults in
    (* Print code before bailing out from the compiler due to errors *)
    (* in the backend of the compiler.  The partially-generated *)
    (* ILX code often contains useful information. *) 
    if (*F# debug && F#*) !ilfiles then printModule (outpath outfile "ilx.txt") mainmodx;

    abortOnError();
    reportPhase "Linking";
        
    let ilxMainModule = eraseILX (tcGlobals.ilg,tcConfig.manager,mainmodx) in
    abortOnError();
    reportPhase "ILX -> IL";
    compact(inputFiles);  
    tcConfig,tcImports,tcGlobals,ilxMainModule,outfile,inputFiles,pdbfile,desiredCLIMetadataVersion
  

let main3(tcConfig,tcImports,tcGlobals,ilxMainModule,outfile,inputFiles,pdbfile,desiredCLIMetadataVersion) = 
        
    (* We collect together the transformations to apply to the assembly to avoid multiple rewrites *)
    let ilxMainModule,trefMorph1 =  
      try staticLink (tcConfig,tcImports,tcGlobals.ilg) ilxMainModule outfile 
      with e -> errorRecoveryPoint e; callExiter 1 in
    abortOnError();
        
    let rewriteAssemblyRefsToMatchLibraries = normalizeAssemblyRefs tcConfig tcImports in
    let ilxMainModule = Ilmorph.module_tref2tref_memoized (trefMorph1 >> Ilmorph.tref_scoref2scoref rewriteAssemblyRefsToMatchLibraries) ilxMainModule in
    abortOnError();    
    reportPhase "Collected Rewrites Applied";
    compact(inputFiles);  
    tcConfig,tcImports,tcGlobals,ilxMainModule,outfile,pdbfile,desiredCLIMetadataVersion

let main4(tcConfig,tcImports,tcGlobals,ilxMainModule,outfile,pdbfile,desiredCLIMetadataVersion) = 
    emitIL (tcConfig,tcImports,tcGlobals.ilg) (outfile,pdbfile) ilxMainModule; 
    reportPhase "IL written";

    writeConfigFile desiredCLIMetadataVersion outfile;
    writeStatsFile  outfile;
     
    let copyOne inF = 
      let inF = Filename.concat (Filename.dirname inF) (Filename.basename inF) in 
      let outF = Filename.concat (Filename.dirname outfile) (Filename.basename inF) in 
      if !progress && String.lowercase outF <> String.lowercase inF then dprintf2 "Copying '%s' to '%s'\n" inF outF;
      copyBinFile inF outF in 
    tcConfig.copyLocalFiles |> List.iter (fun (m,nm) -> 
      let inF = resolveLibFile tcConfig m nm in 
      copyOne inF;
      ignoreAllFailures (fun () -> 
        if Filename.check_suffix (String.lowercase inF) ".dll" &&
           fileExists (Filename.chop_extension inF ^ ".pdb") then 
           copyOne (Filename.chop_extension inF ^ ".pdb")
      ));
    
    abortOnError();
    reportPhase "Done"


let main() = main1() |> main2 |> main3 |> main4
