//-----------------------------------------------------------------------------
// A script utility for using DirectX with F# Interactive (fsi.exe)
//
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//-----------------------------------------------------------------------------

#light
#I "c:/Program Files/Reference Assemblies/Microsoft/Framework/v3.0"
#I "C:/WINDOWS/Microsoft.NET/Framework/v3.0/WPF/"
#r "presentationcore.dll"
#r "presentationframework.dll"
#r "WindowsBase.dll"

#use @"load-wpf.fsx"

open System
open System.Threading  
open System.Windows
open System.Windows.Controls
open System.Windows.Markup
open System.Windows.Media
open System.Windows.Media.Media3D
open System.Windows.Media.Imaging
open System.Windows.Input
open System.Xml

//----------------------------------------------------------------------------
// vectors

let point (v:Vector3D) = Point3D(v.X,v.Y,v.Z)

//----------------------------------------------------------------------------
// MODEL: Compute a model that represents a grid of images from an array of URLs

let squareMesh centre (transform:Vector3D -> Vector3D) = 
    let pc (x,y) = let v = Vector3D (x,y,0.0)
                   let v = centre + transform v
                   point v
    let tc (x,y) = Point(x,1.0 - y) // flip Y coords
    let xys = [ 0.0,0.0; 0.0,1.0; 1.0,0.0; 1.0,1.0 ]

    MeshGeometry3D(TriangleIndices    = Int32Collection [ 0;1;2;1;2;3 ],
                   Positions          = Point3DCollection (xys.Map(pc)),
                   TextureCoordinates = PointCollection (xys.Map(tc)))

let imageMeshFromUrl url (w:float) center = 
    let mesh     = squareMesh center (fun v -> v * w)
    let material = 
        let imBrush    = 
            let fallbackBrush = Brushes.DarkBlue :> Brush
            if url = "" then fallbackBrush else
            try
                let imBrush = ImageBrush(BitmapImage(Uri(url,UriKind.RelativeOrAbsolute)))
                (imBrush :> Brush)
            with
                e -> fallbackBrush 
        EmissiveMaterial(imBrush) 
    GeometryModel3D(mesh,material, BackMaterial = material) :> Model3D

let nx = 20  
let ny = 20

let computeModel(urls: string[]) = 
    let images = 
        if urls.Length = 0 then [] else
        [ for x in 0 .. 20-1  do
            for y in 0 .. 20-1 do
              let fx = float x * 0.05
              let fy = float y * 0.05
              let url = urls.[(x*nx+y) % urls.Length]
              yield imageMeshFromUrl url 0.04 (Vector3D(fx,fy,0.0)) ]
    let lights = AmbientLight(Colors.White) :> Model3D
    Model3DGroup(Children = Model3DCollection (lights :: images ))

//----------------------------------------------------------------------------
// MODEL: Orthographic camera 

let povZ = 10.0
let camera = 
    OrthographicCamera(position=Point3D (0.5,0.5,povZ),
                       lookDirection=Vector3D(0.0,0.0,-1.0),
                       upDirection=Vector3D(0.0,1.0,0.0),
                       width=1.0)
//----------------------------------------------------------------------------
// MODEL: viewport

let viewport = Viewport3D(ClipToBounds=true, Camera = camera)
let modelVisual = ModelVisual3D(Content = computeModel [| |])
viewport.Children.Add(modelVisual)

//----------------------------------------------------------------------------
// CONTROLLER: menus

let mainMenu = Menu()
let fileMenu = MenuItem(Header = "_File")
mainMenu.Items.Add(fileMenu)

let loadMenu = MenuItem(Header = "_Open...")
let exitMenu = MenuItem(Header = "E_xit")
fileMenu.Items.Add(loadMenu)
fileMenu.Items.Add(exitMenu)

let panel  = DockPanel(LastChildFill = true)
panel.Children.Add(mainMenu)
panel.Children.Add(viewport)
DockPanel.SetDock(mainMenu,Dock.Top)

//----------------------------------------------------------------------------
// CONTROLLER: WPF window...

let window = Window(Width = 300.0, Height = 300.0, Topmost = true , 
                    Background = Brushes.Black, 
                    Title = "F# Interactive WPF Test Sample",
                    Visibility = Visibility.Visible,
                    Content = panel)

//----------------------------------------------------------------------------
// CONTROLLER
//
// Handlers for mouse events

window.MouseMove 
   |> IEvent.map (fun args -> args.LeftButton, args.GetPosition(relativeTo=viewport)) 
   |> IEvent.pairwise
   |> IEvent.listen (fun ((a,ap),(b,bp)) -> 
        if a = MouseButtonState.Pressed && b = MouseButtonState.Pressed then 
            let dx = ap.X - bp.X 
            let dy = ap.Y - bp.Y 
            let sf = camera.Width / window.Width
            camera.Position <- camera.Position + Vector3D(dx,-dy,0.0) * sf)


let adjustZoom (p:Point3D) k =
    // Zoom k focused at p 
    let c  = Vector3D(camera.Position.X,camera.Position.Y,povZ)
    let p  = Vector3D(p.X,p.Y,povZ)
    let c  = p * (1.0 - 1.0/k) + c * 1.0/k // = p + 1/k . (c - p) = OP + 1/k.PC 
    camera.Position <- point c
    camera.Width    <- camera.Width * k

let computeWorld (p:Point) =
    // Find world Point3D for viewport Point p 
    let ww,wh = viewport.ActualWidth,viewport.ActualHeight
    let cameraHeight = (camera.Width / ww) * wh
    let x = (0.5 - p.X / ww) * camera.Width + camera.Position.X
    let y = (p.Y / wh - 0.5) * cameraHeight + camera.Position.Y
    point (Vector3D(x,y,0.0))

let zoomFactor = 0.9 

window.MouseWheel.Add(fun args -> let k = if args.Delta>0 then zoomFactor else 1.0/zoomFactor
                                  let p = computeWorld (args.GetPosition(relativeTo=viewport))
                                  adjustZoom p k)

//----------------------------------------------------------------------------
// CONTROLLER: Dialog for Image file selection, update model with slected image list

let refreshModel(urls) = 
    modelVisual.Content <- computeModel(urls)

open Microsoft.Win32      

do loadMenu.Click.Add(fun _ -> 
    let loadD = OpenFileDialog(Multiselect=true,
                               Title="Select multiple image files",
                               InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.MyPictures))
    let res = loadD.ShowDialog()
    if res.HasValue && res.Value then refreshModel loadD.FileNames)

//----------------------------------------------------------------------------
// CONTROLLER: exit

do exitMenu.Click.Add(fun _ -> window.Close())

#if COMPILED
[<STAThread()>] // A Single Threaded Appartment model
do ()           // is required (not default for exe).
#endif

