
//-----------------------------------------------------------------------------
// A sample script for using DirectX with F# Interactive (fsi.exe)
//
// Copyright (c) Microsoft Corporation 2005-2006.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//-----------------------------------------------------------------------------

//Set the current directory, e.g. :
#light

printf "\n\n** Remember to fix up path at top of file   **\n\n"  
System.Environment.CurrentDirectory <-  @"c:\fsharp\src\samples\fsharp\DirectX\Tutorial6"

// Some various paths for different feasible versions of Managed DirectX 
#I @"C:\WINDOWS\Microsoft.NET\Managed DirectX\v9.02.2904" 
#I @"C:\WINDOWS\Microsoft.NET\DirectX for Managed Code\1.0.2902.0"
#I @"C:\WINDOWS\Microsoft.NET\DirectX for Managed Code\1.0.2905.0"
#I @"C:\WINDOWS\Microsoft.NET\Managed DirectX\v9.05.132" 
#r @"Microsoft.DirectX.dll"
#r @"Microsoft.DirectX.Direct3D.dll" 
#r @"Microsoft.DirectX.Direct3Dx.dll" 
#nowarn "40"  
#nowarn "47"  


open Compatibility
open System
open System.ComponentModel
open System.Drawing  // for Colors
open System.Windows.Forms // for Form
open Microsoft.DirectX  // for Matrix
open Microsoft.DirectX.Direct3D // For Device, Cull etc.
open Microsoft.FSharp.Text.Printf
open System.Runtime.InteropServices

//-----------------------------------------------------------------------------
// Float32 arithmetic is standard
//-----------------------------------------------------------------------------

let PI = Float32.of_float(Math.PI)

//-----------------------------------------------------------------------------
// Compute Float32 time values
//-----------------------------------------------------------------------------

let time : unit -> float32 = 
     let startTime = Environment.TickCount 
     (fun () -> Float32.of_int32(Environment.TickCount - startTime))
   
   
//-----------------------------------------------------------------------------
// SceneItems
//-----------------------------------------------------------------------------

/// SceneItems are simply imperative rendering actions for a given form and device.
/// They can partially evaluate some resources when the Device is given.
/// They can partially evaluate some resources when the Form and/or Device is given.
type SceneItem         = SceneItem of (Control -> Device -> unit -> unit)
type SceneItemOnForm   =                         (Device -> unit -> unit)
type SceneItemOnDevice =                                   (unit -> unit)

// A SimpleSceneItem doesn't depend on the Control and doesn't partially evaluate any resources
let SimpleSceneItem (f : Device -> unit) = 
  SceneItem(fun form device () -> f device)

//-----------------------------------------------------------------------------
// A two-state switcher class
//-----------------------------------------------------------------------------

type StartStop = 
  class 
    val mutable a : (unit -> unit) option
    val mutable b : (unit -> unit) option
    new (on,off) as x = 
      let rec turnon()  =
        printf "turnon()\n"; flush stdout;
        on();  x.a <- None; x.b <- Some(turnoff) 
      and     turnoff() =
        printf "turnoff()\n"; flush stdout;
        off(); x.b <- None; x.a <- Some(turnon) 
      { a = Some(turnon);
        b = None }
    member x.Start() = Idioms.lock x (fun () -> match x.a with None -> () | Some f -> f ())
    member x.Stop() = Idioms.lock x (fun () -> match x.b with None -> () | Some f -> f ())
    member x.Started = (x.a = None)
  end


//-----------------------------------------------------------------------------
// Standard DirectX event loops: a blocking loop (not used in 
// this sample)
//-----------------------------------------------------------------------------

let RenderInLoop (form:Form) (render: unit-> unit) = 
  while form.Created do
    render();
    Application.DoEvents();
  done

//-----------------------------------------------------------------------------
// Standard DirectX event loops
//-----------------------------------------------------------------------------

[<DllImport("User32.dll")>]
let PeekMessage( (msg : nativeint), (hWnd : nativeint), (messageFilterMin :uint32), (messageFilterMax:uint32), (flags:uint32)) : bool = false

let message = Bytearray.create 200 
let messageGCHandle = GCHandle.Alloc(message,GCHandleType.Pinned)
let messageAddr = messageGCHandle.AddrOfPinnedObject()
let stillIdle() = 
//  Printf.printf "still idle?\n";
  let res = not (PeekMessage(messageAddr,IntPtr.Zero,0ul,0ul,0ul)) 
  //if not res then Printf.printf "no longer idle!\n";
  res


let RenderOnIdle (form: Form) (device:Device) render = 
  let visible = ref form.Visible 
  let rec drawer   = new EventHandler(fun _ _ -> render()) 
  and     runLoopCheck =
      new EventHandler(fun _ _ -> 
        let r = 
            not form.IsDisposed &&
            device.CheckCooperativeLevel() &&
            (form.WindowState <> FormWindowState.Minimized) &&
            form.Visible &&
            form.Created  
        printf "form.WindowState = %s\n" (form.WindowState.ToString()); flush stdout;
        printf "form.Visible = %s\n" (form.Visible.ToString()); flush stdout;
        printf "form.Created = %s\n" (form.Created.ToString()); flush stdout;
        printf "form.IsDisposed = %s\n" (form.IsDisposed.ToString()); flush stdout;
        printf "device.CheckCooperativeLevel() = %s\n" ((device.CheckCooperativeLevel()).ToString()); flush stdout;
        printf "CheckForMinimized,visible = %b\n" r; flush stdout;
        if r then startstop.Start() else startstop.Stop()) 
      
  and     startstop  : StartStop = 
    device.DeviceLost.AddHandler(runLoopCheck);
    form.Resize.AddHandler(runLoopCheck);
    form.Activated.AddHandler(runLoopCheck);
    form.Closed.AddHandler(runLoopCheck);
    form.VisibleChanged.AddHandler(runLoopCheck);
    new StartStop
      ((fun () -> 
         printf "start! hash(form) = %d\n" (hash(form)); flush stdout;
         if form.IsDisposed then invalid_arg "could not start: form was disposed";
         Application.add_Idle(drawer);),
       (fun () -> 
         printf "stop! hash(form) = %d\n" (hash(form)); flush stdout;
         Application.remove_Idle(drawer))) 
  
  startstop


/// Render one window in a loop (not used in this sample)
let ExclusiveRenderOnIdle (form: Form) (device:Device) render reset = 
  RenderOnIdle form device (fun () -> 
      while (stillIdle() && form.Created && form.Visible) do 
          if device.CheckCooperativeLevel() then 
               printf "."; flush stdout; 
               form.Invalidate()
          else  
              let r = ref 0 in 
              if not (device.CheckCooperativeLevel(r)) then 
                  if !r = Enum.to_int ResultCode.DeviceNotReset then
                      reset()
      done) 


/// Render one window by sending events back through the message queue
let FairRenderOnIdle (form: Form) (device:Device) render reset = 
  RenderOnIdle form device (fun () -> 
      if form.Created && form.Visible && not form.IsDisposed then 
          if device.CheckCooperativeLevel() then 
               //printf "."; flush stdout; 
               form.Invalidate()
          else  
              let r = ref 0  
              if not (device.CheckCooperativeLevel(r)) then 
                  if !r = Enum.to_int ResultCode.DeviceNotReset then
                      reset()
  )
        


//-----------------------------------------------------------------------------
/// Objects of the Canvas3DForm provide a simple-to-use visualization wrapper
/// for a form for display scenes via DirectX.
//-----------------------------------------------------------------------------

type Canvas3DForm = 
  class 
    inherit Form;

    // State.  Some fields are lazy because creation of the item refers to the
    // object being created.
    val device: Device lazy;
    val presentParams : PresentParameters;
    val mutable renderer : StartStop lazy;  
    val mutable items : SceneItemOnForm list;
    val mutable itemsD : SceneItemOnDevice list;

    member x.Device = Lazy.force x.device
    member x.Renderer = Lazy.force x.renderer
    member x.Form = (x :> Form)
    
    new() as x = 
      let presentParams = new PresentParameters() in
      presentParams.Windowed <- true;
      presentParams.SwapEffect <- SwapEffect.Discard;
      presentParams.EnableAutoDepthStencil <- true; // Turn on a Depth stencil
      presentParams.AutoDepthStencilFormat <- DepthFormat.D16; // And the stencil format
      // Store the default adapter
      let adapterOrdinal = Manager.Adapters.Default.Adapter in 
      // Check to see if we can use a pure hardware device
      let caps = Manager.GetDeviceCaps(adapterOrdinal, DeviceType.Hardware) in 
      // Do we support hardware vertex processing?
      let vpflag = 
        if (caps.DeviceCaps.SupportsHardwareTransformAndLight) then 
          CreateFlags.HardwareVertexProcessing
        else 
          CreateFlags.SoftwareVertexProcessing in 
  
      // Do we support a pure device?
      let pflag = 
        if (caps.DeviceCaps.SupportsPureDevice) then
          [CreateFlags.PureDevice]
        else [] in 
      // Create our device
      { inherit Form();
        device = lazy (new Device(adapterOrdinal, DeviceType.Hardware, x,
                                  Enum.combine ([vpflag] @ pflag), 
                                                (* CreateFlags.DisableDriverManagement; *) 
                                                (* CreateFlags.DisableDriverManagementEx *) 
                                  Idioms.ParamArray [presentParams]));
        renderer = lazy(FairRenderOnIdle x.Form x.Device (fun () -> x.Render()) (fun () -> x.Device.Reset(Idioms.ParamArray [x.presentParams])));
        presentParams=presentParams;
        items = [];
        itemsD = [] }
      then  
  
        let OnDeviceLost () =
           printf "lost!\n";
           x.itemsD <- [] in 
        
        let OnDeviceReset () =
          printf "reset!\n";
          x.Device.RenderState.ZBufferEnable <- true;
          x.Device.RenderState.Ambient <- System.Drawing.Color.White;
          x.GetItemsForDevice() in 
        
        let OnDeviceResizing () =
           printf "device resize!\n"; in 
        
        x.SetStyle(Enum.combine [ControlStyles.AllPaintingInWmPaint ;ControlStyles.Opaque] , true);
        x.Visible <- true;
        x.add_Resize(new EventHandler(fun _ _ -> x.Form.Invalidate()));
        x.add_Paint(new PaintEventHandler(fun _ _ -> x.Render()));
        OnDeviceReset();
        x.Device.add_DeviceReset (new EventHandler(fun _ _ -> OnDeviceReset()));
        x.Device.add_DeviceLost (new EventHandler(fun _ _ -> OnDeviceLost()));
        x.Device.add_DeviceResizing (new CancelEventHandler(fun _ _ -> OnDeviceResizing()));
        x.Renderer.Start()
      
    member x.GetItemsForDevice () =
        //printf "GetItemsForDevice\n"; flush stdout;
        // Turn on the zbuffer and turn on ambient lighting 
        printf "GetItemsForDevice!\n";
        x.itemsD <- List.map (fun f -> f(x.Device)) x.items
      
    member x.Scene 
       with set(v) = 
          printf "new scene!\n"; 
          x.items <- v |> List.map (fun (SceneItem v) -> v (x.Form:> Control)); 
          x.Form.Invalidate();
          x.GetItemsForDevice()
             
    member x.Render() = 
      if x.Device.CheckCooperativeLevel() then 
        //printf "R%d" (List.length x.itemsD); flush stdout;
        x.Device.BeginScene();
        x.itemsD |> List.iter (fun f -> f());
        x.Device.EndScene();
        try x.Device.Present()
        with :? DeviceLostException -> ()

    /// Setting this property means that the control is updated under the assumption 
    /// that it is the only control that needs updating during idle cycles.  It gives 
    /// a better frame throughput since a busy loop is used.  The property is off by default.
    member x.RenderExclusive 
      with set(v) =
          x.Renderer.Stop();
          x.renderer <- lazy(if v then ExclusiveRenderOnIdle else FairRenderOnIdle) x.Form x.Device (fun () -> x.Render()) (fun () -> x.Device.Reset(Idioms.ParamArray [x.presentParams]));
          x.Renderer.Start()

  end



//-----------------------------------------------------------------------------
// Events as abstract entities free from any particular control
//-----------------------------------------------------------------------------

type 'a Event = E of (Control -> Idioms.IHandlerEvent<'a>)

let MkEvent mk add remove = E (fun control ->  
   let mp = ref [] in
   { new Idioms.IHandlerEvent<_> 
     with AddHandler(h) = 
        let h' = mk (fun _ a -> h.Invoke(control,a)) in 
        mp := (h,h'):: !mp;
        add control h'
     and  RemoveHandler(h) = 
        if List.mem_assq h !mp then 
          let h' = List.assq h !mp in 
          mp := List.remove_assq h !mp;
          remove control h'
        else ()
     and  Add(f) = add control (mk(fun _ args -> f args)) })

let Paint      = MkEvent (fun h -> new PaintEventHandler(h)) (fun c h -> c.add_Paint(h)) (fun c h -> c.remove_Paint(h))
let MouseDown  = MkEvent (fun h -> new MouseEventHandler(h)) (fun c h -> c.add_MouseDown(h)) (fun c h -> c.remove_MouseDown(h))
let MouseUp    = MkEvent (fun h -> new MouseEventHandler(h)) (fun c h -> c.add_MouseUp(h)) (fun c h -> c.remove_MouseUp(h))
let MouseMove  = MkEvent (fun h -> new MouseEventHandler(h)) (fun c h -> c.add_MouseMove(h)) (fun c h -> c.remove_MouseMove(h))
let MouseLeave = MkEvent (fun h -> new EventHandler(h)) (fun c h -> c.add_MouseLeave(h)) (fun c h -> c.remove_MouseLeave(h))
let KeyUp      = MkEvent (fun h -> new KeyEventHandler(h)) (fun c h -> c.add_KeyUp(h)) (fun c h -> c.remove_KeyUp(h))
let KeyDown    = MkEvent (fun h -> new KeyEventHandler(h)) (fun c h -> c.add_KeyDown(h)) (fun c h -> c.remove_KeyDown(h))
let KeyPress   = MkEvent (fun h -> new KeyPressEventHandler(h)) (fun c h -> c.add_KeyPress(h)) (fun c h -> c.remove_KeyPress(h))


// Here is a non-standard event
let MouseTrack = E (fun form ->
  let mouseDown = ref false in 
  let listeners = new Idioms.EventListeners<_>() in 
  form.add_MouseDown
      (new MouseEventHandler(fun _ _ -> 
          mouseDown := true));
  form.add_MouseUp
      (new MouseEventHandler(fun _ ev -> 
        mouseDown := false;
        if !mouseDown then 
          listeners.Fire(ev)));
  form.add_MouseLeave
      (new EventHandler(fun _ _ -> 
        mouseDown := false));
  form.add_MouseMove
      (new MouseEventHandler(fun _ ev -> 
        if !mouseDown then
          listeners.Fire(ev)));
  listeners.Event)

 
//-----------------------------------------------------------------------------
// Scene items that depend on events
//-----------------------------------------------------------------------------

/// EventE holds state that records the result of applying
/// a computation to the sequence of values generated by the given event.  
/// The event is related to the same control as the generated SceneItem.
let EventE (E e) (initial : 'b) (transform: 'b -> 'a -> 'b) (mk: 'b -> SceneItem) =
  SceneItem(fun control -> 
    let e = e control in
    let latest = ref initial in 
    let invalid = ref false in 
    let (SceneItem latestItemOn) = mk !latest in 
    e.Add(fun x -> 
      latest := transform !latest x;
      invalid := true);
    (fun device -> 
      let (SceneItem latestItem) = mk !latest in 
      let latestItemOnForm = ref (latestItem control device) in
      let makeValid() = 
        let (SceneItem latestItem) = mk !latest in 
        latestItemOnForm := latestItem control device;
        invalid := false in 
      (fun () -> if !invalid then makeValid(); !latestItemOnForm ())))

   
//-----------------------------------------------------------------------------
// A little library of standard rendering functions
//-----------------------------------------------------------------------------

// These ones may depend on the current time/world etc.
let WorldTransformT worldf                   = SimpleSceneItem (fun device -> device.Transform.World <- worldf())
let ClearT          (colorf : unit -> Color) = SimpleSceneItem (fun device -> device.Clear(Enum.combine[ClearFlags.ZBuffer; ClearFlags.Target], colorf (), 1.0f, 0))
let ViewT           viewf                    = SimpleSceneItem (fun device -> device.Transform.View <- viewf())
let ProjectionT     projf                    = SimpleSceneItem (fun device -> device.Transform.Projection <- projf ())

// These ones are invariant
let WorldTransform world = WorldTransformT (fun () -> world)
let Clear          color = ClearT          (fun () -> color)
let View           view  = ViewT           (fun () -> view)
let Projection     proj  = ProjectionT     (fun () -> proj)

// These ones depend on a cached computation over a stream of events
let WorldTransformE e x f                    = EventE e x f WorldTransform
let ClearE          e x f                    = EventE e x f Clear
let ViewE           e x f                    = EventE e x f View
let ProjectionE     e x f                    = EventE e x f Projection


let CullNone = SimpleSceneItem (fun d -> d.RenderState.CullMode <- Cull.None)
let LightingOff = SimpleSceneItem (fun d -> d.RenderState.Lighting <- false)
  
//-----------------------------------------------------------------------------
// A SceneItem for triangles with colored points
//-----------------------------------------------------------------------------

let typeof_PositionColored = (type CustomVertex.PositionColored)

let ColoredPointTrianglesT(trianglesf) =
  SimpleSceneItem(fun device ->         
      let triangles = trianglesf() in 
      let n = Array.length triangles  in 
      if n = 0 then () else 
      let vertexBuffer = 
        new VertexBuffer(typeof_PositionColored,
                         3 * n, device, Usage.None, CustomVertex.PositionColored.Format, 
                         Pool.Managed) in
      let verts = 
        CompatArray.init
          (n * 3)
          (fun i -> 
            let p1,p2,p3 = triangles.(i / 3) in 
            let (a,b,c,(d:Color)) = match i % 3 with | 0 -> p1 | 1 -> p2 | _ -> p3 in
            new CustomVertex.PositionColored(a,b,c,d.ToArgb()))  in 
      let FillVertexBuffer() =
        let stm = vertexBuffer.Lock(0, 0, LockFlags.None) in
        stm.Write(verts); 
        vertexBuffer.Unlock() in 
      vertexBuffer.add_Created(new EventHandler(fun _ _ -> FillVertexBuffer()));
      FillVertexBuffer();
      device.SetStreamSource( 0, vertexBuffer, 0);
      device.VertexFormat <- CustomVertex.PositionColored.Format;
      device.DrawPrimitives(PrimitiveType.TriangleList, 0, n))

let ColoredPointTriangles(triangles) = ColoredPointTrianglesT (fun () -> triangles)
let ColoredPointTrianglesE e x f     = EventE e x f ColoredPointTriangles

//-----------------------------------------------------------------------------
// A SceneItem for mesh data read from a file
//-----------------------------------------------------------------------------

type MeshData= 
  { mesh : Mesh; // Our mesh object in sysmem
    meshMaterials : Material[]; // Materials for our mesh
    meshTextures : Texture[];  } // Textures for our mesh

let LoadMeshFromFile((device : Device), file) =         
   // Load the mesh from the specified file.  'materials' is an
   // 'out' parameter: for these F# currently requires that you pass
   // in a value of type 'ref'.
   let materialsRes = ref (null : ExtendedMaterial[]) in
   let mesh = Mesh.FromFile(file, MeshFlags.SystemMemory,device, materialsRes) in
   let meshExtendedMaterials = !materialsRes in

   // Set the ambient color for each material (D3DX does not do 
   // this).  F# uses copy-out/copy-in for this mutation of a .net struct
   let meshMaterials = 
     CompatArray.map 
       (fun (m : ExtendedMaterial) -> 
          let mutable m3D = m.Material3D in
          m3D.Ambient <- m3D.Diffuse;
          m3D) 
       meshExtendedMaterials in 
     
   let meshTextures =
     CompatArray.map
       (fun (m : ExtendedMaterial) -> TextureLoader.FromFile(device, m.TextureFilename)) 
       meshExtendedMaterials in
   { mesh = mesh;
     meshMaterials = meshMaterials;
     meshTextures = meshTextures }


  // The Render function for mesh data
let MeshFromFile(file) =
  SceneItem(fun form device -> 
    let meshData = LoadMeshFromFile((device : Device), file) in 
    (fun () ->
      // Meshes are divided into subsets, one for each material. Render them in
      // a loop
      for i = 0 to meshData.meshMaterials.Length - 1 do
         // Set the material and texture for this subset
         device.Material <- CompatArray.get meshData.meshMaterials i;
         device.SetTexture(0, CompatArray.get meshData.meshTextures i);
            
         // Draw the mesh subset
         meshData.mesh.DrawSubset(i)
      done))
    

    
//-----------------------------------------------------------------------------
// A sample
//-----------------------------------------------------------------------------


let inputData1 = 
   [|  ((-1.0f,  -1.0f, 0.0f, Color.DarkGoldenrod),
        (1.0f,  -1.0f, 0.0f, Color.MediumOrchid),
        (0.0f,   1.0f, 0.0f, Color.Cornsilk)); |]

let sin x = Float32.of_float(Math.Sin(Float32.to_float x))
let cos x = Float32.of_float(Math.Cos(Float32.to_float x))
let rgen = new Random()
let rand (l,h) = Float32.of_float(rgen.NextDouble()) * (h-l) + l
let randi h = truncate(rgen.NextDouble() * float(h)) 

let inputData2 () = 
  let t = time() in 
  [|  ((-1.0f,  -1.0f, 2.0f * sin(t/650.f), Color.Green),
        (1.0f,  -1.0f, 2.0f * cos(t/550.f), Color.Blue),
        (0.0f,   1.0f, 2.0f * sin(t/450.f), Color.Red)) |]


let canvas = new Canvas3DForm()
canvas.Text <- "F# Direct3D Tutorial 3 - Matrices"
canvas.Size <- new Size(400, 300)


let computeView(x,y) = 
  let initialView = 
    Matrix.LookAtLH( new Vector3( 0.0f, 3.0f, -5.0f ),
                     new Vector3( 0.0f, 0.0f, 0.0f ), 
                     new Vector3( 0.0f, 1.0f, 0.0f ) ) in
  let transform = Matrix.RotationYawPitchRoll(float32(x-200)     / 200.0f,
                                              float32(x-200)     / 200.0f,
                                              float32(x+y-400) / 200.0f) in
  transform * initialView                                         

let scene =
  [ // Turn off culling, so we see the front and back of the triangle
    CullNone;
    
    // Turn off D3D lighting, since we are providing our own vertex colors
    LightingOff;

    // Clear to give a nice background
    Clear(Color.Blue);

    // The world transform makes the scene spin
    WorldTransformT(fun () -> Matrix.RotationY( time()  / 350.0f ));
    
    // The view depends on the MouseTrack event.
    ViewE MouseTrack (computeView(200,200)) (fun _ (x:MouseEventArgs) -> computeView(x.X,x.Y));

    // This projection puts things in perspective
    Projection(Matrix.PerspectiveFovLH(PI / 4.0f, 1.0f, 1.0f, 100.0f ));

    // This one is the star of the show
    MeshFromFile(@"tiger.x");

    // These skewer the poor fellow with some triangles
    ColoredPointTriangles(inputData1);
    ColoredPointTrianglesT(inputData2) ]

canvas.Scene <- scene

#if COMPILED
Application.Run()
#endif

printf "\n\n** You may need to type  at the prompt... **\n"
