/*
 * Decompiled with CFR 0.152.
 */
package org.flywaydb.core.internal.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import javax.sql.DataSource;
import org.flywaydb.core.api.FlywayException;
import org.flywaydb.core.api.logging.Log;
import org.flywaydb.core.api.logging.LogFactory;
import org.flywaydb.core.internal.exception.FlywaySqlException;
import org.flywaydb.core.internal.jdbc.DatabaseType;
import org.flywaydb.core.internal.jdbc.JdbcTemplate;
import org.flywaydb.core.internal.jdbc.JdbcUtils;
import org.flywaydb.core.internal.jdbc.pro.DryRunJdbcProxies;
import org.flywaydb.core.internal.jdbc.pro.DryRunStatementInterceptor;
import org.flywaydb.core.internal.util.ExceptionUtils;

public class JdbcConnectionFactory {
    private static final Log LOG = LogFactory.getLog(JdbcConnectionFactory.class);
    private final DataSource dataSource;
    private final int connectRetries;
    private final DatabaseType databaseType;
    private final String jdbcUrl;
    private final String driverInfo;
    private final String productName;
    private Connection firstConnection;
    private ConnectionInitializer connectionInitializer;
    private final DryRunStatementInterceptor dryRunStatementInterceptor;
    private final boolean supportsBatch;
    private String oracleUser;
    private String oracleConnectIdentifier;
    private String oracleOVersion;
    private String oracleORelease;

    public JdbcConnectionFactory(DataSource dataSource, int connectRetries, DryRunStatementInterceptor dryRunStatementInterceptor) {
        this.dataSource = dataSource;
        this.connectRetries = connectRetries;
        this.firstConnection = JdbcUtils.openConnection(dataSource, connectRetries);
        this.databaseType = DatabaseType.fromJdbcConnection(this.firstConnection);
        DatabaseMetaData databaseMetaData = JdbcUtils.getDatabaseMetaData(this.firstConnection);
        this.jdbcUrl = JdbcConnectionFactory.getJdbcUrl(databaseMetaData);
        this.driverInfo = JdbcConnectionFactory.getDriverInfo(databaseMetaData);
        this.productName = JdbcUtils.getDatabaseProductName(databaseMetaData);
        this.supportsBatch = this.determineBatchSupport(databaseMetaData);
        this.dryRunStatementInterceptor = dryRunStatementInterceptor;
        this.collectExtraInfo(this.databaseType, this.firstConnection, databaseMetaData);
    }

    public void setConnectionInitializer(ConnectionInitializer connectionInitializer) {
        this.connectionInitializer = connectionInitializer;
    }

    private boolean determineBatchSupport(DatabaseMetaData databaseMetaData) {
        try {
            return databaseMetaData.supportsBatchUpdates();
        }
        catch (SQLException e) {
            LOG.debug("Unable to check whether batch updates are supported:\n" + ExceptionUtils.toMessage(e));
            return false;
        }
    }

    private void collectExtraInfo(DatabaseType databaseType, Connection connection, DatabaseMetaData databaseMetaData) {
        try {
            if (databaseType == DatabaseType.ORACLE) {
                JdbcTemplate jdbcTemplate = new JdbcTemplate(connection, databaseType);
                this.oracleUser = jdbcTemplate.queryForString("SELECT USER FROM DUAL", new String[0]);
                String url = databaseMetaData.getURL();
                this.oracleConnectIdentifier = url == null ? "" : url.substring(url.indexOf("//") + 2);
                this.oracleOVersion = jdbcTemplate.queryForString("select banner from v$version where banner like 'Oracle Database%'", new String[0]);
                this.oracleORelease = jdbcTemplate.queryForString("SELECT VERSION FROM PRODUCT_COMPONENT_VERSION WHERE PRODUCT LIKE 'Oracle Database%'", new String[0]).replace(".", "0");
            }
        }
        catch (SQLException e) {
            throw new FlywaySqlException("Unable to collect extra info", e);
        }
    }

    public String getOracleUser() {
        return this.oracleUser;
    }

    public String getOracleConnectIdentifier() {
        return this.oracleConnectIdentifier;
    }

    public String getOracleOVersion() {
        return this.oracleOVersion;
    }

    public String getOracleORelease() {
        return this.oracleORelease;
    }

    public boolean isSupportsBatch() {
        return this.supportsBatch;
    }

    public DatabaseType getDatabaseType() {
        return this.databaseType;
    }

    public String getJdbcUrl() {
        return this.jdbcUrl;
    }

    public String getDriverInfo() {
        return this.driverInfo;
    }

    public String getProductName() {
        return this.productName;
    }

    public Connection openConnection() throws FlywayException {
        Connection connection = this.firstConnection == null ? JdbcUtils.openConnection(this.dataSource, this.connectRetries) : this.firstConnection;
        this.firstConnection = null;
        if (this.connectionInitializer != null) {
            this.connectionInitializer.initialize(this, connection);
        }
        if (this.dryRunStatementInterceptor != null) {
            if (this.databaseType.supportsReadOnlyTransactions()) {
                try {
                    connection.setReadOnly(true);
                }
                catch (SQLException e) {
                    throw new FlywaySqlException("Unable to switch connection to read-only", e);
                }
            }
            return DryRunJdbcProxies.createConnectionProxy(connection, this.dryRunStatementInterceptor);
        }
        return connection;
    }

    private static String getJdbcUrl(DatabaseMetaData databaseMetaData) {
        String url;
        try {
            url = databaseMetaData.getURL();
        }
        catch (SQLException e) {
            throw new FlywaySqlException("Unable to retrieve the JDBC connection URL!", e);
        }
        if (url == null) {
            return "";
        }
        return JdbcConnectionFactory.filterUrl(url);
    }

    static String filterUrl(String url) {
        int questionMark = url.indexOf("?");
        if (questionMark >= 0 && !url.contains("?databaseName=")) {
            url = url.substring(0, questionMark);
        }
        url = url.replaceAll("://.*:.*@", "://");
        return url;
    }

    private static String getDriverInfo(DatabaseMetaData databaseMetaData) {
        try {
            return databaseMetaData.getDriverName() + " " + databaseMetaData.getDriverVersion();
        }
        catch (SQLException e) {
            throw new FlywaySqlException("Unable to read database driver info: " + e.getMessage(), e);
        }
    }

    public static interface ConnectionInitializer {
        public void initialize(JdbcConnectionFactory var1, Connection var2);
    }
}

