/*
 *  Copyright (c) 2019-present, Facebook, Inc.
 *  All rights reserved.
 *
 *  This source code is licensed under the BSD-style license found in the
 *  LICENSE file in the root directory of this source tree.
 */

#include <fizz/extensions/delegatedcred/Serialization.h>
#include <folly/base64.h>
#include <folly/portability/GMock.h>
#include <folly/portability/GTest.h>

using namespace folly;

using namespace testing;

namespace fizz {
namespace extensions {
namespace test {

// @lint-ignore-every PRIVATEKEY

// clang-format off
/*
 *  Delegated credential certificate generated by kP256CredCertKey
 *  Prerequisites:
 *    - P133567922 in config.cfg
 *    - kP256CredCertKey in p256_key.pem
 *  Command: openssl req -new -key p256_key.pem -x509 -nodes -days {days} -config config.cfg
 *  Current cert set to expire in 2119.
 *  Output: Self-signed delegation certificate
 */
// clang-format on
StringPiece kP256CredCert = R"(
-----BEGIN CERTIFICATE-----
MIICKzCCAdGgAwIBAgIJAPi2vMRfOVd0MAoGCCqGSM49BAMCMGIxCzAJBgNVBAYT
AlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQgQ29t
cGFueSBMdGQxHjAcBgNVBAMMFXJldnByb3h5LWRlbGVnYXRlZC1lYzAgFw0xOTA5
MjMwMjAyMzVaGA8yMTE5MDgzMDAyMDIzNVowYjELMAkGA1UEBhMCWFgxFTATBgNV
BAcMDERlZmF1bHQgQ2l0eTEcMBoGA1UECgwTRGVmYXVsdCBDb21wYW55IEx0ZDEe
MBwGA1UEAwwVcmV2cHJveHktZGVsZWdhdGVkLWVjMFkwEwYHKoZIzj0CAQYIKoZI
zj0DAQcDQgAE7EbZMKds65EYciaSULFH4wZKt/OThiUL4uQW9cybr2HIzK68corO
JCeHXOsV3lpYS46b39SBZr1GZprFHH5gHaNuMGwwHQYDVR0OBBYEFMLkRMB4SclK
8K8uYMQBaYw0gNP7MB8GA1UdIwQYMBaAFMLkRMB4SclK8K8uYMQBaYw0gNP7MAwG
A1UdEwQFMAMBAf8wCwYDVR0PBAQDAgHmMA8GCSsGAQQBgtpLLAQCBQAwCgYIKoZI
zj0EAwIDSAAwRQIgB2EWbwWohYziQ2LmY8Qmn8y0WKR6Mbm5aad0rUBvtK4CIQCv
0U6Z/gFrVr0Cb2kc7M37KD9z5eeTwkQuGqs5GXF8Ow==
-----END CERTIFICATE-----
)";

/*
 *  Randomly generated ECDSA-openssl::P256 private key
 *  Command: openssl ecparam -name secp256r1 -genkey
 *  Output: Randomly generated ECDSA-openssl::P256 private key
 */
StringPiece kP256DelegatedCredKey = R"(
-----BEGIN EC PARAMETERS-----
BggqhkjOPQMBBw==
-----END EC PARAMETERS-----
-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIA8/keRkilh8bwUPxH9jiP5SsP4QiZtfofayTsRSI59poAoGCCqGSM49
AwEHoUQDQgAE8mV/wDAabnJbPLuF/qd/FMIWHDlrJI97cwq4obtPHyKFF2ukoG+6
/pXOUrEbsIH+/QBpZsnRHjvxryib97Ay+Q==
-----END EC PRIVATE KEY-----
)";

// clang-format off
/*
 *  Delegated credential generated using kP256CredCert, kP256CredCertKey & kP256DelegatedCredKey
 *  Prerequisites:
 *    - kP256CredCert in cert.pem
 *    - kP256CredCertKey in p256_key.pem
 *    - kP256DelegatedCredKey in p256_dc_key.pem
 *  Command: buck run //fizz/tool:fizz -- gendc -cert cert.pem -key p256_key.pem -credkey p256_dc_key.pem | xxd -p
 *  Output: Hex-encoded delegated credential
 */
// clang-format on

StringPiece kP256DelegatedCredNoLabel = {
    "CI61NAQDAABbMFkwEwYHKoZIzj0CAQYIKoZIzj0"
    "DAQcDQgAE8mV/wDAabnJbPLuF/qd/FMIWHDlrJI"
    "97cwq4obtPHyKFF2ukoG+6/pXOUrEbsIH+/QBpZ"
    "snRHjvxryib97Ay+QQDAEcwRQIgUrbmZ9uq07B0"
    "Myg06hZj0uUq2f04eI4Ci0VIRP7XGhACIQDqckz"
    "1AUqrX1gEw5DVBK8WQ3Q6pYSF4WI4gxIpvcCMEQ=="};

StringPiece kP256DelegatedCred = {
    "-----BEGIN DELEGATED CREDENTIAL-----\n"
    "CI61NAQDAABbMFkwEwYHKoZIzj0CAQYIKoZIzj0"
    "DAQcDQgAE8mV/wDAabnJbPLuF/qd/FMIWHDlrJI"
    "97cwq4obtPHyKFF2ukoG+6/pXOUrEbsIH+/QBpZ"
    "snRHjvxryib97Ay+QQDAEcwRQIgUrbmZ9uq07B0"
    "Myg06hZj0uUq2f04eI4Ci0VIRP7XGhACIQDqckz"
    "1AUqrX1gEw5DVBK8WQ3Q6pYSF4WI4gxIpvcCMEQ==\n"
    "-----END DELEGATED CREDENTIAL-----\n"};

StringPiece kP256DelegatedCredBadLabel = {
    "-----BEGIN DC-----\n"
    "CI61NAQDAABbMFkwEwYHKoZIzj0CAQYIKoZIzj0"
    "DAQcDQgAE8mV/wDAabnJbPLuF/qd/FMIWHDlrJI"
    "97cwq4obtPHyKFF2ukoG+6/pXOUrEbsIH+/QBpZ"
    "snRHjvxryib97Ay+QQDAEcwRQIgUrbmZ9uq07B0"
    "Myg06hZj0uUq2f04eI4Ci0VIRP7XGhACIQDqckz"
    "1AUqrX1gEw5DVBK8WQ3Q6pYSF4WI4gxIpvcCMEQ==\n"
    "-----END DC-----\n"};

StringPiece kP384DelegatedCred = {
    "-----BEGIN DELEGATED CREDENTIAL-----\n"
    "CKpfvQUDAAB4MHYwEAYHKoZIzj0CAQYFK4EEACID"
    "YgAEfDkvSAjmCrrxRNEkpA52igKI5pnE9q/MiKHTJ"
    "/gfmJfhwK3IGI21+zs4DHvR2iowHSa5Ul+J5Fai9o"
    "Nr32wTLUgQyrygUfhwNMuS4ucnd3Lz/EEJUBdkbEL"
    "NnfY98HJcBAMARjBEAiBmz7fIOTsAXTQG55zdTxBy"
    "QiUv7Ub3qtyYnJMaJKmx9gIgQiv40UZcP8GZkvDfg"
    "D/EqLaFsDaP4Fl2PGH9NgXkyo8=\n"
    "-----END DELEGATED CREDENTIAL-----\n"};

TEST(SerializationTest, testValidReadFromPem) {
  auto combinedPem = kP256DelegatedCred.toString() +
      kP256DelegatedCredKey.toString() + kP256CredCert.toString();
  auto dc = loadDCFromPEM(combinedPem);
  EXPECT_NE(dc, nullptr);
  EXPECT_EQ(
      dc->getSigSchemes(),
      std::vector<SignatureScheme>{SignatureScheme::ecdsa_secp256r1_sha256});
}

TEST(SerializationTest, BadLabel) {
  auto combinedPem = kP256DelegatedCredBadLabel.toString() +
      kP256DelegatedCredKey.toString() + kP256CredCert.toString();
  EXPECT_THROW(loadDCFromPEM(combinedPem), std::runtime_error);
}

TEST(SerializationTest, P384DC) {
  auto combinedPem = kP256DelegatedCredBadLabel.toString() +
      kP256DelegatedCredKey.toString() + kP256CredCert.toString();
  EXPECT_THROW(loadDCFromPEM(combinedPem), std::runtime_error);
}

TEST(SerializationTest, TestBuildCombinedPEM) {
  auto credData = folly::base64Decode(kP256DelegatedCredNoLabel.toString());
  std::vector<Extension> credVec;
  credVec.emplace_back(Extension{
      ExtensionType::delegated_credential,
      folly::IOBuf::copyBuffer(std::move(credData))});
  auto cred = getExtension<DelegatedCredential>(std::move(credVec));

  auto combinedPem = kP256DelegatedCred.toString() +
      kP256DelegatedCredKey.toString() + kP256CredCert.toString();
  EXPECT_EQ(
      combinedPem,
      generateDelegatedCredentialPEM(
          std::move(*cred),
          kP256CredCert.toString(),
          kP256DelegatedCredKey.toString()));
}
} // namespace test
} // namespace extensions
} // namespace fizz
