/*
firemime.c - FireMIME function definitions
Copyright (C) 2000 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define _FIREMIME_C

#include <ctype.h>
#include "firemake.h"
#include "firemime.h"

const char firemime_version[] = VERSION;

#define MAXDEPTH 50

static const char *errors[] = {
/* 0 */	"Success",
/* 1 */	"Not all parts processed by handler request",
/* 2 */	"Not all parts processed by handler request",
/* 3 */	"Maximum depth exceeded",
/* 4 */	"No header/body break found",
/* 5 */	"Invalid Content-Type",
/* 6 */	"Invalid parameter list",
/* 7 */	"Missing multipart boundary parameter",
/* 8 */	"Invalid MIME structure (missing multipart closing)",
/* 9 */	"Invalid MIME structure (missing linebreak on boundary line)",
/* 10 */	"Invalid multipart content encoding",
/* 11 */	"Unbalanced header comments",
/* 12 */	"Unrecognized MIME version",
/* 13 */	"Insufficient space",
/* 14 */	"Unrecognized encoding method",
/* 15 */	"Invalid quoted-printable encoding",
/* 16 */	"Invalid HTML/XML in message"
};

static void free_parameters(struct firemime_param *head, struct firestring_estr_t *headers) {
	struct firemime_param *next;
	while (head != NULL) {
		next = head->next;
		firestring_estr_free(&head->name);
		if (head->alloc)
			firestring_estr_free(&head->value);
		free(head);
		head = next;
	}
}

static int firemime_parameters(struct firemime_param **head, struct firestring_estr_t *paramlist) {
	int split, split2;
	struct firemime_param *temp;
	struct firestring_estr_t etemp;
	int start = 0;
	while (1) {
		split = firestring_estr_strchr(paramlist,'=',start);
		if (split == -1)
			return 6;

		temp = firestring_malloc(sizeof(struct firemime_param));
		temp->next = *head;
		temp->name.s = NULL;
		temp->value.s = NULL;
		temp->alloc = 0;
		*head = temp;

		etemp.s = &paramlist->s[start];
		etemp.l = etemp.a = split - start;
		firestring_estr_alloc(&temp->name,etemp.l);
		if (firemime_strip_comments(&temp->name,&etemp) != 0)
			return 11;
		firestring_estr_ip_trim(&temp->name);

		temp->value.s = &paramlist->s[split + 1];
		temp->value.l = temp->value.a = paramlist->l - split - 1;
		firestring_estr_chug(&temp->value);

		if (temp->value.s[0] == '"') {
			/* quoted string */
			temp->value.s++;
			temp->value.l--;
			temp->value.a--;
			split = firestring_estr_strchr(&temp->value,'"',0);
			if (split == -1)
				return 6;
			split2 = firestring_estr_strchr(&temp->value,';',split+1);
			temp->value.l = temp->value.a = split;
			if (split2 == -1)
				return 0;
			else
				start = temp->value.s - paramlist->s + split2 + 1;

		} else {
			/* end at end of string or ; */
			split = firestring_estr_strchr(&temp->value,';',0);
			if (split == -1) {
				temp->alloc = 1;
				firestring_estr_alloc(&etemp,temp->value.l);
				if (firemime_strip_comments(&etemp,&temp->value) != 0) {
					firestring_estr_free(&etemp);
					return 11;
				}
				memcpy(&temp->value,&etemp,sizeof(temp->value));
				firestring_estr_ip_trim(&temp->value);
				return 0;
			} else {
				temp->value.l = temp->value.a = split;
				start = temp->value.s - paramlist->s + split + 1;
				temp->alloc = 1;
				firestring_estr_alloc(&etemp,temp->value.l);
				if (firemime_strip_comments(&etemp,&temp->value) != 0) {
					firestring_estr_free(&etemp);
					return 11;
				}
				memcpy(&temp->value,&etemp,sizeof(temp->value));
				firestring_estr_ip_trim(&temp->value);
			}
		}
	}
}

/* set mime parameters to defaults as this is non-mime */
static void firemime_nonmime(struct firemime_part *stack, int depth) {
	firestring_estr_aestrcpy(&stack[depth].type,&ESTR_S("text"),0);
	firestring_estr_aestrcpy(&stack[depth].subtype,&ESTR_S("plain"),0);
	firestring_estr_aestrcpy(&stack[depth].encoding,&ESTR_S("7bit"),0);
	firestring_estr_aestrcpy(&stack[depth].disposition,&ESTR_S("inline"),0);
}

static int firemime_int_parse(struct firestring_estr_t *part, int (*handler)(struct firemime_part *, int, void *), struct firestring_estr_t *linebreak, struct firemime_part *stack, int depth, struct firestring_estr_t *doublebreak, void *opaque) {
	int split;
	int r = 0;
	struct firestring_estr_t *tempheader;

	if (depth >= MAXDEPTH)
		return 3;

	if (part->l >= linebreak->l && memcmp(part->s,linebreak->s,linebreak->l) == 0) {
		/* starts with a linebreak, it's all body */
		stack[depth].headers.s = NULL;
		stack[depth].headers.l = stack[depth].headers.a = 0;
		memcpy(&stack[depth].body,part,sizeof(stack[depth].body));
	} else {
		split = firestring_estr_estrstr(part,doublebreak,0);
		if (split == -1) {
			/* no break, it's all headers */
			memcpy(&stack[depth].headers,part,sizeof(stack[depth].headers));
			stack[depth].body.s = NULL;
			stack[depth].body.l = stack[depth].body.a = 0;
		} else {
			stack[depth].headers.s = part->s;
			stack[depth].headers.l = stack[depth].headers.a = split + linebreak->l;
			stack[depth].body.s = &part->s[split + doublebreak->l];
			stack[depth].body.l = stack[depth].body.a = part->l - split - doublebreak->l;
		}
	}

	if (depth == 0 ||
			(firestring_estr_estrcasecmp(&stack[depth-1].type,&ESTR_S("message"),0) == 0 &&
			 firestring_estr_estrcasecmp(&stack[depth-1].subtype,&ESTR_S("rfc822"),0) == 0)) {
		struct firestring_estr_t version;

		/* top level, check if this is MIME at all */
		tempheader = firemime_get_header(&stack[depth].headers,&ESTR_S("MIME-Version"),linebreak);
		if (tempheader == NULL) {
			firemime_nonmime(stack,depth);
			goto handler;
		}

		firestring_estr_alloc(&version,tempheader->l);
		if (firemime_strip_comments(&version,tempheader) != 0) {
			firestring_estr_free(&version);
			return 11;
		}
		firestring_estr_ip_trim(&version);

		if (firestring_estr_estrcmp(&version,&ESTR_S("1.0"),0) != 0) {
			firestring_estr_free(&version);
			return 12;
		}

		firestring_estr_free(&version);
	}

	tempheader = firemime_get_header(&stack[depth].headers,&ESTR_S("Content-Type"),linebreak);
	if (tempheader == NULL) {
		firestring_estr_aestrcpy(&stack[depth].type,&ESTR_S("text"),0);
		firestring_estr_aestrcpy(&stack[depth].subtype,&ESTR_S("plain"),0);
	} else {
		int i;
		struct firestring_estr_t tempheader2;

		/* remove parameters */
		i = firestring_estr_strchr(tempheader,';',0);
		if (i != -1) {
			stack[depth].type_params = NULL;
			if (firemime_parameters(&stack[depth].type_params,&(struct firestring_estr_t) { &tempheader->s[i + 1], tempheader->l - i - 1, tempheader->l - i - 1 }) != 0) {
				r = 6;
				goto cleanup;
			}
			tempheader->l = tempheader->a = i;
		}

		/* split type and subtype */
		i = firestring_estr_strchr(tempheader,'/',0);
		if (i == -1) {
			r = 5;
			goto cleanup;
		}
		tempheader2.s = tempheader->s;
		tempheader2.l = tempheader2.a = i;
		firestring_estr_alloc(&stack[depth].type,tempheader2.l);
		if (firemime_strip_comments(&stack[depth].type,&tempheader2) != 0) {
			r = 11;
			goto cleanup;
		}
		firestring_estr_ip_trim(&stack[depth].type);


		tempheader2.s = &tempheader->s[i + 1];
		tempheader2.l = tempheader2.a = tempheader->l -i - 1;
		firestring_estr_alloc(&stack[depth].subtype,tempheader2.l);
		if (firemime_strip_comments(&stack[depth].subtype,&tempheader2) != 0) {
			r = 1;;
			goto cleanup;
		}
		firestring_estr_ip_trim(&stack[depth].subtype);
	}

	tempheader = firemime_get_header(&stack[depth].headers,&ESTR_S("Content-Transfer-Encoding"),linebreak);
	if (tempheader == NULL)
		firestring_estr_aestrcpy(&stack[depth].encoding,&ESTR_S("7bit"),0);
	else {
		int i;

		/* remove parameters */
		i = firestring_estr_strchr(tempheader,';',0);
		if (i != -1)
			tempheader->l = tempheader->a = i;

		firestring_estr_alloc(&stack[depth].encoding,tempheader->l);
		if (firemime_strip_comments(&stack[depth].encoding,tempheader) != 0) {
			r = 11;
			goto cleanup;
		}
		firestring_estr_ip_trim(&stack[depth].encoding);
	}

	tempheader = firemime_get_header(&stack[depth].headers,&ESTR_S("Content-Disposition"),linebreak);
	if (tempheader == NULL)
		firestring_estr_aestrcpy(&stack[depth].disposition,&ESTR_S("inline"),0);
	else {
		int i;

		i = firestring_estr_strchr(tempheader,';',0);
		if (i != -1) {
			stack[depth].disposition_params = NULL;
			if (firemime_parameters(&stack[depth].disposition_params,&(struct firestring_estr_t) { &tempheader->s[i + 1], tempheader->l - i - 1, tempheader->l - i - 1 }) != 0) {
				r = 6;
				goto cleanup;
			}
			tempheader->l = tempheader->a = i;
		}

		firestring_estr_alloc(&stack[depth].disposition,tempheader->l);
		if (firemime_strip_comments(&stack[depth].disposition,tempheader) != 0) {
			r = 11;
			goto cleanup;
		}

		firestring_estr_ip_trim(&stack[depth].disposition);
	}

handler:
	r = handler(stack,depth,opaque);
	if (r != 0)
		goto cleanup;

	if (firestring_estr_estrcasecmp(&stack[depth].type,&ESTR_S("message"),0) == 0 &&
			firestring_estr_estrcasecmp(&stack[depth].subtype,&ESTR_S("rfc822"),0) == 0) {
		int cr;
		cr = firemime_int_parse(&stack[depth].body,handler,linebreak,stack,depth + 1,doublebreak,opaque);
	} else if (firestring_estr_estrcasecmp(&stack[depth].type,&ESTR_S("multipart"),0) == 0) {
		struct firestring_estr_t *boundary, searchboundary;
		long start, end, search;

		if (firestring_estr_estrcasecmp(&stack[depth].encoding,&ESTR_S("7bit"),0) != 0 &&
				firestring_estr_estrcasecmp(&stack[depth].encoding,&ESTR_S("8bit"),0) != 0 &&
				firestring_estr_estrcasecmp(&stack[depth].encoding,&ESTR_S("binary"),0) != 0) {
			r = 10;
			goto cleanup;
		}

		/* we have a multipart type, we need to recurse down */
		boundary = firemime_param_value(stack[depth].type_params,&ESTR_S("boundary"));
		if (boundary == NULL) {
			r = 7;
			goto cleanup;
		}
		
		firestring_estr_alloc(&searchboundary,boundary->l + 2);
		firestring_estr_sprintf(&searchboundary,"--%e",boundary);

		start = search = 0;
		while ((end = firestring_estr_estrstr(&stack[depth].body,&searchboundary,search)) != -1) {
			int cr;
			int last = 0;
			int bodyend;
			search = end + 1;

			/* rule: boundary must start at the beginning of a line */
			if (end != 0) {
				/* not the beginning of the first line */
				if (end < linebreak->l)
					/* no space for a linebreak */
					continue;
				if (firestring_estr_estrcmp(linebreak,&(struct firestring_estr_t) { &stack[depth].body.s[end - linebreak->l], linebreak->l, linebreak->l },0) != 0)
					/* not preceded by a linebreak */
					continue;
			}

			/* check that this has enough room to be the ending line */
			if (end + searchboundary.l + 2 > stack[depth].body.l) {
				firestring_estr_free(&searchboundary);
				r = 8;
				goto cleanup;
			}

			if (memcmp(&stack[depth].body.s[end + searchboundary.l],"--",2) == 0) /* ending delimeter */
				last = 1;

			bodyend = end - linebreak->l;
			end = firestring_estr_estrstr(&stack[depth].body,linebreak,end);
			if (end == -1) {
				firestring_estr_free(&searchboundary);
				r = 9;
				goto cleanup;
			}
			end += linebreak->l;

			if (start == 0) {
				/* first boundary line */
				start = end;
				continue;
			}
			cr = firemime_int_parse(&(struct firestring_estr_t) { &stack[depth].body.s[start], bodyend - start, bodyend - start },handler,linebreak,stack,depth + 1,doublebreak,opaque);
			if (cr >= 2) {
				firestring_estr_free(&searchboundary);
				r = cr;
				goto cleanup;
			} else if (cr == 1)
				r = 1;

			/* check for end of parts */
			if (last == 1) {
				firestring_estr_free(&searchboundary);
				goto cleanup;
			}

			start = end;
		}
		firestring_estr_free(&searchboundary);
		r = 8;
		goto cleanup;
	}

cleanup:
	/* if the string points outside the part, we must have allocated it dynamically; free it */
	firestring_estr_free(&stack[depth].type);
	firestring_estr_free(&stack[depth].subtype);
	firestring_estr_free(&stack[depth].encoding);
	firestring_estr_free(&stack[depth].disposition);
	free_parameters(stack[depth].type_params,&stack[depth].headers);
	free_parameters(stack[depth].disposition_params,&stack[depth].headers);

	memset(&stack[depth],0,sizeof(stack[depth]));

	return r;
}

int firemime_parse(struct firestring_estr_t *message, int (*handler)(struct firemime_part *, int, void *), struct firestring_estr_t *linebreak, void *opaque) {
	struct firemime_part stack[MAXDEPTH];
	struct firestring_estr_t doublebreak;
	int r;

	memset(stack,0,sizeof(struct firemime_part) * MAXDEPTH);

	firestring_estr_alloc(&doublebreak,linebreak->l * 2);
	firestring_estr_sprintf(&doublebreak,"%e%e",linebreak,linebreak);

	r = firemime_int_parse(message,handler,linebreak,stack,0,&doublebreak,opaque);

	firestring_estr_free(&doublebreak);
	return r;
}

int firemime_decode_body(struct firestring_estr_t *dest, struct firemime_part *part, struct firestring_estr_t *linebreak) {
	if (dest->l < part->body.l)
		return 13;
	if (firestring_estr_estrcasecmp(&part->encoding,&ESTR_S("7bit"),0) == 0 ||
			firestring_estr_estrcasecmp(&part->encoding,&ESTR_S("8bit"),0) == 0 ||
			firestring_estr_estrcasecmp(&part->encoding,&ESTR_S("binary"),0) == 0) {
		if (dest->s == part->body.s) /* speed hack for inplace decoding */
			return 0;
		firestring_estr_estrcpy(dest,&part->body,0);
		return 0;
	} else if (firestring_estr_estrcasecmp(&part->encoding,&ESTR_S("base64"),0) == 0) {
		firestring_estr_base64_decode(dest,&part->body);
		return 0;
	} else if (firestring_estr_estrcasecmp(&part->encoding,&ESTR_S("quoted-printable"),0) == 0) {
		int i,o = 0;
		/* we now decode qp */
		for (i = 0; i < part->body.l; i++) {
			if (part->body.s[i] == '=') {
				int c;
				/* equal sign must be followed by a linebreak or two hex characters */
				if (part->body.l - i - 1 >= linebreak->l && memcmp(&part->body.s[i+1],linebreak->s,linebreak->l) == 0) {
					/* soft linebreak */
					i += linebreak->l;
					continue;
				}
				if (part->body.l - i < 3)
					return 15;
				c = firestring_hextoi(&part->body.s[i+1]);
				if (c == -1)
					return 15;
				(unsigned char) dest->s[o++] = c;
				i += 2;
			} else {
				if (part->body.l - i >= linebreak->l && memcmp(&part->body.s[i],linebreak->s,linebreak->l) == 0)
					while ((dest->s[o - 1] == '\t' || dest->s[o - 1] == ' ') && o > 0)
						o--;
				dest->s[o++] = part->body.s[i];
			}
		}
		dest->l = o;
		return 0;
	} else
		return 14;
}

int firemime_strip_markup(struct firestring_estr_t *dest, struct firestring_estr_t *source) {
	int inquote=0, intag=0;
	int i,o = 0;

	for (i = 0; i < source->l; i++) {
		if (inquote) {
			if (source->s[i] == '"')
				inquote = 0;
			continue;
		} else if (intag) {
			if (source->s[i] == '"')
				inquote = 1;
			else if (source->s[i] == '>')
				intag = 0;
			continue;
		} else if (source->s[i] == '<') {
			intag = 1;
		} else if (isspace(source->s[i]) && o > 0 && isspace(dest->s[o - 1])) {
			continue;
		} else
			dest->s[o++] = source->s[i];
	}
	if (inquote || intag)
		return 16;
	dest->l = o;

	firestring_estr_xml_decode(dest,dest);

	return 0;
}

int firemime_strip_comments(struct firestring_estr_t *dest, struct firestring_estr_t *source) {
	int i;
	int incomment = 0;

	if (dest->a < source->l)
		return 13;

	dest->l = 0;
	for (i = 0; i < source->l; i++) {
		if (incomment) {
			if (source->s[i] == ')')
				incomment = 0;
			continue;
		}
		if (source->s[i] == '(') {
			incomment = 1;
			continue;
		}
		dest->s[dest->l++] = source->s[i];
	}
	if (incomment)
		return 11;
	else
		return 0;
}

struct firestring_estr_t *firemime_get_full_header(struct firestring_estr_t *headers, struct firestring_estr_t *headername, struct firestring_estr_t *linebreak) {
	static struct firestring_estr_t ret;
	int i = -1;

	while ((i = firestring_estr_estristr(headers,headername,i+1)) != -1) {
		if (i != 0) {
			if (i < linebreak->l)
				continue;
			if (firestring_estr_estrcmp(&(struct firestring_estr_t) { &headers->s[i - linebreak->l], linebreak->l, linebreak->l }, linebreak,0) != 0)
				continue;
		}
		if (headers->s[i + headername->l] != ':')
			continue;
		/* we're now quite sure that this is the header in question */
		ret.s = &headers->s[i]; /* skip the header name and the colon */
		ret.l = ret.a = headers->l - i;

		/* find the end */
		i = firestring_estr_estrstr(&ret,linebreak,0);
		while (strchr("\t ",ret.s[i + linebreak->l]) != NULL)
			i = firestring_estr_estrstr(&ret,linebreak,i+1);
		ret.l = ret.a = i + linebreak->l;
		return &ret;
	}

	return NULL;
}

struct firestring_estr_t *firemime_get_header(struct firestring_estr_t *headers, struct firestring_estr_t *headername, struct firestring_estr_t *linebreak) {
	static struct firestring_estr_t *ret;

	ret = firemime_get_full_header(headers,headername,linebreak);
	if (ret == NULL)
		return NULL;

	ret->l -= headername->l + linebreak->l + 1;
	ret->a = ret->l;
	ret->s += headername->l + 1;

	firestring_estr_trim(ret);
	
	return ret;
}

struct firestring_estr_t *firemime_param_value(struct firemime_param *paramlist, struct firestring_estr_t *name) {
	while (paramlist != NULL) {
		if (firestring_estr_estrcasecmp(name,&paramlist->name,0) == 0)
			return &paramlist->value;
		paramlist = paramlist->next;
	}
	return NULL;
}

const char *firemime_strerror(int error) {
	return errors[error];
}
