/* -----------------------------------------------------------------------------
 * makespnego makes an RFC 2743 GSS-API InitialContextToken containing an RFC
 * 2478 SPNEGO NegotiationToken.
 *
 * Author: Frank Balluffi
 *
 * Copyright (C) 2002-2005 Frank Balluffi. All rights reserved.
 * -----------------------------------------------------------------------------
 */

#include "../../include/filehelp.h"
#include "../../include/krb5help.h"
#include "../../include/spnego.h"
#include "../../include/spnegohelp.h"

#include <openssl/objects.h>
#include <string.h>

static int makeSpnegoInitial (FILE *       stream,
                              const char * mechType,
                              const char * mechTokenFile,
                              const char * spnegoTokenFile);
static int makeSpnegoTarget (FILE *       stream,
                             long         negResult,
                             const char * supportedMech,
                             const char * responseTokenFile,
                             const char * spnegoTokenFile);
static void printHelp (FILE * stream);

/* -----------------------------------------------------------------------------
 * main is the main entry point.
 *
 * Returns 1 if successful, 0 otherwise.
 * -----------------------------------------------------------------------------
 */

int main (int     argc,
          char ** argv)
{
    if (argc < 2)
    {
        printHelp (stdout);
        return 0;
    }

    if (!strcmp (argv [1], "init"))
    {
        if (argc < 5)
        {
            printHelp (stdout);
            return 0;
        }

        if (!makeSpnegoInitial (stdout,
                                argv [2],
                                argv [3],
                                argv [4]))
        {
            printf ("makespnego failed.\n");
            return 0;
        }
    }
    else if (!strcmp (argv [1], "targ"))
    {
        if (argc < 6)
        {
            printHelp (stdout);
            return 0;
        }

        if (!makeSpnegoTarget (stdout,
                               atol (argv [2]),
                               argv [3],
                               argv [4],
                               argv [5]))
        {
            printf ("makespnego failed.\n");
            return 0;
        }
    }
    else
    {
        printf ("Unrecognized token choice \'%s\'.\n", argv [1]);
        return 0;
    }

    return 1;
}

/* -----------------------------------------------------------------------------
 * makeSpnegoInitial makes an RFC 2743 GSS-API InitialContextToken containing an
 * RFC 2478 SPNEGO NegotiationToken of choice negTokenInit from a mechanism
 * token.
 *
 * Returns 1 if successful, 0 otherwise.
 * -----------------------------------------------------------------------------
 */

static int makeSpnegoInitial (FILE *       stream,
                              const char * mechType,
                              const char * mechTokenFile,
                              const char * spnegoTokenFile)
{
    unsigned char * mechToken         = NULL;
    size_t          mechTokenLength   = 0;
    ASN1_OBJECT *   object            = NULL;
    int             rc                = 1;
    unsigned char * spnegoToken       = NULL;
    size_t          spnegoTokenLength = 0;

    if (!stream        ||
        !mechType      ||
        !mechTokenFile ||
        !spnegoTokenFile)
        return 0;

    object = OBJ_txt2obj (mechType, 1);

    if (!object)
    {
        fprintf (stream, "Unable to parse mechanism type \'%s\'.\n", mechType);
        rc = 0;
        goto cleanup;
    }

    if (!(mechToken = fileToMemory (mechTokenFile,
                                    &mechTokenLength)))
    {
        fprintf (stream, "Unable to read the file \'%s\'.\n", mechTokenFile);
        rc = 0;
        goto cleanup;
    }

    if (!makeSpnegoInitialToken (object,
                                 mechToken,
                                 mechTokenLength,
                                 &spnegoToken,
                                 &spnegoTokenLength))
    {
        fprintf (stream, "Unable to make SPNEGO initial token.\n");
        rc = 0;
        goto cleanup;
    }

    if (!memoryToFile (spnegoTokenFile,
                       spnegoToken,
                       spnegoTokenLength))
    {
        fprintf (stream, "Unable to write the file \'%s\'.\n", spnegoTokenFile);
        rc = 0;
        goto cleanup;
    }

cleanup:

    if (mechToken)
        free (mechToken);

    if (object)
        ASN1_OBJECT_free (object);

    if (spnegoToken)
        free (spnegoToken);

    return rc;
}

/* -----------------------------------------------------------------------------
 * makeSpnegoTarget makes an RFC 2743 GSS-API InitialContextToken containing an
 * RFC 2478 SPNEGO NegotiationToken of choice negTokenTarg from a response
 * token.
 *
 * Returns 1 if successful, 0 otherwise.
 * -----------------------------------------------------------------------------
 */

static int makeSpnegoTarget (FILE *       stream,
                             long         negResult,
                             const char * supportedMech,
                             const char * responseTokenFile,
                             const char * spnegoTokenFile)
{
    ASN1_OBJECT *   object              = NULL;
    unsigned char * responseToken       = NULL;
    size_t          responseTokenLength = 0;
    int             rc                  = 1;
    unsigned char * spnegoToken         = NULL;
    size_t          spnegoTokenLength   = 0;

    if (!stream            ||
        !supportedMech     ||
        !responseTokenFile ||
        !spnegoTokenFile)
        return 0;

    if (!(responseToken = fileToMemory (responseTokenFile,
                                        &responseTokenLength)))
    {
        fprintf (stream, "Unable to read the file \'%s\'.\n", responseTokenFile);
        rc = 0;
        goto cleanup;
    }

    object = OBJ_txt2obj (supportedMech, 1);

    if (!object)
    {
        fprintf (stream, "Unable to parse supported mechanism \'%s\'.\n", supportedMech);
        rc = 0;
        goto cleanup;
    }

    if (!makeSpnegoTargetToken (&negResult,
                                object,
                                responseToken,
                                responseTokenLength,
                                NULL,
                                0,
                                &spnegoToken,
                                &spnegoTokenLength))
    {
        fprintf (stream, "Unable to make SPNEGO target token.\n");
        rc = 0;
        goto cleanup;
    }

    if (!memoryToFile (spnegoTokenFile,
                       spnegoToken,
                       spnegoTokenLength))
    {
        fprintf (stream, "Unable to write the file \'%s\'.\n", spnegoTokenFile);
        rc = 0;
        goto cleanup;
    }

cleanup:

    if (object)
        ASN1_OBJECT_free (object);

    if (responseToken)
        free (responseToken);

    if (spnegoToken)
        free (spnegoToken);

    return rc;
}

/* -----------------------------------------------------------------------------
 * printHelp prints help.
 *
 * Returns nothing.
 * -----------------------------------------------------------------------------
 */

static void printHelp (FILE * stream)
{
    static char help [] = "usage: makespnego init | targ arguments\n\n"
                          "init | targ     token choice: negTokenInit or negTokenTarg\n\n"
                          "if init, arguments are:\n\n"
                          "mech-type       period-delimited mechanism type\n"
                          "mech-token      file containing mechanism token\n"
                          "spnego-token    file to contain SPNEGO GSS-API token\n\n"
                          "if targ, arguments are:\n\n"
                          "neg-result      negotiation result\n"
                          "supported-mech  period-delimited supported mechanism\n"
                          "response-token  file containing response token\n"
                          "spnego-token    file to contain SPNEGO GSS-API token\n";

    if (!stream)
        return;

    fprintf (stream, help);
    return;
}
