/* -----------------------------------------------------------------------------
 * spnegohelp.c defines RFC 2478 SPNEGO GSS-API mechanism helper APIs.
 *
 * Author: Frank Balluffi
 *
 * Copyright (C) 2002-2005 Frank Balluffi. All rights reserved.
 * -----------------------------------------------------------------------------
 */

#include "../../include/spnegohelp.h"
#include "../../include/asn1help.h"
#include "../../include/gssapi.h"
#include "../../include/spnego.h"

#include <string.h>

/* Object identifier definitions. */

static unsigned char _spnegoGssApi [] = {0x2b, 0x06, 0x01, 0x05, 0x05, 0x02};

const ASN1_OBJECT spnegoGssApi = {0, 0, 0, sizeof _spnegoGssApi, _spnegoGssApi, 0};

/* Static helper function declarations. */

static ASN1_OCTET_STRING * makeOctetString (const unsigned char * data,
                                            size_t                length);

/* Functions. */

int makeSpnegoInitialToken (const ASN1_OBJECT *   mechType,
                            const unsigned char * mechToken,
                            size_t                mechTokenLength,
                            unsigned char **      spnegoToken,
                            size_t *              spnegoTokenLength)
{
    unsigned char *                     derNext                 = NULL;
    GSSAPI_INITIAL_CONTEXT_TOKEN_BODY * initialContextTokenBody = NULL;
    SPNEGO_NEGOTIATION_TOKEN *          negotiationToken        = NULL;
    ASN1_OBJECT *                       object                  = NULL;
    int                                 rc                      = 1;

    if (!spnegoToken ||
        !spnegoTokenLength)
        return 0;

    negotiationToken = SPNEGO_NEGOTIATION_TOKEN_new ();

    if (!negotiationToken)
    {
        rc = 0;
        goto cleanup;
    }

    negotiationToken->type = SPNEGO_NEG_TOKEN_INIT_CHOICE;
    negotiationToken->value.negTokenInit = SPNEGO_NEG_TOKEN_INIT_new ();

    if (!negotiationToken->value.negTokenInit)
    {
        rc = 0;
        goto cleanup;
    }

    if (mechType)
    {
        negotiationToken->value.negTokenInit->mechTypes = sk_ASN1_OBJECT_new_null ();

        if (!negotiationToken->value.negTokenInit->mechTypes)
        {
            rc = 0;
            goto cleanup;
        }

        object = ASN1_OBJECT_dup (mechType);

        if (!object)
        {
            rc = 0;
            goto cleanup;
        }

        if (!sk_ASN1_OBJECT_push (negotiationToken->value.negTokenInit->mechTypes, object))
        {
            rc = 0;
            goto cleanup;
        }
    }

    if (mechToken && mechTokenLength)
    {
        negotiationToken->value.negTokenInit->mechToken = makeOctetString (mechToken,
                                                                           mechTokenLength);

        if (!negotiationToken->value.negTokenInit->mechToken)
        {
            rc = 0;
            goto cleanup;
        }
    }

    initialContextTokenBody = (GSSAPI_INITIAL_CONTEXT_TOKEN_BODY *) GSSAPI_INITIAL_CONTEXT_TOKEN_new ();

    if (!initialContextTokenBody)
    {
        rc = 0;
        goto cleanup;
    }

    ASN1_OBJECT_free (initialContextTokenBody->thisMech);
    initialContextTokenBody->thisMech = (ASN1_OBJECT *) &spnegoGssApi;

   /*
    d2i_GSSAPI_INITIAL_CONTEXT_TOKEN sets V_ASN1_OTHER, not V_ASN1_ANY.
    */

    initialContextTokenBody->innerContextToken->type = V_ASN1_OTHER;
    initialContextTokenBody->innerContextToken->value.asn1_string = ASN1_STRING_new ();

    if (!initialContextTokenBody->innerContextToken->value.asn1_string)
    {
        rc = 0;
        goto cleanup;
    }

    initialContextTokenBody->innerContextToken->value.asn1_string->type = V_ASN1_OTHER;
    initialContextTokenBody->innerContextToken->value.asn1_string->flags = 0;
    initialContextTokenBody->innerContextToken->value.asn1_string->length = i2d_SPNEGO_NEGOTIATION_TOKEN (negotiationToken, 0);

    if (initialContextTokenBody->innerContextToken->value.asn1_string->length <= 0)
    {
        rc = 0;
        goto cleanup;
    }

    initialContextTokenBody->innerContextToken->value.asn1_string->data = OPENSSL_malloc (initialContextTokenBody->innerContextToken->value.asn1_string->length);

    if (!initialContextTokenBody->innerContextToken->value.asn1_string->data)
    {
        rc = 0;
        goto cleanup;
    }

   /*
    Because i2d functions modify their second argument, use the variable
    derNext.
    */

    derNext = initialContextTokenBody->innerContextToken->value.asn1_string->data;
    initialContextTokenBody->innerContextToken->value.asn1_string->length = i2d_SPNEGO_NEGOTIATION_TOKEN (negotiationToken,
                                                                                                          &derNext);

    if (initialContextTokenBody->innerContextToken->value.asn1_string->length <= 0)
    {
        rc = 0;
        goto cleanup;
    }

    *spnegoTokenLength = i2d_GSSAPI_INITIAL_CONTEXT_TOKEN ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody,
                                                           0);

    if (*spnegoTokenLength <= 0)
    {
        rc = 0;
        goto cleanup;
    }

    /* Call malloc so the caller can call free. */

    *spnegoToken = malloc (*spnegoTokenLength);

    if (!*spnegoToken)
    {
        rc = 0;
        goto cleanup;
    }

    derNext = *spnegoToken;
    *spnegoTokenLength = i2d_GSSAPI_INITIAL_CONTEXT_TOKEN ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody,
                                                           &derNext);

    if (*spnegoTokenLength <= 0)
    {
        rc = 0;
        goto cleanup;
    }

cleanup:

    if (initialContextTokenBody)
        GSSAPI_INITIAL_CONTEXT_TOKEN_free ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody);

    if (negotiationToken)
        SPNEGO_NEGOTIATION_TOKEN_free (negotiationToken);

    return rc;
}

int makeSpnegoTargetToken (const long *          negResult,
                           const ASN1_OBJECT *   supportedMech,
                           const unsigned char * responseToken,
                           size_t                responseTokenLength,
                           const unsigned char * mechListMic,
                           size_t                mechListMicLength,
                           unsigned char **      spnegoToken,
                           size_t *              spnegoTokenLength)
{
    unsigned char *                     derNext                 = NULL;
    GSSAPI_INITIAL_CONTEXT_TOKEN_BODY * initialContextTokenBody = NULL;
    SPNEGO_NEGOTIATION_TOKEN *          negotiationToken        = NULL;
    int                                 rc                      = 1;

    if (!spnegoToken ||
        !spnegoTokenLength)
        return 0;

    negotiationToken = SPNEGO_NEGOTIATION_TOKEN_new ();

    if (!negotiationToken)
    {
        rc = 0;
        goto cleanup;
    }

    negotiationToken->type = SPNEGO_NEG_TOKEN_TARG_CHOICE;
    negotiationToken->value.negTokenTarg = SPNEGO_NEG_TOKEN_TARG_new ();

    if (!negotiationToken->value.negTokenTarg)
    {
        rc = 0;
        goto cleanup;
    }

    if (negResult)
    {
        negotiationToken->value.negTokenTarg->negResult = ASN1_ENUMERATED_new ();

        if (!negotiationToken->value.negTokenTarg->negResult)
        {
            rc = 0;
            goto cleanup;
        }

        if (!(ASN1_ENUMERATED_set (negotiationToken->value.negTokenTarg->negResult,
                                   *negResult)))
        {
            rc = 0;
            goto cleanup;
        }
    }

    if (supportedMech)
    {
        negotiationToken->value.negTokenTarg->supportedMech = ASN1_OBJECT_dup (supportedMech);

        if (!negotiationToken->value.negTokenTarg->supportedMech)
        {
            rc = 0;
            goto cleanup;
        }
    }

    if (responseToken && responseTokenLength)
    {
        negotiationToken->value.negTokenTarg->responseToken = makeOctetString (responseToken,
                                                                               responseTokenLength);

        if (!negotiationToken->value.negTokenTarg->responseToken)
        {
            rc = 0;
            goto cleanup;
        }
    }

    if (mechListMic && mechListMicLength)
    {
        negotiationToken->value.negTokenTarg->mechListMIC = makeOctetString (mechListMic,
                                                                             mechListMicLength);

        if (!negotiationToken->value.negTokenTarg->mechListMIC)
        {
            rc = 0;
            goto cleanup;
        }
    }

    initialContextTokenBody = (GSSAPI_INITIAL_CONTEXT_TOKEN_BODY *) GSSAPI_INITIAL_CONTEXT_TOKEN_new ();

    if (!initialContextTokenBody)
    {
        rc = 0;
        goto cleanup;
    }

    ASN1_OBJECT_free (initialContextTokenBody->thisMech);
    initialContextTokenBody->thisMech = (ASN1_OBJECT *) &spnegoGssApi;

   /*
    d2i_GSSAPI_INITIAL_CONTEXT_TOKEN sets V_ASN1_OTHER, not V_ASN1_ANY.
    */

    initialContextTokenBody->innerContextToken->type = V_ASN1_OTHER;
    initialContextTokenBody->innerContextToken->value.asn1_string = ASN1_STRING_new ();

    if (!initialContextTokenBody->innerContextToken->value.asn1_string)
    {
        rc = 0;
        goto cleanup;
    }

    initialContextTokenBody->innerContextToken->value.asn1_string->type = V_ASN1_OTHER;
    initialContextTokenBody->innerContextToken->value.asn1_string->flags = 0;
    initialContextTokenBody->innerContextToken->value.asn1_string->length = i2d_SPNEGO_NEGOTIATION_TOKEN (negotiationToken, 0);

    if (initialContextTokenBody->innerContextToken->value.asn1_string->length <= 0)
    {
        rc = 0;
        goto cleanup;
    }

    initialContextTokenBody->innerContextToken->value.asn1_string->data = OPENSSL_malloc (initialContextTokenBody->innerContextToken->value.asn1_string->length);

    if (!initialContextTokenBody->innerContextToken->value.asn1_string->data)
    {
        rc = 0;
        goto cleanup;
    }

   /*
    Because i2d functions modify their second argument, use the variable
    derNext.
    */

    derNext = initialContextTokenBody->innerContextToken->value.asn1_string->data;
    initialContextTokenBody->innerContextToken->value.asn1_string->length = i2d_SPNEGO_NEGOTIATION_TOKEN (negotiationToken,
                                                                                                          &derNext);

    if (initialContextTokenBody->innerContextToken->value.asn1_string->length <= 0)
    {
        rc = 0;
        goto cleanup;
    }

    *spnegoTokenLength = i2d_GSSAPI_INITIAL_CONTEXT_TOKEN ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody,
                                                           0);

    if (*spnegoTokenLength <= 0)
    {
        rc = 0;
        goto cleanup;
    }

    /* Call malloc so the caller can call free. */

    *spnegoToken = malloc (*spnegoTokenLength);

    if (!*spnegoToken)
    {
        rc = 0;
        goto cleanup;
    }

    derNext = *spnegoToken;
    *spnegoTokenLength = i2d_GSSAPI_INITIAL_CONTEXT_TOKEN ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody,
                                                           &derNext);

    if (*spnegoTokenLength <= 0)
    {
        rc = 0;
        goto cleanup;
    }

cleanup:

    if (initialContextTokenBody)
        GSSAPI_INITIAL_CONTEXT_TOKEN_free ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody);

    if (negotiationToken)
        SPNEGO_NEGOTIATION_TOKEN_free (negotiationToken);

    return rc;
}

int parseSpnegoInitialToken (const unsigned char * spnegoToken,
                             size_t                spnegoTokenLength,
                             const ASN1_OBJECT *   mechType,
                             unsigned char **      mechToken,
                             size_t *              mechTokenLength)
{
    unsigned char *                     derNext                 = NULL;
    int                                 foundMechType           = 0;
    int                                 i                       = 0;
    GSSAPI_INITIAL_CONTEXT_TOKEN_BODY * initialContextTokenBody = NULL;
    SPNEGO_NEGOTIATION_TOKEN *          negotiationToken        = NULL;
    int                                 rc                      = 1;

    if (!spnegoToken)
        return 0;

    derNext = (unsigned char *) spnegoToken;
    initialContextTokenBody = (GSSAPI_INITIAL_CONTEXT_TOKEN_BODY *) d2i_GSSAPI_INITIAL_CONTEXT_TOKEN (0,
                                                                                                      &derNext,
                                                                                                      (long) spnegoTokenLength);

    if (!initialContextTokenBody)
    {
        rc = 0;
        goto cleanup;
    }

    if (ASN1_OBJECT_cmp (initialContextTokenBody->thisMech,
                         &spnegoGssApi))
    {
        rc = 0;
        goto cleanup;
    }

    derNext = initialContextTokenBody->innerContextToken->value.sequence->data;
    negotiationToken = d2i_SPNEGO_NEGOTIATION_TOKEN (0,
                                                     &derNext,
                                                     initialContextTokenBody->innerContextToken->value.sequence->length);

    if (!negotiationToken)
    {
        rc = 0;
        goto cleanup;
    }

    if (negotiationToken->type != SPNEGO_NEG_TOKEN_INIT_CHOICE)
    {
        rc = 0;
        goto cleanup;
    }

    if (mechType)
    {
        if (!negotiationToken->value.negTokenInit->mechTypes)
        {
            rc = 0;
            goto cleanup;
        }

        for (i = 0;
             i < sk_ASN1_OBJECT_num (negotiationToken->value.negTokenInit->mechTypes);
             i++)
        {
            if (!ASN1_OBJECT_cmp (sk_ASN1_OBJECT_value (negotiationToken->value.negTokenInit->mechTypes,
                                                        i),
                                  mechType))
            {
                foundMechType = 1;
                break;
            }
        }

        if (!foundMechType)
        {
            rc = 0;
            goto cleanup;
        }
    }

    if (mechToken && mechTokenLength)
    {
        if (!negotiationToken->value.negTokenInit->mechToken->data ||
            !negotiationToken->value.negTokenInit->mechToken->length)
        {
            rc = 0;
            goto cleanup;
        }

        /* Call malloc so the caller can call free. */

        *mechToken = malloc (negotiationToken->value.negTokenInit->mechToken->length);

        if (!*mechToken)
        {
            rc = 0;
            goto cleanup;
        }

        memcpy (*mechToken,
                negotiationToken->value.negTokenInit->mechToken->data,
                negotiationToken->value.negTokenInit->mechToken->length);
        *mechTokenLength = negotiationToken->value.negTokenInit->mechToken->length;
    }

cleanup:

    if (initialContextTokenBody)
        GSSAPI_INITIAL_CONTEXT_TOKEN_free ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody);

    if (negotiationToken)
        SPNEGO_NEGOTIATION_TOKEN_free (negotiationToken);

    return rc;
}

int parseSpnegoTargetToken (const unsigned char * spnegoToken,
                            size_t                spnegoTokenLength,
                            long *                negResult,
                            ASN1_OBJECT **        supportedMech,
                            unsigned char **      responseToken,
                            size_t *              responseTokenLength,
                            unsigned char **      mechListMIC,
                            size_t *              mechListMICLength)
{
    unsigned char *                     derNext                 = NULL;
    GSSAPI_INITIAL_CONTEXT_TOKEN_BODY * initialContextTokenBody = NULL;
    SPNEGO_NEGOTIATION_TOKEN *          negotiationToken        = NULL;
    int                                 rc                      = 1;

    if (!spnegoToken)
        return 0;

    derNext = (unsigned char *) spnegoToken;
    initialContextTokenBody = (GSSAPI_INITIAL_CONTEXT_TOKEN_BODY *) d2i_GSSAPI_INITIAL_CONTEXT_TOKEN (0,
                                                                                                      &derNext,
                                                                                                      (long) spnegoTokenLength);

    if (!initialContextTokenBody)
    {
        rc = 0;
        goto cleanup;
    }

    if (ASN1_OBJECT_cmp (initialContextTokenBody->thisMech,
                         &spnegoGssApi))
    {
        rc = 0;
        goto cleanup;
    }

    derNext = initialContextTokenBody->innerContextToken->value.sequence->data;
    negotiationToken = d2i_SPNEGO_NEGOTIATION_TOKEN (0,
                                                     &derNext,
                                                     initialContextTokenBody->innerContextToken->value.sequence->length);

    if (!negotiationToken)
    {
        rc = 0;
        goto cleanup;
    }

    if (negotiationToken->type != SPNEGO_NEG_TOKEN_TARG_CHOICE)
    {
        rc = 0;
        goto cleanup;
    }

    if (negResult)
    {
        if (!negotiationToken->value.negTokenTarg->negResult)
        {
            rc = 0;
            goto cleanup;
        }

        *negResult = ASN1_ENUMERATED_get (negotiationToken->value.negTokenTarg->negResult);
    }

    if (supportedMech)
    {
        if (!negotiationToken->value.negTokenTarg->supportedMech)
        {
            rc = 0;
            goto cleanup;
        }

        *supportedMech = ASN1_OBJECT_dup (negotiationToken->value.negTokenTarg->supportedMech);

        if (!*supportedMech)
        {
            rc = 0;
            goto cleanup;
        }
    }

    if (responseToken && responseTokenLength)
    {
        if (!negotiationToken->value.negTokenTarg->responseToken->data ||
            !negotiationToken->value.negTokenTarg->responseToken->length)
        {
            rc = 0;
            goto cleanup;
        }

        /* Call malloc so the caller can call free. */

        *responseToken = malloc (negotiationToken->value.negTokenTarg->responseToken->length);

        if (!*responseToken)
        {
            rc = 0;
            goto cleanup;
        }

        memcpy (*responseToken,
                negotiationToken->value.negTokenTarg->responseToken->data,
                negotiationToken->value.negTokenTarg->responseToken->length);
        *responseTokenLength = negotiationToken->value.negTokenTarg->responseToken->length;
    }

    if (mechListMIC && mechListMICLength)
    {
        if (!negotiationToken->value.negTokenTarg->mechListMIC->data ||
            !negotiationToken->value.negTokenTarg->mechListMIC->length)
        {
            rc = 0;
            goto cleanup;
        }

        /* Call malloc so the caller can call free. */

        *mechListMIC = malloc (negotiationToken->value.negTokenTarg->mechListMIC->length);

        if (!*mechListMIC)
        {
            rc = 0;
            goto cleanup;
        }

        memcpy (*mechListMIC,
                negotiationToken->value.negTokenTarg->mechListMIC->data,
                negotiationToken->value.negTokenTarg->mechListMIC->length);
        *mechListMICLength = negotiationToken->value.negTokenTarg->mechListMIC->length;
    }

cleanup:

    if (initialContextTokenBody)
        GSSAPI_INITIAL_CONTEXT_TOKEN_free ((GSSAPI_INITIAL_CONTEXT_TOKEN *) initialContextTokenBody);

    if (negotiationToken)
        SPNEGO_NEGOTIATION_TOKEN_free (negotiationToken);

    return rc;
}

/* Static helper functions. */

static ASN1_OCTET_STRING * makeOctetString (const unsigned char * data,
                                            size_t                length)
{
    ASN1_OCTET_STRING * octetString = NULL;

    octetString = ASN1_OCTET_STRING_new ();

    if (!octetString)
        goto error;

    octetString->data = OPENSSL_malloc (length);

    if (!octetString->data)
        goto error;

    memcpy (octetString->data, data, length);
    octetString->length = (int) length;
    return octetString;

error:

    if (octetString)
        ASN1_OCTET_STRING_free (octetString);

    return NULL;
}
