//-----------------------------------------------------
// author: "Yann Orlarey"
// name: "freeverb -- a Schroeder reverb"
// version: "1.0"
//
// Code generated with Faust 2.0.a30 (http://faust.grame.fr)
//-----------------------------------------------------

#ifndef  __mydsp_H__
#define  __mydsp_H__
/************************************************************************
 ************************************************************************
    FAUST Architecture File
	Copyright (C) 2006-2011 Albert Graef <Dr.Graef@t-online.de>
    ---------------------------------------------------------------------
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as 
	published by the Free Software Foundation; either version 2.1 of the 
	License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
 	License along with the GNU C Library; if not, write to the Free
  	Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
  	02111-1307 USA. 
 ************************************************************************
 ************************************************************************/

/* Pd architecture file, written by Albert Graef <Dr.Graef@t-online.de>.
   This was derived from minimal.cpp included in the Faust distribution.
   Please note that this is to be compiled as a shared library, which is
   then loaded dynamically by Pd as an external. */

#include <stdlib.h>
#include <math.h>
#include <string>

#include "faust/misc.h"
#include "faust/gui/UI.h"
#include "faust/gui/meta.h"
#include "faust/audio/dsp.h"

//using namespace std;

/******************************************************************************
*******************************************************************************

							       VECTOR INTRINSICS

*******************************************************************************
*******************************************************************************/


/***************************************************************************
   Pd UI interface
 ***************************************************************************/

enum ui_elem_type_t {
  UI_BUTTON, UI_CHECK_BUTTON,
  UI_V_SLIDER, UI_H_SLIDER, UI_NUM_ENTRY,
  UI_V_BARGRAPH, UI_H_BARGRAPH,
  UI_END_GROUP, UI_V_GROUP, UI_H_GROUP, UI_T_GROUP
};

struct ui_elem_t {
  ui_elem_type_t type;
  char *label;
  float *zone;
  float init, min, max, step;
};

class PdUI : public UI
{
public:
  const char *name;
  int nelems, level;
  ui_elem_t *elems;
		
  PdUI();
  PdUI(const char *nm, const char *s);
  virtual ~PdUI();

protected:
  std::string path;
  void add_elem(ui_elem_type_t type, const char *label = NULL);
  void add_elem(ui_elem_type_t type, const char *label, float *zone);
  void add_elem(ui_elem_type_t type, const char *label, float *zone,
		float init, float min, float max, float step);
  void add_elem(ui_elem_type_t type, const char *label, float *zone,
		float min, float max);

public:
  virtual void addButton(const char* label, float* zone);
  virtual void addCheckButton(const char* label, float* zone);
  virtual void addVerticalSlider(const char* label, float* zone, float init, float min, float max, float step);
  virtual void addHorizontalSlider(const char* label, float* zone, float init, float min, float max, float step);
  virtual void addNumEntry(const char* label, float* zone, float init, float min, float max, float step);

  virtual void addHorizontalBargraph(const char* label, float* zone, float min, float max);
  virtual void addVerticalBargraph(const char* label, float* zone, float min, float max);
  
  virtual void openTabBox(const char* label);
  virtual void openHorizontalBox(const char* label);
  virtual void openVerticalBox(const char* label);
  virtual void closeBox();
	
  virtual void run();
};

static std::string mangle(const char *name, int level, const char *s)
{
  const char *s0 = s;
  std::string t = "";
  if (!s) return t;
  // Get rid of bogus "0x00" labels in recent Faust revisions. Also, for
  // backward compatibility with old Faust versions, make sure that default
  // toplevel groups and explicit toplevel groups with an empty label are
  // treated alike (these both return "0x00" labels in the latest Faust, but
  // would be treated inconsistently in earlier versions).
  if (!*s || strcmp(s, "0x00") == 0) {
    if (level == 0)
      // toplevel group with empty label, map to dsp name
      s = name;
    else
      // empty label
      s = "";
  }
  while (*s)
    if (isalnum(*s))
      t += *(s++);
    else {
      const char *s1 = s;
      while (*s && !isalnum(*s)) ++s;
      if (s1 != s0 && *s) t += "-";
    }
  return t;
}

static std::string normpath(std::string path)
{
  path = std::string("/")+path;
  int pos = path.find("//");
  while (pos >= 0) {
    path.erase(pos, 1);
    pos = path.find("//");
  }
  size_t len = path.length();
  if (len > 1 && path[len-1] == '/')
    path.erase(len-1, 1);
  return path;
}

static std::string pathcat(std::string path, std::string label)
{
  if (path.empty())
    return normpath(label);
  else if (label.empty())
    return normpath(path);
  else
    return normpath(path+"/"+label);
}

PdUI::PdUI()
{
  nelems = level = 0;
  elems = NULL;
  name = "";
  path = "";
}

PdUI::PdUI(const char *nm, const char *s)
{
  nelems = level = 0;
  elems = NULL;
  name = nm?nm:"";
  path = s?s:"";
}

PdUI::~PdUI()
{
  if (elems) {
    for (int i = 0; i < nelems; i++)
      if (elems[i].label)
	free(elems[i].label);
    free(elems);
  }
}

inline void PdUI::add_elem(ui_elem_type_t type, const char *label)
{
  ui_elem_t *elems1 = (ui_elem_t*)realloc(elems, (nelems+1)*sizeof(ui_elem_t));
  if (elems1)
    elems = elems1;
  else
    return;
  std::string s = pathcat(path, mangle(name, level, label));
  elems[nelems].type = type;
  elems[nelems].label = strdup(s.c_str());
  elems[nelems].zone = NULL;
  elems[nelems].init = 0.0;
  elems[nelems].min = 0.0;
  elems[nelems].max = 0.0;
  elems[nelems].step = 0.0;
  nelems++;
}

inline void PdUI::add_elem(ui_elem_type_t type, const char *label, float *zone)
{
  ui_elem_t *elems1 = (ui_elem_t*)realloc(elems, (nelems+1)*sizeof(ui_elem_t));
  if (elems1)
    elems = elems1;
  else
    return;
  std::string s = pathcat(path, mangle(name, level, label));
  elems[nelems].type = type;
  elems[nelems].label = strdup(s.c_str());
  elems[nelems].zone = zone;
  elems[nelems].init = 0.0;
  elems[nelems].min = 0.0;
  elems[nelems].max = 1.0;
  elems[nelems].step = 1.0;
  nelems++;
}

inline void PdUI::add_elem(ui_elem_type_t type, const char *label, float *zone,
			  float init, float min, float max, float step)
{
  ui_elem_t *elems1 = (ui_elem_t*)realloc(elems, (nelems+1)*sizeof(ui_elem_t));
  if (elems1)
    elems = elems1;
  else
    return;
  std::string s = pathcat(path, mangle(name, level, label));
  elems[nelems].type = type;
  elems[nelems].label = strdup(s.c_str());
  elems[nelems].zone = zone;
  elems[nelems].init = init;
  elems[nelems].min = min;
  elems[nelems].max = max;
  elems[nelems].step = step;
  nelems++;
}

inline void PdUI::add_elem(ui_elem_type_t type, const char *label, float *zone,
			  float min, float max)
{
  ui_elem_t *elems1 = (ui_elem_t*)realloc(elems, (nelems+1)*sizeof(ui_elem_t));
  if (elems1)
    elems = elems1;
  else
    return;
  std::string s = pathcat(path, mangle(name, level, label));
  elems[nelems].type = type;
  elems[nelems].label = strdup(s.c_str());
  elems[nelems].zone = zone;
  elems[nelems].init = 0.0;
  elems[nelems].min = min;
  elems[nelems].max = max;
  elems[nelems].step = 0.0;
  nelems++;
}

void PdUI::addButton(const char* label, float* zone)
{ add_elem(UI_BUTTON, label, zone); }
void PdUI::addCheckButton(const char* label, float* zone)
{ add_elem(UI_CHECK_BUTTON, label, zone); }
void PdUI::addVerticalSlider(const char* label, float* zone, float init, float min, float max, float step)
{ add_elem(UI_V_SLIDER, label, zone, init, min, max, step); }
void PdUI::addHorizontalSlider(const char* label, float* zone, float init, float min, float max, float step)
{ add_elem(UI_H_SLIDER, label, zone, init, min, max, step); }
void PdUI::addNumEntry(const char* label, float* zone, float init, float min, float max, float step)
{ add_elem(UI_NUM_ENTRY, label, zone, init, min, max, step); }

void PdUI::addHorizontalBargraph(const char* label, float* zone, float min, float max)
{ add_elem(UI_H_BARGRAPH, label, zone, min, max); }
void PdUI::addVerticalBargraph(const char* label, float* zone, float min, float max)
{ add_elem(UI_V_BARGRAPH, label, zone, min, max); }

void PdUI::openTabBox(const char* label)
{
  if (!path.empty()) path += "/";
  path += mangle(name, level, label);
  level++;
}
void PdUI::openHorizontalBox(const char* label)
{
  if (!path.empty()) path += "/";
  path += mangle(name, level, label);
  level++;
}
void PdUI::openVerticalBox(const char* label)
{
  if (!path.empty()) path += "/";
  path += mangle(name, level, label);
  level++;
}
void PdUI::closeBox()
{
  int pos = path.rfind("/");
  if (pos < 0) pos = 0;
  path.erase(pos);
  level--;
}

void PdUI::run() {}

/******************************************************************************
*******************************************************************************

			    FAUST DSP

*******************************************************************************
*******************************************************************************/

//----------------------------------------------------------------------------
//  FAUST generated signal processor
//----------------------------------------------------------------------------
		
#ifndef FAUSTFLOAT
#define FAUSTFLOAT float
#endif  



#ifndef FAUSTCLASS 
#define FAUSTCLASS mydsp
#endif

class mydsp : public dsp {
	
  private:
	
	float fVec0[2048];
	float fVec1[2048];
	float fVec2[2048];
	float fVec3[2048];
	float fVec4[2048];
	float fVec5[2048];
	float fVec6[2048];
	float fVec7[2048];
	float fVec12[2048];
	float fVec13[2048];
	float fVec14[2048];
	float fVec15[2048];
	float fVec16[2048];
	float fVec17[2048];
	float fVec18[2048];
	float fVec19[2048];
	float fVec8[1024];
	float fVec20[1024];
	float fVec9[512];
	float fVec10[512];
	float fVec21[512];
	float fVec22[512];
	float fVec11[256];
	float fVec23[256];
	float fRec9[2];
	float fRec8[2];
	float fRec11[2];
	float fRec10[2];
	float fRec13[2];
	float fRec12[2];
	float fRec15[2];
	float fRec14[2];
	float fRec17[2];
	float fRec16[2];
	float fRec19[2];
	float fRec18[2];
	float fRec21[2];
	float fRec20[2];
	float fRec23[2];
	float fRec22[2];
	float fRec6[2];
	float fRec4[2];
	float fRec2[2];
	float fRec0[2];
	float fRec33[2];
	float fRec32[2];
	float fRec35[2];
	float fRec34[2];
	float fRec37[2];
	float fRec36[2];
	float fRec39[2];
	float fRec38[2];
	float fRec41[2];
	float fRec40[2];
	float fRec43[2];
	float fRec42[2];
	float fRec45[2];
	float fRec44[2];
	float fRec47[2];
	float fRec46[2];
	float fRec30[2];
	float fRec28[2];
	float fRec26[2];
	float fRec24[2];
	FAUSTFLOAT fHslider0;
	FAUSTFLOAT fHslider1;
	FAUSTFLOAT fHslider2;
	int IOTA;
	int fSamplingFreq;
	
  public:
	
	void static metadata(Meta* m) { 
		m->declare("author", "Yann Orlarey");
		m->declare("name", "freeverb -- a Schroeder reverb");
		m->declare("version", "1.0");
	}

	virtual int getNumInputs() {
		return 2;
		
	}
	virtual int getNumOutputs() {
		return 2;
		
	}
	virtual int getInputRate(int channel) {
		int rate;
		switch (channel) {
			case 0: {
				rate = 1;
				break;
			}
			case 1: {
				rate = 1;
				break;
			}
			default: {
				rate = -1;
				break;
			}
			
		}
		return rate;
		
	}
	virtual int getOutputRate(int channel) {
		int rate;
		switch (channel) {
			case 0: {
				rate = 1;
				break;
			}
			case 1: {
				rate = 1;
				break;
			}
			default: {
				rate = -1;
				break;
			}
			
		}
		return rate;
		
	}
	
	static void classInit(int samplingFreq) {
		
	}
	
	virtual void instanceInit(int samplingFreq) {
		fSamplingFreq = samplingFreq;
		fHslider0 = FAUSTFLOAT(0.25);
		fHslider1 = FAUSTFLOAT(0.5);
		fHslider2 = FAUSTFLOAT(0.5);
		for (int i0 = 0; (i0 < 2); i0 = (i0 + 1)) {
			fRec9[i0] = 0.f;
			
		}
		IOTA = 0;
		for (int i1 = 0; (i1 < 2048); i1 = (i1 + 1)) {
			fVec0[i1] = 0.f;
			
		}
		for (int i2 = 0; (i2 < 2); i2 = (i2 + 1)) {
			fRec8[i2] = 0.f;
			
		}
		for (int i3 = 0; (i3 < 2); i3 = (i3 + 1)) {
			fRec11[i3] = 0.f;
			
		}
		for (int i4 = 0; (i4 < 2048); i4 = (i4 + 1)) {
			fVec1[i4] = 0.f;
			
		}
		for (int i5 = 0; (i5 < 2); i5 = (i5 + 1)) {
			fRec10[i5] = 0.f;
			
		}
		for (int i6 = 0; (i6 < 2); i6 = (i6 + 1)) {
			fRec13[i6] = 0.f;
			
		}
		for (int i7 = 0; (i7 < 2048); i7 = (i7 + 1)) {
			fVec2[i7] = 0.f;
			
		}
		for (int i8 = 0; (i8 < 2); i8 = (i8 + 1)) {
			fRec12[i8] = 0.f;
			
		}
		for (int i9 = 0; (i9 < 2); i9 = (i9 + 1)) {
			fRec15[i9] = 0.f;
			
		}
		for (int i10 = 0; (i10 < 2048); i10 = (i10 + 1)) {
			fVec3[i10] = 0.f;
			
		}
		for (int i11 = 0; (i11 < 2); i11 = (i11 + 1)) {
			fRec14[i11] = 0.f;
			
		}
		for (int i12 = 0; (i12 < 2); i12 = (i12 + 1)) {
			fRec17[i12] = 0.f;
			
		}
		for (int i13 = 0; (i13 < 2048); i13 = (i13 + 1)) {
			fVec4[i13] = 0.f;
			
		}
		for (int i14 = 0; (i14 < 2); i14 = (i14 + 1)) {
			fRec16[i14] = 0.f;
			
		}
		for (int i15 = 0; (i15 < 2); i15 = (i15 + 1)) {
			fRec19[i15] = 0.f;
			
		}
		for (int i16 = 0; (i16 < 2048); i16 = (i16 + 1)) {
			fVec5[i16] = 0.f;
			
		}
		for (int i17 = 0; (i17 < 2); i17 = (i17 + 1)) {
			fRec18[i17] = 0.f;
			
		}
		for (int i18 = 0; (i18 < 2); i18 = (i18 + 1)) {
			fRec21[i18] = 0.f;
			
		}
		for (int i19 = 0; (i19 < 2048); i19 = (i19 + 1)) {
			fVec6[i19] = 0.f;
			
		}
		for (int i20 = 0; (i20 < 2); i20 = (i20 + 1)) {
			fRec20[i20] = 0.f;
			
		}
		for (int i21 = 0; (i21 < 2); i21 = (i21 + 1)) {
			fRec23[i21] = 0.f;
			
		}
		for (int i22 = 0; (i22 < 2048); i22 = (i22 + 1)) {
			fVec7[i22] = 0.f;
			
		}
		for (int i23 = 0; (i23 < 2); i23 = (i23 + 1)) {
			fRec22[i23] = 0.f;
			
		}
		for (int i24 = 0; (i24 < 1024); i24 = (i24 + 1)) {
			fVec8[i24] = 0.f;
			
		}
		for (int i25 = 0; (i25 < 2); i25 = (i25 + 1)) {
			fRec6[i25] = 0.f;
			
		}
		for (int i26 = 0; (i26 < 512); i26 = (i26 + 1)) {
			fVec9[i26] = 0.f;
			
		}
		for (int i27 = 0; (i27 < 2); i27 = (i27 + 1)) {
			fRec4[i27] = 0.f;
			
		}
		for (int i28 = 0; (i28 < 512); i28 = (i28 + 1)) {
			fVec10[i28] = 0.f;
			
		}
		for (int i29 = 0; (i29 < 2); i29 = (i29 + 1)) {
			fRec2[i29] = 0.f;
			
		}
		for (int i30 = 0; (i30 < 256); i30 = (i30 + 1)) {
			fVec11[i30] = 0.f;
			
		}
		for (int i31 = 0; (i31 < 2); i31 = (i31 + 1)) {
			fRec0[i31] = 0.f;
			
		}
		for (int i32 = 0; (i32 < 2); i32 = (i32 + 1)) {
			fRec33[i32] = 0.f;
			
		}
		for (int i33 = 0; (i33 < 2048); i33 = (i33 + 1)) {
			fVec12[i33] = 0.f;
			
		}
		for (int i34 = 0; (i34 < 2); i34 = (i34 + 1)) {
			fRec32[i34] = 0.f;
			
		}
		for (int i35 = 0; (i35 < 2); i35 = (i35 + 1)) {
			fRec35[i35] = 0.f;
			
		}
		for (int i36 = 0; (i36 < 2048); i36 = (i36 + 1)) {
			fVec13[i36] = 0.f;
			
		}
		for (int i37 = 0; (i37 < 2); i37 = (i37 + 1)) {
			fRec34[i37] = 0.f;
			
		}
		for (int i38 = 0; (i38 < 2); i38 = (i38 + 1)) {
			fRec37[i38] = 0.f;
			
		}
		for (int i39 = 0; (i39 < 2048); i39 = (i39 + 1)) {
			fVec14[i39] = 0.f;
			
		}
		for (int i40 = 0; (i40 < 2); i40 = (i40 + 1)) {
			fRec36[i40] = 0.f;
			
		}
		for (int i41 = 0; (i41 < 2); i41 = (i41 + 1)) {
			fRec39[i41] = 0.f;
			
		}
		for (int i42 = 0; (i42 < 2048); i42 = (i42 + 1)) {
			fVec15[i42] = 0.f;
			
		}
		for (int i43 = 0; (i43 < 2); i43 = (i43 + 1)) {
			fRec38[i43] = 0.f;
			
		}
		for (int i44 = 0; (i44 < 2); i44 = (i44 + 1)) {
			fRec41[i44] = 0.f;
			
		}
		for (int i45 = 0; (i45 < 2048); i45 = (i45 + 1)) {
			fVec16[i45] = 0.f;
			
		}
		for (int i46 = 0; (i46 < 2); i46 = (i46 + 1)) {
			fRec40[i46] = 0.f;
			
		}
		for (int i47 = 0; (i47 < 2); i47 = (i47 + 1)) {
			fRec43[i47] = 0.f;
			
		}
		for (int i48 = 0; (i48 < 2048); i48 = (i48 + 1)) {
			fVec17[i48] = 0.f;
			
		}
		for (int i49 = 0; (i49 < 2); i49 = (i49 + 1)) {
			fRec42[i49] = 0.f;
			
		}
		for (int i50 = 0; (i50 < 2); i50 = (i50 + 1)) {
			fRec45[i50] = 0.f;
			
		}
		for (int i51 = 0; (i51 < 2048); i51 = (i51 + 1)) {
			fVec18[i51] = 0.f;
			
		}
		for (int i52 = 0; (i52 < 2); i52 = (i52 + 1)) {
			fRec44[i52] = 0.f;
			
		}
		for (int i53 = 0; (i53 < 2); i53 = (i53 + 1)) {
			fRec47[i53] = 0.f;
			
		}
		for (int i54 = 0; (i54 < 2048); i54 = (i54 + 1)) {
			fVec19[i54] = 0.f;
			
		}
		for (int i55 = 0; (i55 < 2); i55 = (i55 + 1)) {
			fRec46[i55] = 0.f;
			
		}
		for (int i56 = 0; (i56 < 1024); i56 = (i56 + 1)) {
			fVec20[i56] = 0.f;
			
		}
		for (int i57 = 0; (i57 < 2); i57 = (i57 + 1)) {
			fRec30[i57] = 0.f;
			
		}
		for (int i58 = 0; (i58 < 512); i58 = (i58 + 1)) {
			fVec21[i58] = 0.f;
			
		}
		for (int i59 = 0; (i59 < 2); i59 = (i59 + 1)) {
			fRec28[i59] = 0.f;
			
		}
		for (int i60 = 0; (i60 < 512); i60 = (i60 + 1)) {
			fVec22[i60] = 0.f;
			
		}
		for (int i61 = 0; (i61 < 2); i61 = (i61 + 1)) {
			fRec26[i61] = 0.f;
			
		}
		for (int i62 = 0; (i62 < 256); i62 = (i62 + 1)) {
			fVec23[i62] = 0.f;
			
		}
		for (int i63 = 0; (i63 < 2); i63 = (i63 + 1)) {
			fRec24[i63] = 0.f;
			
		}
		
	}
	
	virtual void init(int samplingFreq) {
		classInit(samplingFreq);
		instanceInit(samplingFreq);
	}
	
	virtual void buildUserInterface(UI* interface) {
		interface->openVerticalBox("freeverb");
		interface->addHorizontalSlider("damp", &fHslider2, 0.5f, 0.f, 1.f, 0.025f);
		interface->addHorizontalSlider("roomsize", &fHslider1, 0.5f, 0.f, 1.f, 0.025f);
		interface->addHorizontalSlider("wet", &fHslider0, 0.25f, 0.f, 1.f, 0.025f);
		interface->closeBox();
		
	}
	
	virtual void compute(int count, FAUSTFLOAT** inputs, FAUSTFLOAT** outputs) {
		FAUSTFLOAT* input0 = inputs[0];
		FAUSTFLOAT* input1 = inputs[1];
		FAUSTFLOAT* output0 = outputs[0];
		FAUSTFLOAT* output1 = outputs[1];
		float fSlow0 = float(fHslider0);
		float fSlow1 = float((0.7f + float((0.28f * float(fHslider1)))));
		float fSlow2 = float((0.4f * float(fHslider2)));
		float fSlow3 = float((1.f - fSlow2));
		float fSlow4 = float((1.f - fSlow0));
		for (int i = 0; (i < count); i = (i + 1)) {
			fRec9[0] = float((float((fSlow2 * fRec9[1])) + float((fSlow3 * fRec8[1]))));
			float fTemp0 = float(input0[i]);
			float fTemp1 = float(input1[i]);
			float fTemp2 = float((0.015f * float((fTemp0 + fTemp1))));
			fVec0[(IOTA & 2047)] = float((float((fSlow1 * fRec9[0])) + fTemp2));
			fRec8[0] = fVec0[((IOTA - 1116) & 2047)];
			fRec11[0] = float((float((fSlow2 * fRec11[1])) + float((fSlow3 * fRec10[1]))));
			fVec1[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec11[0]))));
			fRec10[0] = fVec1[((IOTA - 1188) & 2047)];
			fRec13[0] = float((float((fSlow2 * fRec13[1])) + float((fSlow3 * fRec12[1]))));
			fVec2[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec13[0]))));
			fRec12[0] = fVec2[((IOTA - 1277) & 2047)];
			fRec15[0] = float((float((fSlow2 * fRec15[1])) + float((fSlow3 * fRec14[1]))));
			fVec3[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec15[0]))));
			fRec14[0] = fVec3[((IOTA - 1356) & 2047)];
			fRec17[0] = float((float((fSlow2 * fRec17[1])) + float((fSlow3 * fRec16[1]))));
			fVec4[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec17[0]))));
			fRec16[0] = fVec4[((IOTA - 1422) & 2047)];
			fRec19[0] = float((float((fSlow2 * fRec19[1])) + float((fSlow3 * fRec18[1]))));
			fVec5[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec19[0]))));
			fRec18[0] = fVec5[((IOTA - 1491) & 2047)];
			fRec21[0] = float((float((fSlow2 * fRec21[1])) + float((fSlow3 * fRec20[1]))));
			fVec6[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec21[0]))));
			fRec20[0] = fVec6[((IOTA - 1557) & 2047)];
			fRec23[0] = float((float((fSlow2 * fRec23[1])) + float((fSlow3 * fRec22[1]))));
			fVec7[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec23[0]))));
			fRec22[0] = fVec7[((IOTA - 1617) & 2047)];
			float fTemp3 = float((float((float((float((float((float((float((fRec8[0] + fRec10[0])) + fRec12[0])) + fRec14[0])) + fRec16[0])) + fRec18[0])) + fRec20[0])) + fRec22[0]));
			fVec8[(IOTA & 1023)] = float((fTemp3 + float((0.5f * fRec6[1]))));
			fRec6[0] = fVec8[((IOTA - 556) & 1023)];
			float fRec7 = float((0.f - float((fTemp3 - fRec6[1]))));
			fVec9[(IOTA & 511)] = float((fRec7 + float((0.5f * fRec4[1]))));
			fRec4[0] = fVec9[((IOTA - 441) & 511)];
			float fRec5 = float((fRec4[1] - fRec7));
			fVec10[(IOTA & 511)] = float((fRec5 + float((0.5f * fRec2[1]))));
			fRec2[0] = fVec10[((IOTA - 341) & 511)];
			float fRec3 = float((fRec2[1] - fRec5));
			fVec11[(IOTA & 255)] = float((fRec3 + float((0.5f * fRec0[1]))));
			fRec0[0] = fVec11[((IOTA - 225) & 255)];
			float fRec1 = float((fRec0[1] - fRec3));
			output0[i] = FAUSTFLOAT(float((float((fSlow0 * fRec1)) + float((fSlow4 * fTemp0)))));
			fRec33[0] = float((float((fSlow2 * fRec33[1])) + float((fSlow3 * fRec32[1]))));
			fVec12[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec33[0]))));
			fRec32[0] = fVec12[((IOTA - 1139) & 2047)];
			fRec35[0] = float((float((fSlow2 * fRec35[1])) + float((fSlow3 * fRec34[1]))));
			fVec13[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec35[0]))));
			fRec34[0] = fVec13[((IOTA - 1211) & 2047)];
			fRec37[0] = float((float((fSlow2 * fRec37[1])) + float((fSlow3 * fRec36[1]))));
			fVec14[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec37[0]))));
			fRec36[0] = fVec14[((IOTA - 1300) & 2047)];
			fRec39[0] = float((float((fSlow2 * fRec39[1])) + float((fSlow3 * fRec38[1]))));
			fVec15[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec39[0]))));
			fRec38[0] = fVec15[((IOTA - 1379) & 2047)];
			fRec41[0] = float((float((fSlow2 * fRec41[1])) + float((fSlow3 * fRec40[1]))));
			fVec16[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec41[0]))));
			fRec40[0] = fVec16[((IOTA - 1445) & 2047)];
			fRec43[0] = float((float((fSlow2 * fRec43[1])) + float((fSlow3 * fRec42[1]))));
			fVec17[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec43[0]))));
			fRec42[0] = fVec17[((IOTA - 1514) & 2047)];
			fRec45[0] = float((float((fSlow2 * fRec45[1])) + float((fSlow3 * fRec44[1]))));
			fVec18[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec45[0]))));
			fRec44[0] = fVec18[((IOTA - 1580) & 2047)];
			fRec47[0] = float((float((fSlow2 * fRec47[1])) + float((fSlow3 * fRec46[1]))));
			fVec19[(IOTA & 2047)] = float((fTemp2 + float((fSlow1 * fRec47[0]))));
			fRec46[0] = fVec19[((IOTA - 1640) & 2047)];
			float fTemp4 = float((float((float((float((float((float((float((fRec32[0] + fRec34[0])) + fRec36[0])) + fRec38[0])) + fRec40[0])) + fRec42[0])) + fRec44[0])) + fRec46[0]));
			fVec20[(IOTA & 1023)] = float((fTemp4 + float((0.5f * fRec30[1]))));
			fRec30[0] = fVec20[((IOTA - 579) & 1023)];
			float fRec31 = float((0.f - float((fTemp4 - fRec30[1]))));
			fVec21[(IOTA & 511)] = float((fRec31 + float((0.5f * fRec28[1]))));
			fRec28[0] = fVec21[((IOTA - 464) & 511)];
			float fRec29 = float((fRec28[1] - fRec31));
			fVec22[(IOTA & 511)] = float((fRec29 + float((0.5f * fRec26[1]))));
			fRec26[0] = fVec22[((IOTA - 364) & 511)];
			float fRec27 = float((fRec26[1] - fRec29));
			fVec23[(IOTA & 255)] = float((fRec27 + float((0.5f * fRec24[1]))));
			fRec24[0] = fVec23[((IOTA - 248) & 255)];
			float fRec25 = float((fRec24[1] - fRec27));
			output1[i] = FAUSTFLOAT(float((float((fSlow0 * fRec25)) + float((fSlow4 * fTemp1)))));
			fRec9[1] = fRec9[0];
			IOTA = (IOTA + 1);
			fRec8[1] = fRec8[0];
			fRec11[1] = fRec11[0];
			fRec10[1] = fRec10[0];
			fRec13[1] = fRec13[0];
			fRec12[1] = fRec12[0];
			fRec15[1] = fRec15[0];
			fRec14[1] = fRec14[0];
			fRec17[1] = fRec17[0];
			fRec16[1] = fRec16[0];
			fRec19[1] = fRec19[0];
			fRec18[1] = fRec18[0];
			fRec21[1] = fRec21[0];
			fRec20[1] = fRec20[0];
			fRec23[1] = fRec23[0];
			fRec22[1] = fRec22[0];
			fRec6[1] = fRec6[0];
			fRec4[1] = fRec4[0];
			fRec2[1] = fRec2[0];
			fRec0[1] = fRec0[0];
			fRec33[1] = fRec33[0];
			fRec32[1] = fRec32[0];
			fRec35[1] = fRec35[0];
			fRec34[1] = fRec34[0];
			fRec37[1] = fRec37[0];
			fRec36[1] = fRec36[0];
			fRec39[1] = fRec39[0];
			fRec38[1] = fRec38[0];
			fRec41[1] = fRec41[0];
			fRec40[1] = fRec40[0];
			fRec43[1] = fRec43[0];
			fRec42[1] = fRec42[0];
			fRec45[1] = fRec45[0];
			fRec44[1] = fRec44[0];
			fRec47[1] = fRec47[0];
			fRec46[1] = fRec46[0];
			fRec30[1] = fRec30[0];
			fRec28[1] = fRec28[0];
			fRec26[1] = fRec26[0];
			fRec24[1] = fRec24[0];
			
		}
		
	}

	
};


#include <stdio.h>
#include <string>
#include "m_pd.h"

#define sym(name) xsym(name)
#define xsym(name) #name
#define faust_setup(name) xfaust_setup(name)
#define xfaust_setup(name) name ## _tilde_setup(void)

// time for "active" toggle xfades in secs
#define XFADE_TIME 0.1f

static t_class *faust_class;

struct t_faust {
  t_object x_obj;
#ifdef __MINGW32__
  /* This seems to be necessary as some as yet undetermined Pd routine seems
     to write past the end of x_obj on Windows. */
  int fence; /* dummy field (not used) */
#endif
  mydsp *dsp;
  PdUI *ui;
  std::string *label;
  int active, xfade, n_xfade, rate, n_in, n_out;
  t_sample **inputs, **outputs, **buf;
  t_outlet *out;
  t_sample f;
};

static t_symbol *s_button, *s_checkbox, *s_vslider, *s_hslider, *s_nentry,
  *s_vbargraph, *s_hbargraph;

static inline void zero_samples(int k, int n, t_sample **out)
{
  for (int i = 0; i < k; i++)
#ifdef __STDC_IEC_559__
    /* IEC 559 a.k.a. IEEE 754 floats can be initialized faster like this */
    memset(out[i], 0, n*sizeof(t_sample));
#else
    for (int j = 0; j < n; j++)
      out[i][j] = 0.0f;
#endif
}

static inline void copy_samples(int k, int n, t_sample **out, t_sample **in)
{
  for (int i = 0; i < k; i++)
    memcpy(out[i], in[i], n*sizeof(t_sample));
}

static t_int *faust_perform(t_int *w)
{
  t_faust *x = (t_faust *)(w[1]);
  int n = (int)(w[2]);
  if (!x->dsp || !x->buf) return (w+3);
  AVOIDDENORMALS;
  if (x->xfade > 0) {
    float d = 1.0f/x->n_xfade, f = (x->xfade--)*d;
    d = d/n;
    x->dsp->compute(n, x->inputs, x->buf);
    if (x->active)
      if (x->n_in == x->n_out)
	/* xfade inputs -> buf */
	for (int j = 0; j < n; j++, f -= d)
	  for (int i = 0; i < x->n_out; i++)
	    x->outputs[i][j] = f*x->inputs[i][j]+(1.0f-f)*x->buf[i][j];
      else
	/* xfade 0 -> buf */
	for (int j = 0; j < n; j++, f -= d)
	  for (int i = 0; i < x->n_out; i++)
	    x->outputs[i][j] = (1.0f-f)*x->buf[i][j];
    else
      if (x->n_in == x->n_out)
	/* xfade buf -> inputs */
	for (int j = 0; j < n; j++, f -= d)
	  for (int i = 0; i < x->n_out; i++)
	    x->outputs[i][j] = f*x->buf[i][j]+(1.0f-f)*x->inputs[i][j];
      else
	/* xfade buf -> 0 */
	for (int j = 0; j < n; j++, f -= d)
	  for (int i = 0; i < x->n_out; i++)
	    x->outputs[i][j] = f*x->buf[i][j];
  } else if (x->active) {
    x->dsp->compute(n, x->inputs, x->buf);
    copy_samples(x->n_out, n, x->outputs, x->buf);
  } else if (x->n_in == x->n_out) {
    copy_samples(x->n_out, n, x->buf, x->inputs);
    copy_samples(x->n_out, n, x->outputs, x->buf);
  } else
    zero_samples(x->n_out, n, x->outputs);
  return (w+3);
}

static void faust_dsp(t_faust *x, t_signal **sp)
{
  int n = sp[0]->s_n, sr = (int)sp[0]->s_sr;
  if (x->rate <= 0) {
    /* default sample rate is whatever Pd tells us */
    PdUI *ui = x->ui;
    float *z = NULL;
    if (ui->nelems > 0 &&
	(z = (float*)malloc(ui->nelems*sizeof(float)))) {
      /* save the current control values */
      for (int i = 0; i < ui->nelems; i++)
	if (ui->elems[i].zone)
	  z[i] = *ui->elems[i].zone;
    }
    /* set the proper sample rate; this requires reinitializing the dsp */
    x->rate = sr;
    x->dsp->init(sr);
    if (z) {
      /* restore previous control values */
      for (int i = 0; i < ui->nelems; i++)
	if (ui->elems[i].zone)
	  *ui->elems[i].zone = z[i];
      free(z);
    }
  }
  if (n > 0)
    x->n_xfade = (int)(x->rate*XFADE_TIME/n);
  dsp_add(faust_perform, 2, x, n);
  for (int i = 0; i < x->n_in; i++)
    x->inputs[i] = sp[i+1]->s_vec;
  for (int i = 0; i < x->n_out; i++)
    x->outputs[i] = sp[x->n_in+i+1]->s_vec;
  if (x->buf != NULL)
    for (int i = 0; i < x->n_out; i++) {
      x->buf[i] = (t_sample*)malloc(n*sizeof(t_sample));
      if (x->buf[i] == NULL) {
	for (int j = 0; j < i; j++)
	  free(x->buf[j]);
	free(x->buf);
	x->buf = NULL;
	break;
      }
    }
}

static int pathcmp(const char *s, const char *t)
{
  int n = strlen(s), m = strlen(t);
  if (n == 0 || m == 0)
    return 0;
  else if (t[0] == '/')
    return strcmp(s, t);
  else if (n <= m || s[n-m-1] != '/')
    return strcmp(s+1, t);
  else
    return strcmp(s+n-m, t);
}

static void faust_any(t_faust *x, t_symbol *s, int argc, t_atom *argv)
{
  if (!x->dsp) return;
  PdUI *ui = x->ui;
  if (s == &s_bang) {
    for (int i = 0; i < ui->nelems; i++)
      if (ui->elems[i].label && ui->elems[i].zone) {
	t_atom args[6];
	t_symbol *_s;
	switch (ui->elems[i].type) {
	case UI_BUTTON:
	  _s = s_button;
	  break;
	case UI_CHECK_BUTTON:
	  _s = s_checkbox;
	  break;
	case UI_V_SLIDER:
	  _s = s_vslider;
	  break;
	case UI_H_SLIDER:
	  _s = s_hslider;
	  break;
	case UI_NUM_ENTRY:
	  _s = s_nentry;
	  break;
	case UI_V_BARGRAPH:
	  _s = s_vbargraph;
	  break;
	case UI_H_BARGRAPH:
	  _s = s_hbargraph;
	  break;
	default:
	  continue;
	}
	SETSYMBOL(&args[0], gensym(ui->elems[i].label));
	SETFLOAT(&args[1], *ui->elems[i].zone);
	SETFLOAT(&args[2], ui->elems[i].init);
	SETFLOAT(&args[3], ui->elems[i].min);
	SETFLOAT(&args[4], ui->elems[i].max);
	SETFLOAT(&args[5], ui->elems[i].step);
	outlet_anything(x->out, _s, 6, args);
      }
  } else {
    const char *label = s->s_name;
    int count = 0;
    for (int i = 0; i < ui->nelems; i++)
      if (ui->elems[i].label &&
	  pathcmp(ui->elems[i].label, label) == 0) {
	if (argc == 0) {
	  if (ui->elems[i].zone) {
	    t_atom arg;
	    SETFLOAT(&arg, *ui->elems[i].zone);
	    outlet_anything(x->out, gensym(ui->elems[i].label), 1, &arg);
	  }
	  ++count;
	} else if (argc == 1 &&
		   (argv[0].a_type == A_FLOAT ||
		    argv[0].a_type == A_DEFFLOAT) &&
		   ui->elems[i].zone) {
	  float f = atom_getfloat(argv);
	  *ui->elems[i].zone = f;
	  ++count;
	} else
	  pd_error(x, "[faust] %s: bad control argument: %s",
		   x->label->c_str(), label);
      }
    if (count == 0 && strcmp(label, "active") == 0) {
      if (argc == 0) {
	t_atom arg;
	SETFLOAT(&arg, (float)x->active);
	outlet_anything(x->out, gensym((char*)"active"), 1, &arg);
      } else if (argc == 1 &&
		 (argv[0].a_type == A_FLOAT ||
		  argv[0].a_type == A_DEFFLOAT)) {
	float f = atom_getfloat(argv);
	x->active = (int)f;
	x->xfade = x->n_xfade;
      }
    }
  }
}

static void faust_free(t_faust *x)
{
  if (x->label) delete x->label;
  if (x->dsp) delete x->dsp;
  if (x->ui) delete x->ui;
  if (x->inputs) free(x->inputs);
  if (x->outputs) free(x->outputs);
  if (x->buf) {
    for (int i = 0; i < x->n_out; i++)
      if (x->buf[i]) free(x->buf[i]);
    free(x->buf);
  }
}

static void *faust_new(t_symbol *s, int argc, t_atom *argv)
{
  t_faust *x = (t_faust*)pd_new(faust_class);
  int sr = -1;
  t_symbol *id = NULL;
  x->active = 1;
  for (int i = 0; i < argc; i++)
    if (argv[i].a_type == A_FLOAT || argv[i].a_type == A_DEFFLOAT)
      sr = (int)argv[i].a_w.w_float;
    else if (argv[i].a_type == A_SYMBOL || argv[i].a_type == A_DEFSYMBOL)
      id = argv[i].a_w.w_symbol;
  x->rate = sr;
  if (sr <= 0) sr = 44100;
  x->xfade = 0; x->n_xfade = (int)(sr*XFADE_TIME/64);
  x->inputs = x->outputs = x->buf = NULL;
    x->label = new std::string(sym(mydsp) "~");
  x->dsp = new mydsp();
  x->ui = new PdUI(sym(mydsp), id?id->s_name:NULL);
  if (!x->dsp || !x->ui || !x->label) goto error;
  if (id) {
    *x->label += " ";
    *x->label += id->s_name;
  }
  x->n_in = x->dsp->getNumInputs();
  x->n_out = x->dsp->getNumOutputs();
  if (x->n_in > 0)
    x->inputs = (t_sample**)malloc(x->n_in*sizeof(t_sample*));
  if (x->n_out > 0) {
    x->outputs = (t_sample**)malloc(x->n_out*sizeof(t_sample*));
    x->buf = (t_sample**)malloc(x->n_out*sizeof(t_sample*));
  }
  if ((x->n_in > 0 && x->inputs == NULL) ||
      (x->n_out > 0 && (x->outputs == NULL || x->buf == NULL)))
    goto error;
  for (int i = 0; i < x->n_out; i++)
    x->buf[i] = NULL;
  x->dsp->init(sr);
  x->dsp->buildUserInterface(x->ui);
  for (int i = 0; i < x->n_in; i++)
    inlet_new(&x->x_obj, &x->x_obj.ob_pd, &s_signal, &s_signal);
  x->out = outlet_new(&x->x_obj, 0);
  for (int i = 0; i < x->n_out; i++)
    outlet_new(&x->x_obj, &s_signal);
  return (void *)x;
 error:
  faust_free(x);
  x->dsp = NULL; x->ui = NULL;
  x->inputs = x->outputs = x->buf = NULL;
  return (void *)x;
}

extern "C" void faust_setup(mydsp)
{
  t_symbol *s = gensym(sym(mydsp) "~");
  faust_class =
    class_new(s, (t_newmethod)faust_new, (t_method)faust_free,
	      sizeof(t_faust), CLASS_DEFAULT,
	      A_GIMME, A_NULL);
  class_addmethod(faust_class, (t_method)faust_dsp, gensym((char*)"dsp"), A_NULL);
  class_addanything(faust_class, faust_any);
  class_addmethod(faust_class, nullfn, &s_signal, A_NULL);
  s_button = gensym((char*)"button");
  s_checkbox = gensym((char*)"checkbox");
  s_vslider = gensym((char*)"vslider");
  s_hslider = gensym((char*)"hslider");
  s_nentry = gensym((char*)"nentry");
  s_vbargraph = gensym((char*)"vbargraph");
  s_hbargraph = gensym((char*)"hbargraph");
  /* give some indication that we're loaded and ready to go */
  mydsp dsp = mydsp();
  post("[faust] %s: %d inputs, %d outputs", sym(mydsp) "~",
       dsp.getNumInputs(), dsp.getNumOutputs());
}

#endif
