/*
 * Copyright 2012 Sean Farley
 *
 * This guy called Sean wrote this sweet code. You are hereby granted
 * permission to do whatever you feel like doing with it on the understanding
 * that he's not responsible for anything that results from your use of this
 * sweet code.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <time.h>

#include "fast-hg-common.h"

void dirty(const char* reason) {
  /* we got here because the repo is modified in some way */
  PRINT("dirty repo reason: %s\n", reason);
  printf("*");
}

int parse_dirstate(const char* pwd) {
  FILE          *f        = NULL;
  char           filename[PATH_MAX+1];
  char           filepath[PATH_MAX+1];
  char           dirstate[PATH_MAX+1];
  size_t         i, bytes_read;
  unsigned char  status;             /* needs to be 8 bits */
  unsigned int   mode, size, length; /* each need to be 32 bits */
  time_t         mtime;              /* will only fill 32 bits by dirstate definition */
  mode_t         u;                  /* original mask of the current file or process */
  struct stat    filestat;

  sprintf(dirstate, "%s/.hg/dirstate", pwd);

  f = fopen(dirstate, "rb");
  if (!f) return -1;

  fseek(f, 40, SEEK_SET);       /* skip p1 and p2, each 20 bytes */

  u = umask(0);                 /* copied this from mercurial */
  umask(u);

  i = 0;
  while(!feof(f)) {
    bytes_read = 0;

    bytes_read = fread(&status, 1, 1, f);
    if (!bytes_read) break;     /* either an error happened or the end of file;
                                   either way we break */

    /* if the status is 'n' (normal) then we need to check the file's last
       modified time and size, so if not 'n' then we know it's definitely
       dirty */
    if (status != 'n') {
      dirty("above file's status is not 'n'");
      break;
    }

    fread(&mode, 4, 1, f);      /* file mode */
    fread(&size, 4, 1, f);      /* file's size */
    fread(&mtime, 4, 1, f);     /* file's last modified time */
    fread(&length, 4, 1, f);    /* length of the char array for the filename  */

     /* the dirstate file is written in big-endian, so we convert to network
        (big) endian here */
    mode   = htonl(mode) & 0777 & ~u; /* taken from mercurial/commands.py */
    size   = htonl(size);
    mtime  = htonl(mtime);
    length = htonl(length);


    PRINT("[%zu] status: %c\n", i, status);
    PRINT("[%zu] mode: %.3o\n", i, mode);
    PRINT("[%zu] size: %lld\n", i, (long long)size);
    PRINT("[%zu] mtime: %lld\n", i, (long long)mtime);
    PRINT("[%zu] length: %u\n", i, length);

    fread(filename, 1, length, f);
    filename[length] = 0;       /* let's go ahead and terminate the string */

    PRINT("[%zu] filename: %s\n\n", i, filename);

    sprintf(filepath, "%s/%s", pwd, filename);

    /* size needs to be >= 0 by definition of the dirstate */
    if (lstat(filepath, &filestat)) return -2;

    if (((mode ^ filestat.st_mode) & 0111 & ~u) && ((mode & S_IFLNK) != S_IFLNK)) {
      PRINT("[%zu] filestat.st_mode: %.3o\n", i, filestat.st_mode);
      dirty("above file's mode changed");
      break;
    } else if (mode && mtime != filestat.st_mtime && size != filestat.st_size) {
      /* check that mode != 000 since after updating, certain files are not updated in the
         dirstate? */
      PRINT("[%zu] filestate.st_mtime: %lld\n", i, (long long)filestat.st_mtime);
      PRINT("[%zu] filestate.st_size: %lld\n", i, filestat.st_size);
      dirty("above file's mtime and size both differ");
      break;
    }

    i++;
  }

  fclose(f);                    /* done with file reading */

  return 0;
}

int main() {
  return map(parse_dirstate);
}

