/*
 *  Copyright (C) 2007-8, see the file AUTHORS for copyright owners.
 *
 *  This program can be distributed under the terms of the GNU GPL v2,
 *  or any later version. See the file COPYING.
 */

#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <assert.h>
#include <sys/statvfs.h>
#include <time.h>

#include <ext2fs/ext2fs.h>
#include <ext2fs/ext2_fs.h>
#include "ext2fs.h"

#include <string.h>
#include <stddef.h>

#include "wipe_block.h"

#define ext2_err(rc, ...) \
	com_err("ext2fuse_dbg_msg", rc, __VA_ARGS__)

void init_ext2_stuff()
{
	initialize_ext2_error_table();
}

void fill_statbuf(ext2_ino_t ino, struct ext2_inode *inode,
		struct stat *st)
{
	/* st_dev */
	st->st_ino = ino;
	st->st_mode = inode->i_mode;
	st->st_nlink = inode->i_links_count;
	st->st_uid = inode->i_uid;	/* add in uid_high */
	st->st_gid = inode->i_gid;	/* add in gid_high */
	/* st_rdev */
	st->st_size = inode->i_size;
	st->st_blksize = EXT2_BLOCK_SIZE(fs->super);
	st->st_blocks = inode->i_blocks;

	/* We don't have to implement nanosecs, fs's which don't can return
	 * 0 here
	 */
	/* Using _POSIX_C_SOURCE might also work */
#ifdef __APPLE__
	st->st_atimespec.tv_sec = inode->i_atime;
	st->st_mtimespec.tv_sec = inode->i_mtime;
	st->st_ctimespec.tv_sec = inode->i_ctime;
	st->st_atimespec.tv_nsec = 0;
	st->st_mtimespec.tv_nsec = 0;
	st->st_ctimespec.tv_nsec = 0;
#else
	st->st_atime = inode->i_atime;
	st->st_mtime = inode->i_mtime;
	st->st_ctime = inode->i_ctime;
#ifdef __FreeBSD__
       st->st_atimespec.tv_nsec=0;
       st->st_mtimespec.tv_nsec=0;
       st->st_ctimespec.tv_nsec=0;
#else
       st->st_atim.tv_nsec = 0;
       st->st_mtim.tv_nsec = 0;
       st->st_ctim.tv_nsec = 0;
#endif
#endif
}

int read_inode(ext2_ino_t ino, struct ext2_inode *inode)
{
	int rc = ext2fs_read_inode(fs, ino, inode);
	if(rc)
	{
		ext2_err(rc, "while reading inode %u", ino);
		return 1;
	}
	return 0;
}

int write_inode(ext2_ino_t ino, struct ext2_inode * inode)
{
	int rc = ext2fs_write_inode(fs, ino, inode);
	if (rc)
	{
		ext2_err(rc, "while writing inode %u", ino);
		return 1;
	}
	return 0;
}

/*
 * Given a mode, return the ext2 file type
 */
int ext2_file_type(unsigned int mode)
{
	if(LINUX_S_ISREG(mode))
		return EXT2_FT_REG_FILE;

	if(LINUX_S_ISDIR(mode))
		return EXT2_FT_DIR;

	if(LINUX_S_ISCHR(mode))
		return EXT2_FT_CHRDEV;

	if(LINUX_S_ISBLK(mode))
		return EXT2_FT_BLKDEV;

	if(LINUX_S_ISLNK(mode))
		return EXT2_FT_SYMLINK;

	if(LINUX_S_ISFIFO(mode))
		return EXT2_FT_FIFO;

	if(LINUX_S_ISSOCK(mode))
		return EXT2_FT_SOCK;

	return 0;
}

int get_ino_by_name(ext2_ino_t parent, const char *name,
				ext2_ino_t *ino)
{
	return ext2fs_namei(fs, EXT2_ROOT_INO, parent, name, ino);
}

// Looking up a filename and reading it's inode. checks perms for parent,
// but not inode, as we don't need read for just the inode, but need search
// (execute) bit in parent
int do_lookup(perms_struct perms, ext2_ino_t parent,
			const char *name, ext2_ino_t *ino, struct ext2_inode *inode)
{
	errcode_t rc;

	// check list permission, is a dir, exists, for parent dir
	rc = read_inode(parent, inode);
	if(rc)
	{
		if (rc == EXT2_ET_FILE_NOT_FOUND)
			return ENOENT;
		ext2_err(rc, "while reading inode %u", *ino);
		return rc;
	}
	if (check_perms_in_inode(perms, inode, X_OK))
		return EACCES;

	rc = ext2fs_lookup(fs, parent, name, strlen(name), NULL, ino);
	if(rc)
	{
		ext2_err(rc, "while looking up \"%s\"", name);
		return ENOENT;
	}

	rc = read_inode(*ino, inode);
	if(rc)
	{
		ext2_err(rc, "while reading inode %u", *ino);
		return EIO;
	}

	return 0;
}

/* iterate the directory @ino and fillup the buffer @buf */
int do_dir_iterate(ext2_ino_t ino, int flags,
		int (*func)(struct ext2_dir_entry *, int, int,
		char *, void *), void *buf)
{
	errcode_t rc;

	rc = ext2fs_dir_iterate(fs, ino, flags, NULL, func, buf);
	if(rc)
	{
		ext2_err(rc, "while iterating inode %u", ino);
		return 1;
	}

	return 0;
}

// do_open never creates a file: do_create will always be called instead.
//
ext2_file_t do_open(perms_struct perms, ext2_ino_t ino,
					int flags)
{
	ext2_file_t efile;
	errcode_t rc;
	int newflags;

	// if O_TRUNC and O_RDONLY, correct to O_RDWR
	if (flags & O_TRUNC)
	{
		if (!(flags & (O_RDWR || O_WRONLY)))
			flags |= O_RDWR;
	}

	// check for write access
	if (flags & (O_WRONLY | O_RDWR))
		rc = check_perms(perms, ino, W_OK);
	// check for read access
	else
		rc = check_perms(perms, ino, R_OK);

	if (rc)
	{
		errno = rc;
		return NULL;
	}

	dbg("do_open (ino %u, flags 0%o)", ino, flags);
	// set flags bitmask, as it is actually in a completely different
	// format from the obvious, and only has two bits.
	newflags = (flags & O_CREAT) ? EXT2_FILE_CREATE : 0;
	newflags |= (flags & O_RDWR) ? EXT2_FILE_WRITE : 0;
	newflags |= (flags & O_WRONLY) ? EXT2_FILE_WRITE : 0;
	// TODO: do_open - permission checking for open flags needs checking
		
	rc = ext2fs_file_open(fs, ino, newflags, &efile);
	dbg("ext2fs_file_open() wrote ext2_file {flags 0%o, ino %d}",
		efile->flags, efile->ino);
	
	if ((flags & O_RDWR) && !(efile->flags & EXT2_FILE_WRITE))
	{
		efile->flags |= EXT2_FILE_WRITE;
		dbg("Danger! Danger! EXT2_FILE_WRITE wasn't set!");
	}
	
	if (rc)
	{
		ext2_err(rc, "while opening inode %u", ino);
		errno = rc;
		return NULL;
	}

	if (flags & O_TRUNC)
	{
		rc = do_ftruncate(efile, ino, 0);
		errno = rc;
		return efile;
	}

	// note that efile is just a ptr to something allocated by file_open
	return efile;
}

// do_create: ino and inode are unitialized stores for this routine to fill.
// note that mode gets passed straight into the inode struct, so checks for
// dodgy params to e.g. creat should go in the calling function.
// 
// order of ops:
// 	get inode number (new_inode)
// 	create the inode structure
// 	write it (save it)
// 	set inode bitmap
// 	link in dir
//
// In the worst case, we get an inode allocated but never linked, so only
// wasted space.
//
int do_create(perms_struct perms, ext2_ino_t parent,
			const char *name, mode_t mode,
			ext2_ino_t *ino, struct ext2_inode *inode, int filetype_in_dir)
{
	errcode_t rc;

	dbg("do_create(parent %u, name %s, mode %u, ino*, inode*)",
		parent, name, mode);

	// check we have write permission on parent
	rc = read_inode(parent, inode);
	if(rc)
	{
		ext2_err(rc, "while reading inode %u", *ino);
		return EIO;
	}
	if (check_perms_in_inode(perms, inode, W_OK))
		return EACCES;

	/* create a brand new inode */
	rc = ext2fs_new_inode(fs, parent, mode, 0, ino);
	if(rc)
	{
		if(EXT2_ET_INODE_ALLOC_FAIL)
			rc = ENOSPC;
		return rc;
	}

	/* double-check if the inode is already set */
	if(ext2fs_test_inode_bitmap(fs->inode_map, *ino))
		ext2_err(0, "Warning: inode already set");

	// Ready the inode for writing
	memset(inode, 0, sizeof(struct ext2_inode));

	if (set_perms(perms, inode))
		return EIO;
	inode->i_mode = mode;
	dbg("creating file with mode 0%o", inode->i_mode);

	inode->i_atime = inode->i_ctime = inode->i_mtime = time(NULL);
	inode->i_links_count = 1;
	inode->i_size = 0;

	// Write the inode
	rc = ext2fs_write_new_inode(fs, *ino, inode);
	if(rc)
	{
		ext2_err(rc, "while creating inode %u", *ino);
		return EIO;
	}

	// Update allocation statistics.
	ext2fs_inode_alloc_stats2(fs, *ino, +1, 0);

	// Link it in the directory
	rc = do_link(parent, name, *ino, filetype_in_dir);
	if(rc)
	{
		ext2_err(rc, "while linking \"%s\"", name);
		return rc;
	}

	return 0;
}

static int release_blocks_proc(ext2_filsys fs, blk_t *blocknr,
	int blockcnt EXT2FS_ATTR((unused)), void *private EXT2FS_ATTR((unused)))
{
	blk_t block;

	block = *blocknr;
	ext2fs_block_alloc_stats(fs, block, -1);
	if ((wipe_block_procedure)(block))
	{
		dbg("wipe_block %d failed, aborting", (int) block);
		exit(1);
	}
	return 0;
}

// POSIX says removing open files is not allowed; happily FUSE handles this
// case for us, by renaming to .fuse_hiddenXXX, then removing on unlink.
// See option "hard_remove".
//
// *inode needs to be an up-to-date copy of the inode, as we
// save it back to disk.
// Note that we leave deleted inodes in a perfect state,
// so testing etc is nice and simple
//
void kill_file_by_inode(ext2_ino_t ino, struct ext2_inode *inode)
{
	dbg ("kill_file_by_inode(ino %d, inode* %ld)", ino, (long) inode);

	// For effeciency do_unlink doesn't save a 0-link count for files
	// about to be deleted, so we do it here.
	inode->i_links_count = 0;
	// Set the "deletion time" in the inode, very friendly
	inode->i_dtime = time(NULL);
	if(write_inode(ino, inode))
		return;

	if(!ext2fs_inode_has_valid_blocks(inode))
		return;

	dbg ("deleting blocks for %d...", ino);
	ext2fs_block_iterate(fs, ino, 0, NULL, release_blocks_proc, NULL);
	ext2fs_inode_alloc_stats2(fs, ino, -1, LINUX_S_ISDIR(inode->i_mode));
	ext2fs_unmark_inode_bitmap(fs->inode_map, ino);
	ext2fs_mark_ib_dirty(fs);
	ext2fs_mark_bb_dirty(fs);
	ext2fs_mark_super_dirty(fs);
}

int do_link(ext2_ino_t parent, const char *name, ext2_ino_t ino,
			int filetype)
{
	errcode_t rc;

	rc = ext2fs_link(fs, parent, name, ino, filetype);
	if (rc == EXT2_ET_DIR_NO_SPACE)
	{
		rc = ext2fs_expand_dir(fs, parent);
		if(rc)
		{
			ext2_err(rc, "while expanding directory");
			return EIO;
		}
		rc = ext2fs_link(fs, parent, name, ino, filetype);
	}
	if (rc)
		return EIO;
	return 0;
}


// do_unlink - see do_unlink_on_ino for details...
int do_unlink(ext2_ino_t parent, const char *name, int links)
{
	errcode_t rc;
	ext2_ino_t ino;

	dbg("do_unlink (parent %d, name \"%s\", links %d)",
		parent, name, links);

	rc = ext2fs_namei(fs, EXT2_ROOT_INO, parent, name, &ino);
	if(rc)
	{
		ext2_err(rc, "while trying to resolve filename");
		return ENOENT;
	}

	return do_unlink_on_ino(parent, name, ino, links);
}

// do_unlink_on_ino: used for renaming and deleting files (NOT dirs)
// @links has the number (<= 0) to be added to the inode link count.
//
// Order of ops:
// 	read inode for link count & to check it's a file
// 	reduce link count in copy of inode (no change yet to actual inode)
// 	unlink from parent dir
// 	save altered inode unless...
// 	...link count = 0, in which case kill inode without bothering to
// 	   write new link count value (kill_file does it instead)
//
// 	This means in the worst case we get an unreachable inode with a +ve
// 	link count, as opposed to a reachable non-existant inode
//
int do_unlink_on_ino(ext2_ino_t parent, const char* name, ext2_ino_t ino,
	int links)
{
	errcode_t rc;
	struct ext2_inode inode;

	dbg("do_unlink_on_ino (parent %d, ino %d, links %d)",
		parent, (int) ino, links);

	if(read_inode(ino, &inode))
	{
		ext2_err(0, "while reading ino %u", ino);
		return EIO;
	}

	if(LINUX_S_ISDIR(inode.i_mode))
	{
		dbg("attempted to unlike a directory, oops!\n");
		ext2_err(0, "file is a directory");
		return ENOTDIR;
	}

	/* Assumption: @links is strictly < 1 */
	if (inode.i_links_count > 0)
		inode.i_links_count += links;

	// unlink from parent
	rc = ext2fs_unlink(fs, parent, name, ino, 0);
	if(rc)
		ext2_err(rc, "while unlinking %d", (int) ino);

	// then reduce link count by saving inode
	if (inode.i_links_count < 1)
	{
		kill_file_by_inode(ino, &inode);
	}
	else if (write_inode(ino, &inode))
	{
		ext2_err(rc, "while writing ino %d", (int) ino);
		return EIO;
	}


	return 0;
}


int do_read(struct ext2_file *fh, ext2_ino_t ino, size_t size, off_t off,
			char *buf, unsigned int *bytes)
{
	errcode_t rc;
	__u64 pos;
	dbg("do_read(ext_file {ino %d, flags %d (0%o)},\n\t"\
		"\tino %d, size\n"\
		"\t%d, off_t %d, buf, int* bytes)",
		(int) fh->ino, fh->flags, fh->flags, ino, size, (int) off);

	if(!buf)
		return ENOMEM;

	rc = ext2fs_file_llseek(fh, off, SEEK_SET, &pos);
	if (rc)
	{
		ext2_err(rc, "while seeking in %d by %d", (int) ino, (int) off);
		return EINVAL;
	}

	rc = ext2fs_file_read(fh, buf, size, bytes);
	if (rc)
	{
		ext2_err(rc, "while reading file %d", (int) ino);
		return(rc);
	}

	// TODO: updating atime on reads: implement "noatime" option
	fh->inode.i_atime = time(NULL);
	return ext2fs_write_inode(fs, ino, &fh->inode);
}

// do_write
// order of ops:
// 	write a (sparse) gap if required, by calling recursively
// 	seek to offset
// 	write the data
// 	set the increased file size
//
// Potential issue: crashing after the gap is written will give a file with
// a gap, but no data at the end. I don't think this is really much of an
// issue.
//
int do_write(struct ext2_file *fh, ext2_ino_t ino, const char *buf,
			size_t size, off_t off, unsigned int *bytes)
{
	errcode_t rc;
	__u64 pos;
	dbg("do_write(ext_file {ino %d, flags %d (0%o)},\n\t"\
		"\tino %d, buf, size\n"\
		"\t%d, off_t %d, int* bytes)",
		(int) fh->ino, fh->flags, fh->flags, ino, size, (int) off);


	// sparse file support!
	// posix, man, & gcc say you should be able to seek to after the end,
	// and then be able to read 0's from the gap.
	if (off > (fh->inode).i_size)
	{
		rc = do_ftruncate(fh, ino, off);
		if (rc)
			return rc;
	}
	
	// seek
	rc = ext2fs_file_llseek(fh, off, SEEK_SET, &pos);
	if (rc)
	{
		ext2_err(rc, "while seeking %d by %d", ino, (int) off);
		return EINVAL;
	}

	// write data
	dbg("ext2fs_file_write(fh, buf, size %d, bytes)", (int) size);
	rc = ext2fs_file_write(fh, buf, size, bytes);
	if (rc)
	{
		ext2_err(rc, "while writing file %d", ino);
		return EIO;
	}
	if ((size_t) (*bytes) != size) 
	{
		dbg ("ext2fs_file_write only wrote %d/%d bytes", *bytes,
			(int) size);
		return EIO;
	}
	
	// set file size...
	// if the end of new data is within the file, no increase
	if ((off + (off_t) size) <= (off_t) (fh->inode.i_size))
		return 0; 
	// we have already dealt with the sparse file case
	else if (off > (off_t) fh->inode.i_size)
	{
		dbg("internal error");
		return EIO;
	}
	rc = set_file_size(ino, &fh->inode, off + size);
	if (rc)
		dbg("set_file_size reported error");
	return rc;
}

int do_file_flush(struct ext2_file *fh)
{
	errcode_t rc;
	rc = ext2fs_file_flush(fh);
	return rc;
}

int do_file_close(struct ext2_file *fh)
{
	int rc;
	rc = ext2fs_file_close(fh);
	return rc;	
}

// This is essentially a wrapper for ext2fs_mkdir, which does the grunt
// work of:
// 	picking an inode #
// 	picking a data block #
// 	creating the directory data block structure
// 	creating the inode structure
// 	allocating inode # and block # in bitmasks
// 	writing the structures to disk
// 	linking
// 	inc. parent's link count
//
int do_mkdir(perms_struct perms, ext2_ino_t parent,
	const char *name, mode_t mode, ext2_ino_t *ino)
{
	errcode_t rc;

	dbg("do_mkdir(parent %u, name %s, mode 0%o, ino*)",
		parent, name, mode);

try_again:
	dbg("calling ext2fs_mkdir(fs, parent, 0, name)");
	rc = new_ext2fs_mkdir(perms, parent, name);
	if (rc == EXT2_ET_DIR_NO_SPACE)
	{
		dbg("expanding dir...");
		rc = ext2fs_expand_dir(fs, parent);
		if (rc)
		{
			ext2_err(rc, "while expanding directory");
			return ENOSPC;
		}
		goto try_again;
	}

	if (rc)
	{
		/*if(EXT2_ET_INODE_ALLOC_FAIL)
			rc = ENOSPC;*/
		ext2_err(rc, "while creating \"%s\", rc=%d", name, (int) rc);
		return rc;
	}

	dbg("calling ext2fs_lookup(... name %s ...)", name);
	rc = ext2fs_lookup(fs, parent, name, strlen(name), NULL, ino);
	if (rc)
	{
		ext2_err(rc, "while looking up \"%s\"", name);
		return ENOENT;
	}

	ext2fs_mark_ib_dirty(fs);
	ext2fs_mark_super_dirty(fs);
	return 0;
}

size_t fs_size(void)
{
	if (!fs)
		return 0;

	return fs->super->s_free_blocks_count;
}

int test_root(int a, int b)
{
	int num = b;
	while (a > num)
		num *= b;
	return num == a;
}

int ext2_group_sparse(int group)
{
	if (group <= 1)
		return 1;
	return (test_root(group, 3) || test_root(group, 5) || test_root(group, 7));
}

/**
 * Check if the superblock is used in this group.
 * 0x0001 is a mask, aka. EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER
 */
int ext2_bg_has_super(int group)
{
	if( (fs->super->s_feature_ro_compat & 0x0001) && !(ext2_group_sparse(group)) )
		return 0;
	return 1;
}

int do_statvfs(struct statvfs *stbuf)
{
	unsigned long overhead, groups_count, db_count, itb_per_group;
	int i;

	if(!fs || !stbuf)
		return 1;

	groups_count = (fs->super->s_blocks_count - fs->super->s_first_data_block-1)
			/ EXT2_BLOCKS_PER_GROUP(fs->super) + 1;
	db_count = (groups_count + EXT2_DESC_PER_BLOCK(fs->super) - 1)
			/ EXT2_DESC_PER_BLOCK(fs->super);
	itb_per_group = EXT2_INODES_PER_GROUP(fs->super) / EXT2_INODES_PER_BLOCK(fs->super);
	overhead = fs->super->s_first_data_block;
	for (i = 0; i < groups_count; i++)
	{
		if( ext2_bg_has_super(i))
			overhead += 1 + db_count;
	}
	overhead += groups_count * (2 + itb_per_group);
	stbuf->f_bsize = EXT2_BLOCK_SIZE(fs->super);
	stbuf->f_frsize = EXT2_FRAG_SIZE(fs->super);
	stbuf->f_blocks = fs->super->s_blocks_count - overhead;
	stbuf->f_bfree = fs->super->s_free_blocks_count;

	if (stbuf->f_bfree>=fs->super->s_r_blocks_count)
		stbuf->f_bavail = fs->super->s_free_blocks_count-fs->super->s_r_blocks_count;
	else
		stbuf->f_bavail = 0;
		
	stbuf->f_files = fs->super->s_inodes_count;
	stbuf->f_ffree = stbuf->f_favail = fs->super->s_free_inodes_count;
	stbuf->f_fsid = 0;
	stbuf->f_flag = 0;
	stbuf->f_namemax = EXT2_NAME_LEN;

	return 0;
}

struct rd_struct
{
	ext2_ino_t parent;
	int empty;
};

static int rmdir_proc(ext2_ino_t dir EXT2FS_ATTR((unused)),
		int entry EXT2FS_ATTR((unused)), struct ext2_dir_entry *dirent,
		int offset EXT2FS_ATTR((unused)),
		int blocksize EXT2FS_ATTR((unused)),
		char *buf EXT2FS_ATTR((unused)), void *private)
{
	struct rd_struct *rds = (struct rd_struct *) private;

	dbg("rmdir_proc: iterating over entry %10s", dirent->name);

	if (dirent->inode == 0)
		return 0;
	if (((dirent->name_len & 0xFF) == 1) && (dirent->name[0] == '.'))
		return 0;
	if (((dirent->name_len & 0xFF) == 2) && (dirent->name[0] == '.') &&
	    (dirent->name[1] == '.'))
	{
		rds->parent = dirent->inode;
		return 0;
	}
	rds->empty = 0;
	return 0;
}

int do_rmdir(perms_struct perms, ext2_ino_t parent, const char* name)
{
	errcode_t rc;
	ext2_ino_t ino;

	rc = ext2fs_namei(fs, EXT2_ROOT_INO, parent, name, &ino);
	if (rc)
	{
		ext2_err(rc, "while trying to resolve filename");
		return ENOENT;
	}
	return do_rmdir_on_ino(perms, parent, name, ino);
}


int do_rmdir_on_ino(perms_struct perms, ext2_ino_t parent, const char* name,
		ext2_ino_t ino)
{
	errcode_t rc;
	struct ext2_inode inode;
	struct rd_struct rds;

	if (read_inode(ino, &inode))
	{
		ext2_err(0, "while reading ino %u", ino);
		return EIO;
	}
	// need write access to dir
	if (check_perms_in_inode(perms, &inode, W_OK))
		return EACCES;

	if (!LINUX_S_ISDIR(inode.i_mode))
	{
		ext2_err(0, "file is not a directory");
		return ENOTDIR;
	}

	rds.parent = 0;
	rds.empty = 1;

	rc = ext2fs_dir_iterate2(fs, ino, 0, 0, rmdir_proc, &rds);
	if (rc)
	{
		ext2_err(rc, "while iterating over directory");
		return EIO;
	}

	if (rds.empty == 0)
	{
		ext2_err(0, "directory not empty");
		return ENOTEMPTY;
	}

	// Used to set link count to 0, but this is now done
	// in kill_file_by_inode. Better resiliance this way too.

	// unlink from parent
	rc = ext2fs_unlink(fs, parent, name, ino, 0);
	if(rc)
	{
		ext2_err(rc, "while unlinking ino %d", (int) ino);
		return EIO;
	}

	// remove directory file, freeing blocks
	kill_file_by_inode(ino, &inode);


	// decrease link count in the parent
	if(rds.parent)
	{
		if((rc = read_inode(parent, &inode)))
			return rc;
		if(inode.i_links_count > 1)
			inode.i_links_count--;
		if((rc = write_inode(parent, &inode)))
			return rc;
	}
	else
	{
		dbg("deleted dir has no parent?? what is going on here? dir=%d",
			(int) ino);
	}

	return 0;
}

