/*

    Euchre - a free as in freedom and as in beer version of the 
             euchre card game
  
    Copyright 2002 C Nathan Buckles (nbuckles@bigfoot.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "Debug.hpp"
#include "Hand.hpp"

Hand::Hand() {}
Hand::Hand(const Card& c1, const Card& c2,
	   const Card& c3, const Card& c4,
	   const Card& c5)
{
  itsCards[0] = c1;
  itsCards[1] = c2;
  itsCards[2] = c3;
  itsCards[3] = c4;
  itsCards[4] = c5;
}

Hand::~Hand() {}

const Card& Hand::getCard(int index) const {
  if (index < 0 || index >= NUM_CARDS) {
    index = 0;
  }

  return itsCards[index];
}

Card Hand::removeCard(int index) {
  Card t = itsCards[index];

  itsCards[index].setSuit(Card::NoSuit);
  itsCards[index].setNumber(Card::NoNumber);

  str = getCard(0).getName();
  for (int i = 1; i < Hand::NUM_CARDS; i++) {
    str += ", ";
    str += getCard(i).getName();
  }

  return t;
}

void Hand::removeCard(const Card& card) {
  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (itsCards[i] == card) {
      removeCard(i);
      return;
    }
  }
}

void Hand::setCard(int index, const Card& card) {
  if (index < 0 || index >= NUM_CARDS) {
    index = 0;
  }

  itsCards[index] = card;

  str = getCard(0).getName();
  for (int i = 1; i < Hand::NUM_CARDS; i++) {
    str += ", ";
    str += getCard(i).getName();
  }
}

void Hand::replaceCard(const Card& oldCard, const Card& newCard) {
  LOG("replace\n");
  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (itsCards[i] == oldCard) {
      itsCards[i] = newCard;
      break;
    }
  }

  LOG(" replacing %c%c with %c%c\n", Card::getNumberChar(oldCard.getNumber()), oldCard.getSuitChar(),
      Card::getNumberChar(newCard.getNumber()), newCard.getSuitChar());

  str = getCard(0).getName();
  for (int i = 1; i < Hand::NUM_CARDS; i++) {
    str += ", ";
    str += getCard(i).getName();
  }
}

int Hand::contains(Card::Suit suit, Card::Suit trump) const {
  if (trump == Card::NoSuit) {
    trump = Common::getTrump();
  }
  
  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (itsCards[i].isSuit(suit, trump)) {
      return 1;
    }
  }

  return 0;
}

int Hand::count(Card::Suit suit, Card::Suit trump) const {
  int count = 0;

  if (trump == Card::NoSuit) {
    trump = Common::getTrump();
  }
  
  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (itsCards[i].isSuit(suit, trump)) {
      count++;
    }
  }

  return count;
}

int Hand::getValue(const Card::Suit aTrump) const {
  int total = 0;

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    total += itsCards[i].getValue(aTrump);
  }

  return total;
}

int Hand::cardsLeft() const {
  int ret = 0;

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (itsCards[i].getSuit() != Card::NoSuit) {
      ret++;
    }
  }

  return ret;
}

Card Hand::getBestCard() const {
  Card ret;

  int index = getBestCardIndex();
  if (index != -1) {
    ret = itsCards[index];
  }

  return ret;
}

Card Hand::getWorstCard() const {
  Card ret;

  int index = getWorstCardIndex();
  if (index != -1) {
    ret = itsCards[index];
  }

  return ret;
}

int Hand::getBestCardIndex() const {
  int ret = -1;
  int tmp_score;
  int max_score = -1;

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    /* make sure we don't count an empty card */
    if (itsCards[i].getSuit() != Card::NoSuit) {
      tmp_score = itsCards[i].getValue();
      if (tmp_score > max_score) {
	ret       = i;
	max_score = tmp_score;
      }
    }
  }

  return ret;
}

int Hand::getWorstCardIndex() const {
  int ret = -1;
  int tmp_score;
  int min_score = (Card::JackTrumpVal + 1);

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    /* make sure we don't count an empty card */
    if (itsCards[i].getSuit() != Card::NoSuit) {
      tmp_score = itsCards[i].getValue();
      if (tmp_score < min_score) {
	ret       = i;
	min_score = tmp_score;
      }
    }
  }

  return ret;
}

Card Hand::getBestCard(const Card::Suit suit) const {
  Card ret;

  int index = getBestCardIndex(suit);
  if (index != -1) {
    ret = itsCards[index];
  }

  return ret;
}

Card Hand::getWorstCard(const Card::Suit suit) const {
  Card ret;

  int index = getWorstCardIndex(suit);
  if (index != -1) {
    ret = itsCards[index];
  }

  return ret;
}

int Hand::getBestCardIndex(const Card::Suit suit) const {
  int  ret = -1;
  int  tmp_score;
  int  max_score = -1;

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (itsCards[i].isSuit(suit)) {
      tmp_score = itsCards[i].getValue();
      if (tmp_score > max_score) {
	ret       = i;
	max_score = tmp_score;
      }
    }
  }

  return ret;
}

int Hand::getWorstCardIndex(const Card::Suit suit) const {
  int  ret = -1;
  int  tmp_score;
  int  min_score = (Card::JackTrumpVal + 1);

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (itsCards[i].isSuit(suit)) {
      tmp_score = itsCards[i].getValue();
      if (tmp_score < min_score) {
	ret       = i;
	min_score = tmp_score;
      }
    }
  }

  return ret;
}

Card Hand::getBestNonTrump() const {
  Card       ret;
  int        tmp_score;
  int        max_score = -1;
  Card::Suit trump     = Common::getTrump();

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (! itsCards[i].isSuit(trump)) {
      tmp_score = itsCards[i].getValue();
      if (tmp_score > max_score) {
	ret       = itsCards[i];
	max_score = tmp_score;
      }
    }
  }

  return ret;
}

Card Hand::getWorstNonTrump() const {
  Card       ret;
  int        tmp_score;
  int        min_score = (Card::JackTrumpVal + 1);
  Card::Suit trump     = Common::getTrump();

  for (int i = 0; i < Hand::NUM_CARDS; i++) {
    if (! itsCards[i].isSuit(trump)) {
      tmp_score = itsCards[i].getValue();
      if (tmp_score < min_score) {
	ret       = itsCards[i];
	min_score = tmp_score;
      }
    }
  }

  return ret;
}

Card Hand::getJustBetterCard(const Card& card) const {
  unsigned int points_to_beat = card.getValue();

  Card ret;
  int  my_value = (Card::JackTrumpVal + 1);
  
  for (int i = 0; i < NUM_CARDS; i++) {

    /* if this card is the same suit or is trump */
    if (itsCards[i].getAdjSuit() == card.getAdjSuit() ||
	itsCards[i].getAdjSuit() == Common::getTrump()) {

      /* and it's value is more than the passed in card */
      if (itsCards[i].getValue() > points_to_beat) {

	/* and it's the lowest value qualifier we've seen so far */
	if (itsCards[i].getValue() < my_value) {
	  my_value = itsCards[i].getValue();
	  ret      = itsCards[i];
	}
      }
    }
  }

  return ret;
}

const char* Hand::getName() const {
  return str.c_str();
}

