/*******************************************************************************
* E.S.O. - VLT project
*
* pafchkChecksum.c
*
* who       when        what
* --------  ----------  --------------------------------------------------------
* pbaksai   2013/12/30  CCB-000579: FIMS support for 64 bits.
*                       - Made more compatible outside VLT
* pbaksai   2012-09-19  PPRS-24729: Increased record buffer again. 2048 this
*                       time.
* mpruemm   2009-06-23  Fix rcsId for VLT2009.
* pbaksai   2007-11-30  Increased record buffer; overflow destroyes call stack
*                       in pafchkFileCopy if fromFile has long strings
* rschmutz  2004-09-05  Accept PAF.CHCK.CHECKSUM lines without ending ';'.
* rschmutz  2004-07-15  vltPort.h replaced with _GNU_SOURCE.
* rschmutz  2004-07-15  tmpnam replaced with snprintf+getpid (as before).
* rschmutz  2004-07-11  Updated to support pafchkCreate.
* lsanzana  2004-06-06  Created
*
********************************************************************************
*   NAME
*	pafchkChecksumCreate, pafchkChecksumVerify - PAF checksum utilities
* 
*   SYNOPSIS
*	int pafchkChecksumCreate(const char* filename, const char* progname,
*			         int flags)
*	int pafchkChecksumVerify(const char *filename)
* 
*   DESCRIPTION:
*	Functions to be used by the off-line tools to add o verify the 
*	checksum of a PAF file.
*
*	pafchkChecksumCreate() adds a checksum to the file filename,
*	placing it in keyword PAF.CHCK.CHECKSUM.
*	The progname is placed in keyword PAF.CHCK.NAME.
*	Keyword PAF.CHCK.DAYTIM is normally updated with the current
*	date and time, unless the flag is set to PAFCHK_FLAG_NO_TIMESTAMP.
*	In this case, a fixed date is used (this is useful for tat tests).
*	
*	pafchkChecksumVerify() verifies the checksum stored in file filename.
*
*   RETURN VALUES
*	pafchkChecksumCreate() returns 0 (SUCCESS) or 1 (FAILURE).
*
*	pafchkCheckSumVerify() returns one of the following values:
*	 0 (PAFCHK_RESULT_OK)            : Checksum is correct.
*	 1 (PAFCHK_RESULT_ERROR)	 : Error while accessing <filename>.
*	 2 (PAFCHK_RESULT_BAD_CHECKSUM)  : Checksum is incorrect.
*	 3 (PAFCHK_RESULT_NO_CHECKSUM)   : No checksum found.
*
*   CAUTIONS 
*
*   EXAMPLES
*
*   SEE ALSO
*	pafchkCreate(1) pafchkVerify(1)
*
*   BUGS   
* 
*-------------------------------------------------------------------------------
*/

#include "vltAttrib.h"
#define LINUX
#ifndef __cplusplus
#define _GNU_SOURCE
#endif

#include <stddef.h>
#include <sys/types.h>
#include <sys/time.h>
#include "vltPortGeneral.h"

ATTRIBUTE_UNUSED static const char *rcsId="@(#) $Id: pafchkChecksum.c 250921 2014-01-26 22:50:54Z pbaksai $"; 

/* 
 * System Headers 
 */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>

/*
 * Local Headers 
 */
#include "zlib.h"
extern uLong crc32(uLong crc,  const Bytef *buf, uInt len);
#include "pafchk.h"

/*
 * Local Defines
 */
#define SUCCESS 0
#define FAILURE 1


/************************************************************************* 
 * pafchkValueSet - store a value in line
 *
 * The 1. argument (token) specifies the value that has to be stored
 * (correctly formatted) in the current line (2. argument).
 * The rest of the arguments contain the values to be stored.
 * 
 * token      : name, date or checksum selector.
 * line       : buffer receiving the formatted value.
 * checksum   : new PAF.CHCK.CHECKSUM value (token = 2)
 * progname   : PAF.CHCK.NAME value (token = 3)
 *
 */
static int pafchkValueSet(
    int 	  token,
    char 	  line[2048],
    unsigned long checksum,
    const char*   progname)
{
    char timeString[32];           /* Formatted current time, ESO standard */
    struct tm* gmTimePtr;          /* Used to get current time */
    time_t fileTime;               /* Time used in file name   */
    char *p1;
 
    /*= 1. search beginning of keyword value */
    p1 = strchr(line, '"');
    if (p1 == NULL)
	{
	fprintf(stderr, "Syntax error in file: no value found (%s).\n", line);
	return FAILURE;
	}

    /*= 2. process according to the current token */
    switch (token) 
	{
	/* empty PAF.CHCK.CHECKSUM */
	case 1:
	    /*
	     * No ending ';' is actually not quite correct,
	     * but as this is common, consider this case.
	     */
	    if (strchr(p1 + 1, ';'))
	        strcpy(p1, "\"\";");
	    else
	        strcpy(p1, "\"\"");
	    break;

	/* new PAF.CHCK.CHECKSUM */
	case 2:
	    if (strchr(p1 + 1, ';'))
	        sprintf(p1, "\"%lu\";", checksum);
	    else
	        sprintf(p1, "\"%lu\"", checksum);
	    break;

	/* PAF.CHCK.NAME */
	case 3:
	    sprintf(p1, "\"%s\";", progname);
	    break;

	/* fixed PAF.CHCK.DAYTIM */
	case 4:
	    strcpy(p1, "\"2000-01-01T00:00:00\";");
	    break;

	/* new PAF.CHCK.DAYTIM */
	case 5:
	    fileTime  = time(0);
	    gmTimePtr = gmtime(&fileTime);
	    sprintf(timeString,"\"%4.4d-%2.2d-%2.2dT%2.2d:%2.2d:%2.2d\";",
	            gmTimePtr->tm_year + 1900, gmTimePtr->tm_mon + 1, gmTimePtr->tm_mday,
		    gmTimePtr->tm_hour, gmTimePtr->tm_min, gmTimePtr->tm_sec);
	    strcpy(p1, timeString);
	    break;

	}

    return SUCCESS;
}


/************************************************************************* 
 * pafchkFileCopy
 *
 * pafchkFileCopy is called to either:
 *  1) for checksum verify: copy the PAF file, removing the PAF checksum
 *     to allow to calculate the file checksum.
 *  2) for checksum create: copy the PAF file, removing the PAF checksum,
 *     but updating PAF.CHCK.NAME and PAF.CHCK.DAYTIM.
 *  3) for checksum create: copy the PAF file, storing the new PAF checksum.
 *
 * Action 1 is done when arg. checksum is zero and arg. progname is NULL,
 * Action 2 is done when arg. checksum is zero and arg. progname is not NULL,
 * Action 3 is done when arg. checksum is not zero.
 *  
 * fromFile   : the name of the file to be copied.
 * toFile     : the name of the new file
 * checksum   : zero (for actions 1 and 2), or the new checksum (action 3)
 * progname   : PAF.CHCK.NAME value (for action 2)
 * noTimestamp: for PAF.CHCK.DAYTIME value:
 *		 0: store current timestamp
 *		 1: store a dummy timestamp (used for testing only)
 *
 */
static int pafchkFileCopy(
    const char*	  fromFile,
    const char*	  toFile,
    unsigned long checksum,
    const char*   progname,
    int           noTimestamp)
{
    int   status;
    FILE* fromFh;
    FILE* toFh;
    char  record[2048];
    int   token;
    int   action;

    status = SUCCESS;
    fromFh = NULL;
    toFh   = NULL;
    token  = 0;
    action = (checksum != 0) ? 3 : ((progname) ? 2 : 1);

    /* if error: break */
    do {
    	/*= 1. open files */
        if ((fromFh = fopen(fromFile, "r")) == NULL) 
	    {
	    perror("ERROR");
	    status = FAILURE;
	    break;
	    }
	if ((toFh = fopen(toFile, "w")) == NULL)   
	    {
	    perror("ERROR");
	    status = FAILURE;
	    break;
	    }

	/*= 2. while copying file ... */
	while (! feof(fromFh)) 
	    {
	    /*= 3. skip over empty tokens */
	    if ((fscanf(fromFh, "%[^' ']", record) == 0) &&
                (fscanf(fromFh, "%[^\n]",  record) == 0))
		continue;

	    /*= 4. if no token: search next token ... */
	    if (token == 0) 
		{
		fprintf(toFh, "%s", record);
		if (action != 3 && !strcmp(record,"\nPAF.CHCK.CHECKSUM"))  
		    token = 1;		     
		else if (action == 3 && !strcmp(record,"\nPAF.CHCK.CHECKSUM")) 
		    token = 2;		     
		else if (action == 2 && !strcmp(record,"\nPAF.CHCK.NAME"))
		    token = 3;
		else if (action == 2 && !strcmp(record,"\nPAF.CHCK.DAYTIM"))
		    token = (noTimestamp) ? 4 : 5;
		}

	    /*= 5. ... or if token found: set it's value */
	    else 
		{
		if (pafchkValueSet(token, record, checksum, progname) != 0)
		   {
		   status = FAILURE;
		   break;
		   }
		fprintf(toFh, "%s", record);
		token = 0;
		} 

	    } /* while copying file */
	} 
    while (0); /* if error */

    /*= 6. close files */
    if (fromFh != NULL)
    	fclose(fromFh);
    if (toFh != NULL)
    	{
    	if (fclose(toFh) != 0)
	    {
	    perror("ERROR");
	    status = FAILURE;
	    }
	}

    return status;         
}    


/************************************************************************* 
 * pafchkChecksumGet - retrieves the checksum stored inside a PAF file
 *
 * filename: PAF filename 
 * checksum: pointer to store the found file checksum
 *
 */
static int pafchkChecksumGet(
    const char    *filename,
    unsigned long *checksum)
{
    FILE* fh;
    char record[2048];

    *checksum = 0L;

    /*= 1. open PAF file */
    if ((fh = fopen(filename, "r")) == NULL)
    	{
	perror("ERROR");
	return FAILURE;
	}

    /*= 2. search checksum */
    while (! feof(fh))
        {
        if (fscanf(fh, "%s", record) == 0)
	    continue;
        if (!strcmp(record, "PAF.CHCK.CHECKSUM"))
            {
            fscanf(fh, "%s", record);
            sscanf(record, "\"%lu\";", checksum);
	    break;
            }
        }

    /*= 3. close PAF file */
    fclose(fh);

    return SUCCESS;
}

/************************************************************************* 
 * pafchkChecksumCalc - calculates a PAF file checksum
 *
 * To calculate the checksum of a PAF file, the original PAF file
 * is first copied, making sure that the PAF.CHCK.CHECKSUM keyword
 * is left empty.  Then the checksum of this file is calculated.
 *
 * filename: PAF filename 
 * checksum: pointer to store the calculated file checksum
 *
 */
#define ONEMEGA (1024*1024)

static int pafchkChecksumCalc(
    const char*	   filename,
    unsigned long* checksum)
{
    char    tmpName[32];
    int	    fd;
    char    buffer[ONEMEGA];
    ssize_t len = 0;

    *checksum = 0L;

    /*= 1. create a PAF file with an empty checksum */
    snprintf(tmpName, sizeof(tmpName), "/tmp/pafchk%d.tmp", getpid());
    if (pafchkFileCopy(filename, tmpName, 0L, NULL, 0) != SUCCESS)
    	return FAILURE; 

    /*= 2. read the file and calculate the checksum */
    fd = open(tmpName, O_RDONLY);
    if (fd < 0) 
	{
	perror("ERROR");
	return FAILURE;
	}
    len = read(fd, buffer, ONEMEGA);
    if (unlink(tmpName) != 0)
	{
	perror("ERROR");
	return FAILURE;
	}
    *checksum = crc32(0, (Bytef*)buffer, len);
    
    return SUCCESS;
}


/************************************************************************* 
 * pafchkChecksumCreate - create and store a PAF checksum
 *
 * Creates and stores a checksum in the specified PAF file.
 *
 * filename: PAF filename
 * progname: value stored in keyword PAF.CHCK.NAME.
 * flags:    0: store the current time in PAF.CHCK.DAYTIM.
 *	     1: store a dummy timestamp in PAF.CHCK.DAYTIM.
 *
 */

int pafchkChecksumCreate(
    const char* filename,
    const char* progname,
    int         flags)
{
    int 	  status = SUCCESS;
    int		  noTimestamp;
    char	  tmpName[32];
    unsigned long checksum;
    int		  fd;
    char	  buffer[ONEMEGA];
    ssize_t	  len = 0;

    /*= 1. create a temporary PAF file with an empty checksum */
    noTimestamp = (flags & PAFCHK_FLAG_NO_TIMESTAMP);
    snprintf(tmpName, sizeof(tmpName), "/tmp/pafchk%d.tmp", getpid());
    if (pafchkFileCopy(filename, tmpName, 0L, progname, noTimestamp) != SUCCESS)
    	return FAILURE; 

    /*= 2. read the file and calculate the checksum */
    fd = open(tmpName, O_RDONLY);
    if (fd < 0) 
	{
	perror("ERROR");
	return FAILURE;
	}
    len = read(fd, buffer, ONEMEGA);
    checksum = crc32(0, (Bytef*)buffer, len);

    /*= 3. insert the found checksum in the PAF file */
    status = pafchkFileCopy(tmpName, filename, checksum, progname, noTimestamp);

    /*= 4. delete temporary PAF file */
    if (unlink(tmpName) != 0)
	{
	perror("ERROR");
	return FAILURE;
	}

    return status;
}


/************************************************************************* 
 * pafchkChecksumVerify - verify a PAF checksum
 *
 * Verifies the checksum stored in the specified PAF file.
 *
 * filename: PAF filename.
 *
 * Returns:
 *  0 (PAFCHK_RESULT_OK)            : Checksum is correct.
 *  1 (PAFCHK_RESULT_ERROR)         : Error while accessing <filename>.
 *  2 (PAFCHK_RESULT_BAD_CHECKSUM)  : Checksum is incorrect.
 *  3 (PAFCHK_RESULT_NO_CHECKSUM)   : No checksum found.
 *
 */
int pafchkChecksumVerify(
    const char *filename)
{
    int		  status;
    unsigned long fileChecksum;
    unsigned long calcChecksum;

    /*= 1. retrieve stored file checksum */
    if (pafchkChecksumGet(filename, &fileChecksum) != 0)
    	return PAFCHK_RESULT_ERROR;

    /*= 2. calculate file checksum */
    if (pafchkChecksumCalc(filename, &calcChecksum) != 0)
    	return PAFCHK_RESULT_ERROR;

    /*= 3. compare checksums */
    if (fileChecksum == 0)
        status = PAFCHK_RESULT_NO_CHECKSUM;
    else
        status = (fileChecksum == calcChecksum) ?
	    	 PAFCHK_RESULT_OK : PAFCHK_RESULT_BAD_CHECKSUM;

    return status;
}
    
/*___oOo___*/
