/*                              s d s t c l i n i t. c
  
 * Module name:
        sdstclinit.c
  
 * Function:
        Impelments the standalone (non-DRAMA) SDS Tcl interface.
  
 * Description:
        Provides the function Sds_Init(), which adds SDS Tcl commands
        to Tcl.
        
  
 * Language:
      C
  
 * Support: Tony Farrell, AAO
  
 * History:
      03-Nov-1998 - TJF - Original version (dtclsds.c)
      16-Aug-2001 - KS  - Added Ers.h.
      12-Feb-2002 - KS  - Include strings.h and string.h to prevent compilation
                          warnings on some systems.

  
 *  Copyright (c) Anglo-Australian Telescope Board, 1998.
    Not to be used for commercial purposes without AATB permission.

  
 * Sccs Id:     sdstclinit.c, Release 1.2, 02/18/02
  
 */

#include <strings.h>
#include <string.h>
#include "sdstcl.h"
#include "arg_err.h"
#include "Ers.h"

/*
 *  GetErrorString -> simple routine which removes any dependency on Mess
 *  for returning error strings for each known SDS and ARG error code.
 */
static SDSCONST char * GetErrorString(int error)
{
    switch (error)
    {
    case SDS__BADID:
        return ("SDS:Illegal Identifier");
    case SDS__NOMEM:
        return ("SDS:Insufficient Memory");
    case SDS__NOTSTRUCT:
        return ("SDS:Object is not a structure");
    case SDS__NOITEM:
        return ("SDS:No such item");
    case SDS__NOTPRIM:
        return ("SDS:Object is not a primitive");
    case SDS__TOOLONG:
        return ("SDS:Data too long");
    case SDS__UNDEFINED:
        return ("SDS:Data value is undefined");
    case SDS__LONGNAME:
        return ("SDS:Object name is too long");
    case SDS__EXTRA:
        return ("SDS:Too much extra data");
    case SDS__INVCODE:
        return ("SDS:Invalid type cod");
    case SDS__INVDIMS:
        return ("SDS:Invalid dimensions");
    case SDS__NOTARRAY:
        return ("SDS:Not a structure array");
    case SDS__INDEXERR:
        return ("SDS:Indices invalid");
    case SDS__ILLDEL:
        return ("SDS:Object cannot be deleted");
    case SDS__NOTSDS:
        return ("SDS:Not a valid SDS object");
    case SDS__NOTTOP:
        return ("SDS:Not a top level object ");
    case SDS__EXTERN:
        return ("SDS:Illegal operation on external object");
    case SDS__NOTARR:
        return ("SDS:Object is not an array");
    case SDS__VERSION:
        return ("SDS:Object created by an incompatible SDS version");
    case SDS__FOPEN:
        return ("SDS:Error opening file");
    case SDS__FWRITE:
        return ("SDS:Error writing to a file");
    case SDS__FREAD:
        return ("SDS:Error reading a file");
    case SDS__TYPE:
        return ("SDS:Object has incorrect type");
    case SDS__TIMEOUT:
        return ("SDS:Timeout on semaphore access");
    case SDS__INVPRIMTYPE:
        return ("SDS:Invalid primitive type code");
    case SDS__INVSTRUCTDEF:
        return ("SDS:Invalid structure definition");
    case SDS__SYNTAX:
        return ("SDS:Parsing syntax error");
    case SDS__INVSTRUCTURE:
        return ("SDS:Invalid structure");
    case SDS__INVTYPETYPE:
        return ("SDS:Invalid typedef type");
    case SDS__STRUCTMULTDEF:
        return ("SDS:Multipled defined structure");
    case SDS__INVINT:
        return ("SDS:Invalid integer");
    case SDS__INVTYPE:
        return ("SDS:Invalid type");
    case SDS__INVINPUT:
        return ("SDS:Invalid input");
    case SDS__STRUCTARRAY:
        return ("SDS:Array of structures");
    case SDS__MAXDIMS:
        return ("SDS:Maximum number of dimensions");
    case SDS__NOINPUT:
        return ("SDS:Empty input file");
    case SDS__TCL_ARGERR:
        return ("ARG:Error in command arguments");
    case SDS__TCL_CMDERR:
        return ("ARG:Unspecified error in Tcl command");
    case  ARG__CNVERR:
        return ("ARG:Type Conversion Error");
    case  ARG__NOTSCALAR:
        return ("ARG:Object is not a scalar");
    case  ARG__NOTSTRING:
        return ("ARG:Object is not a strin");
    case  ARG__MALLOCERR:
        return ("ARG:Malloc failed");
    default:
    {
        static char string[40];
        sprintf(string,"Uknown error code (not SDS or ARG) return from an Sds or Arg routine, code = %d (%%x%x)",
                error,error);
        return string;
    }
    } /* switch */
}
/*
 *  Internal SDS error reporting routine.
 *
 *  This is called whenever we get an error from the SDS Tcl commands due
 *  to status being set bad.  It provides consistent formating of error
 *  codes.
 *  
 */
static int TclCmdError(
    Tcl_Interp *interp,
    SDSCONST char *cmd,
    StatusType error)
{
    char value[20]; /* For text of message number */
/*
 *  Format the command name, followed by a colon and the error test
 *  into the Tcl result string.
 */
    if (cmd)
    {
        int len = strlen(cmd);
        strcpy(interp->result,cmd);
        interp->result[len++] = ':';
        sprintf(&interp->result[len],"%s",GetErrorString(error));
    }
/*
 *  Format the errorCode info.
 */
    sprintf(value,"%ld",(long)error);
/*
 *  Set the error code. We set the first value in the array to SDS and
 *  the second to the decimal value.
 */
    Tcl_SetErrorCode(interp,"SDS",value,0);

    if (error != STATUS__OK)
        ErsFlush(&error);

    return TCL_ERROR;

}

/*
 *  Internal SDS command creation routine.  Adds the specified command to
 *  the Sds namespace, if we have Tcl namespace support.  Otherwise, it
 *  is put into the global namespace.
 */
static void CreateCommand(
    Tcl_Interp *interp,
    SDSCONST char * SDSCONST name,
    Tcl_CmdProc *proc,
    ClientData * clientData,
    Tcl_CmdDeleteProc *deleteProc)
{
    Tcl_DString dString;

    Tcl_DStringInit(&dString);

#if TCL_MAJOR_VERSION >= 8
    Tcl_DStringAppend(&dString,"sds::",-1);
#endif
    Tcl_DStringAppend(&dString,(char *)name,-1);

    Tcl_CreateCommand(interp, Tcl_DStringValue(&dString),
                      proc, clientData, deleteProc);
    

}


/*+	S d s _ I n i t
 
 * Function Name:
        Sds_Init

 * Function:
        Add SDS commands to a TCL intepreter.
 
 * Descriptions:
        Add the SDS TCL commands to a TCL intepreter.  This assumes
        a non-DRAMA TCL intepreter.  Otherwise, DTCL should be used.

        The intention is to allow SDS commands to be added to a non-DRAMA
        TCL intpreter.

        Commands are added in the "sds" namespace, unless we don't
        have namespace support, in which case commands are added to
        the global namespace.

 * Include File:
	sdstcl.h

 *  Call:
         (int) = Sds_Init(interp)

 *   Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) interp (Tcl_Interp) The Tcl interperter to add the commands too.
 
 * Author: Tony Farrell, AAO

 *  See Also: SDS manual, DTCL manual.

 *-
 
 *  Copyright (c) Anglo-Australian Telescope Board, 1998.
    Not to be used for commercial purposes without AATB permission.

 * Sccs Id:     sdstclinit.c, Release 1.3, 06/21/99
  
 * History:
      03-Nov-1998 - TJF - Original version.
      01-Sep-1999 - KS  - Allows for non-Unix builds where SDS_VERNUM is
                          undefined. (Normally, this is defined using a -D
                          option in a makefile.)
      
 */

SDSEXTERN int Sds_Init(         
    Tcl_Interp *interp)
{
    char verString[40];
/*
 *  Tell Tcl what package we are.  
 */
 
#ifndef SDS_VERNUM
#define SDS_VERNUM "0.0"
#endif

#if TCL_MAJOR_VERSION >= 8
    sprintf(verString,"%s",SDS_VERNUM);
    if (Tcl_PkgProvide(interp,"Sds",verString) == TCL_ERROR)
        return TCL_ERROR;
#endif

/*
 *  Add the commands
 */

    SdsTclAddCommands(interp, TclCmdError, CreateCommand);
    return TCL_OK;
    
}
