#ifndef _FPCOLMDLCALLIBS_H_
#define _FPCOLMDLCALLIBS_H_

/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: fpcolMdlCalLibs.h 74556 2002-11-24 09:25:26Z flamemgr $"
*
*
* who       when      what
* --------  --------  ----------------------------------------------
* tfarrell  18/10/02  Add powellMode to structure.  Add extra parameters to
*                     the fit functions and typedef (tolerance, maxIts, mode).
* tfarrell  10/10/02  A report from Matthew Colless seems to indicate
*                     that we sometimes need sys/types.h to get fitsio.h
*                     to compile.
* tfarrell  28/08/02  Add fpcolMdlPowell() prototype.
* tfarrell  26/08/02  Add fpcolMdlCalFindExe() prototype.
* tfarrell  24/08/02  Add x and y to FpcalResidualsInfo() structure.
* tfarrell  21/08/02  Declare ModelParNames and ModelParUnits and associated
*                     macros here (definition in fpcolMdlCal.c)
* tfarrell  27/06/02  Add fpcolMdlCalCheckFitType() and
*                     fpcolMdlCalFindButton() prototypes.
* tfarrell  09/05/02  Add support for using offsets in arc-seconds
*                     Add FpcalCurWaveInx type to support different
*                      current wavelengths for each fibre type.
*                     Support multiple offsets for each fibre - one
*                      for each wavelenght.
* tfarrell  12/04/02  Include fpDefines and use fpMDL_NUM_PARS to define NUM_PARAMETERS
* ssmedley  07/09/01  created
*
*/

/************************************************************************
*   NAME
*	fpcalConstructor fpcalDestructor getFpcalData calculateOffsets -
*	Module to store/load/manipulate data used during FLAMES/OzPoz
*	calibration & subsequently to create an astrometric model.
*
*   SYNOPSIS
*       #include "fpcolMdlCalLibs.h"
*
*	struct { ... }FpcalData;
*
*	void fpcalConstructor (FpcalData *pFpcalData);
*
*	void fpcalDestructor (FpcalData *pFpcalData);
*
*	void getFpcalData (
*		const char	*pFitsFilename,
*		FpcalData	*pFpcalData,
*		StatusType	*pStatus);
*
*	void calculateOffsets (
*		FpcalData	*pFpcalData,
*		StatusType	*pStatus);
*
*
*   DESCRIPTION
*
*	fpcalConstructor() will initialise an FpcalData object. You
*	should call fpcalDestructor() whenever you are done with it, or
*	use the object for another set of data.
*
*	fpcalDestructor() tidies up a used FpcalData object. After a
*	call to this function, the object should no longer be used until
*	fpcalConstructor() has been invoked for the object.
*
*	getFpcalData() extracts all relevant data from the FITS file
*	specified by <pFitsFilename> for creating an astrometric model.
*	Note that the FITS fields that are required are specified by
*	#define's in the fpcolMdlCalFields.h header file. If successful,
*	the appropriate data will reside in <pFpcalData>.
*
*	calculateOffsets() will calculate/set the offsets for each fibre
*	button in <pFpcalData>. Note that the offsets are (usually) specified
*	in a separate text file & must be loaded beforehand. (see
*	loadButtonOffsets() in fpcolMdlCalLoadOffsets.h).
*
*   ENVIRONMENT
*
*   CAUTIONS
*
*   EXAMPLES
*
*   SEE ALSO
*
*   BUGS
*
*------------------------------------------------------------------------
*/



#include <sys/types.h>                  /* Sometimes needed by fitsio.h */

#include "status.h"			/* StatusType */
#include "fitsio.h"			/* fitsfile */


#include "fpDefines.h" /* For fpMDL_NUM_PARS */
#include "fpcolMdl.h"
#define NUM_PARAMETERS		fpMDL_NUM_PARS

/*
 * Conversions between radians and arc-seconds.
 */
#define OFFSET_IN_ARCSEC   1 /* Macro which indicates we are using offsets in arc-seconds */
#if OFFSET_IN_ARCSEC
#define RAD2SEC(_rad) ((_rad)/M_PI*180.0*60.0*60.0)
#define SEC2RAD(_sec) ((_sec)/(180.0*60.0*60.0)*M_PI)
#endif
/*
 *  Structure for containing a set of telescope model parameters
 */
typedef struct 
{
  double p[NUM_PARAMETERS];
} FpcalPars;


/*
 * We need a different telescope model for each wavelength we are dealing
 * with.   Each model uses the same telescope model parameters.
 */
#define MAX_WLENS 10
typedef struct 
{
  FpcalPars pars;                    /* Telescope model parameters   */
  int       parsInited;              /* Have the parameters been initilised */
  int       created;                 /* Have the models been created */
  int       wavelenCnt;              /* Number of wavelengths        */
  double    wavelength[MAX_WLENS];   /* Wavelenghts - nanometers     */
  FpilModelType *models[MAX_WLENS];  /* Model for each wavelength    */
} FpcalMdl;

typedef struct
{
  int Giraffe;
  int UVES;
  int FACB;
} FpcalCurWaveIdx;

/*
 * Structure to contain details of the residuals calculated by fitFn().
 */
typedef struct
{
    long x;         /* The fitted model, applied to the RA/DEC minus offset, which is */
    long y;         /*   an approximation to the true position of the button. */
    long initialX;  /* Initial model applied to object sky positions */
    long initialY;
    long fittedX;   /* The position given by the fitted model  by inputing soley  */
    long fittedY;   /*    the apparent RA/DEC. */
} FpcalResidualsInfo;

typedef void (*fitFuncType)(int nobs, FpcalMdl *models, 
                            double tolerance, int maxIts, int mode, 
                            double mjd, double rc, 
                            double dc,
                            double *ra, double *dec, double *dra, double *ddec, int *widx,
                            int *pfit, double *pvar, double *rms, double *psd, 
                            StatusType *status);



typedef struct
{
	int		nFibres_a;
        /*
         * Current wavelength indicies in the "model" item below for each of
         * Giraffe/UVES/FACB.  
         */
       FpcalCurWaveIdx curWaveIndex;

        /*
         * Data read from fits header or equivalent.  Pointers to malloced data, 
         * We have nFibres of each of these
         */
        double          *meanRA;        /* array of mean RA & DEC coorindates */
        double          *meanDEC;
	double		*apparentRA;	/* array of apparent RA & DEC coordinates */
	double		*apparentDEC;   
	double		*properMotionRA;
	double		*properMotionDEC;
	int		*buttons;

        /*
         *  Data read using offset command.  Pointers to malloced data, we
         * have nOffsets of these.
         */
        int             nOffsets;
	double		*offsetRA;
	double		*offsetDEC;
        int             *offsetWaveIndex; /* Wavelength index in FpcalMdl */
        int             *offsetButtonNo;        /* Button number */

        FpcalResidualsInfo *residualsInfo;
        int             fitted;            /* Indicats if residualsInfo is valid */
        /*
         * Field center and telesope model info.
         */
	double		fieldCentreMeanRA;
	double		fieldCentreMeanDEC;
        double          fieldCentreAppRA;
        double          fieldCentreAppDEC;
	double		atmTemp;	/* atmospheric conditions */
	double		atmPressure;
	double		atmHumidity;
	double		equinox;

        double		posAngle;           /* ADC position angle */
	double		observationDate;	/* MJD */
        int		aFit[NUM_PARAMETERS]; /* Flag indiatating parameters to be fitted */
	double		meanAppParams[21];	/* for SLA lib routines */
	int		fieldPlate;
        /*
         * Have we been initialised
         */
        int		bInitialised;
        /*
         * Details of the telescope model
         */
        FpcalMdl        model;


        fitFuncType    fitFunc;
        double         powellTol;
        int            powellMaxIts;
#define POWELL_MODE_1 1
#define POWELL_MODE_2 2
        int            powellMode;
        /*
         *
         */
        char            *fileName;  /* A pointer to the file name is put here */
}FpcalData;
/*
 *  Names for each of the parameters and units.  Values defined in fpcolMdlCal.c.  These
 * macros are also declared again in fpcolMdlCal.c and must have the same value there.
 */
extern char *ModelParNames[];
extern char *ModelParUnits[];
#define LASTLIN 5
#define ROTPAR  6
#define SCALEPAR 7
#define NW_DIST_FIRST 8
#define NW_DIST_LAST  10
#define W_DIST_FIRST 11
#define W_DIST_LAST  12
#define NONPERP 18
/*
 * constructor function for FpcalData objects.
 */
void fpcalConstructor (FpcalData *pFpcalData);

/*
 * destructor function for FpcalData objects.
 */
void fpcalDestructor (FpcalData *pFpcalData);


/*
 * This function extracts all relevant data from the
 * specified fits file using the (fibre) button offsets
 * specified in the offset file.
 */
void getFpcalData
	(const char *pFitsFilename,
	 FpcalData	*pFpcalData,
	 StatusType	*pStatus);

int getFpcalDataText(
                      const char *fileName,
                      const char *cmdName,
                      FpcalData	*pFpcalData);
/*
 * This function sets the offsets for each fibre button.
 * Note: make sure button offsets have been loaded before
 *       calling this function.
 */
void calculateOffsets (FpcalData *pFpcalData, StatusType *pStatus);


/*
 * Function which does the fit.
 */
extern void fpcolMdlFit(int nobs, FpcalMdl *models, 
                        double tolerance, int maxIts, int mode, 
                        double mjd, double rc, double dc,
                        double *ra, double *dec, double *dra, double *ddec, int *widx,
                        int *pfit, double *pvar, double *rms, double *psd, 
                        StatusType *status);
 
/*
 * Alternative fitting function 
 */
extern void fpcolMdlPowell(int nobs, FpcalMdl *models, 
                           double tolerance, int maxIts, int mode, 
                           double mjd, double rc, double dc,
                           double *ra, double *dec, double *dra, double *ddec, int *widx,
                           int *pfit, double *pvar, double *rms, double *psd, 
                           StatusType *status);
 
/*
 * Indicates fibre types.
 */
extern int fpcalIsUVES(unsigned int fibre);
extern int fpcalIsFACB(unsigned int fibre);
extern int fpcalIsGiraffe(unsigned int fibre);
extern int fpcalWaveIdxForFibre(const FpcalCurWaveIdx *curWaveIndex, unsigned int fibre);
/*
 * Warns of attempt to fit linear model parameters
 */
extern void fpcolMdlCalCheckFitType(int *aFit);
/*
 * Find the index of a buttons of the specified button no.
 */
extern int fpcolMdlCalFindButton(
    const int nButtons,   /* Number of elements in buttons */
    const int *buttons,   /* Array of button numbers */
    const int buttonNo);   /* The button to look for */

/*
 * Routine to find the executable directory.
 */
extern void fpcolMdlCalFindExe (
    const char *Arg0,                      /* Zeroth argument to program */
    char *Directory,                       /* Name of 'exe' directory */
    int LDir,                              /* Number of chars in Directory */
    int *Known);      

extern void fpcolMdlDecompLin(
    const double *pars);
#endif
