/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: fpcolBase.c 77339 2003-02-03 05:45:35Z flamemgr $"
* 
*
* who       when      what
* --------  --------  ----------------------------------------------
* tjf       08/06/00  created
* ks        18/08/00  modified definition of INLINE to remove conflicts
*                     under some environments (CodeWarrior in this case).
* ks        02/02/01  relaxed the rather pessimistic clearance being used
*                     for fibre/button collision tests.
* tjf       19/08/02  Enable the check for a collsion between a fibre
*                     and button tail in fpcolBaseColButFib().
*                     Fix problem in fpcolBaseColButFib() were it was
*                     using the wrong ends.  Fix bug in NewLineSegEnds()
*                     which was getting the angle wrong.  These bugs only
*                     showed when we started checking for a fibre-tail
*                     collision and it is sensible that they only showed
*                     then.
* rschmutz 21/12/02   fpcolColInvPos: fibreType arg. added.
* rsc/aka  02/02/03   fpcolColButFib: added fibreClear to fibre/tail check.
*/

/************************************************************************
*   NAME
*       fpcolBase - Base of FP collision detection. 
*
*   SYNOPSIS
*       #include "fpcolInternal.h"
*
*
*   DESCRIPTION
*      This library provides the basic FP collision detection routines.
*      This file is used by both the instrument software and
*      the configure software.  The Instrument software is written to VLT
*      software specifications.  The configure software is a standalone 
*      program written using the AAO's DRAMA style.  This file must be
*      compatible with both styles.  Wrappers are provided to interface
*      this routines to configure (fpcolFpil.c) and the instrument
*      fpcol.c.  
*
*      The calling interface of these routines are appropriate
*      for the FPIL library used by configure and deal in X/Y/Theta where
*      Theta indicates a twist of the fibre.  In OzPoz there is no such twist
*      and this theta value is normally assumed to just indicate the alignment
*      of the tail
*
*      Note on angles.
*          The configure software uses theta = 0 along the positive y
*          axis, instead of the positive x.   So to convert to mathematical
*          orientation, we must rotate by  negative PI/2.   Our internal
*          angles are all mathematical to ensure the matematics we do
*          is correct, so the conversion is done just before the various 
*          types representing line seguments etc. are set up.
*     
*
*      Beware that the OzPoz robot talks in R/Theta for the plate position and
*      alpha for the angle which indicates the tail aligment.  The fpcol
*      module does the appropiate conversions.
*
*      This module has contains larges amounts of code extracted from the
*      AAO 6dF instrument equivalent module.  As a result, its format
*      is not normal VLT format.
*      
*
***
*
*   ENVIRONMENT
*       lopoz - Flames Fibre Positioner.
*       configure - AAO configure program.
*
*   CAUTIONS 
*
*   EXAMPLES
*
*   SEE ALSO
*       fpcol module, AAO FPIL Library.
*   BUGS   
*
*------------------------------------------------------------------------
*/

#include <math.h>
#include <stdio.h>
#include "fpcolBase.h"
#include "fpcolInstDescr.h"

#define PI 3.14159265358979323846

#ifndef DEBUG
#define DEBUG printf
#endif

/*
 *  The INLINE macro allows functions to be inlines under GNU C.
 *  The UNUSED macro allows to prevent unused parameter warnings
 *  under GCC.
 */

#if defined(__GNUC__) || defined(GNUC)
#   define INLINE __inline__ static
#   define UNUSED 
#   ifdef __GNUC_MINOR__
#   if __GNUC_MINOR__ >= 7
#       undef  UNUSED
#       define UNUSED __attribute__ ((unused))
#   endif
#   endif
#else
#   ifndef INLINE
#      define INLINE static
#   endif
#   define UNUSED 
#endif
/*
 *  Definitions
 */
#define SQRD(x)           ((double)(x)*(double)(x))

/*
 *
 *  We have here a bunch of types and associated routines for describing
 *  points, lines etc.
 *
 */

/*
 *  Type describing a point 
 */
typedef struct {
    double x;     /* X ordinate */
    double y;     /* Y ordinate */
} pointType;

/*
 *  Initialise a point from x and y.
 */
INLINE pointType * NewPoint(double x, double y, pointType *point)
{
    point->x = x;
    point->y = y;
    return point;
}

/*
 *  Type describing a line segment using start and end points but for which 
 *  the angle of the line and length of the line are known.  
 */
typedef struct {
    pointType start;  /* Start point */
    pointType end;    /* End point   */
    double angle;     /* Angle line lays along */
    double length;    /* Length of line from start point */
} lineSegType;

/*
 *  Create a lineSegType from a start point, angle and length.
 *
 */
INLINE lineSegType * NewLineSegAngle(pointType start, 
                                      double angle, 
                                      double length,
                                      lineSegType *line)
{
    double cosine;
    double sine;


/*
 *  Copy in basic values.
 */
    line->start  = start;
    line->angle  = angle;
    line->length = length;


    cosine = cos(angle);
    sine   = sin(angle);
/*
 *  Determine other end of the line.
 */
    line->end.x  = start.x + length * cosine;

    line->end.y  = start.y + length * sine;

    return line;

    
}
/*
 *  Create a lineSegType from a start and end points.
 */
INLINE lineSegType * NewLineSegEnds(pointType start, 
                                     pointType end,
                                     lineSegType *line)
{
    double x_len = end.x - start.x;
    double y_len = end.y - start.y;

    line->start  = start;
    line->end    = end;
   
    line->start  = start;

    line->length = sqrt(x_len*x_len + y_len*y_len);

    line->angle  = atan2(y_len, x_len);
    return line;
  
}



/*
 *  Type describing a circle.
 */
typedef struct {
    pointType center;
    double    radius;
} circleType;
/*
 *  Initialise a circle from x, y and radius
 */
INLINE circleType * NewCircleXYR(double x, double y, double radius, 
                             circleType *circle)
{
    circle->center.x = x;
    circle->center.y = y;
    circle->radius   = radius;
    return circle;
}
/*
 *  Initialise a circle from center point and radius
 */
INLINE circleType * NewCircleP(pointType center, double radius, 
                             circleType *circle)
{
    circle->center = center;
    circle->radius = radius;
    return circle;
}

/*
 * Internal Internal Function name:
      MathsTheta
  
 * Description:
      Converts from a positioner system theta value (0 = y axis) to
      a mathematical theta value (0 = x axis).

      
  
 * History:
     18-Jan-2000 - TJF - Original version.
  
 */
INLINE double MathsTheta(double theta)
{
/*
 *   Rotate and then normalize on the way out.
 */
    theta += (PI/2);
    return fmod(theta, (2*PI));
}

/*
 * Internal Internal Function name:
      CirclesOverlap
  
 * Description:
      Determines if two circles overlap.
  
 * History:
     18-Jan-2000 - TJF - Original version.
  
 */
INLINE int  CirclesOverlap (
    const circleType *c1,  /* Circle one */
    const circleType *c2)  /* Circle two */
    
{
/*
 *  This is quite easy.  Just determine the distance between the
 *  two points which given the origin's of the circles.  If that
 *  distance is less then the combined radii, they overlap.
 */
    if ((double)(SQRD(c1->center.x - c2->center.x) +
                 SQRD(c1->center.y - c2->center.y)) 
        <= SQRD(c1->radius + c2->radius))
        return 1;
    else
        return 0;
                                           
}


/*
 * Internal Internal Function name:
      CircleLineOverlap
  
 * Description:
      Determines if a circle and line segement overlap

      To handle non-zero width lines, just expand the circle
      radius by the width of the line

 * History:
     18-Jan-2000 - TJF - Original version.
     20-Feb-2002 - TJF - Should be using absolute value of distance in x
                         in the parallel to y axis case.
  
 */
INLINE int  CircleLineOverlap (
    const circleType  *circle,  /* Circle details */
    const lineSegType *line)    /* Line details   */
    
{
    double radSqrd = SQRD(circle->radius);
/*
 *  a is the length of the line squared
 *  b is the distance from the circle to one end of the line squared
 *  c is the distance from the circle to the other end of the line squared.
 */

    double a = SQRD(line->length);

    double b = SQRD((line->start.x - circle->center.x)) +
               SQRD((line->start.y - circle->center.y)) ;

    double c = SQRD((line->end.x   - circle->center.x)) +
               SQRD((line->end.y   - circle->center.y));
/*
 *  If the distance from either end of the line is longer then the line
 *  length + circle radius, then there can be no collision.
 */
    if (((a+radSqrd) <= b) || ((a+radSqrd) <= c))
        return 0;
    
    else if (line->start.x == line->end.x)
    {
/*
 *      The line is parallel to the Y axis  (the next bit of code would get
 *      a divide by zero error) and the y of point is within the length
 *      of the fibre (ensured by the above check)
 *
 *      Collision occurs if the distance to the line in x is
 *      less then the circle radius. 
 */
        double d = fabs(circle->center.x - line->end.x);
        return ((d <= circle->radius)? 1: 0);        
    }
    else
    {        
/*
 *      We can now use code which determines the distance between a line
 *      and a point since
 *        1.  The first check ensures a perpendicular from line to the
 *            point is within the line-segement of concern
 *        2.  line->end.x - line->start.x != 0
 *
 *     Given m is the gradient of the line between the fibre ends and
 *     b is the intercept, then, d the distance between the point x1,y1 and
 *     the line is given by
 *
 *      d = |  m.x1 - y + b    |   (Note, absolute value only)
 *          |------------------|
 *          | +/- sqrt(m^2 + 1)|
 *
 *
 *      Determine m and b.
 */
        double m = (double)(line->end.y - line->start.y)/
                   (double)(line->end.x - line->start.x);
        double b = line->end.y - m*line->end.x;
        
/*
 *      Determine the denominator and then the numerator.  Note, sqrt()
 *      returns the positive square root only.
 */
        double dd = sqrt(m*m + 1);
        double dn = m * circle->center.x - circle->center.y + b;
        
        double d;
/*
 *      Determine the distance.  (Note, dd is always greater then 1, therefore
 *      no divide by zero is possible)
 */
        if (dn < 0)
            d = dn * -1 / dd;
        else
            d = dn / dd;         
         
        return ((d <= circle->radius) ? 1: 0);
    }
    
}
/*
 * Internal Internal Function name:
      RotatePoint
  
 * Description:
      Rotate a given point around the origin using the sine and cosine of
      the rotation angle.

 * History:
     18-Jan-2000 - TJF - Original version.
  
 */

INLINE void RotatePoint (
    const pointType *point,
    double    sine,
    double    cosine,
    pointType *result )
{
    result->x =      point->x * cosine + point->y * sine;
    result->y = -1 * point->x * sine   + point->y * cosine;
}
  

typedef struct {
    pointType botLeft;   /* Bottom left */
    pointType topLeft;   /* Top left */
    pointType topRight;  /* Top right    */
    pointType botRight;  /* Bottom Right */
} rectangleType;

/*
 * Make a rectangle desribing a thick line parallel to the  X axis.  Don't
 * apply it's angle, but save it for later use.
 */
INLINE void MakeRectangle(const lineSegType *line, 
                          double width,
                          rectangleType *rec)
{
    double yoff,xoff;
    double sine,cosine;
    double halfWidth = width/2;
    rectangleType PreRotation;


/*
 *  
 *  If the angle is so small as to be insignificant, then assume an
 *  angle of zero and just construct it.
 *      return immediately
 */        
    if ((line->angle <= 0.0001)&&(line->angle > -0.0001))
    {
        rec->botLeft.x   = line->start.x;
        rec->botLeft.y   = line->start.y - halfWidth;
        
        rec->topRight.x  = line->start.x + line->length;
        rec->topRight.y  = line->start.y + halfWidth;
        
        
        rec->topLeft.x   = rec->botLeft.x;
        rec->topLeft.y   = rec->topRight.y;
        
        rec->botRight.x  = rec->topRight.x;
        rec->botRight.y  = rec->botLeft.y;
        return;
    }
/*
 *  We need to rotate the rectangle.  Construct a rectangle of the
 *  right size, but with the bottom left corner at the origin and
 *  the bottom layed along the X axis.
 */
    PreRotation.botLeft.x = 0;
    PreRotation.botLeft.y = 0;

    PreRotation.topLeft.x = 0;
    PreRotation.topLeft.y = width;

    PreRotation.topRight.x = line->length;
    PreRotation.topRight.y = width;
    
    PreRotation.botRight.x = line->length;
    PreRotation.botRight.y = 0;
    
/*
 *  Determine sine and cosine.  (We use the negative angle, since
 *  we are effectivly removing a previous rotation.  
 */
    sine   = sin(-1*line->angle);
    cosine = cos(-1*line->angle);

/*
 *  So now point botLeft is at the origin.  Rotate the
 *  other three points by the rectangles angle (the one which put it
 *  on the X axis)
 */
    rec->botLeft = PreRotation.botLeft;
    RotatePoint(&PreRotation.topLeft,  sine, cosine, &rec->topLeft);
    RotatePoint(&PreRotation.topRight, sine, cosine, &rec->topRight);
    RotatePoint(&PreRotation.botRight, sine, cosine, &rec->botRight);
    
/*
 *  So now the rectangle is rotated correctly, we need to offset 
 *  it correctly.  The origin is given by the line start point, except
 *  that y allows for the width.
 */
    xoff =  line->start.x;
    yoff =  line->start.y - width/2;
/*
 *  Offset orgin back to where it should be
 */
    rec->botLeft.y   = yoff;
    rec->topLeft.y  += yoff;
    rec->botRight.y += yoff;
    rec->topRight.y += yoff;   
    

    rec->botLeft.x   = xoff;
    rec->topLeft.x  += xoff;
    rec->botRight.x += xoff;
    rec->topRight.x += xoff;
       
/*
 *  We are done.
 */

}

/*
 *  Rotate a rectangle by a specified angle, with the specified point
 *  to become the origin.
 */
INLINE void RotateRectangle(const rectangleType * const in,
/*                            double        originX,
                              double        originY,*/
                            double        sine,
                            double        cosine,
                            rectangleType *out)
{
/*
    pointType botRight;
    pointType topLeft;
    pointType topRight;
    pointType botRight;
*/
/*
 *  
 */    
/*
    botRight.x = in->botRight.x - originX;
    botRight.y = in->botRight.y - originY;

    topRight.x = in->topRight.x - originX;
    topRight.y = in->topRight.y - originY;

    topLeft.x  = in->topLeft.x  - originX;
    topLeft.y  = in->topLeft.y  - originY;

    botLeft.x  = in->botLeft.x  - originX;
    botLeft.y  = in->botLeft.y  - originY;
*/

/*
 *  Rotate all points.
 */
    RotatePoint(&in->botLeft,  sine, cosine, &out->botLeft);
    RotatePoint(&in->topLeft,  sine, cosine, &out->topLeft);
    RotatePoint(&in->topRight, sine, cosine, &out->topRight);
    RotatePoint(&in->botRight, sine, cosine, &out->botRight);
}

/*
 *  Determine the minimum value of four values.
 */
INLINE double FourPointMin(double a, double b, double c, double d)
{
    double min = a;
    if (b < min)
        min = b;
    if (c < min)
        min = c;
    if (d < min)
        min = d;
    return min;
}
/*
 *  Determine the maximum value of four values.
 */
INLINE double FourPointMax(double a, double b, double c, double d)
{
    double max = a;
    if (b > max)
        max = b;
    if (c > max)
        max = c;
    if (d > max)
        max = d;
    return max;
}

/*
 *  A type that represents the extremes of a shape - that is, the left most,
 *  right most, top most and bottom values.
 */
typedef struct {
    double left;
    double right;
    double top;
    double bottom;
} extremesType;

/*
 *  Initialise an Extremes types from a rectangle.
 */
INLINE void Extremes(const rectangleType *r, extremesType *e)
{

    e->left   = FourPointMin(r->topLeft.x,  r->botLeft.x, 
                           r->topRight.x, r->botRight.x);
    e->right  = FourPointMax(r->topLeft.x,  r->botLeft.x, 
                           r->topRight.x, r->botRight.x);
    
    e->bottom = FourPointMin(r->topLeft.y,  r->botLeft.y, 
                           r->topRight.y, r->botRight.y);
    e->top    = FourPointMax(r->topLeft.y,  r->botLeft.y, 
                           r->topRight.y, r->botRight.y);

}


/*
 *  Given variables supplying the extreme values of two objects, determine
 *  if the objects might have overlap.
 */
INLINE int IsOverlap(
    const extremesType *a, 
    const extremesType *b)
{
/*
 *  If b is totally to the left of a, no overlap is possible
 */
    if (b->right < a->left)
        return 0;
/*
 *  If a is totally to the left of b, no overlap is possible
 */
    else if (a->right < b->left)
        return 0;
/*
 *  If b is totally above a, no overlap is possible
 */
    else if (b->bottom > a->top)
        return 0;
/*
 *  If a is total above b, no overlap is possible
 */
    else if (a->bottom > b->top)
        return 0;
/*
 *  There is some overlap.
 */
    return 1;

}
/*
 * Internal Internal Function name:
      RectanglesCollide
  
 * Description:
      Determines if two rectangles collide

      Straede's conjecture - Two rectangles overlap if

          1.  Rotate Rectangles such that  A is aligned with X axis.  
              There must be overlap in either X or Y directions
          2.  Rotate Rectangles such that B is aligned with the X axis.
              There must be overlap in either X or Y directions

     Note that our rectangles are not normalized, meaning the "angle"
     member gives the angle the rectangle is at relative to the
     X axis and the points of the rectangle have not been adjuects for this.

 * History:
     19-Jan-2000 - TJF - Original version.
  
 */
INLINE int RectanglesCollide (
    const rectangleType *ra,
    const double raAngle,
    const rectangleType *rb,
    const double rbAngle)
{
    rectangleType raInb;        
    rectangleType raInOwn;
    rectangleType rbIna;
    rectangleType rbInOwn;
    extremesType  raExtremes;
    extremesType  rbExtremes;
    extremesType  targetExtremes;
    double        sine;
    double        cosine;


/*
 *  Rotate both rectangles to rectangle a's coorindate system.
 */
    sine   = sin(raAngle);
    cosine = cos(raAngle);
    RotateRectangle(ra, sine, cosine, &raInOwn);
    RotateRectangle(rb, sine, cosine, &rbIna);

/*
 *  Get extremes of rectangle a (in own system) and the rotated rectangle b)
 */
    Extremes(&raInOwn, &raExtremes);
    Extremes(&rbIna,   &targetExtremes);

    if (IsOverlap(&raExtremes, &targetExtremes))
    {
/*
 *      Rotate both rectangles to rectangle a's coorindate system.
 */
        sine   = sin(rbAngle);
        cosine = cos(rbAngle);

        RotateRectangle(rb, sine, cosine, &rbInOwn);
        RotateRectangle(ra, sine, cosine, &raInb);
/*
 *      Get extremes of rectangle b (normalized) and the rotated rectangle a)
 */
        Extremes(&rbInOwn, &rbExtremes);
        Extremes(&raInb,   &targetExtremes);

/*
 *      If we have any overlap, then we have a collision
 */
        return (IsOverlap(&rbExtremes, &targetExtremes));
    }
    return 0;

}
/*
 * Internal Internal Function name:
      ThickLinesCollide
  
 * Description:
      Determines if two line segements of non-zero width cross.


 * History:
     13-Dec-1999 - TJF - Original version.
  
 */
INLINE int ThickLinesCollide (
    
    const lineSegType *line1,   /* Line 1                           */
    double line1width,          /* Line 1 width                     */

    const lineSegType *line2,   /* Line 2                           */
    double line2width)          /* Line 2 width                     */

{   

    rectangleType r1;
    rectangleType r2;

/*
 *  Represent the thick lines as rectangles and then check 
 *  for collision between thoses rectangles.
 */
    MakeRectangle(line1, line1width, &r1);
    MakeRectangle(line2, line2width, &r2);

    return RectanglesCollide(&r1, line1->angle, &r2, line2->angle);
    

}
/*
 * Internal Internal Function name:
      AdjustPivot
  
 * Description:
      In OzPoz, the pivot point of a fibre tends to move as the fibre
      comes out of the retractor due to the shape of the
      retractor mouth.

      The default pivot radius is the applies for bend angles
      of less then XXX.  For greater angles, the point moves.

 * History:
     01-Nov-2001 - TJF - Original version.
  
 */
INLINE void AdjustPivot (
    const pointType   * const buttonEnd,  /* Button end of fibre */
    pointType         * const pivotEnd)   /* pivot end of fibre - adjusted
                                             when returned */
{   
    double anglePivotToCentre;
    double anglePivotToButton;

    
    /*
     * Deterime deltas for line to fibre.
     */
    double deltaX = pivotEnd->x - buttonEnd->x;
    double deltaY = pivotEnd->y - buttonEnd->y;

    double pivotAngle;

    /*
     * Determine the angle of the line to fibre and line to button.
     */
    anglePivotToCentre = atan2(pivotEnd->y, pivotEnd->y);
    anglePivotToButton = atan2(deltaY, deltaX);

    /*
     * Get the absolute pivot angle.
     */
    pivotAngle = fabs(anglePivotToCentre - anglePivotToButton);

    /*
     * If this angle is greater then PI (should not be) then subtract it
     * from 2PI and use the absolute value of the result  (Delta does this,
     * am unsure if it is really needed).
     */
    if (pivotAngle >= M_PI)
      pivotAngle = fabs((2  * M_PI) - pivotAngle);

    /*
     * If the angle is too large, the pivot point starts to move.
     */
    if (pivotAngle > fpcolINST_PIV_ANGLE_MOVE_STARTS)
      {
/*#warning "Pivot move code not yet determined"*/
      }

}
/*
 * Internal Internal Function name:
     fpcolBaseColButFib
  
 * Description:
     OzPoz Button-Fibre collision check routine.

     Here we are checking for two things. 

        1.  A collision between
            a circle of radius fpcolINST_BUTTON_RAD centered at butX, butY and
            the fibre of radius fpcolINST_FIBRE_DIA.  The circle represents 
            the fibre button itself.

        2.  A collision between two thick line segments, 
        one representing the fibre and the other the tail of
        the button. 
        
  
 * History:
     13-Dec-1999 - TJF - Original 6dF version.
     08-Jun-2000 - TJF - OzPoz version
     02-Feb-2001 - KS  - Reduced the size of the button circle used - it was
                         being overly pessimistic.
     01-Mar-2001 - TJF - Adjust the pivot point for the effect caused
                         as the spring comes out of the retractor.
     19-Aug-2002 - TJF - Add a check on collisions between a fibre
                         and a fibre tail - this appears to be necessary,
                         particular for FACB's were we don't want to lay
                         fibres across or under them.  (This involved enabling
                         previously disabled code).  Also - had the start
                         and end points of the fibre line segement wrong, which
                         only triggers problems in the fibre/tail check.
     
     
  
 */
extern int  fpcolBaseColButFib (
        void * clientData UNUSED,
        double    butX,              /* Button X ordinate */
        double    butY,              /* Button Y ordinate */
        double    theta UNUSED,     /* Button rotation   */
        double    fvpX,              /* Fibre button end  X ordinate */
        double    fvpY,              /* Fibre button end, Y ordinate */ 
        double    pivX,              /* Fibre pivot point, X ordinate */
        double    pivY,              /* Fibre pivot point, Y ordinate */
        long int  fibreClear,        /* Clearance to allow between them */
        double    fibreClearSqrd UNUSED)/* fibreClear squared */
{
    circleType  button;       /* Used to represent the button itself      */
    lineSegType fibre;        /* Used to represent the fibre              */


    pointType buttonLocation;
    pointType fibrePivotEnd;
    pointType fibreButtonEnd;
    double distFibre;
    double distPivot;
    long int tailClear;

/*
 *  Create an object representing the button location point,
 *  and fibre end points
 */
    NewPoint(butX, butY, &buttonLocation);
    NewPoint(fvpX, fvpY, &fibreButtonEnd);
    NewPoint(pivX, pivY, &fibrePivotEnd);
/*
 *  Adjust the pivot point if need be.
 */
    AdjustPivot(&fibreButtonEnd, &fibrePivotEnd);
/*
 *  Contruct the circle object representing the fibre.  We increase the
 *  diameter by the fibre clearance when we do the check.
 */
    NewCircleP(buttonLocation,(fpcolINST_BUTTON_RAD + fibreClear),&button);

/*
 *  Generate a line segment description object, given the ends of the fibre.
 */
    distFibre = sqrt(fvpX*fvpX + fvpY*fvpY);
    distPivot = sqrt(pivX*pivX + pivY*pivY);
    if (distFibre <= distPivot)
        NewLineSegEnds(fibreButtonEnd, fibrePivotEnd, &fibre);
    else
        {
        printf("Warning - pivot and fibre ends swapped\n");
        NewLineSegEnds(fibrePivotEnd, fibreButtonEnd, &fibre);
        }


/*
 *  First check.  Is there a collsion between the fibre button and the 
 *  fibre
 */
    if (CircleLineOverlap(&button, &fibre)) 
    {
        return 1;
    }
    else
    {
        lineSegType  tail;     /* Used to represent the tail of the button */

/*
 *      Now we must check the tail of the button against the fibre.
 *
 *      Create a line structure representing the tail
 */
        if (fibreClear > fpcolINST_FIBRE_CLEAR)
	    tailClear = fibreClear - fpcolINST_FIBRE_CLEAR;
        else
	    tailClear = 0;
/* printf("ColButFib: %ld - %ld\n", fibreClear, tailClear); */
        NewLineSegAngle(buttonLocation, MathsTheta(theta), 
                        fpcolINST_BUTTON_TAIL + tailClear, &tail);
/*
 *      Do the check.
 */
        return ThickLinesCollide(&tail, fpcolINST_BUTTON_TD + tailClear, 
                                 &fibre, fpcolINST_FIBRE_DIA);
    }  
}


/*
 * Internal Function name:
     fpcolBaseColButBut
  
 * Description:
     OzPoz Button-Button collision check routine.

     Here we are checking for four things

     1.  An overlap of two circles, representing the two buttons
     2.  A collision between two lines, represent the tails of
         each button
     3.  A collision between a circle and a line segment - fibre A's button
         and fibre B's tail.
     4.  A collision between a circle and a line segment - fibre B's button
         and fibre A's tail.
  
 * History:
     13-Dec-1999 - TJF - Original 6dF version.
     08-Jun-2000 - TJF - OzPoz version
  
 */
extern int  fpcolBaseColButBut (
        void * clientData UNUSED,
        double    butXa,        /* Button A - X ordinate */
        double    butYa,        /* Button A - Y ordinate */
        double    thetaa,       /* Button A rotation     */
        double    butXb,        /* Button B - X ordinate */
        double    butYb,        /* Button B - Y ordinate */
        double    thetab,       /* Button B rotation     */
        long int  buttClear,    /* Tolerance             */
        double    buttClearSqrd UNUSED)/* buttClear squared     */
{
    circleType buttonA;           /* Circle representing button a */
    circleType buttonB;           /* Circle representing button a */

    pointType  buttonLocationA;   /* Point for button a           */
    pointType  buttonLocationB;   /* Point for button b           */

    
    lineSegType  tailA;           /* Tail of button a             */
    lineSegType  tailB;           /* Tail of button b             */
/*
 *  Create the point types for each button
 */
    NewPoint(butXa, butYa, &buttonLocationA);
    NewPoint(butXb, butYb, &buttonLocationB);
    
/*
 *  Create the circles representing each button.  We add the button 
 *  clearance to B's radius to account for the clearnace.
 */
    NewCircleP(buttonLocationA, fpcolINST_BUTTON_RAD, &buttonA);
    NewCircleP(buttonLocationB, fpcolINST_BUTTON_RAD+buttClear, &buttonB);

/*
 *  Check for overlap between the buttons themselves.
 */
    if (CirclesOverlap(&buttonA, &buttonB))
        return 1;
/*
 *  We now need to check button A against B's tail.  This is a circle to
 *  thick line check.  We change the radius of the circle to account
 *  for the clearance and tail width, and then we can use a circle/line segment
 *  overlap check.
 */   
    buttonA.radius = fpcolINST_BUTTON_RAD + buttClear + fpcolINST_BUTTON_THW;

    NewLineSegAngle(buttonLocationB, MathsTheta(thetab), 
                    fpcolINST_BUTTON_TAIL, &tailB);

    if (CircleLineOverlap(&buttonA, &tailB))
        return 1;

/*
 *  We now need to check button B against A's tail.  This is a circle to
 *  thick line check.  Agin We change the radius of the circle to account
 *  for the clearance and tail width and use the circle/line segument overlap
 *  check.
 */   
    buttonB.radius = fpcolINST_BUTTON_RAD + buttClear + fpcolINST_BUTTON_THW;

    NewLineSegAngle(buttonLocationA, MathsTheta(thetaa), 
                    fpcolINST_BUTTON_TAIL, &tailA);

    if (CircleLineOverlap(&buttonB, &tailA))
        return 1;
         
/*
 *  Now we check for a collision between the two tails.  We add the button
 *  clearance to the sizes to allow for it.
 */
    return (ThickLinesCollide(&tailA, fpcolINST_BUTTON_TD+buttClear, 
                              &tailB, fpcolINST_BUTTON_TD+buttClear));

}
/*
 * Internal Function name:
     fpcolBaseColFidBut
  
 * Description:
     OzPoz Fiducial-Button collision check routine.

     Here we are checking two things

     1.  For an overlap between two circles, one representing the
         button and the other the fiducial.

     2.  For a collision between a line segment representing the button
         tail and a circle representing a fiducial.
  
 * History:
     13-Dec-1999 - TJF - Original 6dF version.
     08-Jun-2000 - TJF - OzPoz version
  
 */
extern int  fpcolBaseColFidBut (
        void * clientData UNUSED,
        long int  butx,      /* The button X ordinate   */
        long int  buty,      /* The button Y ordinate   */
        double    theta,     /* The button theta        */
        long int  markx,     /* The fiducial X ordinate */
        long int  marky,     /* The fiducial Y ordinate */
        long int  fidClear,  /* The tolerance           */
        double    fidClearSqrd UNUSED)/* Tolerance squared       */
{
    circleType button;           /* Circle representing button   */
    circleType fiducial;         /* Circle representing fiducial */

    pointType  buttonLocation;   /* Point for button a           */
    pointType  fiducialLocation; /* Point for button b           */

    
/*
 *  Create the point types for the button and fiducial.
 */
    NewPoint(butx,  buty,  &buttonLocation);
    NewPoint(markx, marky, &fiducialLocation);
    
/* 
 *  Create the circles representing each object.  We add the tolerance
 *  to the fiducial radius to account for the required clearance.
 */
    NewCircleP(buttonLocation,   fpcolINST_BUTTON_RAD,       &button);
    NewCircleP(fiducialLocation, fpcolINST_FID_RAD+fidClear, &fiducial);

/*
 *  Check for overlap between the buttons themselves.
 */
    if (CirclesOverlap(&button, &fiducial))
        return 1;

    else
    {
/*
 *      Create a line describing the button tail.
 */
        lineSegType  tail;           
        NewLineSegAngle(buttonLocation, MathsTheta(theta), fpcolINST_BUTTON_TAIL, &tail);
/*
 *      Check for overlap between the tail and the fiducial
 */        
        return (CircleLineOverlap(&fiducial, &tail));
    }
    
}
 
/*
 * Internal Function name:
     fpcolBaseColInvPos
  
 * Description:
     OzPoz Invalid position routine 

     The only invalid position in OzPoz is the location of the ARGUS unit.  Whilst
     this is only on one plate, we intent to assume it is on both to enusre configurations
     will work on both.

     Our first check is for overlap between two circles, one presenting the button and
     one representing the button tail and one enclosing the Argus unit. 

     Later we may add more detailed checking around Argus to allow more valid positoins.
  
     
     For FACB and IFUs an additional check has been added:
     FACBs R range must be between fpcolINST_FACB_MIN_RADIUS and fpcolINST_FACB_MAX_RADIUS,
     IFUs  R range must be between fpcolINST_IFU_MIN_RADIUS  and fpcolINST_IFU_MAX_RADIUS.

 * History:
     13-Dec-1999 - TJF - Original 6dF version.
     19-Apr-2000 - TJf - Add plate argument, but this is not used.
     08-Jun-2000 - TJF - OzPoz version
     21-Dec-2002 - RSC - fibreType argument added, check valid R range
  
 */

/* these #defines must be the same as in FpilExtras.c ... */
#define FLAMES_FIBRE_TYPE_GUIDE   0
#define FLAMES_FIBRE_TYPE_IFU     1

extern int  fpcolBaseColInvPos (
        void * clientData UNUSED,
        unsigned plate UNUSED,    /* Plate to check on */
	int	  fibreType,	  /* Fibre type */
        long int  butx,           /* Button X position */
        long int  buty,           /* Button Y position */
        double    theta  UNUSED,  /* Button theta (Not needed */
        long int  tolerance,      /* Tolerance to allow (button tolerance) */
        double    tolSqrd UNUSED) /* Tolerance squared */
        
{
    circleType button;
    circleType invPos;

    NewCircleXYR(butx, buty, tolerance, &button);

    /* 1. check for ARGUS overlap */
    NewCircleXYR(0,    0,    fpcolINST_ARGUS_RAD, &invPos);
    if (CirclesOverlap(&button, &invPos))
    	return 1;

    /* 2. check FACB R range */
    if (fibreType == FLAMES_FIBRE_TYPE_GUIDE)
    	{
        NewCircleXYR(0,    0,    fpcolINST_FACB_MIN_RADIUS, &invPos);
    	if (CirclesOverlap(&button, &invPos))
    	    return 1;
        NewCircleXYR(0,    0,    fpcolINST_FACB_MAX_RADIUS - 2 * tolerance + 1, &invPos);
    	if (CirclesOverlap(&button, &invPos) == 0)
    	    return 1;
	}
    
    /* 3. check IFU R range */
    if (fibreType == FLAMES_FIBRE_TYPE_IFU)
    	{
        NewCircleXYR(0,    0,    fpcolINST_IFU_MIN_RADIUS, &invPos);
    	if (CirclesOverlap(&button, &invPos))
    	    return 1;
        NewCircleXYR(0,    0,    fpcolINST_IFU_MAX_RADIUS - 2 * tolerance + 1, &invPos);
    	if (CirclesOverlap(&button, &invPos) == 0)
    	    return 1;
	}
    
    return 0;
}



/*___oOo___*/
