# $Id$

"""
This test demonstrates conservative regridding with destination masking
on a grid.

Two Field objects are created, one on a Grid and the other on a Mesh.  
The Grid receives a mask which is used later in the regridding.  The
source Field is set to an analytic function, and a regridding operation 
is performed from the source to the destination Field.  After the 
regridding is completed, the destination Field is compared to the exact 
solution over that domain.
"""

try:
    import numpy as np
except:
    raise ImportError('The Numpy library cannot be found!')

try:
    import ESMF
except:
    raise ImportError('The ESMF library cannot be found!')

from ESMF.test.regrid_test.mesh_regridding_utilities import *
from ESMF.test.regrid_test.grid_regridding_utilities import *

esmp = ESMF.Manager(logkind=ESMF.LogKind.MULTI, debug=True)

parallel = False
if ESMF.pet_count() > 1:
    if ESMF.pet_count() != 4:
        raise NameError('MPI rank must be 4 in parallel mode!')
    parallel = True

# opening remarks
if ESMF.local_pet() == 0:
    print "\ngrid_mesh_regrid_mask"

# create a grid
grid = grid_create([0,0,8,8], [0,0,4,4], domask=True)

# create a Mesh
if parallel:
    mesh, nodeCoord, nodeOwner, elemType, elemConn = \
        mesh_create_50_parallel()
else:
    mesh, nodeCoord, nodeOwner, elemType, elemConn = \
        mesh_create_50()

# create Field objects
srcfield = ESMF.Field(mesh, 'srcfield')
dstfield = ESMF.Field(grid, 'dstfield', mask_values=[0])
exactfield = ESMF.Field(grid, 'exactfield', mask_values=[0])

# initialize the Fields to an analytic function
srcfield = initialize_field_mesh(srcfield, nodeCoord, nodeOwner, elemType, elemConn)
exactfield = initialize_field_grid(exactfield, domask=True)

# run the ESMF regridding
regridSrc2Dst = ESMF.Regrid(srcfield, dstfield, 
                            dst_mask_values=np.array([0]), \
                            regrid_method=ESMF.RegridMethod.BILINEAR, 
                            unmapped_action=ESMF.UnmappedAction.IGNORE)
dstfield = regridSrc2Dst(srcfield, dstfield)

# compare results and output PASS or FAIL
compare_fields_grid(dstfield, exactfield, 20E-1, 10E-16, parallel=parallel, regrid_method=ESMF.RegridMethod.BILINEAR)
