! $Id$
!
! Earth System Modeling Framework
! Copyright 2002-2014, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
!
#define ESMF_FILENAME "ESMF_IOScrip.F90"
!
!     ESMF IOScrip Module
      module ESMF_IOScripMod
!
!------------------------------------------------------------------------------
! INCLUDES
#include "ESMF.h"
!==============================================================================
!BOPI
! !MODULE: ESMF_IOScripMod - Grid IO utility class
!
! !DESCRIPTION:
!
! The code in this file reads the SCRIP Grid files and write out Scrip Weight
! files
!
!------------------------------------------------------------------------------
! !USES:
      use ESMF_UtilTypesMod   
      use ESMF_UtilMod
      use ESMF_InitMacrosMod    ! ESMF initializer macros
      use ESMF_LogErrMod        ! ESMF error handling
      use ESMF_VMMod
      use ESMF_IOUGridMod
      use ESMF_IOGridspecMod
#ifdef ESMF_NETCDF
      use netcdf
#endif

!     NEED TO ADD MORE HERE
      implicit none

!------------------------------------------------------------------------------
! !PRIVATE:
      private
!------------------------------------------------------------------------------
!
! !PUBLIC MEMBER FUNCTIONS:
!
! - ESMF-public methods:
  public ESMF_ScripInq
  public ESMF_ScripInqUnits
  public ESMF_ScripGetVar
  public ESMF_OutputScripWeightFile
  public ESMF_EsmfGetNode
  public ESMF_EsmfGetElement
  public ESMF_OutputScripVarFile
  public ESMF_EsmfInq

!==============================================================================

      contains

!==============================================================================

! -------------------------- ESMF-public method -------------------------------
!------------------------------------------------------------------------------
#undef  ESMF_METHOD
#define ESMF_METHOD "ESMF_ScripInq"
!BOPI
! !ROUTINE: ESMF_ScripInq: Return the dimension, grid_rank, and other related 
!  information from a SCRIP file
!
! !INTERFACE:
  subroutine ESMF_ScripInq(filename, grid_size, grid_corners, &
			  grid_dims, grid_rank,rc)

! !ARGUMENTS:
 
    character(len=*), intent(in)  :: filename
    integer, intent(out), optional :: grid_size
    integer, intent(out), optional :: grid_corners
    integer, pointer, optional     :: grid_dims(:)
    integer, intent(out), optional :: grid_rank
    integer, intent(out), optional :: rc

    integer:: localrc, ncStatus
    integer :: DimId, VarId
    integer :: ncid, local_rank
    character(len=256) :: errmsg

#ifdef ESMF_NETCDF
    ncStatus = nf90_open (path=trim(filename), mode=nf90_nowrite, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, &
      trim(filename), &
      rc)) return

    ! get number of vertices
    errmsg = "dimension grid_size in "//trim(filename)
    if (present(grid_size)) then
      ncStatus = nf90_inq_dimid (ncid, "grid_size", DimId)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return

      ncStatus = nf90_inquire_dimension (ncid, DimId, len=grid_size)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return
    end if

    ! Get vertex dimension
    if (present(grid_corners)) then
      ncStatus = nf90_inq_dimid (ncid, "grid_corners", DimId)
      errmsg = "dimension grid_corners in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return

      ncStatus = nf90_inquire_dimension (ncid, DimId, len=grid_corners)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg, &
        rc)) return
    end if
      
    ! Get vertex dimension
    if (present(grid_rank)) then
      ncStatus = nf90_inq_dimid (ncid, "grid_rank", DimId)
      errmsg = "dimension grid_rank in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return

      ncStatus = nf90_inquire_dimension (ncid, DimId, len=local_rank)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return
      grid_rank = local_rank
    end if
      
    ! get grid dimension
    ! If we didn't get the rank earlier, get it now
    if (present(grid_dims)) then
      if (.not. present(grid_rank)) then
        ncStatus = nf90_inq_dimid (ncid, "grid_rank", DimId)
        errmsg = "dimension grid_rank in "//trim(filename)
        if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
          errmsg,&
          rc)) return

        ncStatus = nf90_inquire_dimension (ncid, DimId, len=local_rank)
        if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
          errmsg,&
        rc)) return
      end if   
      allocate(grid_dims(local_rank))
      ncStatus = nf90_inq_varid (ncid, "grid_dims", VarId)
      errmsg = "dimension grid_dims in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return

      ncStatus = nf90_get_var (ncid, VarId, grid_dims)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return
    end if
    ncStatus = nf90_close(ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, &
      trim(filename), &
      rc)) return
    if (present(rc)) rc=ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
    return
#endif

end subroutine ESMF_ScripInq

#undef  ESMF_METHOD
#define ESMF_METHOD "ESMF_ScripInqUnits"
!BOPI
! !ROUTINE: ESMF_ScripInqUnits: Return the units attribute for
! coordinate variables, assuming all the coordinate variables
! have the same units
!
! !INTERFACE:
subroutine ESMF_ScripInqUnits(filename, units, rc)

! !ARGUMENTS:

    character(len=*), intent(in)   :: filename
    character(len=*), intent(out)   :: units
    integer, intent(out), optional :: rc

    integer:: localrc, ncStatus
    integer :: ncid, VarId, len
    character(len=80) :: buffer, buffer1
    character(len=256) :: errmsg

#ifdef ESMF_NETCDF
    if (present(rc)) rc=ESMF_SUCCESS
    ncStatus = nf90_open (path=trim(filename), mode=nf90_nowrite, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD, &
      ESMF_SRCLINE, &
      trim(filename), &
      rc)) return

    ncStatus = nf90_inq_varid (ncid, "grid_center_lat", VarId)
    errmsg = "variable grid_center_lat in "//trim(filename)
    if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
 
    ncStatus = nf90_inquire_attribute(ncid, VarId, "units", len=len)
    errmsg = "attribute units for grid_center_lat in "//trim(filename)
    if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
    ncStatus = nf90_get_att(ncid, VarId, "units", buffer)
    if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
    
    if (buffer(len:len) .eq. achar(0)) len = len-1
    call ESMF_StringLowerCase(buffer(1:len), rc=rc)
    units = buffer(1:len)
    if (present(rc)) rc=ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
#endif

    return
end subroutine ESMF_ScripInqUnits

#undef  ESMF_METHOD
#define ESMF_METHOD "ESMF_ScripGetVar"
!BOPI
! !ROUTINE: ESMF_ScripGetVar
!

! !INTERFACE:
  subroutine ESMF_ScripGetVar(filename, grid_center_lon, grid_center_lat, &
	      		grid_imask, grid_corner_lon, grid_corner_lat, &
			grid_area, convertToDeg, start, count, rc)
!
! !ARGUMENTS:
    character(len=*), intent(in)  :: filename
    real(ESMF_KIND_R8), intent(inout),optional:: grid_center_lon(:)
    real(ESMF_KIND_R8), intent(inout),optional:: grid_center_lat(:)
    integer, intent(inout), optional:: grid_imask(:)
    real(ESMF_KIND_R8), intent(inout),optional:: grid_corner_lon(:,:)
    real(ESMF_KIND_R8), intent(inout),optional:: grid_corner_lat(:,:)
    real(ESMF_KIND_R8), intent(inout),optional:: grid_area(:)
    logical, intent(in), optional:: convertToDeg
    integer, intent(in), optional:: start, count
    integer, intent(out), optional:: rc

    integer:: ncStatus
    integer:: ncid, VarId
    integer:: len
    character(len=128) :: units
    character(len=256) :: errmsg
    integer:: start1(1), count1(1), start2(2), count2(2)
    integer:: totalcells, grid_corners
    logical :: convertToDegLocal
    integer:: localrc

#ifdef ESMF_NETCDF
    convertToDegLocal = .false.
    if (present(convertToDeg)) convertToDegLocal = convertToDeg

    call ESMF_ScripInq(trim(filename), grid_size=totalcells, &
           grid_corners=grid_corners, rc=localrc)
    if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return

    ncStatus = nf90_open (path=trim(filename), mode=nf90_nowrite, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD, &
      ESMF_SRCLINE,&
      trim(filename), &
      rc)) return

    if (present(start)) then
	start1(1)=start
        start2(2)=start
        start2(1)=1
    else
	start1(1)=1
	start2(:)=1
    endif 
    if (present(count)) then
        count1(1)=count
        count2(2)=count
	count2(1)=grid_corners
    else
	count1(1)=totalcells
        count2(2)=totalcells
        count2(1)=grid_corners
    endif

    ! Read in grid_center_lon and grid_center_lat
    if (present(grid_center_lon)) then
      ncStatus = nf90_inq_varid (ncid, "grid_center_lon", VarId)
      errmsg = "variable grid_center_lon in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ncStatus = nf90_get_var (ncid, VarId, grid_center_lon, start1, count1)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return

      ! get the attribute 'units'
      ncStatus = nf90_inquire_attribute(ncid, VarId, "units", len=len)
      errmsg = "Attribute units for grid_center_lon in "//trim(filename)
      if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
	  errmsg,&
          rc)) return
      ncStatus = nf90_get_att(ncid, VarId, "units", units)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return
      ! if units is not "degrees" or "radians" return errors
      if (units(len:len) .eq. achar(0)) len = len-1
      call ESMF_StringLowerCase(units(1:len), rc=rc)
      if (units(1:len) .ne. 'degrees' .and. units(1:len) .ne. 'radians') then
          call ESMF_LogSetError(rcToCheck=ESMF_FAILURE, & 
                 msg="- units attribute is not degrees or radians", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
          return
      endif
      ! if units is "radians", convert it to degrees
      if (convertToDegLocal) then
         if (units(1:len) .eq. "radians") then
            grid_center_lon(:) = &
                   grid_center_lon(:)*ESMF_COORDSYS_RAD2DEG
         endif
      endif	   
    endif
    if (present(grid_center_lat)) then
      ncStatus = nf90_inq_varid (ncid, "grid_center_lat", VarId)
      errmsg = "variable grid_center_lat in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ncStatus = nf90_get_var (ncid, VarId, grid_center_lat, start1, count1)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ! get the attribute 'units'
      ncStatus = nf90_inquire_attribute(ncid, VarId, "units", len=len)
      errmsg = "Attribute units for grid_center_lat in "//trim(filename)
      if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
	  errmsg,&
          rc)) return
      ncStatus = nf90_get_att(ncid, VarId, "units", units)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ! if units is not "degrees" or "radians" return errors
     if (units(len:len) .eq. achar(0)) len = len-1
      call ESMF_StringLowerCase(units(1:len), rc=rc)
      if (units(1:len) .ne. 'degrees' .and. units(1:len) .ne. 'radians') then
          call ESMF_LogSetError(rcToCheck=ESMF_FAILURE, & 
                 msg="- units attribute is not degrees or radians", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
          return
      endif
      ! if units is "radians", convert it to degree
      if (convertToDegLocal) then
         if (units(1:len) .eq. "radians") then
            grid_center_lat(:) = &
                 grid_center_lat(:)*ESMF_COORDSYS_RAD2DEG
         endif
      endif
    endif

    if (present(grid_imask)) then
      ncStatus = nf90_inq_varid (ncid, "grid_imask", VarId)
      errmsg = "variable grid_imask in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
	grid_imask=1
      ncStatus = nf90_get_var (ncid, VarId, grid_imask, start1, count1)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
    endif

    ! Read in grid_corner_lon and grid_corner_lat
    if (present(grid_corner_lon)) then
      ncStatus = nf90_inq_varid (ncid, "grid_corner_lon", VarId)
      errmsg = "variable grid_corner_lon in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ncStatus = nf90_get_var (ncid, VarId, grid_corner_lon, start2, count2)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ! get the attribute 'units'
      ncStatus = nf90_inquire_attribute(ncid, VarId, "units", len=len)
      errmsg = "attribute units for grid_center_lon in "//trim(filename)
      if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
	  errmsg,&
          rc)) return
      ncStatus = nf90_get_att(ncid, VarId, "units", units)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ! if units is not "degrees" or "radians" return errors
     if (units(len:len) .eq. achar(0)) len = len-1
      call ESMF_StringLowerCase(units(1:len), rc=rc)
      if (units(1:len) .ne. 'degrees' .and. units(1:len) .ne. 'radians') then
          call ESMF_LogSetError(rcToCheck=ESMF_FAILURE, & 
                 msg="- units attribute is not degrees or radians", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
          return
      endif
      ! if units is "radians", convert it to degree
      if (convertToDegLocal) then
         if (units(1:len) .eq. "radians") then
            grid_corner_lon(:,:) = &
               grid_corner_lon(:,:)*ESMF_COORDSYS_RAD2DEG
         endif
      endif	   
    endif
    if (present(grid_corner_lat)) then
      ncStatus = nf90_inq_varid (ncid, "grid_corner_lat", VarId)
      errmsg = "variable grid_corner_lat in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ncStatus = nf90_get_var (ncid, VarId, grid_corner_lat, start2, count2)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ! get the attribute 'units'
      ncStatus = nf90_inquire_attribute(ncid, VarId, "units", len=len)
      errmsg = "Attribute units for grid_center_lon in "//trim(filename)
      if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
	  errmsg,&
          rc)) return
      ncStatus = nf90_get_att(ncid, VarId, "units", units)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
        errmsg,&
        rc)) return
      ! if units is not "degrees" or "radians" return errors
     if (units(len:len) .eq. achar(0)) len = len-1
      call ESMF_StringLowerCase(units(1:len), rc=rc)
      if (units(1:len) .ne. 'degrees' .and. units(1:len) .ne. 'radians') then
          call ESMF_LogSetError(rcToCheck=ESMF_FAILURE, & 
                 msg="- units attribute is not degrees or radians", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
          return
      endif
      ! if units is "radians", convert it to degree
      if (convertToDegLocal) then
         if (units(1:len) .eq. "radians") then
            grid_corner_lat(:,:) = &
                grid_corner_lat(:,:)*ESMF_COORDSYS_RAD2DEG
         endif
      endif
    endif

    ! Read in grid_area and grid_corner_lat
    if (present(grid_area)) then
      ncStatus = nf90_inq_varid (ncid, "grid_area", VarId)
      errmsg = "variable grid_area in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
      ncStatus = nf90_get_var (ncid, VarId, grid_area, start1, count1)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,&
	errmsg,&
        rc)) return
    endif

    ncStatus = nf90_close(ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD, &
      ESMF_SRCLINE,&
      trim(filename),&
      rc)) return

    if(present(rc)) rc = ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
    return
#endif
    
end subroutine ESMF_ScripGetVar

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_OutputScripWeightFile"
!BOPI
! !ROUTINE: ESMF_OUtputScripWeightFile
! output the weight and indices tables together with the source/dest vertex 
! coordinates/masks to the SCRIP format NetCDF file
!

! !INTERFACE:
subroutine ESMF_OutputScripWeightFile (wgtFile, factorList, factorIndexList, &
				      srcFile, dstFile, srcFileType, dstFileType,&
				      title, method, normType, srcArea, dstArea, &
				      srcFrac, dstFrac, largeFileFlag, netcdf4FileFlag, &
				      srcmeshname, dstmeshname, & 
			              srcMissingValue, dstMissingValue, &
				      srcvarname, dstvarname, &
				      srccoordnames, dstcoordnames, rc)
!
! !ARGUMENTS:
      character(len=*), intent(in) :: wgtFile
      real(ESMF_KIND_R8) , intent(in) :: factorList(:)   
      integer(ESMF_KIND_I4) , intent(in), target :: factorIndexList(:,:) 
      character(len=*), optional, intent(in) :: srcFile
      character(len=*), optional, intent(in) :: dstFile
      type(ESMF_FileFormat_Flag), optional, intent(in) :: srcFileType
      type(ESMF_FileFormat_Flag), optional, intent(in) :: dstFileType
      character(len=*), optional, intent(in) :: title
      type(ESMF_RegridMethod_Flag), optional, intent(in) :: method
      type(ESMF_NormType_Flag),    intent(in),   optional :: normType
      real(ESMF_KIND_R8),optional, intent(in) :: srcArea(:),dstArea(:)
      real(ESMF_KIND_R8),optional, intent(in) :: srcFrac(:), dstFrac(:)
      logical, optional, intent(in) :: largeFileFlag, netcdf4FileFlag
      character(len=*), optional, intent(in) :: srcmeshname, dstmeshname
      logical, optional, intent(in) :: srcMissingValue, dstMissingValue
      character(len=*), optional, intent(in) :: srcvarname, dstvarname
      character(len=*), optional, intent(in) :: srccoordnames(:), dstcoordnames(:)
      integer, optional :: rc

      type(ESMF_VM):: vm
      integer :: PetNo, PetCnt

      integer :: total, localCount(1)
      integer :: ncid, ncid1
      integer :: ncStatus
      integer :: status
      integer :: i,j, k, start
      integer :: srcDim, dstDim
      integer:: naDimId, nbDimId, nsDimId, srankDimId, drankDimId, varId
      integer :: nvaDimId, nvbDimId
      integer :: src_grid_rank, dst_grid_rank, src_grid_corner, dst_grid_corner
      integer, pointer :: src_grid_dims(:), dst_grid_dims(:)
      integer :: src_ndims, dst_ndims
      integer :: src_coordids(2), dst_coordids(2)
      integer :: src_dimids(2), dst_dimids(2)
      real(ESMF_KIND_R8), pointer   :: coords(:),area(:),frac(:)
      real(ESMF_KIND_R8), pointer   :: latBuffer(:),lonBuffer(:)
      real(ESMF_KIND_R8), pointer   :: latBuffer1D(:),lonBuffer1D(:)
      real(ESMF_KIND_R8), pointer   :: latBuffer2(:,:),lonBuffer2(:,:)
      real(ESMF_KIND_R8), pointer   :: cornerlon2D(:,:), cornerlat2D(:,:)
      real(ESMF_KIND_R8), pointer   :: cornerlon3D(:,:,:), cornerlat3D(:,:,:)
      real(ESMF_KIND_R8), pointer   :: weightbuf(:), varBuffer(:,:)
      real(ESMF_KIND_R8), pointer   :: varBuffer1D(:)
      real(ESMF_KIND_R8) :: missing_value
      integer(ESMF_KIND_I4), pointer:: indexbuf(:), next(:)
      integer(ESMF_KIND_I4), pointer:: mask(:) 
      integer(ESMF_KIND_I4), pointer:: allCounts(:) 
      character(len=256) :: titlelocal, norm, map_method
      character(len=256) :: esmf_regrid_method,conventions
      type(ESMF_RegridMethod_Flag) :: methodlocal
      character(len=80) :: srcunits, dstunits
      integer :: maxcount
      type(ESMF_FileFormat_Flag) :: srcFileTypeLocal, dstFileTypeLocal
      integer :: srcNodeDim, dstNodeDim, srcCoordDim, dstCoordDim
      integer, parameter :: nf90_noerror = 0
      character(len=256) :: errmsg
      character(len=20) :: varStr
      type(ESMF_Logical) :: largeFileFlaglocal
      type(ESMF_Logical) :: netcdf4FileFlaglocal
      logical            :: faceCoordFlag
      logical            :: srchasbound, dsthasbound
      type(ESMF_NormType_Flag):: localNormType

#ifdef ESMF_NETCDF
      ! write out the indices and weights table sequentially to the output file
      ! first find out the starting index of my portion of table
      ! Global reduce

      if (present(srcFileType)) then
	  srcFileTypeLocal = srcFileType
      else 
	srcFileTypeLocal = ESMF_FILEFORMAT_SCRIP
      endif	

      if (present(dstFileType)) then
	  dstFileTypeLocal = dstFileType
      else 
	dstFileTypeLocal = ESMF_FILEFORMAT_SCRIP
      endif	
 
      if (present(largeFileFlag)) then
	largeFileFlaglocal = largeFileFlag
      else
        largeFileFlaglocal = .false.
      endif

      if (present(netcdf4FileFlag)) then
	netcdf4FileFlaglocal = netcdf4FileFlag
      else
        netcdf4FileFlaglocal = .false.
      endif

      ! Handle optional normType argument
      if (present(normType)) then
         localNormType=normType
      else     
         localNormType=ESMF_NORMTYPE_DSTAREA
      endif

      call ESMF_VMGetCurrent(vm, rc=rc)
      if (rc /= ESMF_SUCCESS) return

      ! set up local pet info
      call ESMF_VMGet(vm, localPet=PetNo, petCount=PetCnt, rc=rc)
      if (rc /= ESMF_SUCCESS) return

      localCount(1)=size(factorList,1)
      allocate(allCounts(PetCnt))
      call ESMF_VMAllGather(vm,localCount,allCounts,1,rc=status)
      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
           ESMF_CONTEXT, rcToReturn=rc)) return

      ! calculate the size of the global weight table
      total = 0
      do i=1,PetCnt
	 total=allCounts(i)+total
      end do
      !print *, PetNo, 'local count ', localCount(1), AllCounts(PetNo+1), total

     !Read the variables from the input grid files at PET0
      if (PetNo == 0) then
         ! Check if srcFile and dstFile exists
         if (.not. present(srcFile)) then
             call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, &
		  msg="- The srcFile argument does not exist on PET0 ", &
                  ESMF_CONTEXT, rcToReturn=rc)
	     return
         endif
         if (.not. present(dstFile)) then
             call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, &
		  msg="- The dstFile argument does not exist on PET0 ",  &
                  ESMF_CONTEXT, rcToReturn=rc)
	     return
         endif
         ! Create output file and create dimensions and variables
	 call c_nc_create(wgtFile, NF90_CLOBBER, &
		largeFileFlaglocal, netcdf4FileFlaglocal, ncid, status)
	 if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
		ESMF_CONTEXT, rcToReturn=rc)) return
         
         ! global variables
          if (present(title)) then
             titlelocal = trim(title)
          else
             titlelocal = "ESMF Offline Regridding Weight Generator"
          endif

          ! Norm type
          if (localNormType .eq. ESMF_NORMTYPE_DSTAREA) then
             norm = "destarea"
          elseif (localNormType .eq. ESMF_NORMTYPE_FRACAREA) then
             norm = "fracarea"
          else
             norm = "unknown"
          endif

         ! Regrid method
         if (present(method)) then
	   methodlocal = method
	   if (methodlocal%regridmethod == ESMF_REGRIDMETHOD_BILINEAR%regridmethod) then
              map_method = "Bilinear remapping"
	      esmf_regrid_method = "Bilinear"
	   elseif (methodlocal%regridmethod == ESMF_REGRIDMETHOD_PATCH%regridmethod) then
	      !scrip_test does not recognize patch remapping
              map_method = "Bilinear remapping"
	      esmf_regrid_method = "Higher-order Patch"
	   elseif (methodlocal%regridmethod == ESMF_REGRIDMETHOD_CONSERVE%regridmethod) then
              map_method = "Conservative remapping"
	      esmf_regrid_method = "First-order Conservative"
	   elseif (methodlocal%regridmethod == ESMF_REGRIDMETHOD_NEAREST_STOD%regridmethod) then
              map_method = "Bilinear remapping"
	      esmf_regrid_method = "Nearest source to destination"
	   elseif (methodlocal%regridmethod == ESMF_REGRIDMETHOD_NEAREST_DTOS%regridmethod) then
              map_method = "Bilinear remapping"
	      esmf_regrid_method = "Nearest destination to source"
	   else
	      !report error
              call ESMF_LogSetError(rcToCheck=ESMF_FAILURE, & 
                  msg="- regridmethod not recongized", & 
                  ESMF_CONTEXT, rcToReturn=rc) 
	      return
	   endif
         else
	   methodlocal = ESMF_REGRIDMETHOD_BILINEAR
           map_method = "Bilinear remapping"
         endif
         conventions = "NCAR-CSM"

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "title", trim(titlelocal))
         errmsg = "Attribute title in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "normalization", trim(norm))
         errmsg = "Attribute normalization in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "map_method", trim(map_method))
         errmsg = "Attribute map_method in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "ESMF_regrid_method", &
			trim(esmf_regrid_method))
         errmsg = "Attribute esmf_regrid_method in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "conventions", trim(conventions))
         errmsg = "Attribute conventions in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

	 ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "domain_a", trim(srcFile))
         errmsg = "Attribute domain_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "domain_b", trim(dstFile))
         errmsg = "Attribute domain_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "grid_file_src", trim(srcFile))
         errmsg = "Attribute grid_file_src in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "grid_file_dst", trim(dstFile))
         errmsg = "Attribute grid_file_dst in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_put_att(ncid, NF90_GLOBAL, "CVS_revision", ESMF_VERSION_STRING)
         errmsg = "Attribute CVS_revision in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
 
        ! Get Source Grid dimension and variables
	if (srcFileTypeLocal == ESMF_FILEFORMAT_SCRIP) then 
          call ESMF_ScripInq(srcFile, grid_rank=src_grid_rank, grid_size=srcDim, &
	      grid_dims=src_grid_dims, grid_corners=src_grid_corner, rc=status)
	  ! The grid_dims for an unstructured grie (grid_rank = 1) is not used
	  ! by SCRIP, thus some of the SCRIP files did not set this value correctly.
          ! This causes the scrip_test generating wrong output data, i.e.
          ! the grid coordinates will not be written in the weight file
          if (src_grid_rank == 1) src_grid_dims(1) = srcDim
          call ESMF_ScripInqUnits(srcFile,units = srcunits, rc=status)
	else if (srcFileTypeLocal == ESMF_FILEFORMAT_GRIDSPEC) then
          allocate(src_grid_dims(2))
          call ESMF_GridspecInq(srcFile, src_ndims, src_grid_dims, &
		dimids = src_dimids, coordids = src_coordids, &
		coord_names = srccoordnames, hasbound=srchasbound, rc=status)
          if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return
	  src_grid_rank = 2
          srcDim = src_grid_dims(1)*src_grid_dims(2)
	  if (src_ndims == 1) then
		src_grid_corner = 2
          else
	        src_grid_corner = 4
          endif
	  srcunits = 'degrees'
	else if (srcFileTypeLocal == ESMF_FILEFORMAT_ESMFMESH) then 
	  ! If bilinear, we have to switch node and elment, so the nodeCount became srcDim and
          ! elementCount becomes srcNodeDim. Hard code src_grid_corner to 3.  The xv_a and xv_b
          ! will be empty   
	  if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_BILINEAR%regridmethod .or. & 
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_PATCH%regridmethod .or. & 
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_STOD%regridmethod .or. & 
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_DTOS%regridmethod) then
            call ESMF_EsmfInq(srcFile, nodeCount=srcDim,  &
	        coordDim = srcCoordDim, elementCount=srcNodeDim, rc=status)
                src_grid_corner =3
	  else
            call ESMF_EsmfInq(srcFile, elementCount=srcDim, maxNodePElement=src_grid_corner, &
	        coordDim = srcCoordDim, nodeCount=srcNodeDim, rc=status)
	  endif
          call ESMF_EsmfInqUnits(srcFile,units = srcunits, rc=status)
          allocate(src_grid_dims(1))
          src_grid_dims(1)=srcDim
          src_grid_rank = 1    
	else if (srcFileTypeLocal == ESMF_FILEFORMAT_UGRID) then 
	  if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_BILINEAR%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_PATCH%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_STOD%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_DTOS%regridmethod) then
            call ESMF_UGridInq(srcFile, srcmeshname, nodeCount=srcDim,  &
	        elementCount=srcNodeDim, units=srcunits, rc=status)
                src_grid_corner =3
	  else
  	    call ESMF_UGridInq(srcFile, srcmeshname, elementCount=srcDim, &
	        maxNodePElement=src_grid_corner, units=srcunits, &
	        nodeCount = srcNodeDim, rc=status)
          endif
          if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return
	  srcCoordDim = 2
          allocate(src_grid_dims(1))
          src_grid_dims(1)=srcDim
          src_grid_rank = 1    
        endif 
        if (dstFileTypelocal == ESMF_FILEFORMAT_SCRIP) then
          call ESMF_ScripInq(dstFile, grid_rank=dst_grid_rank, grid_size=dstDim, &
	     grid_dims=dst_grid_dims, grid_corners=dst_grid_corner, rc=status)
	  ! The grid_dims for an unstructured grie (grid_rank = 1) is not used
	  ! by SCRIP, thus some of the SCRIP files did not set this value correctly.
          ! This causes the scrip_test generating wrong output data, i.e.
          ! the grid coordinates will not be written in the weight file
          if (dst_grid_rank == 1) dst_grid_dims(1) = dstDim
          call ESMF_ScripInqUnits(dstFile,units = dstunits, rc=status)
	else if (dstFileTypeLocal == ESMF_FILEFORMAT_GRIDSPEC) then
          allocate(dst_grid_dims(2))
          call ESMF_GridspecInq(dstFile, dst_ndims, dst_grid_dims, &
		dimids = dst_dimids, coordids = dst_coordids, &
		coord_names = dstcoordnames, hasbound=dsthasbound, rc=status)
          if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return
	  dst_grid_rank = 2
          dstDim = dst_grid_dims(1)*dst_grid_dims(2)
	  if (dst_ndims == 1) then
		dst_grid_corner = 2
          else
	        dst_grid_corner = 4
          endif
	  dstunits = 'degrees'
        else if (dstFileTypeLocal == ESMF_FILEFORMAT_ESMFMESH) then 
	  ! If bilinear, we have to switch node and elment, so the nodeCount became dstDim and
          ! elementCount becomes dstNodeDim. Hard code dst_grid_corner to 3.  The xv_a and xv_b
          ! will be empty   
	  if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_BILINEAR%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_PATCH%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_STOD%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_DTOS%regridmethod) then
            call ESMF_EsmfInq(dstFile, nodeCount=dstDim,  &
	        coordDim = dstCoordDim, elementCount=dstNodeDim, rc=status)
                dst_grid_corner =3
	  else
            call ESMF_EsmfInq(dstFile, elementCount=dstDim, maxNodePElement=dst_grid_corner, &
	      coordDim = dstCoordDim, nodeCount=dstNodeDim, rc=status)    
	  endif
          call ESMF_EsmfInqUnits(dstFile,units = dstunits, rc=status)
          allocate(dst_grid_dims(1))
          dst_grid_dims(1)=dstDim   
          dst_grid_rank = 1
	else if (dstFileTypeLocal == ESMF_FILEFORMAT_UGRID) then 
	  if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_BILINEAR%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_PATCH%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_STOD%regridmethod .or. &
              methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_DTOS%regridmethod) then
            call ESMF_UGridInq(dstFile, dstmeshname, nodeCount=dstDim,  &
	        elementCount=dstNodeDim, units=dstunits, rc=status)
                dst_grid_corner =3
	  else
  	    call ESMF_UGridInq(dstFile, dstmeshname, elementCount=dstDim, &
	        maxNodePElement=dst_grid_corner, units=dstunits, &
	        nodeCount = dstNodeDim, rc=status)
	  endif
          if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
	       	ESMF_CONTEXT, rcToReturn=rc)) return
	  dstCoordDim = 2
          allocate(dst_grid_dims(1))
          dst_grid_dims(1)=dstDim   
          dst_grid_rank = 1
        endif
        ! define dimensions
         ncStatus = nf90_def_dim(ncid,"n_a",srcDim, naDimId)
         errmsg = "Dimension n_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_def_dim(ncid,"n_b",dstDim, nbDimId)
         errmsg = "Dimension n_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_def_dim(ncid,"n_s",total, nsDimId)
         errmsg = "Dimension n_s in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! define max number of vertices
         ncStatus = nf90_def_dim(ncid,"nv_a",src_grid_corner, nvaDimId)
         errmsg = "Dimension nv_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_def_dim(ncid,"nv_b",dst_grid_corner, nvbDimId)
         errmsg = "Dimension nv_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! define max number of vertices
         ncStatus = nf90_def_dim(ncid,"num_wgts",1, VarId)
         errmsg = "Dimension num_wgts in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ! define grid ranks
         ncStatus = nf90_def_dim(ncid,"src_grid_rank",src_grid_rank, srankDimId)
         errmsg = "Dimension src_grid_rank in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_def_dim(ncid,"dst_grid_rank",dst_grid_rank, drankDimId)
         errmsg = "Dimension dst_grid_rank in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! define variables
         ! Grid Dims
         ncStatus = nf90_def_var(ncid,"src_grid_dims",NF90_INT, (/srankDimId/),  VarId)
         errmsg = "Variable src_grid_dims in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus = nf90_def_var(ncid,"dst_grid_dims",NF90_INT, (/drankDimId/),  VarId)
         errmsg = "Variable dst_grid_dims in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! yc_a: source vertex coordinate (latitude)
         ncStatus = nf90_def_var(ncid,"yc_a",NF90_DOUBLE, (/naDimId/),  VarId)
         errmsg = "Variable yc_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(srcunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! yc_b: destination vertex coordinate (latitude)
         ncStatus = nf90_def_var(ncid,"yc_b",NF90_DOUBLE, (/nbDimId/),  VarId)
         errmsg = "Variable yc_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(dstunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! xc_a: source vertex coordinate (longitude)
         ncStatus = nf90_def_var(ncid,"xc_a",NF90_DOUBLE, (/naDimId/),  VarId)
         errmsg = "Variable xc_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(srcunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! xc_b: dest. vertex coordinate (longitude)
         ncStatus = nf90_def_var(ncid,"xc_b",NF90_DOUBLE, (/nbDimId/),  VarId)
         errmsg = "Variable xc_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(dstunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! yv_a: source corner coordinate (latitude)
         ncStatus = nf90_def_var(ncid,"yv_a",NF90_DOUBLE, (/nvaDimId,naDimId/),  VarId)
         errmsg = "Variable yv_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(srcunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! xv_a: source corner coordinate (longitude)
         ncStatus = nf90_def_var(ncid,"xv_a",NF90_DOUBLE, (/nvaDimId,naDimId/),  VarId)
         errmsg = "Variable xv_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(srcunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! yv_b: source corner coordinate (latitude)
         ncStatus = nf90_def_var(ncid,"yv_b",NF90_DOUBLE, (/nvbDimId,nbDimId/),  VarId)
         errmsg = "Variable yv_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(dstunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! xv_b: source corner coordinate (longitude)
         ncStatus = nf90_def_var(ncid,"xv_b",NF90_DOUBLE, (/nvbDimId,nbDimId/),  VarId)
         errmsg = "Variable xv_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", trim(dstunits))
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! mask_a
         ncStatus = nf90_def_var(ncid,"mask_a",NF90_INT, (/naDimId/),  VarId)
         errmsg = "Variable mask_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", "unitless")
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! mask_b
         ncStatus = nf90_def_var(ncid,"mask_b",NF90_INT, (/nbDimId/),  VarId)
         errmsg = "Variable mask_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", "unitless")
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! area_a
         ncStatus = nf90_def_var(ncid,"area_a",NF90_DOUBLE, (/naDimId/),  VarId)
         errmsg = "Variable area_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", "square radians")
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! area_b
         ncStatus = nf90_def_var(ncid,"area_b",NF90_DOUBLE, (/nbDimId/),  VarId)
         errmsg = "Variable area_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", "square radians")
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! frac_a
         ncStatus = nf90_def_var(ncid,"frac_a",NF90_DOUBLE, (/naDimId/),  VarId)
         errmsg = "Variable frac_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", "unitless")
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! frac_b
         ncStatus = nf90_def_var(ncid,"frac_b",NF90_DOUBLE, (/nbDimId/),  VarId)
         errmsg = "Variable frac_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
         ncStatus = nf90_put_att(ncid, VarId, "units", "unitless")
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return


        ! col: sparse matrix weight table
         ncStatus = nf90_def_var(ncid,"col",NF90_INT, (/nsDimId/),  VarId)
         errmsg = "Variable col in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! row: sparse matrix weight table
         ncStatus = nf90_def_var(ncid,"row",NF90_INT, (/nsDimId/),  VarId)
         errmsg = "Variable row in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

        ! S: sparse matrix weight table
         ncStatus = nf90_def_var(ncid,"S",NF90_DOUBLE, (/nsDimId/),  VarId)
         errmsg = "Variable S in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

         ncStatus=nf90_enddef(ncid) 
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   trim(wgtFile),&
           rc)) return
       
        ! write src_grid_dims and dst_grid_dims
	ncStatus=nf90_inq_varid(ncid,"src_grid_dims",VarId)
        ncStatus=nf90_put_var(ncid,VarId, src_grid_dims )          
        errmsg = "Variable src_grid_dims in "//trim(wgtfile)
        if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
	  errmsg,&
          rc)) return

	ncStatus=nf90_inq_varid(ncid,"dst_grid_dims",VarId)
        ncStatus=nf90_put_var(ncid,VarId, dst_grid_dims)          
        errmsg = "Variable dst_grid_dims in "//trim(wgtfile)
        if (CDFCheckError (ncStatus, &
          ESMF_METHOD, &
          ESMF_SRCLINE,&
	  errmsg,&
          rc)) return

        if (srcFileTypeLocal == ESMF_FILEFORMAT_SCRIP) then
           ! Read the srcGrid variables and write them out
           allocate(latBuffer(srcDim), lonBuffer(srcDim))
           call ESMF_ScripGetVar(srcFile, grid_center_lon=lonBuffer, &
	 	grid_center_lat=latBuffer, rc=status)
           if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return

	   ncStatus=nf90_inq_varid(ncid,"xc_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
           errmsg = "Variable xc_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return

           ncStatus=nf90_inq_varid(ncid,"yc_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
           errmsg = "Variable yc_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(latBuffer, lonBuffer)
          ! Write xv_a, yv_a	
           allocate(latBuffer2(src_grid_corner,srcDim),lonBuffer2(src_grid_corner,srcDim))         
           call ESMF_ScripGetVar(srcFile, grid_corner_lon=lonBuffer2, &
	 	grid_corner_lat=latBuffer2, rc=status)
           if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return

	   ncStatus=nf90_inq_varid(ncid,"xv_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)          
           errmsg = "Variable xv_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return

           ncStatus=nf90_inq_varid(ncid,"yv_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, latBuffer2)          
           errmsg = "Variable yv_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(latBuffer2, lonBuffer2) 

           allocate(mask(srcDim))         
           call ESMF_ScripGetVar(srcFile, grid_imask=mask, rc=status)
           if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return
           ncStatus=nf90_inq_varid(ncid,"mask_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, mask)          
           errmsg = "Variable mask_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(mask)
        else if (srcFileTypeLocal == ESMF_FILEFORMAT_GRIDSPEC) then 
           allocate(lonBuffer(srcDim), latBuffer(srcDim))
           if (src_ndims == 1) then
	     allocate(lonBuffer1D(src_grid_dims(1)), latBuffer1D(src_grid_dims(2)))
             if (srchasbound) then
 	        allocate(cornerlon2D(src_grid_corner,src_grid_dims(1)), &
		      cornerlat2D(src_grid_corner,src_grid_dims(2)))
	        call ESMF_GridspecGetVar1D(srcFile, src_coordids, lonBuffer1D, latBuffer1D, &
	              cornerlon = cornerlon2D, cornerlat=cornerlat2D, rc=status)
                if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
          	      ESMF_CONTEXT, rcToReturn=rc)) return
             else 
	        call ESMF_GridspecGetVar1D(srcFile, src_coordids, lonBuffer1D, latBuffer1D, &
	              rc=status)
                if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
          	      ESMF_CONTEXT, rcToReturn=rc)) return
             endif
             k=1
             do j=1,src_grid_dims(2)
	      do i=1,src_grid_dims(1)
                lonBuffer(k)=lonBuffer1D(i)
                latBuffer(k)=latBuffer1D(j)
	        k=k+1
              enddo
            enddo
	    deallocate(lonBuffer1D, latBuffer1D)
	    if (srchasbound) then
 	       allocate(lonBuffer2(src_grid_rank, srcDim),latBuffer2(src_grid_rank,srcDim))
	       k=1
               do j=1,src_grid_dims(2)
	         do i=1,src_grid_dims(1)
                   lonBuffer2(:,k)=cornerlon2D(:,i)
                   latBuffer2(:,k)=cornerlat2D(:,j)
	           k=k+1
                 enddo
               enddo
            endif
      	  else
	     allocate(lonBuffer2(src_grid_dims(1),src_grid_dims(2)), &
		      latBuffer2(src_grid_dims(1),src_grid_dims(2)))
             if (srchasbound) then		      
                allocate(cornerlon3D(src_grid_corner,src_grid_dims(1), src_grid_dims(2)),&
		    cornerlat3D(src_grid_corner,src_grid_dims(1), src_grid_dims(2)))         
 	        call ESMF_GridspecGetVar2D(srcFile, src_coordids, lonBuffer2, latBuffer2, &
		    cornerlon=cornerlon3D, cornerlat=cornerlat3D, rc=status)
                if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
          	   ESMF_CONTEXT, rcToReturn=rc)) return
             else 
 	        call ESMF_GridspecGetVar2D(srcFile, src_coordids, lonBuffer2, latBuffer2, &
		    rc=status)
                if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
          	   ESMF_CONTEXT, rcToReturn=rc)) return
             endif
             lonBuffer = reshape(lonBuffer2, (/srcDim/))
             latBuffer = reshape(latBuffer2, (/srcDim/))
 	     deallocate(lonBuffer2, latBuffer2)
             if (srchasbound) then
  	        allocate(lonBuffer2(src_grid_rank, srcDim),latBuffer2(src_grid_rank,srcDim))
                lonBuffer2=reshape(cornerlon3D, (/src_grid_rank, srcDim/)) 
                latBuffer2=reshape(cornerlat3D, (/src_grid_rank, srcDim/)) 
                deallocate(cornerlon3D, cornerlat3D)
             endif
	   endif

	   ncStatus=nf90_inq_varid(ncid,"xc_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
           errmsg = "Variable xc_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return

           ncStatus=nf90_inq_varid(ncid,"yc_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
           errmsg = "Variable yc_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(latBuffer, lonBuffer)
	   
          ! Write xv_a, yv_a	
	   if (srchasbound) then
	   ncStatus=nf90_inq_varid(ncid,"xv_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)
           errmsg = "Variable xv_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return

           ncStatus=nf90_inq_varid(ncid,"yv_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, latBuffer2)
           errmsg = "Variable yv_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(lonBuffer2, latBuffer2)
           endif

           ! Mask
           allocate(mask(srcDim))         
	   mask(:)=1
	   if (present(srcMissingValue)) then
	     if (srcMissingValue) then
              allocate(varBuffer(src_grid_dims(1),src_grid_dims(2)))
              call ESMF_GridspecGetVarByName(srcFile, srcvarname, src_dimids, &
			        varBuffer, missing_value = missing_value, &
                                rc=status)
              if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
	      k=1
	      do i=1,size(varBuffer,2)
	        do j=1,size(varBuffer,1)
	           if (varBuffer(j,i) == missing_value) mask(k)=0
	           k=k+1
                enddo	
              enddo
	      deallocate(varBuffer)
            endif
           endif

           ncStatus=nf90_inq_varid(ncid,"mask_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, mask)          
           errmsg = "Variable mask_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(mask)
            
        else if (srcFileTypeLocal == ESMF_FILEFORMAT_ESMFMESH) then 
           ! ESMF unstructured grid
           ncStatus=nf90_open(srcFile,NF90_NOWRITE,ncid1)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
	     trim(srcFile),&
             rc)) return
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_BILINEAR%regridmethod &
		.or. methodlocal%regridmethod ==ESMF_REGRIDMETHOD_PATCH%regridmethod &
		.or. methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_STOD%regridmethod &
		.or. methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_DTOS%regridmethod) then
	     ! check if centerCoords exit
             ncStatus=nf90_inq_varid(ncid1,"nodeCoords",VarId)
	     varStr = "nodeCoords"
	   else
             ncStatus=nf90_inq_varid(ncid1,"centerCoords",VarId)
	     varStr = "centerCoords"
           endif
	   if (ncStatus /= nf90_noerror) then
	     write(*,*) "Warning: "//trim(varStr)// &
                " not present in src grid file, so not outputting xc_a and yc_a to weight file."
             write(*,*)
           else 
	     allocate(latBuffer2(srcCoordDim, srcDim))
             allocate(latBuffer(srcDim), lonBuffer(srcDim))
             ncStatus=nf90_get_var(ncid1,VarId, latBuffer2)
             errmsg = "Variable "//trim(varStr)//" in "//trim(srcFile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,&
	       errmsg,&
               rc)) return
	     do i=1,srcDim
		lonBuffer(i)=latBuffer2(1,i)
                latBuffer(i)=latBuffer2(2,i)
             enddo
   	     ncStatus=nf90_inq_varid(ncid,"xc_a",VarId)
             ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
             errmsg = "Variable xc_a in "//trim(wgtfile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,&
	       errmsg,&
               rc)) return

             ncStatus=nf90_inq_varid(ncid,"yc_a",VarId)
             ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
             errmsg = "Variable yc_a in "//trim(wgtfile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,&
    	       errmsg,&
               rc)) return
             deallocate(latBuffer, lonBuffer, latBuffer2)
           endif

           ! only write out xv_a and yv_a when the regrid method is conserve
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_CONSERVE%regridmethod) then 
           ! output xv_a and yv_a is harder, we have to read in the nodeCoords and
           ! elementConn and construct the the latitudes nd longitudes for
           ! all the corner vertices
             allocate(latBuffer2(src_grid_corner,srcDim),lonBuffer2(src_grid_corner,srcDim))         
             call ESMF_EsmfGetVerts(ncid1, srcFile, srcDim, src_grid_corner, srcNodeDim, &
		  latBuffer2, lonBuffer2,status) 
             if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	  ESMF_CONTEXT, rcToReturn=rc)) return

	     ncStatus=nf90_inq_varid(ncid,"xv_a",VarId)
             ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)          
             errmsg = "Variable xv_a in "//trim(wgtfile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE, errmsg, &
               rc)) return

             ncStatus=nf90_inq_varid(ncid,"yv_a",VarId)
             ncStatus=nf90_put_var(ncid,VarId, latBuffer2)          
             errmsg = "Variable yv_a in "//trim(wgtfile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return
             deallocate(latBuffer2, lonBuffer2) 
           endif
           allocate(mask(srcDim))         
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_CONSERVE%regridmethod) then 
             ncStatus=nf90_inq_varid(ncid1,"elementMask",VarId)
	     if (ncStatus /= nf90_noerror) then
               write(*,*) "Warning: elementMask"// &
                " not present in src grid file, so setting mask_a=1 in weight file."
               write(*,*)
               mask = 1
             else 
               ncStatus=nf90_get_var(ncid1,VarId, mask)
               errmsg = "Variable elementMask in "//trim(srcFile)
               if (CDFCheckError (ncStatus, &
                 ESMF_METHOD, &
                 ESMF_SRCLINE,errmsg,&
                 rc)) return
              end if
	   else
	      mask = 1
           endif   
           ncStatus=nf90_inq_varid(ncid,"mask_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, mask)          
           errmsg = "Variable mask_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,errmsg,&
             rc)) return
           deallocate(mask)
           ncStatus=nf90_close(ncid1)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,trim(srcFile),&
             rc)) return
	else if (srcFileTypeLocal == ESMF_FILEFORMAT_UGRID) then 
           ! ESMF unstructured grid
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_CONSERVE%regridmethod) then 
	     ! check if faceCoords exit
	      call ESMF_UGridInq(srcfile, srcmeshname, faceCoordFlag=faceCoordFlag)
	      allocate(latBuffer2(src_grid_corner,srcDim),&
	   	       lonBuffer2(src_grid_corner,srcDim)) 
              if (faceCoordFlag) then
 	        allocate(latBuffer(srcDim), lonBuffer(srcDim))
  	        call ESMF_UGridGetVar(srcfile, srcmeshname, &
		   faceXcoords=lonBuffer, faceYcoords=latBuffer, &
		   faceNodeConnX=lonBuffer2, faceNodeConnY=latBuffer2, rc=status)
              else
  	        write(*,*) "Warning: face coordinates not present in src grid file,"// &
                  " so not outputting xc_a and yc_a to weight file."
                write(*,*)
  	        call ESMF_UGridGetVar(srcfile, srcmeshname, &
		   faceNodeConnX=lonBuffer2, faceNodeConnY=latBuffer2, rc=status)
              endif
	      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
              if (faceCoordFlag) then
   	        ncStatus=nf90_inq_varid(ncid,"xc_a",VarId)
                ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
                errmsg = "Variable xc_a in "//trim(wgtfile)
                if (CDFCheckError (ncStatus, &
                  ESMF_METHOD, &
                  ESMF_SRCLINE,errmsg,&
                  rc)) return
                ncStatus=nf90_inq_varid(ncid,"yc_a",VarId)
                ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
                errmsg = "Variable yc_a in "//trim(wgtfile)
                if (CDFCheckError (ncStatus, &
                  ESMF_METHOD, &
                  ESMF_SRCLINE,errmsg,&
                  rc)) return
                deallocate(latBuffer, lonBuffer)
              endif  
     	      ncStatus=nf90_inq_varid(ncid,"xv_a",VarId)
              ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)          
              errmsg = "Variable xv_b in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                  ESMF_METHOD, &
                  ESMF_SRCLINE,errmsg,&
                  rc)) return
              ncStatus=nf90_inq_varid(ncid,"yv_a",VarId)
              ncStatus=nf90_put_var(ncid,VarId, latBuffer2)          
              errmsg = "Variable yv_b in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                  ESMF_METHOD, &
                  ESMF_SRCLINE,errmsg,&
                  rc)) return
              deallocate(latBuffer2, lonBuffer2)
	  else
	      allocate(latBuffer(srcDim), lonBuffer(srcDim))
  	      call ESMF_UGridGetVar(srcfile, srcmeshname, &
		   nodeXcoords=lonBuffer, nodeYcoords=latBuffer, rc=status)
	      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	  ESMF_CONTEXT, rcToReturn=rc)) return
   	      ncStatus=nf90_inq_varid(ncid,"xc_a",VarId)
              ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
              errmsg = "Variable xc_a in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
                rc)) return
              ncStatus=nf90_inq_varid(ncid,"yc_a",VarId)
              ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
              errmsg = "Variable yc_a in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
              rc)) return
              deallocate(latBuffer, lonBuffer)
	  endif

           ! Write out mask
           allocate(mask(srcDim))         
	   mask(:)=1
	   if (present(srcMissingValue)) then
	     if (srcMissingValue) then
              allocate(varBuffer1D(srcDim))
              call ESMF_UgridGetVarByName(srcFile, srcvarname, &
			        varBuffer1D, missingvalue = missing_value, &
                                rc=status)
              if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
	      do j=1,size(varBuffer1D)
	         if (varBuffer1D(j) == missing_value) mask(j)=0
              enddo
	      deallocate(varBuffer1D)
	     endif
            endif

           ncStatus=nf90_inq_varid(ncid,"mask_a",VarId)
           ncStatus=nf90_put_var(ncid,VarId, mask)          
           errmsg = "Variable mask_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(mask)

        endif 

        ! Read the dstGrid variables and write them out
        if (dstFileTypeLocal == ESMF_FILEFORMAT_SCRIP) then
         allocate(latBuffer(dstDim), lonBuffer(dstDim))
         call ESMF_ScripGetVar(dstFile, grid_center_lon=lonBuffer, &
	 	grid_center_lat=latBuffer, rc=status)
         if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return
	 ncStatus=nf90_inq_varid(ncid,"xc_b",VarId)
         ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
         errmsg = "Variable xc_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,errmsg,&
           rc)) return

         ncStatus=nf90_inq_varid(ncid,"yc_b",VarId)
         ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
         errmsg = "Variable yc_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,errmsg,&
           rc)) return
         deallocate(latBuffer, lonBuffer)

         allocate(latBuffer2(dst_grid_corner,dstDim),lonBuffer2(dst_grid_corner,dstDim)) 
         call ESMF_ScripGetVar(dstFile, grid_corner_lon=lonBuffer2, &
	 	grid_corner_lat=latBuffer2, rc=status)
	 if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return

	 ncStatus=nf90_inq_varid(ncid,"xv_b",VarId)
         ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)          
         errmsg = "Variable xv_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,errmsg,&
           rc)) return

         ncStatus=nf90_inq_varid(ncid,"yv_b",VarId)
         ncStatus=nf90_put_var(ncid,VarId, latBuffer2)          
         errmsg = "Variable yv_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,errmsg,&
           rc)) return
         deallocate(latBuffer2, lonBuffer2)
         allocate(mask(dstDim))         
         call ESMF_ScripGetVar(dstFile, grid_imask=mask, rc=status)
         if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return
         ncStatus=nf90_inq_varid(ncid,"mask_b",VarId)
         ncStatus=nf90_put_var(ncid,VarId, mask)          
         errmsg = "Variable mask_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,errmsg,&
           rc)) return
         deallocate(mask)
        else if (dstFileTypeLocal == ESMF_FILEFORMAT_GRIDSPEC) then 
           allocate(lonBuffer(dstDim), latBuffer(dstDim))
	   ! check if bound variables exist or not
           if (dst_ndims == 1) then
	     allocate(lonBuffer1D(dst_grid_dims(1)), latBuffer1D(dst_grid_dims(2)))
             if (dsthasbound) then
                allocate(cornerlon2D(dst_grid_corner,dst_grid_dims(1)), &
 		      cornerlat2D(dst_grid_corner,dst_grid_dims(2)))
	        call ESMF_GridspecGetVar1D(dstFile, dst_coordids, lonBuffer1D, latBuffer1D,&
			cornerlon=cornerlon2D, cornerlat=cornerlat2D, rc=status)
	     else
	        call ESMF_GridspecGetVar1D(dstFile, dst_coordids, lonBuffer1D, latBuffer1D,&
			rc=status)
	     endif		
             if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
          	   ESMF_CONTEXT, rcToReturn=rc)) return
             k=1
             do j=1,dst_grid_dims(2)
	      do i=1,dst_grid_dims(1)
                lonBuffer(k)=lonBuffer1D(i)
                latBuffer(k)=latBuffer1D(j)
	        k=k+1
              enddo
             enddo
	     deallocate(lonBuffer1D, latBuffer1D)
	     if (dsthasbound) then
	       allocate(lonBuffer2(dst_grid_rank, dstDim),latBuffer2(dst_grid_rank,dstDim))
	       k=1
               do j=1,dst_grid_dims(2)
	         do i=1,dst_grid_dims(1)
                   lonBuffer2(:,k)=cornerlon2D(:,i)
                   latBuffer2(:,k)=cornerlat2D(:,j)
	           k=k+1
                 enddo
               enddo
               deallocate(cornerlon2D, cornerlat2D)
             endif
           else
	     allocate(lonBuffer2(dst_grid_dims(1),dst_grid_dims(2)), &
		      latBuffer2(dst_grid_dims(1),dst_grid_dims(2)))
             if (dsthasbound) then
               allocate(cornerlon3D(dst_grid_corner,dst_grid_dims(1), dst_grid_dims(2)),&
		    cornerlat3D(dst_grid_corner,dst_grid_dims(1), dst_grid_dims(2)))         
	       call ESMF_GridspecGetVar2D(dstFile, dst_coordids, lonBuffer2, latBuffer2, &
		   cornerlon=cornerlon3D, cornerlat=cornerlat3D, rc=status)
             else
	       call ESMF_GridspecGetVar2D(dstFile, dst_coordids, lonBuffer2, latBuffer2, &
		   rc=status)
             endif
             if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
          	   ESMF_CONTEXT, rcToReturn=rc)) return
             lonBuffer = reshape(lonBuffer2, (/dstDim/))
             latBuffer = reshape(latBuffer2, (/dstDim/))
 	     deallocate(lonBuffer2, latBuffer2)
             if (dsthasbound) then
	       allocate(lonBuffer2(dst_grid_rank, dstDim),latBuffer2(dst_grid_rank,dstDim))
               lonBuffer2=reshape(cornerlon3D, (/dst_grid_rank, dstDim/)) 
               latBuffer2=reshape(cornerlat3D, (/dst_grid_rank, dstDim/)) 
               deallocate(cornerlon3D, cornerlat3D)
             endif
	   endif

	   ncStatus=nf90_inq_varid(ncid,"xc_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
           errmsg = "Variable xc_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return

           ncStatus=nf90_inq_varid(ncid,"yc_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
           errmsg = "Variable yc_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(latBuffer, lonBuffer)

          ! Write xv_b, yv_b	
           if (dsthasbound) then
	   ncStatus=nf90_inq_varid(ncid,"xv_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)
           errmsg = "Variable xv_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return

           ncStatus=nf90_inq_varid(ncid,"yv_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, latBuffer2)
           errmsg = "Variable yv_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(lonBuffer2, latBuffer2)
           endif

           ! Mask
           allocate(mask(dstDim))         
	   mask(:)=1
	   if (present(dstMissingValue)) then
	     if (dstMissingValue) then
              allocate(varBuffer(dst_grid_dims(1),dst_grid_dims(2)))
              call ESMF_GridspecGetVarByName(dstFile, dstvarname, dst_dimids, &
			        varBuffer, missing_value = missing_value, &
                                rc=status)
              if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
	      k=1
	      do i=1,size(varBuffer,2)
	        do j=1,size(varBuffer,1)
	           if (varBuffer(j,i) == missing_value) mask(k)=0
	           k=k+1
                enddo	
              enddo
	      deallocate(varBuffer)
	    endif
           endif

           ncStatus=nf90_inq_varid(ncid,"mask_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, mask)          
           errmsg = "Variable mask_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(mask)
            
	else if (dstFileTypeLocal == ESMF_FILEFORMAT_ESMFMESH) then 
           ncStatus=nf90_open(dstFile,NF90_NOWRITE,ncid1)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,errmsg,&
         rc)) return
           ! only write out xv_a and yv_a when the regrid method is conserve
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_BILINEAR%regridmethod &
		.or. methodlocal%regridmethod ==ESMF_REGRIDMETHOD_PATCH%regridmethod &
		.or. methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_STOD%regridmethod &
		.or. methodlocal%regridmethod ==ESMF_REGRIDMETHOD_NEAREST_DTOS%regridmethod) then
	     ! check if centerCoords exit
             ncStatus=nf90_inq_varid(ncid1,"nodeCoords",VarId)
	     varStr = "nodeCoords"
	   else
             ncStatus=nf90_inq_varid(ncid1,"centerCoords",VarId)
	     varStr = "centerCoords"
           endif
	   if (ncStatus /= nf90_noerror) then
	     write(*,*) "Warning: "//trim(varStr)// &
                " not present in dst grid file, so not outputting xc_b and yc_b to weight file."
             write(*,*)
           else 
	     allocate(latBuffer2(dstCoordDim, dstDim))
             allocate(latBuffer(dstDim), lonBuffer(dstDim))
             ncStatus=nf90_get_var(ncid1,VarId, latBuffer2)
             errmsg = "Variable "//varStr//" in "//trim(dstFile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return
	     do i=1,dstDim
		lonBuffer(i)=latBuffer2(1,i)
                latBuffer(i)=latBuffer2(2,i)
             enddo
   	     ncStatus=nf90_inq_varid(ncid,"xc_b",VarId)
             ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
             errmsg = "Variable xc_b in "//trim(wgtfile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return

             ncStatus=nf90_inq_varid(ncid,"yc_b",VarId)
             ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
             errmsg = "Variable yc_b in "//trim(wgtfile)
             if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return
             deallocate(latBuffer, lonBuffer, latBuffer2)
           endif
           
           ! output xv_b and yv_b is harder, we have to read in the nodeCoords and
           ! elementConn and construct the the latitudes and longitudes for
           ! all the corner vertices
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_CONSERVE%regridmethod) then 
           allocate(latBuffer2(dst_grid_corner,dstDim),lonBuffer2(dst_grid_corner,dstDim))         
           call ESMF_EsmfGetVerts(ncid1, dstFile, dstDim, dst_grid_corner, dstNodeDim, &
		latBuffer2, lonBuffer2, status) 
           if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return

	   ncStatus=nf90_inq_varid(ncid,"xv_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)          
           errmsg = "Variable xv_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,errmsg,&
             rc)) return

           ncStatus=nf90_inq_varid(ncid,"yv_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, latBuffer2)          
           errmsg = "Variable yv_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,errmsg,&
             rc)) return
           deallocate(latBuffer2, lonBuffer2) 
           endif
           ! Write mask_b
           allocate(mask(dstDim))         
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_CONSERVE%regridmethod) then 
             ncStatus=nf90_inq_varid(ncid1,"elementMask",VarId)
	     if (ncStatus /= nf90_noerror) then
               write(*,*) "Warning: elementMask"// &
                " not present in dst grid file, so setting mask_b=1 in weight file."
               write(*,*)
               mask = 1
             else 
               ncStatus=nf90_get_var(ncid1,VarId, mask)
               errmsg = "Variable elementMask in "//trim(dstFile)
               if (CDFCheckError (ncStatus, &
                 ESMF_METHOD, &
                 ESMF_SRCLINE,errmsg,&
                 rc)) return
             end if
           else
             ! make mask all 1
             mask=1
           endif
           ncStatus=nf90_inq_varid(ncid,"mask_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, mask)          
           errmsg = "Variable mask_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,errmsg,&
             rc)) return
           deallocate(mask)
           ncStatus=nf90_close(ncid1)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,trim(dstFile),&
             rc)) return
	else if (dstFileTypeLocal == ESMF_FILEFORMAT_UGRID) then 
  	   if (methodlocal%regridmethod ==ESMF_REGRIDMETHOD_CONSERVE%regridmethod) then 
	     ! check if faceCoords exit
	      call ESMF_UGridInq(dstfile, dstmeshname, faceCoordFlag=faceCoordFlag)
	        allocate(latBuffer2(dst_grid_corner,dstDim),&
			lonBuffer2(dst_grid_corner,dstDim)) 
              if (faceCoordFlag) then
  	        allocate(latBuffer(dstDim), lonBuffer(dstDim))
  	        call ESMF_UGridGetVar(dstfile, dstmeshname, &
		   faceXcoords=lonBuffer, faceYcoords=latBuffer, &
		   faceNodeConnX=lonBuffer2, faceNodeConnY=latBuffer2, rc=status)
              else
  	        write(*,*) "Warning: face coordinates not present in dst grid file,"// &
                  " so not outputting xc_a and yc_a to weight file."
                write(*,*)
  	        call ESMF_UGridGetVar(dstfile, dstmeshname, &
		   faceNodeConnX=lonBuffer2, faceNodeConnY=latBuffer2, rc=status)
              endif
	      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
              if (faceCoordFlag) then
   	        ncStatus=nf90_inq_varid(ncid,"xc_b",VarId)
                ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
                errmsg = "Variable xc_a in "//trim(wgtfile)
                if (CDFCheckError (ncStatus, &
                  ESMF_METHOD, &
                  ESMF_SRCLINE,errmsg,&
                  rc)) return
                ncStatus=nf90_inq_varid(ncid,"yc_b",VarId)
                ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
                errmsg = "Variable yc_a in "//trim(wgtfile)
                if (CDFCheckError (ncStatus, &
                  ESMF_METHOD, &
                  ESMF_SRCLINE,errmsg,&
                rc)) return
                deallocate(latBuffer, lonBuffer)
              endif
   	      ncStatus=nf90_inq_varid(ncid,"xv_b",VarId)
              ncStatus=nf90_put_var(ncid,VarId, lonBuffer2)          
              errmsg = "Variable xv_b in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
                rc)) return

              ncStatus=nf90_inq_varid(ncid,"yv_b",VarId)
              ncStatus=nf90_put_var(ncid,VarId, latBuffer2)          
              errmsg = "Variable yv_b in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
                rc)) return
              deallocate(latBuffer2, lonBuffer2)
	  else
	      allocate(latBuffer(dstDim), lonBuffer(dstDim))
  	      call ESMF_UGridGetVar(dstfile, dstmeshname, &
		   nodeXcoords=lonBuffer, nodeYcoords=latBuffer, rc=rc)
   	      ncStatus=nf90_inq_varid(ncid,"xc_b",VarId)
              ncStatus=nf90_put_var(ncid,VarId, lonBuffer)          
              errmsg = "Variable xc_a in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
                rc)) return
              ncStatus=nf90_inq_varid(ncid,"yc_b",VarId)
              ncStatus=nf90_put_var(ncid,VarId, latBuffer)          
              errmsg = "Variable yc_a in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
              rc)) return
              deallocate(latBuffer, lonBuffer)
	  endif

           ! Write out mask
           allocate(mask(dstDim))         
	   mask(:)=1
	   if (present(dstMissingValue)) then
	     if (dstMissingValue) then
              allocate(varBuffer1D(dstDim))
              call ESMF_UgridGetVarByName(dstFile, dstvarname, &
			        varBuffer1D, missingvalue = missing_value, &
                                rc=status)
              if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
                  ESMF_CONTEXT, rcToReturn=rc)) return
	      do j=1,size(varBuffer1D)
	         if (varBuffer1D(j) == missing_value) mask(j)=0
              enddo
	      deallocate(varBuffer1D)
	     endif
            endif

           ncStatus=nf90_inq_varid(ncid,"mask_b",VarId)
           ncStatus=nf90_put_var(ncid,VarId, mask)          
           errmsg = "Variable mask_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
             ESMF_METHOD, &
             ESMF_SRCLINE,&
  	     errmsg,&
             rc)) return
           deallocate(mask)
        endif 

         ! Write area_a
         ncStatus=nf90_inq_varid(ncid,"area_a",VarId)
         if (present(srcArea)) then
           ncStatus=nf90_put_var(ncid,VarId, srcArea)          
           errmsg = "Variable area_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return 
         else
          ! Just set these to 0.0, because not provided
           allocate(area(srcDim))         
           area=0.0
           ncStatus=nf90_put_var(ncid,VarId, area)          
           errmsg = "Variable area_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return
           deallocate(area)
         endif

         ! Write area_b
         ncStatus=nf90_inq_varid(ncid,"area_b",VarId)
         if (present(dstArea)) then
           ncStatus=nf90_put_var(ncid,VarId, dstArea)          
           errmsg = "Variable area_b in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return 
         else
          ! Just set these to 0.0, because not provided
           allocate(area(dstDim))         
           area=0.0
           ncStatus=nf90_put_var(ncid,VarId, area)          
           errmsg = "Variable area_a in "//trim(wgtfile)
           if (CDFCheckError (ncStatus, &
               ESMF_METHOD, &
               ESMF_SRCLINE,errmsg,&
               rc)) return
           deallocate(area)
         endif

         ! Write frac_a
         ncStatus=nf90_inq_varid(ncid,"frac_a",VarId)
	 if (present(srcFrac)) then
            ncStatus=nf90_put_var(ncid,VarId, srcFrac)          
	 else
            allocate(frac(srcDim))         
            frac=0.0
            ncStatus=nf90_put_var(ncid,VarId, frac)          
            deallocate(frac)
	 endif
         errmsg = "Variable frac_a in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,errmsg,&
           rc)) return

         ! Write frac_b
         ncStatus=nf90_inq_varid(ncid,"frac_b",VarId)
	 if (present(dstFrac)) then
            ncStatus=nf90_put_var(ncid,VarId, dstFrac)          
	 else
            allocate(frac(dstDim))         
            frac=1.0
            ncStatus=nf90_put_var(ncid,VarId, frac)          
            deallocate(frac)
         endif
         errmsg = "Variable frac_b in "//trim(wgtfile)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,errmsg,&
           rc)) return
         deallocate(src_grid_dims, dst_grid_dims)
    end if

    ! Block all other PETs until the NetCDF file has been created
    call ESMF_VMBarrier(vm)

    ! find the max of allCounts(i) and allocate colrow
    maxcount=0
    do i=1,PetCnt
	if (allCounts(i) > maxcount) maxcount = allCounts(i)
    enddo

    if (PetNo == 0) then 
	! First write out its own weight and indices, then receive the data from 
	! other PETs and write them out
	start = 1
        ! allocate indexbuf and weightbuf to receive data from other PETs
        allocate(indexbuf(maxcount*2), weightbuf(maxcount))
        do i=1, PetCnt
          ! write the local factorList and factorIndexList first
          localCount(1)=allCounts(i)
 	  if (i==1) then 
  	    !do j=1,localCount(1)
            !    indexbuf(j) = factorIndexList(1,j)
            !enddo
            next => factorIndexList(1,:)
            ncStatus=nf90_inq_varid(ncid,"col",VarId)
   	    ncStatus=nf90_put_var(ncid,VarId, next,(/start/),localCount)          
            errmsg = "Variable col in "//trim(wgtfile)
            if (CDFCheckError (ncStatus, &
              ESMF_METHOD, &
              ESMF_SRCLINE,errmsg,&
              rc)) return
            !do j=1,localCount(1)
            !  indexbuf(j) = factorIndexList(2,j)
            !enddo
            next => factorIndexList(2,:)
            ncStatus=nf90_inq_varid(ncid,"row",VarId)
   	    ncStatus=nf90_put_var(ncid,VarId, next,(/start/),localCount)          
            errmsg = "Variable row in "//trim(wgtfile)
            if (CDFCheckError (ncStatus, &
              ESMF_METHOD, &
              ESMF_SRCLINE,errmsg,&
              rc)) return

            ncStatus=nf90_inq_varid(ncid,"S",VarId)
            ncStatus=nf90_put_var(ncid,VarId, factorList, (/start/),localCount)
            errmsg = "Variable S in "//trim(wgtfile)
            if (CDFCheckError (ncStatus, &
              ESMF_METHOD, &
              ESMF_SRCLINE,errmsg,&
              rc)) return
	  else 
            if (localCount(1) > 0) then
              ! receive the factorList and factorIndexList 
              call ESMF_VMRecv(vm, indexbuf, localCount(1)*2, i-1, rc=status)
	      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	  ESMF_CONTEXT, rcToReturn=rc)) return
              call ESMF_VMRecv(vm, weightbuf, localCount(1), i-1, rc=status)
	      if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	  ESMF_CONTEXT, rcToReturn=rc)) return

              ncStatus=nf90_inq_varid(ncid,"col",VarId)
       	      ncStatus=nf90_put_var(ncid,VarId, indexbuf,(/start/),localCount)          
              errmsg = "Variable col in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
                rc)) return
              next => indexbuf(localCount(1)+1:localCount(1)*2)
              ncStatus=nf90_inq_varid(ncid,"row",VarId)
     	      ncStatus=nf90_put_var(ncid,VarId, next ,(/start/),localCount)          
              errmsg = "Variable row in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
                rc)) return

              ncStatus=nf90_inq_varid(ncid,"S",VarId)
              ncStatus=nf90_put_var(ncid,VarId, weightbuf, (/start/),localCount)
              errmsg = "Variable S in "//trim(wgtfile)
              if (CDFCheckError (ncStatus, &
                ESMF_METHOD, &
                ESMF_SRCLINE,errmsg,&
                rc)) return
            end if
          end if
          start = start + localCount(1)
       end do
    else
       allocate(indexbuf(localcount(1)*2))
       if (localcount(1) > 0) then
         do j=1,localCount(1)
             indexbuf(j) = factorIndexList(1,j)
             indexbuf(j+localCount(1)) = factorIndexList(2,j)
         enddo
         ! a non-root PET, send the results to PET 0
         call ESMF_VMSend(vm, indexbuf, localCount(1)*2, 0, rc=status)
         if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return
         call ESMF_VMSend(vm, factorList, localCount(1), 0, rc=status)
	 if (ESMF_LogFoundError(status, ESMF_ERR_PASSTHRU, &
        	ESMF_CONTEXT, rcToReturn=rc)) return
       end if
    end if
       
    call ESMF_VMBarrier(vm)
    deallocate(allCounts)
    if (PetNo == 0) then
       ncStatus = nf90_close(ncid)                        
       if (CDFCheckError (ncStatus, &
         ESMF_METHOD, &
         ESMF_SRCLINE, trim(wgtfile),&
         rc)) return
       deallocate(weightbuf)
    end if
    deallocate(indexbuf)
    if (present(rc)) rc = ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
#endif

    return
end subroutine ESMF_OutputScripWeightFile

#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_OutputScripVarFile"
!
subroutine ESMF_OutputScripVarFile(filename, varname, varbuffer, rc)

    character(len=*), intent(in)   :: filename
    character(len=*), intent(in)   :: varname
    real(ESMF_KIND_R8), pointer    :: varbuffer(:)
    integer                       :: rc

    integer :: ncid, dimid, varid
    integer :: localrc, ncStatus
    integer :: varsize
    character(len=256) :: errmsg

#ifdef ESMF_NETCDF

    ncStatus = nf90_open (path=trim(filename), mode=nf90_write, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, trim(filename), &
      rc)) return

    ! define a new varilable with dimension 'n_b'
    ncStatus = nf90_inq_dimid(ncid, 'n_b', dimid)
    errmsg = "Dimension n_b in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

   ncStatus = nf90_inquire_dimension (ncid, dimid, len=varsize)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

   if (size(varbuffer,1) /= varsize) then
      call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_BAD, &
           msg="- the variable size is inconsistent with the dest grid dimension", &
           ESMF_CONTEXT, rcToReturn=rc)
      return
    endif 
	  	      
    ncStatus = nf90_redef(ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, trim(filename), &
      rc)) return
    
    ncStatus = nf90_def_var(ncid, varname, NF90_DOUBLE, (/dimid/), varid)
    errmsg = "Variable src_grid_dims in "//trim(filename)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return

    ncStatus = nf90_enddef(ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, trim(filename), &
      rc)) return

    ncStatus = nf90_put_var(ncid, varid, varbuffer)
    errmsg = "Variable "//trim(varname)//" in "//trim(filename)
         if (CDFCheckError (ncStatus, &
           ESMF_METHOD, &
           ESMF_SRCLINE,&
	   errmsg,&
           rc)) return
    ncStatus = nf90_close(ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, &
      trim(filename), &
      rc)) return
    rc=ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
    return
#endif

end subroutine ESMF_OutputScripVarFile 
 
#undef  ESMF_METHOD
#define ESMF_METHOD "ESMF_EsmfInq"
!BOPI
! !ROUTINE: ESMF_EsmfInq: Return the dimension information
!  information from a ESMF Unstructured grid file
!
! !INTERFACE:
subroutine ESMF_EsmfInq(filename, nodeCount, elementCount, &
	      		maxNodePElement, coordDim, &
			haveNodeMask, haveElmtMask, rc)    

! !ARGUMENTS:

    character(len=*), intent(in)   :: filename
    integer, intent(out), optional :: nodeCount
    integer, intent(out), optional :: elementCount
    integer, intent(out), optional :: maxNodePElement
    integer, intent(out), optional :: coordDim
    logical, intent(out), optional :: haveNodeMask
    logical, intent(out), optional :: haveElmtMask
    integer, intent(out), optional :: rc

    integer:: localrc, ncStatus
    integer :: DimId, VarId
    integer :: ncid, local_rank
    character(len=256):: errmsg
    integer, parameter :: nf90_noerror = 0

#ifdef ESMF_NETCDF
    if (present(rc)) rc=ESMF_SUCCESS
    ncStatus = nf90_open (path=trim(filename), mode=nf90_nowrite, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD, &
      ESMF_SRCLINE, trim(filename), rc)) return

    ! get number of elements
    if (present(nodeCount)) then
      ncStatus = nf90_inq_dimid (ncid, "nodeCount", DimId)
      errmsg = "Dimension nodeCount in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return

      ncStatus = nf90_inquire_dimension (ncid, DimId, len=nodeCount)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return
    end if

    ! get number of elements
    if (present(elementCount)) then
      ncStatus = nf90_inq_dimid (ncid, "elementCount", DimId) 
      errmsg = "Dimension elementCount in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return

      ncStatus = nf90_inquire_dimension (ncid, DimId, len=elementCount)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return
    end if

    ! get number of elements
    if (present(maxNodePElement)) then
      ncStatus = nf90_inq_dimid (ncid, "maxNodePElement", DimId)
      errmsg = "Dimension maxNodePElement in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return

      ncStatus = nf90_inquire_dimension (ncid, DimId, len=maxNodePElement)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return
    end if

    ! get number of elements
    if (present(coordDim)) then
      ncStatus = nf90_inq_dimid (ncid, "coordDim", DimId)
      errmsg = "Dimension coordDim in "//trim(filename)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return

      ncStatus = nf90_inquire_dimension (ncid, DimId, len=coordDim)
      if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return
    end if

    ! check if elementMask exit
    if (present(haveElmtMask)) then
      ncStatus = nf90_inq_varid (ncid, "elementMask", VarId)
      if (ncStatus == nf90_noerror) then
           haveElmtMask = .true.
      else
	   haveElmtMask = .false.
      end if
    end if

    ! check if nodeMask exit
    if (present(haveNodeMask)) then
      ncStatus = nf90_inq_varid (ncid, "nodeMask", VarId)
      if (ncStatus == nf90_noerror) then
           haveNodeMask = .true.
      else
	   haveNodeMask = .false.
      end if
    end if

    if (present(rc)) rc=ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
#endif

    return
end subroutine ESMF_EsmfInq

#undef  ESMF_METHOD
#define ESMF_METHOD "ESMF_EsmfInqUnits"
!BOPI
! !ROUTINE: ESMF_EsmfInqUnits: Return the units attribute for
! coordinate variables, assuming all the coordinate variables
! have the same units
!
! !INTERFACE:
subroutine ESMF_EsmfInqUnits(filename, units, rc)

! !ARGUMENTS:

    character(len=*), intent(in)   :: filename
    character(len=*), intent(out)   :: units
    integer, intent(out), optional :: rc

    integer:: localrc, ncStatus
    integer :: ncid, VarId, len
    character(len=80) :: buffer
    character(len=256) :: errmsg
    integer, parameter :: nf90_noerror = 0
    integer :: coordDim,DimID

#ifdef ESMF_NETCDF
    if (present(rc)) rc=ESMF_SUCCESS
    ncStatus = nf90_open (path=trim(filename), mode=nf90_nowrite, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD, &
      ESMF_SRCLINE, trim(filename),rc)) return

    ! Get vertex dimension
    ncStatus = nf90_inq_dimid (ncid, "coordDim", DimId)
    errmsg = "Dimension coordDim in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_inquire_dimension (ncid, DimId, len=coordDim)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ! Get units
    ncStatus = nf90_inq_varid (ncid, "nodeCoords", VarId)
    errmsg = "Variable nodeCoords in "//trim(filename)
    if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return

    ncStatus = nf90_inquire_attribute(ncid, VarId, "units", len=len)
    ! only require units for 2D
    if (coordDim==2) then
       if (CDFCheckError (ncStatus, &
            ESMF_METHOD, &
            ESMF_SRCLINE,errmsg,&
            rc)) return
    else    
       ! if no units are present set to default and leave
       if (ncStatus /= nf90_noerror) then
          units="UNKNOWN"
          if (present(rc)) rc=ESMF_SUCCESS
          return
       endif
    endif

    ncStatus = nf90_get_att(ncid, VarId, "units", buffer)
    if (CDFCheckError (ncStatus, &
        ESMF_METHOD, &
        ESMF_SRCLINE,errmsg,&
        rc)) return

    if (buffer(len:len) .eq. achar(0)) len = len-1
    call ESMF_StringLowerCase(buffer(1:len), rc=rc)
    units = buffer(1:len)
    if (present(rc)) rc=ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
#endif

    return
end subroutine ESMF_EsmfInqUnits

#undef  ESMF_METHOD
#define ESMF_METHOD "ESMF_EsmfGetNode"
subroutine ESMF_EsmfGetNode (filename, nodeCoords, nodeMask, &
			    convertToDeg, rc)

    character(len=*), intent(in)   :: filename
    real(ESMF_KIND_R8), pointer    :: nodeCoords (:,:)
    integer(ESMF_KIND_I4), pointer, optional :: nodeMask (:)
    logical, intent(in), optional  :: convertToDeg
    integer, intent(out), optional :: rc

    type(ESMF_VM) :: vm
    integer :: PetNo, PetCnt

    integer :: ncid
    integer :: ncStatus
    integer :: RecCnt (2)

    integer :: DimId
    integer :: nodeCnt, ElmtCount, MaxNodePerElmt, NodeDim
    integer :: localCount, remain

    integer :: VarNo
    character(len=256)::errmsg
    character(len=80) :: units
    integer :: len
    logical :: convertToDegLocal

#ifdef ESMF_NETCDF
    convertToDegLocal = .false.
    if (present(convertToDeg)) convertToDegLocal = convertToDeg

    call ESMF_VMGetCurrent(vm, rc=rc)
    if (rc /= ESMF_SUCCESS) return
    ! set up local pet info
    call ESMF_VMGet(vm, localPet=PetNo, petCount=PetCnt, rc=rc)
    if (rc /= ESMF_SUCCESS) return

    ncStatus = nf90_open (path=trim(filename), mode=nf90_nowrite, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, trim(filename), &
      rc)) return

    ! get number of vertices
    ncStatus = nf90_inq_dimid (ncid, "nodeCount", DimId)
    errmsg = "Dimension nodeCount in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_inquire_dimension (ncid, DimId, len=nodeCnt)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ! Get vertex dimension
    ncStatus = nf90_inq_dimid (ncid, "coordDim", DimId)
    errmsg = "Dimension coordDim in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_inquire_dimension (ncid, DimId, len=NodeDim)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ! allocate memory for verticies
    allocate (nodeCoords (NodeDim, nodeCnt))

    RecCnt(:) = ubound(nodeCoords)
    !print *, "nodeCoords:",nodeCnt, NodeDim

    ! read vertex data
    ncStatus = nf90_inq_varid (ncid, "nodeCoords", VarNo)
    errmsg = "Variable nodeCoords in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_get_var (ncid, VarNo, nodeCoords, start=(/1,1/), count=(/NodeDim, nodeCnt/))
    errmsg = "Variable nodeCoords in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ! Check units, but only if 2D
    if (nodeDim==2) then 
       ncStatus = nf90_inquire_attribute(ncid, VarNo, "units", len=len)
       if (CDFCheckError (ncStatus, &
            ESMF_METHOD, &
            ESMF_SRCLINE,errmsg,&
            rc)) return

       ncStatus = nf90_get_att(ncid, VarNo, "units", units)
       if (CDFCheckError (ncStatus, &
            ESMF_METHOD, &
            ESMF_SRCLINE,errmsg,&
            rc)) return
       ! if len != 7, something is wrong, check the value.  If it starts 
       ! with Degres/degrees/Radians/radians, ignore the garbage after the
       ! word.  Otherwise, return the whole thing
     if (units(len:len) .eq. achar(0)) len = len-1
       call ESMF_StringLowerCase(units(1:len), rc=rc)
       if (units(1:len) .ne. 'degrees' .and. &
            units(1:len) .ne. 'radians') then
          call ESMF_LogSetError(rcToCheck=ESMF_FAILURE, & 
                 msg="- units attribute is not degrees or radians", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
          return
       endif

       ! if units is "radians", convert it to degree
       if (convertToDegLocal) then
          if (units(1:len) .eq. "radians") then
             nodeCoords(:,:) = &
                 nodeCoords(:,:)*ESMF_COORDSYS_RAD2DEG
          endif
       endif
    endif
    
    ! get nodeMask
    if (present(nodeMask)) then
       allocate(nodeMask(nodeCnt))
       ncStatus = nf90_inq_varid (ncid, "nodeMask", VarNo)
       errmsg = "Variable nodeMask in "//trim(filename)
       if (CDFCheckError (ncStatus, &
          ESMF_METHOD,  &
          ESMF_SRCLINE, errmsg, &
          rc)) return

       ncStatus = nf90_get_var (ncid, VarNo, nodeMask, start=(/1/), count=(/nodeCnt/))
       if (CDFCheckError (ncStatus, &
          ESMF_METHOD,  &
          ESMF_SRCLINE, errmsg, &
          rc)) return
    endif

    ncStatus = nf90_close (ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, trim(filename), &
      rc)) return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
    return
#endif

end subroutine ESMF_EsmfGetNode

#undef  ESMF_METHOD
#define ESMF_METHOD "ESMF_EsmfGetElement"
subroutine ESMF_EsmfGetElement (filename, elementConn, &
				 elmtNums, startElmt, elementMask, &
				 elementArea, rc)

    character(len=*), intent(in)   :: filename
    integer(ESMF_KIND_I4), pointer :: elementConn (:,:)
    integer(ESMF_KIND_I4), pointer :: elmtNums (:)
    integer,           intent(out) :: startElmt
    integer(ESMF_KIND_I4), pointer, optional :: elementMask (:)
    real(ESMF_KIND_R8), pointer, optional :: elementArea (:)
    integer, intent(out), optional :: rc

    type(ESMF_VM) :: vm
    integer :: PetNo, PetCnt

    integer :: ncid
    integer :: ncStatus
    integer :: RecCnt (2)

    integer :: DimId
    integer :: nodeCnt, ElmtCount, MaxNodePerElmt, NodeDim
    integer :: localCount, remain

    integer :: VarNo
    character(len=256)::errmsg
    character(len=80) :: units
    integer :: len

#ifdef ESMF_NETCDF

    call ESMF_VMGetCurrent(vm, rc=rc)
    if (rc /= ESMF_SUCCESS) return
    ! set up local pet info
    call ESMF_VMGet(vm, localPet=PetNo, petCount=PetCnt, rc=rc)
    if (rc /= ESMF_SUCCESS) return

    ncStatus = nf90_open (path=trim(filename), mode=nf90_nowrite, ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, trim(filename), &
      rc)) return

    ! get number of elmts
    ncStatus = nf90_inq_dimid (ncid, "elementCount", DimId)
    errmsg = "Dimension elementCount in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_inquire_dimension (ncid, DimId, len=ElmtCount)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ! Get max_verts_per_elmt
    ncStatus = nf90_inq_dimid (ncid, "maxNodePElement", DimId)
    errmsg = "Dimension maxNodePElement in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_inquire_dimension (ncid, DimId, len=MaxNodePerElmt)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ! Decompose the elmt array evenly on all the PEs
    localcount = elmtCount/PetCnt
    remain = mod(elmtCount,PetCnt)
    startElmt = localcount * PetNo+1
    if (PetNo == (PetCnt-1)) localcount = localcount+remain

    ! allocate memory for elmts
    allocate (elementConn (MaxNodePerElmt, localcount))
    allocate (elmtNums (localcount))
    
    ! read elmt_verts data
    ncStatus = nf90_inq_varid (ncid, "elementConn", VarNo)
    errmsg = "Variable elementConn in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_get_var (ncid, VarNo, elementConn, start=(/1,startElmt/), count=(/MaxNodePerElmt, localcount/))
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return
    
    ! read num_elmt_verts
    ncStatus = nf90_inq_varid (ncid, "numElementConn", VarNo)
    errmsg = "Variable numElementConn in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ncStatus = nf90_get_var (ncid, VarNo, elmtNums, start=(/startElmt/), count=(/localcount/))
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    if (present(elementMask)) then
       allocate(elementMask(localcount))
       ncStatus = nf90_inq_varid (ncid, "elementMask", VarNo)
       errmsg = "Variable elementMask in "//trim(filename)
       if (CDFCheckError (ncStatus, &
          ESMF_METHOD,  &
          ESMF_SRCLINE, errmsg, &
          rc)) return

       ncStatus = nf90_get_var (ncid, VarNo, elementMask, start=(/startElmt/), count=(/localcount/))
       if (CDFCheckError (ncStatus, &
          ESMF_METHOD,  &
          ESMF_SRCLINE, errmsg, &
          rc)) return

    end if

    if (present(elementArea)) then
       allocate(elementArea(localcount))
       ncStatus = nf90_inq_varid (ncid, "elementArea", VarNo)
       errmsg = "Variable elementArea in "//trim(filename)
       if (CDFCheckError (ncStatus, &
          ESMF_METHOD,  &
          ESMF_SRCLINE, errmsg, &
          rc)) return

       ncStatus = nf90_get_var (ncid, VarNo, elementArea, start=(/startElmt/), &
			        count=(/localcount/))
       if (CDFCheckError (ncStatus, &
          ESMF_METHOD,  &
          ESMF_SRCLINE, errmsg, &
          rc)) return
    end if

    ncStatus = nf90_close (ncid=ncid)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, trim(filename), &
      rc)) return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
    return
#endif

end subroutine ESMF_EsmfGetElement


!
!  Get the NodeCoords and ElementConn from the ESMF unstructured file and construct
!   corner vertices table similar to what stored in the SCRIP file
!
#undef  ESMF_METHOD
#define ESMF_METHOD "EsmfGetVerts"
subroutine ESMF_EsmfGetVerts(ncid, filename, numElements, numNodePElement, numNodes, &
		latBuffer, lonBuffer, rc) 

    integer, intent(in)  :: ncid
    character(len=*), intent(in) :: filename
    integer, intent(in)  :: numElements
    integer, intent(in)  :: numNodePElement
    integer, intent(in)  :: numNodes
    real(ESMF_KIND_R8), intent(inout)   :: latBuffer(:,:)
    real(ESMF_KIND_R8), intent(inout)   :: lonBuffer(:,:)
    integer, intent(out), optional      :: rc

    integer :: ncStatus
    real(ESMF_KIND_R8), allocatable :: nodeCoords(:,:)
    integer(ESMF_KIND_I4), allocatable :: elementConn(:,:)
    integer :: varId, fillValue
    integer, parameter :: fillValue1 = -9999
    integer :: i,j, index
    character(len=256)::errmsg

#ifdef ESMF_NETCDF
    allocate(nodeCoords(2, numNodes))
    allocate(elementConn(numNodePElement, numElements))

    ! Get NodeCoords    
    ncStatus = nf90_inq_varid(ncid, "nodeCoords", varId)
    errmsg = "Variable nodeCoords in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return
    ncStatus = nf90_get_var(ncid, varId, nodeCoords)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return

    ! Get elementConn table
    ncStatus = nf90_inq_varid(ncid, "elementConn", varId)
    errmsg = "Variable elementConn in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return
    ncStatus = nf90_get_var(ncid, varId, elementConn)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return
    ! Get fill value
    ncStatus = nf90_get_att(ncid, varId, "_FillValue", fillValue)
    errmsg = "Attribute _FillValue for elementConn in "//trim(filename)
    if (CDFCheckError (ncStatus, &
      ESMF_METHOD,  &
      ESMF_SRCLINE, errmsg, &
      rc)) return
    ! Fill latBuffer and lonBuffer
    do i=1, numElements
      do j=1, numNodePElement
         if (elementConn(j,i) /= fillValue) then
            index = elementConn(j,i)
            latBuffer(j,i) = nodeCoords(2,index)
            lonBuffer(j,i) = nodeCoords(1,index)
         else
            latBuffer(j,i) = fillValue1
            lonBuffer(j,i) = fillValue1
         endif
      enddo
    enddo
    if (present(rc)) rc=ESMF_SUCCESS
    return
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
    return
#endif
end subroutine ESMF_EsmfGetVerts

!-----------------------------------------------------------------------

!
!  check CDF file error code
!
#undef  ESMF_METHOD
#define ESMF_METHOD "CDFCheckError"
function CDFCheckError (ncStatus, module, fileName, lineNo, errmsg, rc)

    logical                       :: CDFCheckError

    integer,          intent(in)  :: ncStatus
    character(len=*), intent(in)  :: module
    character(len=*), intent(in)  :: fileName
    integer,          intent(in)  :: lineNo
    character(len=*), intent(in)  :: errmsg
    integer, intent(out),optional :: rc

    integer, parameter :: nf90_noerror = 0

    CDFCheckError = .FALSE.

#ifdef ESMF_NETCDF
    if ( ncStatus .ne. nf90_noerror) then
        call ESMF_LogWrite (msg="netCDF Status Return Error", logmsgFlag=ESMF_LOGMSG_ERROR, &
            line=lineNo, file=fileName, method=module)
        print '("NetCDF Error: ", A, " : ", A)', &
	trim(errmsg),trim(nf90_strerror(ncStatus))
        call ESMF_LogFlush()
        if (present(rc)) rc = ESMF_FAILURE
 	CDFCheckError = .TRUE.
    else
       if (present(rc)) rc = ESMF_SUCCESS
       return
    end if
#else
    call ESMF_LogSetError(rcToCheck=ESMF_RC_LIB_NOT_PRESENT, & 
                 msg="- ESMF_NETCDF not defined when lib was compiled", & 
                 ESMF_CONTEXT, rcToReturn=rc) 
    return
#endif

end function CDFCheckError


end module ESMF_IOScripMod
