#ifndef LOGGER_H
#define LOGGER_H

#include<stdio.h>
#include<stdarg.h>

/** \brief logger function that reports just a message */
typedef void (*LogFunction)( const char* , ... ) ;

/** \brief logger function that reports a message, along with some identifier (e.g. the name of the calling method) */
typedef void (*LogFunctionWithID)( const char* , const char* , ... ) ;

/** \brief logger function to explicitly set level */
typedef void (*SetLevelFunction)( int ) ;

/** \brief logger function to incrementally change the message level */
typedef void (*LevelIncrementFunction)( void ) ;

/** \brief used to explicitly set the minimum level of printed messages */
enum LogLevels {
	Logger_LevelNothing ,	/** \brief< output no messages */
	Logger_LevelError ,	/** \brief< output only error messages */
	Logger_LevelWarn ,	/** \brief< output warning and error messages */
	Logger_LevelInfo ,	/** \brief< output info, warning, error messages */
	Logger_LevelDebug ,	/** \brief< output info, warning, debug, error messages */
	Logger_LevelTrace ,	/** \brief< output info, warning, debug, trace, error messages */
	Logger_LevelEverything	/** \brief< output all messages */
} ;


/**
\brief the singleton Logger object

All data to be logged are passed through a reference to
this Logger object, which in turn decides (based on that
message's severity level) whether to print said message.

To make Logger more object-like (in the sense of a living,
self-referential object as found in OO languages) the 
"member functions" all fetch a reference to the singleton
instance when needed.

With this being C and not C++, we must handle some steps
manually.  For example: a C struct cannot have functions
unto itself, but instead supports function pointers as
member variables, which must in turn be assigned to other
functions.  Hence the need for the Logger_init() function.

*/

typedef struct {


	/**
	\brief where to send non-error messages; defaults to stdout

	Set this using Logger_setOutputDest()
	*/
	FILE* _output ;

	/**
	\brief where to send error messages; defaults to stderr
	Set this using Logger_setErrorDest()
	*/
	FILE* _error ;

	/**
	\brief messages with a level less than or equal to this will be printed

	set this using Logger_setLevel()
	*/

	int _level ;

	/** \brief log trace messages */
	LogFunction trace ;

	/** \brief log trace messages, with an identifier */
	LogFunctionWithID traceWithID ;

	/** \brief log debug messages */
	LogFunction debug ;

	/** \brief log debug messages, with an identifier */
	LogFunctionWithID debugWithID ;

	/** \brief log informational (standard) messages */
	LogFunction info ;

	/** \brief log informational messages, with an identifier */
	LogFunctionWithID infoWithID ;

	/** \brief log warning messages */
	LogFunction warn ;

	/** \brief log warning messages, with an identifier */
	LogFunctionWithID warnWithID ;

	/** \brief log error messages */
	LogFunction error ;

	/** \brief log error messages, with an identifier */
	LogFunctionWithID errorWithID ;

	/** \brief explicitly set the level of printed messages, using one of the constants from LogLevels */
	SetLevelFunction setLevel ;

	/** \brief increase the number of messages output */
	LevelIncrementFunction levelUp ;

	/** \brief decrease the number of messages output */
	LevelIncrementFunction levelDown ;

} Logger ;

/** \brief return a reference to the singleton Logger instance */
Logger* Logger_getInstance() ;

/** \brief set destination of non-error messages */
void Logger_setOutputDest( FILE* dest ) ;

/** \brief set destination of error messages */
void Logger_setErrorDest( FILE* dest ) ;

/**
\brief initialize the singleton Logger instance

This essentially comes down to assigning function pointers
to the Logger struct's member variables.

- default log level is "info"
*/

void Logger_init() ;

#endif /* #ifndef LOGGER_H */
