#include<stdio.h>
#include<stdarg.h>	/* va_start() , va_end(), va_list */

#include "Logger.h"

/* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =  */

/* BEGIN: private functions and data */

/*
LogLevels and LogLabels (in the public header file) must be kept in sync.
*/
static const char* LogLabels[] = {
	NULL   , /* not used */
	"[!] " ,
	"[W] " ,
	""     ,
	"[D] " ,
	"[T] " ,
	NULL     /* not used */
} ;

/* - - - - - - - - - - - - - - - - - - - - */

/** \brief return the current log level */
static int Logger_getLevel() ;

static void Logger_printMessage( int level , const char* format , va_list args ) ;
static void Logger_trace( const char* format , ... ) ;
static void Logger_debug( const char* format , ... ) ;
static void Logger_info( const char* format , ... ) ;
static void Logger_warn( const char* format , ... ) ;
static void Logger_error( const char* format , ... ) ;

static void Logger_printMessageWithID( int level , const char* id , const char* format , va_list args ) ;
static void Logger_traceWithID( const char* id , const char* format , ... ) ;
static void Logger_debugWithID( const char* id , const char* format , ... ) ;
static void Logger_infoWithID( const char* id , const char* format , ... ) ;
static void Logger_warnWithID( const char* id , const char* format , ... ) ;
static void Logger_errorWithID( const char* id , const char* format , ... ) ;

/** \brief explicitly set the level of printed messages, using one of the constants from LogLevels */
static void Logger_setLevel( int level ) ;

/** \brief increase the number of messages output */
static void Logger_levelUp( void ) ;

/** \brief decrease the number of messages output */
static void Logger_levelDown( void ) ;

/** \brief get the output channel for non-error messages */
FILE* Logger_getOutputDest() ;

/** \brief get the output channel for error messages */
FILE* Logger_getErrorDest() ;

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_printMessage( int level , const char* format , va_list args ){

	if( level <= Logger_getLevel() ){

		FILE* output ;

		if( level == Logger_LevelError ){
			output = Logger_getErrorDest() ;
		} else {
			output = Logger_getOutputDest() ;
		}


		fprintf( output , "%s" , LogLabels[level] ) ;
		vfprintf( output , format , args ) ;
		fprintf( output , "\n" ) ;

	}

	return ;

} /* Logger_printMessage() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_trace( const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessage( Logger_LevelTrace , format , args ) ;
	return ;

} /* Logger_trace() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_debug( const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessage( Logger_LevelDebug , format , args ) ;
	return ;

} /* Logger_debug() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_info( const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessage( Logger_LevelInfo , format , args ) ;
	return ;

} /* Logger_info() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_warn( const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessage( Logger_LevelWarn , format , args ) ;
	return ;

} /* Logger_warn() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_error( const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessage( Logger_LevelError , format , args ) ;
	return ;

} /* Logger_error() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_printMessageWithID( int level , const char* id , const char* format , va_list args ){

	if( level <= Logger_getLevel() ){

		FILE* output ;

		if( level == Logger_LevelError ){
			output = Logger_getErrorDest() ;
		} else {
			output = Logger_getOutputDest() ;
		}


		fprintf( output , "%s%s: " , LogLabels[level] , id ) ;
		vfprintf( output , format , args ) ;
		fprintf( output , "\n" ) ;

	}

	return ;

} /* Logger_printMessageWithID() */


/* - - - - - - - - - - - - - - - - - - - - */

void Logger_traceWithID( const char* id , const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessageWithID( Logger_LevelTrace , id , format , args ) ;
	return ;

} /* Logger_traceWithID() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_debugWithID( const char* id , const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessageWithID( Logger_LevelDebug , id , format , args ) ;
	return ;

} /* Logger_debugWithID() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_infoWithID( const char* id , const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessageWithID( Logger_LevelInfo , id , format , args ) ;
	return ;

} /* Logger_infoWithID() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_warnWithID( const char* id , const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessageWithID( Logger_LevelWarn , id , format , args ) ;
	return ;

} /* Logger_warnWithID() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_errorWithID( const char* id , const char* format , ... ){

	va_list args ;
	va_start( args , format ) ;
	Logger_printMessageWithID( Logger_LevelError , id , format , args ) ;
	return ;

} /* Logger_errorWithID() */

/* - - - - - - - - - - - - - - - - - - - - */

int Logger_getLevel(){
	return( Logger_getInstance()->_level ) ;
} /* Logger_getLevel() */

void Logger_setLevel( int level ){

	Logger_getInstance()->_level = level ;
	return ;

} /* Logger_setLevel() */

/* - - - - - - - - - - - - - - - - - - - - */

FILE* Logger_getOutputDest(){
	return( Logger_getInstance()->_output ) ;
} /* Logger_setOutputDest() */

void Logger_setOutputDest( FILE* dest ){
	Logger_getInstance()->_output = dest ;
	return ;
} /* Logger_setOutputDest() */

/* - - - - - - - - - - - - - - - - - - - - */

FILE* Logger_getErrorDest(){
	return( Logger_getInstance()->_error ) ;
} /* Logger_setErrorDest() */

void Logger_setErrorDest( FILE* dest ){
	Logger_getInstance()->_error = dest ;
	return ;
} /* Logger_setErrorDest() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_levelUp( void ){

	if( Logger_getLevel() < Logger_LevelEverything ){
		++( Logger_getInstance()->_level ) ;
	}

	return ;

} /* Logger_levelUp() */

/* - - - - - - - - - - - - - - - - - - - - */

void Logger_levelDown( void ){

	if( Logger_getLevel() > Logger_LevelNothing ){
		--( Logger_getInstance()->_level ) ;
	}

	return ;

} /* Logger_levelDown() */

/* = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =  */

/* BEGIN: public/global functions */

Logger* Logger_getInstance(){

	static Logger _instance ;

	/* printf( "\n\nLOGGER: ADDRESS %p\n\n" , &_instance ) ; */

	return( &_instance ) ;

} /* Logger_getInstance() */


/* - - - - - - - - - - - - - - - - - - - - */

void Logger_init(){

	Logger* obj = Logger_getInstance() ;

	obj->_output = stdout ;
	obj->_error = stderr ;

	obj->_level = Logger_LevelInfo ;

	obj->trace =  Logger_trace ;
	obj->debug =  Logger_debug ;
	obj->info =  Logger_info ;
	obj->warn =  Logger_warn ;
	obj->error =  Logger_error ;


	obj->traceWithID =  Logger_traceWithID ;
	obj->debugWithID =  Logger_debugWithID ;
	obj->infoWithID =  Logger_infoWithID ;
	obj->warnWithID =  Logger_warnWithID ;
	obj->errorWithID =  Logger_errorWithID ;

	obj->setLevel = Logger_setLevel ;

	obj->levelUp = Logger_levelUp ;
	obj->levelDown = Logger_levelDown ;

	return ;

} /* Logger_init() */

/* - - - - - - - - - - - - - - - - - - - - */

/* EOF Logger.c */
