/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.RecoverySource;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.StoreStats;
import org.elasticsearch.indices.recovery.RecoverySettings;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentFragment;
import org.elasticsearch.xcontent.ToXContentObject;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentFactory;

public class RecoveryState
implements ToXContentFragment,
Writeable {
    private Stage stage;
    private final Index index;
    private final Translog translog;
    private final VerifyIndex verifyIndex;
    private final Timer timer;
    private RecoverySource recoverySource;
    private ShardId shardId;
    @Nullable
    private DiscoveryNode sourceNode;
    private DiscoveryNode targetNode;
    private boolean primary;

    public RecoveryState(ShardRouting shardRouting, DiscoveryNode targetNode, @Nullable DiscoveryNode sourceNode) {
        this(shardRouting, targetNode, sourceNode, new Index());
    }

    public RecoveryState(ShardRouting shardRouting, DiscoveryNode targetNode, @Nullable DiscoveryNode sourceNode, Index index) {
        assert (shardRouting.initializing()) : "only allow initializing shard routing to be recovered: " + shardRouting;
        RecoverySource recoverySource = shardRouting.recoverySource();
        assert (recoverySource.getType() == RecoverySource.Type.PEER == (sourceNode != null)) : "peer recovery requires source node, recovery type: " + (Object)((Object)recoverySource.getType()) + " source node: " + sourceNode;
        this.shardId = shardRouting.shardId();
        this.primary = shardRouting.primary();
        this.recoverySource = recoverySource;
        this.sourceNode = sourceNode;
        this.targetNode = targetNode;
        this.stage = Stage.INIT;
        this.index = index;
        this.translog = new Translog();
        this.verifyIndex = new VerifyIndex();
        this.timer = new Timer();
        this.timer.start();
    }

    public RecoveryState(StreamInput in) throws IOException {
        this.timer = new Timer(in);
        this.stage = Stage.fromId(in.readByte());
        this.shardId = new ShardId(in);
        this.recoverySource = RecoverySource.readFrom(in);
        this.targetNode = new DiscoveryNode(in);
        this.sourceNode = in.readOptionalWriteable(DiscoveryNode::new);
        this.index = new Index(in);
        this.translog = new Translog(in);
        this.verifyIndex = new VerifyIndex(in);
        this.primary = in.readBoolean();
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        this.timer.writeTo(out);
        out.writeByte(this.stage.id());
        this.shardId.writeTo(out);
        this.recoverySource.writeTo(out);
        this.targetNode.writeTo(out);
        out.writeOptionalWriteable(this.sourceNode);
        this.index.writeTo(out);
        this.translog.writeTo(out);
        this.verifyIndex.writeTo(out);
        out.writeBoolean(this.primary);
    }

    public ShardId getShardId() {
        return this.shardId;
    }

    public synchronized Stage getStage() {
        return this.stage;
    }

    protected void validateAndSetStage(Stage expected, Stage next) {
        if (this.stage != expected) {
            assert (false) : "can't move recovery to stage [" + (Object)((Object)next) + "]. current stage: [" + (Object)((Object)this.stage) + "] (expected [" + (Object)((Object)expected) + "])";
            throw new IllegalStateException("can't move recovery to stage [" + (Object)((Object)next) + "]. current stage: [" + (Object)((Object)this.stage) + "] (expected [" + (Object)((Object)expected) + "])");
        }
        this.stage = next;
    }

    public synchronized void validateCurrentStage(Stage expected) {
        if (this.stage != expected) {
            assert (false) : "expected stage [" + (Object)((Object)expected) + "]; but current stage is [" + (Object)((Object)this.stage) + "]";
            throw new IllegalStateException("expected stage [" + (Object)((Object)expected) + "] but current stage is [" + (Object)((Object)this.stage) + "]");
        }
    }

    public synchronized RecoveryState setStage(Stage stage) {
        switch (stage) {
            case INIT: {
                this.stage = Stage.INIT;
                this.getIndex().reset();
                this.getVerifyIndex().reset();
                this.getTranslog().reset();
                break;
            }
            case INDEX: {
                this.validateAndSetStage(Stage.INIT, stage);
                this.getIndex().start();
                break;
            }
            case VERIFY_INDEX: {
                this.validateAndSetStage(Stage.INDEX, stage);
                this.getIndex().stop();
                this.getVerifyIndex().start();
                break;
            }
            case TRANSLOG: {
                this.validateAndSetStage(Stage.VERIFY_INDEX, stage);
                this.getVerifyIndex().stop();
                this.getTranslog().start();
                break;
            }
            case FINALIZE: {
                assert (this.getIndex().bytesStillToRecover() >= 0L) : "moving to stage FINALIZE without completing file details";
                this.validateAndSetStage(Stage.TRANSLOG, stage);
                this.getTranslog().stop();
                break;
            }
            case DONE: {
                this.validateAndSetStage(Stage.FINALIZE, stage);
                this.getTimer().stop();
                break;
            }
            default: {
                throw new IllegalArgumentException("unknown RecoveryState.Stage [" + (Object)((Object)stage) + "]");
            }
        }
        return this;
    }

    public synchronized RecoveryState setLocalTranslogStage() {
        return this.setStage(Stage.TRANSLOG);
    }

    public synchronized RecoveryState setRemoteTranslogStage() {
        return this.setStage(Stage.TRANSLOG);
    }

    public Index getIndex() {
        return this.index;
    }

    public VerifyIndex getVerifyIndex() {
        return this.verifyIndex;
    }

    public Translog getTranslog() {
        return this.translog;
    }

    public Timer getTimer() {
        return this.timer;
    }

    public RecoverySource getRecoverySource() {
        return this.recoverySource;
    }

    @Nullable
    public DiscoveryNode getSourceNode() {
        return this.sourceNode;
    }

    public DiscoveryNode getTargetNode() {
        return this.targetNode;
    }

    public boolean getPrimary() {
        return this.primary;
    }

    public static RecoveryState readRecoveryState(StreamInput in) throws IOException {
        return new RecoveryState(in);
    }

    @Override
    public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        builder.field("id", this.shardId.id());
        builder.field("type", this.recoverySource.getType());
        builder.field("stage", this.stage.toString());
        builder.field("primary", this.primary);
        builder.timeField("start_time_in_millis", "start_time", this.timer.startTime);
        if (this.timer.stopTime > 0L) {
            builder.timeField("stop_time_in_millis", "stop_time", this.timer.stopTime);
        }
        builder.humanReadableField("total_time_in_millis", "total_time", new TimeValue(this.timer.time()));
        if (this.recoverySource.getType() == RecoverySource.Type.PEER) {
            builder.startObject("source");
            builder.field("id", this.sourceNode.getId());
            builder.field("host", this.sourceNode.getHostName());
            builder.field("transport_address", this.sourceNode.getAddress().toString());
            builder.field("ip", this.sourceNode.getHostAddress());
            builder.field("name", this.sourceNode.getName());
            builder.endObject();
        } else {
            builder.startObject("source");
            this.recoverySource.addAdditionalFields(builder, params);
            builder.endObject();
        }
        builder.startObject("target");
        builder.field("id", this.targetNode.getId());
        builder.field("host", this.targetNode.getHostName());
        builder.field("transport_address", this.targetNode.getAddress().toString());
        builder.field("ip", this.targetNode.getHostAddress());
        builder.field("name", this.targetNode.getName());
        builder.endObject();
        builder.startObject("index");
        this.index.toXContent(builder, params);
        builder.endObject();
        builder.startObject("translog");
        this.translog.toXContent(builder, params);
        builder.endObject();
        builder.startObject("verify_index");
        this.verifyIndex.toXContent(builder, params);
        builder.endObject();
        return builder;
    }

    public static class Index
    extends Timer
    implements ToXContentFragment,
    Writeable {
        protected final RecoveryFilesDetails fileDetails;
        public static final long UNKNOWN = -1L;
        private long sourceThrottlingInNanos = -1L;
        private long targetThrottleTimeInNanos = -1L;

        public Index() {
            this(new RecoveryFilesDetails());
        }

        public Index(RecoveryFilesDetails recoveryFilesDetails) {
            this.fileDetails = recoveryFilesDetails;
        }

        public Index(StreamInput in) throws IOException {
            super(in);
            this.fileDetails = new RecoveryFilesDetails(in);
            this.sourceThrottlingInNanos = in.readLong();
            this.targetThrottleTimeInNanos = in.readLong();
        }

        @Override
        public synchronized void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.fileDetails.writeTo(out);
            out.writeLong(this.sourceThrottlingInNanos);
            out.writeLong(this.targetThrottleTimeInNanos);
        }

        public synchronized List<FileDetail> fileDetails() {
            return Collections.unmodifiableList(new ArrayList<FileDetail>(this.fileDetails.values()));
        }

        @Override
        public synchronized void reset() {
            super.reset();
            this.fileDetails.clear();
            this.sourceThrottlingInNanos = -1L;
            this.targetThrottleTimeInNanos = -1L;
        }

        public synchronized void addFileDetail(String name, long length, boolean reused) {
            this.fileDetails.addFileDetails(name, length, reused);
        }

        public synchronized void setFileDetailsComplete() {
            this.fileDetails.setComplete();
        }

        public synchronized void addRecoveredBytesToFile(String name, long bytes) {
            this.fileDetails.addRecoveredBytesToFile(name, bytes);
        }

        public synchronized void resetRecoveredBytesOfFile(String name) {
            this.fileDetails.resetRecoveredBytesOfFile(name);
        }

        public synchronized void addRecoveredFromSnapshotBytesToFile(String name, long bytes) {
            this.fileDetails.addRecoveredFromSnapshotBytesToFile(name, bytes);
        }

        public synchronized void addSourceThrottling(long timeInNanos) {
            this.sourceThrottlingInNanos = this.sourceThrottlingInNanos == -1L ? timeInNanos : (this.sourceThrottlingInNanos += timeInNanos);
        }

        public synchronized void addTargetThrottling(long timeInNanos) {
            this.targetThrottleTimeInNanos = this.targetThrottleTimeInNanos == -1L ? timeInNanos : (this.targetThrottleTimeInNanos += timeInNanos);
        }

        public synchronized TimeValue sourceThrottling() {
            return TimeValue.timeValueNanos(this.sourceThrottlingInNanos);
        }

        public synchronized TimeValue targetThrottling() {
            return TimeValue.timeValueNanos(this.targetThrottleTimeInNanos);
        }

        public synchronized int totalFileCount() {
            return this.fileDetails.size();
        }

        public synchronized int totalRecoverFiles() {
            int total = 0;
            for (FileDetail file : this.fileDetails.values()) {
                if (file.reused()) continue;
                ++total;
            }
            return total;
        }

        public synchronized int recoveredFileCount() {
            int count = 0;
            for (FileDetail file : this.fileDetails.values()) {
                if (!file.fullyRecovered()) continue;
                ++count;
            }
            return count;
        }

        public synchronized float recoveredFilesPercent() {
            int total = 0;
            int recovered = 0;
            for (FileDetail file : this.fileDetails.values()) {
                if (file.reused()) continue;
                ++total;
                if (!file.fullyRecovered()) continue;
                ++recovered;
            }
            if (total == 0 && this.fileDetails.size() == 0) {
                return 0.0f;
            }
            if (total == recovered) {
                return 100.0f;
            }
            float result = 100.0f * ((float)recovered / (float)total);
            return result;
        }

        public synchronized long totalBytes() {
            long total = 0L;
            for (FileDetail file : this.fileDetails.values()) {
                total += file.length();
            }
            return total;
        }

        public synchronized long recoveredBytes() {
            long recovered = 0L;
            for (FileDetail file : this.fileDetails.values()) {
                recovered += file.recovered();
            }
            return recovered;
        }

        public synchronized long recoveredFromSnapshotBytes() {
            long recoveredFromSnapshot = 0L;
            for (FileDetail fileDetail : this.fileDetails.values()) {
                recoveredFromSnapshot += fileDetail.recoveredFromSnapshot();
            }
            return recoveredFromSnapshot;
        }

        public synchronized long totalRecoverBytes() {
            long total = 0L;
            for (FileDetail file : this.fileDetails.values()) {
                if (file.reused()) continue;
                total += file.length();
            }
            return total;
        }

        public synchronized long bytesStillToRecover() {
            if (!this.fileDetails.isComplete()) {
                return -1L;
            }
            long total = 0L;
            for (FileDetail file : this.fileDetails.values()) {
                if (file.reused()) continue;
                total += file.length() - file.recovered();
            }
            return total;
        }

        public synchronized float recoveredBytesPercent() {
            long total = 0L;
            long recovered = 0L;
            for (FileDetail file : this.fileDetails.values()) {
                if (file.reused()) continue;
                total += file.length();
                recovered += file.recovered();
            }
            if (total == 0L && this.fileDetails.size() == 0) {
                return 0.0f;
            }
            if (total == recovered) {
                return 100.0f;
            }
            return 100.0f * (float)recovered / (float)total;
        }

        public synchronized int reusedFileCount() {
            int reused = 0;
            for (FileDetail file : this.fileDetails.values()) {
                if (!file.reused()) continue;
                ++reused;
            }
            return reused;
        }

        public synchronized long reusedBytes() {
            long reused = 0L;
            for (FileDetail file : this.fileDetails.values()) {
                if (!file.reused()) continue;
                reused += file.length();
            }
            return reused;
        }

        @Override
        public synchronized XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject("size");
            builder.humanReadableField("total_in_bytes", "total", new ByteSizeValue(this.totalBytes()));
            builder.humanReadableField("reused_in_bytes", "reused", new ByteSizeValue(this.reusedBytes()));
            builder.humanReadableField("recovered_in_bytes", "recovered", new ByteSizeValue(this.recoveredBytes()));
            builder.humanReadableField("recovered_from_snapshot_in_bytes", "recovered_from_snapshot", new ByteSizeValue(this.recoveredFromSnapshotBytes()));
            builder.field("percent", String.format(Locale.ROOT, "%1.1f%%", Float.valueOf(this.recoveredBytesPercent())));
            builder.endObject();
            builder.startObject("files");
            builder.field("total", this.totalFileCount());
            builder.field("reused", this.reusedFileCount());
            builder.field("recovered", this.recoveredFileCount());
            builder.field("percent", String.format(Locale.ROOT, "%1.1f%%", Float.valueOf(this.recoveredFilesPercent())));
            this.fileDetails.toXContent(builder, params);
            builder.endObject();
            builder.humanReadableField("total_time_in_millis", "total_time", new TimeValue(this.time()));
            builder.humanReadableField("source_throttle_time_in_millis", "source_throttle_time", this.sourceThrottling());
            builder.humanReadableField("target_throttle_time_in_millis", "target_throttle_time", this.targetThrottling());
            return builder;
        }

        public synchronized String toString() {
            try {
                XContentBuilder builder = XContentFactory.jsonBuilder().prettyPrint();
                builder.startObject();
                this.toXContent(builder, EMPTY_PARAMS);
                builder.endObject();
                return Strings.toString(builder);
            }
            catch (IOException e) {
                return "{ \"error\" : \"" + e.getMessage() + "\"}";
            }
        }

        public synchronized FileDetail getFileDetails(String dest) {
            return this.fileDetails.get(dest);
        }
    }

    public static final class Stage
    extends Enum<Stage> {
        public static final /* enum */ Stage INIT = new Stage(0);
        public static final /* enum */ Stage INDEX = new Stage(1);
        public static final /* enum */ Stage VERIFY_INDEX = new Stage(2);
        public static final /* enum */ Stage TRANSLOG = new Stage(3);
        public static final /* enum */ Stage FINALIZE = new Stage(4);
        public static final /* enum */ Stage DONE = new Stage(5);
        private static final Stage[] STAGES;
        private final byte id;
        private static final /* synthetic */ Stage[] $VALUES;

        public static Stage[] values() {
            return (Stage[])$VALUES.clone();
        }

        public static Stage valueOf(String name) {
            return Enum.valueOf(Stage.class, name);
        }

        private Stage(byte id) {
            this.id = id;
        }

        public byte id() {
            return this.id;
        }

        public static Stage fromId(byte id) {
            if (id < 0 || id >= STAGES.length) {
                throw new IllegalArgumentException("No mapping for id [" + id + "]");
            }
            return STAGES[id];
        }

        private static /* synthetic */ Stage[] $values() {
            return new Stage[]{INIT, INDEX, VERIFY_INDEX, TRANSLOG, FINALIZE, DONE};
        }

        static {
            $VALUES = Stage.$values();
            STAGES = new Stage[Stage.values().length];
            for (Stage stage : Stage.values()) {
                assert (stage.id() < STAGES.length && stage.id() >= 0);
                Stage.STAGES[stage.id] = stage;
            }
        }
    }

    public static class Translog
    extends Timer
    implements ToXContentFragment,
    Writeable {
        public static final int UNKNOWN = -1;
        private int recovered;
        private int total = -1;
        private int totalOnStart = -1;
        private int totalLocal = -1;

        public Translog() {
        }

        public Translog(StreamInput in) throws IOException {
            super(in);
            this.recovered = in.readVInt();
            this.total = in.readVInt();
            this.totalOnStart = in.readVInt();
            if (in.getVersion().onOrAfter(Version.V_7_4_0)) {
                this.totalLocal = in.readVInt();
            }
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeVInt(this.recovered);
            out.writeVInt(this.total);
            out.writeVInt(this.totalOnStart);
            if (out.getVersion().onOrAfter(Version.V_7_4_0)) {
                out.writeVInt(this.totalLocal);
            }
        }

        @Override
        public synchronized void reset() {
            super.reset();
            this.recovered = 0;
            this.total = -1;
            this.totalOnStart = -1;
            this.totalLocal = -1;
        }

        public synchronized void incrementRecoveredOperations() {
            ++this.recovered;
            assert (this.total == -1 || this.total >= this.recovered) : "total, if known, should be > recovered. total [" + this.total + "], recovered [" + this.recovered + "]";
        }

        public synchronized void incrementRecoveredOperations(int ops) {
            this.recovered += ops;
            assert (this.total == -1 || this.total >= this.recovered) : "total, if known, should be > recovered. total [" + this.total + "], recovered [" + this.recovered + "]";
        }

        public synchronized void decrementRecoveredOperations(int ops) {
            this.recovered -= ops;
            assert (this.recovered >= 0) : "recovered operations must be non-negative. Because [" + this.recovered + "] after decrementing [" + ops + "]";
            assert (this.total == -1 || this.total >= this.recovered) : "total, if known, should be > recovered. total [" + this.total + "], recovered [" + this.recovered + "]";
        }

        public synchronized int recoveredOperations() {
            return this.recovered;
        }

        public synchronized int totalOperations() {
            return this.total;
        }

        public synchronized void totalOperations(int total) {
            int n = this.total = this.totalLocal == -1 ? total : this.totalLocal + total;
            assert (total == -1 || this.total >= this.recovered) : "total, if known, should be > recovered. total [" + total + "], recovered [" + this.recovered + "]";
        }

        public synchronized int totalOperationsOnStart() {
            return this.totalOnStart;
        }

        public synchronized void totalOperationsOnStart(int total) {
            this.totalOnStart = this.totalLocal == -1 ? total : this.totalLocal + total;
        }

        public synchronized void totalLocal(int totalLocal) {
            assert (totalLocal >= this.recovered) : totalLocal + " < " + this.recovered;
            this.totalLocal = totalLocal;
        }

        public synchronized int totalLocal() {
            return this.totalLocal;
        }

        public synchronized float recoveredPercent() {
            if (this.total == -1) {
                return -1.0f;
            }
            if (this.total == 0) {
                return 100.0f;
            }
            return (float)this.recovered * 100.0f / (float)this.total;
        }

        @Override
        public synchronized XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.field("recovered", this.recovered);
            builder.field("total", this.total);
            builder.field("percent", String.format(Locale.ROOT, "%1.1f%%", Float.valueOf(this.recoveredPercent())));
            builder.field("total_on_start", this.totalOnStart);
            builder.humanReadableField("total_time_in_millis", "total_time", new TimeValue(this.time()));
            return builder;
        }
    }

    public static class VerifyIndex
    extends Timer
    implements ToXContentFragment,
    Writeable {
        private volatile long checkIndexTime;

        public VerifyIndex() {
        }

        public VerifyIndex(StreamInput in) throws IOException {
            super(in);
            this.checkIndexTime = in.readVLong();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeVLong(this.checkIndexTime);
        }

        @Override
        public void reset() {
            super.reset();
            this.checkIndexTime = 0L;
        }

        public long checkIndexTime() {
            return this.checkIndexTime;
        }

        public void checkIndexTime(long checkIndexTime) {
            this.checkIndexTime = checkIndexTime;
        }

        @Override
        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.humanReadableField("check_index_time_in_millis", "check_index_time", new TimeValue(this.checkIndexTime));
            builder.humanReadableField("total_time_in_millis", "total_time", new TimeValue(this.time()));
            return builder;
        }
    }

    public static class Timer
    implements Writeable {
        protected long startTime = 0L;
        protected long startNanoTime = 0L;
        protected long time = -1L;
        protected long stopTime = 0L;

        public Timer() {
        }

        public Timer(StreamInput in) throws IOException {
            this.startTime = in.readVLong();
            this.startNanoTime = in.readVLong();
            this.stopTime = in.readVLong();
            this.time = in.readVLong();
        }

        @Override
        public synchronized void writeTo(StreamOutput out) throws IOException {
            out.writeVLong(this.startTime);
            out.writeVLong(this.startNanoTime);
            out.writeVLong(this.stopTime);
            out.writeVLong(this.time());
        }

        public synchronized void start() {
            assert (this.startTime == 0L) : "already started";
            this.startTime = System.currentTimeMillis();
            this.startNanoTime = System.nanoTime();
        }

        public synchronized long startTime() {
            return this.startTime;
        }

        public synchronized long time() {
            if (this.startNanoTime == 0L) {
                return 0L;
            }
            if (this.time >= 0L) {
                return this.time;
            }
            return Math.max(0L, TimeValue.nsecToMSec(System.nanoTime() - this.startNanoTime));
        }

        public synchronized long stopTime() {
            return this.stopTime;
        }

        public synchronized void stop() {
            assert (this.stopTime == 0L) : "already stopped";
            this.stopTime = Math.max(System.currentTimeMillis(), this.startTime);
            this.time = TimeValue.nsecToMSec(System.nanoTime() - this.startNanoTime);
            assert (this.time >= 0L);
        }

        public synchronized void reset() {
            this.startTime = 0L;
            this.startNanoTime = 0L;
            this.time = -1L;
            this.stopTime = 0L;
        }

        public long getStartNanoTime() {
            return this.startNanoTime;
        }
    }

    static final class Fields {
        static final String ID = "id";
        static final String TYPE = "type";
        static final String STAGE = "stage";
        static final String PRIMARY = "primary";
        static final String START_TIME = "start_time";
        static final String START_TIME_IN_MILLIS = "start_time_in_millis";
        static final String STOP_TIME = "stop_time";
        static final String STOP_TIME_IN_MILLIS = "stop_time_in_millis";
        static final String TOTAL_TIME = "total_time";
        static final String TOTAL_TIME_IN_MILLIS = "total_time_in_millis";
        static final String SOURCE = "source";
        static final String HOST = "host";
        static final String TRANSPORT_ADDRESS = "transport_address";
        static final String IP = "ip";
        static final String NAME = "name";
        static final String TARGET = "target";
        static final String INDEX = "index";
        static final String TRANSLOG = "translog";
        static final String TOTAL_ON_START = "total_on_start";
        static final String VERIFY_INDEX = "verify_index";
        static final String RECOVERED = "recovered";
        static final String RECOVERED_IN_BYTES = "recovered_in_bytes";
        static final String RECOVERED_FROM_SNAPSHOT = "recovered_from_snapshot";
        static final String RECOVERED_FROM_SNAPSHOT_IN_BYTES = "recovered_from_snapshot_in_bytes";
        static final String CHECK_INDEX_TIME = "check_index_time";
        static final String CHECK_INDEX_TIME_IN_MILLIS = "check_index_time_in_millis";
        static final String LENGTH = "length";
        static final String LENGTH_IN_BYTES = "length_in_bytes";
        static final String FILES = "files";
        static final String TOTAL = "total";
        static final String TOTAL_IN_BYTES = "total_in_bytes";
        static final String REUSED = "reused";
        static final String REUSED_IN_BYTES = "reused_in_bytes";
        static final String PERCENT = "percent";
        static final String DETAILS = "details";
        static final String SIZE = "size";
        static final String SOURCE_THROTTLE_TIME = "source_throttle_time";
        static final String SOURCE_THROTTLE_TIME_IN_MILLIS = "source_throttle_time_in_millis";
        static final String TARGET_THROTTLE_TIME = "target_throttle_time";
        static final String TARGET_THROTTLE_TIME_IN_MILLIS = "target_throttle_time_in_millis";

        Fields() {
        }
    }

    public static class RecoveryFilesDetails
    implements ToXContentFragment,
    Writeable {
        protected final Map<String, FileDetail> fileDetails = new HashMap<String, FileDetail>();
        protected boolean complete;

        public RecoveryFilesDetails() {
        }

        RecoveryFilesDetails(StreamInput in) throws IOException {
            int size = in.readVInt();
            for (int i = 0; i < size; ++i) {
                FileDetail file = new FileDetail(in);
                this.fileDetails.put(file.name, file);
            }
            this.complete = in.getVersion().onOrAfter(StoreStats.RESERVED_BYTES_VERSION) ? in.readBoolean() : !this.fileDetails.isEmpty();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            FileDetail[] files = this.values().toArray(new FileDetail[0]);
            out.writeVInt(files.length);
            for (FileDetail file : files) {
                file.writeTo(out);
            }
            if (out.getVersion().onOrAfter(StoreStats.RESERVED_BYTES_VERSION)) {
                out.writeBoolean(this.complete);
            }
        }

        @Override
        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            if (params.paramAsBoolean("detailed", false)) {
                builder.startArray("details");
                for (FileDetail file : this.values()) {
                    file.toXContent(builder, params);
                }
                builder.endArray();
            }
            return builder;
        }

        public void addFileDetails(String name, long length, boolean reused) {
            assert (!this.complete) : "addFileDetail for [" + name + "] when file details are already complete";
            FileDetail existing = this.fileDetails.put(name, new FileDetail(name, length, reused));
            assert (existing == null) : "file [" + name + "] is already reported";
        }

        public void addRecoveredBytesToFile(String name, long bytes) {
            FileDetail file = this.fileDetails.get(name);
            assert (file != null) : "file [" + name + "] hasn't been reported";
            file.addRecoveredBytes(bytes);
        }

        public void resetRecoveredBytesOfFile(String name) {
            FileDetail file = this.fileDetails.get(name);
            assert (file != null) : "file [" + name + "] hasn't been reported";
            file.resetRecoveredBytes();
        }

        public void addRecoveredFromSnapshotBytesToFile(String name, long bytes) {
            FileDetail file = this.fileDetails.get(name);
            assert (file != null) : "file [" + name + "] hasn't been reported";
            file.addRecoveredFromSnapshotBytes(bytes);
        }

        public FileDetail get(String name) {
            return this.fileDetails.get(name);
        }

        public void setComplete() {
            this.complete = true;
        }

        public int size() {
            return this.fileDetails.size();
        }

        public boolean isEmpty() {
            return this.fileDetails.isEmpty();
        }

        public void clear() {
            this.fileDetails.clear();
            this.complete = false;
        }

        public Collection<FileDetail> values() {
            return this.fileDetails.values();
        }

        public boolean isComplete() {
            return this.complete;
        }
    }

    public static class FileDetail
    implements ToXContentObject,
    Writeable {
        private String name;
        private long length;
        private long recovered;
        private boolean reused;
        private long recoveredFromSnapshot;

        public FileDetail(String name, long length, boolean reused) {
            assert (name != null);
            this.name = name;
            this.length = length;
            this.reused = reused;
        }

        public FileDetail(StreamInput in) throws IOException {
            this.name = in.readString();
            this.length = in.readVLong();
            this.recovered = in.readVLong();
            this.reused = in.readBoolean();
            if (in.getVersion().onOrAfter(RecoverySettings.SNAPSHOT_RECOVERIES_SUPPORTED_VERSION)) {
                this.recoveredFromSnapshot = in.readLong();
            }
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.name);
            out.writeVLong(this.length);
            out.writeVLong(this.recovered);
            out.writeBoolean(this.reused);
            if (out.getVersion().onOrAfter(RecoverySettings.SNAPSHOT_RECOVERIES_SUPPORTED_VERSION)) {
                out.writeLong(this.recoveredFromSnapshot);
            }
        }

        void addRecoveredBytes(long bytes) {
            assert (!this.reused) : "file is marked as reused, can't update recovered bytes";
            assert (bytes >= 0L) : "can't recovered negative bytes. got [" + bytes + "]";
            this.recovered += bytes;
        }

        void resetRecoveredBytes() {
            assert (!this.reused) : "file is marked as reused, can't update recovered bytes";
            this.recovered = 0L;
        }

        void addRecoveredFromSnapshotBytes(long bytes) {
            assert (!this.reused) : "file is marked as reused, can't update recovered bytes";
            assert (bytes >= 0L) : "can't recovered negative bytes. got [" + bytes + "]";
            this.recoveredFromSnapshot += bytes;
            this.recovered += bytes;
        }

        public String name() {
            return this.name;
        }

        public long length() {
            return this.length;
        }

        public long recovered() {
            return this.recovered;
        }

        public long recoveredFromSnapshot() {
            return this.recoveredFromSnapshot;
        }

        public boolean reused() {
            return this.reused;
        }

        boolean fullyRecovered() {
            return !this.reused && this.length == this.recovered;
        }

        @Override
        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject();
            builder.field("name", this.name);
            builder.humanReadableField("length_in_bytes", "length", new ByteSizeValue(this.length));
            builder.field("reused", this.reused);
            builder.humanReadableField("recovered_in_bytes", "recovered", new ByteSizeValue(this.recovered));
            builder.humanReadableField("recovered_from_snapshot_in_bytes", "recovered_from_snapshot", new ByteSizeValue(this.recoveredFromSnapshot));
            builder.endObject();
            return builder;
        }

        public boolean equals(Object obj) {
            if (obj instanceof FileDetail) {
                FileDetail other = (FileDetail)obj;
                return this.name.equals(other.name) && this.length == other.length() && this.reused == other.reused() && this.recovered == other.recovered && this.recoveredFromSnapshot == other.recoveredFromSnapshot;
            }
            return false;
        }

        public int hashCode() {
            int result = this.name.hashCode();
            result = 31 * result + Long.hashCode(this.length);
            result = 31 * result + Long.hashCode(this.recovered);
            result = 31 * result + Long.hashCode(this.recoveredFromSnapshot);
            result = 31 * result + (this.reused ? 1 : 0);
            return result;
        }

        public String toString() {
            return "file (name [" + this.name + "], reused [" + this.reused + "], length [" + this.length + "], recovered [" + this.recovered + "], recovered from snapshot [" + this.recoveredFromSnapshot + "]) ";
        }
    }
}

