/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.process;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Phaser;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.LocalNodeMasterListener;
import org.elasticsearch.cluster.NotMasterException;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.persistent.PersistentTasksClusterService;
import org.elasticsearch.persistent.PersistentTasksCustomMetadata;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.ml.action.OpenJobAction;
import org.elasticsearch.xpack.core.ml.action.StartDataFrameAnalyticsAction;
import org.elasticsearch.xpack.core.ml.dataframe.DataFrameAnalyticsConfig;
import org.elasticsearch.xpack.core.ml.job.config.Job;
import org.elasticsearch.xpack.ml.dataframe.persistence.DataFrameAnalyticsConfigProvider;
import org.elasticsearch.xpack.ml.job.JobManager;
import org.elasticsearch.xpack.ml.job.persistence.JobResultsProvider;

public class MlMemoryTracker
implements LocalNodeMasterListener {
    private static final Duration RECENT_UPDATE_THRESHOLD = Duration.ofMinutes(1L);
    private final Logger logger = LogManager.getLogger(MlMemoryTracker.class);
    private final Map<String, Long> memoryRequirementByAnomalyDetectorJob = new ConcurrentHashMap<String, Long>();
    private final Map<String, Long> memoryRequirementByDataFrameAnalyticsJob = new ConcurrentHashMap<String, Long>();
    private final Map<String, Map<String, Long>> memoryRequirementByTaskName;
    private final List<ActionListener<Void>> fullRefreshCompletionListeners = new ArrayList<ActionListener<Void>>();
    private final ThreadPool threadPool;
    private final ClusterService clusterService;
    private final JobManager jobManager;
    private final JobResultsProvider jobResultsProvider;
    private final DataFrameAnalyticsConfigProvider configProvider;
    private final Phaser stopPhaser;
    private volatile AtomicInteger phase = new AtomicInteger(0);
    private volatile boolean isMaster;
    private volatile boolean stopped;
    private volatile Instant lastUpdateTime;
    private volatile Duration reassignmentRecheckInterval;

    public MlMemoryTracker(Settings settings, ClusterService clusterService, ThreadPool threadPool, JobManager jobManager, JobResultsProvider jobResultsProvider, DataFrameAnalyticsConfigProvider configProvider) {
        this.threadPool = threadPool;
        this.clusterService = clusterService;
        this.jobManager = jobManager;
        this.jobResultsProvider = jobResultsProvider;
        this.configProvider = configProvider;
        this.stopPhaser = new Phaser(1);
        TreeMap<String, Map<String, Long>> memoryRequirementByTaskName = new TreeMap<String, Map<String, Long>>();
        memoryRequirementByTaskName.put("xpack/ml/job", this.memoryRequirementByAnomalyDetectorJob);
        memoryRequirementByTaskName.put("xpack/ml/data_frame/analytics", this.memoryRequirementByDataFrameAnalyticsJob);
        this.memoryRequirementByTaskName = Collections.unmodifiableMap(memoryRequirementByTaskName);
        this.setReassignmentRecheckInterval((TimeValue)PersistentTasksClusterService.CLUSTER_TASKS_ALLOCATION_RECHECK_INTERVAL_SETTING.get(settings));
        clusterService.addLocalNodeMasterListener((LocalNodeMasterListener)this);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(PersistentTasksClusterService.CLUSTER_TASKS_ALLOCATION_RECHECK_INTERVAL_SETTING, this::setReassignmentRecheckInterval);
    }

    private void setReassignmentRecheckInterval(TimeValue recheckInterval) {
        this.reassignmentRecheckInterval = Duration.ofNanos(recheckInterval.getNanos());
    }

    public void onMaster() {
        this.isMaster = true;
        try {
            this.asyncRefresh();
        }
        catch (Exception ex) {
            this.logger.warn("unexpected failure while attempting asynchronous refresh on new master assignment", (Throwable)ex);
        }
        this.logger.trace("ML memory tracker on master");
    }

    public void offMaster() {
        this.isMaster = false;
        this.logger.trace("ML memory tracker off master");
        this.clear();
    }

    public void awaitAndClear(ActionListener<Void> listener) {
        this.logger.trace("awaiting and clearing memory tracker");
        assert (!this.stopPhaser.isTerminated());
        assert (this.stopPhaser.getRegisteredParties() > 0);
        assert (this.stopPhaser.getUnarrivedParties() > 0);
        this.threadPool.executor("ml_utility").execute(() -> {
            try {
                int newPhase = this.stopPhaser.arriveAndAwaitAdvance();
                assert (newPhase > 0);
                this.clear();
                this.phase.incrementAndGet();
                this.logger.trace("completed awaiting and clearing memory tracker");
                listener.onResponse(null);
            }
            catch (Exception e) {
                this.logger.warn("failed to wait for all refresh requests to complete", (Throwable)e);
                listener.onFailure(e);
            }
        });
    }

    private void clear() {
        this.logger.trace("clearing ML Memory tracker contents");
        for (Map<String, Long> memoryRequirementByJob : this.memoryRequirementByTaskName.values()) {
            memoryRequirementByJob.clear();
        }
        this.lastUpdateTime = null;
    }

    public void stop() {
        this.stopped = true;
        this.logger.trace("ML memory tracker stop called");
        assert (!this.stopPhaser.isTerminated());
        assert (this.stopPhaser.getRegisteredParties() > 0);
        assert (this.stopPhaser.getUnarrivedParties() > 0);
        this.stopPhaser.arriveAndAwaitAdvance();
        assert (this.stopPhaser.getPhase() > 0);
        this.logger.debug("ML memory tracker stopped");
    }

    public boolean isRecentlyRefreshed() {
        return this.isRecentlyRefreshed(this.reassignmentRecheckInterval);
    }

    public boolean isRecentlyRefreshed(Duration customDuration) {
        Instant localLastUpdateTime = this.lastUpdateTime;
        return this.isMaster && localLastUpdateTime != null && localLastUpdateTime.plus(RECENT_UPDATE_THRESHOLD).plus(customDuration).isAfter(Instant.now());
    }

    public Long getAnomalyDetectorJobMemoryRequirement(String jobId) {
        return this.getJobMemoryRequirement("xpack/ml/job", jobId);
    }

    public Long getDataFrameAnalyticsJobMemoryRequirement(String id) {
        return this.getJobMemoryRequirement("xpack/ml/data_frame/analytics", id);
    }

    public Long getJobMemoryRequirement(String taskName, String id) {
        if (!this.isMaster) {
            return null;
        }
        Map<String, Long> memoryRequirementByJob = this.memoryRequirementByTaskName.get(taskName);
        if (memoryRequirementByJob == null) {
            return null;
        }
        return memoryRequirementByJob.get(id);
    }

    public void removeAnomalyDetectorJob(String jobId) {
        this.memoryRequirementByAnomalyDetectorJob.remove(jobId);
    }

    public void removeDataFrameAnalyticsJob(String id) {
        this.memoryRequirementByDataFrameAnalyticsJob.remove(id);
    }

    public boolean asyncRefresh() {
        if (this.isMaster) {
            try {
                ActionListener listener = ActionListener.wrap(aVoid -> this.logger.trace("Job memory requirement refresh request completed successfully"), e -> this.logIfNecessary(() -> this.logger.warn("Failed to refresh job memory requirements", (Throwable)e)));
                this.threadPool.executor("ml_utility").execute(() -> this.refresh((PersistentTasksCustomMetadata)this.clusterService.state().getMetadata().custom("persistent_tasks"), (ActionListener<Void>)listener));
                return true;
            }
            catch (EsRejectedExecutionException e2) {
                this.logger.warn("Couldn't schedule ML memory update - node might be shutting down", (Throwable)e2);
            }
        }
        return false;
    }

    public void refreshAnomalyDetectorJobMemoryAndAllOthers(String jobId, ActionListener<Long> listener) {
        if (!this.isMaster) {
            listener.onFailure((Exception)new NotMasterException("Request to refresh anomaly detector memory requirements on non-master node"));
            return;
        }
        PersistentTasksCustomMetadata persistentTasks = (PersistentTasksCustomMetadata)this.clusterService.state().getMetadata().custom("persistent_tasks");
        this.refresh(persistentTasks, (ActionListener<Void>)ActionListener.wrap(aVoid -> this.refreshAnomalyDetectorJobMemory(jobId, listener), arg_0 -> listener.onFailure(arg_0)));
    }

    public void addDataFrameAnalyticsJobMemoryAndRefreshAllOthers(String id, long mem, ActionListener<Void> listener) {
        if (!this.isMaster) {
            listener.onFailure((Exception)new NotMasterException("Request to put data frame analytics memory requirement on non-master node"));
            return;
        }
        this.memoryRequirementByDataFrameAnalyticsJob.put(id, mem + DataFrameAnalyticsConfig.PROCESS_MEMORY_OVERHEAD.getBytes());
        PersistentTasksCustomMetadata persistentTasks = (PersistentTasksCustomMetadata)this.clusterService.state().getMetadata().custom("persistent_tasks");
        this.refresh(persistentTasks, listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void refresh(PersistentTasksCustomMetadata persistentTasks, ActionListener<Void> onCompletion) {
        List<ActionListener<Void>> list = this.fullRefreshCompletionListeners;
        synchronized (list) {
            this.fullRefreshCompletionListeners.add(onCompletion);
            if (this.fullRefreshCompletionListeners.size() > 1) {
                return;
            }
        }
        ActionListener refreshComplete = ActionListener.wrap(aVoid -> {
            List<ActionListener<Void>> list = this.fullRefreshCompletionListeners;
            synchronized (list) {
                assert (!this.fullRefreshCompletionListeners.isEmpty());
                if (this.isMaster) {
                    this.lastUpdateTime = Instant.now();
                    for (ActionListener<Void> listener : this.fullRefreshCompletionListeners) {
                        listener.onResponse(null);
                    }
                    this.logger.trace("ML memory tracker last update time now [{}] and listeners called", (Object)this.lastUpdateTime);
                } else {
                    NotMasterException e = new NotMasterException("Node ceased to be master during ML memory tracker refresh");
                    for (ActionListener<Void> listener : this.fullRefreshCompletionListeners) {
                        listener.onFailure((Exception)e);
                    }
                }
                this.fullRefreshCompletionListeners.clear();
            }
        }, e -> {
            List<ActionListener<Void>> list = this.fullRefreshCompletionListeners;
            synchronized (list) {
                assert (!this.fullRefreshCompletionListeners.isEmpty());
                for (ActionListener<Void> listener : this.fullRefreshCompletionListeners) {
                    listener.onFailure(e);
                }
                this.logIfNecessary(() -> this.logger.warn("ML memory tracker last update failed and listeners called", (Throwable)e));
                this.fullRefreshCompletionListeners.clear();
            }
        });
        if (persistentTasks == null) {
            refreshComplete.onResponse(null);
        } else {
            List mlDataFrameAnalyticsJobTasks = persistentTasks.tasks().stream().filter(task -> "xpack/ml/data_frame/analytics".equals(task.getTaskName())).collect(Collectors.toList());
            ActionListener refreshDataFrameAnalyticsJobs = ActionListener.wrap(aVoid -> this.refreshAllDataFrameAnalyticsJobTasks(mlDataFrameAnalyticsJobTasks, (ActionListener<Void>)refreshComplete), arg_0 -> ((ActionListener)refreshComplete).onFailure(arg_0));
            List mlAnomalyDetectorJobTasks = persistentTasks.tasks().stream().filter(task -> "xpack/ml/job".equals(task.getTaskName())).collect(Collectors.toList());
            this.iterateAnomalyDetectorJobTasks(mlAnomalyDetectorJobTasks.iterator(), (ActionListener<Void>)refreshDataFrameAnalyticsJobs);
        }
    }

    private void iterateAnomalyDetectorJobTasks(Iterator<PersistentTasksCustomMetadata.PersistentTask<?>> iterator, ActionListener<Void> refreshComplete) {
        if (iterator.hasNext()) {
            OpenJobAction.JobParams jobParams = (OpenJobAction.JobParams)iterator.next().getParams();
            this.refreshAnomalyDetectorJobMemory(jobParams.getJobId(), (ActionListener<Long>)ActionListener.wrap(mem -> this.threadPool.executor("ml_utility").execute(() -> this.iterateAnomalyDetectorJobTasks(iterator, refreshComplete)), arg_0 -> refreshComplete.onFailure(arg_0)));
        } else {
            refreshComplete.onResponse(null);
        }
    }

    private void refreshAllDataFrameAnalyticsJobTasks(List<PersistentTasksCustomMetadata.PersistentTask<?>> mlDataFrameAnalyticsJobTasks, ActionListener<Void> listener) {
        if (mlDataFrameAnalyticsJobTasks.isEmpty()) {
            listener.onResponse(null);
            return;
        }
        Set<String> jobsWithTasks = mlDataFrameAnalyticsJobTasks.stream().map(task -> ((StartDataFrameAnalyticsAction.TaskParams)task.getParams()).getId()).collect(Collectors.toSet());
        this.configProvider.getConfigsForJobsWithTasksLeniently(jobsWithTasks, (ActionListener<List<DataFrameAnalyticsConfig>>)ActionListener.wrap(analyticsConfigs -> {
            for (DataFrameAnalyticsConfig analyticsConfig : analyticsConfigs) {
                this.memoryRequirementByDataFrameAnalyticsJob.put(analyticsConfig.getId(), analyticsConfig.getModelMemoryLimit().getBytes() + DataFrameAnalyticsConfig.PROCESS_MEMORY_OVERHEAD.getBytes());
            }
            listener.onResponse(null);
        }, arg_0 -> listener.onFailure(arg_0)));
    }

    public void refreshAnomalyDetectorJobMemory(String jobId, ActionListener<Long> listener) {
        if (!this.isMaster) {
            listener.onFailure((Exception)new NotMasterException("Request to refresh anomaly detector memory requirement on non-master node"));
            return;
        }
        if (this.stopPhaser.register() != this.phase.get()) {
            this.stopPhaser.arriveAndDeregister();
            this.logger.info(() -> new ParameterizedMessage("[{}] not refreshing anomaly detector memory as node is shutting down", (Object)jobId));
            listener.onFailure((Exception)((Object)new EsRejectedExecutionException("Couldn't run ML memory update - node is shutting down")));
            return;
        }
        ActionListener phaserListener = ActionListener.wrap(r -> {
            this.stopPhaser.arriveAndDeregister();
            listener.onResponse(r);
        }, e -> {
            this.stopPhaser.arriveAndDeregister();
            listener.onFailure(e);
        });
        try {
            this.jobResultsProvider.getEstablishedMemoryUsage(jobId, null, null, establishedModelMemoryBytes -> {
                if (establishedModelMemoryBytes <= 0L) {
                    this.setAnomalyDetectorJobMemoryToLimit(jobId, (ActionListener<Long>)phaserListener);
                } else {
                    Long memoryRequirementBytes = establishedModelMemoryBytes + Job.PROCESS_MEMORY_OVERHEAD.getBytes();
                    this.memoryRequirementByAnomalyDetectorJob.put(jobId, memoryRequirementBytes);
                    phaserListener.onResponse((Object)memoryRequirementBytes);
                }
            }, e -> {
                this.logIfNecessary(() -> this.logger.error(() -> new ParameterizedMessage("[{}] failed to calculate anomaly detector job established model memory requirement", (Object)jobId), (Throwable)e));
                this.setAnomalyDetectorJobMemoryToLimit(jobId, (ActionListener<Long>)phaserListener);
            });
        }
        catch (Exception e2) {
            this.logIfNecessary(() -> this.logger.error(() -> new ParameterizedMessage("[{}] failed to calculate anomaly detector job established model memory requirement", (Object)jobId), (Throwable)e2));
            this.setAnomalyDetectorJobMemoryToLimit(jobId, (ActionListener<Long>)phaserListener);
        }
    }

    private void setAnomalyDetectorJobMemoryToLimit(String jobId, ActionListener<Long> listener) {
        this.jobManager.getJob(jobId, (ActionListener<Job>)ActionListener.wrap(job -> {
            Long memoryLimitMb;
            Long l = memoryLimitMb = job.getAnalysisLimits() != null ? job.getAnalysisLimits().getModelMemoryLimit() : null;
            if (memoryLimitMb == null) {
                memoryLimitMb = 4096L;
            }
            Long memoryRequirementBytes = ByteSizeValue.ofMb((long)memoryLimitMb).getBytes() + Job.PROCESS_MEMORY_OVERHEAD.getBytes();
            this.memoryRequirementByAnomalyDetectorJob.put(jobId, memoryRequirementBytes);
            listener.onResponse((Object)memoryRequirementBytes);
        }, e -> {
            if (e instanceof ResourceNotFoundException) {
                this.logger.trace("[{}] anomaly detector job deleted during ML memory update", (Object)jobId);
            } else {
                this.logIfNecessary(() -> this.logger.error(() -> new ParameterizedMessage("[{}] failed to get anomaly detector job during ML memory update", (Object)jobId), (Throwable)e));
            }
            this.memoryRequirementByAnomalyDetectorJob.remove(jobId);
            listener.onResponse(null);
        }));
    }

    private void logIfNecessary(Runnable log) {
        if (this.isMaster && !this.stopped) {
            log.run();
        }
    }
}

