/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc.oidc;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSelector;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.source.JWKSource;
import com.nimbusds.jose.proc.BadJOSEException;
import com.nimbusds.jose.proc.JWSKeySelector;
import com.nimbusds.jose.proc.JWSVerificationKeySelector;
import com.nimbusds.jose.proc.SecurityContext;
import com.nimbusds.jose.util.IOUtils;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.oauth2.sdk.AuthorizationCode;
import com.nimbusds.oauth2.sdk.AuthorizationCodeGrant;
import com.nimbusds.oauth2.sdk.ErrorObject;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.TokenErrorResponse;
import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.oauth2.sdk.auth.ClientSecretJWT;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.BearerTokenError;
import com.nimbusds.oauth2.sdk.util.JSONObjectUtils;
import com.nimbusds.openid.connect.sdk.AuthenticationErrorResponse;
import com.nimbusds.openid.connect.sdk.AuthenticationResponse;
import com.nimbusds.openid.connect.sdk.AuthenticationResponseParser;
import com.nimbusds.openid.connect.sdk.AuthenticationSuccessResponse;
import com.nimbusds.openid.connect.sdk.Nonce;
import com.nimbusds.openid.connect.sdk.OIDCTokenResponse;
import com.nimbusds.openid.connect.sdk.claims.AccessTokenHash;
import com.nimbusds.openid.connect.sdk.token.OIDCTokens;
import com.nimbusds.openid.connect.sdk.validators.AccessTokenValidator;
import com.nimbusds.openid.connect.sdk.validators.IDTokenValidator;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;
import org.apache.commons.codec.Charsets;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthenticationException;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.impl.nio.conn.PoolingNHttpClientConnectionManager;
import org.apache.http.impl.nio.reactor.DefaultConnectingIOReactor;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.nio.conn.NHttpClientConnectionManager;
import org.apache.http.nio.conn.NoopIOSessionStrategy;
import org.apache.http.nio.conn.ssl.SSLIOSessionStrategy;
import org.apache.http.nio.reactor.ConnectingIOReactor;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.SpecialPermission;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.util.concurrent.ListenableFuture;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.security.authc.RealmConfig;
import org.elasticsearch.xpack.core.security.authc.RealmSettings;
import org.elasticsearch.xpack.core.security.authc.oidc.OpenIdConnectRealmSettings;
import org.elasticsearch.xpack.core.ssl.SSLConfiguration;
import org.elasticsearch.xpack.core.ssl.SSLService;
import org.elasticsearch.xpack.security.authc.oidc.OpenIdConnectProviderConfiguration;
import org.elasticsearch.xpack.security.authc.oidc.OpenIdConnectToken;
import org.elasticsearch.xpack.security.authc.oidc.RelyingPartyConfiguration;

public class OpenIdConnectAuthenticator {
    private final RealmConfig realmConfig;
    private final OpenIdConnectProviderConfiguration opConfig;
    private final RelyingPartyConfiguration rpConfig;
    private final SSLService sslService;
    private AtomicReference<IDTokenValidator> idTokenValidator = new AtomicReference();
    private final CloseableHttpAsyncClient httpClient;
    private final ResourceWatcherService watcherService;
    private static final Logger LOGGER = LogManager.getLogger(OpenIdConnectAuthenticator.class);

    public OpenIdConnectAuthenticator(RealmConfig realmConfig, OpenIdConnectProviderConfiguration opConfig, RelyingPartyConfiguration rpConfig, SSLService sslService, ResourceWatcherService watcherService) {
        this.realmConfig = realmConfig;
        this.opConfig = opConfig;
        this.rpConfig = rpConfig;
        this.sslService = sslService;
        this.httpClient = this.createHttpClient();
        this.watcherService = watcherService;
        this.idTokenValidator.set(this.createIdTokenValidator(true));
    }

    OpenIdConnectAuthenticator(RealmConfig realmConfig, OpenIdConnectProviderConfiguration opConfig, RelyingPartyConfiguration rpConfig, SSLService sslService, IDTokenValidator idTokenValidator, ResourceWatcherService watcherService) {
        this.realmConfig = realmConfig;
        this.opConfig = opConfig;
        this.rpConfig = rpConfig;
        this.sslService = sslService;
        this.httpClient = this.createHttpClient();
        this.idTokenValidator.set(idTokenValidator);
        this.watcherService = watcherService;
    }

    public void authenticate(OpenIdConnectToken token, ActionListener<JWTClaimsSet> listener) {
        try {
            AuthenticationResponse authenticationResponse = AuthenticationResponseParser.parse((URI)new URI(token.getRedirectUrl()));
            Nonce expectedNonce = token.getNonce();
            State expectedState = token.getState();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("OpenID Connect Provider redirected user to [{}]. Expected Nonce is [{}] and expected State is [{}]", (Object)token.getRedirectUrl(), (Object)expectedNonce, (Object)expectedState);
            }
            if (authenticationResponse instanceof AuthenticationErrorResponse) {
                ErrorObject error = ((AuthenticationErrorResponse)authenticationResponse).getErrorObject();
                listener.onFailure((Exception)((Object)new ElasticsearchSecurityException("OpenID Connect Provider response indicates authentication failureCode=[{}], Description=[{}]", new Object[]{error.getCode(), error.getDescription()})));
                return;
            }
            AuthenticationSuccessResponse response = authenticationResponse.toSuccessResponse();
            this.validateState(expectedState, response.getState());
            this.validateResponseType(response);
            if (this.rpConfig.getResponseType().impliesCodeFlow()) {
                AuthorizationCode code = response.getAuthorizationCode();
                this.exchangeCodeForToken(code, (ActionListener<Tuple<AccessToken, JWT>>)ActionListener.wrap(tokens -> {
                    AccessToken accessToken = (AccessToken)tokens.v1();
                    JWT idToken = (JWT)tokens.v2();
                    this.validateAccessToken(accessToken, idToken);
                    this.getUserClaims(accessToken, idToken, expectedNonce, true, listener);
                }, arg_0 -> listener.onFailure(arg_0)));
            } else {
                JWT idToken = response.getIDToken();
                AccessToken accessToken = response.getAccessToken();
                this.validateAccessToken(accessToken, idToken);
                this.getUserClaims(accessToken, idToken, expectedNonce, true, listener);
            }
        }
        catch (ElasticsearchSecurityException e) {
            listener.onFailure((Exception)((Object)e));
        }
        catch (Exception e) {
            listener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to consume the OpenID connect response. ", e, new Object[0])));
        }
    }

    private void getUserClaims(@Nullable AccessToken accessToken, JWT idToken, Nonce expectedNonce, boolean shouldRetry, ActionListener<JWTClaimsSet> claimsListener) {
        try {
            JWTClaimsSet verifiedIdTokenClaims = this.idTokenValidator.get().validate(idToken, expectedNonce).toJWTClaimsSet();
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Received and validated the Id Token for the user: [{}]", (Object)verifiedIdTokenClaims);
            }
            Map verifiedIdTokenClaimsObject = verifiedIdTokenClaims.toJSONObject();
            JWTClaimsSet idTokenClaim = new JWTClaimsSet.Builder().claim("id_token_hint", (Object)idToken.serialize()).build();
            OpenIdConnectAuthenticator.mergeObjects((Map<String, Object>)verifiedIdTokenClaimsObject, idTokenClaim.toJSONObject());
            JWTClaimsSet enrichedVerifiedIdTokenClaims = JWTClaimsSet.parse((Map)verifiedIdTokenClaimsObject);
            if (accessToken != null && this.opConfig.getUserinfoEndpoint() != null) {
                this.getAndCombineUserInfoClaims(accessToken, enrichedVerifiedIdTokenClaims, claimsListener);
            } else {
                if (accessToken == null && this.opConfig.getUserinfoEndpoint() != null) {
                    LOGGER.debug("UserInfo endpoint is configured but the OP didn't return an access token so we can't query it");
                } else if (accessToken != null && this.opConfig.getUserinfoEndpoint() == null) {
                    LOGGER.debug("OP returned an access token but the UserInfo endpoint is not configured.");
                }
                claimsListener.onResponse((Object)enrichedVerifiedIdTokenClaims);
            }
        }
        catch (BadJOSEException e) {
            if (shouldRetry && !JWSAlgorithm.Family.HMAC_SHA.contains((Object)this.rpConfig.getSignatureAlgorithm()) && this.opConfig.getJwkSetPath().startsWith("https://")) {
                ((ReloadableJWKSource)((JWSVerificationKeySelector)this.idTokenValidator.get().getJWSKeySelector()).getJWKSource()).triggerReload((ActionListener<Void>)ActionListener.wrap(v -> this.getUserClaims(accessToken, idToken, expectedNonce, false, claimsListener), ex -> {
                    LOGGER.trace("Attempted and failed to refresh JWK cache upon token validation failure", (Throwable)e);
                    claimsListener.onFailure(ex);
                }));
            } else {
                claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to parse or validate the ID Token", (Exception)((Object)e), new Object[0])));
            }
        }
        catch (JOSEException | com.nimbusds.oauth2.sdk.ParseException | ParseException e) {
            claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to parse or validate the ID Token", (Exception)e, new Object[0])));
        }
    }

    private void validateAccessToken(AccessToken accessToken, JWT idToken) {
        try {
            if (this.rpConfig.getResponseType().equals((Object)ResponseType.parse((String)"id_token token")) || this.rpConfig.getResponseType().equals((Object)ResponseType.parse((String)"code"))) {
                assert (accessToken != null) : "Access Token cannot be null for Response Type " + this.rpConfig.getResponseType().toString();
                boolean isValidationOptional = this.rpConfig.getResponseType().equals((Object)ResponseType.parse((String)"code"));
                if (!accessToken.getType().toString().equals("Bearer")) {
                    throw new ElasticsearchSecurityException("Invalid access token type [{}], while [Bearer] was expected", new Object[]{accessToken.getType()});
                }
                String atHashValue = idToken.getJWTClaimsSet().getStringClaim("at_hash");
                if (!Strings.hasText((String)atHashValue)) {
                    if (!isValidationOptional) {
                        throw new ElasticsearchSecurityException("Failed to verify access token. ID Token doesn't contain at_hash claim ", new Object[0]);
                    }
                } else {
                    AccessTokenHash atHash = new AccessTokenHash(atHashValue);
                    JWSAlgorithm jwsAlgorithm = JWSAlgorithm.parse((String)idToken.getHeader().getAlgorithm().getName());
                    AccessTokenValidator.validate((AccessToken)accessToken, (JWSAlgorithm)jwsAlgorithm, (AccessTokenHash)atHash);
                }
            } else if (this.rpConfig.getResponseType().equals((Object)ResponseType.parse((String)"id_token")) && accessToken != null) {
                LOGGER.warn("Access Token incorrectly returned from the OpenId Connect Provider while using \"id_token\" response type.");
            }
        }
        catch (Exception e) {
            throw new ElasticsearchSecurityException("Failed to verify access token.", e, new Object[0]);
        }
    }

    private JWKSet readJwkSetFromFile(String jwkSetPath) throws IOException, ParseException {
        Path path = this.realmConfig.env().configFile().resolve(jwkSetPath);
        String jwkSet = new String(Files.readAllBytes(path), StandardCharsets.UTF_8);
        return JWKSet.parse((String)jwkSet);
    }

    private void validateResponseType(AuthenticationSuccessResponse response) {
        if (!this.rpConfig.getResponseType().equals((Object)response.impliedResponseType())) {
            throw new ElasticsearchSecurityException("Unexpected response type [{}], while [{}] is configured", new Object[]{response.impliedResponseType(), this.rpConfig.getResponseType()});
        }
    }

    private void validateState(State expectedState, State state) {
        if (null == state) {
            throw new ElasticsearchSecurityException("Failed to validate the response, the response did not contain a state parameter", new Object[0]);
        }
        if (null == expectedState) {
            throw new ElasticsearchSecurityException("Failed to validate the response, the user's session did not contain a state parameter", new Object[0]);
        }
        if (!state.equals((Object)expectedState)) {
            throw new ElasticsearchSecurityException("Invalid state parameter [{}], while [{}] was expected", new Object[]{state, expectedState});
        }
    }

    private void getAndCombineUserInfoClaims(AccessToken accessToken, final JWTClaimsSet verifiedIdTokenClaims, final ActionListener<JWTClaimsSet> claimsListener) {
        try {
            HttpGet httpGet = new HttpGet(this.opConfig.getUserinfoEndpoint());
            httpGet.setHeader("Authorization", "Bearer " + accessToken.getValue());
            AccessController.doPrivileged(() -> {
                this.httpClient.execute((HttpUriRequest)httpGet, (FutureCallback)new FutureCallback<HttpResponse>(){

                    public void completed(HttpResponse result) {
                        OpenIdConnectAuthenticator.this.handleUserinfoResponse(result, verifiedIdTokenClaims, (ActionListener<JWTClaimsSet>)claimsListener);
                    }

                    public void failed(Exception ex) {
                        claimsListener.onFailure((Exception)new ElasticsearchSecurityException("Failed to get claims from the Userinfo Endpoint.", ex, new Object[0]));
                    }

                    public void cancelled() {
                        claimsListener.onFailure((Exception)new ElasticsearchSecurityException("Failed to get claims from the Userinfo Endpoint. Request was cancelled", new Object[0]));
                    }
                });
                return null;
            });
        }
        catch (Exception e) {
            claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to get user information from the UserInfo endpoint.", e, new Object[0])));
        }
    }

    private void handleUserinfoResponse(HttpResponse httpResponse, JWTClaimsSet verifiedIdTokenClaims, ActionListener<JWTClaimsSet> claimsListener) {
        try {
            HttpEntity entity = httpResponse.getEntity();
            Header encodingHeader = entity.getContentEncoding();
            Charset encoding = encodingHeader == null ? StandardCharsets.UTF_8 : Charsets.toCharset((String)encodingHeader.getValue());
            Header contentHeader = entity.getContentType();
            String contentAsString = EntityUtils.toString((HttpEntity)entity, (Charset)encoding);
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Received UserInfo Response from OP with status [{}] and content [{}] ", (Object)httpResponse.getStatusLine().getStatusCode(), (Object)contentAsString);
            }
            if (httpResponse.getStatusLine().getStatusCode() == 200) {
                if (ContentType.parse((String)contentHeader.getValue()).getMimeType().equals("application/json")) {
                    JWTClaimsSet userInfoClaims = JWTClaimsSet.parse((String)contentAsString);
                    this.validateUserInfoResponse(userInfoClaims, verifiedIdTokenClaims.getSubject(), claimsListener);
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace("Successfully retrieved user information: [{}]", (Object)userInfoClaims);
                    }
                    Map combinedClaims = verifiedIdTokenClaims.toJSONObject();
                    OpenIdConnectAuthenticator.mergeObjects((Map<String, Object>)combinedClaims, userInfoClaims.toJSONObject());
                    claimsListener.onResponse((Object)JWTClaimsSet.parse((Map)combinedClaims));
                } else if (ContentType.parse((String)contentHeader.getValue()).getMimeType().equals("application/jwt")) {
                    claimsListener.onFailure((Exception)new IllegalStateException("Unable to parse Userinfo Response. Signed/encrypted JWTs arenot currently supported"));
                } else {
                    claimsListener.onFailure((Exception)new IllegalStateException("Unable to parse Userinfo Response. Content type was expected to be [application/json] or [appliation/jwt] but was [" + contentHeader.getValue() + "]"));
                }
            } else {
                Header wwwAuthenticateHeader = httpResponse.getFirstHeader("WWW-Authenticate");
                if (Strings.hasText((String)wwwAuthenticateHeader.getValue())) {
                    BearerTokenError error = BearerTokenError.parse((String)wwwAuthenticateHeader.getValue());
                    claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to get user information from the UserInfo endpoint. Code=[{}], Description=[{}]", new Object[]{error.getCode(), error.getDescription()})));
                } else {
                    claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to get user information from the UserInfo endpoint. Code=[{}], Description=[{}]", new Object[]{httpResponse.getStatusLine().getStatusCode(), httpResponse.getStatusLine().getReasonPhrase()})));
                }
            }
        }
        catch (Exception e) {
            claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to get user information from the UserInfo endpoint.", e, new Object[0])));
        }
    }

    private void validateUserInfoResponse(JWTClaimsSet userInfoClaims, String expectedSub, ActionListener<JWTClaimsSet> claimsListener) {
        if (userInfoClaims.getSubject().isEmpty()) {
            claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Userinfo Response did not contain a sub Claim", new Object[0])));
        } else if (!userInfoClaims.getSubject().equals(expectedSub)) {
            claimsListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Userinfo Response is not valid as it is for subject [{}] while the ID Token was for subject [{}]", new Object[]{userInfoClaims.getSubject(), expectedSub})));
        }
    }

    private void exchangeCodeForToken(AuthorizationCode code, final ActionListener<Tuple<AccessToken, JWT>> tokensListener) {
        try {
            AuthorizationCodeGrant codeGrant = new AuthorizationCodeGrant(code, this.rpConfig.getRedirectUri());
            HttpPost httpPost = new HttpPost(this.opConfig.getTokenEndpoint());
            httpPost.setHeader("Content-type", "application/x-www-form-urlencoded");
            ArrayList<BasicNameValuePair> params = new ArrayList<BasicNameValuePair>();
            for (Map.Entry entry : codeGrant.toParameters().entrySet()) {
                params.add(new BasicNameValuePair((String)entry.getKey(), (String)((List)entry.getValue()).get(0)));
            }
            if (this.rpConfig.getClientAuthenticationMethod().equals((Object)ClientAuthenticationMethod.CLIENT_SECRET_BASIC)) {
                UsernamePasswordCredentials creds = new UsernamePasswordCredentials(URLEncoder.encode(this.rpConfig.getClientId().getValue(), StandardCharsets.UTF_8.name()), URLEncoder.encode(this.rpConfig.getClientSecret().toString(), StandardCharsets.UTF_8.name()));
                httpPost.addHeader(new BasicScheme().authenticate((Credentials)creds, (HttpRequest)httpPost, null));
            } else if (this.rpConfig.getClientAuthenticationMethod().equals((Object)ClientAuthenticationMethod.CLIENT_SECRET_POST)) {
                params.add(new BasicNameValuePair("client_id", this.rpConfig.getClientId().getValue()));
                params.add(new BasicNameValuePair("client_secret", this.rpConfig.getClientSecret().toString()));
            } else if (this.rpConfig.getClientAuthenticationMethod().equals((Object)ClientAuthenticationMethod.CLIENT_SECRET_JWT)) {
                ClientSecretJWT clientSecretJWT = new ClientSecretJWT(this.rpConfig.getClientId(), this.opConfig.getTokenEndpoint(), this.rpConfig.getClientAuthenticationJwtAlgorithm(), new Secret(this.rpConfig.getClientSecret().toString()));
                for (Map.Entry entry : clientSecretJWT.toParameters().entrySet()) {
                    params.add(new BasicNameValuePair((String)entry.getKey(), (String)((List)entry.getValue()).get(0)));
                }
            } else {
                tokensListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to exchange code for Id Token using Token Endpoint.Expected client authentication method to be one of " + OpenIdConnectRealmSettings.CLIENT_AUTH_METHODS + " but was [" + this.rpConfig.getClientAuthenticationMethod() + "]", new Object[0])));
            }
            httpPost.setEntity((HttpEntity)new UrlEncodedFormEntity(params));
            SpecialPermission.check();
            AccessController.doPrivileged(() -> {
                this.httpClient.execute((HttpUriRequest)httpPost, (FutureCallback)new FutureCallback<HttpResponse>(){

                    public void completed(HttpResponse result) {
                        OpenIdConnectAuthenticator.this.handleTokenResponse(result, (ActionListener<Tuple<AccessToken, JWT>>)tokensListener);
                    }

                    public void failed(Exception ex) {
                        tokensListener.onFailure((Exception)new ElasticsearchSecurityException("Failed to exchange code for Id Token using the Token Endpoint.", ex, new Object[0]));
                    }

                    public void cancelled() {
                        String message = "Failed to exchange code for Id Token using the Token Endpoint. Request was cancelled";
                        tokensListener.onFailure((Exception)new ElasticsearchSecurityException("Failed to exchange code for Id Token using the Token Endpoint. Request was cancelled", new Object[0]));
                    }
                });
                return null;
            });
        }
        catch (JOSEException | UnsupportedEncodingException | AuthenticationException e) {
            tokensListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to exchange code for Id Token using the Token Endpoint.", (Exception)e, new Object[0])));
        }
    }

    private void handleTokenResponse(HttpResponse httpResponse, ActionListener<Tuple<AccessToken, JWT>> tokensListener) {
        try {
            HttpEntity entity = httpResponse.getEntity();
            Header encodingHeader = entity.getContentEncoding();
            Header contentHeader = entity.getContentType();
            if (!ContentType.parse((String)contentHeader.getValue()).getMimeType().equals("application/json")) {
                tokensListener.onFailure((Exception)new IllegalStateException("Unable to parse Token Response. Content type was expected to be [application/json] but was [" + contentHeader.getValue() + "]"));
                return;
            }
            Charset encoding = encodingHeader == null ? StandardCharsets.UTF_8 : Charsets.toCharset((String)encodingHeader.getValue());
            RestStatus responseStatus = RestStatus.fromCode((int)httpResponse.getStatusLine().getStatusCode());
            if (RestStatus.OK != responseStatus) {
                String json = EntityUtils.toString((HttpEntity)entity, (Charset)encoding);
                LOGGER.warn("Received Token Response from OP with status [{}] and content [{}]", (Object)responseStatus, (Object)json);
                if (RestStatus.BAD_REQUEST == responseStatus) {
                    TokenErrorResponse tokenErrorResponse = TokenErrorResponse.parse((JSONObject)JSONObjectUtils.parse((String)json));
                    tokensListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to exchange code for Id Token. Code=[{}], Description=[{}]", new Object[]{tokenErrorResponse.getErrorObject().getCode(), tokenErrorResponse.getErrorObject().getDescription()})));
                } else {
                    tokensListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to exchange code for Id Token", new Object[0])));
                }
            } else {
                OIDCTokenResponse oidcTokenResponse = OIDCTokenResponse.parse((JSONObject)JSONObjectUtils.parse((String)EntityUtils.toString((HttpEntity)entity, (Charset)encoding)));
                OIDCTokens oidcTokens = oidcTokenResponse.getOIDCTokens();
                AccessToken accessToken = oidcTokens.getAccessToken();
                JWT idToken = oidcTokens.getIDToken();
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Successfully exchanged code for ID Token [{}] and Access Token [{}]", (Object)idToken, (Object)OpenIdConnectAuthenticator.truncateToken(accessToken.toString()));
                }
                if (idToken == null) {
                    tokensListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Token Response did not contain an ID Token or parsing of the JWT failed.", new Object[0])));
                    return;
                }
                tokensListener.onResponse((Object)new Tuple((Object)accessToken, (Object)idToken));
            }
        }
        catch (Exception e) {
            tokensListener.onFailure((Exception)((Object)new ElasticsearchSecurityException("Failed to exchange code for Id Token using the Token Endpoint. Unable to parse Token Response", e, new Object[0])));
        }
    }

    private static String truncateToken(String input) {
        if (!Strings.hasText((String)input) || input.length() <= 4) {
            return input;
        }
        return input.substring(0, 2) + "***" + input.substring(input.length() - 2);
    }

    private CloseableHttpAsyncClient createHttpClient() {
        try {
            SpecialPermission.check();
            return AccessController.doPrivileged(() -> {
                DefaultConnectingIOReactor ioReactor = new DefaultConnectingIOReactor();
                String sslKey = RealmSettings.realmSslPrefix((RealmConfig.RealmIdentifier)this.realmConfig.identifier());
                SSLConfiguration sslConfiguration = this.sslService.getSSLConfiguration(sslKey);
                SSLContext clientContext = this.sslService.sslContext(sslConfiguration);
                HostnameVerifier verifier = SSLService.getHostnameVerifier((SSLConfiguration)sslConfiguration);
                Registry registry = RegistryBuilder.create().register("http", (Object)NoopIOSessionStrategy.INSTANCE).register("https", (Object)new SSLIOSessionStrategy(clientContext, verifier)).build();
                PoolingNHttpClientConnectionManager connectionManager = new PoolingNHttpClientConnectionManager((ConnectingIOReactor)ioReactor, registry);
                connectionManager.setDefaultMaxPerRoute(((Integer)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_MAX_ENDPOINT_CONNECTIONS)).intValue());
                connectionManager.setMaxTotal(((Integer)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_MAX_CONNECTIONS)).intValue());
                RequestConfig requestConfig = RequestConfig.custom().setConnectTimeout(Math.toIntExact(((TimeValue)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_CONNECT_TIMEOUT)).getMillis())).setConnectionRequestTimeout(Math.toIntExact(((TimeValue)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_CONNECTION_READ_TIMEOUT)).getSeconds())).setSocketTimeout(Math.toIntExact(((TimeValue)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_SOCKET_TIMEOUT)).getMillis())).build();
                HttpAsyncClientBuilder httpAsyncClientBuilder = HttpAsyncClients.custom().setConnectionManager((NHttpClientConnectionManager)connectionManager).setDefaultRequestConfig(requestConfig);
                if (this.realmConfig.hasSetting(OpenIdConnectRealmSettings.HTTP_PROXY_HOST)) {
                    httpAsyncClientBuilder.setProxy(new HttpHost((String)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_PROXY_HOST), ((Integer)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_PROXY_PORT)).intValue(), (String)this.realmConfig.getSetting(OpenIdConnectRealmSettings.HTTP_PROXY_SCHEME)));
                }
                CloseableHttpAsyncClient httpAsyncClient = httpAsyncClientBuilder.build();
                httpAsyncClient.start();
                return httpAsyncClient;
            });
        }
        catch (PrivilegedActionException e) {
            throw new IllegalStateException("Unable to create a HttpAsyncClient instance", e);
        }
    }

    IDTokenValidator createIdTokenValidator(boolean addFileWatcherIfRequired) {
        try {
            IDTokenValidator idTokenValidator;
            JWSAlgorithm requestedAlgorithm = this.rpConfig.getSignatureAlgorithm();
            int allowedClockSkew = Math.toIntExact(((TimeValue)this.realmConfig.getSetting(OpenIdConnectRealmSettings.ALLOWED_CLOCK_SKEW)).getMillis());
            if (JWSAlgorithm.Family.HMAC_SHA.contains((Object)requestedAlgorithm)) {
                Secret clientSecret = new Secret(this.rpConfig.getClientSecret().toString());
                idTokenValidator = new IDTokenValidator(this.opConfig.getIssuer(), this.rpConfig.getClientId(), requestedAlgorithm, clientSecret);
            } else {
                String jwkSetPath = this.opConfig.getJwkSetPath();
                if (jwkSetPath.startsWith("http://")) {
                    throw new IllegalArgumentException("The [http] protocol is not supported as it is insecure. Use [https] instead");
                }
                if (jwkSetPath.startsWith("https://")) {
                    JWSVerificationKeySelector keySelector = new JWSVerificationKeySelector(requestedAlgorithm, new ReloadableJWKSource(new URL(jwkSetPath)));
                    idTokenValidator = new IDTokenValidator(this.opConfig.getIssuer(), this.rpConfig.getClientId(), (JWSKeySelector)keySelector, null);
                } else {
                    if (addFileWatcherIfRequired) {
                        this.setMetadataFileWatcher(jwkSetPath);
                    }
                    JWKSet jwkSet = this.readJwkSetFromFile(jwkSetPath);
                    idTokenValidator = new IDTokenValidator(this.opConfig.getIssuer(), this.rpConfig.getClientId(), requestedAlgorithm, jwkSet);
                }
            }
            idTokenValidator.setMaxClockSkew(allowedClockSkew);
            return idTokenValidator;
        }
        catch (IOException | ParseException e) {
            throw new IllegalStateException("Unable to create a IDTokenValidator instance", e);
        }
    }

    private void setMetadataFileWatcher(String jwkSetPath) throws IOException {
        Path path = this.realmConfig.env().configFile().resolve(jwkSetPath);
        FileWatcher watcher = new FileWatcher(path);
        watcher.addListener((Object)new FileListener(LOGGER, () -> this.idTokenValidator.set(this.createIdTokenValidator(false))));
        this.watcherService.add((ResourceWatcher)watcher, ResourceWatcherService.Frequency.MEDIUM);
    }

    static Map<String, Object> mergeObjects(Map<String, Object> idToken, Map<String, Object> userInfo) {
        for (Map.Entry<String, Object> entry : idToken.entrySet()) {
            Object value1 = entry.getValue();
            Object value2 = userInfo.get(entry.getKey());
            if (value2 == null) continue;
            if (value1 instanceof JSONArray) {
                idToken.put(entry.getKey(), OpenIdConnectAuthenticator.mergeArrays((JSONArray)value1, value2));
                continue;
            }
            if (value1 instanceof Map) {
                idToken.put(entry.getKey(), OpenIdConnectAuthenticator.mergeObjects((Map<String, Object>)((Map)value1), value2));
                continue;
            }
            if (value1.getClass().equals(value2.getClass())) continue;
            if (value1 instanceof Boolean && value2 instanceof String && String.valueOf(value1).equals(value2)) {
                idToken.put(entry.getKey(), value1);
                continue;
            }
            if (value2 instanceof Boolean && value1 instanceof String && String.valueOf(value2).equals(value1)) {
                idToken.put(entry.getKey(), value2);
                continue;
            }
            throw new IllegalStateException("Error merging ID token and userinfo claim value for claim [" + entry.getKey() + "]. Cannot merge [" + value1.getClass().getName() + "] with [" + value2.getClass().getName() + "]");
        }
        for (Map.Entry<String, Object> entry : userInfo.entrySet()) {
            if (idToken.containsKey(entry.getKey())) continue;
            idToken.put(entry.getKey(), entry.getValue());
        }
        return idToken;
    }

    private static Map<String, Object> mergeObjects(Map<String, Object> jsonObject1, Object jsonObject2) {
        if (jsonObject2 == null) {
            return jsonObject1;
        }
        if (jsonObject2 instanceof Map) {
            return OpenIdConnectAuthenticator.mergeObjects(jsonObject1, (Map)jsonObject2);
        }
        throw new IllegalStateException("Error while merging ID token and userinfo claims. Cannot merge a Map with a [" + jsonObject2.getClass().getName() + "]");
    }

    private static JSONArray mergeArrays(JSONArray jsonArray1, Object jsonArray2) {
        if (jsonArray2 == null) {
            return jsonArray1;
        }
        if (jsonArray2 instanceof JSONArray) {
            return OpenIdConnectAuthenticator.mergeArrays(jsonArray1, (JSONArray)jsonArray2);
        }
        if (jsonArray2 instanceof String) {
            jsonArray1.add(jsonArray2);
        }
        return jsonArray1;
    }

    private static JSONArray mergeArrays(JSONArray jsonArray1, JSONArray jsonArray2) {
        jsonArray1.addAll((Collection)jsonArray2);
        return jsonArray1;
    }

    protected void close() {
        try {
            this.httpClient.close();
        }
        catch (IOException e) {
            LOGGER.debug("Unable to close the HttpAsyncClient", (Throwable)e);
        }
    }

    class ReloadableJWKSource<C extends SecurityContext>
    implements JWKSource<C> {
        private volatile JWKSet cachedJwkSet = new JWKSet();
        private final AtomicReference<ListenableFuture<Void>> reloadFutureRef = new AtomicReference();
        private final URL jwkSetPath;

        private ReloadableJWKSource(URL jwkSetPath) {
            this.jwkSetPath = jwkSetPath;
            this.triggerReload((ActionListener<Void>)ActionListener.wrap(success -> LOGGER.trace("Successfully loaded and cached remote JWKSet on startup"), failure -> LOGGER.trace("Failed to load and cache remote JWKSet on startup", (Throwable)failure)));
        }

        public List<JWK> get(JWKSelector jwkSelector, C context) {
            return jwkSelector.select(this.cachedJwkSet);
        }

        void triggerReload(ActionListener<Void> toNotify) {
            ListenableFuture<Void> future = this.reloadFutureRef.get();
            while (future == null) {
                future = new ListenableFuture<Void>();
                if (this.reloadFutureRef.compareAndSet(null, future)) {
                    this.reloadAsync(future);
                    continue;
                }
                future = this.reloadFutureRef.get();
            }
            future.addListener(toNotify);
        }

        void reloadAsync(final ListenableFuture<Void> future) {
            try {
                HttpGet httpGet = new HttpGet(this.jwkSetPath.toURI());
                AccessController.doPrivileged(() -> {
                    OpenIdConnectAuthenticator.this.httpClient.execute((HttpUriRequest)httpGet, (FutureCallback)new FutureCallback<HttpResponse>(){

                        public void completed(HttpResponse result) {
                            try {
                                ReloadableJWKSource.this.cachedJwkSet = JWKSet.parse((String)IOUtils.readInputStreamToString((InputStream)result.getEntity().getContent(), (Charset)StandardCharsets.UTF_8));
                                ReloadableJWKSource.this.reloadFutureRef.set(null);
                                LOGGER.trace("Successfully refreshed and cached remote JWKSet");
                                future.onResponse(null);
                            }
                            catch (Exception e) {
                                this.failed(e);
                            }
                        }

                        public void failed(Exception ex) {
                            future.onFailure((Exception)new ElasticsearchSecurityException("Failed to retrieve remote JWK set.", ex, new Object[0]));
                            ReloadableJWKSource.this.reloadFutureRef.set(null);
                        }

                        public void cancelled() {
                            future.onFailure((Exception)new ElasticsearchSecurityException("Failed to retrieve remote JWK set. Request was cancelled.", new Object[0]));
                            ReloadableJWKSource.this.reloadFutureRef.set(null);
                        }
                    });
                    return null;
                });
            }
            catch (Exception e) {
                future.onFailure(e);
                this.reloadFutureRef.set(null);
            }
        }
    }

    private static class FileListener
    implements FileChangesListener {
        private final Logger logger;
        private final CheckedRunnable<Exception> onChange;

        private FileListener(Logger logger, CheckedRunnable<Exception> onChange) {
            this.logger = logger;
            this.onChange = onChange;
        }

        public void onFileCreated(Path file) {
            this.onFileChanged(file);
        }

        public void onFileDeleted(Path file) {
            this.onFileChanged(file);
        }

        public void onFileChanged(Path file) {
            try {
                this.onChange.run();
            }
            catch (Exception e) {
                this.logger.warn((Message)new ParameterizedMessage("An error occurred while reloading file {}", (Object)file), (Throwable)e);
            }
        }
    }
}

