/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.ContextPreservingActionListener;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.xpack.core.common.IteratingActionListener;
import org.elasticsearch.xpack.core.security.authc.Authentication;
import org.elasticsearch.xpack.core.security.authc.AuthenticationResult;
import org.elasticsearch.xpack.core.security.authc.AuthenticationToken;
import org.elasticsearch.xpack.core.security.authc.Realm;
import org.elasticsearch.xpack.core.security.user.User;
import org.elasticsearch.xpack.security.authc.Authenticator;
import org.elasticsearch.xpack.security.authc.support.RealmUserLookup;

class RealmsAuthenticator
implements Authenticator {
    private static final Logger logger = LogManager.getLogger(RealmsAuthenticator.class);
    private final String nodeName;
    private final AtomicLong numInvalidation;
    private final Cache<String, Realm> lastSuccessfulAuthCache;
    private boolean authenticationTokenExtracted = false;

    RealmsAuthenticator(String nodeName, AtomicLong numInvalidation, Cache<String, Realm> lastSuccessfulAuthCache) {
        this.nodeName = nodeName;
        this.numInvalidation = numInvalidation;
        this.lastSuccessfulAuthCache = lastSuccessfulAuthCache;
    }

    @Override
    public String name() {
        return "realms";
    }

    @Override
    public AuthenticationToken extractCredentials(Authenticator.Context context) {
        AuthenticationToken authenticationToken = this.extractToken(context);
        if (authenticationToken != null) {
            this.authenticationTokenExtracted = true;
        }
        return authenticationToken;
    }

    @Override
    public boolean canBeFollowedByNullTokenHandler() {
        return false == this.authenticationTokenExtracted;
    }

    @Override
    public void authenticate(Authenticator.Context context, ActionListener<Authenticator.Result> listener) {
        if (context.getMostRecentAuthenticationToken() == null) {
            listener.onFailure((Exception)new ElasticsearchSecurityException("authentication token must present for realms authentication", RestStatus.UNAUTHORIZED, new Object[0]));
            return;
        }
        assert (context.getMostRecentAuthenticationToken() != null) : "null token should be handled by fallback authenticator";
        this.consumeToken(context, listener);
    }

    AuthenticationToken extractToken(Authenticator.Context context) {
        try {
            for (Realm realm : context.getDefaultOrderedRealmList()) {
                AuthenticationToken token = realm.token(context.getThreadContext());
                if (token == null) continue;
                logger.trace("Found authentication credentials [{}] for principal [{}] in request [{}]", (Object)token.getClass().getName(), (Object)token.principal(), (Object)context.getRequest());
                return token;
            }
        }
        catch (Exception e) {
            logger.warn("An exception occurred while attempting to find authentication credentials", (Throwable)e);
            throw context.getRequest().exceptionProcessingRequest(e, null);
        }
        if (!context.getUnlicensedRealms().isEmpty()) {
            logger.warn("No authentication credential could be extracted using realms [{}]. Realms [{}] were skipped because they are not permitted on the current license", (Object)Strings.collectionToCommaDelimitedString(context.getDefaultOrderedRealmList()), (Object)Strings.collectionToCommaDelimitedString(context.getUnlicensedRealms()));
        }
        return null;
    }

    private void consumeToken(Authenticator.Context context, ActionListener<Authenticator.Result> listener) {
        AuthenticationToken authenticationToken = context.getMostRecentAuthenticationToken();
        List<Realm> realmsList = this.getRealmList(context, authenticationToken.principal());
        logger.trace("Checking token of type [{}] against [{}] realm(s)", (Object)authenticationToken.getClass().getName(), (Object)realmsList.size());
        long startInvalidation = this.numInvalidation.get();
        LinkedHashMap messages = new LinkedHashMap();
        AtomicReference authenticatedByRef = new AtomicReference();
        AtomicReference authenticationResultRef = new AtomicReference();
        BiConsumer<Realm, ActionListener> realmAuthenticatingConsumer = (realm, userListener) -> {
            if (realm.supports(authenticationToken)) {
                logger.trace("Trying to authenticate [{}] using realm [{}] with token [{}] ", (Object)authenticationToken.principal(), realm, (Object)authenticationToken.getClass().getName());
                realm.authenticate(authenticationToken, ActionListener.wrap(result -> {
                    assert (result != null) : "Realm " + realm + " produced a null authentication result";
                    logger.debug("Authentication of [{}] using realm [{}] with token [{}] was [{}]", (Object)authenticationToken.principal(), realm, (Object)authenticationToken.getClass().getSimpleName(), result);
                    if (result.getStatus() == AuthenticationResult.Status.SUCCESS) {
                        authenticatedByRef.set(new Authentication.RealmRef(realm.name(), realm.type(), this.nodeName));
                        authenticationResultRef.set(result);
                        if (this.lastSuccessfulAuthCache != null && startInvalidation == this.numInvalidation.get()) {
                            this.lastSuccessfulAuthCache.put((Object)authenticationToken.principal(), realm);
                        }
                        userListener.onResponse((Object)result.getUser());
                    } else {
                        context.getRequest().realmAuthenticationFailed(authenticationToken, realm.name());
                        if (result.getStatus() == AuthenticationResult.Status.TERMINATE) {
                            if (result.getException() != null) {
                                logger.info((Message)new ParameterizedMessage("Authentication of [{}] was terminated by realm [{}] - {}", new Object[]{authenticationToken.principal(), realm.name(), result.getMessage()}), (Throwable)result.getException());
                            } else {
                                logger.info("Authentication of [{}] was terminated by realm [{}] - {}", (Object)authenticationToken.principal(), (Object)realm.name(), (Object)result.getMessage());
                            }
                            userListener.onFailure(result.getException());
                        } else {
                            if (result.getMessage() != null) {
                                messages.put(realm, new Tuple((Object)result.getMessage(), (Object)result.getException()));
                            }
                            userListener.onResponse(null);
                        }
                    }
                }, ex -> {
                    logger.warn((Message)new ParameterizedMessage("An error occurred while attempting to authenticate [{}] against realm [{}]", (Object)authenticationToken.principal(), (Object)realm.name()), (Throwable)ex);
                    userListener.onFailure(ex);
                }));
            } else {
                userListener.onResponse(null);
            }
        };
        IteratingActionListener authenticatingListener = new IteratingActionListener((ActionListener)ContextPreservingActionListener.wrapPreservingContext((ActionListener)ActionListener.wrap(user -> {
            if (user == null) {
                this.consumeNullUser(context, messages, listener);
            } else {
                AuthenticationResult result = (AuthenticationResult)authenticationResultRef.get();
                assert (result != null) : "authentication result must not be null when user is not null";
                context.getThreadContext().putTransient(AuthenticationResult.THREAD_CONTEXT_KEY, (Object)result);
                listener.onResponse((Object)Authenticator.Result.success(new Authentication(user, (Authentication.RealmRef)authenticatedByRef.get(), null)));
            }
        }, e -> {
            if (e != null) {
                listener.onFailure((Exception)context.getRequest().exceptionProcessingRequest((Exception)e, authenticationToken));
            } else {
                listener.onFailure((Exception)context.getRequest().authenticationFailed(authenticationToken));
            }
        }), (ThreadContext)context.getThreadContext()), realmAuthenticatingConsumer, realmsList, context.getThreadContext());
        try {
            authenticatingListener.run();
        }
        catch (Exception e2) {
            logger.debug((Message)new ParameterizedMessage("Authentication of [{}] with token [{}] failed", (Object)authenticationToken.principal(), (Object)authenticationToken.getClass().getName()), (Throwable)e2);
            listener.onFailure((Exception)context.getRequest().exceptionProcessingRequest(e2, authenticationToken));
        }
    }

    private void consumeNullUser(Authenticator.Context context, Map<Realm, Tuple<String, Exception>> messages, ActionListener<Authenticator.Result> listener) {
        messages.forEach((realm, tuple) -> {
            String message = (String)tuple.v1();
            String cause = tuple.v2() == null ? "" : " (Caused by " + tuple.v2() + ")";
            logger.warn("Authentication to realm {} failed - {}{}", (Object)realm.name(), (Object)message, (Object)cause);
        });
        if (!context.getUnlicensedRealms().isEmpty()) {
            logger.warn("Authentication failed using realms [{}]. Realms [{}] were skipped because they are not permitted on the current license", (Object)Strings.collectionToCommaDelimitedString(context.getDefaultOrderedRealmList()), (Object)Strings.collectionToCommaDelimitedString(context.getUnlicensedRealms()));
        }
        logger.trace("Failed to authenticate request [{}]", (Object)context.getRequest());
        listener.onFailure((Exception)context.getRequest().authenticationFailed(context.getMostRecentAuthenticationToken()));
    }

    public void lookupRunAsUser(Authenticator.Context context, Authentication authentication, ActionListener<Tuple<User, Authentication.RealmRef>> listener) {
        assert (authentication.getLookedUpBy() == null) : "authentication already has a lookup realm";
        String runAsUsername = context.getThreadContext().getHeader("es-security-runas-user");
        if (runAsUsername != null && !runAsUsername.isEmpty()) {
            logger.trace("Looking up run-as user [{}] for authenticated user [{}]", (Object)runAsUsername, (Object)authentication.getUser().principal());
            RealmUserLookup lookup = new RealmUserLookup(this.getRealmList(context, runAsUsername), context.getThreadContext());
            long startInvalidationNum = this.numInvalidation.get();
            lookup.lookup(runAsUsername, (ActionListener<Tuple<User, Realm>>)ActionListener.wrap(tuple -> {
                if (tuple == null) {
                    logger.debug("Cannot find run-as user [{}] for authenticated user [{}]", (Object)runAsUsername, (Object)authentication.getUser().principal());
                    listener.onResponse(null);
                } else {
                    User foundUser = Objects.requireNonNull((User)tuple.v1());
                    Realm realm = Objects.requireNonNull((Realm)tuple.v2());
                    if (this.lastSuccessfulAuthCache != null && startInvalidationNum == this.numInvalidation.get()) {
                        this.lastSuccessfulAuthCache.computeIfAbsent((Object)runAsUsername, s -> realm);
                    }
                    logger.trace("Using run-as user [{}] with authenticated user [{}]", (Object)foundUser, (Object)authentication.getUser().principal());
                    listener.onResponse((Object)new Tuple((Object)((User)tuple.v1()), (Object)new Authentication.RealmRef(((Realm)tuple.v2()).name(), ((Realm)tuple.v2()).type(), this.nodeName)));
                }
            }, e -> listener.onFailure((Exception)context.getRequest().exceptionProcessingRequest((Exception)e, context.getMostRecentAuthenticationToken()))));
        } else if (runAsUsername == null) {
            listener.onResponse(null);
        } else {
            logger.debug("user [{}] attempted to runAs with an empty username", (Object)authentication.getUser().principal());
            listener.onFailure((Exception)context.getRequest().runAsDenied(new Authentication(new User(runAsUsername, null, authentication.getUser()), authentication.getAuthenticatedBy(), null), context.getMostRecentAuthenticationToken()));
        }
    }

    private List<Realm> getRealmList(Authenticator.Context context, String principal) {
        int index;
        Realm lastSuccess;
        List<Realm> orderedRealmList = context.getDefaultOrderedRealmList();
        if (this.lastSuccessfulAuthCache != null && (lastSuccess = (Realm)this.lastSuccessfulAuthCache.get((Object)principal)) != null && (index = orderedRealmList.indexOf(lastSuccess)) > 0) {
            ArrayList<Realm> smartOrder = new ArrayList<Realm>(orderedRealmList.size());
            smartOrder.add(lastSuccess);
            for (int i = 0; i < orderedRealmList.size(); ++i) {
                if (i == index) continue;
                smartOrder.add(orderedRealmList.get(i));
            }
            assert (smartOrder.size() == orderedRealmList.size() && smartOrder.containsAll(orderedRealmList)) : "Element mismatch between SmartOrder=" + smartOrder + " and DefaultOrder=" + orderedRealmList;
            return Collections.unmodifiableList(smartOrder);
        }
        return orderedRealmList;
    }
}

