/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.watcher;

import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.component.LifecycleListener;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.xpack.core.watcher.WatcherMetadata;
import org.elasticsearch.xpack.core.watcher.WatcherState;
import org.elasticsearch.xpack.watcher.WatcherService;
import org.elasticsearch.xpack.watcher.watch.WatchStoreUtils;

public class WatcherLifeCycleService
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(WatcherLifeCycleService.class);
    private final AtomicReference<WatcherState> state = new AtomicReference<WatcherState>(WatcherState.STARTED);
    private final AtomicReference<List<ShardRouting>> previousShardRoutings = new AtomicReference(Collections.emptyList());
    private volatile boolean shutDown = false;
    private volatile WatcherService watcherService;
    private final EnumSet<WatcherState> stopStates = EnumSet.of(WatcherState.STOPPED, WatcherState.STOPPING);

    WatcherLifeCycleService(ClusterService clusterService, WatcherService watcherService) {
        this.watcherService = watcherService;
        clusterService.addListener((ClusterStateListener)this);
        clusterService.addLifecycleListener(new LifecycleListener(){

            public void beforeStop() {
                WatcherLifeCycleService.this.shutDown();
            }
        });
    }

    synchronized void shutDown() {
        this.state.set(WatcherState.STOPPING);
        this.shutDown = true;
        this.clearAllocationIds();
        this.watcherService.shutDown(() -> {
            this.state.set(WatcherState.STOPPED);
            logger.info("watcher has stopped and shutdown");
        });
    }

    public void clusterChanged(ClusterChangedEvent event) {
        if (event.state().blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) || this.shutDown) {
            this.clearAllocationIds();
            return;
        }
        if (Strings.isNullOrEmpty((String)event.state().nodes().getMasterNodeId())) {
            this.pauseExecution("no master node");
            return;
        }
        if (event.state().getBlocks().hasGlobalBlockWithLevel(ClusterBlockLevel.WRITE)) {
            this.pauseExecution("write level cluster block");
            return;
        }
        boolean isWatcherStoppedManually = this.isWatcherStoppedManually(event.state());
        boolean isStoppedOrStopping = this.stopStates.contains(this.state.get());
        if (!event.state().nodes().getLocalNode().isDataNode() && !isWatcherStoppedManually && isStoppedOrStopping) {
            this.state.set(WatcherState.STARTING);
            this.watcherService.start(event.state(), () -> this.state.set(WatcherState.STARTED));
            return;
        }
        if (isWatcherStoppedManually) {
            if (this.state.get() == WatcherState.STARTED) {
                this.clearAllocationIds();
                boolean stopping = this.state.compareAndSet(WatcherState.STARTED, WatcherState.STOPPING);
                if (stopping) {
                    this.watcherService.stop("watcher manually marked to shutdown by cluster state update", () -> {
                        boolean stopped = this.state.compareAndSet(WatcherState.STOPPING, WatcherState.STOPPED);
                        if (stopped) {
                            logger.info("watcher has stopped");
                        } else {
                            logger.info("watcher has not been stopped. not currently in a stopping state, current state [{}]", (Object)this.state.get());
                        }
                    });
                }
            }
            return;
        }
        DiscoveryNode localNode = event.state().nodes().getLocalNode();
        RoutingNode routingNode = event.state().getRoutingNodes().node(localNode.getId());
        if (routingNode == null) {
            this.pauseExecution("routing node in cluster state undefined. network issue?");
            return;
        }
        IndexMetadata watcherIndexMetadata = WatchStoreUtils.getConcreteIndex(".watches", event.state().metadata());
        if (watcherIndexMetadata == null) {
            this.pauseExecution("no watcher index found");
            return;
        }
        String watchIndex = watcherIndexMetadata.getIndex().getName();
        List localShards = routingNode.shardsWithState(watchIndex, new ShardRoutingState[]{ShardRoutingState.RELOCATING, ShardRoutingState.STARTED});
        if (localShards.isEmpty()) {
            this.pauseExecution("no local watcher shards found");
            return;
        }
        Set localShardIds = localShards.stream().map(ShardRouting::shardId).collect(Collectors.toSet());
        List allShards = event.state().routingTable().index(watchIndex).shardsWithState(ShardRoutingState.STARTED);
        allShards.addAll(event.state().routingTable().index(watchIndex).shardsWithState(ShardRoutingState.RELOCATING));
        List localAffectedShardRoutings = allShards.stream().filter(shardRouting -> localShardIds.contains(shardRouting.shardId())).sorted(Comparator.comparing(ShardRouting::hashCode)).collect(Collectors.toList());
        if (!this.previousShardRoutings.get().equals(localAffectedShardRoutings)) {
            if (this.watcherService.validate(event.state())) {
                this.previousShardRoutings.set(localAffectedShardRoutings);
                if (this.state.get() == WatcherState.STARTED) {
                    this.watcherService.reload(event.state(), "new local watcher shard allocation ids");
                } else if (isStoppedOrStopping) {
                    this.state.set(WatcherState.STARTING);
                    this.watcherService.start(event.state(), () -> this.state.set(WatcherState.STARTED));
                }
            } else {
                this.clearAllocationIds();
                this.state.set(WatcherState.STOPPED);
            }
        }
    }

    private void pauseExecution(String reason) {
        if (this.clearAllocationIds()) {
            this.watcherService.pauseExecution(reason);
        }
        this.state.set(WatcherState.STARTED);
    }

    private boolean isWatcherStoppedManually(ClusterState state) {
        WatcherMetadata watcherMetadata = (WatcherMetadata)state.getMetadata().custom("watcher");
        return watcherMetadata != null && watcherMetadata.manuallyStopped();
    }

    private boolean clearAllocationIds() {
        List previousIds = this.previousShardRoutings.getAndSet(Collections.emptyList());
        return !previousIds.isEmpty();
    }

    List<ShardRouting> shardRoutings() {
        return this.previousShardRoutings.get();
    }

    public Supplier<WatcherState> getState() {
        return () -> this.state.get();
    }
}

