/*
	description: "Generic conformance routines."
	date:		"$Date: 2012-06-09 13:57:06 -0700 (Sat, 09 Jun 2012) $"
	revision:	"$Revision: 88842 $"
	copyright:	"Copyright (c) 1985-2012, Eiffel Software."
	license:	"GPL version 2 see http://www.eiffel.com/licensing/gpl.txt)"
	licensing_options:	"Commercial license is available at http://www.eiffel.com/licensing"
	copying: "[
			This file is part of Eiffel Software's Runtime.
			
			Eiffel Software's Runtime is free software; you can
			redistribute it and/or modify it under the terms of the
			GNU General Public License as published by the Free
			Software Foundation, version 2 of the License
			(available at the URL listed under "license" above).
			
			Eiffel Software's Runtime is distributed in the hope
			that it will be useful,	but WITHOUT ANY WARRANTY;
			without even the implied warranty of MERCHANTABILITY
			or FITNESS FOR A PARTICULAR PURPOSE.
			See the	GNU General Public License for more details.
			
			You should have received a copy of the GNU General Public
			License along with Eiffel Software's Runtime; if not,
			write to the Free Software Foundation, Inc.,
			51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
		]"
	source: "[
			 Eiffel Software
			 356 Storke Road, Goleta, CA 93117 USA
			 Telephone 805-685-1006, Fax 805-685-6869
			 Website http://www.eiffel.com
			 Customer support http://support.eiffel.com
		]"
*/

/*
doc:<file name="gen_conf.c" header="eif_gen_conf.h" version="$Id: gen_conf.c 88842 2012-06-09 20:57:06Z manus $" summary="Generic conformance">
*/

#include "eif_portable.h"
#include "rt_macros.h"
#include "eif_globals.h"
#include "rt_struct.h"
#include "rt_gen_conf.h"
#include "rt_gen_types.h"
#include "rt_malloc.h"
#include "rt_threads.h"
#include "rt_garcol.h"
#include "rt_cecil.h"
#include "rt_assert.h"
#include <ctype.h>
#include <string.h>

/*------------------------------------------------------------------*/
/* Debugging flag. If set, the names of the generated types will be */
/* output to the file 'logfile'. Simple facility.                   */
/*------------------------------------------------------------------*/

/*
doc:	<attribute name="eif_par_table" return_type="struct eif_par_types **" export="shared">
doc:		<summary>Parent tables. Defined by compiler C generated code in `eparents.c'. Changes to this table after melt are stored in melted file and processed by `updated.c' and stored in `eif_par_table2'.</summary>
doc:		<access>Read only through macro `par_info'.</access>
doc:		<indexing>base id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_par_table_size" return_type="EIF_TYPE_INDEX" export="shared">
doc:		<summary>Size of `eif_par_table' table.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_par_table2" return_type="struct eif_par_types **" export="shared">
doc:		<summary>Same as `eif_par_table' except that this one contains both the static definition generated by compiler in `eparents.c' and the melted definition contained in melted file.</summary>
doc:		<access>Read only through macro `par_info'.</access>
doc:		<indexing>base id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_par_table2_size" return_type="EIF_TYPE_INDEX" export="shared">
doc:		<summary>Size of `eif_par_table2' table.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
*/
rt_shared struct eif_par_types **eif_par_table = NULL;
rt_shared EIF_TYPE_INDEX    eif_par_table_size = 0;
rt_shared struct eif_par_types **eif_par_table2 = NULL;
rt_shared EIF_TYPE_INDEX    eif_par_table2_size = 0;

/*
doc:	<attribute name="eif_cid_map" return_type="EIF_TYPE_INDEX *" export="public">
doc:		<summary>Compound id map. Maps compiler generated IDs to themselves and run-time generated IDs to their base IDs. In other word, map full dynamic type to dynamic type. Table is dynamically reallocated except in multithreaded mode where it is by default initialized to a count which represent the greatest type id (MAX_DTYPE).</summary>
doc:		<access>Read/Write (read with macros To_dtype)</access>
doc:		<indexing>full type id</indexing>
doc:		<result>base id</result>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_cid_size" return_type="int" export="private">
doc:		<summary>Number of elements in following structures `eif_cid_map', `eif_derivations', `eif_con_tab', `eif_conf_tab'.</summary>
doc:		<access>Read/Write</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
*/
rt_public EIF_TYPE_INDEX  *eif_cid_map = NULL;
rt_private int  eif_cid_size = 0;

/*
doc:	<attribute name="egc_any_dtype" return_type="EIF_TYPE_INDEX" export="public">
doc:		<summary>Type of ANY. Used to create ARRAY [ANY] from the runtime (e.g. used for `strip'). Value computed in `eif_gen_conf_init'.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization in `eif_gen_conf_init'.</synchronization>
doc:	</attribute>
doc:	<attribute name="tuple_static_type" return_type="EIF_TYPE_INDEX" export="private">
doc:		<summary>Base id of TUPLE.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization in `eif_gen_conf_init'.</synchronization>
doc:		<fixme>Shouldn't we use `egc_tup_dtype' instead?</fixme>
doc:	</attribute>
*/
rt_public EIF_TYPE_INDEX egc_any_dtype = INVALID_DTYPE; /* Precise value determined in init */
rt_private EIF_TYPE_INDEX tuple_static_type = INVALID_DTYPE;

/*------------------------------------------------------------------*/
/* Structure representing a generic derivation. We also use this    */
/* for BIT types to remember their sizes.                           */
/*------------------------------------------------------------------*/

typedef struct eif_gen_der {
	uint32              size;       /* Size of type array/ nr. of bits in BIT type */
	EIF_TYPE_INDEX      hcode;      /* Hash code to speedup search */
	EIF_TYPE_INDEX      *typearr;   /* Array of types (cid) */
	EIF_TYPE_INDEX      *gen_seq;   /* Id sequence which generates this type */
	EIF_TYPE_INDEX      *ptypes;    /* Parent types */
	EIF_TYPE_INDEX      id;         /* Run-time generated id */
	EIF_TYPE_INDEX      base_id;    /* Compiler generated (base) id */
	EIF_TYPE_INDEX      first_id;   /* First matching compiler gen. id */
	EIF_TYPE_INDEX		annotation;	/* Annotation flag for that type. */
	char                *name;      /* Full type name */
	char                is_expanded;/* Is it an expanded type? */
	char                is_bit;     /* Is it a BIT type? */
	char                is_tuple;   /* Is it a TUPLE type? */
	struct eif_gen_der  *next;      /* Next derivation */
} EIF_GEN_DER;
/*------------------------------------------------------------------*/
/* Structure for conformance information. The `lower' ids are the   */
/* usual compiler generated ids. The others are generated here.     */
/*                                                                  */
/* All ids are full type ids                                        */
/* Indexing: full type ids                                          */
/*------------------------------------------------------------------*/

typedef struct {
	EIF_TYPE_INDEX  min_low_id;     /* Minimal lower conforming id */
	EIF_TYPE_INDEX  max_low_id;     /* Maximal lower conforming id */
	EIF_TYPE_INDEX  min_high_id;    /* Minimal high conforming id */
	EIF_TYPE_INDEX  max_high_id;    /* Maximal high conforming id */
	unsigned char   *low_tab;       /* Bit table for lower ids */
	unsigned char   *high_tab;      /* Bit table for high ids */
	unsigned char   *low_comp;      /* Bit table for computed lower conf. */
	unsigned char   *high_comp;     /* Bit table for computed high conf. */
#ifdef EIF_ASSERTIONS
		/* End of all the above arrays for efficient bound checks. */
	unsigned char *low_tab_end;
	unsigned char *high_tab_end;
	unsigned char *low_comp_end;
	unsigned char *high_comp_end;
#endif
} EIF_CONF_TAB;
/*
doc:	<attribute name="eif_first_gen_id" return_type="EIF_TYPE_INDEX" export="private">
doc:		<summary>base id of first generic type. All values below `eif_first_gen_id' do not represent generic classes.</summary>
doc:		<access>Read</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization in `eif_gen_conf_init'.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_next_gen_id" return_type="EIF_TYPE_INDEX" export="shared">
doc:		<summary>ID for next new generic derivation encountered during runtime execution.</summary>
doc:		<access>Read/Write</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
*/
rt_private EIF_TYPE_INDEX  eif_first_gen_id = 0;
rt_shared EIF_TYPE_INDEX  eif_next_gen_id  = 0;

/*
doc:	<attribute name="eif_conf_tab" return_type="EIF_CONF_TAB **" export="private">
doc:		<summary>Conformance tables.</summary>
doc:		<access>Read/Write</access>
doc:		<indexing>full type id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_derivations" return_type="EIF_GEN_DER **" export="private">
doc:		<summary>Generic derivations.</summary>
doc:		<access>Read/Write</access>
doc:		<indexing>full type id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
*/
rt_private EIF_CONF_TAB **eif_conf_tab = NULL;
rt_private EIF_GEN_DER **eif_derivations = NULL;

#ifndef EIF_THREADS
/*
doc:	<attribute name="cid_array" return_type="EIF_TYPE_INDEX [3]" export="private">
doc:		<summary>Static array used by `eif_gen_cid' for `dftype' that are not generics.</summary>
doc:		<access>Read/Write</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>Private per thread data</synchronization>
doc:	</attribute>
*/
rt_private EIF_TYPE_INDEX cid_array [3];

/*
doc:	<attribute name="non_generic_type_names" return_type="char **" export="private">
doc:		<summary>Array indexed by non generic type id which contains their associated type name.</summary>
doc:		<access>Read/Write</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>Private per thread data</synchronization>
doc:	</attribute>
*/
rt_private char ** non_generic_type_names = NULL;
#endif

/*------------------------------------------------------------------*/
/* THREADS.                                                         */
/* Calls to public routines are indirected and protected by a mutex */
/* The indirection avoids problems with recursive calls.            */
/*------------------------------------------------------------------*/

#ifdef  EIF_THREADS

/*
doc:	<attribute name="eif_gen_mutex" return_type="" export="shared">
doc:		<summary>Calls to public routines are indirected and protected by current mutex. Indirection is needed to avoids problem with recursive calls.</summary>
doc:		<access>Read</access>
doc:		<thread_safety>Safe</thread_safety>
doc:	</attribute>
*/
rt_shared EIF_CS_TYPE *eif_gen_mutex = NULL;

rt_public EIF_TYPE_INDEX eifthd_compound_id (EIF_TYPE_INDEX, EIF_TYPE_INDEX, EIF_TYPE_INDEX *);
rt_public EIF_TYPE_INDEX eifthd_final_id (EIF_TYPE_INDEX *, EIF_TYPE_INDEX **, EIF_TYPE_INDEX, int );
rt_shared uint32 eifthd_gen_count_with_dftype (EIF_TYPE_INDEX );
rt_shared char eifthd_gen_typecode_with_dftype (EIF_TYPE_INDEX , uint32);
rt_public EIF_TYPE_INDEX eifthd_gen_param_id (EIF_TYPE_INDEX , uint32);
rt_public EIF_REFERENCE eifthd_gen_create (EIF_REFERENCE , uint32);
rt_shared EIF_TYPE_INDEX eifthd_register_bit_type (uint16);
rt_shared EIF_TYPE_INDEX eifthd_typeof_array_of (EIF_TYPE_INDEX);
rt_shared EIF_TYPE_INDEX eifthd_typeof_type_of (EIF_TYPE_INDEX);
rt_public char *eifthd_gen_typename (EIF_REFERENCE );
rt_shared EIF_TYPE_INDEX *eifthd_gen_cid (EIF_TYPE_INDEX);
rt_shared EIF_TYPE_INDEX eifthd_gen_id_from_cid (EIF_TYPE_INDEX *, EIF_TYPE_INDEX *);
rt_public int eifthd_gen_conf (EIF_TYPE_INDEX, EIF_TYPE_INDEX);

#define EIFMTX_LOCK \
	{\
		RT_GET_CONTEXT \
		EIF_ASYNC_SAFE_CS_LOCK(eif_gen_mutex)

#define EIFMTX_UNLOCK \
		EIF_ASYNC_SAFE_CS_UNLOCK(eif_gen_mutex); \
	}

#else
/* Noop for locks in non-multithreaded mode. */
#define EIFMTX_LOCK
#define EIFMTX_UNLOCK
#endif
/*------------------------------------------------------------------*/

rt_private size_t eif_typename_len (EIF_TYPE_INDEX dftype);
rt_private void eif_create_typename (EIF_TYPE_INDEX, char*);
rt_private EIF_GEN_DER *eif_new_gen_der(uint32, EIF_TYPE_INDEX*, EIF_TYPE_INDEX, char, char, EIF_TYPE_INDEX, EIF_TYPE_INDEX);
rt_private void eif_expand_tables(int);
rt_private EIF_TYPE_INDEX eif_id_of (EIF_TYPE_INDEX**, EIF_TYPE_INDEX**, EIF_TYPE_INDEX);
rt_private void eif_compute_ctab (EIF_TYPE_INDEX);
rt_private EIF_CONF_TAB *eif_new_conf_tab (EIF_TYPE_INDEX, EIF_TYPE_INDEX, EIF_TYPE_INDEX, EIF_TYPE_INDEX);
rt_private void eif_enlarge_conf_tab (EIF_CONF_TAB *, EIF_TYPE_INDEX);
rt_private uint16 eif_gen_seq_len (EIF_TYPE_INDEX);
rt_private void eif_put_gen_seq (EIF_TYPE_INDEX, EIF_TYPE_INDEX*, EIF_TYPE_INDEX*);

/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/

#ifdef EIF_THREADS

/*------------------------------------------------------------------*/
/* Public features protected with a MUTEX.                          */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_compound_id (EIF_TYPE_INDEX current_dftype, EIF_TYPE_INDEX base_id, EIF_TYPE_INDEX *types)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = eifthd_compound_id (current_dftype, base_id, types);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_final_id (EIF_TYPE_INDEX *ttable, EIF_TYPE_INDEX **gttable, EIF_TYPE_INDEX dftype, int offset)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = eifthd_final_id (ttable, gttable, dftype, offset);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared uint32 eif_gen_count_with_dftype (EIF_TYPE_INDEX dftype)
{
	uint32 result;

	EIFMTX_LOCK;
	result = eifthd_gen_count_with_dftype (dftype);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared char eif_gen_typecode_with_dftype (EIF_TYPE_INDEX dftype, uint32 pos)
{
	char    result;

	EIFMTX_LOCK;
	result = eifthd_gen_typecode_with_dftype (dftype, pos);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_gen_param_id (EIF_TYPE_INDEX dftype, uint32 pos)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = eifthd_gen_param_id (dftype, pos);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_register_bit_type (uint16 size)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = eifthd_register_bit_type (size);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_typeof_array_of (EIF_TYPE_INDEX dtype)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = eifthd_typeof_array_of (dtype);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_typeof_type_of (EIF_TYPE_INDEX dtype)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = eifthd_typeof_type_of (dtype);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX *eif_gen_cid (EIF_TYPE_INDEX dftype)
{
	EIF_TYPE_INDEX   *result;

	EIFMTX_LOCK;
	result = eifthd_gen_cid (dftype);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_gen_id_from_cid (EIF_TYPE_INDEX *a_cidarr, EIF_TYPE_INDEX *dtype_map)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = eifthd_gen_id_from_cid (a_cidarr, dtype_map);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_public int eif_gen_conf (EIF_TYPE_INDEX source_type, EIF_TYPE_INDEX target_type)
{
	int result;

	EIFMTX_LOCK;
	result = eifthd_gen_conf (source_type, target_type);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/
/* Rename public features if EIF_THREADS is on.                     */
/*------------------------------------------------------------------*/

#define eif_compound_id           eifthd_compound_id
#define eif_final_id              eifthd_final_id
#define eif_gen_count_with_dftype eifthd_gen_count_with_dftype
#define eif_gen_typecode_with_dftype  eifthd_gen_typecode_with_dftype
#define eif_gen_param_id          eifthd_gen_param_id
#define eif_register_bit_type     eifthd_register_bit_type
#define eif_typeof_array_of       eifthd_typeof_array_of
#define eif_typeof_type_of        eifthd_typeof_type_of
#define eif_gen_cid               eifthd_gen_cid
#define eif_gen_id_from_cid       eifthd_gen_id_from_cid
#define eif_gen_conf              eifthd_gen_conf

#endif

/*------------------------------------------------------------------*/
/* Initialize all structures                                        */
/* Called only once before root object is created.                  */
/*------------------------------------------------------------------*/

rt_shared void eif_gen_conf_init (EIF_TYPE_INDEX max_dtype)
{
	EIF_TYPE_INDEX dt;
	char   *cname;
	struct eif_par_types **pt;

#ifdef EIF_THREADS
		/* Since we want to avoid any locks to happen on the access on 
		 * eif_cid_map, we make sure that `eif_cid_map' can't be resized
		 * by giving the maximum size it can have, ie 0x0000FFFF */
	eif_cid_size = 65535;
#else
	eif_cid_size = max_dtype + 32;
#endif
	eif_first_gen_id = eif_next_gen_id = max_dtype + 1;

	/* Set `eif_par_table2' if it is null. */

	if (eif_par_table2 == NULL)
	{
		eif_par_table2 = eif_par_table;
		eif_par_table2_size = eif_par_table_size;
	}

	eif_cid_map = (EIF_TYPE_INDEX *) cmalloc (eif_cid_size * sizeof (EIF_TYPE_INDEX));

	if (eif_cid_map == NULL)
		enomem();

	eif_derivations = (EIF_GEN_DER **) cmalloc(eif_cid_size * sizeof (EIF_GEN_DER*));

	if (eif_derivations == NULL)
		enomem();

	eif_conf_tab = (EIF_CONF_TAB **) cmalloc(eif_cid_size * sizeof (EIF_CONF_TAB*));

	if (eif_conf_tab == NULL)
		enomem();

	/* Setup a 1-1 mapping and initialize the arrays */

	for (dt = 0; dt < eif_cid_size; ++dt)
	{
		eif_cid_map [dt]     = dt;
		eif_derivations [dt] = NULL;
		eif_conf_tab [dt]    = NULL;
	}

	/* Now initialize egc_xxx_dtypes */

	for (dt = 0, pt = eif_par_table2; dt <= eif_par_table2_size; ++dt, ++pt)
	{
		if ((*pt == (struct eif_par_types *)0) || EIF_IS_SEPARATE_TYPE (System((*pt)->dtype)))
			continue;

		cname = System((*pt)->dtype).cn_generator;

		if ((strcmp("ANY",cname)==0))
		{
			egc_any_dtype = dt;
		}

		if ((strcmp("TUPLE",cname)==0))
		{
			tuple_static_type = dt;
		}
	}

		/* Initialize `non_generic_type_names' for root thread now that `eif_first_gen_id' is
		 * properly computed. Indeed the first call to `eif_gen_conf_thread_init' is done
		 * before `eif_first_gen_id' is initialized and therefore does not allocate anything
		 * since `eif_first_gen_id' is zero. The second call will do things properly.
		 */
	eif_gen_conf_thread_init();
}

/*
doc:	<routine name="eif_gen_conf_thread_init" return_type="void" export="shared">
doc:		<summary>Initialize per thread data used for generic conformance. Root thread initialization is done in `eif_gen_conf_init' as when we are called by the `eif_thr_register' routine, the value `eif_first_gen_id' is still zero.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</routine>
*/
rt_shared void eif_gen_conf_thread_init (void) {
	RT_GET_CONTEXT

	/* Initialize `cid_array' */
	cid_array [0] = 1;  /* count */
	cid_array [1] = 0;  /* id */
	cid_array [2] = TERMINATOR; /* Terminator */

	if (eif_first_gen_id > 0) {
		non_generic_type_names = (char **) eif_rt_xcalloc (eif_first_gen_id, sizeof (char *));
	}
}

/*
doc:	<routine name="eif_gen_conf_thread_cleanup" return_type="void" export="shared">
doc:		<summary>Initialize per thread data used for generic conformance.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</routine>
*/
rt_shared void eif_gen_conf_thread_cleanup (void) {
	RT_GET_CONTEXT
	char *l_name;
	int i = 0;

	for (; i < eif_first_gen_id; i++) {
		l_name = non_generic_type_names [i];
		if (l_name) {
			eif_rt_xfree (l_name);
		}
	}
	eif_rt_xfree (non_generic_type_names);
	non_generic_type_names = NULL;
}

/*------------------------------------------------------------------*/
/* Clean up.                                                        */
/* Called from reclaim, and free all global variables allocated     */
/* for the Generic Conformance.                                     */
/*------------------------------------------------------------------*/
rt_shared void eif_gen_conf_cleanup (void) 
{
	/* Free in reverse order of allocation. */

	int i, j;

	REQUIRE ("eif_conf_tab not null", eif_conf_tab);
	REQUIRE ("eif_derivations not null", eif_derivations);
	REQUIRE ("eif_cid_map not null", eif_cid_map);

	/* Recursively free eif_conf_tab */
	for (i = 0; i < eif_cid_size; i++) {
		EIF_CONF_TAB *tmp = eif_conf_tab [i];	

		if (tmp == NULL)
			continue;

		if (tmp->low_tab) {
			eif_rt_xfree (tmp->low_tab);	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->low_tab)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->high_tab) {
			eif_rt_xfree (tmp->high_tab);	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->high_tab)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->low_comp) {
			eif_rt_xfree (tmp->low_comp);	 	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->low_comp)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->high_comp)	{
			eif_rt_xfree (tmp->high_comp);	 	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->high_comp)));
#endif	/* LMALLOC_CHECK */
		}
		eif_rt_xfree (tmp);
	}
	eif_rt_xfree (eif_conf_tab);	

	/* Recursively free eif_derivations. */
	for (i = 0; i < eif_cid_size; i++) {
		EIF_GEN_DER *tmp = eif_derivations [i];

		if (tmp == NULL)
			continue;
		if (tmp->typearr) {
			eif_rt_xfree (tmp->typearr);
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->typearr)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->gen_seq) {
			eif_rt_xfree (tmp->gen_seq);
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->gen_seq)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->ptypes) {
			eif_rt_xfree (tmp->ptypes);
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->ptypes)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->name) {
			eif_rt_xfree (tmp->name);			/* char * */
		}
		for (j = i + 1; j < eif_cid_size; j++) {
			if (eif_derivations [j] == tmp)	
				eif_derivations[j] = NULL;
				
		}
		eif_rt_xfree (tmp);
	}
	eif_rt_xfree (eif_derivations);	

	eif_rt_xfree (eif_cid_map);
#ifndef EIF_THREADS
	eif_gen_conf_thread_cleanup();
#endif
} /* eif_gen_conf_cleanup () */

/*------------------------------------------------------------------*/
/* Compute id for `types'. `cache' is used to cache the result in   */
/* the generated C code if possible.                                */
/*                                                                  */
/* cache   : To cache result;                                       */
/* base_id : Base id of type                                        */
/* types   : Id array                                               */
/* Result  : Resulting id;                                          */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_compound_id (EIF_TYPE_INDEX current_dftype, EIF_TYPE_INDEX base_id, EIF_TYPE_INDEX *types)
{
	if ((types != NULL) && (*(types+1) != TERMINATOR)) {
		EIF_TYPE_INDEX   outtab [256], *outtable, *intable;
		intable  = types+1;
		outtable = outtab;
		return eif_id_of (&intable, &outtable, current_dftype);
	} else {
		return base_id;
	}
}

/*------------------------------------------------------------------*/
/* Compute id for `gttable' (generic type list for feature in final */
/* mode).                                                           */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_final_id (EIF_TYPE_INDEX *ttable, EIF_TYPE_INDEX **gttable, EIF_TYPE_INDEX dftype, int offset)

{
	EIF_TYPE_INDEX   result, *gtp;
	EIF_TYPE_INDEX	dtype = To_dtype(dftype);
	int	table_index = dtype - offset;

	if (gttable != NULL) {
		gtp = gttable [table_index];

		if ((gtp != NULL) && (*(gtp+1) != TERMINATOR)) {
			*gtp = dtype;
			return eif_compound_id (dftype, ttable[table_index], gtp);
		}
	}

	result = ttable[table_index];

	return result;
}
/*------------------------------------------------------------------*/
/* Number of generic parameters of `obj's type. Can ONLY be used for*/
/* TUPLE and its descendants!                                       */
/*------------------------------------------------------------------*/

rt_public uint32 eif_gen_count_with_dftype (EIF_TYPE_INDEX dftype)
{
	EIF_GEN_DER *gdp;

	REQUIRE("Valid type", dftype < eif_next_gen_id);

	gdp = eif_derivations [dftype];

	return (gdp ? gdp->size : 0);
}

/*------------------------------------------------------------------*/
/* Number of generic parameters of `obj's type. Can ONLY be used for*/
/* TUPLE                                                            */
/*------------------------------------------------------------------*/

rt_public uint32 eif_tuple_count (EIF_REFERENCE obj)
{
	return (obj ? RT_SPECIAL_COUNT(obj) - 1 : 0);
}

/*------------------------------------------------------------------*/
/* Are all generic parameters of basic types? Can ONLY be used for  */
/* TUPLE                                                            */
/*------------------------------------------------------------------*/

rt_shared int eif_tuple_is_atomic (EIF_REFERENCE obj)
{
	EIF_VALUE *l_item = (EIF_VALUE *) obj;
	unsigned int count;
	
	if (obj == NULL) {
			/* This is atomic */
		return 1;
	}

	CHECK("Tuple object", HEADER(obj)->ov_flags & EO_TUPLE);
	count = RT_SPECIAL_COUNT(obj);

		/* Don't forget that first element of TUPLE is the BOOLEAN
		 * `object_comparison' attribute. */
	l_item++;
	for (; count > 0 ; count--) {
		if (eif_is_reference_tuple_item(l_item)) {
				/* It has a reference. This is therefore not atomic */
			return 0;
		}
	}
		/* No reference found. It is atomic */
	return 1;
}

/*------------------------------------------------------------------*/
/* Typecode of generic type at position `pos' in `obj'. ONLY for    */
/* TUPLE                                                            */
/*------------------------------------------------------------------*/

rt_public char eif_gen_typecode (EIF_REFERENCE obj, uint32 pos)
{
	if (obj == NULL) {
		return (char) 0;
	} else {
		char result;
			/* Critical section as we might compute a new `eif_anc_id_map' entry */
		EIFMTX_LOCK;
		result = eif_gen_typecode_with_dftype (Dftype(obj), pos);
		EIFMTX_UNLOCK;
		return result;
	}
}

rt_shared char eif_gen_typecode_with_dftype (EIF_TYPE_INDEX dftype, uint32 pos)
{
	EIF_TYPE_INDEX gtype;
	EIF_GEN_DER *gdp;

		/* Check type validity */
	REQUIRE ("dftype is less than maximum computed id", dftype < eif_next_gen_id);
	REQUIRE ("We have routines, so we must have tuples.", tuple_static_type < MAX_DTYPE);

	gdp = eif_derivations [dftype];

	CHECK ("gdp not null", gdp != (EIF_GEN_DER *)0);
	CHECK ("Not a bit type", !gdp->is_bit);

	CHECK ("Valid generic position min", pos > 0);
	CHECK ("Valid generic position max", pos <= gdp->size);

	gtype = gdp->typearr [pos-1];

	if (RT_IS_NONE_TYPE(gtype)) {
		return EIF_REFERENCE_CODE;
	}

	return EIF_TUPLE_CODE(System(eif_cid_map[gtype]));
}

/*------------------------------------------------------------------*/
/* Typecode string for target/argument types of a ROUTINE object.   */
/* ONLY for ROUTINE!                                                */
/*------------------------------------------------------------------*/

rt_public EIF_REFERENCE eif_gen_typecode_str (EIF_REFERENCE obj)
{
	EIF_GET_CONTEXT

	EIF_REFERENCE ret;	/* Return value. */
	EIF_TYPE_INDEX dftype, gtype;
	int len;
	uint32 pos;
	EIF_GEN_DER *gdp;
	char *strp;

	REQUIRE ("obj not null", obj != (EIF_REFERENCE )0);

	dftype = Dftype(obj);

	REQUIRE ("Valid dftype", dftype < eif_next_gen_id);

	gdp = eif_derivations [dftype];

	CHECK ("gdp not null", gdp != (EIF_GEN_DER *)0);
	CHECK ("Not a bit type", !gdp->is_bit);
	CHECK ("Not a routine object", gdp->size > 1);

		/* Type of call target */
	gtype = gdp->typearr [0];

		/* Now treat the arguments.  This is necessarily a TUPLE */
	dftype = gdp->typearr [1];

	CHECK ("Valid dftype", dftype < eif_next_gen_id);
	CHECK ("Routines implies we have tuples", tuple_static_type < MAX_DTYPE);

	gdp = eif_derivations [dftype];

	CHECK ("gdp not null", gdp != (EIF_GEN_DER *)0);
	CHECK ("Not a bit type", !gdp->is_bit);

		/* Create a string for gdp->size + 1 characters */
	len = gdp->size + 1;

	ret = emalloc(egc_str_dtype);
	RT_GC_PROTECT(ret);
		/* Protect address in case it moves */

	nstcall = 0;
	RT_STRING_MAKE(ret, (EIF_INTEGER) len);
	RT_STRING_SET_COUNT(ret, len);

	/* We know the `area' is the very first reference
	 * of the STRING object, hence the simple de-referencing.
	 */

	strp = *(EIF_REFERENCE*)ret;

	*strp = EIF_TUPLE_CODE(System(eif_cid_map[gtype]));
	strp++;

	for (pos = 0; pos < gdp->size; pos++, strp++) {
		gtype = gdp->typearr [pos];
		if (RT_IS_NONE_TYPE(gtype)) {
			*strp = EIF_REFERENCE_CODE;
		} else {
			*strp = EIF_TUPLE_CODE(System(eif_cid_map[gtype]));
		}
	}

	RT_GC_WEAN(ret);			/* Remove protection */

	return ret;	
}

/*------------------------------------------------------------------*/
/* Type of generic parameter in `obj' at position `pos'.            */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_gen_param_id (EIF_TYPE_INDEX dftype, uint32 pos)

{
	EIF_GEN_DER *gdp;

	REQUIRE("Valid type", dftype < eif_next_gen_id);
	REQUIRE("pos positive", pos > 0);

	gdp = eif_derivations [dftype];

	CHECK("A generic type", gdp && (!gdp->is_bit));
	CHECK("Valid generic parameter position", (pos <= gdp->size));

	return gdp->typearr [pos-1];
}
/*------------------------------------------------------------------*/
/* Register a bit type. Return its type id.                         */
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_register_bit_type (uint16 size)
{
	EIF_TYPE_INDEX dftype;
	EIF_GEN_DER *gdp, *prev;

	/* Search for BIT type of size *intab */

	dftype = egc_bit_dtype;
	gdp    = eif_derivations [dftype];
	prev   = NULL;

	while (gdp != NULL) {
		if (size == gdp->size) {
			break; /* Found */
		}
		prev = gdp;
		gdp  = gdp->next;
	}

	if (!gdp) {
			/* Not found: we need a new id */
		gdp = eif_new_gen_der(size, NULL, dftype, '1', (char) 0, 0, 0);

		if (prev == (EIF_GEN_DER *)0) {
			eif_derivations [dftype] = gdp;
		} else {
			prev->next = gdp;
		}
		eif_derivations[gdp->id] = gdp; /* Self-reference */
	}

	return gdp->id;
}
/*------------------------------------------------------------------*/
/* Type id for ARRAY [something], where 'something' is a reference  */
/* type.                                                            */
/* dftype : full type id;                                            */
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_typeof_array_of (EIF_TYPE_INDEX dftype)
{
	EIF_TYPE_INDEX   typearr [4], result;

	REQUIRE("Valid ARRAY generic type", dftype <= MAX_DTYPE);
	REQUIRE("Valid ARRAY reference type", egc_arr_dtype <= MAX_DTYPE);

	typearr [0] = INVALID_DTYPE;	/* No static call context */
	typearr [1] = egc_arr_dtype;	/* Base type of ARRAY     */
	typearr [2] = dftype;			/* Parameter type */
	typearr [3] = TERMINATOR;

	result = eif_compound_id (0, typearr[1], typearr);
	return result;
}

/*------------------------------------------------------------------*/
/* Type id for TYPE [dftype]                                        */
/* dftype : full type id;                                           */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_typeof_type_of (EIF_TYPE_INDEX dftype)
{
	EIF_TYPE_INDEX   typearr [4], result, l_type;
	struct cecil_info *cecil_type;
	uint32 sk_type;			/* Generic information for dftype */
	uint32 *t;				/* To walk through the patterns array */
	int matched = 0;
	size_t index = 0;

	REQUIRE("Valid actual generic type", (dftype <= MAX_DTYPE) || (RT_IS_NONE_TYPE(dftype)));

		/* Get the CECIL description for TYPE. */
	cecil_type = (struct cecil_info *) ct_value (&egc_ce_type, "TYPE");

		/* Get the SK_type for X in TYPE [X] were are trying to build. */
	if (RT_IS_NONE_TYPE(dftype)) {
			/* For NONE we use the reference generic derivation of TYPE. */
		sk_type = SK_REF;
	} else {
		l_type = To_dtype(dftype);
		if (l_type == egc_bit_dtype) {
			EIF_GEN_DER *der = eif_derivations[dftype];
			sk_type = SK_BIT | der->size;
		} else {
			sk_type = eif_dtype_to_sk_type (l_type);
		}
	}

		/* Now try to find the proper TYPE generic derivation. */
	t = cecil_type->patterns;
	while ((*t != SK_INVALID) && (!matched)) {
		matched = (*t++ == sk_type);
		index++;
	}

	if (matched == 1) {
		typearr [0] = INVALID_DTYPE;	/* No static call context */
		typearr [1] = cecil_type->dynamic_types[index - 1];		/* Base type of TYPE */
		typearr [2] = dftype;			/* Parameter type */
		typearr [3] = TERMINATOR;

		result = eif_compound_id (0, typearr[1], typearr);
	} else {
		result = INVALID_DTYPE;
	}

	return result;
}

/*------------------------------------------------------------------*/
/* Full type name of `obj' as STRING object.                        */
/*------------------------------------------------------------------*/

rt_public EIF_REFERENCE eif_gen_typename_of_type (EIF_TYPE_INDEX current_dftype)
{
	char    *name;
	EIF_REFERENCE ret;	/* Return value. */

	EIFMTX_LOCK;
	name = eif_typename (current_dftype);
	EIFMTX_UNLOCK;

	ret = makestr(name, strlen(name));
	return ret;
}
/*------------------------------------------------------------------*/
/* CID which generates `dftype'. First entry is the length of the   */
/* compound id.                                                     */
/*                                                                  */
/* dftype : full type id;                                           */
/* Result : base ids;                                               */
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX *eif_gen_cid (EIF_TYPE_INDEX dftype)
{
	RT_GET_CONTEXT
	uint16 len;
	EIF_GEN_DER *gdp;

	if ((RT_IS_NONE_TYPE(dftype)) || (dftype < eif_first_gen_id)) {
		CHECK("valid_cid_array", (cid_array[0] == 1) && (cid_array[2] == TERMINATOR));
		cid_array [1] = dftype;
		return cid_array;
	}

	/* It's a run-time generated id */

	gdp = eif_derivations [dftype];

	if (gdp->gen_seq) {
		return gdp->gen_seq;        /* already computed */
	}
	/* Compute size of array */

	len = eif_gen_seq_len (dftype);
	gdp->gen_seq = (EIF_TYPE_INDEX *) cmalloc ((len+2)*sizeof(EIF_TYPE_INDEX));
	if (!gdp->gen_seq) {
		enomem();
	}

	gdp->gen_seq [0] = len;
	gdp->gen_seq [len+1] = TERMINATOR;

	/* Fill array */

	len = 1;

	eif_put_gen_seq (dftype, gdp->gen_seq, &len);

	return gdp->gen_seq;
}
/*------------------------------------------------------------------*/
/* Create an id from a type array 'a_cidarr'. If 'dtype_map' is not   */
/* NULL, use it to map old to new dtypes ('retrieve')               */
/* Format:                                                          */
/* First entry: count                                               */
/* Then 'count' type ids, then TERMINATOR                           */
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_gen_id_from_cid (EIF_TYPE_INDEX *a_cidarr, EIF_TYPE_INDEX *dtype_map)
{
	EIF_TYPE_INDEX   dftype;
	EIF_TYPE_INDEX   count, i, dtype;

	REQUIRE ("Valid cid array", a_cidarr);

	count   = *a_cidarr;
	*a_cidarr = 0;	/* Not used anyway. */

	if (dtype_map) {
			/* We need to map old dtypes to new dtypes */
		for (i = 1; i <= count; i++) {
			dtype = a_cidarr [i];

				/* Read annotation if any. */
			while (RT_HAS_ANNOTATION_TYPE(dtype)) {
				i++;
				dtype = a_cidarr [i];
			}

			if (dtype <= MAX_DTYPE) {
				a_cidarr [i] = dtype_map[dtype];
			} else if (dtype == TUPLE_TYPE) {
					/* We simply skip number of generic
					 * parameters of the tuple as they are not really used
					 * and only update TUPLE dynamic type */
				i = i + TUPLE_OFFSET;
				a_cidarr [i]  = dtype_map [a_cidarr [i]];
			} else if (dtype == FORMAL_TYPE) {
					/* We skip formal position. */
				i++;
			}
		}
	}

	a_cidarr [count+1] = TERMINATOR;
	dftype  = eif_compound_id (0, *(a_cidarr+1), a_cidarr);
	*a_cidarr = count;

	return dftype;
}

/*------------------------------------------------------------------*/
/* Conformance test. Does `source_type' conform to `target_type'?   */
/* This only applies to instantiated type.                          */
/*                                                                  */
/* Source_type : full type id;                                      */
/* Target_type : full type id;                                      */
/*------------------------------------------------------------------*/

rt_public int eif_gen_conf (EIF_TYPE_INDEX stype, EIF_TYPE_INDEX ttype)
{
	EIF_CONF_TAB *stab;
	EIF_GEN_DER *sgdp, *tgdp;
	EIF_TYPE_INDEX *ptypes;
	uint32 i, idx;
	int result;
	unsigned char mask;

	if (ttype > MAX_DTYPE) {
			/* Target is NONE (attached/detachable). */
		CHECK("NONE type", RT_IS_NONE_TYPE(ttype));
		if (ttype == DETACHABLE_NONE_TYPE) {
				/* Target is detachable so we can accept any other NONE types. */
			return RT_IS_NONE_TYPE(stype);
		} else {
				/* Target is attached NONE, so we can only accept attached NONE. */
			return stype == ttype;
		}
	}

	if (stype == ttype) {
		return 1;
	}

	if (EIF_IS_EXPANDED_TYPE(System(eif_cid_map[ttype]))) {
		/* Expanded target no conformance because types are different */
		return 0;
	} else if (stype > MAX_DTYPE) {
			/* Target is not expanded and source is NONE, then there is conformance if compatible. */
		CHECK("NONE type", RT_IS_NONE_TYPE(stype));
		if (eif_is_attached_type(ttype)) {
				/* Target is attached, so we can only accept attached NONE. */
			return stype == ATTACHED_NONE_TYPE;
		} else {
				/* Target is detachable so we can accept anything. */
			return 1;
		}
	}

	stab = eif_conf_tab[stype];

	if (stab == NULL) {
		eif_compute_ctab (stype);
		stab = eif_conf_tab[stype];
	}

	if (ttype < eif_first_gen_id) {
		/* Lower id */

		if ((ttype >= stab->min_low_id) && (ttype <= stab->max_low_id)) {
			idx = ttype-stab->min_low_id;
			mask = (unsigned char) (1 << (idx % 8));

			return (mask == ((stab->low_tab)[idx/8] & mask)) ? 1 : 0;
		}
	} else {
		/* High id */

		if ((ttype < stab->min_high_id) || (ttype > stab->max_high_id)) {
				/* We need to enlarge the table */
			eif_enlarge_conf_tab (stab, ttype);
		}

		/* Now ttype is in the table range */

		idx  = (ttype - stab->min_high_id);
		mask = (unsigned char) (1 << (idx % 8));

		/* If we have computed it already, return result 
		 * We check first if the computed value is '1', if so, it means both that we already
		 * computed it and that is True.
		 * If the computed value is '0' we check if we compute a value, if so we return 0
		 * because we already know the computed value, otherwise we compute it
		 */

		if (mask == ((stab->high_tab)[idx/8] & mask))
			return 1;
		if (mask == ((stab->high_comp)[idx/8] & mask))
			return 0;

		/* We have to compute it now (once!) */

		sgdp = eif_derivations [stype];
		tgdp = eif_derivations [ttype];
		result = 0;

		if
			(RT_IS_ATTACHED_TYPE(tgdp->annotation) &&
			(!RT_IS_ATTACHED_TYPE(sgdp->annotation) && !(sgdp->is_expanded))
		) {
				/* If target is attached but source is not, we can clearly dismiss this. */
			goto done;
		}

		if (stype >= eif_first_gen_id) {
				/* Both ids generated here */
			if (sgdp->first_id == tgdp->first_id) {
				/* Both have the same base class */

				/* Check BIT types. BIT n conforms to BIT m
				   iff n <= m. 
				*/

				if (sgdp->is_bit) {
					result = ((sgdp->size <= tgdp->size) ? 1 : 0);
					goto done;
				}

				/* Same base class. If nr. of generics
				   differs, both are TUPLEs.
				*/

				if (tgdp->size > sgdp->size) {
					/* Source and target are TUPLES but
					   source has fewer parameters */
					goto done;
				}

				for (i = 0; i < tgdp->size; ++i) {
					stype = (sgdp->typearr) [i];
					ttype = (tgdp->typearr) [i];

					if (stype == ttype)
						continue; /* Same types - avoid recursion */

					if (!eif_gen_conf (stype, ttype)) {
						goto done;
					}
				}

				result = 1;
				goto done;
			}
		}

		/* Target is generic.
		   We need to check every parent of the source
		   against the target */

		ptypes = sgdp->ptypes;

		result = 0;

		while (!result && (*ptypes != TERMINATOR))
		{
			result = eif_gen_conf (*ptypes, ttype);
			++ptypes;
		}

done:
		/* Register that we have computed it */
		(stab->high_comp)[idx/8] |= mask;

		if (result)
			(stab->high_tab)[idx/8] |= mask;

		return result;
	}

	return 0;
}
/*------------------------------------------------------------------*/
/* Private routines.                                                */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* Computation of a new id.                                         */
/*                                                                  */
/* intab      : dtypes array                                        */
/* outtab     : List of computed ids for generics.                  */
/* obj_type   : Full type of object; Used to replace a              */
/*              formal generic by an actual generic of the object.  */
/* annotation : Annotation of type if any (see ANNOTATION_TYPE_MASK)*/
/*------------------------------------------------------------------*/

rt_private EIF_TYPE_INDEX eif_id_of (EIF_TYPE_INDEX **intab, EIF_TYPE_INDEX **outtab, EIF_TYPE_INDEX obj_type)
{
	EIF_TYPE_INDEX   dftype, gcount = 0, i, hcode;
	EIF_TYPE_INDEX   *save_otab, type_annotation;
	int     pos, mcmp, require_detachable = 0;
	char    is_expanded, is_tuple;
	EIF_GEN_DER *gdp, *prev;

		/* Get full type */
	dftype = **intab;

	CHECK("Not terminator", dftype != TERMINATOR);

		/* Read possible annotations. */
	if (RT_HAS_ANNOTATION_TYPE(dftype)) {
		if (RT_IS_DETACHABLE_TYPE(dftype)) {
				/* We compute the type normally, but then we make sure the type is detachable at the end. */
			require_detachable = 1;
			type_annotation = 0;
		} else {
			type_annotation = dftype;
		}
		while (RT_HAS_ANNOTATION_TYPE(dftype)) {
			(*intab)++;
			dftype = **intab;
		}
	} else {
		type_annotation = 0;
	}

	if (dftype <= MAX_DTYPE) {
		if (EIF_IS_EXPANDED_TYPE(System (eif_cid_map[dftype]))) {
			is_expanded = '1';
			type_annotation = 0;
		} else {
			is_expanded = (char) 0;
		}
	} else {
		is_expanded = (char) 0;
	}

		/* Check if type is NONE. */
	if (RT_IS_NONE_TYPE(dftype)) {
		(*intab)++;
		if (RT_IS_ATTACHED_TYPE(type_annotation)) {
				/* Requested attached, so we use our special value for attached NONE. */
			dftype = ATTACHED_NONE_TYPE;
		} else if (require_detachable) {
				/* An explicit request was made to be detachable. */
			dftype = DETACHABLE_NONE_TYPE;
		}
		**outtab = dftype;
		(*outtab)++;
		return dftype;
	}

		/* Check whether it's a TUPLE Type */
	if (dftype == TUPLE_TYPE) {
		(*intab)++;
		gcount = **intab;       /* Number of generic params */
		(*intab)++;
		dftype = **intab;       /* Base id for TUPLE */

			/* Can we really have a TUPLE that is expanded? I guess so for now. */
		if (EIF_IS_EXPANDED_TYPE(System (eif_cid_map[dftype]))) {
			is_expanded = '1';
			type_annotation = 0;
		} else {
			is_expanded = (char) 0;
		}
		is_tuple = '1';
	} else {
		is_tuple = (char) 0;
	}

	if (dftype == FORMAL_TYPE) {
			/* formal generic */
		(*intab)++;
		pos = **intab;	/* Position of formal generic */

		gdp = eif_derivations [obj_type];
		dftype = gdp->typearr [pos-1];

		(*intab)++;
		if (RT_IS_ATTACHED_TYPE(type_annotation)) {
			dftype = eif_attached_type (dftype);
		} else if (require_detachable) {
			dftype = eif_non_attached_type (dftype);
		}
		**outtab = dftype;
		(*outtab)++;

		return dftype;
	}

	if (dftype == egc_bit_dtype) {
		(*intab)++;
		dftype = eif_register_bit_type ((uint16) (**intab));
		**outtab = dftype;
		(*intab)++;
		(*outtab)++;
		return dftype;
	}

	if (dftype >= eif_first_gen_id) {
		/* It's an already created gen. type */
		(*intab)++;
		if (RT_IS_ATTACHED_TYPE(type_annotation)) {
			dftype = eif_attached_type (dftype);
		} else if (require_detachable) {
			dftype = eif_non_attached_type (dftype);
		}
		**outtab = dftype;
		(*outtab)++;
		return dftype;
	}

	/* It's an ordinary id generated by the compiler */

	if (!is_tuple) {
		gcount = par_info(dftype)->nb_generics;
	}

	if (!is_tuple && (gcount == 0) && !type_annotation) {
		/* Neither a generic type nor a TUPLE type without annotation.*/
		(*intab)++;

		dftype = (require_detachable ? eif_non_attached_type (dftype) : dftype);
		**outtab = dftype;
		(*outtab)++;

		return dftype;
	}

	save_otab = *outtab;
	(*intab)++;

	for (hcode = 0, i = gcount; i; --i) {
		hcode = hcode + eif_id_of (intab, outtab, obj_type);
	}

		/* Search all the generic derivations associated to the base class `dftype' */
	gdp  = eif_derivations [dftype];
	prev = NULL;

	while (gdp != NULL) {
		if (
			(hcode == gdp->hcode) && (is_expanded == gdp->is_expanded) &&
			(gcount == gdp->size) && (type_annotation == gdp->annotation))
		{
			mcmp = 0;
			if (gcount > 0) {
				mcmp = memcmp((char*)save_otab, (char*)(gdp->typearr),gcount*sizeof(EIF_TYPE_INDEX));
			}
			if (mcmp == 0) {
				break; /* Found */
			}
		}
		prev = gdp;
		gdp  = gdp->next;
	}

	if (gdp == (EIF_GEN_DER *)0) {
			/* Not found: we need a new id */
		if ((gcount == 0) && (!is_tuple)) {
				/* This is a non-generic type which is used with some annotation.
				   First we generate the ID without annotation and then compute the
				   type with the annotation. */
			CHECK ("has annotation", type_annotation);
			CHECK ("is not tuple", is_tuple == (char) 0);
			if (prev == NULL) {
				prev = eif_new_gen_der (0, NULL, dftype, is_expanded, (char) 0, 0, 0);
				eif_derivations[dftype] = prev;
				if (is_expanded) {
					gdp = prev;
				}
			}
			if (!gdp) {
				gdp = eif_new_gen_der(gcount, save_otab, dftype, is_expanded, (char) 0, hcode, type_annotation);
			}
		} else {
			gdp = eif_new_gen_der(gcount, save_otab, dftype, is_expanded, is_tuple, hcode, type_annotation);
			if (prev == NULL) {
				eif_derivations [dftype] = gdp;
			}
		}
		if (prev) {
			prev->next = gdp;
		}
		eif_derivations[gdp->id] = gdp; /* Self-reference */
	}

	/* Put full id */
	*outtab = save_otab;
	dftype = gdp->id;
	dftype = (require_detachable ? eif_non_attached_type (dftype) : dftype);
	**outtab = dftype;
	(*outtab)++;

	return dftype;
}
/*------------------------------------------------------------------*/
/* Create a new generic derivation. Actually we create one for every*/
/* type, generic or not.                                            */
/*                                                                  */
/* size     : Nr. of bits in a bit type; nr. of generics in a ge-   */
/*            neric type; 0 otherwise.                              */
/* typearr  : Ids of generic paramenters; null pointer if not a     */
/*            generic type                                          */
/* base_id  : Base id of type                                       */
/* is_exp   : Is it expanded?                                       */
/* is_tuple : Is it a tuple?                                        */
/* hcode    : Hash code for faster search                           */
/*------------------------------------------------------------------*/

rt_private EIF_GEN_DER *eif_new_gen_der(uint32 size, EIF_TYPE_INDEX *typearr, EIF_TYPE_INDEX base_id, char is_exp, char is_tuple, EIF_TYPE_INDEX hcode, EIF_TYPE_INDEX annotation)
{
	EIF_GEN_DER *result;
	EIF_TYPE_INDEX *tp, dt;
	char *cname;
	struct eif_par_types **pt;

	result = (EIF_GEN_DER *) cmalloc(sizeof (EIF_GEN_DER));

	if (result == NULL)
		enomem();

	if (typearr == NULL) {
		/* It's not a generic type. If size > 0 then it's a BIT type */

		result->size        = size;
		result->hcode       = hcode;
		result->typearr     = NULL;
		result->gen_seq     = NULL;      /* Generated on request only */
		result->ptypes      = NULL;      /* Generated on request only */
		result->id          = ((size > 0) ? eif_next_gen_id++ : base_id);
		result->base_id     = base_id;
		result->first_id    = INVALID_DTYPE;
		result->annotation  = annotation;
		result->is_expanded = is_exp;
		result->is_bit      = ((size > 0) ? '1' : (char) 0);
		result->is_tuple    = is_tuple;
		result->name        = NULL;       /* Generated on request only */
				/* `name' must be allocated dynamically. */
		result->next        = (EIF_GEN_DER *)0;

		if (size > 0)
			goto finish;

		/* Just a simple, compiler generated id */

		goto finish_simple;
	}

		/* Large array */
	tp = (EIF_TYPE_INDEX *) cmalloc((size + 1)*sizeof(EIF_TYPE_INDEX));
	if (tp == NULL)
		enomem();

	tp[size]=TERMINATOR;

	if (size > 0) {
		memcpy (tp, typearr, size*sizeof(EIF_TYPE_INDEX));
	}

	result->size        = size;
	result->hcode       = hcode;
	result->typearr     = tp;
	result->gen_seq     = NULL;      /* Generated on request only */
	result->ptypes      = NULL;      /* Generated on request only */
	result->id          = eif_next_gen_id++;
	result->base_id     = base_id;
	result->first_id    = INVALID_DTYPE;
	result->annotation  = annotation;
	result->is_expanded = is_exp;
	result->is_bit      = (char) 0;
	result->is_tuple    = is_tuple;
	result->name        = NULL;       /* Generated on request only */
				/* `name' must be allocated dynamically. */
	result->next        = (EIF_GEN_DER *)0;

finish:

	/* Expand tables if necessary */

	if (eif_next_gen_id >= eif_cid_size)
		eif_expand_tables (eif_next_gen_id + 32);

	eif_cid_map [result->id] = base_id;

finish_simple:

	/* Now find first entry in parent table
	   which has the same class name as `base_id'.
	*/

	cname = System ((par_info(base_id))->dtype).cn_generator;

	for (dt = 0, pt = eif_par_table2; dt <= eif_par_table2_size; ++dt, ++pt) {
		if (*pt == (struct eif_par_types *)0)
			continue;

		if (strcmp (cname,System((*pt)->dtype).cn_generator) == 0) {
			result->first_id = dt;
			break;
		}
	}

	return result;
}
/*------------------------------------------------------------------*/
/* Create new conformance table.                                    */
/*                                                                  */
/* All ids are full type ids                                        */
/*------------------------------------------------------------------*/

rt_private EIF_CONF_TAB *eif_new_conf_tab(EIF_TYPE_INDEX min_low, EIF_TYPE_INDEX max_low, EIF_TYPE_INDEX min_high, EIF_TYPE_INDEX max_high) {
	EIF_CONF_TAB *result;
	EIF_TYPE_INDEX size;
	unsigned char *tab;

	result = (EIF_CONF_TAB *) cmalloc(sizeof (EIF_CONF_TAB));

	if (result == NULL) {
		enomem();
	} else {
		result->min_low_id = min_low;
		result->max_low_id = max_low;
		result->min_high_id = min_high;
		result->max_high_id = max_high;

		if (min_low <= max_low) {
			size = (max_low - min_low + 8)/8;
			tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
			if (!tab) {
				enomem ();
			} else {
				result->low_tab = tab;

				tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
				if (!tab) {
					eif_rt_xfree (result->low_tab);
					enomem ();
				} else {
					result->low_comp = tab;
#ifdef EIF_ASSERTIONS
					result->low_tab_end = result->low_tab + size;
					result->low_comp_end = result->low_comp + size;
#endif
				}
			}
		} else {
			result->low_tab = NULL;
			result->low_comp = NULL;
#ifdef EIF_ASSERTIONS
			result->low_tab_end = NULL;
			result->low_comp_end = NULL;
#endif
		}


		if (min_high <= max_high) {
			size = (max_high - min_high + 8)/8;
			tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
			if (!tab) {
				enomem ();
			} else  {
				result->high_tab = tab;

				tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
				if (!tab) {
					eif_rt_xfree(result->high_tab);
					enomem ();
				} else {
					result->high_comp = tab;
#ifdef EIF_ASSERTIONS
					result->high_tab_end = result->high_tab + size;
					result->high_comp_end = result->high_comp + size;
#endif
				}
			}
		} else {
			result->high_tab = NULL;
			result->high_comp = NULL;
#ifdef EIF_ASSERTIONS
			result->high_tab_end = NULL;
			result->high_comp_end = NULL;
#endif
		}
	}

	ENSURE("result not null", result);
	return result;
}

/*------------------------------------------------------------------*/
/* Enlarge conformance table to include `new_id'                    */
/*                                                                  */
/* New_id: full type id                                             */
/*------------------------------------------------------------------*/

rt_private void eif_enlarge_conf_tab(EIF_CONF_TAB *table, EIF_TYPE_INDEX new_id)
{
	unsigned char *tab, *comp, *old_tab, *old_comp;
	int offset, is_low;
	EIF_TYPE_INDEX min_old, max_old, min_new, max_new, size, old_size;

	is_low = 0;

	if (new_id < eif_first_gen_id) {
		/* It's a lower id */

		is_low  = 1;
		min_old = min_new = table->min_low_id;
		max_old = max_new = table->max_low_id;

		if (new_id < min_new)
			min_new = new_id - (new_id % 8);    /* alignment */

		if (new_id > max_new)
			max_new = new_id;

		old_tab  = table->low_tab;
		old_comp = table->low_comp;
	} else {
		/* It's a high id */

		min_old = min_new = table->min_high_id;
		max_old = max_new = table->max_high_id;

		if (new_id < min_new)
			min_new = new_id - (new_id % 8);    /* alignment */

		if (new_id > max_new)
			max_new = new_id;

		old_tab  = table->high_tab;
		old_comp = table->high_comp;
	}

	if (min_old <= max_old) {
		old_size = (max_old - min_old + 8)/8;
	} else {
		old_size = 0;
	}

	size = (max_new - min_new + 8)/8;
	tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
	if (!tab)
		enomem ();
	comp = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
	if (!comp)
		enomem ();

		/* Initialize new tables from old tables */
	if (min_old <= max_old)
	{
		offset = (min_old - min_new) / 8;

		memcpy ((void *)(tab + offset), (void *)old_tab, old_size);
		memcpy ((void *)(comp + offset), (void *)old_comp, old_size);
	}

	/* Free old tables if they were not small (i.e. static) */

	if (old_tab) {
		eif_rt_xfree (old_tab);
	}
	if (old_comp) {
		eif_rt_xfree (old_comp);
	}

	/* Now update structure values */

	if (is_low) {
		table->min_low_id = min_new;
		table->max_low_id = max_new;
		table->low_tab = tab;
		table->low_comp = comp;
#ifdef EIF_ASSERTIONS
		table->low_tab_end = tab + size;
		table->low_comp_end = comp + size;
#endif
	} else {
		table->min_high_id = min_new;
		table->max_high_id = max_new;
		table->high_tab = tab;
		table->high_comp = comp;
#ifdef EIF_ASSERTIONS
		table->high_tab_end = tab + size;
		table->high_comp_end = comp + size;
#endif
	}
}

/*------------------------------------------------------------------*/
/* Expand `eif_cid_map', `eif_conf_tab' , `eif_derivations' and     */
/* to `new_size'                                                    */
/*------------------------------------------------------------------*/

rt_private void eif_expand_tables(int new_size)
{
#ifdef EIF_THREADS
	eif_panic ("Cannot resize Generic conformance tables in multithreaded mode.");
#else
	EIF_GEN_DER **new;
	EIF_CONF_TAB **tab;
	EIF_TYPE_INDEX *map;
	int         i;

	new = (EIF_GEN_DER **) crealloc((char*)eif_derivations, new_size*sizeof (EIF_GEN_DER*));

	if (new == NULL)
		enomem();

	eif_derivations = new;

	tab = (EIF_CONF_TAB **) crealloc((char*)eif_conf_tab, new_size*sizeof (EIF_CONF_TAB*));

	if (tab == NULL)
		enomem();

	eif_conf_tab = tab;

	map = (EIF_TYPE_INDEX *) crealloc((char*)eif_cid_map, new_size*sizeof (EIF_TYPE_INDEX));

	if (map == NULL)
		enomem();

	eif_cid_map = map;

	for (i = eif_cid_size; i < new_size; ++i)
	{
		eif_cid_map [i]     = 0;
		eif_derivations [i] = NULL;
		eif_conf_tab [i]    = NULL;
	}

	eif_cid_size = new_size;
#endif
}
/*------------------------------------------------------------------*/
/* Full type name for type `dftype' as C string.                    */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/
rt_private char *rt_attached_none_name_type = "!NONE";
rt_private char *rt_detachable_none_name_type = "NONE";

rt_public char *eif_typename (EIF_TYPE_INDEX dftype)
{
	EIF_GEN_DER *gdp;
	char    *result;
			
	REQUIRE("Valid type", (dftype < eif_next_gen_id) || (RT_IS_NONE_TYPE(dftype)));

	if (RT_IS_NONE_TYPE (dftype)) {
		if (dftype == DETACHABLE_NONE_TYPE) {
			result = rt_detachable_none_name_type;
		} else {
			CHECK("attached NONE", dftype == ATTACHED_NONE_TYPE);
			result = rt_attached_none_name_type;
		}
	} else if (dftype < eif_first_gen_id) {
		RT_GET_CONTEXT
		result = non_generic_type_names [dftype];
		if (result == NULL) {
			char *l_class_name = System(par_info(dftype)->dtype).cn_generator;

			if (EIF_NEEDS_EXPANDED_KEYWORD(System (dftype))) {
				result = cmalloc (10 + strlen (l_class_name));
				if (!result) {
					enomem();
				} else {
					result [0] = '\0';
					strcat (result, "expanded ");
				}
			} else if (EIF_NEEDS_REFERENCE_KEYWORD(System (dftype))) {
				result = cmalloc (11 + strlen (l_class_name));
				if (!result) {
					enomem();
				} else {
					result [0] = '\0';
					strcat (result, "reference ");
				}
			} else {
				result = cmalloc (strlen (l_class_name) + 1);
				if (!result) {
					enomem();
				} else {
					result [0] = '\0';
				}
			}
			strcat (result, l_class_name);
			non_generic_type_names[dftype] = result;
		}
	} else {
		gdp = eif_derivations [dftype];
		CHECK("gdp_computed", gdp);
		if (gdp->name != NULL) {    /* Already computed */
			result = gdp->name;	/* Allocated dynamically! */
		} else {
				/* Create dynamic buffer for string */
			result = cmalloc (eif_typename_len (dftype) + 1);
			if (result == NULL) {
				enomem(); 
			} else {
				*result = '\0';
				eif_create_typename (dftype, result);
				CHECK ("Not computed", !(gdp->name))
				gdp->name = result;
			}
		}
	}

	return result;
}
/*------------------------------------------------------------------*/
/* Produce full type name of `dftype' in `result'.                  */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private void eif_create_typename (EIF_TYPE_INDEX dftype, char *result)
{
	EIF_GEN_DER *gdp;
	EIF_TYPE_INDEX       *gp, dtype, i;
	size_t n;
	int         size;
	int	needs_expanded = 0, needs_reference = 0;

	if (dftype > MAX_DTYPE) {
		CHECK("NONE type", RT_IS_NONE_TYPE(dftype));
		if (dftype == DETACHABLE_NONE_TYPE) {
			strcat(result, rt_detachable_none_name_type);
		} else {
			CHECK("attached NONE", dftype == ATTACHED_NONE_TYPE);
			strcat(result, rt_attached_none_name_type);
		}
	} else {
		needs_expanded = EIF_NEEDS_EXPANDED_KEYWORD(System(eif_cid_map[dftype]));
		needs_reference = EIF_NEEDS_REFERENCE_KEYWORD(System(eif_cid_map[dftype]));

		if (dftype < eif_first_gen_id) {
			if (needs_expanded) {
				strcat (result, "expanded ");
			} else if (needs_reference) {
				strcat (result, "reference ");
			}
				/* Compiler generated id */
			strcat (result, System(par_info(dftype)->dtype).cn_generator);
		} else {
				/* We have created this id */
			gdp = eif_derivations [dftype];
			if (gdp->name != NULL) {    /* Already computed */
				strcat (result, gdp->name);
			} else {
/*				if (!RT_IS_FROZEN_TYPE(gdp->annotation)) {
					strcat (result, "variant ");
				} */
				if (RT_IS_ATTACHED_TYPE(gdp->annotation)) {
					strcat (result, "!");
				}
				if (gdp->is_bit) {
					size = gdp->size;
					strcat (result, "BIT ");
					n = strlen(result);
					while (size) {
						size /= 10;
						++n;
					}

					size = gdp->size;
					result [n] = '\0';
					for (--n; size; --n) {
						result [n] = (char) (size % 10) + '0';
						size /= 10;
					}
				} else {
						/* Generic case */
					i = (EIF_TYPE_INDEX) gdp->size;

					if (needs_expanded) {
						strcat (result, "expanded ");
					} else if (needs_reference) {
						strcat (result, "reference ");
					}

					strcat (result, System(par_info(gdp->base_id)->dtype).cn_generator);

					if (i > 0) {
						strcat (result, " [");
						gp = gdp->typearr;
						while (i--) {
							dtype = *gp;
							eif_create_typename (dtype, result);
							++gp;
							if (i) {
								strcat (result, ", ");
							}
						}
						strcat(result, "]");
					}
				}
			}
		}
	}
}
/*------------------------------------------------------------------*/
/* Compute length of string needed for full type name of `dftype'   */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private size_t eif_typename_len (EIF_TYPE_INDEX dftype)
{
	EIF_GEN_DER *gdp;
	EIF_TYPE_INDEX *gp, l_dftype;
	uint32 i;
	size_t len = 0;
	int	size, needs_expanded, needs_reference;

	if (dftype > MAX_DTYPE) {
		CHECK ("NONE type", RT_IS_NONE_TYPE(dftype));
		if (dftype == DETACHABLE_NONE_TYPE) {
			len += 4;
		} else {
			CHECK("attached NONE", dftype == ATTACHED_NONE_TYPE);
			len += 5;
		}
	} else {
		needs_expanded = EIF_NEEDS_EXPANDED_KEYWORD(System(eif_cid_map[dftype]));
		needs_reference = EIF_NEEDS_REFERENCE_KEYWORD(System(eif_cid_map[dftype]));

		if (dftype < eif_first_gen_id) {
			if (needs_expanded) {
				len += 9; /* for expanded followed by space */
			} else if (needs_reference) {
				len += 10; /* for reference followed by space */
			}
				/* Compiler generated id */
			len += strlen (System(par_info(dftype)->dtype).cn_generator);
		} else {
				/* We have created this id */
			gdp = eif_derivations [dftype];
			if (gdp->name != NULL) {    /* Already computed */
				len += strlen (gdp->name);
			} else {
/*				if (!RT_IS_FROZEN_TYPE(gdp->annotation)) {
					len += 7 + 1; *//* for variant followed by a space */
/*				} */
				if (RT_IS_ATTACHED_TYPE(gdp->annotation)) {
					len += 1;	/* for ! */
				}
				if (gdp->is_bit) {
					size = gdp->size;
					len += 4; /* for BIT followed by a space */
					while (size) {
						size /= 10;
						len++;
					}
				} else {
						/* Generic case */
					i = gdp->size;
					len += strlen (System(par_info(gdp->base_id)->dtype).cn_generator);

					if (needs_expanded) {
						len += 9; /* for expanded followed by space */
					} else if (needs_reference) {
						len += 10; /* for reference followed by space */
					}

					if (i > 0) {
							/* Numbers of `[', `]' and `, ' needed in the type specification. */
						len += 3 + (i-1)*2;
						gp = gdp->typearr;
						while (i--) {
							l_dftype = *gp;
							len += eif_typename_len (l_dftype);
							++gp;
						}
					}
				}
			}
		}
	}

	return len;
}
/*------------------------------------------------------------------*/
/* Compute length of generating id sequence for `dftype'            */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private uint16 eif_gen_seq_len (EIF_TYPE_INDEX dftype)
{
	EIF_GEN_DER *gdp;
	uint32 i;
	uint16 len;

	REQUIRE ("dftype is not an annotation", !RT_HAS_ANNOTATION_TYPE(dftype));
	REQUIRE ("dftype is not a formal generic parameter", dftype != FORMAL_TYPE);
	REQUIRE ("dftype is not a tuple", dftype != TUPLE_TYPE);
	REQUIRE ("dftype is not a terminator", dftype != TERMINATOR);

		/* Simple id */
	if ((RT_IS_NONE_TYPE(dftype)) || (dftype < eif_first_gen_id)) {
		len = 1;
	} else {
			/* It's a generic type or a BIT type */
		gdp = eif_derivations[dftype];

			/* If there is an annotation, then we should increase by one. */
		if (gdp->annotation) {
			len = 1;
		} else {
			len = 0;
		}

			/* Is it a BIT type? */
		if (gdp->is_bit) {
			len += 2;
		} else {
				/* Is it a TUPLE? */
			if (gdp->is_tuple) {
					/* Size is TUPLE_OFFSET because we need to take into account
					 * TUPLE_TYPE constant, number of generic parameters
					 * in seqence for tuple type */
				len += TUPLE_OFFSET;
			}

			i = gdp->size;
				/* Add 1 for the base ID. */
			len = len + 1; 
			while (i) {
				i--;
				len = len + eif_gen_seq_len (gdp->typearr [i]);
			}
		}
	}

	return len;
}
/*------------------------------------------------------------------*/
/* Produce generating id sequence for `dftype' in `typearr'.        */
/*                                                                  */
/* dftype    : full type id                                         */
/* typearr   : Base type ids                                        */
/* idx       : index where to put id                                */
/*------------------------------------------------------------------*/

rt_private void eif_put_gen_seq (EIF_TYPE_INDEX dftype, EIF_TYPE_INDEX *typearr, EIF_TYPE_INDEX *idx)
{
	EIF_GEN_DER *gdp;
	uint32 i, len;

	REQUIRE ("dftype is not an annotation", !RT_HAS_ANNOTATION_TYPE(dftype));
	REQUIRE ("dftype is not a formal generic parameter", dftype != FORMAL_TYPE);
	REQUIRE ("dftype is not a tuple", dftype != TUPLE_TYPE);
	REQUIRE ("dftype is not a terminator", dftype != TERMINATOR);

	/* Simple id */

	if ((RT_IS_NONE_TYPE(dftype)) || (dftype < eif_first_gen_id)) {
		typearr [*idx] = dftype;
		(*idx)++;
	} else {

			/* It's a generic type or a BIT type */
		gdp = eif_derivations[dftype];

		if (gdp->annotation) {
			typearr [*idx] = gdp->annotation;
			(*idx)++;
		}

			/* Is it a BIT type? */
		if (gdp->is_bit) {
			typearr [*idx] = egc_bit_dtype;    /* Bit type */
			(*idx)++;
			CHECK("Valid number of bits", rt_valid_type_index(gdp->size));
			typearr [*idx] = (EIF_TYPE_INDEX) (gdp->size); /* Nr of bits */
			(*idx)++;
		} else {
				/* Is it a TUPLE type? */
			if (gdp->is_tuple) {
				typearr [*idx] = TUPLE_TYPE;                   /* TUPLE type */
				(*idx)++;
				CHECK("Valid number of generics", rt_valid_type_index(gdp->size));
				typearr [*idx] = (EIF_TYPE_INDEX) (gdp->size);   /* Nr of generics */
				(*idx)++;
			}

			typearr [*idx] = gdp->base_id;
			(*idx)++;

			len = gdp->size;

			for (i = 0; i < len; ++i) {
				eif_put_gen_seq (gdp->typearr [i], typearr, idx);
			}
		}
	}
}

/*------------------------------------------------------------------*/
/* Compute if `dftype' is attached or not.                          */
/*------------------------------------------------------------------*/

rt_public EIF_BOOLEAN eif_is_attached_type (EIF_TYPE_INDEX dftype)
{
	if (RT_IS_NONE_TYPE(dftype)) {
		return EIF_TEST(dftype == ATTACHED_NONE_TYPE);
	} else {
		EIF_GEN_DER *gdp = eif_derivations [dftype];

		return EIF_TEST(gdp && RT_IS_ATTACHED_TYPE(gdp->annotation)); 
	}
}

/*------------------------------------------------------------------*/
/* Compute if `dftype' has a default value, i.e. detachable         */
/* reference type or expanded type.                                 */
/*------------------------------------------------------------------*/

rt_public EIF_BOOLEAN eif_gen_has_default (EIF_TYPE_INDEX dftype)
{
	if (RT_IS_NONE_TYPE(dftype)) {
		return EIF_TEST(dftype == DETACHABLE_NONE_TYPE);
	} else {
		EIF_GEN_DER *gdp = eif_derivations [dftype];

		return EIF_TEST(!gdp || !RT_IS_ATTACHED_TYPE(gdp->annotation) || gdp->is_expanded); 
	}
}

/*------------------------------------------------------------------*/
/* Compute if `dftype' has a default value, i.e. detachable         */
/* reference type or expanded type.                                 */
/*------------------------------------------------------------------*/

rt_public EIF_BOOLEAN eif_gen_is_expanded (EIF_TYPE_INDEX dftype)
{
	if (RT_IS_NONE_TYPE(dftype)) {
		return EIF_FALSE;
	} else {
		EIF_GEN_DER *gdp = eif_derivations [dftype];
		if (gdp) {
			return EIF_TEST(gdp->is_expanded);
		} else {
				/* It is not a generic derivation, we can avoid the conversion dftype -> dtype. */
			CHECK("Same as dtype", eif_cid_map[dftype] == dftype);
			return EIF_TEST(EIF_IS_EXPANDED_TYPE(System(dftype)));
		}

		return EIF_TEST(gdp && gdp->is_expanded); 
	}
}


/*------------------------------------------------------------------*/
/* Compute the associated detachable type of `dftype' if any,       */
/* otherwise `dftype'.                                              */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_non_attached_type (EIF_TYPE_INDEX dftype)
{
	EIF_GEN_DER *gdp;
	EIF_TYPE_INDEX l_result;
	EIF_TYPE_INDEX *saved_out, *outtable, *saved_in, *intable;
	uint32 nb, tuple_added_size;

	if (RT_IS_NONE_TYPE(dftype)) {
		l_result = DETACHABLE_NONE_TYPE;
	} else {
		gdp = eif_derivations [dftype];

		if (gdp && (RT_IS_ATTACHED_TYPE(gdp->annotation))) {
			nb = gdp->size;
			if (nb || (gdp->is_tuple)) {
				if (gdp->is_tuple) {
					tuple_added_size = 2;
				} else {
					tuple_added_size = 0;
				} 
					/* Case of a generic class. */
				outtable = (EIF_TYPE_INDEX *) cmalloc (sizeof(EIF_TYPE_INDEX) * (nb + 2 + tuple_added_size));
				if (!outtable) {
					enomem();
				}
				intable = (EIF_TYPE_INDEX *) cmalloc (sizeof(EIF_TYPE_INDEX) * (nb + 2 + tuple_added_size));
				if (!intable) {
					eif_rt_xfree(outtable);
					enomem();
				}
				saved_out = outtable;
				saved_in = intable;
				if (tuple_added_size) {
					intable[0] = TUPLE_TYPE;
					CHECK("valid cound", nb < 0xFFFF);
					intable[1] = (EIF_TYPE_INDEX) nb;
				}
				intable[tuple_added_size] = To_dtype(dftype);
				memcpy (intable + (tuple_added_size + 1), gdp->typearr, sizeof(EIF_TYPE_INDEX) * nb);
				intable[nb + tuple_added_size + 1] = TERMINATOR;
				l_result = eif_id_of (&intable, &outtable, dftype);

				eif_rt_xfree(saved_out);
				eif_rt_xfree(saved_in);
			} else {
					/* Case of a non-generic class, simply takes the Dtype. */
				l_result = To_dtype(dftype);
			}
		} else {
			l_result = dftype;
		}
	}
	return l_result;
}

/*------------------------------------------------------------------*/
/* Compute the associated attached type of `dftype' if any,         */
/* otherwise `dftype'.                                              */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE_INDEX eif_attached_type (EIF_TYPE_INDEX dftype)
{
	EIF_GEN_DER *gdp;
	EIF_TYPE_INDEX l_result;
	EIF_TYPE_INDEX *saved_out, *outtable, *saved_in, *intable;
	uint32 nb, tuple_added_size;

	if (RT_IS_NONE_TYPE(dftype)) {
		l_result = ATTACHED_NONE_TYPE;
	} else {
		gdp = eif_derivations [dftype];

		if (!gdp || (!RT_IS_ATTACHED_TYPE(gdp->annotation))) {
			tuple_added_size = 0;
			if (gdp) {
				if (gdp->is_bit) {
					nb = 1;
				} else {
					nb = gdp->size;
					if (gdp->is_tuple) {
							/* + 2 because we need to store TUPLE_TYPE followed by
							* the actual generic parameter count. */
						tuple_added_size = 2;
					}
				}
			} else {
				nb = 0;
			}
				/* + 3 because we need additional space for the attached mark, the dtype and the terminator. */
			outtable = (EIF_TYPE_INDEX *) cmalloc (sizeof(EIF_TYPE_INDEX) * (nb + tuple_added_size + 3));
			if (!outtable) {
				enomem();
			}
				/* + 3 because we need additional space for the attached mark, the dtype and the terminator. */
			intable = (EIF_TYPE_INDEX *) cmalloc (sizeof(EIF_TYPE_INDEX) * (nb + tuple_added_size + 3));
			if (!intable) {
				eif_rt_xfree(outtable);
				enomem();
			}
			saved_out = outtable;
			saved_in = intable;
			intable[0] = ATTACHED_TYPE;
			intable[tuple_added_size + 1] = To_dtype(dftype);
			if (gdp) {
				if (gdp->is_bit) {
					CHECK("not tuple", tuple_added_size == 0);
					intable [2] = (EIF_TYPE_INDEX) gdp->size;
				} else {
					if (tuple_added_size) {
						intable[1] = TUPLE_TYPE;
						CHECK("valid cound", nb < 0xFFFF);
						intable[2] = (EIF_TYPE_INDEX) nb;
					}
					memcpy (intable + (tuple_added_size + 2), gdp->typearr, sizeof(EIF_TYPE_INDEX) * nb);
				}
			}
			intable[nb + tuple_added_size + 2] = TERMINATOR;
			l_result = eif_id_of (&intable, &outtable, dftype);

			eif_rt_xfree(saved_out);
			eif_rt_xfree(saved_in);
		} else {
			l_result = dftype;
		}
	}
	return l_result;
}

/*------------------------------------------------------------------*/
/* Compute conformance table for `dftype'                           */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private void eif_compute_ctab (EIF_TYPE_INDEX dftype)

{
	EIF_TYPE_INDEX outtab [256], *outtable, *intable;
	EIF_TYPE_INDEX min_low, max_low, min_high, max_high, pftype, dtype, *ptypes, type_annotation;
	int i, count, offset, pcount;
	int repeat_parent_iteration, add_non_attached_parent;
	unsigned char *src, *dest, *src_comp, *dest_comp, mask;
	char is_expanded;
	struct eif_par_types *pt;
	EIF_CONF_TAB *ctab, *pctab;
	EIF_GEN_DER *gdp;

		/* Get parent table */
	dtype = To_dtype(dftype);
	gdp = eif_derivations [dftype];
		/* `gdp' might be NULL in the case of non-generic classes, thus we build a very simple entry. */
	if (gdp == NULL) {
		CHECK("same type", dftype == dtype);
		if (EIF_IS_EXPANDED_TYPE(System(dtype))) {
			gdp = eif_new_gen_der (0, NULL, dtype, '1', (char) 0, 0, 0);
		} else {
			gdp = eif_new_gen_der (0, NULL, dtype, (char) 0, (char) 0, 0, 0);
		}
		eif_derivations [dftype] = gdp;
	}

	CHECK ("gdp not null", gdp);

		/* Compiler generated id */
	pt = par_info (dtype);
	CHECK ("Parents never NULL", pt->parents);
	is_expanded = pt->is_expanded;

		/* Let's compute the number of parent types. */
	pcount = 0;
	intable = pt->parents;
		/* Skip annotation entry. */
	intable++;
	pftype = *intable++;
	while (pftype != TERMINATOR) {
		while ((pftype != PARENT_TYPE_SEPARATOR) && (pftype != TERMINATOR)) {
			pftype = *intable++;
		}
		if (pftype == PARENT_TYPE_SEPARATOR) {
			pftype = *intable++;
		}
		pcount++;
	}

	if (gdp->annotation) {
			/* We propagate the annotation to the parent, that is to say if class A inherit B, when
			 * building the parents of !A, we actually want !B. */
		type_annotation = gdp->annotation;
		pcount++;
		repeat_parent_iteration = 0;
		add_non_attached_parent = 1;
	} else if (gdp->is_expanded) {
			/* An expanded type conforms to all the attached version of its parents
			 * and the non-attached ones. */
		type_annotation = ATTACHED_TYPE;
		pcount *= 2; /* We double the number of parents for the non-attached ones. */
		repeat_parent_iteration = 1;
		add_non_attached_parent = 0;
	} else {
		type_annotation = 0;
		repeat_parent_iteration = 0;
		add_non_attached_parent = 0;
	}

				/* Compute the ranges of the bit tables */
	min_low = eif_next_gen_id;
	max_low = 0;
	min_high = eif_next_gen_id;
	max_high = 0;

		/* Type conforms to itself */
	if (dftype < eif_first_gen_id) {
		min_low = max_low = dftype;
	} else {
		min_high = max_high = dftype;
	}

		/* Create table of parent types (+1 for TERMINATOR) */
	ptypes = (EIF_TYPE_INDEX *) cmalloc (sizeof (EIF_TYPE_INDEX) * (pcount + 1));
	if (ptypes == NULL)
		enomem ();

	gdp->ptypes = ptypes;
	if (pcount) {
		if (add_non_attached_parent) {
				/* We are processing an attached type, which naturally conforms to its
				 * detachable version. */
			pftype = eif_non_attached_type (dftype);
			CHECK ("Cannot be the same type", dftype != pftype);
			*(ptypes++) = pftype;
		}

non_attached_parents:
		intable = pt->parents;
			/* If there is an annotation and if it is not ANY (ANY being
			 * the only class for which the parent description is made of
			 * {0, 0xFFFF}), we put the annotation for all parents. */
		if (type_annotation && (intable[1] != TERMINATOR)) {
			intable[0] = type_annotation;
		} else {
				/* annotation slot is useless, so skip it. */
			intable++;
		}

		outtable = outtab;
		while (*intable != TERMINATOR) {
			pftype = eif_id_of (&intable, &outtable, dftype);
			if (*intable == PARENT_TYPE_SEPARATOR) {
				intable++;
				if (type_annotation) {
					*intable = type_annotation;
				} else {
						/* annotation slot is useless, so skip it. */
					intable++;
				}
			}

				/* Register parent type */
			*(ptypes++) = pftype;
		}
		if (repeat_parent_iteration) {
				/* We are processing an expanded type, and we need to conform to all the detachable
				 * parents. */
			repeat_parent_iteration = 0;
			type_annotation = 0;
			goto non_attached_parents;
		}
		*ptypes = TERMINATOR;

		ptypes = gdp->ptypes;
		while (*ptypes != TERMINATOR) {
			pftype = *ptypes++;
			ctab = eif_conf_tab [pftype];

			if (ctab == NULL) {
				eif_compute_ctab (pftype);
				ctab = eif_conf_tab [pftype];
			}

			if (ctab->min_low_id < min_low)
				min_low = ctab->min_low_id;
			if (ctab->max_low_id > max_low)
				max_low = ctab->max_low_id;
			if (ctab->min_high_id < min_high)
				min_high = ctab->min_high_id;
			if (ctab->max_high_id > max_high)
				max_high = ctab->max_high_id;
		}
	} else {
		*ptypes = TERMINATOR;
	}

		/* Create a new table */
		/* Make sure that the min values are == 0 mod 8 */
	min_low  -= (min_low % 8);
	min_high -= (min_high % 8);

	ctab = eif_new_conf_tab (min_low, max_low, min_high, max_high);

	eif_conf_tab [dftype] = ctab;

		/* Fill bit tables */
	if (pcount) {
		ptypes = gdp->ptypes;

		while (*ptypes != TERMINATOR) {
			pftype = *ptypes++;
			pctab = eif_conf_tab [pftype];

			if ((min_low <= max_low) && (pctab->min_low_id <= pctab->max_low_id)) {
				count  = (pctab->max_low_id-pctab->min_low_id+8)/8;
				offset = (pctab->min_low_id - min_low)/8;
				src  = pctab->low_tab;
				dest = ctab->low_tab + offset;
				src_comp = pctab->low_comp;
				dest_comp = ctab->low_comp + offset;

				for (i = count; i; --i) {
					/* We conform to everything our parent
					   conforms to */

					*dest |= *src;

					/* Consider only those bits as already
					   computed for which conformance holds
					   because we may conform to something
					   to which the parent does not! */
					   
					*(dest_comp) |= ((*src) & (*src_comp));
					++dest;
					++src;
					++src_comp;
					++dest_comp;
				}
				CHECK ("valid_src", src <= pctab->low_tab_end);
				CHECK ("valid_src_comp", src_comp <= pctab->low_comp_end);
				CHECK ("valid_dest", dest <= ctab->low_tab_end);
				CHECK ("valid_dest_comp", dest_comp <= ctab->low_comp_end);
			}

			if ((min_high <= max_high) && (pctab->min_high_id <= pctab->max_high_id)) {
				count  = (pctab->max_high_id-pctab->min_high_id+8)/8;
				offset = (pctab->min_high_id - min_high)/8;
				src  = pctab->high_tab;
				dest = ctab->high_tab + offset;
				src_comp = pctab->high_comp;
				dest_comp = ctab->high_comp + offset;

				for (i = count; i; --i) {
					/* We conform to everything our parent
					   conforms to */
					*dest |= *src;

					/* Consider only those bits as already
					   computed for which conformance holds
					   because we may conform to something
					   to which the parent does not! */
					   
					*(dest_comp) |= ((*src) & (*src_comp));
					++dest;
					++src;
					++src_comp;
					++dest_comp;
				}
				CHECK ("valid_src", src <= pctab->high_tab_end);
				CHECK ("valid_src_comp", src_comp <= pctab->high_comp_end);
				CHECK ("valid_dest", dest <= ctab->high_tab_end);
				CHECK ("valid_dest_comp", dest_comp <= ctab->high_comp_end);
			}
		}
	}

	/* Put own type in table if it's not expanded */

	if (is_expanded)
		return;

	if (dftype < eif_first_gen_id) {
		offset = (dftype - min_low);
		mask   = (char) (1 << (offset % 8));
		CHECK("valid low_tab index", (ctab->low_tab + (offset / 8)) < ctab->low_tab_end);
		(ctab->low_tab)[offset/8] |= mask;
		CHECK("valid low_comp index", (ctab->low_comp + (offset / 8)) < ctab->low_comp_end);
		(ctab->low_comp)[offset/8] |= mask;
	} else {
		offset = (dftype - min_high);
		mask   = (char) (1 << (offset % 8));
		CHECK("valid high_tab index", (ctab->high_tab + (offset / 8)) < ctab->high_tab_end);
		(ctab->high_tab)[offset/8] |= mask;
		CHECK("valid high_comp index", (ctab->high_comp + (offset / 8)) < ctab->high_comp_end);
		(ctab->high_comp)[offset/8] |= mask;
	}
}

/*
doc:</file>
*/
