/*
	description: "Generic conformance routines."
	date:		"$Date: 2022-11-23 17:02:25 +0000 (Wed, 23 Nov 2022) $"
	revision:	"$Revision: 106407 $"
	copyright:	"Copyright (c) 1985-2019, Eiffel Software."
	license:	"GPL version 2 see http://www.eiffel.com/licensing/gpl.txt)"
	licensing_options:	"Commercial license is available at http://www.eiffel.com/licensing"
	copying: "[
			This file is part of Eiffel Software's Runtime.

			Eiffel Software's Runtime is free software; you can
			redistribute it and/or modify it under the terms of the
			GNU General Public License as published by the Free
			Software Foundation, version 2 of the License
			(available at the URL listed under "license" above).

			Eiffel Software's Runtime is distributed in the hope
			that it will be useful,	but WITHOUT ANY WARRANTY;
			without even the implied warranty of MERCHANTABILITY
			or FITNESS FOR A PARTICULAR PURPOSE.
			See the	GNU General Public License for more details.

			You should have received a copy of the GNU General Public
			License along with Eiffel Software's Runtime; if not,
			write to the Free Software Foundation, Inc.,
			51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
		]"
	source: "[
			 Eiffel Software
			 356 Storke Road, Goleta, CA 93117 USA
			 Telephone 805-685-1006, Fax 805-685-6869
			 Website http://www.eiffel.com
			 Customer support http://support.eiffel.com
		]"
*/

/*
doc:<file name="gen_conf.c" header="eif_gen_conf.h" version="$Id: gen_conf.c 106407 2022-11-23 17:02:25Z jfiat $" summary="Generic conformance">
*/

#include "eif_portable.h"
#include "rt_macros.h"
#include "eif_globals.h"
#include "rt_struct.h"
#include "rt_gen_conf.h"
#include "rt_gen_types.h"
#include "rt_malloc.h"
#include "rt_threads.h"
#include "rt_garcol.h"
#include "rt_cecil.h"
#include "rt_assert.h"
#include "rt_hashin.h"
#include <ctype.h>
#include <string.h>
#ifdef WORKBENCH
#include "rt_interp.h"
#endif
#include "rt_globals_access.h"
#include "rt_string.h"

/*------------------------------------------------------------------*/
/* Debugging flag. If set, the names of the generated types will be */
/* output to the file 'logfile'. Simple facility.                   */
/*------------------------------------------------------------------*/

/*
doc:	<attribute name="eif_par_table" return_type="struct eif_par_types **" export="shared">
doc:		<summary>Parent tables. Defined by compiler C generated code in `eparents.c'. Changes to this table after melt are stored in melted file and processed by `updated.c' and stored in `eif_par_table2'.</summary>
doc:		<access>Read only through macro `par_info'.</access>
doc:		<indexing>base id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_par_table_size" return_type="EIF_TYPE_INDEX" export="shared">
doc:		<summary>Size of `eif_par_table' table.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_par_table2" return_type="struct eif_par_types **" export="shared">
doc:		<summary>Same as `eif_par_table' except that this one contains both the static definition generated by compiler in `eparents.c' and the melted definition contained in melted file.</summary>
doc:		<access>Read only through macro `par_info'.</access>
doc:		<indexing>base id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_par_table2_size" return_type="EIF_TYPE_INDEX" export="shared">
doc:		<summary>Size of `eif_par_table2' table.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization.</synchronization>
doc:	</attribute>
*/
rt_shared struct eif_par_types **eif_par_table = NULL;
rt_shared EIF_TYPE_INDEX    eif_par_table_size = 0;
rt_shared struct eif_par_types **eif_par_table2 = NULL;
rt_shared EIF_TYPE_INDEX    eif_par_table2_size = 0;

/*
doc:	<attribute name="eif_cid_map" return_type="EIF_TYPE_INDEX *" export="public">
doc:		<summary>Compound id map. Maps compiler generated IDs to themselves and run-time generated IDs to their base IDs. In other word, map full dynamic type to dynamic type. Table is dynamically reallocated except in multithreaded mode where it is by default initialized to a count which represent the greatest type id (MAX_DTYPE).</summary>
doc:		<access>Read/Write (read with macros To_dtype)</access>
doc:		<indexing>full type id</indexing>
doc:		<result>base id</result>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_cid_size" return_type="int" export="private">
doc:		<summary>Number of elements in following structures `eif_cid_map', `eif_derivations', `eif_con_tab', `eif_conf_tab'.</summary>
doc:		<access>Read/Write</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
*/
rt_public EIF_TYPE_INDEX  *eif_cid_map = NULL;
rt_private int  eif_cid_size = 0;

/*
doc:	<attribute name="egc_any_dtype" return_type="EIF_TYPE_INDEX" export="public">
doc:		<summary>Type of ANY. Used to create ARRAY [ANY] from the runtime (e.g. used for `strip'). Value computed in `eif_gen_conf_init'.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization in `eif_gen_conf_init'.</synchronization>
doc:	</attribute>
doc:	<attribute name="tuple_static_type" return_type="EIF_TYPE_INDEX" export="private">
doc:		<summary>Base id of TUPLE.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization in `eif_gen_conf_init'.</synchronization>
doc:		<fixme>Shouldn't we use `egc_tup_dtype' instead?</fixme>
doc:	</attribute>
*/
rt_public EIF_TYPE_INDEX egc_any_dtype = INVALID_DTYPE; /* Precise value determined in init */
rt_private EIF_TYPE_INDEX tuple_static_type = INVALID_DTYPE;

/*----------------------------------------------*/
/* Structure representing a generic derivation. */
/*----------------------------------------------*/

typedef struct eif_gen_der {
	struct eif_gen_der  *next;      /* Next derivation */
	EIF_TYPE			*types;     /* Array of types (cid) */
	EIF_TYPE_INDEX      *gen_seq;   /* Id sequence which generates this type */
	EIF_TYPE_INDEX      *ptypes;    /* Parent types */
	uint32              size;       /* Size of type array. */
	uint32			    hcode;      /* Hash code to speedup search */
	EIF_TYPE_INDEX      dftype;     /* Run-time generated id */
	EIF_TYPE_INDEX      dtype;      /* Compiler generated (base) id */
	EIF_TYPE_INDEX      base_dtype; /* Base compiler generated (base) id of current type. */
	char                is_expanded;/* Is it an expanded type? */
	char                is_tuple;   /* Is it a TUPLE type? */
} EIF_GEN_DER;

/*------------------------------------------------------------------*/
/* Structure for conformance information. The `lower' ids are the   */
/* usual compiler generated ids. The others are generated here.     */
/*                                                                  */
/* All ids are full type ids                                        */
/* Indexing: full type ids                                          */
/*------------------------------------------------------------------*/

typedef struct {
	EIF_TYPE_INDEX  min_low_id;     /* Minimal lower conforming id */
	EIF_TYPE_INDEX  max_low_id;     /* Maximal lower conforming id */
	EIF_TYPE_INDEX  min_high_id;    /* Minimal high conforming id */
	EIF_TYPE_INDEX  max_high_id;    /* Maximal high conforming id */
	unsigned char   *low_tab;       /* Bit table for lower ids */
	unsigned char   *high_tab;      /* Bit table for high ids */
	unsigned char   *low_comp;      /* Bit table for computed lower conf. */
	unsigned char   *high_comp;     /* Bit table for computed high conf. */
#ifdef EIF_ASSERTIONS
		/* End of all the above arrays for efficient bound checks. */
	unsigned char *low_tab_end;
	unsigned char *high_tab_end;
	unsigned char *low_comp_end;
	unsigned char *high_comp_end;
#endif
} EIF_CONF_TAB;
/*
doc:	<attribute name="eif_first_gen_id" return_type="EIF_TYPE_INDEX" export="private">
doc:		<summary>base id of first generic type. All values below `eif_first_gen_id' do not represent generic classes.</summary>
doc:		<access>Read</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None since initialized during runtime initialization in `eif_gen_conf_init'.</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_next_gen_id" return_type="EIF_TYPE_INDEX" export="shared">
doc:		<summary>ID for next new generic derivation encountered during runtime execution.</summary>
doc:		<access>Read/Write</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
*/
rt_private EIF_TYPE_INDEX  eif_first_gen_id = 0;
rt_shared EIF_TYPE_INDEX  eif_next_gen_id  = 0;

/*
doc:	<attribute name="eif_conf_tab" return_type="EIF_CONF_TAB **" export="private">
doc:		<summary>Conformance tables.</summary>
doc:		<access>Read/Write</access>
doc:		<indexing>full type id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
doc:	<attribute name="eif_derivations" return_type="EIF_GEN_DER **" export="private">
doc:		<summary>Generic derivations.</summary>
doc:		<access>Read/Write</access>
doc:		<indexing>full type id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
*/
rt_private EIF_CONF_TAB **eif_conf_tab = NULL;
rt_private EIF_GEN_DER **eif_derivations = NULL;

/*
doc:	<attribute name="eif_type_names" return_type="struct htable" export="private">
doc:		<summary>Table containings all the type names in the system.</summary>
doc:		<access>Read/Write</access>
doc:		<indexing>Encoded type id</indexing>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>eif_gen_mutex</synchronization>
doc:	</attribute>
*/
rt_private struct htable eif_type_names;

#ifndef EIF_THREADS
/*
doc:	<attribute name="cid_array" return_type="EIF_TYPE_INDEX [4]" export="private">
doc:		<summary>Static array used by `eif_gen_cid' for `dftype' that are not generics.</summary>
doc:		<access>Read/Write</access>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>Private per thread data</synchronization>
doc:	</attribute>
*/
rt_private EIF_TYPE_INDEX cid_array [4];
#endif

/*------------------------------------------------------------------*/
/* THREADS.                                                         */
/* Calls to public routines are indirected and protected by a mutex */
/* The indirection avoids problems with recursive calls.            */
/*------------------------------------------------------------------*/

#ifdef  EIF_THREADS

/*
doc:	<attribute name="eif_gen_mutex" return_type="" export="shared">
doc:		<summary>Calls to public routines are indirected and protected by current mutex. Indirection is needed to avoids problem with recursive calls.</summary>
doc:		<access>Read</access>
doc:		<thread_safety>Safe</thread_safety>
doc:	</attribute>
*/
rt_shared EIF_CS_TYPE *eif_gen_mutex = NULL;

rt_private EIF_TYPE rt_thd_compound_id_with_context (struct rt_id_of_context *a_context, EIF_TYPE_INDEX, const EIF_TYPE_INDEX *);
rt_private EIF_TYPE rt_thd_final_id (EIF_TYPE_INDEX *, EIF_TYPE_INDEX **, EIF_TYPE_INDEX, int );
rt_private char rt_thd_gen_typecode_with_dftype (EIF_TYPE_INDEX , uint32);
rt_private EIF_TYPE rt_thd_gen_param (EIF_TYPE_INDEX , uint32);
rt_private EIF_TYPE_INDEX rt_thd_typeof_array_of (EIF_TYPE);
rt_private EIF_TYPE_INDEX rt_thd_typeof_type_of (EIF_TYPE);
rt_private EIF_TYPE_INDEX *rt_thd_gen_cid (EIF_TYPE, int);
rt_private int rt_thd_gen_conf2 (EIF_TYPE, EIF_TYPE);
rt_private char *rt_thd_typename(EIF_TYPE);
rt_private EIF_REFERENCE rt_thd_typename_of_type(EIF_TYPE);

#define EIFMTX_LOCK \
	{\
		RT_GET_CONTEXT \
		EIF_ASYNC_SAFE_CS_LOCK(eif_gen_mutex)

#define EIFMTX_UNLOCK \
		EIF_ASYNC_SAFE_CS_UNLOCK(eif_gen_mutex); \
	}

#else
/* Noop for locks in non-multithreaded mode. */
#define EIFMTX_LOCK
#define EIFMTX_UNLOCK
#endif
/*------------------------------------------------------------------*/

rt_private size_t rt_typename_len (EIF_TYPE, int);
rt_private void rt_typename (EIF_TYPE, char*, int);
rt_private EIF_GEN_DER *rt_new_gen_der(uint32, EIF_TYPE*, EIF_TYPE_INDEX, char, char, uint32);
rt_private void rt_expand_tables(int);
rt_private EIF_TYPE rt_id_of (struct rt_id_of_context *a_context, const EIF_TYPE_INDEX**, EIF_TYPE**, EIF_TYPE_INDEX, int);
rt_private void rt_compute_ctab (EIF_TYPE_INDEX);
rt_private EIF_CONF_TAB *rt_new_conf_tab (EIF_TYPE_INDEX, EIF_TYPE_INDEX, EIF_TYPE_INDEX, EIF_TYPE_INDEX);
rt_private void rt_enlarge_conf_tab (EIF_CONF_TAB *, EIF_TYPE_INDEX);
rt_private uint16 rt_gen_seq_len (EIF_TYPE);
rt_private void rt_put_gen_seq (EIF_TYPE, EIF_TYPE_INDEX*, EIF_TYPE_INDEX*, int);

/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/

/*
doc:	<routine name="eif_compound_id" return_type="EIF_TYPE" export="public">
doc:		<summary>Given a current dynamic type `dftype' and a sequence of dtypes, compute the corresponding dynamic type.</summary>
doc:		<param name="current_dftype" type="EIF_TYPE_INDEX">Context in which dynamic type is computed.</param>
doc:		<param name="types" type="EIF_TYPE_INDEX *">Arrays of dtypes representing a dynamic type.</param>
doc:		<return>A dynamic type which was either already computed, or a new one.</return>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None required, done internally.</synchronization>
doc:	</routine>
*/
rt_public EIF_TYPE eif_compound_id (EIF_TYPE_INDEX current_dftype, const EIF_TYPE_INDEX *types)
{
	return rt_compound_id_with_context (NULL, current_dftype, types);
}


#ifdef EIF_THREADS

/*------------------------------------------------------------------*/
/* Public features protected with a MUTEX.                          */
/*------------------------------------------------------------------*/
rt_shared EIF_TYPE rt_compound_id_with_context (struct rt_id_of_context *a_context, EIF_TYPE_INDEX current_dftype, const EIF_TYPE_INDEX *types)
{
	EIF_TYPE   result;

	EIFMTX_LOCK;
	result = rt_thd_compound_id_with_context (a_context, current_dftype, types);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_public EIF_TYPE eif_final_id (EIF_TYPE_INDEX *ttable, EIF_TYPE_INDEX **gttable, EIF_TYPE_INDEX dftype, int offset)
{
	EIF_TYPE   result;

	EIFMTX_LOCK;
	result = rt_thd_final_id (ttable, gttable, dftype, offset);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared char eif_gen_typecode_with_dftype (EIF_TYPE_INDEX dftype, uint32 pos)
{
	char    result;

	EIFMTX_LOCK;
	result = rt_thd_gen_typecode_with_dftype (dftype, pos);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_public EIF_TYPE eif_gen_param (EIF_TYPE_INDEX dftype, uint32 pos)
{
	EIF_TYPE result;

	EIFMTX_LOCK;
	result = rt_thd_gen_param (dftype, pos);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_typeof_array_of (EIF_TYPE dtype)
{
	EIF_TYPE_INDEX   result;

	EIFMTX_LOCK;
	result = rt_thd_typeof_array_of (dtype);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX rt_typeof_type_of (EIF_TYPE ftype)
{
	EIF_TYPE_INDEX result;

	EIFMTX_LOCK;
	result = rt_thd_typeof_type_of (ftype);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX *eif_gen_cid (EIF_TYPE a_dftype, int use_old_annotations)
{
	EIF_TYPE_INDEX   *result;

	EIFMTX_LOCK;
	result = rt_thd_gen_cid (a_dftype, use_old_annotations);
	EIFMTX_UNLOCK;

	return result;
}
/*------------------------------------------------------------------*/

rt_public int eif_gen_conf2 (EIF_TYPE source_type, EIF_TYPE target_type)
{
	int result;

	EIFMTX_LOCK;
	result = rt_thd_gen_conf2 (source_type, target_type);
	EIFMTX_UNLOCK;

	return result;
}

/*------------------------------------------------------------------*/

rt_public char * eif_typename (EIF_TYPE ftype)
{
	char * result;

	EIFMTX_LOCK;
	result = rt_thd_typename (ftype);
	EIFMTX_UNLOCK;

	return result;
}

/*------------------------------------------------------------------*/

rt_public char * eif_typename_of_type (EIF_TYPE ftype)
{
	char * result;

	EIFMTX_LOCK;
	result = rt_thd_typename_of_type (ftype);
	EIFMTX_UNLOCK;

	return result;
}

/*------------------------------------------------------------------*/
/* Rename public features if EIF_THREADS is on.                     */
/*------------------------------------------------------------------*/

#define rt_compound_id_with_context rt_thd_compound_id_with_context
#define eif_final_id              rt_thd_final_id
#define eif_gen_typecode_with_dftype  rt_thd_gen_typecode_with_dftype
#define eif_gen_param             rt_thd_gen_param
#define eif_typeof_array_of       rt_thd_typeof_array_of
#define rt_typeof_type_of        rt_thd_typeof_type_of
#define eif_gen_cid               rt_thd_gen_cid
#define eif_gen_conf2             rt_thd_gen_conf2
#define eif_typename              rt_thd_typename
#define eif_typename_of_type      rt_thd_typename_of_type

#endif

/*------------------------------------------------------------------*/
/* Initialize all structures                                        */
/* Called only once before root object is created.                  */
/*------------------------------------------------------------------*/

rt_shared void eif_gen_conf_init (EIF_TYPE_INDEX max_dtype)
{
	EIF_TYPE_INDEX dt;
	const char   *cname;
	struct eif_par_types **pt;

#ifdef EIF_THREADS
		/* Since we want to avoid any locks to happen on the access on
		 * eif_cid_map, we make sure that `eif_cid_map' can't be resized
		 * by giving the maximum size it can have, ie 0x0000FFFF */
	eif_cid_size = 65535;
#else
	eif_cid_size = max_dtype + 32;
#endif
	eif_first_gen_id = eif_next_gen_id = max_dtype + 1;

	/* Set `eif_par_table2' if it is null. */

	if (eif_par_table2 == NULL)
	{
		eif_par_table2 = eif_par_table;
		eif_par_table2_size = eif_par_table_size;
	}

	eif_cid_map = (EIF_TYPE_INDEX *) cmalloc (eif_cid_size * sizeof (EIF_TYPE_INDEX));

	if (eif_cid_map == NULL)
		enomem();

	eif_derivations = (EIF_GEN_DER **) cmalloc(eif_cid_size * sizeof (EIF_GEN_DER*));

	if (eif_derivations == NULL)
		enomem();

		/* Create hash-table containing all the type names. */
	memset(&eif_type_names, 0, sizeof(struct htable));
	ht_create (&eif_type_names, eif_cid_size, sizeof(char *));

	eif_conf_tab = (EIF_CONF_TAB **) cmalloc(eif_cid_size * sizeof (EIF_CONF_TAB*));

	if (eif_conf_tab == NULL)
		enomem();

	/* Setup a 1-1 mapping and initialize the arrays */

	for (dt = 0; dt < eif_cid_size; ++dt)
	{
		eif_cid_map [dt]     = dt;
		eif_derivations [dt] = NULL;
		eif_conf_tab [dt]    = NULL;
	}

	/* Now initialize egc_xxx_dtypes */

	for (dt = 0, pt = eif_par_table2; dt < eif_par_table2_size; ++dt, ++pt)
	{
		if (*pt == (struct eif_par_types *)0)
			continue;

		cname = System((*pt)->dtype).cn_generator;

		if ((strcmp("ANY",cname)==0))
		{
			egc_any_dtype = dt;
		}

		if ((strcmp("TUPLE",cname)==0))
		{
			tuple_static_type = dt;
		}
	}

	eif_gen_conf_thread_init();
}

/*
doc:	<routine name="eif_gen_conf_thread_init" return_type="void" export="shared">
doc:		<summary>Initialize per thread data used for generic conformance. Root thread initialization is done in `eif_gen_conf_init' as when we are called by the `eif_thr_register' routine, the value `eif_first_gen_id' is still zero.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</routine>
*/
rt_shared void eif_gen_conf_thread_init (void) {
}

/*
doc:	<routine name="eif_gen_conf_thread_cleanup" return_type="void" export="shared">
doc:		<summary>Free per thread data used for generic conformance.</summary>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</routine>
*/
rt_shared void eif_gen_conf_thread_cleanup (void) {
}

/*------------------------------------------------------------------*/
/* Clean up.                                                        */
/* Called from reclaim, and free all global variables allocated     */
/* for the Generic Conformance.                                     */
/*------------------------------------------------------------------*/
rt_shared void eif_gen_conf_cleanup (void)
{
	/* Free in reverse order of allocation. */

	int i, j;
	size_t k;

	REQUIRE ("eif_conf_tab not null", eif_conf_tab);
	REQUIRE ("eif_derivations not null", eif_derivations);
	REQUIRE ("eif_cid_map not null", eif_cid_map);

	/* Recursively free eif_conf_tab */
	for (i = 0; i < eif_cid_size; i++) {
		EIF_CONF_TAB *tmp = eif_conf_tab [i];

		if (tmp == NULL)
			continue;

		if (tmp->low_tab) {
			eif_rt_xfree (tmp->low_tab);	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->low_tab)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->high_tab) {
			eif_rt_xfree (tmp->high_tab);	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->high_tab)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->low_comp) {
			eif_rt_xfree (tmp->low_comp);	 	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->low_comp)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->high_comp)	{
			eif_rt_xfree (tmp->high_comp);	 	/* unsigned char * */
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->high_comp)));
#endif	/* LMALLOC_CHECK */
		}
		eif_rt_xfree (tmp);
	}
	eif_rt_xfree (eif_conf_tab);

	/* Recursively free eif_derivations. */
	for (i = 0; i < eif_cid_size; i++) {
		EIF_GEN_DER *tmp = eif_derivations [i];

		if (tmp == NULL)
			continue;
		if (tmp->types) {
			eif_rt_xfree (tmp->types);
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->types)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->gen_seq) {
			eif_rt_xfree (tmp->gen_seq);
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->gen_seq)));
#endif	/* LMALLOC_CHECK */
		}
		if (tmp->ptypes) {
			eif_rt_xfree (tmp->ptypes);
		} else {
#ifdef LMALLOC_CHECK
			CHECK ("", !(is_in_lm (tmp->ptypes)));
#endif	/* LMALLOC_CHECK */
		}
		for (j = i + 1; j < eif_cid_size; j++) {
			if (eif_derivations [j] == tmp)
				eif_derivations[j] = NULL;

		}
		eif_rt_xfree (tmp);
	}
	eif_rt_xfree (eif_derivations);

		/* Free any allocated strings used to store type names and
		 * then the table itself. */
	for (k = 0 ; k < eif_type_names.h_capacity; k++) {
		char **l_name = (char **) (((char *) eif_type_names.h_values) + (k * sizeof(char *)));
		if (*l_name) {
			eif_rt_xfree (*l_name);
		}
	}
	ht_release(&eif_type_names);
	memset(&eif_type_names, 0, sizeof(struct htable));

	eif_rt_xfree (eif_cid_map);
#ifndef EIF_THREADS
	eif_gen_conf_thread_cleanup();
#endif
} /* eif_gen_conf_cleanup () */

/*------------------------------------------------------------------*/
/* Compute id for `types'. `cache' is used to cache the result in   */
/* the generated C code if possible.                                */
/*                                                                  */
/* types   : Id array                                               */
/* Result  : Resulting id;                                          */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE rt_compound_id_with_context (struct rt_id_of_context *a_context, EIF_TYPE_INDEX current_dftype, const EIF_TYPE_INDEX *types)
{
	EIF_TYPE outtab [256], *outtable;

	REQUIRE("types not NULL", types);
	REQUIRE("types not empty", types [0] != TERMINATOR);

	outtable = outtab;
	return rt_id_of (a_context, &types, &outtable, current_dftype, 0);
}


/*------------------------------------------------------------------*/
/* Compute id for `gttable' (generic type list for feature in final */
/* mode).                                                           */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE eif_final_id (EIF_TYPE_INDEX *ttable, EIF_TYPE_INDEX **gttable, EIF_TYPE_INDEX dftype, int offset)
{
	EIF_TYPE_INDEX *gtp;
	EIF_TYPE_INDEX dtype = To_dtype(dftype);
	EIF_TYPE non_generic_type;
	int	table_index = dtype - offset;

	REQUIRE("gttable not NULL", gttable);

	gtp = gttable [table_index];

	if ((gtp != NULL) && (*(gtp+1) != TERMINATOR)) {
			/* Type is actually generic. */
		return eif_compound_id (dftype, gtp);
	} else {
			/* Optimization for non-generic type. */
		non_generic_type.id = ttable[table_index];
		non_generic_type.annotations = 0;
		return non_generic_type;
	}
}

/*------------------------------------------------------------------*/
/* Number of generic parameters of `obj's type. Can ONLY be used for*/
/* TUPLE and its descendants!                                       */
/*------------------------------------------------------------------*/

rt_public uint32 eif_gen_count_with_dftype (EIF_TYPE_INDEX dftype)
{
	EIF_GEN_DER *gdp;

	REQUIRE("Valid type", dftype < eif_next_gen_id);

	gdp = eif_derivations [dftype];

	return (gdp ? gdp->size : 0);
}

/*------------------------------------------------------------------*/
/* Number of generic parameters of `obj's type. Can ONLY be used for*/
/* TUPLE                                                            */
/*------------------------------------------------------------------*/

rt_public uint32 eif_tuple_count (EIF_REFERENCE obj)
{
	return (obj ? RT_SPECIAL_COUNT(obj) - 1 : 0);
}

/*------------------------------------------------------------------*/
/* Are all generic parameters of basic types? Can ONLY be used for  */
/* TUPLE                                                            */
/*------------------------------------------------------------------*/

rt_shared int eif_tuple_is_atomic (EIF_REFERENCE obj)
{
	EIF_VALUE *l_item = (EIF_VALUE *) obj;
	unsigned int count;

	if (obj == NULL) {
			/* This is atomic */
		return 1;
	}

	CHECK("Tuple object", HEADER(obj)->ov_flags & EO_TUPLE);
	count = RT_SPECIAL_COUNT(obj);

		/* Don't forget that first element of TUPLE is the BOOLEAN
		 * `object_comparison' attribute. */
	l_item++;
	for (; count > 0 ; count--) {
		if (eif_is_reference_tuple_item(l_item)) {
				/* It has a reference. This is therefore not atomic */
			return 0;
		}
	}
		/* No reference found. It is atomic */
	return 1;
}

/*------------------------------------------------------------------*/
/* Typecode of generic type at position `pos' in `obj'. ONLY for    */
/* TUPLE                                                            */
/*------------------------------------------------------------------*/

rt_public char eif_gen_typecode (EIF_REFERENCE obj, uint32 pos)
{
	if (obj == NULL) {
		return (char) 0;
	} else {
		char result;
			/* Critical section as we might compute a new `eif_anc_id_map' entry */
		EIFMTX_LOCK;
		result = eif_gen_typecode_with_dftype (Dftype(obj), pos);
		EIFMTX_UNLOCK;
		return result;
	}
}

rt_shared char eif_gen_typecode_with_dftype (EIF_TYPE_INDEX dftype, uint32 pos)
{
	EIF_TYPE_INDEX gtype;
	EIF_GEN_DER *gdp;

		/* Check type validity */
	REQUIRE ("dftype is less than maximum computed id", dftype < eif_next_gen_id);
	REQUIRE ("We have routines, so we must have tuples.", tuple_static_type < MAX_DTYPE);

	gdp = eif_derivations [dftype];

	CHECK ("gdp not null", gdp != (EIF_GEN_DER *)0);

	CHECK ("Valid generic position min", pos > 0);
	CHECK ("Valid generic position max", pos <= gdp->size);

	gtype = gdp->types [pos-1].id;

	if (RT_CONF_IS_NONE_TYPE(gtype)) {
		return EIF_REFERENCE_CODE;
	}

	return EIF_TUPLE_CODE(System(eif_cid_map[gtype]));
}

/*------------------------------------------------------------------*/
/* Typecode string for target/argument types of a ROUTINE object.   */
/* ONLY for ROUTINE!                                                */
/*------------------------------------------------------------------*/

rt_public EIF_REFERENCE eif_gen_typecode_str (EIF_REFERENCE obj)
{
	EIF_GET_CONTEXT

	EIF_REFERENCE ret;	/* Return value. */
	EIF_TYPE_INDEX dftype, gtype;
	int len;
	uint32 pos;
	EIF_GEN_DER *gdp;
	char *strp;

	REQUIRE ("obj not null", obj != (EIF_REFERENCE )0);

	dftype = Dftype(obj);

	REQUIRE ("Valid dftype", dftype < eif_next_gen_id);

	gdp = eif_derivations [dftype];

	CHECK ("gdp not null", gdp != (EIF_GEN_DER *)0);
	CHECK ("Not a routine object", gdp->size > 0);

		/* Treat the arguments. This is necessarily a TUPLE. */
	dftype = gdp->types [0].id;

	CHECK ("Valid dftype", dftype < eif_next_gen_id);
	CHECK ("Routines implies we have tuples", tuple_static_type < MAX_DTYPE);

	gdp = eif_derivations [dftype];

	CHECK ("gdp not null", gdp != (EIF_GEN_DER *)0);

		/* Create a string for gdp->size characters */
	len = gdp->size;

	ret = emalloc(egc_str_dtype);
	RT_GC_PROTECT(ret);
		/* Protect address in case it moves */

	nstcall = 0;
	RT_STRING_MAKE(ret, (EIF_INTEGER) len);
	RT_STRING_SET_COUNT(ret, len);

	/* We know the `area' is the very first reference
	 * of the STRING object, hence the simple de-referencing.
	 */

	strp = *(EIF_REFERENCE*)ret;

	for (pos = 0; pos < gdp->size; pos++, strp++) {
		gtype = gdp->types [pos].id;
		if (RT_CONF_IS_NONE_TYPE(gtype)) {
			*strp = EIF_REFERENCE_CODE;
		} else {
			*strp = EIF_TUPLE_CODE(System(eif_cid_map[gtype]));
		}
	}

	RT_GC_WEAN(ret);			/* Remove protection */

	return ret;
}

/*------------------------------------------------------------------*/
/* Annotated type of generic parameter in `obj' at position `pos'.  */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE eif_gen_param (EIF_TYPE_INDEX dftype, uint32 pos)

{
	EIF_GEN_DER *gdp;

	REQUIRE("Valid type", dftype < eif_next_gen_id);
	REQUIRE("pos positive", pos > 0);

	gdp = eif_derivations [dftype];

	CHECK("A generic type", gdp);
	CHECK("Valid generic parameter position", (pos <= gdp->size));

	return gdp->types [pos-1];
}

/*------------------------------------------------------------------*/
/* Type id for allocation ARRAY [something], where 'something' is a */
/* reference type.                                                  */
/* dftype : full type id;                                           */
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX eif_typeof_array_of (EIF_TYPE type)
{
	EIF_TYPE_INDEX l_types [4], result;

	REQUIRE("Valid ARRAY generic type", type.id <= MAX_DTYPE);
	REQUIRE("Valid ARRAY reference type", egc_arr_dtype <= MAX_DTYPE);
	REQUIRE("Valid annotations", RT_CONF_IS_VALID_ANNOTATION(type.annotations));

	l_types [0] = egc_arr_dtype;	/* Base type of ARRAY     */
		/* We add the 0xFF00 because we have the annotations in the type array
		 * which enables us to differentiate them from normal type IDs. */
	l_types [1] = type.annotations | 0xFF00;	/* Parameter type's annotations */
	l_types [2] = type.id;		/* Parameter type */
	l_types [3] = TERMINATOR;

	result = eif_compound_id (0, l_types).id;
	return result;
}

/*------------------------------------------------------------------*/
/* Type id for TYPE [a_type]                                        */
/* a_type : full type id with annotations;                          */
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX rt_typeof_type_of (EIF_TYPE a_type)
{
	EIF_TYPE_INDEX l_types [4], result, l_type;
	struct cecil_info *cecil_type;
	uint32 sk_type;			/* Generic information for a_type */
	uint32 *t;				/* To walk through the patterns array */
	int matched = 0;
	size_t index = 0;

	REQUIRE("Valid actual generic type", (a_type.id <= MAX_DTYPE) || (RT_CONF_IS_NONE_TYPE(a_type.id)));

		/* Get the CECIL description for TYPE. */
	cecil_type = (struct cecil_info *) ct_value (&egc_ce_type, "TYPE");

		/* Get the SK_type for X in TYPE [X] were are trying to build. */
	if (RT_CONF_IS_NONE_TYPE(a_type.id)) {
			/* For NONE we use the reference generic derivation of TYPE. */
		sk_type = SK_REF;
	} else {
		l_type = To_dtype(a_type.id);
		sk_type = eif_dtype_to_sk_type (l_type);
	}

		/* Now try to find the proper TYPE generic derivation. */
	t = cecil_type->patterns;
	while ((*t != SK_INVALID) && (!matched)) {
		matched = (*t++ == sk_type);
		index++;
	}

	if (matched == 1) {
		l_types [0] = cecil_type->dynamic_types[index - 1];		/* Base type of TYPE */
		if (a_type.annotations) {
				/* We add the 0xFF00 because we have the annotations in the type array
				 * which enables us to differentiate them from normal type IDs. */
			l_types[1] = a_type.annotations | 0xFF00;
			l_types[2] = a_type.id;			/* Parameter type */
			l_types[3] = TERMINATOR;
		} else {
			l_types[1] = a_type.id;			/* Parameter type */
			l_types[2] = TERMINATOR;
		}

		result = eif_compound_id (0, l_types).id;
	} else {
		result = INVALID_DTYPE;
	}

	return result;
}

/*------------------------------------------------------------------*/
/* CID which generates `dftype'. First entry is the length of the   */
/* compound id.                                                     */
/*                                                                  */
/* dftype : full type id;                                           */
/* Result : base ids;                                               */
/*------------------------------------------------------------------*/

rt_shared EIF_TYPE_INDEX *eif_gen_cid (EIF_TYPE a_dftype, int use_old_annotations)
{
	RT_GET_CONTEXT
	uint16 len;
	EIF_GEN_DER *gdp;
	EIF_TYPE_INDEX l_dftype;

	l_dftype = a_dftype.id;

	if ((RT_CONF_IS_NONE_TYPE(l_dftype)) || (l_dftype < eif_first_gen_id)) {
		if (a_dftype.annotations) {
				/* Size of 2 in the array. */
			cid_array [0] = 2;
				/* We add the 0xFF00 because we have the annotations in the type array
				 * which enables us to differentiate them from normal type IDs. */
			cid_array [1] = a_dftype.annotations | 0xFF00;
			if (use_old_annotations) {
					/* For backward compatibility for C storable we annotations used
					 * to be 0xFF1x we add the UNUSABLE_FLAG so that old systems can
					 * still retrieve void-safe data generated by a newer version of
					 * the runtime. */
				cid_array [1] |= UNUSABLE_FLAG;
			}
			cid_array [2] = l_dftype;
			cid_array [3] = TERMINATOR;
		} else {
				/* Size of 1 in the array. */
			cid_array [0] = 1;
			cid_array [1] = l_dftype;
			cid_array [2] = TERMINATOR;
		}
		return cid_array;
	}

	/* It's a run-time generated id */

	gdp = eif_derivations [l_dftype];

	if (gdp->gen_seq) {
		return gdp->gen_seq;        /* already computed */
	}
	/* Compute size of array */

	len = rt_gen_seq_len (a_dftype);
	gdp->gen_seq = (EIF_TYPE_INDEX *) cmalloc ((len+2)*sizeof(EIF_TYPE_INDEX));
	if (!gdp->gen_seq) {
		enomem();
	}

	gdp->gen_seq [0] = len;
	gdp->gen_seq [len+1] = TERMINATOR;

	/* Fill array */

	len = 1;

	rt_put_gen_seq (a_dftype, gdp->gen_seq, &len, use_old_annotations);

	return gdp->gen_seq;
}
/*------------------------------------------------------------------*/
/* Conformance test. Does `source_type' conform to `target_type'?   */
/* This only applies to instantiated type.                          */
/*                                                                  */
/* Source_type : full type id;                                      */
/* Target_type : full type id;                                      */
/*------------------------------------------------------------------*/

rt_public int eif_gen_conf2 (EIF_TYPE stype, EIF_TYPE ttype)
{
	EIF_CONF_TAB *stab;
	EIF_GEN_DER *sgdp, *tgdp;
	EIF_TYPE_INDEX *ptypes;
	uint32 i, idx;
	int result;
	unsigned char mask;

		/* If types are identical, then they conform if they have the same annotations. */
	if (stype.id == ttype.id) {
		return rt_is_conforming_annotation(stype, ttype);
	}

	CHECK("Type are differents", ttype.id != stype.id);

	if (ttype.id > MAX_DTYPE) {
			/* Target is NONE so we can only accept NONE but `stype' is not the same as `ttype'. */
		CHECK("NONE type", RT_CONF_IS_NONE_TYPE(ttype.id));
		return 0;
	}

	if (EIF_IS_EXPANDED_TYPE(System(eif_cid_map[ttype.id]))) {
			/* Expanded target no conformance because types are different */
		return 0;
	} else if (stype.id > MAX_DTYPE) {
			/* Target is not expanded and not NONE, but source is NONE.
			 * It only conforms if target is detachable. */
		CHECK("NONE type", RT_CONF_IS_NONE_TYPE(stype.id));
		return RT_CONF_IS_DETACHABLE_FLAG(ttype.annotations) || !RT_CONF_HAS_ATTACHMENT_MARK_FLAG(ttype.annotations);
	} else {
		if (EIF_IS_EXPANDED_TYPE(System(eif_cid_map[stype.id]))) {
				/* Source is expanded, we force the attachment mark, otherwise
				* eweasel test#conform001 will say that A [INTEGER] does not
				* conform to A [attached ANY] which is clearly not the case. Note
				* that this is because in `rt_id_of' we clear the annotations when
				* type is expanded. We clear it to maintain as much backward compatibility
				* with existing code. */
			CHECK("no attachment marks", !RT_CONF_HAS_ATTACHMENT_MARK_FLAG(stype.annotations) || RT_CONF_IS_ATTACHED_FLAG(stype.annotations));
			stype.annotations |= ATTACHED_FLAG;
		}
			/* If `nnotations are not compatible, the types are not conforming. */
		if (!rt_is_conforming_annotation(stype, ttype)) {
			return 0;
		}
	}

	stab = eif_conf_tab[stype.id];

	if (stab == NULL) {
		rt_compute_ctab (stype.id);
		stab = eif_conf_tab[stype.id];
	}

	if (ttype.id < eif_first_gen_id) {
		/* Lower id */

		if ((ttype.id >= stab->min_low_id) && (ttype.id <= stab->max_low_id)) {
			idx = ttype.id - stab->min_low_id;
			mask = (unsigned char) (1 << (idx % 8));

			return (mask == ((stab->low_tab)[idx/8] & mask)) ? 1 : 0;
		}
	} else {
		/* High id */

		if ((ttype.id < stab->min_high_id) || (ttype.id > stab->max_high_id)) {
				/* We need to enlarge the table */
			rt_enlarge_conf_tab (stab, ttype.id);
		}

		/* Now ttype is in the table range */

		idx  = (ttype.id - stab->min_high_id);
		mask = (unsigned char) (1 << (idx % 8));

		/* If we have computed it already, return result
		 * We check first if the computed value is '1', if so, it means both that we already
		 * computed it and that is True.
		 * If the computed value is '0' we check if we compute a value, if so we return 0
		 * because we already know the computed value, otherwise we compute it
		 */

		if (mask == ((stab->high_tab)[idx/8] & mask))
			return 1;
		if (mask == ((stab->high_comp)[idx/8] & mask))
			return 0;

		/* We have to compute it now (once!) */

		sgdp = eif_derivations [stype.id];
		tgdp = eif_derivations [ttype.id];
		result = 0;

		/* Check annotations. */

		if (stype.id >= eif_first_gen_id) {
				/* Both ids generated here */
			if (sgdp->base_dtype == tgdp->base_dtype) {
				/* Both have the same base class, so we can simply check that the
				 * adapted parents are the same.*/

					/* Same base class. If nr. of generics differs, both are TUPLEs. */
				if (tgdp->size > sgdp->size) {
						/* Source and target are TUPLES but source has fewer parameters
						 * is not conforming. */
					CHECK("result is false", !result);
					goto done;
				}

				for (i = 0; i < tgdp->size; ++i) {
					stype = sgdp->types[i];
					ttype = tgdp->types[i];

					if ((stype.id == ttype.id) && (stype.annotations == ttype.annotations)) {
							/* Trivially the same type, so we can skip it and thus avoiding recursion. */
						continue;
					} else if (!eif_gen_conf2 (stype, ttype)) {
							/* Not conforming. */
						CHECK("result is false", !result);
						goto done;
					}
				}

				result = 1;
				goto done;
			}
		}

		/* Target is generic. But source is not.
		   We need to check every parent of the source
		   against the target */

		ptypes = sgdp->ptypes;
		result = 0;
			/* Annotations are now irrelevant, we are clearing them. */
		ttype.annotations = 0;
		stype.annotations = 0;
		while (!result && (*ptypes != TERMINATOR)) {
			stype.id = *ptypes;
			result = eif_gen_conf2 (stype, ttype);
			++ptypes;
		}

done:
		/* Register that we have computed it */
		(stab->high_comp)[idx/8] |= mask;

		if (result)
			(stab->high_tab)[idx/8] |= mask;

		return result;
	}

	return 0;
}
/*------------------------------------------------------------------*/
/* Private routines.                                                */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* Computation of a new id.                                         */
/*                                                                  */
/* intab      : dtypes array                                        */
/* outtab     : List of computed ids for generics.                  */
/* obj_type   : Full type of object; Used to replace a              */
/*              formal generic by an actual generic of the object.  */
/* annotation : Annotation of type if any (see ANNOTATION_TYPE_MASK)*/
/*------------------------------------------------------------------*/
/*
doc:	<routine name="rt_annotations_from_generated_id" return_type="EIF_TYPE_INDEX" export="private">
doc:		<summary>Given a generated ID which is an annotation, get the corresponding flag for {EIF_TYPE}.annotations.</summary>
doc:		<param name="annotation" type="EIF_TYPE_INDEX">A compiler generated annotation ID to be converted.</param>
doc:		<return>The annotation for an EIF_TYPE instance.</return>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>none</synchronization>
doc:	</routine>
*/
rt_private rt_inline EIF_TYPE_INDEX rt_annotations_from_generated_id (EIF_TYPE_INDEX annotation) {
	EIF_TYPE_INDEX result;

	REQUIRE("Is annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(annotation));

		/* Note that compiler generated IDs have the 0x0010 bit set when they have
		 * an attachment mark (for backward compatibility) but since we never use that
		 * bit in {EIF_TYPE}.annotations we can safely leave it there. */
	result = annotation & 0x00FF;

	ENSURE("valid_annotation", RT_CONF_IS_VALID_ANNOTATION(result));

	return result;
}

/*
doc:	<routine name="rt_merged_annotation" return_type="EIF_TYPE_INDEX" export="shared">
doc:		<summary>Given two consecutives annotation, compute the resulting one. For example, if the first annotation is ATTACHED_FLAG and the second one is DETACHABLE_FLAG, then the resulting is just ATTACHED_FLAG. If they are DETACHABLE_FLAG and SEPARATE_FLAG, then the resulting is a combination of both.</summary>
doc:		<param name="first_annotation" type="EIF_TYPE_INDEX">First annotation.</param>
doc:		<param name="second_annotation" type="EIF_TYPE_INDEX">Second annotation to be merged into `first_annotation'.</param>
doc:		<return>The combined annotation.</return>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>none</synchronization>
doc:	</routine>
*/
rt_shared EIF_TYPE_INDEX rt_merged_annotation (EIF_TYPE_INDEX first_annotation, EIF_TYPE_INDEX second_annotation)
{
	EIF_TYPE_INDEX result = 0;
	EIF_TYPE_INDEX l_annotation;

	REQUIRE("Is first annotation?", RT_CONF_IS_VALID_ANNOTATION(first_annotation));
	REQUIRE("Is second annotation?", RT_CONF_IS_VALID_ANNOTATION(second_annotation));

	if (!first_annotation) {
			/* Simple case where there is no annotation. Currently this is only possible when we encounter a
			 * formal generic parameter that needs to be evaluated in a context, in that case we take whatever
			 * annotations from the context. In other words in a type LIST [separate A], the annotation of
			 * G is separate. */
		result = second_annotation;
	} else if (!second_annotation) {
			/* Simple case where there is no annotation. Note this is for efficiency
			 * since the code below in the `else' clause would handled this properly. */
		result = first_annotation;
	} else {
			/* 1. Merge attachment marks. */
		l_annotation = (first_annotation & (ATTACHED_FLAG | DETACHABLE_FLAG));
		if (l_annotation) {
				/* First annotation has an attachment mark, we will use it as it overrides
				 * whatever is coming next. */
			result |= l_annotation;
		} else {
				/* No attachment mark specified, we use the one from `second_annotation'. */
			result |= (second_annotation & (ATTACHED_FLAG | DETACHABLE_FLAG));
		}

			/* 2. Merge separate status. */
		l_annotation = (first_annotation & SEPARATE_FLAG);
		if (l_annotation) {
			result |= l_annotation;
		} else {
			result |= (second_annotation & SEPARATE_FLAG);
		}

			/* 3. Merge variance status. */
		l_annotation = (first_annotation & (VARIANT_FLAG | FROZEN_FLAG | POLY_FLAG));
		if (l_annotation) {
			result |= l_annotation;
		} else {
			result |= (second_annotation & (VARIANT_FLAG | FROZEN_FLAG | POLY_FLAG));
		}
	}
	return result;
}


/*
doc:	<routine name="rt_id_of" return_type="EIF_TYPE" export="private">
doc:		<summary>Given a set of compiler generated IDs in `intab' we evaluate them in the context of `obj_type' to compute a type with its annotations. If it has some actual generic parameters, their computed type is stored in the type sequence `outtab'.</summary>
doc:		<param name="a_context" type="struct rt_id_of_context *">State context used during computation. If present we can state if the evaluation of the compiler generated IDs required the `ojb_type'. Mostly used for the C store/retrieve to see if we have qualified anchored type.</param>
doc:		<param name="intab" type="const EIF_TYPE_INDEX **">Compiler generated IDs. They include both annotations and type IDs.</param>
doc:		<param name="outtab" type="EIF_TYPE **">Set of type forming the computed type actual generic parameters. For example LIST [A], would create the following array [{A, ATTACHED_FLAG}] (ATTACHED_FLAG since by default types are attached).</param>
			<param name="obj_type" type="EIF_TYPE_INDEX">Type of object that can be used to resolve type of actual generic parameters or anchors.</a_param>
			<param name="a_depth" type="int">Depth of the recursion. 0 means first level. It is used so that we remove the POLY_FLAG tag in annotations at the first level since this is something that can only appear in formal generic parameters.</a_param>
doc:		<return>The computed type ID with its annotations.</return>
doc:		<thread_safety>Safe.</thread_safety>
doc:		<synchronization>Through `eif_gen_mutex'.</synchronization>
doc:	</routine>
*/
rt_private EIF_TYPE rt_id_of (struct rt_id_of_context *a_context, const EIF_TYPE_INDEX **intab, EIF_TYPE **outtab, EIF_TYPE_INDEX obj_type, int a_depth)
{
	EIF_TYPE_INDEX l_type_entry, gcount = 0, i, nb;
	EIF_TYPE_INDEX l_dtype;
	EIF_TYPE *save_outtab, result, l_type;
	int     mcmp;
	uint32	pos, hcode;
	int l_routine_id, l_just_computed = 0;
	char    is_expanded, is_tuple;
	EIF_GEN_DER *gdp, *prev;

		/* Setup result to an invalid type to begine with. */
	result.id = INVALID_DTYPE;
	result.annotations = 0;

	if (a_context && a_context->is_invalid) {
			/* Previous computation was broek, so here we just return a dummy dynamic type.
			 * Callers should check that `is_invalid' is not set to `1' before accessing
			 * the computed type. */
		return result;
	}

		/* Get full type */
	l_type_entry = **intab;

	CHECK("Not terminator", l_type_entry != TERMINATOR);

		/* Read possible annotations. They can be more in the case we are trying to resolve
		 * a type such as `attached like x' where `x' is defined as `separate detachable X',
		 * the resulting type is actually `attached separate X'. */
	if (RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(l_type_entry)) {
		result.annotations = rt_annotations_from_generated_id (l_type_entry);
		(*intab)++;
		l_type_entry = **intab;
		while (RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(l_type_entry)) {
			result.annotations = rt_merged_annotation (result.annotations, rt_annotations_from_generated_id (l_type_entry));
			(*intab)++;
			l_type_entry = **intab;
		}
	}

		/* FIXME: Manu 2015/07/13: The following is to remove a limitation of the Eiffel code
		 * generation which should never generate a POLY_FLAG as the annotation of a type.
		 * POLY_FLAG can only appear in actual generic parameter. */
	if (a_depth == 0) {
		result.annotations &= ~POLY_FLAG;
	}

	if (l_type_entry <= MAX_DTYPE) {
		l_dtype = eif_cid_map[l_type_entry];
		if (EIF_IS_EXPANDED_TYPE(System (l_dtype))) {
				/* Type is expanded, we have to remove attached/detachable or frozen annotations
				 * since the nature of the type carries this with itself. */
			is_expanded = '1';
			result.annotations &= ~(DETACHABLE_FLAG | ATTACHED_FLAG | FROZEN_FLAG);
		} else {
			is_expanded = (char) 0;
			if (EIF_IS_FROZEN_TYPE(System (l_dtype))) {
					/* Type is a frozen class, no need for the frozen annotation. */
				result.annotations &= ~FROZEN_FLAG;
			}
		}
	} else {
		is_expanded = (char) 0;
	}

		/* Check if type is NONE. */
	if (RT_CONF_IS_NONE_TYPE(l_type_entry)) {
		(*intab)++;
		result.id = l_type_entry;
		if (RT_CONF_IS_DETACHABLE_FLAG(result.annotations)) {
				/* We clear the detachable flag as otherwise we could end up with 2
				 * types if one is declared as `A [detachable C]' which will be generated as A [C]
				 * and the other as `A [detachable like x]' which will have the detachable mark.
				 * See eweasel test#conform001, test#reflection005. */
			result.annotations &= ~DETACHABLE_FLAG;
		}
		**outtab = result;
		(*outtab)++;
		return result;
	}

		/* Check whether it's a TUPLE Type */
	if (l_type_entry == TUPLE_TYPE) {
		(*intab)++;
		gcount = **intab;       /* Number of generic params */
		(*intab)++;
		l_type_entry = **intab;       /* Base id for TUPLE */
		is_tuple = '1';
	} else {
		is_tuple = (char) 0;
	}

	if (l_type_entry == FORMAL_TYPE) {
			/* formal generic */
		(*intab)++;
		pos = **intab;	/* Position of formal generic */
		(*intab)++;

		if (a_context && a_context->has_no_context) {
				/* We cannot compute the proper type since no valid
				 * `obj_type' was provided. We flag `a_context' and return
				 * a dummy dynamic type. */
			a_context->is_invalid = 1;
			result.id = 0;
		} else {
			gdp = eif_derivations [obj_type];
			CHECK("Valid generic parameter position", pos <= gdp->size);
			result.annotations = rt_merged_annotation (result.annotations, gdp->types [pos - 1].annotations);
			result.id = gdp->types [pos-1].id;
		}
		l_just_computed = 1;

	} else if (l_type_entry == QUALIFIED_FEATURE_TYPE) {
			/* Qualified anchor type: #routine_IDs, qualifier type, rid_1, rid_2, ... */
		(*intab)++;
		nb = **intab; /* Length of the chain not including the qualifier. */
			/* Get type of qualifier. */
		(*intab)++;
			/* We save `outtab' as we do not care about the value
			 * stored in it by the call to `rt_id_of' since we are
			 * just going to use the return value to calculate
			 * the dynamic type by processing the chain of routine IDs. */
		save_outtab = *outtab;
		l_type = rt_id_of (a_context, intab, outtab, obj_type, 0);
		*outtab = save_outtab;
		if (a_context && a_context->is_invalid) {
				/* Context is invalid, so we just read all the routine IDs before
				 * returning a dummy dynamic type. */
			for (i = nb; i; --i) {
				l_routine_id = (int) **intab;
				if (l_routine_id & 0x8000) {
						/* This is a routine ID whose value is greater than 0x7FFF */
					(*intab)++;
					l_routine_id = ((l_routine_id & 0x7FFF) << 16) | (int) **intab;
				}
				(*intab)++;
			}
			result.id = 0;
		} else {
				/* Using the computed dynamic type as a starting point, compute the type
				 * for each element in the chain. */
			for (i = nb; i; --i) {
				l_routine_id = (int) **intab;
				if (l_routine_id & 0x8000) {
						/* This is a routine ID whose value is greater than 0x7FFF */
					(*intab)++;
					l_routine_id = ((l_routine_id & 0x7FFF) << 16) | (int) **intab;
				}
#ifdef WORKBENCH
				l_type = wtype_gen(l_routine_id, To_dtype(l_type.id), l_type.id);
#else
				l_type = eif_final_id (egc_routines_types [l_routine_id],
					egc_routines_gen_types [l_routine_id], l_type.id,
					egc_routines_offset [l_routine_id]);
#endif
				(*intab)++;
			}
				/* Now that the chain has been computed, we construct the result
				 * in particular the annotations that needs to be merged. */
			result.id = l_type.id;
			result.annotations = rt_merged_annotation (result.annotations, l_type.annotations);
		}
		l_just_computed = 1;

	/* This is only for melted mode only. For frozen and finalized mode we do not use this
	 * Instead we do this via code generation through `generate_cid_array' and `generate_cid_init'. */
#ifdef WORKBENCH
	} else if (l_type_entry == LIKE_CURRENT_TYPE) {
		(*intab)++;
		if (a_context && a_context->has_no_context) {
				/* We cannot compute the proper type since no valid
				 * `obj_type' was provided. We flag `a_context' and return
				 * a dummy dynamic type. */
			a_context->is_invalid = 1;
			result.id = 0;
		} else {
			result.id = obj_type;
		}
		l_just_computed = 1;

	} else if (l_type_entry == LIKE_FEATURE_TYPE) {
		(*intab)++;
		l_routine_id = (int) **intab;
		if (l_routine_id & 0x8000) {
				/* This is a routine ID whose value is greater than 0x7FFF */
			(*intab)++;
			l_routine_id = ((l_routine_id & 0x7FFF) << 16) | (int) **intab;
		}
		(*intab)++;
		if (a_context && a_context->has_no_context) {
				/* We cannot compute the proper type since no valid
				 * `obj_type' was provided. We flag `a_context' and return
				 * a dummy dynamic type. */
			a_context->is_invalid = 1;
			result.id = 0;
		} else {
			l_type = wtype_gen(l_routine_id, To_dtype(obj_type), obj_type);
			result.id = l_type.id;
			result.annotations = rt_merged_annotation(result.annotations, l_type.annotations);
		}
		l_just_computed = 1;

	} else if (l_type_entry == LIKE_ARG_TYPE) {
		(*intab)++;
		pos = (int) **intab;
		(*intab)++;
			/* We save `outtab' as we do not care about the value
			 * stored in it by the call to `rt_id_of' since we are
			 * just going to use the return value to calculate
			 * the dynamic type by processing the chain of routine IDs. */
		save_outtab = *outtab;
		l_type = rt_id_of (a_context, intab, outtab, obj_type, 0);
		*outtab = save_outtab;
		if (a_context && a_context->is_invalid) {
				/* Here we return a dummy dynamic type. If this is part of a more complex
				 * type, it won't stop that computation, but it won't be a correct type.
				 * Callers should check that `is_invalid' is not set to `1' before accessing
				 * the computed type. */
			result.id = 0;
		} else {
			EIF_REFERENCE l_ref = rt_melted_arg(pos);
			if (l_ref) {
				if (egc_is_experimental) {
					l_type = eif_new_type(Dftype(l_ref), ATTACHED_FLAG);
				} else {
					l_type = eif_new_type(Dftype(l_ref), 0);
				}
			}
			result.id = l_type.id;
			result.annotations = rt_merged_annotation (result.annotations, l_type.annotations);
		}
		l_just_computed = 1;
#endif
	} else if (l_type_entry >= eif_first_gen_id) {
			/* This is either an already computed type in the type array. */
		(*intab)++;
		result.id = l_type_entry;
		l_just_computed = 1;
	}

	if (!l_just_computed) {
			/* It's an ordinary id generated by the compiler */

		if (!is_tuple) {
			gcount = par_info(l_type_entry)->nb_generics;
		}

		if (!is_tuple && (gcount == 0)) {
			/* Neither a generic type nor a TUPLE type without annotation.*/
			(*intab)++;

			result.id = l_type_entry;
			if (RT_CONF_IS_DETACHABLE_FLAG(result.annotations)) {
				/* We clear the detachable flag as otherwise we could end up with 2
				* types if one is declared as `A [detachable C]' which will be generated as A [C]
				* and the other as `A [detachable like x]' which will have the detachable mark.
				* See eweasel test#conform001, test#reflection005. */
				result.annotations &= ~DETACHABLE_FLAG;
			}
			**outtab = result;
			(*outtab)++;

			return result;
		}

		save_outtab = *outtab;
		(*intab)++;

			/* Compute types of actual generic parameters and the corresponding
			 * hashcode using the FNV hash (http://www.isthe.com/chongo/tech/comp/fnv/). */
		a_depth++;
		for (hcode = 2166136261, i = gcount; i; --i) {
			l_type = rt_id_of(a_context, intab, outtab, obj_type, a_depth);
			hcode = (hcode * 16777619) ^ eif_encoded_type(l_type);
		}

			/* Only continue if context is valid. */
		if (!a_context || !a_context->is_invalid) {
				/* Search all the generic derivations associated to the base class `l_type_entry' */
			gdp  = eif_derivations [l_type_entry];
			prev = NULL;

			while (gdp != NULL) {
					/* We do not pay attention to the annotation if any at the first level, because they
					 * actually do not control the dynamic type of the result.
					 * This fixes eweasel test#attach069. */
				if ((hcode == gdp->hcode) && (is_expanded == gdp->is_expanded) && (gcount == gdp->size)) {
					mcmp = 0;
					if (gcount > 0) {
						mcmp = memcmp((char*)save_outtab, (char*)(gdp->types),gcount*sizeof(EIF_TYPE_INDEX));
					}
					if (mcmp == 0) {
						break; /* Found */
					}
				}
				prev = gdp;
				gdp  = gdp->next;
			}

			if (gdp == (EIF_GEN_DER *)0) {
					/* Not found: we need a new id */
				if ((gcount == 0) && (!is_tuple)) {
						/* This is a non-generic type which is used with some annotation.
						   First we generate the ID without annotation and then compute the
						   type with the annotation. */
					//CHECK ("has annotation", type_annotation);
					CHECK ("is not tuple", is_tuple == (char) 0);
					if (prev == NULL) {
						prev = rt_new_gen_der (0, NULL, l_type_entry, is_expanded, (char) 0, 0);
						eif_derivations[l_type_entry] = prev;
						if (is_expanded) {
							gdp = prev;
						}
					}
					if (!gdp) {
						gdp = rt_new_gen_der(gcount, save_outtab, l_type_entry, is_expanded, (char) 0, hcode);
					}
				} else {
					gdp = rt_new_gen_der(gcount, save_outtab, l_type_entry, is_expanded, is_tuple, hcode);
					if (prev == NULL) {
						eif_derivations [l_type_entry] = gdp;
					}
				}
				if (prev) {
					prev->next = gdp;
				}
				eif_derivations[gdp->dftype] = gdp; /* Self-reference */
			}

			/* Put full id */
			*outtab = save_outtab;
			result.id = gdp->dftype;
		}
	} else {
			/* Perform some normalization on annotations if type is not NONE.
			 * See eweasel test#exec373.
			 */
		if (result.id <= MAX_DTYPE)
		{
			l_dtype = eif_cid_map[result.id];
			if (EIF_IS_EXPANDED_TYPE(System (l_dtype))) {
					/* Type was computed and is expanded, it does not carry the attachment mark.
					 * See eweasel test#reflection005 when processing A [C [STRING], ANY]. */
				result.annotations &= ~(DETACHABLE_FLAG | ATTACHED_FLAG | FROZEN_FLAG);
			} else if (EIF_IS_FROZEN_TYPE(System (l_dtype))) {
					/* Type is a frozen class, no need for the frozen annotation. */
				result.annotations &= ~FROZEN_FLAG;
			}
		} else {
			CHECK("NONE type", RT_CONF_IS_NONE_TYPE(result.id));
			/* FIXME: If it has the frozen flag, should we clean it? */
		}
	}

	if (RT_CONF_IS_DETACHABLE_FLAG(result.annotations)) {
			/* We clear the detachable flag as otherwise we could end up with 2
			 * types if one is declared as `A [detachable C]' which will be generated as A [C]
			 * and the other as `A [detachable like x]' which will have the detachable mark.
			 * See eweasel test#conform001, test#reflection005. */
		result.annotations &= ~DETACHABLE_FLAG;
	}

	**outtab = result;
	(*outtab)++;

		/* If caller requested the next entry in the `intab' array, we simply return it. */
	if (a_context && a_context->next_address_requested) {
		a_context->next_address = *intab;
	}

	return result;
}
/*------------------------------------------------------------------*/
/* Create a new generic derivation. Actually we create one for every*/
/* type, generic or not.                                            */
/*                                                                  */
/* size     : Nr. of generics in a generic type; 0 otherwise.       */
/* a_types  : Ids of generic paramenters; null pointer if not a     */
/*            generic type                                          */
/* base_id  : Base id of type                                       */
/* is_exp   : Is it expanded?                                       */
/* is_tuple : Is it a tuple?                                        */
/* hcode    : Hash code for faster search                           */
/*------------------------------------------------------------------*/

rt_private EIF_GEN_DER *rt_new_gen_der(uint32 size, EIF_TYPE *a_types, EIF_TYPE_INDEX base_id, char is_exp, char is_tuple, uint32 hcode)
{
	EIF_GEN_DER *result;
	EIF_TYPE *tp;
	EIF_TYPE_INDEX dt;
	const char *cname;
	struct eif_par_types **pt;

	result = (EIF_GEN_DER *) cmalloc(sizeof (EIF_GEN_DER));

	if (result == NULL)
		enomem();

	result->size = size;
	result->hcode = hcode;
	result->gen_seq = NULL;
	result->ptypes = NULL;
	result->is_expanded = is_exp;
	result->is_tuple = is_tuple;
	result->dtype = base_id;
	result->base_dtype = base_id;	/* Type is not generic, there is only one entry in parent table. */
	result->next = NULL;

	if (a_types == NULL) {
			/* Just a simple, compiler generated id */
		CHECK("Not generic type", size == 0);
		result->types = NULL;
		result->dftype = base_id;
		return result;
	} else {
		CHECK("Generic type or tuple", (size > 0) || is_tuple);

		tp = (EIF_TYPE *) cmalloc((size + 1)*sizeof(EIF_TYPE));
		if (!tp) {
			enomem();
		} else {
			tp[size].id = TERMINATOR;
			tp[size].annotations = 0;
			memcpy (tp, a_types, size*sizeof(EIF_TYPE));

			result->types       = tp;
			result->dftype      = eif_next_gen_id++;

				/* Expand tables if necessary */
			if (eif_next_gen_id >= eif_cid_size)
				rt_expand_tables (eif_next_gen_id + 32);

			eif_cid_map [result->dftype] = base_id;

				/* Now find first entry in parent table
				   which has the same class name as `base_id'.
				   We stop when reaching currently found `base_id'.
				*/
			CHECK("Type is generic, base_id > 0", base_id > 0);
			cname = System ((par_info(base_id))->dtype).cn_generator;
			for (dt = 0, pt = eif_par_table2; dt < eif_par_table2_size; ++dt, ++pt) {
				if (*pt && (strcmp (cname,System((*pt)->dtype).cn_generator) == 0)) {
					result->base_dtype = dt;
					break;
				}
			}
				/* Lowest dtype we found should be smaller than the one associated to `dftype'. */
			CHECK("valid base_dtype", result->base_dtype <= result->dtype);
		}
		return result;
	}
}
/*------------------------------------------------------------------*/
/* Create new conformance table.                                    */
/*                                                                  */
/* All ids are full type ids                                        */
/*------------------------------------------------------------------*/

rt_private EIF_CONF_TAB *rt_new_conf_tab(EIF_TYPE_INDEX min_low, EIF_TYPE_INDEX max_low, EIF_TYPE_INDEX min_high, EIF_TYPE_INDEX max_high) {
	EIF_CONF_TAB *result;
	EIF_TYPE_INDEX size;
	unsigned char *tab;

	result = (EIF_CONF_TAB *) cmalloc(sizeof (EIF_CONF_TAB));

	if (result == NULL) {
		enomem();
	} else {
		result->min_low_id = min_low;
		result->max_low_id = max_low;
		result->min_high_id = min_high;
		result->max_high_id = max_high;

		if (min_low <= max_low) {
			size = (max_low - min_low + 8)/8;
			tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
			if (!tab) {
				enomem ();
			} else {
				result->low_tab = tab;

				tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
				if (!tab) {
					eif_rt_xfree (result->low_tab);
					enomem ();
				} else {
					result->low_comp = tab;
#ifdef EIF_ASSERTIONS
					result->low_tab_end = result->low_tab + size;
					result->low_comp_end = result->low_comp + size;
#endif
				}
			}
		} else {
			result->low_tab = NULL;
			result->low_comp = NULL;
#ifdef EIF_ASSERTIONS
			result->low_tab_end = NULL;
			result->low_comp_end = NULL;
#endif
		}


		if (min_high <= max_high) {
			size = (max_high - min_high + 8)/8;
			tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
			if (!tab) {
				enomem ();
			} else  {
				result->high_tab = tab;

				tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
				if (!tab) {
					eif_rt_xfree(result->high_tab);
					enomem ();
				} else {
					result->high_comp = tab;
#ifdef EIF_ASSERTIONS
					result->high_tab_end = result->high_tab + size;
					result->high_comp_end = result->high_comp + size;
#endif
				}
			}
		} else {
			result->high_tab = NULL;
			result->high_comp = NULL;
#ifdef EIF_ASSERTIONS
			result->high_tab_end = NULL;
			result->high_comp_end = NULL;
#endif
		}
	}

	ENSURE("result not null", result);
	return result;
}

/*------------------------------------------------------------------*/
/* Enlarge conformance table to include `new_id'                    */
/*                                                                  */
/* New_id: full type id                                             */
/*------------------------------------------------------------------*/

rt_private void rt_enlarge_conf_tab(EIF_CONF_TAB *table, EIF_TYPE_INDEX new_id)
{
	unsigned char *tab, *comp, *old_tab, *old_comp;
	int offset, is_low;
	EIF_TYPE_INDEX min_old, max_old, min_new, max_new, size, old_size;

	is_low = 0;

	if (new_id < eif_first_gen_id) {
		/* It's a lower id */

		is_low  = 1;
		min_old = min_new = table->min_low_id;
		max_old = max_new = table->max_low_id;

		if (new_id < min_new)
			min_new = new_id - (new_id % 8);    /* alignment */

		if (new_id > max_new)
			max_new = new_id;

		old_tab  = table->low_tab;
		old_comp = table->low_comp;
	} else {
		/* It's a high id */

		min_old = min_new = table->min_high_id;
		max_old = max_new = table->max_high_id;

		if (new_id < min_new)
			min_new = new_id - (new_id % 8);    /* alignment */

		if (new_id > max_new)
			max_new = new_id;

		old_tab  = table->high_tab;
		old_comp = table->high_comp;
	}

	if (min_old <= max_old) {
		old_size = (max_old - min_old + 8)/8;
	} else {
		old_size = 0;
	}

	size = (max_new - min_new + 8)/8;
	tab = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
	if (!tab)
		enomem ();
	comp = (unsigned char *) eif_rt_xcalloc (size, sizeof (unsigned char));
	if (!comp)
		enomem ();

		/* Initialize new tables from old tables */
	if (min_old <= max_old)
	{
		offset = (min_old - min_new) / 8;

		memcpy ((void *)(tab + offset), (void *)old_tab, old_size);
		memcpy ((void *)(comp + offset), (void *)old_comp, old_size);
	}

	/* Free old tables if they were not small (i.e. static) */

	if (old_tab) {
		eif_rt_xfree (old_tab);
	}
	if (old_comp) {
		eif_rt_xfree (old_comp);
	}

	/* Now update structure values */

	if (is_low) {
		table->min_low_id = min_new;
		table->max_low_id = max_new;
		table->low_tab = tab;
		table->low_comp = comp;
#ifdef EIF_ASSERTIONS
		table->low_tab_end = tab + size;
		table->low_comp_end = comp + size;
#endif
	} else {
		table->min_high_id = min_new;
		table->max_high_id = max_new;
		table->high_tab = tab;
		table->high_comp = comp;
#ifdef EIF_ASSERTIONS
		table->high_tab_end = tab + size;
		table->high_comp_end = comp + size;
#endif
	}
}

/*------------------------------------------------------------------*/
/* Expand `eif_cid_map', `eif_conf_tab' , `eif_derivations' and     */
/* to `new_size'                                                    */
/*------------------------------------------------------------------*/

rt_private void rt_expand_tables(int new_size)
{
#ifdef EIF_THREADS
	eif_panic ("Cannot resize Generic conformance tables in multithreaded mode.");
#else
	EIF_GEN_DER **new;
	EIF_CONF_TAB **tab;
	EIF_TYPE_INDEX *map;
	int         i;

	new = (EIF_GEN_DER **) crealloc((char*)eif_derivations, new_size*sizeof (EIF_GEN_DER*));

	if (new == NULL)
		enomem();

	eif_derivations = new;

	tab = (EIF_CONF_TAB **) crealloc((char*)eif_conf_tab, new_size*sizeof (EIF_CONF_TAB*));

	if (tab == NULL)
		enomem();

	eif_conf_tab = tab;

	map = (EIF_TYPE_INDEX *) crealloc((char*)eif_cid_map, new_size*sizeof (EIF_TYPE_INDEX));

	if (map == NULL)
		enomem();

	eif_cid_map = map;

	for (i = eif_cid_size; i < new_size; ++i)
	{
		eif_cid_map [i]     = 0;
		eif_derivations [i] = NULL;
		eif_conf_tab [i]    = NULL;
	}

	eif_cid_size = new_size;
#endif
}
/*------------------------------------------------------------------*/
/* Full type name for type `dftype' as C string.                    */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/
rt_private char *rt_none_name_type = "NONE";

/*
doc:	<routine name="eif_typename_of_type" return_type="EIF_REFERENCE" export="public">
doc:		<summary>Given a type ID with annotations, returns the Eiffel 8-bit string representation of that type.</summary>
doc:		<param name="ftype" type="EIF_TYPE">Dynamic type.</param>
doc:		<return>Returns type name of `dftype'.</return>
doc:		<thread_safety>Not Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</routine>
*/
rt_public EIF_REFERENCE eif_typename_of_type (EIF_TYPE ftype)
{
	char    *name;

	name = eif_typename (ftype);
	return makestr(name, strlen(name));
}

/*
doc:	<routine name="eif_typename_of_type" return_type="EIF_REFERENCE" export="public">
doc:		<summary>Given a type ID with annotations, returns the Eiffel 32-bit string representation of that type.</summary>
doc:		<param name="ftype" type="EIF_TYPE">Dynamic type.</param>
doc:		<return>Returns type name of `dftype'.</return>
doc:		<thread_safety>Not Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</routine>
*/
rt_public EIF_REFERENCE eif_typename_of_type_32 (EIF_TYPE ftype)
{
	EIF_REFERENCE result;

	result = string_32_from_char_8 (eif_typename (ftype));
	if (result == (EIF_REFERENCE) 0) {
		enomem(MTC_NOARG);
	}
	return result;
}

rt_public char *rt_typename_id (EIF_TYPE_INDEX dftype)
{
	return eif_typename_id(dftype);
}

/*
doc:	<routine name="eif_typename" return_type="char *" export="shared">
doc:		<summary>Given a type ID with annotations, returns the string representation of that type. Internally we cache those strings in `eif_type_names'.</summary>
doc:		<param name="dftype" type="EIF_TYPE_INDEX">Dynamic type.</param>
doc:		<return>Returns type name of `dftype'.</return>
doc:		<thread_safety>Not Safe</thread_safety>
doc:		<synchronization>None</synchronization>
doc:	</routine>
*/
rt_public char *eif_typename (EIF_TYPE ftype)
{
	char **l_table_entry;
	char *l_result;
	rt_uint_ptr l_array_index;

	REQUIRE("Valid type", (ftype.id < eif_next_gen_id) || (RT_CONF_IS_NONE_TYPE(ftype.id)));

		/* We use the encoded version of the type to build the search key in `eif_type_names'.
		 * We add +1 because our hash-table implementation does not handle 0 for the key. */
	l_array_index = eif_encoded_type (ftype) + 1;
	l_table_entry = ht_first(&eif_type_names, l_array_index);
	if (!l_table_entry) {
			/* Per documentation, the table is full and we could not find the key `l_array_index'. */
		if (ht_resize (&eif_type_names, eif_type_names.h_capacity + eif_type_names.h_capacity / 2)) {
			enomem();
		} else {
				/* We reiterate the lookup in the resized table. */
			l_table_entry = (EIF_REFERENCE *) ht_first (&eif_type_names, l_array_index);
			CHECK("has_result", l_table_entry);
		}
	}
	CHECK("has_result", l_table_entry);
	if (!*l_table_entry) {
			/* The type name was not yet computed. We compute it and store it immediately
			 * in location pointed by `l_table_entry'. This is the beauty of using `ht_first' as we 
			 * avoid a second lookup. */

		size_t len = rt_typename_len (ftype, 0) + 1;
				/* Create dynamic buffer for string */
		l_result = cmalloc (len);
		if (l_result == NULL) {
			enomem();
		} else {
			*l_result = '\0';
			rt_typename (ftype, l_result, 0);
				/* Store computed result in `eif_type_names'. */
			*l_table_entry = l_result;
		}
	} else {
			/* Get previously computed value. */
		l_result = *l_table_entry;
	}
	return l_result;
}
/*------------------------------------------------------------------*/
/* Produce full type name of `dftype' in `result'.                  */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private void rt_typename (EIF_TYPE a_type, char *result, int a_level)
{
	EIF_GEN_DER *gdp;
	EIF_TYPE *gp;
	EIF_TYPE_INDEX i;
	int	needs_expanded = 0, needs_reference = 0;
	rt_uint_ptr l_array_index;
	char **l_table_entry;

	if (!a_type.annotations) {
			/* This is a type with no explicit annotations, so it is
			 * by default detachable. */
		if (a_level != 0) {
				/* It is also an actual generic parameter, so it is poly too. */ 
			a_type.annotations |= (DETACHABLE_FLAG | POLY_FLAG);
		} else {
			a_type.annotations |= DETACHABLE_FLAG;
		}
	}
	if (a_type.annotations) {
			/* There are some annotations, let's process them. First variance,
			 * and if is RT_CONF_IS_POLY_TYPE then no need to print anything as this
			 * is the default. */
		if (RT_CONF_IS_FROZEN_FLAG(a_type.annotations)) {
			strcat (result, "frozen ");
		} else if (RT_CONF_IS_POLY_FLAG(a_type.annotations)) {
		} else {
			CHECK("has variant or no variance mark", RT_CONF_IS_VARIANT_FLAG(a_type.annotations) || !RT_CONF_HAS_VARIANT_MARK_FLAG(a_type.annotations));
			/* We ignore them for now: 
			strcat (result, "variant ");
			*/
		}

		if (egc_is_experimental) {
				/* If it is detachable and not expanded, we print the detachable mark.
				 * Otherwise nothing as attached is default. */
			if (!RT_CONF_IS_ATTACHED_FLAG(a_type.annotations)) {
				if (!EIF_IS_EXPANDED_TYPE(System (eif_cid_map[a_type.id]))) {
					strcat(result, "detachable ");
				}
			}
		} else {
			if (RT_CONF_IS_ATTACHED_FLAG(a_type.annotations)) {
					/* If it is attached, we print the attached mark. */
				strcat(result, "!");
			}
		}

		if (RT_CONF_IS_SEPARATE_FLAG(a_type.annotations)) {
			strcat (result, "separate ");
		}
	}

	if (a_type.id > MAX_DTYPE) {
		CHECK("NONE type", RT_CONF_IS_NONE_TYPE(a_type.id));
		strcat(result, rt_none_name_type);
	} else {
		needs_expanded = EIF_NEEDS_EXPANDED_KEYWORD(System(eif_cid_map[a_type.id]));
		needs_reference = EIF_NEEDS_REFERENCE_KEYWORD(System(eif_cid_map[a_type.id]));

		if (a_type.id < eif_first_gen_id) {
			if (needs_expanded) {
				strcat (result, "expanded ");
			} else if (needs_reference) {
				strcat (result, "reference ");
			}
				/* Compiler generated id */
			strcat (result, System(par_info(a_type.id)->dtype).cn_generator);
		} else {
				/* We have created this id. First see if we have computed its name yet.
				 * It is important that we do not recursively call `eif_typename' here
				 * as we do not expect the table to resize which could happen if `a_type'
				 * had not yet been computed. */

				/* We use the encoded version of the type to build the search key in `eif_type_names'.
				 * We add +1 because our hash-table implementation does not handle 0 for the key. */
			l_array_index = eif_encoded_type(a_type) + 1;
			l_table_entry = ht_first(&eif_type_names, l_array_index);
			if (l_table_entry && *l_table_entry) {	/* Already computed */
				strcat (result, *l_table_entry);
			} else {
					/* Generic case */
				gdp = eif_derivations [a_type.id];
				i = (EIF_TYPE_INDEX) gdp->size;

				if (needs_expanded) {
					strcat (result, "expanded ");
				} else if (needs_reference) {
					strcat (result, "reference ");
				}

				strcat (result, System(par_info(gdp->dtype)->dtype).cn_generator);

				if (i > 0) {
					strcat (result, " [");
					gp = gdp->types;
					while (i--) {
						rt_typename (*gp, result, a_level + 1);
						++gp;
						if (i) {
							strcat (result, ", ");
						}
					}
					strcat(result, "]");
				}
			}
		}
	}
}
/*------------------------------------------------------------------*/
/* Compute length of string needed for full type name of `dftype'   */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private size_t rt_typename_len (EIF_TYPE a_type, int a_level)
{
	EIF_GEN_DER *gdp;
	EIF_TYPE *gp;
	uint32 i;
	size_t len = 0;
	int	needs_expanded, needs_reference;
	rt_uint_ptr l_array_index;
	char **l_table_entry;

	if (!a_type.annotations) {
			/* This is a type with no explicit annotations, so it is
			 * by default detachable. */
		if (a_level != 0) {
				/* It is also an actual generic parameter, so it is poly too. */ 
			a_type.annotations |= (DETACHABLE_FLAG | POLY_FLAG);
		} else {
			a_type.annotations |= DETACHABLE_FLAG;
		}
	}
	if (a_type.annotations) {
			/* There are some annotations, let's process them. First variance,
			 * and if is RT_CONF_IS_POLY_TYPE then no need to print anything as this
			 * is the default. */
		if (RT_CONF_IS_FROZEN_FLAG(a_type.annotations)) {
			len += 7;	/* for frozen followed by space */
		} else if (RT_CONF_IS_POLY_FLAG(a_type.annotations)) {
		} else {
			CHECK("has variant or no variance mark", RT_CONF_IS_VARIANT_FLAG(a_type.annotations) || !RT_CONF_HAS_VARIANT_MARK_FLAG(a_type.annotations));
			//len += 8;	/* for variant followed by space */
		}

		if (egc_is_experimental) {
				/* If it is detachable and not expanded, we print the detachable mark.
				 * Otherwise nothing as attached is default. */
			if (!RT_CONF_IS_ATTACHED_FLAG(a_type.annotations)) {
				if (!EIF_IS_EXPANDED_TYPE(System (eif_cid_map[a_type.id]))) {
					len += 11;
				}
			}
		} else {
			if (RT_CONF_IS_ATTACHED_FLAG(a_type.annotations)) {
					/* If it is attached, we print the attached mark. */
				len +=1;
			}
		}

		if (RT_CONF_IS_SEPARATE_FLAG(a_type.annotations)) {
			len += 9;	/* for separate followed by space */
		}
	}

	if (a_type.id > MAX_DTYPE) {
		CHECK ("NONE type", RT_CONF_IS_NONE_TYPE(a_type.id));
		len += 4;
	} else {
		needs_expanded = EIF_NEEDS_EXPANDED_KEYWORD(System(eif_cid_map[a_type.id]));
		needs_reference = EIF_NEEDS_REFERENCE_KEYWORD(System(eif_cid_map[a_type.id]));

		if (a_type.id < eif_first_gen_id) {
			if (needs_expanded) {
				len += 9; /* for expanded followed by space */
			} else if (needs_reference) {
				len += 10; /* for reference followed by space */
			}
				/* Compiler generated id */
			len += strlen (System(par_info(a_type.id)->dtype).cn_generator);
		} else {
				/* We have created this id. First see if we have computed its name yet.
				 * It is important that we do not recursively call `eif_typename' here
				 * as we do not expect the table to resize which could happen if `a_type'
				 * had not yet been computed. */

				/* We use the encoded version of the type to build the search key in `eif_type_names'.
				 * We add +1 because our hash-table implementation does not handle 0 for the key. */
			l_array_index = eif_encoded_type(a_type) + 1;
			l_table_entry = ht_first(&eif_type_names, l_array_index);
			if (l_table_entry && *l_table_entry) {	/* Already computed */
				len += strlen (*l_table_entry);
			} else {
					/* Generic case */
				gdp = eif_derivations[a_type.id];
				i = gdp->size;
				len += strlen (System(par_info(gdp->dtype)->dtype).cn_generator);

				if (needs_expanded) {
					len += 9; /* for expanded followed by space */
				} else if (needs_reference) {
					len += 10; /* for reference followed by space */
				}

				if (i > 0) {
						/* Numbers of `[', `]' and `, ' needed in the type specification. */
					len += 3 + (i-1)*2;
					gp = gdp->types;
					while (i--) {
						len += rt_typename_len (*gp, a_level + 1);
						++gp;
					}
				}
			}
		}
	}

	return len;
}
/*------------------------------------------------------------------*/
/* Compute length of generating id sequence for `dftype'            */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private uint16 rt_gen_seq_len (EIF_TYPE a_type)
{
	EIF_GEN_DER *gdp;
	uint32 i;
	uint16 len;

	REQUIRE ("annotations is not an annotation", RT_CONF_IS_VALID_ANNOTATION(a_type.annotations));
	REQUIRE ("dftype is not an annotation", !RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(a_type.id));
	REQUIRE ("dftype is not a formal generic parameter", a_type.id != FORMAL_TYPE);
	REQUIRE ("dftype is not a tuple", a_type.id != TUPLE_TYPE);
	REQUIRE ("dftype is not a terminator", a_type.id != TERMINATOR);

	if (a_type.annotations != 0) {
			/* There is an annotation, we need to make room for it. */
		len = 1;
	} else {
		len = 0;
	}
		/* Simple id */
	if ((RT_CONF_IS_NONE_TYPE(a_type.id)) || (a_type.id < eif_first_gen_id)) {
		len++;
	} else {
			/* It's a generic type */
		gdp = eif_derivations[a_type.id];

			/* Is it a TUPLE? */
		if (gdp->is_tuple) {
				/* Size is TUPLE_OFFSET because we need to take into account
				 * TUPLE_TYPE constant, number of generic parameters
				 * in seqence for tuple type */
			len += TUPLE_OFFSET;
		}

		i = gdp->size;
			/* Add 1 for the base ID. */
		len = len + 1;
		while (i) {
			i--;
			len = len + rt_gen_seq_len (gdp->types [i]);
		}
	}

	return len;
}
/*------------------------------------------------------------------*/
/* Produce generating id sequence for `dftype' in `a_types'.        */
/*                                                                  */
/* dftype    : full type id                                         */
/* a_types   : Base type ids                                        */
/* idx       : index where to put id                                */
/*------------------------------------------------------------------*/

rt_private void rt_put_gen_seq (EIF_TYPE a_type, EIF_TYPE_INDEX *a_types, EIF_TYPE_INDEX *idx, int use_old_annotations)
{
	EIF_GEN_DER *gdp;
	uint32 i, len;

	REQUIRE ("annotations is not an annotation", RT_CONF_IS_VALID_ANNOTATION(a_type.annotations));
	REQUIRE ("dftype is not an annotation", !RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(a_type.id));
	REQUIRE ("dftype is not a formal generic parameter", a_type.id != FORMAL_TYPE);
	REQUIRE ("dftype is not a tuple", a_type.id != TUPLE_TYPE);
	REQUIRE ("dftype is not a terminator", a_type.id != TERMINATOR);

	if (a_type.annotations != 0) {
			/* We add the 0xFF00 because we have the annotations in the type array
			 * which enables us to differentiate them from normal type IDs. */
		a_types [*idx] = a_type.annotations | 0xFF00;
		if (use_old_annotations) {
				/* For backward compatibility for C storable we annotations used
				 * to be 0xFF1x we add the UNUSABLE_FLAG so that old systems can
				 * still retrieve void-safe data generated by a newer version of
				 * the runtime. */
			a_types [*idx] |= UNUSABLE_FLAG;
		}
		(*idx)++;
	}

	/* Simple id */

	if ((RT_CONF_IS_NONE_TYPE(a_type.id)) || (a_type.id < eif_first_gen_id)) {
		a_types [*idx] = a_type.id;
		(*idx)++;
	} else {
			/* It's a generic type */
		gdp = eif_derivations[a_type.id];

			/* Is it a TUPLE type? */
		if (gdp->is_tuple) {
			a_types [*idx] = TUPLE_TYPE;                   /* TUPLE type */
			(*idx)++;
			CHECK("Valid number of generics", rt_valid_type_index(gdp->size));
			a_types [*idx] = (EIF_TYPE_INDEX) (gdp->size);   /* Nr of generics */
			(*idx)++;
		}

		a_types [*idx] = gdp->dtype;
		(*idx)++;

		len = gdp->size;

		for (i = 0; i < len; ++i) {
			rt_put_gen_seq (gdp->types [i], a_types, idx, use_old_annotations);
		}
	}
}

/*------------------------------------------------------------------*/
/* Compute if `ftype' is attached or not.                           */
/*------------------------------------------------------------------*/

rt_public EIF_BOOLEAN eif_is_attached_type2 (EIF_TYPE ftype)
{
		/* A type is attached if it is marked attached or if it is expanded. */
	return EIF_TEST(RT_CONF_IS_ATTACHED_FLAG(ftype.annotations) ||
		( 
			!RT_CONF_IS_NONE_TYPE(ftype.id) &&
			EIF_IS_EXPANDED_TYPE(System(eif_cid_map[ftype.id]))
		)
	);
}

/*------------------------------------------------------------------*/
/* Compute if `dftype' represents a deferred type                   */
/*------------------------------------------------------------------*/

rt_public EIF_BOOLEAN eif_gen_is_deferred (EIF_TYPE_INDEX dftype)
{
	if (RT_CONF_IS_NONE_TYPE(dftype)) {
		return EIF_FALSE;
	} else {
		return EIF_TEST(EIF_IS_DEFERRED_TYPE(System(eif_cid_map[dftype])));
	}
}

/*------------------------------------------------------------------*/
/* Compute if `dftype' has a default value, i.e. detachable         */
/* reference type or expanded type.                                 */
/*------------------------------------------------------------------*/

rt_public EIF_BOOLEAN eif_gen_has_default (EIF_TYPE ftype)
{
	if (!RT_CONF_IS_ATTACHED_FLAG(ftype.annotations)) {
		return EIF_TRUE;
	} else {
		return eif_gen_is_expanded(ftype.id);
	}
}

/*------------------------------------------------------------------*/
/* Compute if `dftype' represents an expanded type                  */
/*------------------------------------------------------------------*/

rt_public EIF_BOOLEAN eif_gen_is_expanded (EIF_TYPE_INDEX dftype)
{
	if (RT_CONF_IS_NONE_TYPE(dftype)) {
		return EIF_FALSE;
	} else {
		EIF_GEN_DER *gdp = eif_derivations [dftype];
		if (gdp) {
			return EIF_TEST(gdp->is_expanded);
		} else {
				/* It is not a generic derivation, we can avoid the conversion dftype -> dtype. */
			CHECK("Same as dtype", eif_cid_map[dftype] == dftype);
			return EIF_TEST(EIF_IS_EXPANDED_TYPE(System(dftype)));
		}
	}
}

/*------------------------------------------------------------------*/
/* Compute the associated detachable type of `ftype' if any,       */
/* otherwise `ftype'.                                              */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE eif_non_attached_type2 (EIF_TYPE ftype)
{
		/* Since types are by default detachable, we simply remove
		 * all attachment marks. */
	ftype.annotations &= ~(ATTACHED_FLAG | DETACHABLE_FLAG);
	return ftype;
}

/*------------------------------------------------------------------*/
/* Compute the associated attached type of `ftype' if any,         */
/* otherwise `ftype'.                                              */
/*------------------------------------------------------------------*/

rt_public EIF_TYPE eif_attached_type2 (EIF_TYPE ftype)
{
	if (RT_CONF_IS_NONE_TYPE (ftype.id) || !EIF_IS_EXPANDED_TYPE(System(eif_cid_map[ftype.id]))) {
		ftype.annotations &= ~DETACHABLE_FLAG;
		ftype.annotations |= ATTACHED_FLAG;
	} else {
		CHECK("No marks", !RT_CONF_HAS_ATTACHMENT_MARK_FLAG(ftype.annotations));
	}
	return ftype;
}

/*------------------------------------------------------------------*/
/* Compute conformance table for `dftype'                           */
/*                                                                  */
/* dftype : full type id                                            */
/*------------------------------------------------------------------*/

rt_private void rt_compute_ctab (EIF_TYPE_INDEX dftype)
{
	EIF_TYPE outtab [256], *outtable;
	EIF_TYPE_INDEX *intable;
	const EIF_TYPE_INDEX *l_intable;
	EIF_TYPE_INDEX min_low, max_low, min_high, max_high, pftype, dtype, *ptypes;
	int i, count, offset, pcount;
	unsigned char *src, *dest, *src_comp, *dest_comp, mask;
	char is_expanded;
	struct eif_par_types *pt;
	EIF_CONF_TAB *ctab, *pctab;
	EIF_GEN_DER *gdp;

		/* Get parent table */
	dtype = To_dtype(dftype);
	gdp = eif_derivations [dftype];

		/* `gdp' might be NULL in the case of non-generic classes, thus we build a very simple entry. */
	if (gdp == NULL) {
		CHECK("same type", dftype == dtype);
		if (EIF_IS_EXPANDED_TYPE(System(dtype))) {
			gdp = rt_new_gen_der (0, NULL, dtype, '1', (char) 0, 0);
		} else {
			gdp = rt_new_gen_der (0, NULL, dtype, (char) 0, (char) 0, 0);
		}
		eif_derivations [dftype] = gdp;
	}

	CHECK ("gdp not null", gdp);

		/* Compiler generated id */
	pt = par_info (dtype);
	CHECK ("Parents never NULL", pt->parents);
	is_expanded = pt->is_expanded;

		/* Let's compute the number of parent types. */
	pcount = pt->nb_parents;
	intable = pt->parents;

				/* Compute the ranges of the bit tables */
	min_low = eif_next_gen_id;
	max_low = 0;
	min_high = eif_next_gen_id;
	max_high = 0;

		/* Type conforms to itself */
	if (dftype < eif_first_gen_id) {
		min_low = max_low = dftype;
	} else {
		min_high = max_high = dftype;
	}

		/* Create table of parent types (+1 for TERMINATOR) */
	ptypes = (EIF_TYPE_INDEX *) cmalloc (sizeof (EIF_TYPE_INDEX) * (pcount + 1));
	if (ptypes == NULL)
		enomem ();

	gdp->ptypes = ptypes;
	if (pcount) {
		outtable = outtab;
		l_intable = intable;
		while (*l_intable != TERMINATOR) {
			pftype = rt_id_of (NULL, &l_intable, &outtable, dftype, 0).id;
			if (*l_intable == PARENT_TYPE_SEPARATOR) {
				l_intable++;
			} else {
				CHECK("Terminator expected", *l_intable == TERMINATOR);
			}

				/* Register parent type */
			*(ptypes++) = pftype;
		}
		*ptypes = TERMINATOR;

		ptypes = gdp->ptypes;
		while (*ptypes != TERMINATOR) {
			pftype = *ptypes++;
			ctab = eif_conf_tab [pftype];

			if (ctab == NULL) {
				rt_compute_ctab (pftype);
				ctab = eif_conf_tab [pftype];
			}

			if (ctab->min_low_id < min_low)
				min_low = ctab->min_low_id;
			if (ctab->max_low_id > max_low)
				max_low = ctab->max_low_id;
			if (ctab->min_high_id < min_high)
				min_high = ctab->min_high_id;
			if (ctab->max_high_id > max_high)
				max_high = ctab->max_high_id;
		}
	} else {
		*ptypes = TERMINATOR;
	}

		/* Create a new table */
		/* Make sure that the min values are == 0 mod 8 */
	min_low  -= (min_low % 8);
	min_high -= (min_high % 8);

	ctab = rt_new_conf_tab (min_low, max_low, min_high, max_high);

	eif_conf_tab [dftype] = ctab;

		/* Fill bit tables */
	if (pcount) {
		ptypes = gdp->ptypes;

		while (*ptypes != TERMINATOR) {
			pftype = *ptypes++;
			pctab = eif_conf_tab [pftype];

			if ((min_low <= max_low) && (pctab->min_low_id <= pctab->max_low_id)) {
				count  = (pctab->max_low_id-pctab->min_low_id+8)/8;
				offset = (pctab->min_low_id - min_low)/8;
				src  = pctab->low_tab;
				dest = ctab->low_tab + offset;
				src_comp = pctab->low_comp;
				dest_comp = ctab->low_comp + offset;

				for (i = count; i; --i) {
					/* We conform to everything our parent
					   conforms to */

					*dest |= *src;

					/* Consider only those bits as already
					   computed for which conformance holds
					   because we may conform to something
					   to which the parent does not! */

					*(dest_comp) |= ((*src) & (*src_comp));
					++dest;
					++src;
					++src_comp;
					++dest_comp;
				}
				CHECK ("valid_src", src <= pctab->low_tab_end);
				CHECK ("valid_src_comp", src_comp <= pctab->low_comp_end);
				CHECK ("valid_dest", dest <= ctab->low_tab_end);
				CHECK ("valid_dest_comp", dest_comp <= ctab->low_comp_end);
			}

			if ((min_high <= max_high) && (pctab->min_high_id <= pctab->max_high_id)) {
				count  = (pctab->max_high_id-pctab->min_high_id+8)/8;
				offset = (pctab->min_high_id - min_high)/8;
				src  = pctab->high_tab;
				dest = ctab->high_tab + offset;
				src_comp = pctab->high_comp;
				dest_comp = ctab->high_comp + offset;

				for (i = count; i; --i) {
					/* We conform to everything our parent
					   conforms to */
					*dest |= *src;

					/* Consider only those bits as already
					   computed for which conformance holds
					   because we may conform to something
					   to which the parent does not! */

					*(dest_comp) |= ((*src) & (*src_comp));
					++dest;
					++src;
					++src_comp;
					++dest_comp;
				}
				CHECK ("valid_src", src <= pctab->high_tab_end);
				CHECK ("valid_src_comp", src_comp <= pctab->high_comp_end);
				CHECK ("valid_dest", dest <= ctab->high_tab_end);
				CHECK ("valid_dest_comp", dest_comp <= ctab->high_comp_end);
			}
		}
	}

	/* Put own type in table if it's not expanded */

	if (is_expanded)
		return;

	if (dftype < eif_first_gen_id) {
		offset = (dftype - min_low);
		mask   = (char) (1 << (offset % 8));
		CHECK("valid low_tab index", (ctab->low_tab + (offset / 8)) < ctab->low_tab_end);
		(ctab->low_tab)[offset/8] |= mask;
		CHECK("valid low_comp index", (ctab->low_comp + (offset / 8)) < ctab->low_comp_end);
		(ctab->low_comp)[offset/8] |= mask;
	} else {
		offset = (dftype - min_high);
		mask   = (char) (1 << (offset % 8));
		CHECK("valid high_tab index", (ctab->high_tab + (offset / 8)) < ctab->high_tab_end);
		(ctab->high_tab)[offset/8] |= mask;
		CHECK("valid high_comp index", (ctab->high_comp + (offset / 8)) < ctab->high_comp_end);
		(ctab->high_comp)[offset/8] |= mask;
	}
}

/*
doc:</file>
*/
