note
	description: "[
		Object representing The Field table
		Conceptually, each row in the Field table is owned by one, and only one, row in the TypeDef table
	]"
	date: "$Date: 2024-03-19 18:44:23 +0000 (Tue, 19 Mar 2024) $"
	revision: "$Revision: 107698 $"
	EIS: "name=Field Table", "src=https://www.ecma-international.org/wp-content/uploads/ECMA-335_6th_edition_june_2012.pdf#page=249&zoom=100,116,324", "protocol=uri"

class
	PE_FIELD_TABLE_ENTRY

inherit
	PE_TABLE_ENTRY_BASE
		redefine
			same_as
		end

	DEBUG_OUTPUT

create
	make_with_data

feature {NONE} -- Initialization

	make_with_data (a_flags: INTEGER_32; a_name_index: NATURAL_32; a_signature_index: NATURAL_32)
		do
			flags := a_flags.to_integer_16
			check no_truncation: flags.to_integer_32 = a_flags end

			create name_index.make_with_index (a_name_index)
			create signature_index.make_with_index (a_signature_index)
		end

feature -- Status

	same_as (e: like Current): BOOLEAN
			-- Is `e` same as `Current`?
			-- note: used to detect if an entry is already recorded.
			--| Flags.CompilerControlled = 1 then this row is ignored completely in duplicate checking
			--| There shall be no duplicate rows in the Field table, based upon  owner+Name+Signature (where owner is the owning row in the TypeDef table, as
			--| described above) (Note however that if Flags.CompilerControlled = 1, then this
			--| row is completely excluded from duplicate checking)
		do
			Result := Precursor (e)
				or else (
					e.flags /= 1 and then
					e.flags = flags and then
					e.name_index.is_equal (name_index) and then
					e.signature_index.is_equal (signature_index)
				)
		end

feature -- Access

	flags: INTEGER_16
			-- a 2-byte bitmask of type FieldAttributes
			-- see https://www.ecma-international.org/wp-content/uploads/ECMA-335_6th_edition_june_2012.pdf#page=276&zoom=100,116,708

	name_index: PE_STRING
			-- an index into the String heap.

	signature_index: PE_BLOB
			-- an index into the Blob heap.

feature -- Status report	

	debug_output: STRING
			-- String that should be displayed in debugger to represent `Current'.
		do
			Result := "{Field} "
			Result := Result + " name[" + name_index.debug_output + "]"
			Result := Result + " signature[" + signature_index.debug_output + "]"
		end

feature -- Enum: Flags

	FieldAccessMask: INTEGER = 0x0007

	PrivateScope: INTEGER = 0x0000
	Private: INTEGER = 0x0001
	FamANDAssem: INTEGER = 0x0002
	Assembly: INTEGER = 0x0003
	Family: INTEGER = 0x0004
	FamORAssem: INTEGER = 0x0005
	Public: INTEGER = 0x0006

		-- other attribs
	Static: INTEGER = 0x0010
	InitOnly: INTEGER = 0x0020
	Literal: INTEGER = 0x0040
	NotSerialized: INTEGER = 0x0080
	SpecialName: INTEGER = 0x0200

		-- pinvoke
	PinvokeImpl: INTEGER = 0x2000

		-- runtime
	ReservedMask: INTEGER = 0x9500
	RTSpecialName: INTEGER = 0x0400
	HasFieldMarshal: INTEGER = 0x1000
	HasDefault: INTEGER = 0x8000
	HasFieldRVA: INTEGER = 0x0100

feature -- Operations

	table_index: NATURAL_32
		once
			Result := {PE_TABLES}.tfield
		end

	render (a_sizes: SPECIAL [NATURAL_32]; a_dest: ARRAY [NATURAL_8]): NATURAL_32
			-- <Precursor>
		local
			l_bytes: NATURAL_32
		do
				-- Write the flags to the destination buffer `a_dest`.
			{BYTE_ARRAY_HELPER}.put_integer_16 (a_dest, flags, 0)

				-- Initialize the number of bytes written
			l_bytes := 2

				-- Write the name_index, signature_index
				-- to the buffer and update the number of bytes.
			l_bytes := l_bytes + name_index.render (a_sizes, a_dest, l_bytes)
			l_bytes := l_bytes + signature_index.render (a_sizes, a_dest, l_bytes)

				-- Return the total number of bytes written.
			Result := l_bytes
		end

	rendering_size (a_sizes: SPECIAL [NATURAL_32]): NATURAL_32
		local
			l_bytes: NATURAL_32
		do
				-- Initialize the number of bytes readed.
			l_bytes := 2 -- flags

				-- Get the name_index, signature_index and
				-- update the number of bytes.

			l_bytes := l_bytes + name_index.rendering_size (a_sizes)
			l_bytes := l_bytes + signature_index.rendering_size (a_sizes)

				-- Return the number of bytes readed.
			Result := l_bytes
		end

end
