#include <cstdlib>
#include <sstream>
#include <getopt.h>
#include "sentence.hpp"
#include "models.hpp"

using namespace std;
using namespace maeda;

void PrintUsageAndDie(ostream &out) {
  out << "Usage: eda [Options]" << endl;
  out << "Options:" << endl;
  out << "  -i, --in-format FORMAT   input format of test data(eda/conll/kytea/score)" << endl;
  out << "  -o, --out-format FORMAT  output format of parse trees(eda/conll)" << endl;
  out << "  -m, --model FILE         model file for analysis" << endl;
  out << "      --labeling           label edges" << endl;
  out << "  -n, --named-entity       solution search without breaking given named entities" << endl;
  exit(1);
}


int main(int argc, char *argv[]) {
  string model_file;   // ǥե

  StreamReader reader = ReadFromEdaDependencyStream;

  StreamWriter writer = WriteToEdaDependencyStream;

  bool labeling_p = false;

  bool score_output_p = false;

  bool named_entity_p = false;

  while (1) {
    enum Opt {
      OPT_IN_FORMAT = 1, OPT_OUT_FORMAT, OPT_MODEL, OPT_LABELING,
      OPT_NAMED_ENTITY,
    };
    static struct option longopts [] = {
      {"in-format"        , required_argument, 0, OPT_IN_FORMAT},
      {"out-format"       , required_argument, 0, OPT_OUT_FORMAT},
      {"model"            , required_argument, 0, OPT_MODEL},
      {"labeling"         , no_argument      , 0, OPT_LABELING},
      {"named-entity"     , no_argument      , 0, OPT_NAMED_ENTITY},
      {0, 0, 0, 0},
    };
    const char *shortopts = "i:o:m:n";

    int option_index = 0;

    int c = getopt_long_only(argc, argv, shortopts, longopts, &option_index);

    if (c == -1) {
      break;
    } else {
      switch (c) {
      case 'i':
      case OPT_IN_FORMAT:
        if (string(optarg) == "eda") {
          reader = ReadFromEdaDependencyStream;
        } else if (string(optarg) == "conll") {
          reader = ReadFromCoNLLDependencyStream;
        } else if (string(optarg) == "kytea") {
          reader = ReadFromKyteaOutputStream;
        } else {
          cerr << "Error: invalid format: " << optarg << endl;
          exit(1);
        }
        break;
      case 'o':
      case OPT_OUT_FORMAT:
        if (string(optarg) == "eda") {
          writer = WriteToEdaDependencyStream;
        } else if (string(optarg) == "conll") {
          writer = WriteToCoNLLDependencyStream;
        } else if (string(optarg) == "score") {
          writer = NULL;
          score_output_p = true;
        } else {
          cerr << "Error: invalid format: " << optarg << endl;
          exit(1);
        }
        break;
      case 'm':
      case OPT_MODEL:
        if (optarg == NULL) {
          cerr << "Error: model filename expected" << endl;
          exit(1);
        } else {
          model_file = string(optarg);
        }
        break;
      case 'n':
      case OPT_NAMED_ENTITY:
        named_entity_p = true;
        break;
      case OPT_LABELING:
        labeling_p = true;
        break;
      default:
        PrintUsageAndDie(cout);
      }
    }
  }

  if (named_entity_p && writer != WriteToEdaDependencyStream) {
    cerr << "Warning: -n is only supported for eda output" << endl;
    named_entity_p = false;
  }


  if (model_file == "") {
    PrintUsageAndDie(cout);
    return 0;
  }

  if (labeling_p) {
    if (reader == ReadFromCoNLLDependencyStream
        && writer == WriteToCoNLLDependencyStream) {
      EdgeLabelModel model;
      model.Load(model_file);

      while (1) {
        Sentence sentence;
        if (!reader(cin, &sentence)) {
          break;
        }
        model.LabelEdges(&sentence);
        writer(cout, sentence);
      }
    } else {
      cerr << "Error: Labeling is only supported for conll type i/o"
           << endl;
      exit(1);
    }
    // ǰΤ
    return 0;
  }

  // parse sentences
  UnlabeledParseModel model;
  model.Load(model_file);

  int sentence_count = 0;
  while (1) {
    Sentence sentence;

    if (!reader(cin, &sentence)) {
      break;
    }

    sentence_count++;
    if (sentence.id.empty()) {
      stringstream ss;
      ss << sentence_count;
      sentence.id = ss.str();
    }

    if (score_output_p) {
      cout << "ID=" << sentence.id << endl;
      model.PrintArcScores(cout, sentence);
    } else {
      model.ParseSentenceWithMST(named_entity_p, &sentence);
      writer(cout, sentence);
    }
  }
  return 0;
}
