/*
 * ecgen, tool for generating Elliptic curve domain parameters
 * Copyright (C) 2017-2018 J08nY
 */
#include "poly.h"

// HP range [2, 159] + [2001, 10000]
/**
 * Table of Low-Weight Binary Irreducible Polynomials
 * Gadiel Seroussi
 * Computer Systems Laboratory
 * HPL-98-135
 * August, 1998
 *
 * https://www.hpl.hp.com/techreports/98/HPL-98-135.pdf
 */
static polynomial_t hp_trinomials[] = {
    {2, 1},       {3, 1},       {4, 1},       {5, 2},       {6, 1},
    {7, 1},       {9, 1},       {10, 3},      {11, 2},      {12, 3},
    {14, 5},      {15, 1},      {17, 3},      {18, 3},      {20, 3},
    {21, 2},      {22, 1},      {23, 5},      {25, 3},      {28, 1},
    {29, 2},      {30, 1},      {31, 3},      {33, 10},     {34, 7},
    {35, 2},      {36, 9},      {39, 4},      {41, 3},      {42, 7},
    {44, 5},      {46, 1},      {47, 5},      {49, 9},      {52, 3},
    {54, 9},      {55, 7},      {57, 4},      {58, 19},     {60, 1},
    {62, 29},     {63, 1},      {65, 18},     {66, 3},      {68, 9},
    {71, 6},      {73, 25},     {74, 35},     {76, 21},     {79, 9},
    {81, 4},      {84, 5},      {86, 21},     {87, 13},     {89, 38},
    {90, 27},     {92, 21},     {93, 2},      {94, 21},     {95, 11},
    {97, 6},      {98, 11},     {100, 15},    {102, 29},    {103, 9},
    {105, 4},     {106, 15},    {108, 17},    {110, 33},    {111, 10},
    {113, 9},     {118, 33},    {119, 8},     {121, 18},    {123, 2},
    {124, 19},    {126, 21},    {127, 1},     {129, 5},     {130, 3},
    {132, 17},    {134, 57},    {135, 11},    {137, 21},    {140, 15},
    {142, 21},    {145, 52},    {146, 71},    {147, 14},    {148, 27},
    {150, 53},    {151, 3},     {153, 1},     {154, 15},    {155, 62},
    {156, 9},     {159, 31},    {2001, 169},  {2004, 441},  {2006, 917},
    {2007, 205},  {2009, 54},   {2010, 459},  {2015, 42},   {2017, 330},
    {2020, 81},   {2022, 349},  {2023, 165},  {2025, 274},  {2026, 475},
    {2028, 93},   {2031, 386},  {2033, 881},  {2034, 143},  {2036, 71},
    {2039, 155},  {2041, 735},  {2044, 45},   {2047, 3},    {2049, 124},
    {2052, 323},  {2054, 201},  {2055, 11},   {2057, 245},  {2058, 343},
    {2060, 387},  {2063, 48},   {2065, 97},   {2066, 71},   {2070, 237},
    {2073, 253},  {2074, 231},  {2076, 851},  {2079, 35},   {2081, 467},
    {2082, 523},  {2086, 261},  {2087, 141},  {2089, 150},  {2094, 645},
    {2095, 256},  {2097, 119},  {2098, 19},   {2100, 35},   {2102, 33},
    {2103, 98},   {2105, 153},  {2106, 111},  {2111, 249},  {2113, 385},
    {2114, 155},  {2116, 25},   {2118, 385},  {2119, 84},   {2121, 304},
    {2122, 91},   {2124, 45},   {2126, 881},  {2127, 539},  {2129, 21},
    {2130, 239},  {2132, 213},  {2135, 47},   {2137, 331},  {2140, 283},
    {2142, 69},   {2143, 345},  {2145, 19},   {2146, 595},  {2148, 549},
    {2150, 569},  {2151, 224},  {2153, 582},  {2156, 405},  {2158, 93},
    {2159, 6},    {2161, 766},  {2162, 47},   {2164, 561},  {2166, 693},
    {2167, 840},  {2169, 55},   {2170, 411},  {2174, 225},  {2175, 128},
    {2177, 554},  {2178, 15},   {2180, 111},  {2182, 93},   {2183, 162},
    {2185, 51},   {2186, 51},   {2188, 99},   {2190, 441},  {2191, 111},
    {2193, 71},   {2196, 539},  {2198, 893},  {2199, 49},   {2201, 143},
    {2206, 793},  {2207, 438},  {2209, 142},  {2210, 539},  {2212, 423},
    {2214, 1041}, {2215, 39},   {2217, 455},  {2218, 603},  {2220, 7},
    {2222, 333},  {2225, 47},   {2228, 425},  {2230, 637},  {2231, 654},
    {2233, 249},  {2236, 63},   {2238, 1053}, {2239, 120},  {2241, 20},
    {2242, 7},    {2244, 399},  {2247, 217},  {2249, 126},  {2250, 927},
    {2252, 75},   {2255, 729},  {2257, 829},  {2258, 983},  {2262, 57},
    {2263, 273},  {2265, 151},  {2266, 343},  {2268, 115},  {2270, 369},
    {2271, 560},  {2273, 630},  {2274, 239},  {2276, 21},   {2279, 276},
    {2281, 715},  {2282, 975},  {2284, 889},  {2286, 249},  {2287, 651},
    {2289, 136},  {2292, 89},   {2295, 259},  {2297, 405},  {2300, 95},
    {2303, 80},   {2305, 424},  {2306, 551},  {2308, 31},   {2310, 233},
    {2311, 148},  {2313, 221},  {2314, 879},  {2316, 21},   {2318, 245},
    {2319, 161},  {2321, 543},  {2322, 83},   {2324, 717},  {2327, 32},
    {2329, 105},  {2331, 14},   {2332, 349},  {2334, 1125}, {2335, 553},
    {2337, 523},  {2338, 211},  {2340, 39},   {2342, 65},   {2343, 415},
    {2345, 29},   {2346, 987},  {2348, 731},  {2351, 950},  {2353, 328},
    {2356, 183},  {2358, 161},  {2359, 172},  {2361, 646},  {2364, 643},
    {2367, 610},  {2369, 77},   {2372, 1139}, {2375, 198},  {2377, 381},
    {2378, 243},  {2380, 1},    {2382, 429},  {2383, 49},   {2385, 607},
    {2388, 11},   {2390, 629},  {2391, 956},  {2393, 59},   {2394, 423},
    {2396, 173},  {2399, 107},  {2401, 61},   {2402, 251},  {2404, 67},
    {2407, 91},   {2409, 1198}, {2410, 807},  {2412, 25},   {2414, 29},
    {2415, 154},  {2417, 225},  {2418, 311},  {2420, 77},   {2422, 1117},
    {2423, 102},  {2425, 678},  {2428, 301},  {2430, 477},  {2431, 303},
    {2433, 305},  {2434, 507},  {2436, 145},  {2438, 929},  {2439, 404},
    {2441, 339},  {2442, 127},  {2444, 1115}, {2447, 786},  {2449, 621},
    {2450, 191},  {2452, 331},  {2454, 357},  {2455, 313},  {2457, 238},
    {2460, 35},   {2463, 1172}, {2465, 531},  {2466, 599},  {2468, 99},
    {2470, 217},  {2473, 225},  {2474, 899},  {2478, 765},  {2479, 72},
    {2481, 710},  {2484, 523},  {2485, 142},  {2487, 155},  {2489, 315},
    {2492, 141},  {2495, 497},  {2497, 1171}, {2500, 135},  {2502, 45},
    {2503, 316},  {2505, 131},  {2508, 25},   {2510, 1113}, {2511, 110},
    {2513, 99},   {2514, 183},  {2516, 563},  {2519, 579},  {2521, 426},
    {2526, 141},  {2527, 640},  {2529, 49},   {2534, 185},  {2537, 77},
    {2538, 315},  {2540, 209},  {2542, 97},   {2543, 240},  {2545, 982},
    {2546, 891},  {2548, 373},  {2550, 333},  {2551, 103},  {2553, 28},
    {2554, 1123}, {2556, 349},  {2559, 23},   {2561, 201},  {2562, 203},
    {2564, 561},  {2566, 37},   {2567, 122},  {2569, 69},   {2572, 535},
    {2574, 5},    {2575, 867},  {2577, 674},  {2580, 105},  {2583, 31},
    {2585, 263},  {2586, 1047}, {2590, 1017}, {2591, 219},  {2593, 297},
    {2594, 863},  {2596, 145},  {2598, 225},  {2599, 289},  {2601, 406},
    {2604, 435},  {2606, 1181}, {2607, 34},   {2609, 425},  {2610, 427},
    {2614, 553},  {2615, 518},  {2617, 462},  {2618, 71},   {2620, 835},
    {2623, 409},  {2625, 112},  {2626, 43},   {2628, 47},   {2630, 177},
    {2631, 139},  {2633, 1241}, {2639, 144},  {2641, 736},  {2642, 551},
    {2644, 597},  {2646, 297},  {2647, 513},  {2649, 689},  {2652, 519},
    {2655, 53},   {2657, 242},  {2660, 5},    {2663, 458},  {2665, 772},
    {2666, 663},  {2667, 254},  {2668, 819},  {2670, 229},  {2671, 46},
    {2673, 530},  {2674, 967},  {2676, 93},   {2679, 286},  {2681, 635},
    {2682, 463},  {2686, 789},  {2687, 225},  {2689, 36},   {2692, 577},
    {2694, 621},  {2695, 123},  {2697, 170},  {2698, 963},  {2700, 3},
    {2702, 257},  {2703, 67},   {2706, 515},  {2708, 423},  {2711, 690},
    {2713, 840},  {2716, 255},  {2718, 369},  {2719, 102},  {2721, 826},
    {2722, 127},  {2724, 121},  {2727, 430},  {2729, 96},   {2730, 343},
    {2732, 845},  {2735, 933},  {2737, 226},  {2738, 923},  {2740, 109},
    {2742, 149},  {2743, 447},  {2745, 484},  {2748, 25},   {2750, 629},
    {2751, 49},   {2753, 716},  {2754, 231},  {2756, 159},  {2759, 842},
    {2761, 108},  {2762, 1319}, {2764, 687},  {2766, 1285}, {2767, 102},
    {2769, 269},  {2770, 567},  {2772, 135},  {2775, 802},  {2778, 1095},
    {2780, 51},   {2783, 168},  {2785, 349},  {2786, 339},  {2790, 837},
    {2791, 490},  {2793, 343},  {2796, 769},  {2799, 880},  {2801, 279},
    {2804, 609},  {2806, 729},  {2807, 270},  {2809, 1342}, {2812, 453},
    {2814, 621},  {2815, 84},   {2817, 109},  {2820, 815},  {2823, 592},
    {2825, 288},  {2826, 135},  {2828, 1103}, {2831, 186},  {2833, 409},
    {2836, 1113}, {2839, 1033}, {2841, 370},  {2842, 1231}, {2844, 25},
    {2847, 329},  {2849, 114},  {2850, 1411}, {2852, 1145}, {2854, 313},
    {2855, 41},   {2857, 756},  {2860, 603},  {2862, 405},  {2863, 139},
    {2865, 212},  {2868, 915},  {2871, 272},  {2873, 75},   {2876, 605},
    {2878, 781},  {2879, 149},  {2881, 1201}, {2882, 1431}, {2884, 529},
    {2887, 469},  {2889, 76},   {2890, 31},   {2892, 309},  {2895, 358},
    {2897, 15},   {2898, 91},   {2900, 303},  {2903, 279},  {2905, 321},
    {2906, 1155}, {2910, 1301}, {2911, 685},  {2913, 238},  {2914, 351},
    {2916, 21},   {2918, 237},  {2919, 149},  {2921, 480},  {2922, 559},
    {2927, 974},  {2929, 651},  {2934, 713},  {2937, 172},  {2938, 499},
    {2940, 49},   {2942, 1425}, {2943, 320},  {2945, 146},  {2946, 551},
    {2950, 397},  {2951, 872},  {2953, 33},   {2956, 823},  {2959, 69},
    {2961, 86},   {2962, 319},  {2964, 83},   {2966, 861},  {2967, 1028},
    {2969, 561},  {2970, 583},  {2972, 693},  {2975, 192},  {2977, 126},
    {2978, 375},  {2980, 381},  {2982, 669},  {2983, 330},  {2985, 166},
    {2986, 343},  {2988, 313},  {2991, 292},  {2993, 569},  {2994, 303},
    {2996, 345},  {2998, 669},  {2999, 1011}, {3001, 975},  {3004, 351},
    {3007, 963},  {3009, 1349}, {3012, 1327}, {3015, 308},  {3017, 108},
    {3018, 203},  {3020, 413},  {3023, 734},  {3025, 757},  {3028, 135},
    {3031, 55},   {3033, 238},  {3034, 399},  {3036, 391},  {3038, 633},
    {3039, 436},  {3041, 776},  {3042, 415},  {3044, 69},   {3046, 1021},
    {3049, 765},  {3050, 651},  {3052, 363},  {3057, 110},  {3058, 811},
    {3060, 405},  {3062, 1053}, {3063, 32},   {3065, 432},  {3066, 455},
    {3068, 215},  {3071, 65},   {3073, 184},  {3076, 475},  {3078, 105},
    {3079, 174},  {3081, 64},   {3084, 109},  {3086, 1281}, {3087, 49},
    {3089, 261},  {3090, 279},  {3092, 45},   {3094, 769},  {3095, 419},
    {3097, 1162}, {3100, 45},   {3102, 225},  {3103, 124},  {3105, 833},
    {3108, 61},   {3110, 1421}, {3111, 199},  {3113, 191},  {3116, 461},
    {3118, 525},  {3119, 315},  {3121, 493},  {3124, 861},  {3126, 449},
    {3127, 139},  {3129, 23},   {3130, 867},  {3132, 123},  {3134, 89},
    {3135, 356},  {3137, 587},  {3140, 1115}, {3142, 981},  {3143, 8},
    {3145, 112},  {3148, 1171}, {3150, 253},  {3151, 1254}, {3153, 98},
    {3156, 565},  {3159, 103},  {3161, 858},  {3162, 315},  {3164, 113},
    {3167, 672},  {3169, 1123}, {3170, 783},  {3172, 301},  {3174, 81},
    {3175, 646},  {3177, 484},  {3178, 915},  {3180, 1085}, {3182, 1205},
    {3183, 1225}, {3185, 204},  {3186, 891},  {3188, 129},  {3191, 495},
    {3193, 211},  {3194, 1059}, {3196, 175},  {3198, 841},  {3199, 54},
    {3201, 674},  {3204, 31},   {3207, 704},  {3209, 81},   {3210, 1303},
    {3212, 1559}, {3214, 1197}, {3215, 614},  {3217, 67},   {3220, 19},
    {3222, 145},  {3223, 784},  {3225, 101},  {3228, 1225}, {3230, 501},
    {3233, 575},  {3234, 511},  {3236, 887},  {3238, 409},  {3239, 98},
    {3241, 127},  {3244, 1249}, {3246, 1221}, {3247, 426},  {3249, 149},
    {3252, 567},  {3254, 1485}, {3255, 124},  {3257, 806},  {3258, 203},
    {3260, 237},  {3263, 939},  {3268, 73},   {3270, 237},  {3271, 333},
    {3273, 1408}, {3274, 775},  {3276, 69},   {3279, 446},  {3281, 47},
    {3282, 783},  {3286, 397},  {3287, 717},  {3289, 43},   {3292, 61},
    {3294, 249},  {3295, 594},  {3297, 7},    {3298, 639},  {3300, 55},
    {3302, 605},  {3303, 1336}, {3305, 806},  {3306, 127},  {3308, 717},
    {3310, 1},    {3311, 618},  {3313, 436},  {3314, 1019}, {3316, 1641},
    {3318, 585},  {3319, 58},   {3321, 20},   {3322, 567},  {3324, 173},
    {3326, 1145}, {3327, 875},  {3329, 525},  {3330, 191},  {3332, 587},
    {3335, 636},  {3337, 370},  {3338, 1155}, {3343, 73},   {3345, 796},
    {3348, 177},  {3350, 1401}, {3351, 731},  {3353, 389},  {3356, 339},
    {3359, 99},   {3364, 85},   {3366, 257},  {3367, 136},  {3369, 1541},
    {3372, 47},   {3374, 417},  {3375, 49},   {3377, 236},  {3378, 623},
    {3380, 659},  {3382, 217},  {3383, 956},  {3385, 603},  {3388, 169},
    {3390, 1381}, {3391, 465},  {3393, 1615}, {3398, 245},  {3399, 416},
    {3401, 531},  {3402, 387},  {3404, 173},  {3407, 507},  {3409, 244},
    {3410, 1023}, {3412, 325},  {3414, 93},   {3415, 1272}, {3417, 32},
    {3418, 15},   {3420, 423},  {3422, 1121}, {3423, 11},   {3425, 189},
    {3426, 1071}, {3430, 153},  {3431, 153},  {3436, 159},  {3438, 393},
    {3439, 147},  {3441, 394},  {3444, 69},   {3447, 404},  {3449, 917},
    {3452, 1145}, {3455, 21},   {3457, 120},  {3458, 519},  {3460, 1495},
    {3462, 225},  {3463, 289},  {3465, 304},  {3466, 43},   {3468, 921},
    {3470, 917},  {3471, 314},  {3473, 720},  {3474, 735},  {3476, 525},
    {3478, 465},  {3479, 155},  {3481, 546},  {3484, 1329}, {3486, 1085},
    {3487, 120},  {3489, 518},  {3492, 57},   {3495, 254},  {3497, 1025},
    {3498, 567},  {3500, 375},  {3503, 993},  {3505, 103},  {3510, 81},
    {3511, 1141}, {3513, 41},   {3516, 667},  {3519, 569},  {3521, 129},
    {3522, 399},  {3524, 1439}, {3527, 476},  {3529, 270},  {3532, 1561},
    {3534, 973},  {3535, 162},  {3537, 218},  {3540, 75},   {3542, 345},
    {3543, 377},  {3545, 998},  {3546, 151},  {3548, 255},  {3550, 1269},
    {3551, 183},  {3556, 127},  {3558, 397},  {3559, 69},   {3561, 257},
    {3562, 927},  {3564, 225},  {3569, 1028}, {3570, 699},  {3572, 1143},
    {3574, 889},  {3575, 339},  {3577, 348},  {3580, 915},  {3582, 713},
    {3583, 747},  {3585, 7},    {3588, 843},  {3590, 1713}, {3591, 509},
    {3593, 72},   {3594, 59},   {3596, 383},  {3599, 114},  {3601, 669},
    {3604, 637},  {3606, 861},  {3607, 142},  {3609, 1016}, {3612, 215},
    {3614, 29},   {3615, 47},   {3617, 377},  {3618, 1539}, {3620, 231},
    {3622, 481},  {3625, 279},  {3628, 957},  {3630, 729},  {3631, 90},
    {3633, 553},  {3634, 651},  {3636, 391},  {3639, 76},   {3641, 1626},
    {3642, 771},  {3644, 1365}, {3647, 45},   {3649, 394},  {3650, 1691},
    {3652, 721},  {3654, 273},  {3655, 112},  {3657, 928},  {3658, 1471},
    {3660, 61},   {3662, 1365}, {3663, 130},  {3665, 189},  {3668, 269},
    {3671, 101},  {3673, 544},  {3676, 609},  {3678, 501},  {3679, 21},
    {3681, 115},  {3682, 471},  {3684, 81},   {3686, 81},   {3687, 889},
    {3689, 759},  {3690, 839},  {3694, 1129}, {3695, 62},   {3697, 91},
    {3698, 1719}, {3700, 675},  {3702, 1281}, {3703, 429},  {3705, 148},
    {3706, 1195}, {3708, 147},  {3710, 797},  {3711, 1735}, {3713, 413},
    {3714, 459},  {3719, 488},  {3721, 31},   {3726, 609},  {3727, 42},
    {3729, 184},  {3730, 1191}, {3732, 1327}, {3734, 1305}, {3735, 46},
    {3737, 287},  {3738, 75},   {3740, 95},   {3743, 279},  {3745, 684},
    {3750, 1013}, {3751, 435},  {3753, 407},  {3754, 1611}, {3756, 291},
    {3759, 208},  {3761, 30},   {3762, 383},  {3764, 1307}, {3767, 672},
    {3769, 300},  {3770, 107},  {3772, 61},   {3775, 1416}, {3777, 1414},
    {3780, 63},   {3782, 1785}, {3783, 272},  {3785, 87},   {3786, 1027},
    {3788, 1173}, {3791, 45},   {3793, 481},  {3796, 127},  {3798, 1337},
    {3799, 202},  {3801, 112},  {3804, 349},  {3807, 68},   {3809, 938},
    {3810, 323},  {3812, 1799}, {3815, 143},  {3817, 252},  {3822, 29},
    {3823, 609},  {3825, 437},  {3828, 1217}, {3830, 713},  {3831, 310},
    {3833, 35},   {3834, 567},  {3836, 681},  {3838, 273},  {3839, 503},
    {3841, 840},  {3842, 1331}, {3844, 1063}, {3846, 693},  {3847, 108},
    {3849, 71},   {3850, 583},  {3852, 169},  {3854, 765},  {3855, 1399},
    {3857, 50},   {3858, 459},  {3860, 35},   {3863, 834},  {3865, 289},
    {3866, 315},  {3870, 913},  {3871, 264},  {3873, 32},   {3876, 157},
    {3879, 121},  {3881, 810},  {3882, 1775}, {3884, 45},   {3886, 273},
    {3887, 915},  {3889, 340},  {3892, 289},  {3894, 1197}, {3895, 777},
    {3897, 310},  {3900, 65},   {3902, 1845}, {3903, 350},  {3905, 26},
    {3906, 251},  {3908, 855},  {3911, 1673}, {3913, 393},  {3914, 531},
    {3916, 445},  {3918, 117},  {3919, 285},  {3921, 785},  {3924, 245},
    {3927, 367},  {3929, 1440}, {3930, 199},  {3932, 1563}, {3937, 252},
    {3938, 1835}, {3942, 57},   {3943, 1125}, {3945, 427},  {3946, 1155},
    {3948, 293},  {3950, 873},  {3951, 752},  {3953, 698},  {3954, 503},
    {3956, 429},  {3959, 891},  {3961, 756},  {3962, 255},  {3964, 735},
    {3966, 337},  {3967, 357},  {3969, 196},  {3970, 163},  {3972, 595},
    {3974, 861},  {3975, 322},  {3977, 221},  {3983, 11},   {3985, 1038},
    {3988, 1017}, {3990, 469},  {3991, 168},  {3993, 1468}, {3996, 19},
    {3998, 153},  {3999, 1250}, {4001, 137},  {4004, 1479}, {4007, 705},
    {4009, 124},  {4014, 125},  {4015, 249},  {4017, 22},   {4018, 1467},
    {4020, 375},  {4023, 985},  {4025, 599},  {4030, 93},   {4031, 1805},
    {4033, 223},  {4034, 1163}, {4036, 157},  {4038, 953},  {4039, 1408},
    {4041, 410},  {4044, 1659}, {4046, 981},  {4047, 158},  {4049, 215},
    {4050, 71},   {4052, 17},   {4054, 981},  {4055, 854},  {4057, 871},
    {4058, 419},  {4060, 435},  {4062, 765},  {4063, 118},  {4065, 356},
    {4066, 847},  {4068, 825},  {4070, 1529}, {4071, 661},  {4073, 575},
    {4074, 595},  {4079, 224},  {4081, 78},   {4084, 1435}, {4086, 1445},
    {4087, 769},  {4089, 463},  {4090, 79},   {4092, 1491}, {4094, 321},
    {4095, 616},  {4097, 1232}, {4098, 3},    {4100, 615},  {4102, 57},
    {4103, 1278}, {4105, 252},  {4106, 123},  {4108, 1615}, {4111, 201},
    {4113, 913},  {4114, 219},  {4116, 245},  {4119, 1015}, {4122, 595},
    {4124, 345},  {4125, 2},    {4126, 133},  {4127, 276},  {4129, 1315},
    {4132, 873},  {4134, 1045}, {4135, 1554}, {4137, 862},  {4138, 1899},
    {4140, 549},  {4142, 525},  {4143, 421},  {4145, 51},   {4146, 131},
    {4148, 1857}, {4150, 573},  {4151, 186},  {4154, 231},  {4156, 1165},
    {4158, 133},  {4159, 81},   {4161, 1361}, {4162, 2047}, {4164, 2045},
    {4166, 645},  {4167, 698},  {4169, 105},  {4170, 387},  {4174, 1297},
    {4175, 1698}, {4177, 805},  {4180, 273},  {4182, 1985}, {4183, 1984},
    {4185, 341},  {4186, 1411}, {4188, 947},  {4191, 454},  {4193, 200},
    {4196, 1565}, {4199, 225},  {4201, 76},   {4202, 1031}, {4204, 1327},
    {4206, 381},  {4207, 1846}, {4209, 826},  {4210, 1491}, {4212, 243},
    {4214, 473},  {4215, 34},   {4217, 347},  {4218, 287},  {4220, 525},
    {4223, 144},  {4225, 9},    {4228, 145},  {4230, 533},  {4231, 756},
    {4233, 5},    {4234, 799},  {4236, 459},  {4238, 389},  {4239, 904},
    {4241, 273},  {4242, 503},  {4244, 387},  {4246, 273},  {4247, 783},
    {4249, 387},  {4255, 754},  {4257, 578},  {4260, 99},   {4262, 1301},
    {4263, 217},  {4265, 362},  {4266, 727},  {4268, 147},  {4271, 1775},
    {4273, 385},  {4274, 1859}, {4276, 1471}, {4278, 309},  {4279, 945},
    {4281, 226},  {4282, 1671}, {4284, 81},   {4287, 679},  {4289, 689},
    {4290, 99},   {4295, 302},  {4300, 175},  {4302, 569},  {4303, 84},
    {4305, 116},  {4308, 901},  {4311, 455},  {4313, 1016}, {4314, 1071},
    {4316, 431},  {4318, 973},  {4319, 1850}, {4321, 390},  {4322, 111},
    {4324, 987},  {4326, 485},  {4327, 540},  {4329, 833},  {4330, 1351},
    {4332, 593},  {4335, 826},  {4337, 1991}, {4338, 287},  {4340, 725},
    {4342, 1477}, {4343, 678},  {4345, 271},  {4346, 1911}, {4348, 429},
    {4350, 301},  {4351, 900},  {4353, 1042}, {4356, 667},  {4358, 645},
    {4359, 595},  {4361, 272},  {4362, 539},  {4364, 1523}, {4366, 1773},
    {4367, 197},  {4369, 154},  {4372, 151},  {4374, 729},  {4375, 253},
    {4377, 155},  {4378, 1939}, {4380, 745},  {4382, 1541}, {4383, 886},
    {4386, 175},  {4390, 1029}, {4391, 84},   {4394, 1535}, {4396, 925},
    {4398, 933},  {4399, 585},  {4401, 394},  {4402, 351},  {4404, 543},
    {4406, 137},  {4409, 218},  {4410, 211},  {4412, 1769}, {4415, 1269},
    {4417, 442},  {4420, 865},  {4422, 1069}, {4423, 271},  {4425, 533},
    {4428, 63},   {4431, 47},   {4433, 350},  {4434, 207},  {4436, 273},
    {4439, 869},  {4441, 909},  {4442, 1187}, {4444, 1839}, {4445, 254},
    {4446, 589},  {4447, 21},   {4449, 644},  {4450, 1603}, {4452, 905},
    {4454, 621},  {4455, 1189}, {4457, 329},  {4458, 255},  {4460, 155},
    {4463, 582},  {4465, 351},  {4466, 639},  {4468, 1621}, {4470, 2117},
    {4471, 471},  {4473, 112},  {4474, 799},  {4476, 595},  {4479, 1165},
    {4481, 273},  {4482, 919},  {4484, 975},  {4486, 1749}, {4487, 843},
    {4489, 193},  {4490, 1907}, {4492, 895},  {4494, 401},  {4495, 1},
    {4497, 299},  {4500, 5},    {4502, 657},  {4505, 794},  {4508, 285},
    {4511, 173},  {4513, 196},  {4514, 323},  {4516, 715},  {4518, 577},
    {4519, 1191}, {4521, 650},  {4522, 1327}, {4524, 1339}, {4526, 849},
    {4527, 674},  {4529, 345},  {4530, 151},  {4534, 381},  {4535, 993},
    {4537, 21},   {4538, 2223}, {4540, 715},  {4542, 897},  {4543, 358},
    {4545, 523},  {4548, 831},  {4550, 297},  {4551, 730},  {4553, 704},
    {4554, 207},  {4557, 98},   {4558, 1869}, {4559, 1356}, {4561, 835},
    {4562, 87},   {4564, 165},  {4566, 905},  {4567, 126},  {4569, 2110},
    {4570, 2079}, {4572, 81},   {4574, 69},   {4575, 638},  {4578, 1419},
    {4583, 1358}, {4585, 223},  {4586, 483},  {4588, 265},  {4590, 497},
    {4591, 1254}, {4596, 371},  {4599, 25},   {4601, 68},   {4602, 675},
    {4604, 1907}, {4606, 2013}, {4607, 1026}, {4609, 780},  {4610, 239},
    {4612, 1557}, {4614, 965},  {4615, 232},  {4617, 287},  {4618, 975},
    {4620, 77},   {4623, 136},  {4625, 42},   {4626, 367},  {4628, 1779},
    {4633, 1035}, {4634, 1535}, {4636, 901},  {4639, 448},  {4641, 149},
    {4642, 1311}, {4644, 189},  {4647, 959},  {4649, 207},  {4650, 567},
    {4652, 95},   {4655, 474},  {4657, 417},  {4658, 1943}, {4660, 885},
    {4662, 297},  {4663, 841},  {4665, 1663}, {4668, 917},  {4671, 392},
    {4673, 147},  {4676, 603},  {4678, 2245}, {4679, 1146}, {4681, 216},
    {4682, 119},  {4684, 819},  {4686, 285},  {4687, 1330}, {4689, 421},
    {4690, 283},  {4692, 21},   {4694, 1725}, {4695, 1522}, {4697, 386},
    {4698, 351},  {4700, 873},  {4703, 248},  {4705, 1033}, {4706, 2151},
    {4708, 1543}, {4710, 2257}, {4711, 426},  {4713, 67},   {4716, 97},
    {4719, 466},  {4724, 617},  {4726, 1693}, {4727, 408},  {4729, 73},
    {4732, 157},  {4734, 2193}, {4735, 412},  {4737, 248},  {4740, 1941},
    {4742, 2201}, {4743, 857},  {4745, 281},  {4746, 363},  {4748, 1257},
    {4750, 541},  {4751, 1946}, {4753, 2368}, {4754, 923},  {4756, 279},
    {4758, 401},  {4759, 883},  {4761, 170},  {4764, 477},  {4767, 178},
    {4769, 119},  {4770, 507},  {4772, 213},  {4775, 2381}, {4777, 348},
    {4780, 321},  {4782, 1397}, {4783, 369},  {4785, 382},  {4788, 9},
    {4790, 1233}, {4791, 707},  {4793, 413},  {4798, 117},  {4799, 158},
    {4801, 1146}, {4802, 291},  {4805, 1922}, {4806, 2349}, {4807, 12},
    {4809, 91},   {4810, 1171}, {4815, 638},  {4817, 491},  {4818, 459},
    {4820, 515},  {4823, 1094}, {4825, 558},  {4828, 913},  {4830, 1493},
    {4831, 337},  {4833, 458},  {4836, 301},  {4839, 238},  {4846, 1477},
    {4847, 1085}, {4849, 184},  {4850, 243},  {4852, 561},  {4854, 633},
    {4855, 172},  {4857, 34},   {4858, 355},  {4860, 81},   {4863, 166},
    {4865, 1551}, {4868, 593},  {4870, 213},  {4871, 1251}, {4873, 1798},
    {4874, 555},  {4876, 585},  {4878, 1457}, {4879, 1342}, {4881, 1276},
    {4884, 91},   {4886, 1989}, {4887, 139},  {4889, 45},   {4890, 1967},
    {4892, 2381}, {4894, 469},  {4895, 623},  {4897, 210},  {4900, 697},
    {4902, 1253}, {4903, 774},  {4905, 296},  {4908, 111},  {4910, 377},
    {4911, 1955}, {4913, 693},  {4914, 259},  {4916, 1793}, {4918, 889},
    {4919, 380},  {4921, 277},  {4922, 2387}, {4924, 1545}, {4926, 693},
    {4927, 21},   {4929, 49},   {4930, 2143}, {4932, 1039}, {4934, 413},
    {4935, 133},  {4937, 63},   {4940, 383},  {4943, 494},  {4945, 1011},
    {4948, 601},  {4950, 517},  {4951, 217},  {4953, 938},  {4956, 559},
    {4958, 1709}, {4959, 920},  {4961, 843},  {4966, 2205}, {4967, 1292},
    {4969, 1084}, {4970, 1131}, {4972, 709},  {4974, 1229}, {4975, 2301},
    {4977, 644},  {4978, 315},  {4980, 111},  {4982, 1701}, {4983, 1070},
    {4985, 384},  {4986, 239},  {4988, 375},  {4991, 201},  {4993, 39},
    {4994, 1791}, {4998, 161},  {5001, 637},  {5004, 671},  {5007, 464},
    {5009, 38},   {5012, 33},   {5014, 1197}, {5015, 206},  {5017, 385},
    {5018, 731},  {5020, 97},   {5022, 329},  {5023, 777},  {5025, 1379},
    {5028, 1089}, {5030, 2373}, {5031, 496},  {5033, 1122}, {5034, 387},
    {5036, 1295}, {5038, 45},   {5039, 489},  {5041, 646},  {5042, 1607},
    {5044, 1165}, {5047, 415},  {5049, 133},  {5050, 2467}, {5052, 833},
    {5054, 1493}, {5055, 211},  {5057, 126},  {5058, 1411}, {5060, 161},
    {5062, 141},  {5063, 795},  {5065, 126},  {5066, 783},  {5068, 1635},
    {5071, 258},  {5073, 476},  {5074, 2119}, {5076, 1467}, {5078, 2201},
    {5079, 35},   {5081, 1562}, {5082, 35},   {5084, 173},  {5087, 210},
    {5089, 88},   {5090, 147},  {5092, 843},  {5094, 1377}, {5095, 489},
    {5097, 454},  {5098, 243},  {5100, 675},  {5103, 148},  {5105, 72},
    {5106, 1307}, {5108, 191},  {5110, 2209}, {5111, 279},  {5113, 559},
    {5114, 899},  {5116, 1387}, {5118, 2045}, {5119, 130},  {5121, 457},
    {5122, 199},  {5124, 279},  {5127, 1519}, {5129, 1514}, {5130, 603},
    {5132, 15},   {5134, 1629}, {5135, 141},  {5137, 1701}, {5138, 1571},
    {5140, 1015}, {5143, 1767}, {5145, 308},  {5146, 1771}, {5148, 103},
    {5150, 213},  {5151, 724},  {5153, 708},  {5159, 494},  {5161, 315},
    {5162, 419},  {5166, 117},  {5167, 2193}, {5169, 1196}, {5170, 1411},
    {5172, 2247}, {5174, 189},  {5175, 347},  {5177, 963},  {5180, 2055},
    {5183, 1869}, {5185, 373},  {5186, 1755}, {5188, 2305}, {5190, 677},
    {5191, 330},  {5193, 1006}, {5196, 617},  {5199, 1546}, {5201, 125},
    {5202, 1519}, {5204, 917},  {5206, 465},  {5207, 588},  {5209, 156},
    {5210, 2219}, {5212, 487},  {5215, 354},  {5217, 1507}, {5218, 2335},
    {5220, 205},  {5222, 1637}, {5223, 1787}, {5225, 503},  {5226, 2607},
    {5228, 327},  {5231, 1188}, {5233, 205},  {5234, 275},  {5236, 475},
    {5238, 765},  {5239, 549},  {5241, 101},  {5242, 895},  {5244, 2321},
    {5246, 2477}, {5247, 235},  {5249, 197},  {5250, 1371}, {5252, 627},
    {5254, 1305}, {5255, 167},  {5257, 1537}, {5258, 1359}, {5260, 1455},
    {5263, 769},  {5265, 41},   {5266, 483},  {5268, 297},  {5270, 341},
    {5271, 1106}, {5273, 1395}, {5276, 609},  {5278, 2337}, {5279, 1299},
    {5287, 196},  {5289, 1424}, {5292, 315},  {5295, 1133}, {5297, 446},
    {5298, 575},  {5300, 2195}, {5302, 2205}, {5303, 434},  {5305, 1863},
    {5310, 837},  {5311, 142},  {5313, 598},  {5314, 7},    {5316, 1067},
    {5318, 125},  {5319, 1573}, {5321, 351},  {5322, 1199}, {5324, 15},
    {5326, 1545}, {5327, 723},  {5329, 1219}, {5330, 803},  {5332, 171},
    {5334, 545},  {5337, 2279}, {5338, 1647}, {5340, 731},  {5343, 485},
    {5345, 339},  {5346, 171},  {5348, 2259}, {5351, 707},  {5353, 1419},
    {5356, 51},   {5358, 2485}, {5359, 81},   {5361, 35},   {5364, 863},
    {5367, 194},  {5369, 152},  {5370, 1919}, {5372, 1151}, {5375, 188},
    {5377, 547},  {5378, 1059}, {5380, 423},  {5383, 1509}, {5385, 376},
    {5386, 2055}, {5388, 2261}, {5390, 2409}, {5391, 434},  {5393, 215},
    {5394, 1523}, {5396, 2255}, {5398, 357},  {5399, 485},  {5401, 420},
    {5404, 387},  {5406, 1205}, {5407, 955},  {5409, 1720}, {5410, 2343},
    {5412, 917},  {5415, 331},  {5417, 1256}, {5418, 303},  {5420, 945},
    {5422, 445},  {5423, 311},  {5425, 186},  {5428, 117},  {5430, 513},
    {5431, 2163}, {5433, 791},  {5436, 1229}, {5438, 1289}, {5439, 203},
    {5441, 1254}, {5442, 1635}, {5444, 1235}, {5446, 865},  {5447, 860},
    {5449, 103},  {5450, 479},  {5452, 2691}, {5454, 893},  {5455, 91},
    {5457, 767},  {5458, 1839}, {5460, 29},   {5463, 167},  {5465, 54},
    {5468, 95},   {5471, 198},  {5473, 589},  {5474, 1247}, {5478, 305},
    {5479, 370},  {5481, 167},  {5484, 1029}, {5487, 1834}, {5489, 1290},
    {5490, 1143}, {5492, 135},  {5494, 537},  {5495, 279},  {5497, 1729},
    {5500, 175},  {5502, 2537}, {5503, 237},  {5505, 98},   {5508, 729},
    {5511, 554},  {5513, 1814}, {5514, 535},  {5516, 1563}, {5519, 773},
    {5521, 1209}, {5524, 2113}, {5526, 1505}, {5527, 318},  {5529, 1756},
    {5530, 1467}, {5532, 1017}, {5534, 1425}, {5535, 217},  {5537, 492},
    {5540, 1089}, {5542, 957},  {5543, 248},  {5545, 298},  {5548, 403},
    {5550, 1273}, {5551, 712},  {5553, 116},  {5556, 269},  {5558, 1301},
    {5559, 938},  {5561, 773},  {5562, 995},  {5564, 705},  {5566, 2185},
    {5567, 239},  {5569, 4},    {5572, 1053}, {5574, 901},  {5575, 699},
    {5577, 328},  {5580, 81},   {5582, 1697}, {5583, 1037}, {5585, 1338},
    {5586, 371},  {5588, 39},   {5590, 1381}, {5591, 870},  {5593, 124},
    {5594, 15},   {5596, 853},  {5598, 101},  {5599, 565},  {5601, 487},
    {5602, 963},  {5604, 2251}, {5606, 1905}, {5607, 43},   {5609, 287},
    {5610, 271},  {5612, 135},  {5615, 461},  {5617, 6},    {5620, 1251},
    {5622, 2465}, {5623, 709},  {5625, 28},   {5626, 943},  {5628, 863},
    {5630, 357},  {5631, 1502}, {5633, 143},  {5634, 759},  {5636, 555},
    {5639, 1365}, {5641, 568},  {5644, 2233}, {5649, 49},   {5650, 967},
    {5652, 929},  {5654, 149},  {5655, 434},  {5657, 165},  {5660, 39},
    {5662, 2637}, {5663, 90},   {5665, 714},  {5666, 1215}, {5670, 41},
    {5671, 37},   {5673, 82},   {5676, 1259}, {5679, 271},  {5681, 326},
    {5684, 87},   {5687, 2007}, {5689, 1384}, {5690, 1623}, {5692, 561},
    {5694, 1609}, {5695, 732},  {5700, 705},  {5703, 82},   {5705, 248},
    {5706, 2199}, {5708, 1151}, {5710, 409},  {5711, 441},  {5713, 27},
    {5714, 1775}, {5718, 1361}, {5719, 375},  {5721, 1562}, {5724, 103},
    {5726, 309},  {5727, 1310}, {5729, 515},  {5732, 2267}, {5734, 2629},
    {5735, 744},  {5737, 507},  {5740, 115},  {5742, 1085}, {5743, 1621},
    {5745, 544},  {5746, 279},  {5750, 377},  {5751, 127},  {5753, 1721},
    {5754, 1235}, {5756, 371},  {5758, 805},  {5759, 1431}, {5761, 615},
    {5762, 2759}, {5764, 2019}, {5767, 124},  {5769, 83},   {5772, 343},
    {5775, 266},  {5777, 134},  {5778, 1071}, {5780, 1535}, {5782, 2377},
    {5783, 954},  {5785, 1161}, {5788, 1585}, {5790, 197},  {5791, 2101},
    {5793, 437},  {5794, 2383}, {5796, 831},  {5798, 1853}, {5799, 934},
    {5801, 129},  {5802, 1015}, {5806, 369},  {5807, 731},  {5809, 369},
    {5810, 47},   {5812, 295},  {5814, 2233}, {5815, 1083}, {5817, 343},
    {5818, 763},  {5819, 1058}, {5820, 61},   {5822, 2145}, {5823, 916},
    {5825, 323},  {5826, 2027}, {5831, 275},  {5833, 264},  {5834, 935},
    {5836, 91},   {5838, 153},  {5839, 169},  {5841, 250},  {5844, 1809},
    {5847, 1013}, {5849, 105},  {5850, 27},   {5854, 2217}, {5855, 1334},
    {5857, 516},  {5858, 2735}, {5860, 1899}, {5863, 58},   {5865, 1249},
    {5868, 373},  {5871, 287},  {5873, 113},  {5874, 267},  {5876, 243},
    {5879, 2556}, {5881, 81},   {5882, 975},  {5886, 297},  {5887, 142},
    {5889, 541},  {5892, 63},   {5895, 781},  {5897, 390},  {5898, 247},
    {5900, 125},  {5902, 1981}, {5903, 380},  {5905, 172},  {5908, 2101},
    {5910, 449},  {5911, 2545}, {5913, 292},  {5919, 1843}, {5921, 311},
    {5922, 495},  {5924, 1017}, {5926, 325},  {5927, 1305}, {5932, 2673},
    {5935, 84},   {5937, 49},   {5940, 99},   {5942, 245},  {5943, 1075},
    {5948, 117},  {5951, 216},  {5953, 1767}, {5954, 95},   {5956, 385},
    {5958, 2297}, {5959, 1864}, {5961, 2081}, {5962, 2599}, {5964, 1475},
    {5966, 609},  {5967, 2359}, {5969, 24},   {5970, 399},  {5972, 315},
    {5975, 284},  {5977, 657},  {5978, 1151}, {5980, 1951}, {5982, 45},
    {5983, 1960}, {5985, 203},  {5986, 895},  {5988, 1067}, {5990, 2729},
    {5991, 680},  {5993, 1133}, {5998, 1765}, {5999, 362},  {6001, 648},
    {6002, 2895}, {6004, 2287}, {6006, 1025}, {6007, 435},  {6009, 346},
    {6010, 2907}, {6012, 533},  {6015, 91},   {6017, 261},  {6018, 1315},
    {6020, 645},  {6022, 2217}, {6023, 143},  {6025, 586},  {6026, 915},
    {6027, 98},   {6030, 1377}, {6031, 613},  {6033, 406},  {6034, 439},
    {6036, 1577}, {6039, 1556}, {6041, 1476}, {6044, 2525}, {6047, 411},
    {6049, 988},  {6050, 1227}, {6052, 127},  {6055, 1242}, {6057, 2378},
    {6058, 1155}, {6060, 243},  {6065, 129},  {6066, 67},   {6068, 1149},
    {6070, 37},   {6071, 789},  {6073, 1753}, {6074, 35},   {6076, 1387},
    {6078, 1241}, {6079, 382},  {6081, 1081}, {6084, 279},  {6086, 557},
    {6087, 1948}, {6089, 1062}, {6090, 11},   {6092, 1475}, {6094, 973},
    {6095, 2186}, {6097, 346},  {6098, 1395}, {6100, 577},  {6103, 376},
    {6105, 823},  {6106, 2079}, {6108, 213},  {6111, 467},  {6113, 1283},
    {6116, 873},  {6118, 1765}, {6119, 287},  {6121, 490},  {6124, 19},
    {6129, 1421}, {6132, 135},  {6134, 2445}, {6135, 572},  {6137, 1682},
    {6140, 135},  {6142, 2245}, {6143, 1005}, {6145, 348},  {6146, 71},
    {6150, 1297}, {6151, 538},  {6153, 77},   {6156, 25},   {6159, 1165},
    {6161, 1556}, {6162, 603},  {6164, 1179}, {6167, 33},   {6169, 1204},
    {6170, 1179}, {6172, 3},    {6174, 1029}, {6175, 823},  {6177, 430},
    {6178, 1719}, {6180, 245},  {6182, 1773}, {6183, 56},   {6185, 308},
    {6188, 1223}, {6190, 877},  {6191, 1074}, {6193, 919},  {6196, 1807},
    {6198, 213},  {6199, 109},  {6201, 922},  {6202, 867},  {6204, 691},
    {6207, 1363}, {6209, 935},  {6210, 711},  {6214, 913},  {6215, 134},
    {6217, 711},  {6218, 383},  {6220, 889},  {6222, 693},  {6223, 1033},
    {6225, 317},  {6226, 1035}, {6228, 2691}, {6230, 1317}, {6231, 359},
    {6233, 777},  {6236, 1701}, {6238, 2353}, {6239, 921},  {6241, 592},
    {6244, 475},  {6246, 1565}, {6247, 586},  {6249, 1000}, {6250, 363},
    {6255, 2318}, {6257, 852},  {6258, 647},  {6260, 1601}, {6262, 261},
    {6263, 1466}, {6265, 502},  {6268, 973},  {6270, 737},  {6271, 54},
    {6273, 1106}, {6274, 999},  {6276, 1787}, {6278, 749},  {6279, 290},
    {6281, 2021}, {6282, 1895}, {6286, 741},  {6287, 104},  {6289, 237},
    {6290, 1127}, {6292, 2317}, {6294, 57},   {6295, 82},   {6297, 190},
    {6300, 7},    {6302, 2105}, {6303, 467},  {6305, 1052}, {6306, 2079},
    {6308, 405},  {6311, 1155}, {6313, 355},  {6316, 21},   {6318, 333},
    {6319, 1213}, {6321, 1},    {6322, 387},  {6324, 1273}, {6326, 1029},
    {6330, 623},  {6332, 2781}, {6334, 1977}, {6335, 368},  {6337, 2086},
    {6338, 459},  {6340, 1137}, {6342, 1309}, {6343, 939},  {6345, 2566},
    {6348, 75},   {6351, 1234}, {6353, 1025}, {6354, 1155}, {6356, 1115},
    {6358, 229},  {6359, 305},  {6362, 1967}, {6364, 67},   {6366, 273},
    {6367, 142},  {6370, 1147}, {6372, 135},  {6375, 313},  {6377, 275},
    {6378, 2643}, {6380, 1085}, {6382, 2061}, {6383, 1013}, {6385, 364},
    {6386, 543},  {6388, 2731}, {6390, 581},  {6391, 324},  {6393, 103},
    {6394, 1447}, {6396, 91},   {6399, 146},  {6401, 1146}, {6402, 947},
    {6404, 1779}, {6406, 613},  {6407, 231},  {6409, 72},   {6412, 1131},
    {6415, 663},  {6417, 19},   {6418, 1095}, {6420, 903},  {6422, 621},
    {6425, 231},  {6426, 207},  {6428, 791},  {6430, 1333}, {6431, 3},
    {6433, 810},  {6434, 651},  {6436, 1141}, {6438, 425},  {6439, 630},
    {6441, 41},   {6444, 1647}, {6446, 2633}, {6447, 1547}, {6449, 777},
    {6450, 1707}, {6452, 381},  {6454, 825},  {6455, 732},  {6457, 840},
    {6458, 119},  {6460, 2073}, {6463, 93},   {6465, 224},  {6468, 217},
    {6470, 477},  {6471, 1910}, {6473, 1211}, {6474, 563},  {6476, 1535},
    {6478, 213},  {6479, 84},   {6481, 538},  {6484, 43},   {6487, 880},
    {6489, 706},  {6490, 111},  {6492, 1169}, {6495, 283},  {6497, 1736},
    {6498, 2079}, {6500, 375},  {6502, 465},  {6503, 330},  {6505, 1911},
    {6510, 61},   {6511, 2673}, {6513, 682},  {6514, 2571}, {6518, 513},
    {6519, 725},  {6521, 695},  {6522, 675},  {6524, 1283}, {6526, 2137},
    {6527, 242},  {6529, 765},  {6534, 45},   {6535, 31},   {6537, 542},
    {6538, 31},   {6540, 333},  {6545, 1322}, {6546, 279},  {6548, 945},
    {6550, 2401}, {6551, 941},  {6553, 265},  {6554, 159},  {6556, 885},
    {6558, 153},  {6559, 879},  {6561, 1834}, {6562, 651},  {6564, 525},
    {6567, 1112}, {6569, 1410}, {6570, 1323}, {6575, 884},  {6577, 2341},
    {6578, 1587}, {6580, 2577}, {6583, 859},  {6585, 476},  {6588, 1617},
    {6590, 957},  {6591, 1999}, {6593, 279},  {6594, 2679}, {6596, 665},
    {6598, 885},  {6599, 189},  {6601, 457},  {6604, 1279}, {6606, 2365},
    {6607, 813},  {6609, 286},  {6612, 289},  {6614, 2105}, {6615, 136},
    {6617, 71},   {6618, 2379}, {6620, 341},  {6622, 645},  {6623, 491},
    {6625, 549},  {6628, 2259}, {6630, 1617}, {6631, 823},  {6633, 535},
    {6634, 559},  {6636, 1189}, {6638, 3185}, {6639, 2135}, {6641, 167},
    {6642, 1063}, {6644, 917},  {6647, 77},   {6649, 202},  {6650, 1391},
    {6652, 1143}, {6654, 1809}, {6655, 54},   {6657, 839},  {6660, 475},
    {6662, 2901}, {6663, 416},  {6665, 147},  {6666, 999},  {6668, 221},
    {6670, 673},  {6671, 930},  {6673, 256},  {6674, 983},  {6676, 3141},
    {6678, 41},   {6679, 550},  {6681, 124},  {6684, 1275}, {6686, 617},
    {6687, 872},  {6689, 1764}, {6690, 1911}, {6692, 2237}, {6694, 2989},
    {6695, 3072}, {6697, 2613}, {6700, 625},  {6702, 1145}, {6703, 675},
    {6705, 218},  {6706, 1683}, {6708, 189},  {6710, 1397}, {6711, 2929},
    {6713, 11},   {6714, 751},  {6716, 1781}, {6719, 455},  {6721, 210},
    {6722, 2387}, {6724, 1017}, {6726, 21},   {6727, 397},  {6729, 631},
    {6730, 3},    {6732, 965},  {6735, 1241}, {6737, 1197}, {6740, 41},
    {6743, 240},  {6745, 136},  {6746, 1563}, {6748, 1273}, {6750, 2781},
    {6751, 1104}, {6753, 35},   {6754, 1483}, {6756, 225},  {6757, 466},
    {6758, 3105}, {6759, 29},   {6761, 15},   {6764, 989},  {6767, 300},
    {6769, 97},   {6772, 321},  {6774, 733},  {6775, 1293}, {6777, 925},
    {6780, 655},  {6782, 3321}, {6783, 518},  {6785, 452},  {6786, 171},
    {6788, 2345}, {6791, 107},  {6793, 571},  {6794, 707},  {6796, 895},
    {6798, 1029}, {6799, 1240}, {6801, 140},  {6804, 49},   {6807, 1568},
    {6810, 679},  {6812, 1115}, {6815, 518},  {6817, 682},  {6818, 1571},
    {6820, 747},  {6822, 1425}, {6825, 133},  {6826, 435},  {6828, 63},
    {6830, 537},  {6831, 115},  {6833, 35},   {6836, 473},  {6838, 357},
    {6839, 96},   {6841, 427},  {6846, 2497}, {6847, 2247}, {6849, 221},
    {6852, 839},  {6855, 254},  {6857, 386},  {6858, 747},  {6860, 531},
    {6862, 1405}, {6863, 956},  {6865, 2494}, {6866, 647},  {6868, 43},
    {6871, 1719}, {6873, 2056}, {6874, 2631}, {6876, 287},  {6881, 152},
    {6882, 1883}, {6884, 383},  {6886, 565},  {6887, 495},  {6889, 652},
    {6892, 421},  {6895, 256},  {6898, 439},  {6900, 915},  {6902, 645},
    {6903, 2981}, {6905, 2513}, {6906, 1667}, {6908, 1613}, {6910, 1953},
    {6911, 1031}, {6913, 414},  {6918, 1653}, {6921, 1046}, {6924, 93},
    {6927, 653},  {6929, 3216}, {6930, 699},  {6932, 627},  {6935, 678},
    {6937, 582},  {6938, 963},  {6940, 399},  {6942, 1949}, {6943, 211},
    {6948, 63},   {6951, 434},  {6953, 1820}, {6954, 735},  {6956, 1727},
    {6958, 2113}, {6959, 1070}, {6961, 570},  {6964, 13},   {6966, 249},
    {6967, 2269}, {6969, 1073}, {6970, 1867}, {6972, 455},  {6974, 369},
    {6975, 992},  {6977, 285},  {6980, 485},  {6983, 168},  {6985, 439},
    {6986, 579},  {6988, 445},  {6990, 1673}, {6991, 510},  {6993, 166},
    {6996, 1025}, {6998, 1241}, {6999, 623},  {7001, 1631}, {7002, 3375},
    {7004, 291},  {7006, 981},  {7007, 914},  {7009, 1932}, {7012, 273},
    {7014, 633},  {7015, 1483}, {7017, 175},  {7018, 1695}, {7020, 117},
    {7022, 1337}, {7023, 242},  {7025, 1701}, {7026, 195},  {7030, 793},
    {7031, 1029}, {7033, 16},   {7034, 2679}, {7036, 1353}, {7038, 2961},
    {7039, 669},  {7041, 587},  {7042, 2871}, {7044, 23},   {7046, 1961},
    {7047, 238},  {7049, 794},  {7050, 1379}, {7052, 3471}, {7055, 803},
    {7057, 937},  {7058, 1103}, {7060, 765},  {7062, 2597}, {7063, 265},
    {7065, 599},  {7068, 549},  {7070, 1217}, {7071, 137},  {7073, 2202},
    {7074, 259},  {7076, 677},  {7077, 674},  {7079, 351},  {7081, 333},
    {7082, 423},  {7084, 345},  {7087, 1245}, {7089, 2519}, {7092, 1929},
    {7095, 67},   {7097, 936},  {7098, 2695}, {7100, 921},  {7103, 63},
    {7105, 1954}, {7106, 3123}, {7108, 235},  {7110, 41},   {7111, 1665},
    {7113, 1618}, {7116, 3417}, {7118, 1113}, {7119, 935},  {7121, 953},
    {7124, 89},   {7126, 2785}, {7127, 42},   {7129, 130},  {7132, 273},
    {7134, 729},  {7135, 553},  {7137, 337},  {7140, 225},  {7143, 2426},
    {7145, 1127}, {7146, 259},  {7148, 935},  {7150, 33},   {7151, 660},
    {7153, 301},  {7156, 1117}, {7159, 1332}, {7161, 49},   {7162, 1647},
    {7164, 33},   {7166, 3521}, {7167, 3529}, {7169, 1110}, {7170, 1887},
    {7172, 2549}, {7175, 584},  {7177, 918},  {7180, 415},  {7182, 245},
    {7183, 30},   {7185, 1064}, {7190, 897},  {7191, 1201}, {7193, 2412},
    {7194, 1763}, {7196, 1607}, {7199, 21},   {7201, 153},  {7202, 1631},
    {7203, 686},  {7204, 1695}, {7206, 2349}, {7207, 246},  {7209, 460},
    {7210, 3111}, {7215, 19},   {7217, 891},  {7218, 3263}, {7220, 2975},
    {7222, 2733}, {7223, 245},  {7225, 333},  {7226, 2807}, {7228, 1165},
    {7230, 2421}, {7231, 2185}, {7233, 1435}, {7236, 75},   {7239, 1610},
    {7241, 308},  {7242, 87},   {7244, 497},  {7246, 2505}, {7247, 726},
    {7249, 274},  {7250, 2427}, {7252, 1885}, {7254, 933},  {7255, 283},
    {7257, 3472}, {7258, 2731}, {7260, 121},  {7263, 256},  {7265, 776},
    {7266, 139},  {7270, 829},  {7271, 1325}, {7273, 789},  {7276, 1621},
    {7278, 1741}, {7279, 3247}, {7281, 179},  {7284, 625},  {7286, 1793},
    {7287, 1349}, {7289, 1253}, {7290, 1431}, {7292, 3033}, {7295, 1961},
    {7297, 570},  {7300, 925},  {7302, 1521}, {7303, 3280}, {7305, 602},
    {7306, 7},    {7308, 287},  {7310, 3557}, {7311, 341},  {7313, 45},
    {7314, 2787}, {7316, 1415}, {7318, 2625}, {7319, 330},  {7321, 202},
    {7322, 1587}, {7324, 523},  {7326, 381},  {7327, 2947}, {7329, 70},
    {7332, 643},  {7334, 237},  {7335, 1421}, {7337, 828},  {7338, 1351},
    {7340, 311},  {7342, 577},  {7343, 170},  {7345, 1344}, {7346, 671},
    {7348, 2863}, {7350, 573},  {7351, 601},  {7353, 410},  {7354, 2443},
    {7356, 1179}, {7359, 2864}, {7361, 155},  {7362, 355},  {7364, 377},
    {7366, 397},  {7367, 213},  {7369, 1041}, {7370, 1683}, {7372, 1429},
    {7374, 829},  {7375, 172},  {7377, 407},  {7378, 523},  {7380, 105},
    {7382, 2537}, {7383, 20},   {7385, 912},  {7386, 751},  {7388, 629},
    {7391, 35},   {7393, 1407}, {7394, 1127}, {7398, 145},  {7399, 2715},
    {7401, 1814}, {7402, 871},  {7404, 297},  {7409, 854},  {7410, 2179},
    {7412, 1797}, {7417, 634},  {7418, 867},  {7422, 2961}, {7423, 285},
    {7425, 371},  {7426, 1371}, {7428, 343},  {7431, 65},   {7433, 420},
    {7434, 2295}, {7436, 315},  {7438, 3285}, {7439, 2505}, {7441, 1053},
    {7444, 679},  {7446, 1217}, {7447, 1},    {7449, 625},  {7450, 379},
    {7452, 1701}, {7455, 284},  {7458, 3127}, {7460, 1535}, {7462, 345},
    {7463, 2939}, {7465, 277},  {7466, 2087}, {7468, 789},  {7471, 400},
    {7473, 457},  {7476, 423},  {7478, 1797}, {7479, 1069}, {7481, 887},
    {7484, 3305}, {7487, 2940}, {7489, 2644}, {7490, 2843}, {7492, 1015},
    {7495, 241},  {7497, 49},   {7500, 125},  {7503, 628},  {7505, 1451},
    {7506, 135},  {7508, 3503}, {7511, 51},   {7513, 2697}, {7518, 3653},
    {7519, 15},   {7521, 1501}, {7522, 2331}, {7524, 75},   {7526, 533},
    {7527, 1507}, {7529, 953},  {7530, 307},  {7532, 705},  {7534, 2349},
    {7535, 2726}, {7537, 1377}, {7538, 735},  {7540, 2127}, {7542, 549},
    {7543, 1173}, {7545, 1679}, {7548, 1689}, {7551, 145},  {7553, 492},
    {7556, 2835}, {7558, 1501}, {7559, 185},  {7561, 846},  {7562, 1547},
    {7564, 1845}, {7566, 209},  {7567, 151},  {7569, 3409}, {7570, 3303},
    {7572, 2413}, {7574, 197},  {7575, 2174}, {7577, 846},  {7578, 423},
    {7580, 761},  {7583, 653},  {7585, 268},  {7586, 3251}, {7588, 655},
    {7590, 869},  {7591, 1236}, {7593, 1210}, {7594, 3471}, {7596, 55},
    {7598, 1469}, {7599, 2066}, {7601, 2555}, {7602, 555},  {7604, 1257},
    {7607, 111},  {7609, 495},  {7612, 1671}, {7614, 945},  {7620, 203},
    {7622, 581},  {7623, 637},  {7625, 474},  {7626, 291},  {7628, 3773},
    {7630, 2833}, {7631, 92},   {7634, 2963}, {7636, 733},  {7638, 2673},
    {7639, 3729}, {7641, 925},  {7642, 663},  {7644, 161},  {7646, 1233},
    {7647, 1105}, {7649, 2222}, {7650, 199},  {7652, 2165}, {7655, 3629},
    {7657, 1069}, {7658, 3627}, {7660, 2287}, {7662, 2473}, {7663, 1203},
    {7665, 97},   {7666, 1351}, {7668, 77},   {7671, 769},  {7673, 120},
    {7674, 1891}, {7678, 2325}, {7679, 1497}, {7681, 1500}, {7682, 479},
    {7684, 1419}, {7686, 581},  {7687, 867},  {7689, 728},  {7692, 1683},
    {7694, 2685}, {7695, 119},  {7697, 282},  {7698, 111},  {7700, 177},
    {7702, 541},  {7703, 2786}, {7705, 1962}, {7706, 831},  {7708, 13},
    {7710, 3853}, {7711, 3084}, {7713, 1060}, {7714, 367},  {7716, 1157},
    {7719, 1993}, {7721, 1382}, {7722, 15},   {7724, 1967}, {7727, 1007},
    {7729, 594},  {7730, 1979}, {7732, 49},   {7734, 2089}, {7735, 717},
    {7738, 3499}, {7740, 47},   {7742, 377},  {7743, 3007}, {7745, 1364},
    {7748, 1155}, {7750, 721},  {7751, 273},  {7753, 1459}, {7758, 1633},
    {7759, 42},   {7761, 2074}, {7762, 231},  {7764, 499},  {7766, 1341},
    {7767, 410},  {7769, 2933}, {7770, 1943}, {7772, 2879}, {7774, 3585},
    {7775, 243},  {7777, 150},  {7780, 325},  {7783, 118},  {7785, 818},
    {7788, 435},  {7790, 1533}, {7791, 749},  {7793, 233},  {7794, 675},
    {7796, 1841}, {7799, 137},  {7801, 558},  {7802, 2103}, {7807, 1533},
    {7809, 2233}, {7810, 699},  {7812, 413},  {7814, 2121}, {7815, 994},
    {7817, 47},   {7818, 1291}, {7820, 35},   {7822, 2929}, {7823, 458},
    {7825, 966},  {7826, 3483}, {7828, 163},  {7830, 1281}, {7831, 825},
    {7833, 613},  {7838, 3789}, {7839, 374},  {7841, 2829}, {7847, 270},
    {7849, 967},  {7850, 3591}, {7852, 2235}, {7854, 213},  {7855, 1087},
    {7857, 215},  {7858, 3643}, {7860, 2505}, {7863, 1568}, {7865, 654},
    {7866, 79},   {7868, 1227}, {7870, 981},  {7871, 2462}, {7873, 154},
    {7874, 855},  {7876, 3165}, {7878, 129},  {7884, 215},  {7887, 664},
    {7889, 2400}, {7890, 531},  {7892, 1113}, {7895, 197},  {7897, 940},
    {7900, 1681}, {7902, 2729}, {7903, 612},  {7905, 1612}, {7908, 1333},
    {7910, 221},  {7911, 197},  {7913, 1466}, {7919, 756},  {7921, 2206},
    {7922, 1335}, {7924, 1089}, {7926, 1653}, {7927, 321},  {7929, 1178},
    {7932, 515},  {7934, 137},  {7935, 133},  {7937, 617},  {7938, 891},
    {7940, 795},  {7942, 1057}, {7943, 80},   {7945, 2967}, {7946, 51},
    {7948, 3001}, {7951, 498},  {7953, 413},  {7956, 1557}, {7958, 1121},
    {7959, 424},  {7961, 1412}, {7962, 1463}, {7964, 1253}, {7966, 1153},
    {7967, 483},  {7969, 1054}, {7972, 1695}, {7975, 387},  {7977, 1249},
    {7980, 15},   {7983, 2678}, {7985, 504},  {7988, 1679}, {7990, 1041},
    {7991, 2160}, {7993, 2073}, {7994, 2423}, {7996, 2581}, {7998, 1989},
    {7999, 952},  {8001, 127},  {8004, 3087}, {8006, 2201}, {8007, 554},
    {8009, 3159}, {8010, 687},  {8012, 1541}, {8014, 1753}, {8015, 776},
    {8017, 141},  {8020, 1713}, {8022, 917},  {8025, 229},  {8026, 1843},
    {8028, 279},  {8030, 53},   {8033, 2240}, {8034, 695},  {8036, 131},
    {8039, 98},   {8041, 1305}, {8044, 3627}, {8046, 1389}, {8047, 1444},
    {8049, 1009}, {8054, 1025}, {8055, 1069}, {8057, 960},  {8058, 1115},
    {8060, 297},  {8062, 2449}, {8063, 273},  {8065, 253},  {8068, 1603},
    {8070, 413},  {8071, 358},  {8073, 883},  {8076, 1731}, {8078, 669},
    {8079, 1523}, {8081, 768},  {8082, 239},  {8084, 2519}, {8086, 1597},
    {8087, 989},  {8089, 967},  {8090, 3759}, {8092, 351},  {8094, 2889},
    {8095, 907},  {8097, 2513}, {8098, 2667}, {8100, 9},    {8103, 2333},
    {8105, 1611}, {8106, 771},  {8108, 627},  {8110, 217},  {8111, 1130},
    {8113, 3058}, {8114, 1667}, {8116, 1057}, {8119, 10},   {8121, 3494},
    {8124, 269},  {8127, 218},  {8129, 548},  {8130, 719},  {8134, 25},
    {8135, 468},  {8137, 739},  {8138, 1043}, {8140, 1683}, {8143, 1890},
    {8145, 728},  {8146, 1911}, {8148, 133},  {8150, 89},   {8151, 3841},
    {8153, 1020}, {8154, 995},  {8156, 3641}, {8159, 1026}, {8161, 292},
    {8162, 1391}, {8164, 471},  {8166, 381},  {8167, 2553}, {8169, 481},
    {8172, 539},  {8175, 661},  {8177, 905},  {8178, 2611}, {8180, 185},
    {8182, 1477}, {8183, 1139}, {8185, 777},  {8188, 1155}, {8190, 1869},
    {8191, 714},  {8193, 1055}, {8194, 3847}, {8196, 2477}, {8198, 657},
    {8199, 2758}, {8201, 365},  {8204, 1727}, {8207, 5},    {8209, 943},
    {8210, 3507}, {8212, 1773}, {8215, 469},  {8217, 94},   {8218, 1443},
    {8220, 327},  {8222, 2345}, {8223, 308},  {8225, 2588}, {8226, 447},
    {8230, 2149}, {8231, 3564}, {8233, 1407}, {8234, 215},  {8236, 351},
    {8239, 1462}, {8241, 1694}, {8244, 343},  {8246, 173},  {8247, 2050},
    {8250, 3399}, {8252, 2861}, {8254, 3157}, {8255, 1496}, {8257, 1530},
    {8258, 1407}, {8260, 225},  {8262, 693},  {8263, 2232}, {8265, 161},
    {8268, 477},  {8271, 44},   {8273, 3500}, {8274, 419},  {8276, 915},
    {8278, 193},  {8279, 2439}, {8281, 226},  {8282, 351},  {8284, 2197},
    {8286, 3957}, {8287, 1755}, {8289, 98},   {8292, 637},  {8295, 938},
    {8297, 2846}, {8300, 185},  {8302, 1305}, {8303, 3789}, {8305, 417},
    {8310, 3333}, {8311, 1470}, {8313, 4117}, {8316, 405},  {8319, 458},
    {8321, 2985}, {8324, 1149}, {8327, 596},  {8329, 775},  {8330, 1107},
    {8332, 651},  {8334, 3285}, {8335, 129},  {8337, 643},  {8340, 553},
    {8342, 2921}, {8343, 1667}, {8345, 777},  {8348, 1737}, {8350, 3177},
    {8351, 170},  {8353, 2548}, {8354, 1515}, {8358, 677},  {8359, 1905},
    {8362, 655},  {8364, 2443}, {8366, 2381}, {8367, 154},  {8370, 1643},
    {8372, 17},   {8374, 4125}, {8375, 2316}, {8377, 2035}, {8378, 3851},
    {8380, 1815}, {8385, 2369}, {8386, 907},  {8388, 691},  {8390, 3797},
    {8391, 1336}, {8393, 312},  {8398, 3481}, {8399, 4191}, {8401, 1105},
    {8402, 287},  {8404, 805},  {8407, 1557}, {8409, 2506}, {8412, 1049},
    {8414, 3197}, {8415, 1061}, {8417, 2240}, {8418, 2187}, {8420, 2751},
    {8423, 168},  {8425, 3},    {8426, 1539}, {8428, 129},  {8430, 361},
    {8431, 97},   {8433, 1255}, {8434, 1459}, {8436, 401},  {8439, 43},
    {8441, 992},  {8442, 1863}, {8444, 905},  {8446, 3201}, {8447, 213},
    {8449, 25},   {8455, 412},  {8457, 230},  {8460, 127},  {8463, 397},
    {8465, 936},  {8466, 451},  {8470, 1713}, {8471, 827},  {8473, 2398},
    {8476, 85},   {8478, 405},  {8479, 1425}, {8481, 641},  {8482, 331},
    {8484, 1421}, {8487, 4112}, {8489, 3096}, {8492, 497},  {8495, 122},
    {8497, 540},  {8500, 753},  {8502, 3157}, {8503, 91},   {8505, 109},
    {8506, 3555}, {8508, 1207}, {8510, 2709}, {8511, 4177}, {8513, 3044},
    {8516, 1221}, {8518, 721},  {8519, 36},   {8521, 2646}, {8522, 3603},
    {8524, 1575}, {8526, 637},  {8527, 435},  {8529, 2296}, {8530, 2019},
    {8532, 787},  {8535, 2996}, {8537, 242},  {8540, 465},  {8543, 969},
    {8545, 1944}, {8548, 519},  {8550, 893},  {8551, 639},  {8553, 4171},
    {8556, 2011}, {8558, 4193}, {8559, 2161}, {8562, 939},  {8564, 801},
    {8566, 2581}, {8569, 1429}, {8572, 1081}, {8575, 457},  {8577, 439},
    {8580, 143},  {8582, 1817}, {8583, 3448}, {8585, 137},  {8586, 407},
    {8588, 29},   {8591, 431},  {8593, 94},   {8599, 2385}, {8601, 734},
    {8602, 1431}, {8604, 1381}, {8607, 203},  {8609, 434},  {8612, 287},
    {8614, 1585}, {8615, 3713}, {8617, 1150}, {8618, 1947}, {8620, 1197},
    {8623, 1483}, {8625, 256},  {8626, 103},  {8631, 857},  {8633, 3234},
    {8636, 2345}, {8638, 3961}, {8639, 441},  {8641, 81},   {8644, 103},
    {8646, 4293}, {8647, 145},  {8649, 280},  {8650, 783},  {8652, 833},
    {8654, 2529}, {8655, 2863}, {8657, 884},  {8658, 3323}, {8660, 1157},
    {8663, 2306}, {8665, 2416}, {8666, 2111}, {8668, 1839}, {8670, 1061},
    {8671, 1329}, {8673, 28},   {8674, 3403}, {8676, 927},  {8679, 1370},
    {8681, 1175}, {8682, 947},  {8684, 3815}, {8686, 2653}, {8689, 855},
    {8692, 1095}, {8694, 1809}, {8695, 2283}, {8697, 164},  {8698, 2583},
    {8700, 1295}, {8703, 281},  {8705, 59},   {8708, 417},  {8711, 360},
    {8713, 1785}, {8714, 231},  {8718, 3465}, {8719, 7},    {8721, 467},
    {8722, 1551}, {8724, 2279}, {8727, 748},  {8729, 68},   {8730, 959},
    {8732, 525},  {8734, 2817}, {8735, 1326}, {8737, 3235}, {8740, 2835},
    {8742, 1053}, {8743, 90},   {8745, 79},   {8746, 4183}, {8748, 63},
    {8750, 53},   {8751, 1922}, {8753, 693},  {8754, 2871}, {8756, 1181},
    {8759, 3675}, {8762, 2159}, {8766, 1677}, {8767, 70},   {8769, 188},
    {8770, 603},  {8772, 37},   {8775, 508},  {8777, 257},  {8780, 495},
    {8783, 860},  {8785, 1597}, {8791, 463},  {8793, 272},  {8796, 1585},
    {8798, 1301}, {8799, 656},  {8801, 309},  {8802, 2139}, {8804, 549},
    {8806, 1773}, {8807, 3569}, {8809, 1089}, {8812, 2295}, {8814, 1497},
    {8815, 1174}, {8817, 1654}, {8820, 147},  {8823, 1028}, {8825, 303},
    {8826, 4275}, {8828, 1133}, {8830, 3889}, {8831, 891},  {8833, 328},
    {8834, 683},  {8836, 1171}, {8838, 1653}, {8839, 285},  {8841, 539},
    {8842, 4143}, {8844, 355},  {8847, 1100}, {8849, 2324}, {8850, 1191},
    {8852, 1697}, {8855, 138},  {8857, 2493}, {8860, 3577}, {8862, 49},
    {8863, 1099}, {8865, 2713}, {8868, 623},  {8870, 1029}, {8871, 2422},
    {8873, 689},  {8874, 2239}, {8876, 381},  {8878, 3949}, {8879, 2369},
    {8881, 990},  {8884, 2121}, {8886, 469},  {8887, 1107}, {8889, 1216},
    {8890, 2007}, {8892, 249},  {8895, 958},  {8897, 1439}, {8898, 287},
    {8900, 2285}, {8903, 222},  {8905, 433},  {8908, 3031}, {8910, 1749},
    {8911, 963},  {8913, 158},  {8916, 2259}, {8919, 2479}, {8921, 6},
    {8922, 787},  {8924, 797},  {8926, 853},  {8927, 1425}, {8929, 358},
    {8930, 83},   {8932, 507},  {8934, 1125}, {8935, 861},  {8937, 3733},
    {8938, 2295}, {8940, 529},  {8943, 3635}, {8945, 563},  {8946, 967},
    {8948, 2637}, {8951, 506},  {8953, 25},   {8956, 4231}, {8958, 1029},
    {8959, 225},  {8961, 176},  {8962, 691},  {8964, 939},  {8966, 1233},
    {8967, 2744}, {8969, 435},  {8970, 2191}, {8972, 1619}, {8974, 981},
    {8975, 1128}, {8977, 966},  {8980, 3093}, {8982, 169},  {8983, 345},
    {8985, 2674}, {8986, 495},  {8988, 1513}, {8990, 333},  {8991, 2020},
    {8993, 2079}, {8994, 1051}, {8999, 1386}, {9001, 1599}, {9002, 3995},
    {9004, 649},  {9006, 1477}, {9007, 222},  {9009, 29},   {9012, 1053},
    {9014, 1217}, {9017, 155},  {9018, 1327}, {9020, 287},  {9022, 861},
    {9023, 4350}, {9025, 744},  {9026, 35},   {9028, 2935}, {9031, 714},
    {9033, 1615}, {9034, 2095}, {9036, 449},  {9039, 1741}, {9041, 813},
    {9042, 103},  {9044, 2819}, {9047, 287},  {9049, 2490}, {9050, 1319},
    {9052, 2725}, {9054, 609},  {9055, 946},  {9057, 2345}, {9058, 235},
    {9060, 151},  {9062, 3929}, {9063, 2320}, {9065, 2022}, {9068, 2565},
    {9071, 2276}, {9074, 207},  {9076, 2395}, {9078, 2681}, {9079, 2179},
    {9081, 190},  {9084, 1699}, {9086, 2445}, {9089, 440},  {9092, 29},
    {9094, 2769}, {9095, 2546}, {9097, 1260}, {9100, 55},   {9103, 4200},
    {9105, 1493}, {9106, 1723}, {9108, 575},  {9111, 487},  {9113, 3563},
    {9114, 539},  {9116, 1841}, {9119, 917},  {9121, 249},  {9126, 181},
    {9127, 1120}, {9129, 23},   {9130, 2991}, {9132, 207},  {9134, 1545},
    {9135, 944},  {9137, 2163}, {9138, 2671}, {9140, 1365}, {9142, 4545},
    {9143, 1268}, {9145, 546},  {9148, 1921}, {9150, 469},  {9151, 1330},
    {9153, 682},  {9154, 3087}, {9156, 281},  {9158, 809},  {9159, 3470},
    {9161, 1046}, {9162, 2807}, {9164, 1913}, {9169, 229},  {9170, 1103},
    {9175, 2806}, {9177, 884},  {9180, 1153}, {9182, 1769}, {9185, 207},
    {9186, 3159}, {9188, 4305}, {9190, 2217}, {9191, 3936}, {9193, 1216},
    {9198, 1365}, {9199, 109},  {9201, 146},  {9202, 211},  {9204, 645},
    {9206, 4085}, {9207, 397},  {9209, 2181}, {9210, 2603}, {9212, 3839},
    {9214, 397},  {9215, 1217}, {9217, 1956}, {9218, 2919}, {9220, 1875},
    {9223, 1023}, {9225, 181},  {9226, 4315}, {9228, 145},  {9231, 3899},
    {9233, 545},  {9234, 315},  {9238, 49},   {9239, 1629}, {9241, 909},
    {9242, 239},  {9244, 441},  {9247, 48},   {9249, 1663}, {9250, 531},
    {9252, 327},  {9255, 298},  {9257, 1364}, {9258, 3843}, {9260, 2729},
    {9262, 1549}, {9263, 3173}, {9265, 1123}, {9268, 3183}, {9270, 837},
    {9271, 1488}, {9273, 502},  {9274, 1219}, {9276, 135},  {9278, 2145},
    {9279, 1526}, {9281, 2049}, {9282, 2307}, {9284, 1883}, {9287, 2520},
    {9289, 358},  {9290, 1731}, {9292, 3087}, {9294, 3373}, {9295, 2149},
    {9297, 1838}, {9298, 2139}, {9300, 135},  {9302, 605},  {9303, 208},
    {9305, 3344}, {9306, 675},  {9308, 1559}, {9310, 3141}, {9311, 2657},
    {9313, 924},  {9316, 1233}, {9319, 477},  {9321, 2627}, {9322, 4027},
    {9324, 665},  {9327, 4165}, {9329, 2910}, {9330, 4263}, {9332, 2081},
    {9334, 3141}, {9335, 1176}, {9337, 546},  {9340, 1339}, {9343, 2700},
    {9345, 353},  {9350, 1221}, {9351, 56},   {9353, 3221}, {9356, 1773},
    {9358, 453},  {9359, 1227}, {9361, 369},  {9364, 3297}, {9366, 2245},
    {9367, 3172}, {9369, 116},  {9370, 2311}, {9372, 269},  {9375, 4459},
    {9377, 914},  {9378, 1347}, {9380, 561},  {9382, 3297}, {9383, 1124},
    {9385, 523},  {9386, 3863}, {9390, 2601}, {9391, 1239}, {9393, 623},
    {9394, 3471}, {9396, 369},  {9398, 2769}, {9399, 1295}, {9401, 429},
    {9402, 267},  {9404, 2009}, {9407, 260},  {9409, 405},  {9414, 673},
    {9415, 1678}, {9417, 218},  {9418, 823},  {9420, 2701}, {9423, 1280},
    {9425, 699},  {9431, 3113}, {9433, 42},   {9436, 225},  {9439, 105},
    {9441, 479},  {9442, 4611}, {9444, 3513}, {9447, 458},  {9449, 584},
    {9450, 1431}, {9454, 4189}, {9455, 2807}, {9457, 4},    {9463, 327},
    {9465, 197},  {9468, 1695}, {9470, 2289}, {9471, 1459}, {9473, 656},
    {9476, 2583}, {9478, 909},  {9479, 200},  {9481, 4060}, {9484, 853},
    {9486, 945},  {9487, 211},  {9489, 2089}, {9490, 999},  {9492, 373},
    {9494, 477},  {9497, 2301}, {9500, 1175}, {9502, 621},  {9503, 99},
    {9508, 1141}, {9510, 2349}, {9511, 399},  {9513, 686},  {9516, 903},
    {9518, 1797}, {9519, 1573}, {9521, 116},  {9522, 243},  {9524, 3207},
    {9527, 1650}, {9529, 43},   {9530, 1623}, {9532, 2103}, {9534, 1193},
    {9535, 1119}, {9537, 565},  {9540, 3059}, {9542, 1185}, {9543, 533},
    {9545, 539},  {9546, 47},   {9548, 335},  {9551, 126},  {9553, 1498},
    {9558, 2041}, {9559, 702},  {9561, 3059}, {9562, 687},  {9564, 387},
    {9566, 1901}, {9567, 1883}, {9569, 60},   {9570, 487},  {9575, 888},
    {9577, 1668}, {9578, 1503}, {9580, 1771}, {9582, 3177}, {9583, 790},
    {9585, 2524}, {9586, 1867}, {9588, 525},  {9590, 2061}, {9591, 841},
    {9593, 284},  {9594, 499},  {9598, 4309}, {9599, 14},   {9601, 963},
    {9607, 262},  {9609, 1526}, {9610, 2883}, {9612, 93},   {9614, 4577},
    {9615, 361},  {9617, 1385}, {9620, 557},  {9622, 2145}, {9623, 2429},
    {9625, 508},  {9628, 31},   {9630, 37},   {9631, 295},  {9633, 2333},
    {9634, 3075}, {9636, 515},  {9639, 700},  {9641, 3287}, {9642, 3591},
    {9644, 1671}, {9647, 1227}, {9649, 657},  {9652, 2733}, {9654, 1969},
    {9655, 3558}, {9657, 1192}, {9658, 3691}, {9660, 57},   {9663, 446},
    {9665, 2763}, {9666, 435},  {9668, 225},  {9671, 1086}, {9673, 3538},
    {9674, 759},  {9676, 3367}, {9678, 4277}, {9679, 864},  {9681, 910},
    {9682, 3175}, {9684, 2405}, {9686, 2333}, {9687, 2054}, {9689, 84},
    {9690, 1503}, {9692, 3381}, {9695, 207},  {9697, 1201}, {9700, 565},
    {9702, 1533}, {9705, 1802}, {9708, 1753}, {9711, 917},  {9713, 2933},
    {9714, 1227}, {9716, 263},  {9719, 3492}, {9721, 171},  {9727, 760},
    {9729, 938},  {9730, 663},  {9732, 3747}, {9734, 4125}, {9735, 2086},
    {9737, 275},  {9738, 3663}, {9740, 3015}, {9743, 869},  {9745, 2361},
    {9746, 2559}, {9748, 889},  {9750, 833},  {9751, 1093}, {9753, 1078},
    {9754, 3891}, {9756, 1701}, {9758, 1557}, {9759, 706},  {9761, 2588},
    {9762, 4455}, {9766, 585},  {9767, 1668}, {9769, 1390}, {9770, 1091},
    {9772, 741},  {9774, 401},  {9775, 537},  {9777, 773},  {9780, 711},
    {9782, 161},  {9783, 88},   {9785, 2036}, {9786, 4147}, {9788, 2007},
    {9790, 177},  {9791, 390},  {9793, 339},  {9796, 321},  {9798, 2677},
    {9799, 139},  {9801, 284},  {9804, 1165}, {9807, 337},  {9810, 711},
    {9814, 1545}, {9817, 4420}, {9820, 105},  {9822, 305},  {9823, 144},
    {9825, 2563}, {9828, 207},  {9830, 1541}, {9831, 1159}, {9833, 104},
    {9839, 389},  {9841, 1788}, {9844, 565},  {9846, 2349}, {9847, 3537},
    {9849, 329},  {9852, 3379}, {9854, 317},  {9855, 803},  {9857, 4386},
    {9858, 4235}, {9860, 2997}, {9863, 1809}, {9865, 2604}, {9868, 339},
    {9870, 2681}, {9871, 903},  {9873, 3842}, {9874, 4279}, {9876, 183},
    {9878, 573},  {9879, 1181}, {9882, 747},  {9884, 389},  {9886, 3673},
    {9887, 710},  {9889, 2019}, {9890, 2583}, {9895, 787},  {9897, 430},
    {9900, 11},   {9903, 40},   {9905, 219},  {9906, 2027}, {9908, 2699},
    {9911, 483},  {9913, 1899}, {9914, 95},   {9916, 4483}, {9918, 381},
    {9919, 1185}, {9921, 901},  {9922, 2691}, {9926, 1445}, {9927, 1987},
    {9929, 1382}, {9930, 331},  {9932, 2397}, {9935, 2216}, {9937, 451},
    {9940, 2059}, {9942, 133},  {9943, 3069}, {9945, 1882}, {9946, 2355},
    {9948, 1535}, {9950, 2453}, {9951, 1334}, {9953, 539},  {9954, 343},
    {9956, 851},  {9959, 381},  {9961, 2707}, {9964, 2691}, {9966, 1701},
    {9967, 4399}, {9969, 295},  {9970, 2587}, {9972, 519},  {9974, 2045},
    {9975, 124},  {9977, 2954}, {9978, 1483}, {9980, 707},  {9982, 993},
    {9983, 785},  {9985, 1974}, {9986, 1143}, {9988, 3129}, {9990, 573},
    {9991, 495},  {9993, 121},  {9996, 1447}, {9998, 4013}, {9999, 2951}};

// ANSI range [160, 2000]
/**
 * ANSI X9.62 irreducible trinomials
 *
 * Accredited Standards Committee X9:
 * Harold Deal, Chairman
 * Alice Droogan, Vice Chairman
 * Cynthia Fuller, Secretariat
 * The X9F subcommittee on Data and Information Security:
 * Glenda Barnes, Chairman
 * The X9F1 working group:
 * M. Blake Greenlee, Chairman
 */
static polynomial_t ansi_trinomials[] = {
    {161, 18},   {162, 27},   {166, 37},   {167, 6},    {169, 34},
    {170, 11},   {172, 1},    {174, 13},   {175, 6},    {177, 8},
    {178, 31},   {180, 3},    {182, 81},   {183, 56},   {185, 24},
    {186, 11},   {191, 9},    {193, 15},   {194, 87},   {196, 3},
    {198, 9},    {199, 34},   {201, 14},   {202, 55},   {204, 27},
    {207, 43},   {209, 6},    {210, 7},    {212, 105},  {214, 73},
    {215, 23},   {217, 45},   {218, 11},   {220, 7},    {223, 33},
    {225, 32},   {228, 113},  {231, 26},   {233, 74},   {234, 31},
    {236, 5},    {238, 73},   {239, 36},   {241, 70},   {242, 95},
    {244, 111},  {247, 82},   {249, 35},   {250, 103},  {252, 15},
    {253, 46},   {255, 52},   {257, 12},   {258, 71},   {260, 15},
    {263, 93},   {265, 42},   {266, 47},   {268, 25},   {270, 53},
    {271, 58},   {273, 23},   {274, 67},   {276, 63},   {278, 5},
    {279, 5},    {281, 93},   {282, 35},   {284, 53},   {286, 69},
    {287, 71},   {289, 21},   {292, 37},   {294, 33},   {295, 48},
    {297, 5},    {300, 5},    {302, 41},   {303, 1},    {305, 102},
    {308, 15},   {310, 93},   {313, 79},   {314, 15},   {316, 63},
    {318, 45},   {319, 36},   {321, 31},   {322, 67},   {324, 51},
    {327, 34},   {329, 50},   {330, 99},   {332, 89},   {333, 2},
    {337, 55},   {340, 45},   {342, 125},  {343, 75},   {345, 22},
    {346, 63},   {348, 103},  {350, 53},   {351, 34},   {353, 69},
    {354, 99},   {358, 57},   {359, 68},   {362, 63},   {364, 9},
    {366, 29},   {367, 21},   {369, 91},   {370, 139},  {372, 111},
    {375, 16},   {377, 41},   {378, 43},   {380, 47},   {382, 81},
    {383, 90},   {385, 6},    {386, 83},   {388, 159},  {390, 9},
    {391, 28},   {393, 7},    {394, 135},  {396, 25},   {399, 26},
    {401, 152},  {402, 171},  {404, 65},   {406, 141},  {407, 71},
    {409, 87},   {412, 147},  {414, 13},   {415, 102},  {417, 107},
    {418, 199},  {420, 7},    {422, 149},  {423, 25},   {425, 12},
    {426, 63},   {428, 105},  {431, 120},  {433, 33},   {436, 165},
    {438, 65},   {439, 49},   {441, 7},    {444, 81},   {446, 105},
    {447, 73},   {449, 134},  {450, 47},   {455, 38},   {457, 16},
    {458, 203},  {460, 19},   {462, 73},   {463, 93},   {465, 31},
    {468, 27},   {470, 9},    {471, 1},    {473, 200},  {474, 191},
    {476, 9},    {478, 121},  {479, 104},  {481, 138},  {484, 105},
    {486, 81},   {487, 94},   {489, 83},   {490, 219},  {492, 7},
    {494, 17},   {495, 76},   {497, 78},   {498, 155},  {500, 27},
    {503, 3},    {505, 156},  {506, 23},   {508, 9},    {510, 69},
    {511, 10},   {513, 26},   {514, 67},   {516, 21},   {518, 33},
    {519, 79},   {521, 32},   {522, 39},   {524, 167},  {526, 97},
    {527, 47},   {529, 42},   {532, 1},    {534, 161},  {537, 94},
    {538, 195},  {540, 9},    {543, 16},   {545, 122},  {550, 193},
    {551, 135},  {553, 39},   {556, 153},  {558, 73},   {559, 34},
    {561, 71},   {564, 163},  {566, 153},  {567, 28},   {569, 77},
    {570, 67},   {574, 13},   {575, 146},  {577, 25},   {580, 237},
    {582, 85},   {583, 130},  {585, 88},   {588, 35},   {590, 93},
    {593, 86},   {594, 19},   {596, 273},  {599, 30},   {601, 201},
    {602, 215},  {604, 105},  {606, 165},  {607, 105},  {609, 31},
    {610, 127},  {612, 81},   {614, 45},   {615, 211},  {617, 200},
    {618, 295},  {620, 9},    {622, 297},  {623, 68},   {625, 133},
    {626, 251},  {628, 223},  {631, 307},  {633, 101},  {634, 39},
    {636, 217},  {639, 16},   {641, 11},   {642, 119},  {646, 249},
    {647, 5},    {649, 37},   {650, 3},    {651, 14},   {652, 93},
    {654, 33},   {655, 88},   {657, 38},   {658, 55},   {660, 11},
    {662, 21},   {663, 107},  {665, 33},   {668, 147},  {670, 153},
    {671, 15},   {673, 28},   {676, 31},   {679, 66},   {682, 171},
    {684, 209},  {686, 197},  {687, 13},   {689, 14},   {690, 79},
    {692, 299},  {694, 169},  {695, 177},  {697, 267},  {698, 215},
    {700, 75},   {702, 37},   {705, 17},   {708, 15},   {711, 92},
    {713, 41},   {714, 23},   {716, 183},  {718, 165},  {719, 150},
    {721, 9},    {722, 231},  {724, 207},  {726, 5},    {727, 180},
    {729, 58},   {730, 147},  {732, 343},  {735, 44},   {737, 5},
    {738, 347},  {740, 135},  {742, 85},   {743, 90},   {745, 258},
    {746, 351},  {748, 19},   {750, 309},  {751, 18},   {753, 158},
    {754, 19},   {756, 45},   {758, 233},  {759, 98},   {761, 3},
    {762, 83},   {767, 168},  {769, 120},  {772, 7},    {774, 185},
    {775, 93},   {777, 29},   {778, 375},  {780, 13},   {782, 329},
    {783, 68},   {785, 92},   {791, 30},   {793, 253},  {794, 143},
    {798, 53},   {799, 25},   {801, 217},  {804, 75},   {806, 21},
    {807, 7},    {809, 15},   {810, 159},  {812, 29},   {814, 21},
    {815, 333},  {817, 52},   {818, 119},  {820, 123},  {822, 17},
    {823, 9},    {825, 38},   {826, 255},  {828, 189},  {831, 49},
    {833, 149},  {834, 15},   {838, 61},   {839, 54},   {841, 144},
    {842, 47},   {844, 105},  {845, 2},    {846, 105},  {847, 136},
    {849, 253},  {850, 111},  {852, 159},  {855, 29},   {857, 119},
    {858, 207},  {860, 35},   {861, 14},   {862, 349},  {865, 1},
    {866, 75},   {868, 145},  {870, 301},  {871, 378},  {873, 352},
    {876, 149},  {879, 11},   {881, 78},   {882, 99},   {884, 173},
    {887, 147},  {889, 127},  {890, 183},  {892, 31},   {894, 173},
    {895, 12},   {897, 113},  {898, 207},  {900, 1},    {902, 21},
    {903, 35},   {905, 117},  {906, 123},  {908, 143},  {911, 204},
    {913, 91},   {916, 183},  {918, 77},   {919, 36},   {921, 221},
    {924, 31},   {926, 365},  {927, 403},  {930, 31},   {932, 177},
    {935, 417},  {937, 217},  {938, 207},  {942, 45},   {943, 24},
    {945, 77},   {948, 189},  {951, 260},  {953, 168},  {954, 131},
    {956, 305},  {959, 143},  {961, 18},   {964, 103},  {966, 201},
    {967, 36},   {969, 31},   {972, 7},    {975, 19},   {977, 15},
    {979, 178},  {982, 177},  {983, 230},  {985, 222},  {986, 3},
    {988, 121},  {990, 161},  {991, 39},   {993, 62},   {994, 223},
    {996, 65},   {998, 101},  {999, 59},   {1001, 17},  {1007, 75},
    {1009, 55},  {1010, 99},  {1012, 115}, {1014, 385}, {1015, 186},
    {1020, 135}, {1022, 317}, {1023, 7},   {1025, 294}, {1026, 35},
    {1028, 119}, {1029, 98},  {1030, 93},  {1031, 68},  {1033, 108},
    {1034, 75},  {1036, 411}, {1039, 21},  {1041, 412}, {1042, 439},
    {1044, 41},  {1047, 10},  {1049, 141}, {1050, 159}, {1052, 291},
    {1054, 105}, {1055, 24},  {1057, 198}, {1058, 27},  {1062, 49},
    {1063, 168}, {1065, 463}, {1071, 7},   {1078, 361}, {1079, 230},
    {1081, 24},  {1082, 407}, {1084, 189}, {1085, 62},  {1086, 189},
    {1087, 112}, {1089, 91},  {1090, 79},  {1092, 23},  {1094, 57},
    {1095, 139}, {1097, 14},  {1098, 83},  {1100, 35},  {1102, 117},
    {1103, 65},  {1105, 21},  {1106, 195}, {1108, 327}, {1110, 417},
    {1111, 13},  {1113, 107}, {1116, 59},  {1119, 283}, {1121, 62},
    {1122, 427}, {1126, 105}, {1127, 27},  {1129, 103}, {1130, 551},
    {1134, 129}, {1135, 9},   {1137, 277}, {1138, 31},  {1140, 141},
    {1142, 357}, {1145, 227}, {1146, 131}, {1148, 23},  {1151, 90},
    {1153, 241}, {1154, 75},  {1156, 307}, {1158, 245}, {1159, 66},
    {1161, 365}, {1164, 19},  {1166, 189}, {1167, 133}, {1169, 114},
    {1170, 27},  {1174, 133}, {1175, 476}, {1177, 16},  {1178, 375},
    {1180, 25},  {1182, 77},  {1183, 87},  {1185, 134}, {1186, 171},
    {1188, 75},  {1190, 233}, {1191, 196}, {1193, 173}, {1196, 281},
    {1198, 405}, {1199, 114}, {1201, 171}, {1202, 287}, {1204, 43},
    {1206, 513}, {1207, 273}, {1209, 118}, {1210, 243}, {1212, 203},
    {1214, 257}, {1215, 302}, {1217, 393}, {1218, 91},  {1220, 413},
    {1223, 255}, {1225, 234}, {1226, 167}, {1228, 27},  {1230, 433},
    {1231, 105}, {1233, 151}, {1234, 427}, {1236, 49},  {1238, 153},
    {1239, 4},   {1241, 54},  {1242, 203}, {1246, 25},  {1247, 14},
    {1249, 187}, {1252, 97},  {1255, 589}, {1257, 289}, {1260, 21},
    {1263, 77},  {1265, 119}, {1266, 7},   {1268, 345}, {1270, 333},
    {1271, 17},  {1273, 168}, {1276, 217}, {1278, 189}, {1279, 216},
    {1281, 229}, {1282, 231}, {1284, 223}, {1286, 153}, {1287, 470},
    {1289, 99},  {1294, 201}, {1295, 38},  {1297, 198}, {1298, 399},
    {1300, 75},  {1302, 77},  {1305, 326}, {1306, 39},  {1308, 495},
    {1310, 333}, {1311, 476}, {1313, 164}, {1314, 19},  {1319, 129},
    {1321, 52},  {1324, 337}, {1326, 397}, {1327, 277}, {1329, 73},
    {1332, 95},  {1334, 617}, {1335, 392}, {1337, 75},  {1338, 315},
    {1340, 125}, {1343, 348}, {1345, 553}, {1348, 553}, {1350, 237},
    {1351, 39},  {1353, 371}, {1354, 255}, {1356, 131}, {1358, 117},
    {1359, 98},  {1361, 56},  {1362, 655}, {1364, 239}, {1366, 1},
    {1367, 134}, {1369, 88},  {1372, 181}, {1374, 609}, {1375, 52},
    {1377, 100}, {1380, 183}, {1383, 130}, {1385, 12},  {1386, 219},
    {1388, 11},  {1390, 129}, {1391, 3},   {1393, 300}, {1396, 97},
    {1398, 601}, {1399, 55},  {1401, 92},  {1402, 127}, {1404, 81},
    {1407, 47},  {1409, 194}, {1410, 383}, {1412, 125}, {1414, 429},
    {1415, 282}, {1417, 342}, {1420, 33},  {1422, 49},  {1423, 15},
    {1425, 28},  {1426, 103}, {1428, 27},  {1430, 33},  {1431, 17},
    {1433, 387}, {1434, 363}, {1436, 83},  {1438, 357}, {1441, 322},
    {1442, 395}, {1444, 595}, {1446, 421}, {1447, 195}, {1449, 13},
    {1452, 315}, {1454, 297}, {1455, 52},  {1457, 314}, {1458, 243},
    {1460, 185}, {1463, 575}, {1465, 39},  {1466, 311}, {1468, 181},
    {1470, 49},  {1471, 25},  {1473, 77},  {1476, 21},  {1478, 69},
    {1479, 49},  {1481, 32},  {1482, 411}, {1486, 85},  {1487, 140},
    {1489, 252}, {1490, 279}, {1492, 307}, {1495, 94},  {1497, 49},
    {1500, 25},  {1503, 80},  {1505, 246}, {1508, 599}, {1510, 189},
    {1511, 278}, {1513, 399}, {1514, 299}, {1516, 277}, {1518, 69},
    {1519, 220}, {1521, 229}, {1524, 27},  {1526, 473}, {1527, 373},
    {1529, 60},  {1530, 207}, {1534, 225}, {1535, 404}, {1537, 46},
    {1540, 75},  {1542, 365}, {1543, 445}, {1545, 44},  {1548, 63},
    {1550, 189}, {1551, 557}, {1553, 252}, {1554, 99},  {1556, 65},
    {1558, 9},   {1559, 119}, {1561, 339}, {1562, 95},  {1564, 7},
    {1566, 77},  {1567, 127}, {1569, 319}, {1570, 667}, {1572, 501},
    {1575, 17},  {1577, 341}, {1578, 731}, {1580, 647}, {1582, 121},
    {1583, 20},  {1585, 574}, {1586, 399}, {1588, 85},  {1590, 169},
    {1591, 15},  {1593, 568}, {1596, 3},   {1599, 643}, {1601, 548},
    {1602, 783}, {1604, 317}, {1606, 153}, {1607, 87},  {1609, 231},
    {1612, 771}, {1615, 103}, {1617, 182}, {1618, 211}, {1620, 27},
    {1623, 17},  {1625, 69},  {1628, 603}, {1630, 741}, {1631, 668},
    {1633, 147}, {1634, 227}, {1636, 37},  {1638, 173}, {1639, 427},
    {1641, 287}, {1642, 231}, {1647, 310}, {1649, 434}, {1650, 579},
    {1652, 45},  {1655, 53},  {1657, 16},  {1660, 37},  {1663, 99},
    {1665, 176}, {1666, 271}, {1668, 459}, {1671, 202}, {1673, 90},
    {1674, 755}, {1676, 363}, {1678, 129}, {1679, 20},  {1681, 135},
    {1687, 31},  {1689, 758}, {1692, 359}, {1694, 501}, {1695, 29},
    {1697, 201}, {1698, 459}, {1700, 225}, {1703, 161}, {1705, 52},
    {1708, 93},  {1710, 201}, {1711, 178}, {1713, 250}, {1716, 221},
    {1719, 113}, {1721, 300}, {1722, 39},  {1724, 261}, {1726, 753},
    {1729, 94},  {1734, 461}, {1735, 418}, {1737, 403}, {1738, 267},
    {1740, 259}, {1742, 869}, {1743, 173}, {1745, 369}, {1746, 255},
    {1748, 567}, {1750, 457}, {1751, 482}, {1753, 775}, {1756, 99},
    {1759, 165}, {1764, 105}, {1767, 250}, {1769, 327}, {1770, 279},
    {1772, 371}, {1774, 117}, {1775, 486}, {1777, 217}, {1778, 635},
    {1780, 457}, {1782, 57},  {1783, 439}, {1785, 214}, {1788, 819},
    {1790, 593}, {1791, 190}, {1793, 114}, {1798, 69},  {1799, 312},
    {1801, 502}, {1802, 843}, {1804, 747}, {1806, 101}, {1807, 123},
    {1809, 521}, {1810, 171}, {1814, 545}, {1815, 163}, {1817, 479},
    {1818, 495}, {1820, 11},  {1823, 684}, {1825, 9},   {1828, 273},
    {1830, 381}, {1831, 51},  {1833, 518}, {1836, 243}, {1838, 53},
    {1839, 836}, {1841, 66},  {1844, 339}, {1846, 901}, {1847, 180},
    {1849, 49},  {1854, 885}, {1855, 39},  {1857, 688}, {1860, 13},
    {1862, 149}, {1863, 260}, {1865, 53},  {1866, 11},  {1870, 121},
    {1871, 261}, {1873, 199}, {1878, 253}, {1879, 174}, {1881, 370},
    {1884, 669}, {1886, 833}, {1887, 353}, {1889, 29},  {1890, 371},
    {1895, 873}, {1900, 235}, {1902, 733}, {1903, 778}, {1905, 344},
    {1906, 931}, {1908, 945}, {1911, 67},  {1913, 462}, {1918, 477},
    {1919, 105}, {1921, 468}, {1924, 327}, {1926, 357}, {1927, 25},
    {1929, 31},  {1932, 277}, {1934, 413}, {1935, 103}, {1937, 231},
    {1938, 747}, {1940, 113}, {1943, 11},  {1945, 91},  {1946, 51},
    {1948, 603}, {1950, 9},   {1951, 121}, {1953, 17},  {1956, 279},
    {1958, 89},  {1959, 371}, {1961, 771}, {1962, 99},  {1964, 21},
    {1966, 801}, {1967, 26},  {1969, 175}, {1974, 165}, {1975, 841},
    {1977, 238}, {1980, 33},  {1983, 113}, {1985, 311}, {1986, 891},
    {1988, 555}, {1990, 133}, {1991, 546}, {1993, 103}, {1994, 15},
    {1996, 307}, {1999, 367}};

// HP range [2, 159] + [2001, 10000]
/**
 * Table of Low-Weight Binary Irreducible Polynomials
 * Gadiel Seroussi
 * Computer Systems Laboratory
 * HPL-98-135
 * August, 1998
 *
 * http://www.hpl.hp.com/techreports/98/HPL-98-135.pdf
 */
static polynomial_t hp_pentanomials[] = {
    {8, 4, 3, 1},       {13, 4, 3, 1},      {16, 5, 3, 1},
    {19, 5, 2, 1},      {24, 4, 3, 1},      {26, 4, 3, 1},
    {27, 5, 2, 1},      {32, 7, 3, 2},      {37, 6, 4, 1},
    {38, 6, 5, 1},      {40, 5, 4, 3},      {43, 6, 4, 3},
    {45, 4, 3, 1},      {48, 5, 3, 2},      {50, 4, 3, 2},
    {51, 6, 3, 1},      {53, 6, 2, 1},      {56, 7, 4, 2},
    {59, 7, 4, 2},      {61, 5, 2, 1},      {64, 4, 3, 1},
    {67, 5, 2, 1},      {69, 6, 5, 2},      {70, 5, 3, 1},
    {72, 10, 9, 3},     {75, 6, 3, 1},      {77, 6, 5, 2},
    {78, 6, 5, 3},      {80, 9, 4, 2},      {82, 8, 3, 1},
    {83, 7, 4, 2},      {85, 8, 2, 1},      {88, 7, 6, 2},
    {91, 8, 5, 1},      {96, 10, 9, 6},     {99, 6, 3, 1},
    {101, 7, 6, 1},     {104, 4, 3, 1},     {107, 9, 7, 4},
    {109, 5, 4, 2},     {112, 5, 4, 3},     {114, 5, 3, 2},
    {115, 8, 7, 5},     {116, 4, 2, 1},     {117, 5, 2, 1},
    {120, 4, 3, 1},     {122, 6, 2, 1},     {125, 7, 6, 5},
    {128, 7, 2, 1},     {131, 8, 3, 2},     {133, 9, 8, 2},
    {136, 5, 3, 2},     {138, 8, 7, 1},     {139, 8, 5, 3},
    {141, 10, 4, 1},    {143, 5, 3, 2},     {144, 7, 4, 2},
    {149, 10, 9, 7},    {152, 6, 3, 2},     {157, 6, 5, 2},
    {158, 8, 6, 5},     {2002, 22, 21, 11}, {2003, 12, 10, 8},
    {2005, 17, 12, 7},  {2008, 26, 23, 13}, {2011, 17, 15, 4},
    {2012, 19, 15, 4},  {2013, 5, 4, 2},    {2014, 9, 7, 6},
    {2016, 21, 15, 7},  {2018, 20, 7, 3},   {2019, 20, 7, 2},
    {2021, 19, 14, 1},  {2024, 40, 35, 9},  {2027, 11, 10, 3},
    {2029, 12, 7, 4},   {2030, 13, 12, 2},  {2032, 7, 6, 2},
    {2035, 9, 8, 4},    {2037, 19, 18, 3},  {2038, 16, 11, 6},
    {2040, 7, 2, 1},    {2042, 16, 8, 7},   {2043, 9, 7, 4},
    {2045, 7, 6, 4},    {2046, 12, 11, 3},  {2048, 19, 14, 13},
    {2050, 15, 13, 8},  {2051, 13, 6, 5},   {2053, 21, 13, 6},
    {2056, 13, 12, 3},  {2059, 14, 12, 10}, {2061, 19, 4, 1},
    {2062, 16, 3, 2},   {2064, 17, 9, 2},   {2067, 17, 13, 8},
    {2068, 18, 10, 7},  {2069, 18, 9, 8},   {2071, 11, 5, 3},
    {2072, 13, 10, 3},  {2075, 9, 7, 4},    {2077, 15, 14, 4},
    {2078, 16, 6, 5},   {2080, 4, 3, 1},    {2083, 21, 11, 10},
    {2084, 4, 2, 1},    {2085, 9, 8, 3},    {2088, 18, 11, 5},
    {2090, 9, 4, 1},    {2091, 12, 9, 5},   {2092, 17, 15, 7},
    {2093, 16, 15, 7},  {2096, 19, 4, 2},   {2099, 15, 12, 9},
    {2101, 25, 22, 9},  {2104, 19, 15, 9},  {2107, 17, 10, 2},
    {2108, 21, 5, 3},   {2109, 10, 5, 1},   {2110, 12, 9, 6},
    {2112, 16, 13, 7},  {2115, 11, 10, 1},  {2117, 24, 16, 11},
    {2120, 17, 14, 6},  {2123, 14, 11, 3},  {2125, 24, 17, 14},
    {2128, 23, 9, 1},   {2131, 13, 6, 5},   {2133, 24, 22, 4},
    {2134, 23, 13, 2},  {2136, 15, 12, 9},  {2138, 13, 9, 2},
    {2139, 14, 4, 1},   {2141, 16, 3, 1},   {2144, 13, 7, 3},
    {2147, 8, 3, 2},    {2149, 17, 9, 2},   {2152, 24, 13, 7},
    {2154, 10, 7, 5},   {2155, 10, 9, 8},   {2157, 14, 4, 1},
    {2160, 31, 25, 14}, {2163, 12, 9, 7},   {2165, 10, 4, 2},
    {2168, 11, 9, 3},   {2171, 7, 6, 4},    {2172, 6, 4, 1},
    {2173, 15, 8, 4},   {2176, 15, 8, 1},   {2179, 8, 7, 2},
    {2181, 18, 12, 7},  {2184, 11, 10, 5},  {2187, 22, 11, 1},
    {2189, 19, 8, 7},   {2192, 8, 5, 3},    {2194, 15, 13, 9},
    {2195, 23, 22, 16}, {2197, 6, 5, 2},    {2200, 20, 15, 5},
    {2202, 15, 3, 2},   {2203, 14, 6, 5},   {2204, 11, 7, 1},
    {2205, 14, 7, 4},   {2208, 21, 16, 6},  {2211, 20, 14, 3},
    {2213, 20, 19, 4},  {2216, 24, 7, 2},   {2219, 22, 12, 11},
    {2221, 17, 16, 6},  {2223, 17, 6, 2},   {2224, 21, 19, 5},
    {2226, 19, 16, 7},  {2227, 14, 9, 8},   {2229, 17, 8, 7},
    {2232, 19, 17, 4},  {2234, 7, 6, 1},    {2235, 20, 17, 11},
    {2237, 7, 4, 2},    {2240, 23, 7, 1},   {2243, 27, 15, 2},
    {2245, 22, 12, 11}, {2246, 23, 15, 6},  {2248, 9, 4, 3},
    {2251, 19, 16, 13}, {2253, 19, 14, 2},  {2254, 10, 9, 2},
    {2256, 14, 9, 6},   {2259, 16, 10, 6},  {2260, 12, 4, 1},
    {2261, 14, 12, 7},  {2264, 15, 7, 2},   {2267, 18, 17, 8},
    {2269, 15, 10, 7},  {2272, 21, 10, 9},  {2275, 15, 12, 1},
    {2277, 10, 4, 2},   {2278, 17, 14, 7},  {2280, 13, 4, 2},
    {2283, 20, 13, 4},  {2285, 8, 6, 2},    {2288, 17, 16, 7},
    {2290, 23, 6, 5},   {2291, 13, 10, 2},  {2293, 10, 8, 3},
    {2294, 21, 17, 10}, {2296, 15, 10, 1},  {2298, 15, 13, 3},
    {2299, 16, 6, 1},   {2301, 15, 9, 8},   {2302, 15, 8, 1},
    {2304, 8, 7, 5},    {2307, 11, 7, 2},   {2309, 12, 10, 8},
    {2312, 19, 6, 4},   {2315, 17, 15, 4},  {2317, 17, 4, 2},
    {2320, 13, 11, 5},  {2323, 16, 3, 2},   {2325, 14, 8, 5},
    {2326, 13, 10, 7},  {2328, 15, 9, 2},   {2330, 15, 5, 1},
    {2333, 18, 15, 8},  {2336, 15, 10, 8},  {2339, 10, 3, 2},
    {2341, 24, 18, 16}, {2344, 27, 26, 14}, {2347, 11, 10, 2},
    {2349, 31, 16, 9},  {2350, 21, 19, 4},  {2352, 23, 20, 2},
    {2354, 14, 11, 6},  {2355, 12, 11, 6},  {2357, 10, 9, 8},
    {2360, 19, 10, 8},  {2362, 13, 10, 6},  {2363, 9, 7, 4},
    {2365, 21, 14, 5},  {2366, 16, 13, 6},  {2368, 13, 11, 8},
    {2370, 12, 11, 6},  {2371, 20, 18, 17}, {2373, 17, 14, 5},
    {2374, 24, 16, 13}, {2376, 7, 5, 4},    {2379, 22, 9, 3},
    {2381, 18, 12, 2},  {2384, 21, 19, 1},  {2386, 11, 9, 1},
    {2387, 8, 7, 6},    {2389, 31, 12, 10}, {2392, 31, 13, 3},
    {2395, 17, 8, 7},   {2397, 22, 17, 4},  {2398, 15, 13, 11},
    {2400, 20, 19, 17}, {2403, 11, 8, 2},   {2405, 17, 14, 5},
    {2406, 14, 12, 5},  {2408, 23, 6, 4},   {2411, 12, 2, 1},
    {2413, 11, 6, 1},   {2416, 23, 6, 5},   {2419, 22, 16, 6},
    {2421, 11, 8, 4},   {2424, 21, 16, 6},  {2426, 20, 4, 3},
    {2427, 8, 6, 5},    {2429, 22, 14, 7},  {2432, 29, 22, 19},
    {2435, 18, 6, 2},   {2437, 9, 4, 3},    {2440, 12, 7, 5},
    {2443, 15, 13, 4},  {2445, 23, 20, 10}, {2446, 18, 13, 6},
    {2448, 21, 10, 4},  {2451, 10, 4, 3},   {2453, 21, 14, 11},
    {2456, 12, 5, 3},   {2458, 23, 20, 18}, {2459, 17, 7, 4},
    {2461, 19, 18, 10}, {2462, 22, 13, 8},  {2464, 5, 4, 3},
    {2467, 18, 14, 2},  {2469, 26, 16, 11}, {2471, 15, 6, 3},
    {2472, 12, 3, 1},   {2475, 12, 11, 9},  {2476, 17, 3, 2},
    {2477, 19, 17, 6},  {2480, 20, 5, 2},   {2482, 11, 7, 6},
    {2483, 12, 11, 2},  {2486, 19, 14, 9},  {2488, 23, 13, 9},
    {2490, 8, 7, 5},    {2491, 25, 16, 12}, {2493, 18, 15, 7},
    {2494, 13, 8, 2},   {2496, 12, 3, 1},   {2498, 8, 7, 4},
    {2499, 13, 12, 9},  {2501, 22, 21, 5},  {2504, 19, 8, 6},
    {2506, 17, 11, 3},  {2507, 13, 8, 1},   {2509, 14, 13, 3},
    {2512, 29, 21, 7},  {2515, 8, 7, 5},    {2517, 14, 4, 1},
    {2518, 18, 13, 2},  {2520, 31, 15, 13}, {2522, 16, 10, 5},
    {2523, 23, 17, 14}, {2524, 15, 6, 4},   {2525, 7, 6, 5},
    {2528, 19, 9, 4},   {2530, 14, 5, 3},   {2531, 6, 2, 1},
    {2532, 26, 22, 13}, {2533, 10, 3, 1},   {2535, 24, 19, 16},
    {2536, 21, 10, 9},  {2539, 10, 9, 3},   {2541, 11, 8, 7},
    {2544, 21, 20, 6},  {2547, 22, 10, 3},  {2549, 10, 9, 5},
    {2552, 28, 3, 2},   {2555, 9, 6, 2},    {2557, 18, 17, 7},
    {2558, 18, 8, 1},   {2560, 9, 3, 1},    {2563, 12, 11, 10},
    {2565, 25, 16, 14}, {2568, 8, 5, 2},    {2570, 18, 15, 14},
    {2571, 18, 16, 9},  {2573, 12, 11, 3},  {2576, 7, 2, 1},
    {2578, 15, 7, 3},   {2579, 23, 6, 1},   {2581, 26, 14, 12},
    {2582, 22, 19, 15}, {2584, 25, 19, 12}, {2587, 23, 12, 10},
    {2588, 13, 8, 1},   {2589, 29, 11, 10}, {2592, 15, 12, 5},
    {2595, 24, 17, 2},  {2597, 16, 8, 7},   {2600, 14, 13, 7},
    {2602, 11, 6, 1},   {2603, 18, 8, 7},   {2605, 19, 14, 5},
    {2608, 15, 11, 2},  {2611, 27, 17, 10}, {2612, 21, 14, 6},
    {2613, 14, 12, 9},  {2616, 17, 8, 7},   {2619, 17, 10, 1},
    {2621, 8, 7, 1},    {2622, 11, 5, 3},   {2624, 15, 10, 4},
    {2627, 20, 17, 11}, {2629, 13, 9, 6},   {2632, 19, 5, 3},
    {2634, 20, 11, 5},  {2635, 25, 21, 14}, {2636, 18, 11, 10},
    {2637, 9, 6, 4},    {2638, 10, 3, 1},   {2640, 23, 11, 9},
    {2643, 16, 13, 10}, {2645, 18, 11, 10}, {2648, 15, 8, 1},
    {2650, 17, 13, 5},  {2651, 7, 5, 4},    {2653, 17, 4, 2},
    {2654, 20, 16, 13}, {2656, 19, 11, 5},  {2658, 6, 3, 2},
    {2659, 20, 18, 16}, {2661, 17, 14, 2},  {2662, 14, 12, 7},
    {2664, 27, 21, 19}, {2669, 18, 4, 2},   {2672, 18, 7, 1},
    {2675, 13, 10, 9},  {2677, 17, 8, 6},   {2678, 15, 6, 5},
    {2680, 15, 9, 4},   {2683, 11, 6, 1},   {2684, 14, 12, 3},
    {2685, 8, 2, 1},    {2688, 21, 10, 6},  {2690, 12, 9, 3},
    {2691, 14, 10, 8},  {2693, 10, 5, 3},   {2696, 17, 15, 12},
    {2699, 32, 30, 29}, {2701, 12, 10, 5},  {2704, 12, 9, 7},
    {2705, 12, 10, 5},  {2707, 9, 6, 4},    {2709, 10, 9, 3},
    {2710, 7, 3, 1},    {2712, 21, 12, 7},  {2714, 12, 8, 7},
    {2715, 30, 26, 15}, {2717, 14, 8, 3},   {2720, 25, 18, 1},
    {2723, 9, 6, 5},    {2725, 21, 17, 2},  {2726, 10, 6, 1},
    {2728, 21, 7, 5},   {2731, 15, 11, 2},  {2733, 19, 8, 7},
    {2734, 9, 5, 4},    {2736, 16, 3, 1},   {2739, 12, 9, 5},
    {2741, 6, 5, 4},    {2744, 19, 18, 10}, {2746, 9, 7, 2},
    {2747, 15, 11, 6},  {2749, 22, 18, 17}, {2752, 15, 4, 2},
    {2755, 13, 7, 6},   {2757, 24, 23, 12}, {2758, 17, 5, 4},
    {2760, 29, 26, 7},  {2763, 12, 10, 6},  {2765, 16, 10, 3},
    {2768, 25, 19, 15}, {2771, 13, 12, 5},  {2773, 30, 25, 20},
    {2774, 28, 3, 2},   {2776, 7, 3, 2},    {2777, 22, 21, 17},
    {2779, 20, 17, 9},  {2781, 28, 27, 10}, {2782, 22, 10, 9},
    {2784, 29, 21, 15}, {2787, 19, 18, 3},  {2788, 21, 16, 2},
    {2789, 14, 12, 8},  {2792, 12, 7, 2},   {2794, 11, 9, 4},
    {2795, 10, 8, 4},   {2797, 19, 6, 1},   {2798, 20, 14, 5},
    {2800, 17, 14, 6},  {2802, 18, 14, 3},  {2803, 18, 16, 13},
    {2805, 24, 8, 2},   {2808, 15, 13, 1},  {2810, 23, 10, 9},
    {2811, 10, 9, 7},   {2813, 13, 7, 6},   {2816, 21, 19, 8},
    {2818, 15, 9, 1},   {2819, 10, 6, 5},   {2821, 16, 6, 4},
    {2822, 18, 17, 3},  {2824, 15, 14, 10}, {2827, 19, 10, 6},
    {2829, 9, 6, 4},    {2830, 17, 15, 13}, {2832, 27, 18, 1},
    {2834, 15, 13, 7},  {2835, 20, 13, 5},  {2837, 17, 8, 3},
    {2838, 20, 4, 1},   {2840, 20, 15, 9},  {2843, 7, 3, 2},
    {2845, 10, 9, 1},   {2846, 23, 15, 4},  {2848, 15, 8, 1},
    {2851, 10, 7, 1},   {2853, 14, 8, 1},   {2856, 15, 13, 3},
    {2858, 17, 9, 7},   {2859, 29, 20, 11}, {2861, 20, 16, 10},
    {2864, 21, 17, 15}, {2866, 9, 7, 2},    {2867, 15, 13, 10},
    {2869, 8, 6, 1},    {2870, 12, 11, 1},  {2872, 21, 5, 2},
    {2874, 13, 6, 3},   {2875, 20, 16, 2},  {2877, 10, 7, 4},
    {2880, 13, 10, 6},  {2883, 16, 13, 12}, {2885, 13, 11, 6},
    {2886, 20, 14, 9},  {2888, 11, 4, 1},   {2891, 16, 15, 10},
    {2893, 27, 7, 2},   {2894, 16, 14, 9},  {2896, 29, 6, 1},
    {2899, 19, 10, 1},  {2901, 11, 3, 2},   {2902, 14, 10, 9},
    {2904, 27, 15, 6},  {2907, 17, 14, 1},  {2908, 19, 13, 10},
    {2909, 23, 22, 4},  {2912, 16, 9, 2},   {2915, 18, 7, 5},
    {2917, 16, 15, 4},  {2920, 19, 9, 5},   {2923, 11, 6, 5},
    {2924, 12, 4, 1},   {2925, 12, 4, 3},   {2926, 20, 14, 1},
    {2928, 24, 21, 11}, {2930, 9, 4, 1},    {2931, 13, 8, 1},
    {2932, 14, 7, 6},   {2933, 15, 14, 13}, {2935, 13, 12, 7},
    {2936, 5, 3, 2},    {2939, 30, 17, 5},  {2941, 23, 18, 17},
    {2944, 5, 3, 2},    {2947, 22, 20, 11}, {2948, 17, 3, 2},
    {2949, 17, 7, 4},   {2952, 17, 13, 2},  {2954, 9, 6, 5},
    {2955, 12, 10, 6},  {2957, 19, 14, 3},  {2958, 23, 13, 5},
    {2960, 12, 3, 2},   {2963, 21, 14, 5},  {2965, 25, 22, 15},
    {2968, 29, 27, 4},  {2971, 18, 13, 2},  {2973, 18, 10, 4},
    {2974, 11, 3, 1},   {2976, 21, 10, 3},  {2979, 12, 11, 6},
    {2981, 13, 2, 1},   {2984, 17, 9, 6},   {2987, 8, 3, 2},
    {2989, 18, 9, 7},   {2990, 26, 22, 9},  {2992, 23, 3, 1},
    {2995, 9, 6, 4},    {2997, 12, 6, 5},   {3000, 15, 12, 9},
    {3002, 22, 21, 10}, {3003, 12, 11, 5},  {3005, 14, 12, 5},
    {3006, 15, 9, 6},   {3008, 15, 13, 1},  {3010, 25, 12, 10},
    {3011, 22, 8, 6},   {3013, 23, 6, 2},   {3014, 17, 15, 5},
    {3016, 38, 25, 9},  {3019, 16, 6, 1},   {3021, 22, 10, 1},
    {3022, 14, 12, 1},  {3024, 32, 3, 2},   {3026, 19, 18, 13},
    {3027, 17, 16, 4},  {3029, 11, 6, 4},   {3030, 12, 9, 4},
    {3032, 17, 15, 4},  {3035, 21, 20, 2},  {3037, 7, 6, 3},
    {3040, 27, 21, 3},  {3043, 18, 16, 15}, {3045, 17, 14, 11},
    {3047, 19, 15, 4},  {3048, 18, 3, 2},   {3051, 19, 17, 16},
    {3053, 22, 20, 15}, {3054, 21, 4, 3},   {3055, 13, 7, 1},
    {3056, 5, 4, 3},    {3059, 15, 10, 1},  {3061, 22, 15, 1},
    {3064, 25, 11, 9},  {3067, 18, 16, 13}, {3069, 34, 26, 19},
    {3070, 20, 13, 8},  {3072, 11, 10, 5},  {3074, 17, 9, 3},
    {3075, 16, 14, 10}, {3077, 12, 10, 8},  {3080, 21, 19, 16},
    {3082, 9, 6, 1},    {3083, 23, 20, 18}, {3085, 25, 14, 12},
    {3088, 20, 13, 11}, {3091, 12, 7, 5},   {3093, 14, 11, 8},
    {3096, 33, 29, 14}, {3098, 18, 17, 11}, {3099, 14, 13, 11},
    {3101, 10, 7, 3},   {3104, 23, 9, 5},   {3106, 6, 2, 1},
    {3107, 14, 12, 11}, {3109, 26, 20, 19}, {3112, 17, 15, 1},
    {3114, 19, 15, 4},  {3115, 25, 18, 16}, {3117, 19, 8, 4},
    {3120, 18, 17, 11}, {3122, 22, 7, 6},   {3123, 15, 10, 4},
    {3125, 24, 21, 18}, {3128, 30, 19, 11}, {3131, 22, 8, 7},
    {3133, 6, 4, 3},    {3136, 15, 12, 10}, {3138, 29, 19, 13},
    {3139, 14, 11, 10}, {3141, 23, 18, 12}, {3144, 23, 21, 8},
    {3146, 18, 11, 6},  {3147, 17, 10, 7},  {3149, 22, 3, 2},
    {3152, 21, 17, 6},  {3154, 19, 17, 6},  {3155, 15, 12, 2},
    {3157, 24, 14, 10}, {3158, 19, 9, 5},   {3160, 7, 6, 2},
    {3163, 18, 13, 10}, {3165, 17, 13, 10}, {3166, 18, 10, 1},
    {3168, 33, 31, 18}, {3171, 19, 14, 13}, {3173, 20, 17, 14},
    {3176, 13, 10, 5},  {3179, 22, 12, 2},  {3181, 12, 10, 3},
    {3184, 11, 10, 2},  {3187, 9, 8, 2},    {3189, 19, 18, 12},
    {3190, 12, 4, 1},   {3192, 25, 8, 7},   {3195, 19, 14, 1},
    {3197, 22, 16, 14}, {3200, 11, 6, 4},   {3202, 24, 12, 3},
    {3203, 14, 7, 3},   {3205, 17, 9, 2},   {3206, 15, 8, 6},
    {3208, 16, 13, 3},  {3211, 12, 10, 5},  {3213, 30, 16, 1},
    {3216, 21, 11, 3},  {3218, 10, 9, 8},   {3219, 24, 10, 3},
    {3221, 11, 6, 5},   {3224, 23, 19, 1},  {3226, 9, 7, 5},
    {3227, 8, 7, 6},    {3229, 12, 9, 7},   {3231, 15, 9, 8},
    {3232, 12, 9, 7},   {3235, 21, 11, 8},  {3237, 19, 8, 4},
    {3240, 12, 3, 2},   {3242, 27, 13, 7},  {3243, 22, 13, 5},
    {3245, 11, 10, 4},  {3248, 15, 8, 1},   {3250, 15, 11, 8},
    {3251, 9, 6, 5},    {3253, 10, 5, 3},   {3256, 31, 26, 2},
    {3259, 22, 4, 1},   {3261, 18, 12, 10}, {3262, 15, 13, 7},
    {3264, 17, 5, 2},   {3265, 18, 16, 7},  {3266, 19, 2, 1},
    {3267, 20, 19, 10}, {3269, 22, 3, 2},   {3272, 23, 10, 1},
    {3275, 24, 13, 10}, {3277, 25, 22, 1},  {3278, 22, 12, 1},
    {3280, 16, 15, 6},  {3283, 30, 28, 21}, {3284, 24, 17, 13},
    {3285, 18, 4, 1},   {3288, 21, 18, 11}, {3290, 11, 7, 3},
    {3291, 18, 7, 1},   {3293, 20, 18, 15}, {3296, 19, 14, 13},
    {3299, 18, 17, 14}, {3301, 24, 10, 4},  {3304, 19, 17, 3},
    {3307, 15, 10, 2},  {3309, 23, 20, 6},  {3312, 14, 9, 3},
    {3315, 12, 8, 2},   {3317, 22, 17, 7},  {3320, 17, 10, 4},
    {3323, 28, 14, 10}, {3325, 25, 19, 10}, {3328, 17, 9, 2},
    {3331, 18, 17, 11}, {3333, 16, 8, 7},   {3334, 6, 4, 1},
    {3336, 11, 10, 5},  {3339, 22, 16, 12}, {3340, 11, 7, 5},
    {3341, 25, 19, 12}, {3342, 9, 6, 5},    {3344, 30, 27, 15},
    {3346, 15, 6, 1},   {3347, 23, 18, 16}, {3349, 20, 19, 17},
    {3352, 21, 20, 19}, {3354, 10, 9, 3},   {3355, 10, 6, 4},
    {3357, 24, 17, 15}, {3358, 19, 8, 6},   {3360, 18, 15, 5},
    {3361, 12, 10, 4},  {3362, 11, 7, 4},   {3363, 14, 10, 2},
    {3365, 24, 15, 2},  {3368, 7, 5, 1},    {3370, 15, 10, 1},
    {3371, 30, 29, 18}, {3373, 14, 6, 4},   {3376, 11, 9, 1},
    {3379, 25, 20, 9},  {3381, 7, 4, 1},    {3384, 21, 9, 3},
    {3386, 19, 9, 2},   {3387, 26, 25, 16}, {3389, 17, 15, 4},
    {3392, 23, 13, 6},  {3394, 13, 12, 3},  {3395, 22, 10, 6},
    {3396, 13, 6, 1},   {3397, 19, 4, 1},   {3400, 14, 13, 6},
    {3403, 15, 12, 6},  {3405, 24, 9, 2},   {3406, 22, 13, 12},
    {3408, 16, 15, 6},  {3411, 14, 8, 5},   {3413, 14, 9, 6},
    {3416, 28, 27, 1},  {3419, 12, 9, 3},   {3421, 19, 14, 5},
    {3424, 22, 15, 6},  {3427, 16, 12, 1},  {3428, 17, 16, 13},
    {3429, 16, 12, 6},  {3432, 25, 2, 1},   {3433, 28, 25, 12},
    {3434, 14, 13, 12}, {3435, 15, 14, 5},  {3437, 18, 16, 10},
    {3440, 27, 16, 1},  {3442, 8, 7, 3},    {3443, 26, 19, 3},
    {3445, 21, 5, 2},   {3446, 21, 17, 8},  {3448, 17, 11, 6},
    {3450, 11, 8, 3},   {3451, 19, 14, 9},  {3453, 16, 6, 1},
    {3454, 25, 23, 21}, {3456, 19, 18, 9},  {3459, 19, 18, 12},
    {3461, 20, 10, 7},  {3464, 11, 6, 3},   {3467, 28, 26, 6},
    {3469, 38, 16, 6},  {3472, 17, 14, 7},  {3475, 30, 16, 13},
    {3477, 16, 15, 12}, {3480, 19, 15, 13}, {3482, 15, 5, 4},
    {3483, 12, 5, 2},   {3485, 8, 7, 4},    {3488, 12, 11, 1},
    {3490, 16, 12, 3},  {3491, 19, 14, 7},  {3493, 19, 18, 1},
    {3494, 25, 19, 9},  {3496, 35, 21, 4},  {3499, 29, 24, 4},
    {3501, 15, 8, 2},   {3502, 15, 13, 6},  {3504, 23, 17, 10},
    {3506, 13, 5, 3},   {3507, 21, 14, 6},  {3508, 10, 7, 6},
    {3509, 23, 12, 7},  {3512, 37, 35, 6},  {3514, 11, 9, 4},
    {3515, 17, 10, 9},  {3517, 22, 14, 12}, {3518, 16, 14, 9},
    {3520, 32, 29, 3},  {3523, 23, 12, 2},  {3525, 10, 7, 5},
    {3526, 12, 11, 10}, {3528, 25, 18, 7},  {3530, 10, 9, 5},
    {3531, 18, 3, 1},   {3533, 30, 3, 2},   {3536, 12, 7, 5},
    {3538, 13, 6, 5},   {3539, 16, 2, 1},   {3541, 23, 7, 2},
    {3544, 21, 14, 2},  {3547, 26, 23, 12}, {3549, 14, 6, 3},
    {3552, 15, 9, 6},   {3553, 13, 3, 2},   {3554, 24, 23, 17},
    {3555, 28, 25, 15}, {3557, 14, 8, 5},   {3560, 17, 3, 2},
    {3563, 18, 15, 6},  {3565, 22, 17, 12}, {3566, 8, 6, 1},
    {3567, 24, 20, 12}, {3568, 21, 12, 10}, {3571, 30, 13, 3},
    {3573, 13, 8, 2},   {3576, 19, 10, 3},  {3578, 17, 9, 5},
    {3579, 20, 14, 6},  {3581, 22, 15, 2},  {3584, 25, 12, 10},
    {3586, 19, 14, 8},  {3587, 26, 6, 5},   {3589, 30, 28, 8},
    {3592, 38, 33, 14}, {3595, 28, 14, 2},  {3597, 22, 9, 3},
    {3598, 24, 5, 1},   {3600, 9, 5, 2},    {3602, 10, 2, 1},
    {3603, 23, 11, 6},  {3605, 8, 7, 4},    {3608, 15, 14, 10},
    {3610, 12, 5, 2},   {3611, 18, 7, 1},   {3613, 17, 7, 6},
    {3616, 25, 18, 7},  {3619, 13, 12, 5},  {3621, 22, 21, 16},
    {3623, 10, 9, 7},   {3624, 29, 27, 12}, {3626, 26, 25, 13},
    {3627, 7, 6, 4},    {3629, 15, 10, 2},  {3632, 26, 17, 5},
    {3635, 15, 8, 2},   {3637, 7, 6, 5},    {3638, 28, 8, 1},
    {3640, 20, 15, 10}, {3643, 14, 13, 8},  {3645, 21, 14, 6},
    {3646, 20, 17, 6},  {3648, 23, 7, 2},   {3651, 15, 13, 6},
    {3653, 10, 9, 8},   {3656, 17, 12, 11}, {3659, 18, 13, 2},
    {3661, 16, 11, 6},  {3664, 35, 24, 14}, {3666, 30, 20, 11},
    {3667, 15, 6, 4},   {3669, 22, 7, 4},   {3670, 23, 4, 3},
    {3672, 19, 17, 8},  {3674, 27, 15, 11}, {3675, 30, 10, 9},
    {3677, 25, 20, 7},  {3680, 14, 13, 7},  {3683, 15, 13, 10},
    {3685, 9, 4, 3},    {3688, 32, 13, 11}, {3691, 26, 9, 2},
    {3692, 6, 5, 3},    {3693, 26, 20, 18}, {3696, 36, 33, 22},
    {3699, 24, 21, 5},  {3701, 4, 2, 1},    {3704, 14, 13, 1},
    {3707, 11, 6, 1},   {3709, 16, 14, 6},  {3712, 13, 12, 7},
    {3715, 20, 18, 11}, {3716, 24, 11, 4},  {3717, 18, 15, 4},
    {3718, 23, 18, 10}, {3720, 17, 15, 11}, {3722, 15, 7, 5},
    {3723, 18, 6, 4},   {3724, 10, 9, 8},   {3725, 21, 14, 8},
    {3728, 9, 4, 2},    {3731, 26, 20, 5},  {3733, 8, 7, 3},
    {3736, 33, 22, 18}, {3739, 18, 10, 5},  {3741, 16, 15, 4},
    {3742, 25, 18, 11}, {3744, 27, 14, 2},  {3746, 22, 9, 7},
    {3747, 32, 22, 11}, {3748, 19, 11, 8},  {3749, 15, 4, 1},
    {3752, 9, 4, 2},    {3755, 15, 13, 8},  {3757, 18, 16, 5},
    {3758, 21, 20, 9},  {3760, 23, 9, 1},   {3763, 23, 10, 2},
    {3765, 28, 19, 12}, {3766, 21, 15, 1},  {3768, 14, 7, 2},
    {3771, 13, 10, 9},  {3773, 10, 9, 4},   {3774, 24, 9, 4},
    {3776, 7, 5, 4},    {3778, 9, 5, 1},    {3779, 23, 8, 2},
    {3781, 10, 9, 6},   {3784, 29, 13, 6},  {3787, 14, 6, 1},
    {3789, 16, 15, 4},  {3790, 22, 21, 17}, {3792, 20, 7, 5},
    {3794, 17, 4, 3},   {3795, 8, 7, 5},    {3797, 16, 8, 6},
    {3800, 24, 23, 21}, {3802, 16, 15, 8},  {3803, 18, 15, 6},
    {3805, 18, 12, 9},  {3806, 9, 7, 5},    {3808, 29, 18, 4},
    {3811, 9, 8, 4},    {3813, 11, 8, 7},   {3814, 22, 21, 11},
    {3816, 19, 13, 9},  {3818, 17, 8, 6},   {3819, 16, 6, 3},
    {3820, 20, 11, 3},  {3821, 8, 7, 6},    {3824, 19, 13, 2},
    {3826, 23, 8, 1},   {3827, 18, 13, 8},  {3829, 13, 9, 6},
    {3832, 35, 13, 2},  {3835, 15, 5, 4},   {3837, 22, 18, 3},
    {3840, 27, 9, 1},   {3843, 16, 5, 2},   {3845, 11, 10, 9},
    {3848, 29, 18, 13}, {3851, 29, 24, 19}, {3853, 12, 7, 5},
    {3856, 39, 25, 3},  {3859, 14, 8, 7},   {3861, 31, 10, 2},
    {3862, 18, 16, 5},  {3864, 19, 15, 9},  {3867, 20, 14, 6},
    {3868, 13, 12, 9},  {3869, 24, 22, 13}, {3872, 10, 3, 2},
    {3874, 20, 8, 3},   {3875, 11, 10, 4},  {3877, 17, 11, 4},
    {3878, 19, 9, 2},   {3880, 27, 5, 1},   {3883, 20, 9, 2},
    {3885, 15, 8, 3},   {3888, 45, 42, 6},  {3890, 20, 19, 10},
    {3891, 17, 9, 2},   {3893, 16, 13, 2},  {3896, 15, 7, 5},
    {3898, 25, 9, 1},   {3899, 21, 20, 12}, {3901, 26, 6, 1},
    {3904, 17, 13, 2},  {3907, 15, 4, 1},   {3909, 14, 12, 11},
    {3910, 28, 22, 13}, {3912, 24, 11, 2},  {3915, 25, 22, 9},
    {3917, 16, 12, 11}, {3920, 15, 13, 8},  {3922, 26, 21, 1},
    {3923, 24, 21, 3},  {3925, 18, 16, 5},  {3926, 17, 16, 12},
    {3928, 8, 7, 5},    {3931, 23, 9, 4},   {3933, 30, 19, 3},
    {3934, 28, 12, 5},  {3935, 20, 15, 8},  {3936, 15, 5, 3},
    {3939, 28, 19, 10}, {3940, 21, 5, 2},   {3941, 19, 11, 6},
    {3944, 31, 29, 28}, {3947, 22, 10, 5},  {3949, 28, 22, 3},
    {3952, 11, 6, 5},   {3955, 24, 8, 5},   {3957, 18, 16, 10},
    {3958, 27, 4, 2},   {3960, 29, 15, 2},  {3963, 13, 8, 1},
    {3965, 14, 3, 2},   {3968, 25, 18, 14}, {3971, 10, 7, 2},
    {3973, 13, 11, 8},  {3976, 36, 3, 1},   {3978, 19, 9, 7},
    {3979, 25, 9, 2},   {3980, 16, 9, 4},   {3981, 21, 11, 8},
    {3982, 21, 13, 8},  {3984, 19, 5, 2},   {3986, 12, 8, 7},
    {3987, 11, 4, 2},   {3989, 6, 5, 2},    {3992, 27, 8, 6},
    {3994, 19, 12, 9},  {3995, 12, 9, 8},   {3997, 16, 13, 3},
    {4000, 31, 18, 17}, {4002, 24, 11, 6},  {4003, 14, 12, 5},
    {4005, 17, 12, 2},  {4006, 17, 6, 1},   {4008, 24, 9, 6},
    {4010, 18, 15, 2},  {4011, 28, 15, 9},  {4012, 21, 20, 8},
    {4013, 18, 17, 15}, {4016, 33, 32, 23}, {4019, 7, 6, 1},
    {4021, 24, 19, 16}, {4022, 7, 4, 2},    {4024, 16, 9, 7},
    {4026, 23, 20, 10}, {4027, 9, 7, 4},    {4028, 22, 12, 7},
    {4029, 14, 11, 4},  {4032, 15, 13, 6},  {4035, 25, 16, 1},
    {4037, 21, 14, 10}, {4040, 29, 20, 15}, {4042, 23, 21, 13},
    {4043, 6, 5, 1},    {4045, 22, 12, 10}, {4048, 21, 5, 2},
    {4051, 24, 14, 7},  {4053, 17, 14, 5},  {4056, 21, 17, 6},
    {4059, 13, 6, 3},   {4061, 28, 20, 10}, {4064, 33, 29, 7},
    {4067, 24, 8, 5},   {4069, 18, 9, 1},   {4072, 13, 10, 6},
    {4075, 21, 10, 6},  {4076, 19, 15, 7},  {4077, 24, 18, 2},
    {4078, 29, 18, 16}, {4080, 15, 9, 6},   {4082, 16, 10, 3},
    {4083, 19, 10, 4},  {4085, 12, 11, 6},  {4088, 15, 7, 2},
    {4091, 23, 12, 10}, {4093, 23, 22, 18}, {4096, 27, 15, 1},
    {4099, 19, 18, 2},  {4101, 18, 10, 6},  {4104, 30, 13, 3},
    {4107, 27, 6, 2},   {4109, 19, 12, 1},  {4110, 16, 12, 1},
    {4112, 23, 12, 1},  {4115, 16, 12, 1},  {4117, 19, 18, 13},
    {4118, 22, 12, 1},  {4120, 26, 17, 9},  {4121, 12, 6, 3},
    {4123, 10, 9, 1},   {4128, 31, 2, 1},   {4130, 21, 7, 5},
    {4131, 18, 14, 5},  {4133, 19, 14, 12}, {4136, 31, 29, 15},
    {4139, 15, 7, 2},   {4141, 28, 23, 2},  {4144, 15, 10, 1},
    {4147, 14, 10, 5},  {4149, 28, 8, 2},   {4152, 25, 13, 3},
    {4153, 20, 13, 5},  {4155, 17, 10, 7},  {4157, 19, 17, 10},
    {4160, 27, 18, 12}, {4163, 22, 15, 5},  {4165, 18, 13, 2},
    {4168, 17, 10, 6},  {4171, 24, 8, 1},   {4172, 13, 9, 5},
    {4173, 16, 12, 10}, {4176, 26, 15, 5},  {4178, 18, 2, 1},
    {4179, 14, 6, 3},   {4181, 26, 18, 1},  {4184, 29, 15, 7},
    {4187, 15, 14, 6},  {4189, 16, 10, 1},  {4190, 11, 7, 6},
    {4192, 15, 11, 5},  {4194, 18, 5, 2},   {4195, 21, 13, 2},
    {4197, 16, 13, 6},  {4198, 19, 14, 6},  {4200, 12, 5, 2},
    {4203, 10, 4, 1},   {4205, 8, 4, 3},    {4208, 15, 6, 3},
    {4211, 14, 12, 6},  {4213, 18, 15, 10}, {4216, 13, 3, 2},
    {4219, 24, 6, 4},   {4221, 13, 10, 3},  {4222, 18, 10, 1},
    {4224, 8, 3, 2},    {4226, 30, 23, 10}, {4227, 19, 11, 2},
    {4229, 7, 6, 1},    {4232, 18, 13, 7},  {4235, 15, 3, 2},
    {4237, 21, 19, 14}, {4240, 31, 6, 1},   {4243, 14, 12, 10},
    {4245, 26, 20, 15}, {4248, 11, 6, 2},   {4250, 7, 6, 4},
    {4251, 23, 18, 17}, {4252, 28, 10, 7},  {4253, 21, 12, 11},
    {4254, 11, 5, 3},   {4256, 15, 13, 8},  {4258, 9, 7, 5},
    {4259, 17, 12, 7},  {4261, 23, 20, 18}, {4264, 15, 14, 5},
    {4267, 24, 8, 2},   {4269, 19, 17, 16}, {4270, 18, 11, 7},
    {4272, 11, 8, 1},   {4275, 13, 12, 10}, {4277, 26, 3, 1},
    {4280, 17, 14, 5},  {4283, 30, 27, 15}, {4285, 20, 18, 11},
    {4286, 12, 5, 4},   {4288, 5, 4, 3},    {4291, 18, 14, 10},
    {4292, 22, 18, 7},  {4293, 20, 19, 17}, {4294, 9, 8, 7},
    {4296, 14, 5, 2},   {4297, 15, 4, 2},   {4298, 30, 24, 11},
    {4299, 17, 16, 14}, {4301, 12, 11, 2},  {4304, 12, 7, 2},
    {4306, 9, 5, 1},    {4307, 28, 24, 2},  {4309, 23, 15, 10},
    {4310, 13, 10, 2},  {4312, 21, 5, 2},   {4315, 9, 8, 4},
    {4317, 20, 18, 11}, {4320, 21, 20, 14}, {4323, 16, 6, 3},
    {4325, 22, 16, 12}, {4328, 20, 17, 15}, {4331, 12, 9, 2},
    {4333, 21, 17, 10}, {4334, 17, 12, 6},  {4336, 18, 11, 5},
    {4339, 25, 22, 16}, {4341, 28, 14, 11}, {4344, 8, 7, 5},
    {4347, 10, 8, 5},   {4349, 13, 11, 10}, {4352, 33, 27, 20},
    {4354, 11, 5, 3},   {4355, 24, 23, 14}, {4357, 16, 7, 6},
    {4360, 21, 4, 2},   {4363, 15, 8, 2},   {4365, 22, 12, 10},
    {4368, 27, 12, 6},  {4370, 26, 15, 11}, {4371, 18, 15, 5},
    {4373, 8, 7, 1},    {4376, 18, 7, 1},   {4379, 16, 13, 8},
    {4381, 22, 19, 12}, {4384, 27, 19, 17}, {4385, 19, 17, 10},
    {4387, 23, 18, 4},  {4388, 18, 8, 5},   {4389, 15, 10, 3},
    {4392, 20, 19, 5},  {4393, 15, 13, 9},  {4395, 33, 25, 18},
    {4397, 25, 18, 11}, {4400, 37, 35, 3},  {4403, 15, 14, 4},
    {4405, 21, 13, 2},  {4407, 21, 3, 2},   {4408, 9, 8, 7},
    {4411, 22, 6, 3},   {4413, 23, 22, 20}, {4414, 19, 6, 4},
    {4416, 31, 10, 6},  {4418, 23, 19, 7},  {4419, 13, 11, 8},
    {4421, 32, 22, 18}, {4424, 21, 19, 13}, {4426, 11, 6, 1},
    {4427, 27, 22, 16}, {4429, 16, 6, 4},   {4430, 14, 4, 1},
    {4432, 25, 5, 3},   {4435, 12, 5, 2},   {4437, 20, 8, 2},
    {4438, 8, 6, 1},    {4440, 26, 21, 14}, {4443, 16, 6, 4},
    {4448, 11, 9, 8},   {4451, 17, 12, 8},  {4453, 26, 15, 2},
    {4456, 24, 9, 7},   {4459, 13, 9, 6},   {4461, 15, 10, 6},
    {4462, 21, 16, 3},  {4464, 13, 3, 1},   {4467, 30, 29, 11},
    {4469, 19, 18, 16}, {4472, 30, 7, 2},   {4475, 16, 5, 3},
    {4477, 37, 22, 4},  {4478, 25, 21, 8},  {4480, 28, 21, 15},
    {4483, 8, 7, 2},    {4485, 8, 2, 1},    {4488, 38, 35, 13},
    {4491, 18, 7, 3},   {4493, 22, 8, 1},   {4496, 17, 10, 7},
    {4498, 23, 8, 1},   {4499, 27, 8, 5},   {4501, 7, 6, 1},
    {4503, 7, 5, 1},    {4504, 12, 9, 6},   {4506, 13, 9, 3},
    {4507, 14, 9, 4},   {4509, 14, 5, 4},   {4510, 20, 13, 9},
    {4512, 12, 3, 1},   {4515, 8, 6, 3},    {4517, 21, 16, 11},
    {4520, 39, 25, 23}, {4523, 17, 12, 7},  {4525, 16, 11, 8},
    {4528, 23, 13, 9},  {4531, 8, 5, 1},    {4532, 25, 22, 20},
    {4533, 15, 4, 1},   {4536, 25, 11, 7},  {4539, 11, 8, 6},
    {4541, 12, 11, 2},  {4544, 25, 14, 7},  {4546, 27, 23, 16},
    {4547, 6, 5, 1},    {4549, 23, 12, 7},  {4552, 34, 27, 18},
    {4555, 25, 22, 12}, {4556, 26, 21, 12}, {4560, 14, 13, 7},
    {4563, 20, 11, 1},  {4565, 18, 6, 2},   {4568, 22, 17, 6},
    {4571, 10, 6, 1},   {4573, 24, 15, 6},  {4576, 26, 19, 9},
    {4577, 9, 8, 4},    {4579, 10, 6, 2},   {4580, 21, 12, 5},
    {4581, 27, 13, 10}, {4582, 14, 12, 1},  {4584, 19, 15, 9},
    {4587, 13, 6, 1},   {4589, 26, 18, 16}, {4592, 21, 16, 11},
    {4593, 22, 16, 12}, {4594, 30, 11, 4},  {4595, 15, 14, 1},
    {4597, 34, 13, 12}, {4598, 32, 28, 9},  {4600, 17, 14, 1},
    {4603, 24, 22, 5},  {4605, 8, 5, 2},    {4608, 23, 20, 13},
    {4611, 23, 13, 10}, {4613, 27, 14, 10}, {4616, 27, 23, 5},
    {4619, 10, 8, 7},   {4621, 28, 15, 4},  {4622, 21, 20, 17},
    {4624, 26, 23, 10}, {4627, 16, 8, 5},   {4629, 20, 3, 2},
    {4630, 14, 3, 2},   {4631, 25, 17, 7},  {4632, 19, 16, 2},
    {4635, 21, 16, 1},  {4637, 18, 17, 3},  {4638, 30, 24, 15},
    {4640, 25, 8, 7},   {4643, 15, 13, 10}, {4645, 24, 11, 5},
    {4646, 16, 13, 9},  {4648, 8, 5, 3},    {4651, 29, 24, 8},
    {4653, 30, 21, 13}, {4654, 21, 20, 1},  {4656, 27, 25, 4},
    {4659, 8, 7, 5},    {4661, 26, 21, 2},  {4664, 37, 6, 5},
    {4666, 24, 17, 11}, {4667, 36, 25, 20}, {4669, 18, 14, 10},
    {4670, 29, 12, 2},  {4672, 27, 25, 23}, {4674, 10, 5, 4},
    {4675, 26, 25, 21}, {4677, 19, 11, 10}, {4680, 10, 9, 3},
    {4683, 35, 28, 14}, {4685, 24, 10, 1},  {4688, 13, 7, 6},
    {4691, 26, 21, 3},  {4693, 26, 13, 6},  {4696, 18, 17, 11},
    {4699, 36, 7, 5},   {4701, 23, 21, 18}, {4702, 17, 3, 2},
    {4704, 11, 8, 1},   {4707, 17, 16, 11}, {4709, 21, 12, 11},
    {4712, 25, 16, 6},  {4714, 19, 17, 6},  {4715, 26, 24, 7},
    {4717, 30, 17, 16}, {4718, 16, 14, 1},  {4720, 24, 19, 9},
    {4721, 10, 9, 7},   {4722, 24, 19, 18}, {4723, 19, 14, 2},
    {4725, 13, 10, 8},  {4728, 26, 21, 14}, {4730, 19, 12, 10},
    {4731, 14, 6, 5},   {4733, 14, 9, 5},   {4736, 15, 10, 1},
    {4738, 12, 7, 5},   {4739, 27, 26, 6},  {4741, 21, 10, 8},
    {4744, 19, 14, 6},  {4747, 25, 24, 8},  {4749, 29, 19, 16},
    {4752, 29, 5, 2},   {4755, 19, 6, 4},   {4757, 30, 22, 17},
    {4760, 28, 11, 9},  {4762, 7, 6, 4},    {4763, 17, 16, 7},
    {4765, 30, 19, 7},  {4766, 25, 22, 21}, {4768, 25, 17, 3},
    {4771, 24, 5, 4},   {4773, 26, 12, 5},  {4774, 16, 13, 6},
    {4776, 27, 23, 6},  {4778, 18, 5, 3},   {4779, 14, 8, 1},
    {4781, 22, 19, 5},  {4784, 30, 29, 7},  {4786, 20, 3, 1},
    {4787, 19, 6, 5},   {4789, 14, 7, 2},   {4792, 29, 18, 4},
    {4794, 19, 12, 7},  {4795, 23, 7, 2},   {4796, 18, 8, 7},
    {4797, 15, 14, 3},  {4800, 29, 19, 11}, {4803, 7, 5, 4},
    {4804, 18, 15, 4},  {4808, 13, 4, 2},   {4811, 13, 10, 2},
    {4812, 22, 15, 13}, {4813, 18, 7, 6},   {4814, 22, 10, 9},
    {4816, 33, 31, 25}, {4819, 28, 25, 7},  {4821, 16, 13, 2},
    {4822, 33, 17, 13}, {4824, 25, 16, 3},  {4826, 24, 19, 17},
    {4827, 29, 20, 14}, {4829, 20, 14, 2},  {4832, 33, 30, 5},
    {4834, 19, 17, 2},  {4835, 28, 10, 5},  {4837, 14, 9, 1},
    {4838, 17, 12, 7},  {4840, 25, 21, 2},  {4841, 19, 13, 3},
    {4842, 33, 29, 18}, {4843, 22, 21, 3},  {4844, 16, 15, 1},
    {4845, 38, 21, 17}, {4848, 28, 27, 6},  {4851, 24, 17, 2},
    {4853, 17, 7, 4},   {4856, 27, 23, 21}, {4859, 25, 22, 14},
    {4861, 15, 7, 6},   {4862, 15, 8, 1},   {4864, 29, 22, 17},
    {4866, 23, 15, 2},  {4867, 18, 11, 1},  {4869, 22, 16, 4},
    {4872, 24, 9, 2},   {4875, 26, 16, 10}, {4877, 23, 20, 6},
    {4880, 32, 21, 7},  {4882, 28, 7, 4},   {4883, 26, 8, 2},
    {4885, 27, 21, 12}, {4888, 35, 21, 8},  {4891, 25, 19, 16},
    {4893, 15, 12, 2},  {4896, 21, 19, 12}, {4898, 25, 12, 2},
    {4899, 33, 28, 6},  {4901, 20, 18, 14}, {4904, 27, 18, 15},
    {4906, 22, 21, 6},  {4907, 14, 10, 3},  {4909, 30, 16, 11},
    {4912, 30, 29, 7},  {4915, 16, 9, 1},   {4917, 34, 5, 4},
    {4920, 11, 9, 5},   {4923, 10, 3, 1},   {4925, 19, 6, 2},
    {4928, 29, 9, 3},   {4931, 23, 21, 18}, {4933, 13, 11, 8},
    {4936, 27, 22, 15}, {4938, 19, 9, 3},   {4939, 24, 8, 1},
    {4941, 17, 14, 1},  {4942, 13, 12, 9},  {4944, 19, 6, 4},
    {4946, 23, 9, 1},   {4947, 24, 6, 4},   {4949, 30, 25, 2},
    {4952, 29, 11, 5},  {4954, 15, 10, 7},  {4955, 27, 12, 5},
    {4957, 11, 6, 5},   {4960, 25, 20, 6},  {4962, 23, 11, 6},
    {4963, 20, 17, 15}, {4964, 13, 10, 5},  {4965, 13, 12, 6},
    {4968, 26, 25, 17}, {4971, 12, 6, 2},   {4973, 26, 22, 16},
    {4976, 42, 7, 1},   {4979, 29, 11, 8},  {4981, 33, 15, 12},
    {4984, 23, 16, 9},  {4987, 10, 7, 4},   {4989, 15, 8, 2},
    {4990, 23, 19, 3},  {4992, 15, 8, 6},   {4995, 28, 25, 19},
    {4996, 18, 15, 13}, {4997, 31, 11, 6},  {4999, 13, 11, 9},
    {5000, 17, 15, 7},  {5002, 33, 13, 8},  {5003, 18, 16, 8},
    {5005, 22, 13, 6},  {5006, 12, 3, 2},   {5008, 21, 10, 3},
    {5010, 17, 8, 6},   {5011, 29, 20, 15}, {5013, 38, 35, 15},
    {5016, 15, 12, 9},  {5019, 23, 19, 2},  {5021, 14, 11, 2},
    {5024, 33, 23, 14}, {5026, 20, 15, 11}, {5027, 28, 9, 5},
    {5029, 22, 3, 2},   {5032, 25, 6, 2},   {5035, 24, 21, 10},
    {5037, 22, 15, 3},  {5040, 24, 5, 3},   {5043, 23, 17, 8},
    {5045, 25, 24, 2},  {5046, 17, 11, 4},  {5048, 34, 15, 10},
    {5051, 22, 17, 15}, {5053, 26, 18, 11}, {5056, 22, 9, 6},
    {5059, 20, 19, 14}, {5061, 16, 11, 7},  {5064, 21, 12, 11},
    {5067, 12, 6, 2},   {5069, 18, 5, 3},   {5070, 16, 13, 7},
    {5072, 25, 5, 3},   {5075, 18, 15, 5},  {5077, 18, 11, 5},
    {5080, 21, 7, 6},   {5083, 25, 19, 6},  {5085, 20, 14, 13},
    {5086, 23, 18, 14}, {5088, 22, 21, 3},  {5091, 18, 13, 10},
    {5093, 15, 6, 2},   {5096, 19, 18, 13}, {5099, 24, 2, 1},
    {5101, 18, 16, 15}, {5102, 22, 18, 11}, {5104, 30, 13, 2},
    {5107, 17, 2, 1},   {5109, 26, 19, 7},  {5112, 42, 33, 9},
    {5115, 14, 8, 5},   {5117, 19, 14, 6},  {5120, 33, 27, 5},
    {5123, 11, 8, 5},   {5125, 23, 18, 4},  {5126, 22, 14, 9},
    {5128, 24, 15, 6},  {5131, 21, 7, 4},   {5133, 15, 11, 8},
    {5136, 27, 25, 4},  {5139, 36, 22, 1},  {5141, 24, 14, 5},
    {5142, 12, 11, 3},  {5144, 15, 11, 5},  {5147, 12, 10, 2},
    {5149, 30, 22, 2},  {5152, 15, 8, 1},   {5154, 16, 5, 2},
    {5155, 15, 14, 1},  {5156, 14, 6, 1},   {5157, 8, 6, 3},
    {5158, 21, 20, 2},  {5160, 35, 31, 13}, {5163, 20, 17, 5},
    {5164, 12, 7, 3},   {5165, 18, 16, 10}, {5168, 15, 11, 5},
    {5171, 18, 3, 2},   {5173, 23, 22, 2},  {5176, 24, 21, 3},
    {5178, 15, 13, 10}, {5179, 30, 12, 9},  {5181, 23, 7, 2},
    {5182, 16, 13, 6},  {5184, 20, 11, 5},  {5187, 16, 8, 5},
    {5189, 28, 22, 2},  {5192, 11, 8, 2},   {5194, 25, 15, 13},
    {5195, 27, 11, 6},  {5197, 20, 15, 7},  {5198, 27, 10, 8},
    {5200, 33, 27, 19}, {5203, 14, 4, 3},   {5205, 9, 8, 6},
    {5208, 23, 22, 2},  {5211, 20, 19, 1},  {5213, 19, 8, 7},
    {5214, 12, 9, 8},   {5216, 27, 19, 2},  {5219, 23, 3, 2},
    {5221, 12, 11, 8},  {5224, 29, 15, 1},  {5227, 24, 10, 7},
    {5229, 28, 19, 13}, {5230, 13, 12, 11}, {5232, 21, 11, 2},
    {5235, 16, 6, 1},   {5237, 12, 10, 7},  {5240, 33, 27, 21},
    {5243, 10, 9, 1},   {5245, 28, 23, 3},  {5248, 27, 18, 1},
    {5251, 35, 25, 14}, {5253, 24, 17, 14}, {5256, 9, 5, 2},
    {5259, 10, 8, 4},   {5261, 22, 10, 9},  {5262, 22, 20, 1},
    {5264, 16, 11, 9},  {5267, 31, 25, 14}, {5269, 23, 6, 1},
    {5272, 29, 18, 10}, {5274, 17, 11, 10}, {5275, 16, 10, 1},
    {5277, 9, 7, 6},    {5280, 13, 11, 1},  {5281, 29, 9, 4},
    {5282, 13, 10, 9},  {5283, 25, 17, 2},  {5284, 16, 9, 5},
    {5285, 25, 24, 23}, {5286, 20, 14, 7},  {5288, 11, 10, 1},
    {5290, 17, 9, 4},   {5291, 24, 18, 5},  {5293, 25, 9, 2},
    {5294, 14, 3, 1},   {5296, 8, 3, 2},    {5299, 17, 12, 1},
    {5301, 14, 8, 1},   {5304, 27, 6, 5},   {5306, 15, 4, 1},
    {5307, 12, 10, 4},  {5308, 18, 14, 11}, {5309, 7, 6, 1},
    {5312, 22, 3, 2},   {5315, 18, 14, 3},  {5317, 17, 4, 3},
    {5320, 19, 17, 3},  {5323, 15, 11, 2},  {5325, 18, 14, 11},
    {5328, 30, 27, 9},  {5331, 18, 5, 3},   {5333, 25, 24, 19},
    {5335, 16, 15, 11}, {5336, 23, 21, 8},  {5339, 17, 16, 15},
    {5341, 22, 15, 13}, {5342, 27, 9, 3},   {5344, 17, 11, 10},
    {5347, 24, 22, 11}, {5349, 16, 13, 10}, {5350, 15, 14, 5},
    {5352, 23, 20, 1},  {5354, 22, 7, 6},   {5355, 16, 11, 2},
    {5357, 29, 24, 10}, {5360, 19, 11, 1},  {5362, 15, 7, 1},
    {5363, 20, 7, 1},   {5365, 14, 13, 8},  {5366, 20, 3, 2},
    {5368, 19, 18, 3},  {5371, 18, 15, 6},  {5373, 20, 14, 13},
    {5374, 24, 19, 14}, {5376, 7, 4, 1},    {5379, 23, 17, 4},
    {5381, 13, 11, 10}, {5382, 26, 24, 17}, {5384, 17, 15, 4},
    {5387, 28, 21, 8},  {5389, 18, 10, 9},  {5392, 7, 3, 2},
    {5395, 18, 13, 1},  {5397, 9, 5, 2},    {5400, 33, 17, 3},
    {5402, 18, 15, 9},  {5403, 14, 11, 6},  {5405, 17, 14, 12},
    {5408, 30, 23, 1},  {5411, 28, 10, 3},  {5413, 21, 18, 7},
    {5414, 25, 2, 1},   {5416, 34, 31, 19}, {5419, 25, 24, 23},
    {5421, 30, 29, 6},  {5424, 16, 15, 13}, {5426, 18, 11, 8},
    {5427, 18, 12, 11}, {5429, 40, 22, 5},  {5432, 37, 34, 23},
    {5434, 19, 18, 15}, {5435, 24, 18, 14}, {5437, 21, 8, 2},
    {5440, 24, 15, 10}, {5443, 20, 10, 6},  {5445, 20, 19, 7},
    {5448, 43, 33, 15}, {5451, 30, 19, 16}, {5453, 13, 4, 2},
    {5456, 21, 14, 10}, {5459, 36, 20, 2},  {5461, 30, 28, 5},
    {5462, 15, 11, 1},  {5464, 15, 5, 2},   {5466, 32, 23, 10},
    {5467, 22, 8, 5},   {5469, 32, 23, 2},  {5470, 13, 12, 1},
    {5472, 21, 15, 3},  {5475, 18, 16, 13}, {5476, 30, 23, 11},
    {5477, 23, 18, 10}, {5480, 13, 10, 3},  {5482, 20, 17, 9},
    {5483, 22, 2, 1},   {5485, 16, 8, 2},   {5486, 14, 9, 8},
    {5488, 21, 19, 9},  {5491, 24, 14, 1},  {5493, 15, 12, 7},
    {5496, 13, 11, 4},  {5498, 19, 12, 7},  {5499, 29, 21, 18},
    {5501, 27, 19, 2},  {5504, 20, 19, 1},  {5506, 23, 20, 12},
    {5507, 15, 10, 7},  {5509, 11, 9, 4},   {5510, 19, 15, 2},
    {5512, 13, 11, 6},  {5515, 19, 3, 2},   {5517, 40, 13, 11},
    {5518, 26, 23, 7},  {5520, 10, 5, 2},   {5522, 26, 15, 2},
    {5523, 26, 20, 13}, {5525, 24, 21, 6},  {5528, 25, 18, 14},
    {5531, 20, 15, 1},  {5533, 19, 16, 10}, {5536, 20, 5, 3},
    {5538, 28, 22, 7},  {5539, 14, 9, 7},   {5541, 15, 9, 4},
    {5544, 11, 10, 2},  {5546, 23, 12, 6},  {5547, 22, 4, 3},
    {5549, 33, 11, 10}, {5552, 15, 9, 8},   {5554, 16, 5, 2},
    {5555, 24, 20, 5},  {5557, 39, 35, 8},  {5560, 55, 46, 10},
    {5563, 22, 19, 13}, {5565, 17, 14, 5},  {5568, 33, 22, 7},
    {5570, 17, 16, 3},  {5571, 27, 11, 2},  {5573, 18, 11, 6},
    {5576, 27, 23, 6},  {5578, 15, 9, 3},   {5579, 25, 20, 12},
    {5581, 20, 6, 4},   {5584, 13, 9, 7},   {5587, 21, 16, 7},
    {5589, 29, 27, 8},  {5592, 29, 20, 7},  {5595, 21, 12, 2},
    {5597, 20, 6, 3},   {5600, 8, 5, 3},    {5603, 22, 20, 4},
    {5605, 32, 23, 1},  {5608, 22, 5, 2},   {5611, 20, 9, 7},
    {5613, 20, 17, 14}, {5614, 6, 3, 1},    {5616, 30, 15, 6},
    {5618, 17, 15, 2},  {5619, 11, 8, 6},   {5621, 12, 11, 4},
    {5624, 21, 18, 14}, {5627, 13, 9, 8},   {5629, 23, 12, 4},
    {5632, 17, 15, 5},  {5635, 17, 10, 3},  {5637, 8, 6, 5},
    {5638, 20, 17, 13}, {5640, 29, 27, 23}, {5642, 33, 23, 2},
    {5643, 10, 3, 1},   {5645, 19, 16, 2},  {5646, 23, 8, 5},
    {5647, 17, 16, 9},  {5648, 21, 15, 8},  {5651, 10, 7, 2},
    {5653, 16, 11, 5},  {5656, 27, 12, 9},  {5658, 9, 4, 3},
    {5659, 12, 8, 2},   {5661, 26, 14, 10}, {5664, 31, 29, 11},
    {5667, 10, 8, 1},   {5668, 34, 27, 3},  {5669, 27, 4, 1},
    {5672, 26, 25, 10}, {5674, 18, 15, 13}, {5675, 17, 15, 4},
    {5677, 27, 26, 2},  {5678, 31, 21, 8},  {5680, 40, 17, 2},
    {5682, 35, 16, 1},  {5683, 22, 18, 11}, {5685, 10, 7, 5},
    {5686, 27, 7, 3},   {5688, 41, 20, 11}, {5691, 21, 15, 4},
    {5693, 24, 4, 2},   {5696, 26, 21, 14}, {5697, 19, 8, 2},
    {5698, 14, 9, 5},   {5699, 12, 8, 5},   {5701, 16, 6, 5},
    {5702, 22, 15, 8},  {5704, 27, 25, 14}, {5707, 24, 21, 16},
    {5709, 22, 9, 1},   {5712, 10, 9, 3},   {5715, 28, 6, 3},
    {5716, 14, 13, 8},  {5717, 21, 6, 5},   {5720, 32, 11, 2},
    {5722, 19, 18, 14}, {5723, 28, 19, 6},  {5725, 22, 9, 3},
    {5728, 25, 24, 7},  {5730, 25, 18, 10}, {5731, 13, 11, 8},
    {5733, 10, 8, 5},   {5736, 25, 18, 10}, {5738, 30, 17, 4},
    {5739, 18, 15, 13}, {5741, 16, 6, 4},   {5744, 21, 11, 8},
    {5747, 13, 4, 2},   {5748, 18, 17, 2},  {5749, 26, 19, 18},
    {5752, 21, 20, 7},  {5755, 8, 7, 2},    {5757, 22, 18, 16},
    {5760, 29, 23, 10}, {5763, 18, 13, 1},  {5765, 17, 10, 7},
    {5766, 13, 10, 4},  {5768, 21, 14, 3},  {5770, 9, 7, 2},
    {5771, 16, 14, 13}, {5773, 12, 11, 8},  {5774, 15, 5, 4},
    {5776, 19, 12, 1},  {5779, 21, 7, 4},   {5781, 24, 14, 6},
    {5784, 25, 22, 6},  {5786, 20, 19, 7},  {5787, 30, 20, 15},
    {5789, 21, 14, 12}, {5792, 33, 13, 11}, {5795, 15, 14, 3},
    {5797, 33, 26, 23}, {5800, 25, 7, 6},   {5803, 16, 15, 10},
    {5804, 13, 8, 1},   {5805, 10, 7, 3},   {5808, 43, 10, 1},
    {5811, 21, 17, 2},  {5813, 14, 13, 1},  {5816, 17, 15, 7},
    {5821, 18, 11, 4},  {5824, 23, 17, 10}, {5827, 28, 16, 10},
    {5828, 17, 8, 5},   {5829, 24, 6, 5},   {5830, 27, 23, 11},
    {5832, 21, 18, 13}, {5835, 13, 5, 2},   {5837, 13, 9, 2},
    {5840, 23, 14, 2},  {5842, 21, 18, 7},  {5843, 20, 7, 6},
    {5845, 18, 5, 1},   {5846, 20, 17, 3},  {5848, 19, 8, 6},
    {5851, 9, 6, 1},    {5852, 15, 13, 10}, {5853, 46, 36, 18},
    {5856, 23, 15, 6},  {5859, 28, 25, 16}, {5861, 16, 2, 1},
    {5862, 18, 3, 2},   {5864, 27, 11, 10}, {5866, 30, 23, 2},
    {5867, 17, 6, 2},   {5869, 26, 23, 4},  {5870, 15, 9, 5},
    {5872, 19, 15, 10}, {5875, 20, 13, 9},  {5877, 37, 36, 30},
    {5878, 24, 9, 1},   {5880, 21, 10, 3},  {5883, 19, 6, 4},
    {5884, 14, 11, 10}, {5885, 38, 29, 14}, {5888, 23, 10, 4},
    {5890, 13, 8, 3},   {5891, 20, 15, 9},  {5893, 28, 3, 1},
    {5894, 21, 15, 2},  {5896, 30, 23, 1},  {5899, 35, 21, 10},
    {5901, 23, 15, 12}, {5904, 32, 19, 5},  {5906, 23, 18, 15},
    {5907, 25, 17, 2},  {5909, 34, 24, 14}, {5912, 33, 22, 13},
    {5914, 25, 13, 7},  {5915, 30, 29, 28}, {5916, 20, 15, 3},
    {5917, 21, 12, 6},  {5918, 13, 12, 8},  {5920, 16, 15, 6},
    {5923, 18, 10, 5},  {5925, 24, 7, 3},   {5928, 16, 7, 2},
    {5929, 16, 6, 4},   {5930, 17, 13, 10}, {5931, 30, 26, 14},
    {5933, 15, 11, 6},  {5934, 28, 25, 4},  {5936, 27, 4, 1},
    {5938, 23, 7, 3},   {5939, 12, 5, 3},   {5941, 20, 18, 11},
    {5944, 23, 21, 14}, {5945, 9, 5, 3},    {5946, 33, 20, 3},
    {5947, 14, 11, 4},  {5949, 23, 21, 18}, {5950, 32, 13, 8},
    {5952, 25, 23, 2},  {5955, 28, 8, 5},   {5957, 18, 6, 2},
    {5960, 37, 35, 25}, {5963, 35, 28, 6},  {5965, 27, 19, 12},
    {5968, 21, 14, 4},  {5971, 11, 3, 2},   {5973, 20, 8, 3},
    {5974, 23, 19, 3},  {5976, 18, 9, 6},   {5979, 29, 25, 2},
    {5981, 21, 14, 13}, {5984, 17, 7, 1},   {5987, 23, 18, 4},
    {5989, 36, 20, 2},  {5992, 29, 9, 2},   {5994, 17, 9, 5},
    {5995, 26, 25, 20}, {5996, 20, 18, 3},  {5997, 10, 8, 1},
    {6000, 23, 21, 12}, {6003, 25, 19, 14}, {6005, 22, 13, 8},
    {6008, 30, 27, 15}, {6011, 36, 30, 5},  {6013, 31, 24, 16},
    {6014, 21, 14, 9},  {6016, 35, 34, 2},  {6019, 19, 18, 11},
    {6021, 32, 22, 17}, {6024, 39, 33, 26}, {6028, 18, 8, 7},
    {6029, 22, 17, 3},  {6032, 44, 21, 14}, {6035, 28, 14, 13},
    {6037, 10, 7, 1},   {6038, 21, 17, 11}, {6040, 25, 11, 5},
    {6042, 21, 3, 2},   {6043, 15, 12, 6},  {6045, 19, 16, 10},
    {6046, 16, 3, 2},   {6048, 17, 15, 8},  {6051, 34, 17, 1},
    {6053, 15, 11, 2},  {6054, 24, 22, 21}, {6056, 7, 6, 1},
    {6059, 39, 34, 20}, {6061, 31, 25, 6},  {6062, 15, 7, 2},
    {6063, 28, 24, 9},  {6064, 23, 18, 11}, {6067, 30, 18, 10},
    {6069, 21, 5, 2},   {6072, 28, 15, 13}, {6075, 26, 24, 6},
    {6077, 17, 11, 8},  {6080, 19, 8, 6},   {6082, 20, 2, 1},
    {6083, 33, 26, 6},  {6085, 16, 13, 7},  {6088, 23, 10, 3},
    {6091, 15, 9, 4},   {6093, 14, 9, 6},   {6096, 20, 11, 2},
    {6099, 22, 16, 14}, {6101, 24, 20, 11}, {6102, 22, 17, 16},
    {6104, 13, 11, 6},  {6107, 23, 12, 10}, {6109, 17, 11, 4},
    {6110, 26, 19, 13}, {6112, 35, 12, 1},  {6114, 11, 6, 5},
    {6115, 22, 19, 10}, {6117, 34, 33, 30}, {6120, 4, 3, 1},
    {6122, 32, 24, 11}, {6123, 25, 22, 6},  {6125, 22, 8, 6},
    {6126, 20, 15, 5},  {6127, 31, 9, 4},   {6128, 34, 15, 2},
    {6130, 24, 22, 7},  {6131, 36, 34, 7},  {6133, 16, 7, 3},
    {6136, 17, 7, 5},   {6138, 26, 11, 10}, {6139, 14, 13, 3},
    {6141, 16, 13, 6},  {6144, 26, 7, 1},   {6147, 29, 24, 6},
    {6148, 25, 18, 16}, {6149, 23, 20, 14}, {6152, 28, 27, 13},
    {6154, 23, 12, 10}, {6155, 34, 33, 4},  {6157, 17, 16, 14},
    {6158, 14, 6, 5},   {6160, 38, 15, 10}, {6163, 17, 12, 7},
    {6165, 25, 15, 12}, {6166, 5, 4, 2},    {6168, 20, 11, 2},
    {6171, 22, 10, 8},  {6173, 10, 9, 1},   {6176, 29, 15, 1},
    {6179, 16, 8, 1},   {6181, 22, 13, 6},  {6184, 39, 13, 12},
    {6186, 15, 6, 4},   {6187, 26, 25, 18}, {6189, 26, 8, 5},
    {6192, 20, 5, 2},   {6194, 21, 9, 2},   {6195, 30, 24, 23},
    {6197, 25, 14, 4},  {6200, 29, 10, 7},  {6203, 20, 15, 10},
    {6205, 14, 12, 1},  {6206, 22, 7, 5},   {6208, 25, 23, 14},
    {6211, 19, 18, 5},  {6212, 27, 16, 14}, {6213, 15, 14, 12},
    {6216, 39, 30, 9},  {6219, 17, 14, 4},  {6221, 21, 12, 3},
    {6224, 13, 4, 2},   {6227, 18, 9, 8},   {6229, 12, 7, 6},
    {6232, 17, 3, 1},   {6234, 22, 18, 7},  {6235, 23, 18, 10},
    {6237, 32, 14, 9},  {6240, 11, 10, 2},  {6242, 32, 28, 27},
    {6243, 17, 10, 5},  {6245, 28, 26, 4},  {6248, 18, 7, 2},
    {6251, 35, 34, 31}, {6252, 15, 10, 6},  {6253, 14, 10, 2},
    {6254, 30, 29, 4},  {6256, 11, 10, 5},  {6259, 18, 12, 3},
    {6261, 17, 14, 12}, {6264, 17, 16, 7},  {6266, 10, 7, 5},
    {6267, 26, 8, 5},   {6269, 31, 8, 3},   {6272, 17, 10, 6},
    {6275, 15, 14, 5},  {6277, 16, 10, 6},  {6280, 9, 7, 5},
    {6283, 14, 11, 4},  {6284, 16, 13, 8},  {6285, 26, 14, 5},
    {6288, 34, 25, 5},  {6291, 29, 6, 5},   {6293, 30, 20, 18},
    {6296, 35, 19, 10}, {6298, 15, 10, 1},  {6299, 25, 13, 2},
    {6301, 17, 8, 7},   {6304, 13, 3, 1},   {6307, 28, 27, 5},
    {6309, 23, 13, 10}, {6310, 11, 7, 2},   {6312, 35, 33, 14},
    {6314, 16, 11, 7},  {6315, 42, 40, 31}, {6317, 11, 6, 1},
    {6320, 29, 28, 10}, {6323, 19, 15, 14}, {6325, 18, 4, 2},
    {6327, 22, 15, 10}, {6328, 15, 6, 1},   {6329, 21, 16, 1},
    {6331, 21, 12, 1},  {6333, 22, 14, 2},  {6336, 22, 15, 9},
    {6339, 17, 10, 7},  {6341, 16, 14, 9},  {6344, 21, 15, 2},
    {6346, 31, 22, 16}, {6347, 11, 6, 2},   {6349, 8, 6, 1},
    {6350, 11, 7, 6},   {6352, 21, 11, 4},  {6355, 30, 27, 16},
    {6357, 37, 27, 12}, {6360, 13, 11, 1},  {6361, 31, 17, 3},
    {6363, 32, 23, 14}, {6365, 21, 14, 5},  {6368, 31, 9, 1},
    {6369, 19, 11, 10}, {6371, 30, 18, 17}, {6373, 21, 18, 9},
    {6374, 12, 9, 8},   {6376, 28, 27, 5},  {6379, 30, 10, 5},
    {6381, 33, 26, 3},  {6384, 34, 29, 7},  {6387, 28, 14, 4},
    {6389, 19, 17, 4},  {6392, 39, 34, 10}, {6395, 29, 7, 4},
    {6397, 17, 6, 4},   {6398, 12, 11, 10}, {6400, 37, 12, 3},
    {6403, 15, 10, 4},  {6405, 18, 10, 7},  {6408, 31, 12, 10},
    {6410, 18, 15, 2},  {6411, 23, 18, 14}, {6413, 18, 17, 10},
    {6414, 34, 15, 9},  {6416, 29, 15, 7},  {6419, 9, 7, 4},
    {6421, 24, 16, 14}, {6423, 26, 19, 1},  {6424, 29, 18, 5},
    {6427, 35, 29, 18}, {6429, 18, 16, 13}, {6432, 26, 13, 7},
    {6435, 15, 9, 4},   {6437, 23, 18, 4},  {6440, 21, 18, 14},
    {6442, 19, 18, 5},  {6443, 25, 17, 16}, {6445, 30, 10, 4},
    {6448, 25, 23, 8},  {6451, 26, 14, 13}, {6453, 29, 24, 14},
    {6456, 31, 25, 17}, {6459, 25, 17, 2},  {6461, 36, 15, 6},
    {6462, 24, 3, 2},   {6464, 25, 22, 6},  {6466, 26, 8, 7},
    {6467, 7, 6, 2},    {6469, 24, 22, 6},  {6472, 31, 30, 2},
    {6475, 18, 13, 2},  {6477, 31, 18, 15}, {6480, 11, 10, 1},
    {6482, 17, 8, 3},   {6483, 10, 5, 1},   {6485, 21, 8, 6},
    {6486, 23, 11, 1},  {6488, 21, 13, 7},  {6491, 17, 16, 6},
    {6493, 38, 11, 4},  {6494, 29, 27, 5},  {6496, 21, 5, 2},
    {6499, 26, 9, 2},   {6501, 24, 21, 14}, {6504, 23, 15, 9},
    {6506, 15, 7, 5},   {6507, 25, 12, 9},  {6508, 20, 11, 3},
    {6509, 14, 12, 7},  {6512, 29, 27, 13}, {6515, 27, 26, 12},
    {6516, 18, 12, 3},  {6517, 24, 11, 2},  {6520, 37, 29, 11},
    {6523, 12, 6, 3},   {6525, 16, 12, 10}, {6528, 16, 7, 2},
    {6530, 15, 9, 5},   {6531, 17, 14, 12}, {6532, 15, 10, 7},
    {6533, 18, 11, 9},  {6536, 25, 10, 9},  {6539, 12, 8, 2},
    {6541, 14, 11, 7},  {6542, 33, 29, 9},  {6543, 28, 21, 7},
    {6544, 19, 15, 1},  {6547, 12, 11, 10}, {6549, 34, 15, 6},
    {6552, 27, 22, 6},  {6555, 28, 24, 22}, {6557, 34, 30, 17},
    {6560, 19, 13, 11}, {6563, 35, 30, 21}, {6565, 19, 15, 8},
    {6566, 10, 3, 1},   {6568, 19, 15, 1},  {6571, 14, 13, 6},
    {6572, 20, 15, 11}, {6573, 17, 10, 5},  {6574, 29, 15, 14},
    {6576, 27, 25, 19}, {6579, 32, 15, 6},  {6581, 10, 8, 6},
    {6582, 17, 13, 8},  {6584, 37, 23, 7},  {6586, 34, 32, 15},
    {6587, 27, 4, 2},   {6589, 30, 25, 7},  {6592, 45, 42, 1},
    {6595, 33, 8, 5},   {6597, 18, 14, 5},  {6600, 21, 19, 16},
    {6602, 14, 8, 3},   {6603, 30, 20, 8},  {6605, 25, 15, 14},
    {6608, 9, 4, 2},    {6610, 27, 26, 8},  {6611, 13, 10, 3},
    {6613, 28, 22, 15}, {6616, 33, 9, 3},   {6619, 7, 6, 1},
    {6621, 21, 19, 6},  {6624, 15, 14, 9},  {6626, 20, 3, 1},
    {6627, 26, 7, 5},   {6629, 11, 9, 4},   {6632, 27, 20, 17},
    {6635, 14, 13, 9},  {6637, 29, 26, 21}, {6640, 43, 32, 9},
    {6643, 42, 25, 9},  {6645, 23, 18, 3},  {6646, 9, 8, 5},
    {6648, 25, 19, 16}, {6651, 22, 19, 14}, {6653, 8, 7, 3},
    {6656, 19, 15, 1},  {6658, 31, 30, 19}, {6659, 20, 16, 1},
    {6661, 14, 12, 1},  {6664, 24, 15, 6},  {6667, 13, 8, 2},
    {6669, 16, 7, 5},   {6672, 26, 21, 5},  {6675, 16, 10, 1},
    {6677, 26, 9, 7},   {6680, 29, 7, 3},   {6682, 38, 21, 5},
    {6683, 12, 5, 3},   {6685, 18, 11, 1},  {6688, 55, 32, 9},
    {6691, 15, 12, 1},  {6693, 10, 6, 4},   {6696, 45, 19, 7},
    {6698, 18, 14, 11}, {6699, 14, 12, 3},  {6701, 19, 8, 4},
    {6704, 11, 10, 6},  {6707, 32, 15, 14}, {6709, 18, 12, 7},
    {6712, 31, 26, 2},  {6715, 17, 6, 1},   {6717, 16, 11, 7},
    {6718, 14, 9, 7},   {6720, 12, 9, 7},   {6723, 21, 18, 4},
    {6725, 23, 3, 2},   {6728, 21, 8, 2},   {6731, 32, 27, 6},
    {6733, 8, 5, 3},    {6734, 21, 14, 9},  {6736, 27, 14, 2},
    {6738, 35, 24, 5},  {6739, 29, 24, 1},  {6741, 18, 15, 11},
    {6742, 9, 4, 3},    {6744, 29, 21, 15}, {6747, 22, 21, 1},
    {6749, 18, 16, 4},  {6752, 11, 8, 2},   {6755, 30, 20, 3},
    {6760, 37, 15, 7},  {6762, 28, 4, 3},   {6763, 32, 21, 11},
    {6765, 23, 3, 2},   {6766, 17, 5, 4},   {6768, 18, 15, 10},
    {6770, 34, 31, 24}, {6771, 36, 13, 12}, {6773, 32, 20, 6},
    {6776, 25, 11, 6},  {6778, 34, 33, 1},  {6779, 16, 6, 2},
    {6781, 16, 15, 4},  {6784, 16, 15, 1},  {6787, 25, 5, 2},
    {6789, 28, 19, 9},  {6790, 22, 8, 5},   {6792, 27, 25, 24},
    {6795, 17, 2, 1},   {6797, 20, 10, 6},  {6800, 26, 25, 1},
    {6802, 29, 25, 3},  {6803, 32, 21, 16}, {6805, 24, 20, 7},
    {6806, 25, 23, 16}, {6808, 37, 19, 13}, {6809, 27, 10, 5},
    {6811, 21, 20, 8},  {6813, 24, 19, 16}, {6814, 32, 26, 25},
    {6816, 22, 5, 2},   {6819, 21, 11, 8},  {6821, 19, 13, 8},
    {6823, 14, 5, 1},   {6824, 21, 14, 10}, {6827, 26, 12, 8},
    {6829, 20, 16, 6},  {6832, 35, 32, 25}, {6834, 23, 14, 1},
    {6835, 23, 4, 2},   {6837, 21, 17, 14}, {6840, 14, 9, 3},
    {6842, 18, 17, 7},  {6843, 26, 9, 6},   {6844, 20, 15, 5},
    {6845, 26, 20, 6},  {6848, 29, 22, 18}, {6850, 21, 7, 4},
    {6851, 31, 30, 4},  {6853, 28, 27, 15}, {6854, 21, 14, 1},
    {6856, 30, 29, 17}, {6859, 16, 12, 9},  {6861, 21, 10, 3},
    {6864, 11, 5, 2},   {6867, 16, 4, 2},   {6869, 17, 10, 1},
    {6870, 27, 22, 21}, {6872, 25, 13, 2},  {6875, 22, 21, 4},
    {6877, 15, 14, 2},  {6878, 21, 15, 5},  {6879, 9, 6, 2},
    {6880, 31, 30, 19}, {6883, 32, 26, 25}, {6885, 22, 21, 12},
    {6888, 24, 11, 9},  {6890, 16, 15, 8},  {6891, 30, 15, 10},
    {6893, 20, 12, 6},  {6894, 12, 9, 8},   {6896, 29, 17, 11},
    {6897, 27, 18, 8},  {6899, 6, 5, 2},    {6901, 6, 4, 1},
    {6904, 36, 13, 11}, {6907, 23, 12, 2},  {6909, 36, 35, 8},
    {6912, 25, 15, 12}, {6914, 16, 11, 4},  {6915, 18, 15, 7},
    {6916, 13, 4, 1},   {6917, 22, 13, 8},  {6919, 15, 14, 2},
    {6920, 8, 3, 1},    {6922, 25, 15, 13}, {6923, 17, 9, 2},
    {6925, 24, 9, 7},   {6926, 29, 9, 3},   {6928, 8, 7, 5},
    {6931, 21, 14, 13}, {6933, 35, 26, 4},  {6934, 15, 10, 9},
    {6936, 37, 31, 30}, {6939, 16, 12, 6},  {6941, 27, 12, 11},
    {6944, 23, 21, 8},  {6945, 17, 3, 2},   {6946, 23, 16, 14},
    {6947, 25, 19, 18}, {6949, 16, 5, 2},   {6950, 11, 9, 3},
    {6952, 32, 21, 19}, {6955, 13, 7, 4},   {6957, 24, 10, 3},
    {6960, 14, 9, 3},   {6962, 16, 11, 1},  {6963, 24, 20, 6},
    {6965, 18, 8, 7},   {6968, 35, 32, 17}, {6971, 28, 20, 6},
    {6973, 36, 34, 14}, {6976, 19, 18, 9},  {6978, 23, 18, 3},
    {6979, 11, 8, 6},   {6981, 17, 16, 11}, {6982, 21, 16, 8},
    {6984, 8, 3, 2},    {6987, 21, 6, 4},   {6989, 15, 12, 2},
    {6992, 27, 11, 9},  {6994, 28, 12, 3},  {6995, 25, 12, 4},
    {6997, 27, 17, 4},  {7000, 27, 26, 11}, {7003, 16, 6, 5},
    {7005, 14, 11, 4},  {7008, 33, 25, 6},  {7010, 14, 13, 6},
    {7011, 5, 2, 1},    {7013, 8, 5, 2},    {7016, 23, 10, 7},
    {7019, 33, 14, 13}, {7021, 23, 6, 4},   {7024, 22, 15, 10},
    {7027, 20, 9, 2},   {7028, 25, 16, 2},  {7029, 29, 23, 16},
    {7032, 19, 13, 9},  {7035, 29, 9, 8},   {7037, 26, 24, 20},
    {7040, 19, 18, 7},  {7043, 15, 12, 6},  {7045, 19, 6, 3},
    {7048, 39, 17, 4},  {7051, 7, 6, 3},    {7053, 13, 6, 1},
    {7054, 17, 11, 1},  {7056, 27, 24, 10}, {7059, 31, 17, 4},
    {7061, 24, 23, 1},  {7064, 11, 5, 2},   {7066, 21, 14, 11},
    {7067, 19, 11, 10}, {7069, 28, 17, 14}, {7072, 37, 26, 17},
    {7075, 31, 26, 20}, {7078, 11, 10, 2},  {7080, 27, 18, 16},
    {7083, 26, 22, 2},  {7085, 20, 15, 14}, {7086, 22, 6, 3},
    {7088, 32, 5, 2},   {7090, 21, 15, 1},  {7091, 9, 7, 4},
    {7093, 22, 18, 2},  {7094, 23, 13, 12}, {7096, 27, 17, 13},
    {7099, 16, 4, 1},   {7101, 8, 7, 1},    {7102, 28, 21, 8},
    {7104, 15, 10, 4},  {7107, 8, 2, 1},    {7109, 26, 21, 8},
    {7112, 35, 5, 2},   {7114, 13, 12, 3},  {7115, 34, 9, 5},
    {7117, 27, 26, 10}, {7120, 17, 15, 5},  {7122, 35, 13, 2},
    {7123, 26, 22, 6},  {7125, 11, 10, 5},  {7128, 19, 13, 2},
    {7130, 35, 24, 21}, {7131, 21, 10, 7},  {7133, 36, 34, 10},
    {7136, 16, 3, 2},   {7138, 20, 7, 6},   {7139, 26, 21, 11},
    {7141, 26, 9, 1},   {7142, 27, 20, 15}, {7144, 33, 13, 3},
    {7147, 20, 17, 9},  {7149, 9, 7, 6},    {7152, 37, 23, 16},
    {7154, 15, 14, 8},  {7155, 34, 21, 18}, {7157, 20, 18, 16},
    {7158, 33, 23, 12}, {7160, 27, 26, 9},  {7163, 7, 3, 2},
    {7165, 21, 17, 2},  {7168, 13, 10, 6},  {7171, 22, 18, 3},
    {7173, 28, 26, 19}, {7174, 16, 13, 4},  {7176, 33, 31, 21},
    {7178, 16, 7, 6},   {7179, 30, 20, 15}, {7181, 24, 13, 11},
    {7184, 33, 27, 2},  {7186, 19, 10, 1},  {7187, 18, 14, 12},
    {7188, 23, 19, 7},  {7189, 18, 14, 8},  {7192, 29, 14, 6},
    {7195, 32, 30, 7},  {7197, 17, 14, 12}, {7198, 11, 2, 1},
    {7200, 29, 23, 3},  {7205, 21, 5, 2},   {7208, 29, 23, 21},
    {7211, 29, 8, 6},   {7212, 29, 26, 4},  {7213, 25, 15, 8},
    {7214, 28, 7, 2},   {7216, 15, 8, 1},   {7219, 23, 6, 5},
    {7221, 13, 12, 10}, {7224, 15, 12, 10}, {7227, 29, 22, 7},
    {7229, 19, 4, 2},   {7232, 35, 12, 9},  {7234, 29, 22, 18},
    {7235, 19, 18, 7},  {7237, 21, 10, 2},  {7238, 19, 14, 6},
    {7240, 33, 14, 2},  {7243, 25, 10, 1},  {7245, 26, 23, 9},
    {7248, 17, 7, 4},   {7251, 19, 14, 13}, {7253, 32, 13, 7},
    {7256, 22, 21, 7},  {7259, 38, 19, 1},  {7261, 28, 3, 2},
    {7262, 32, 13, 12}, {7264, 41, 36, 19}, {7267, 32, 17, 6},
    {7268, 14, 4, 3},   {7269, 33, 32, 30}, {7272, 23, 10, 4},
    {7274, 38, 31, 18}, {7275, 8, 2, 1},    {7277, 24, 23, 7},
    {7280, 37, 6, 4},   {7282, 26, 8, 7},   {7283, 34, 22, 12},
    {7285, 29, 18, 5},  {7288, 25, 23, 17}, {7291, 29, 24, 1},
    {7293, 34, 14, 8},  {7294, 22, 13, 10}, {7296, 37, 7, 2},
    {7298, 15, 13, 11}, {7299, 26, 11, 3},  {7301, 23, 22, 7},
    {7304, 37, 30, 17}, {7307, 19, 15, 2},  {7309, 23, 15, 2},
    {7312, 25, 24, 3},  {7315, 25, 10, 4},  {7317, 20, 19, 4},
    {7320, 35, 28, 10}, {7323, 26, 11, 2},  {7325, 35, 29, 6},
    {7328, 41, 30, 26}, {7330, 11, 10, 9},  {7331, 28, 19, 6},
    {7333, 26, 21, 5},  {7336, 34, 21, 5},  {7339, 14, 5, 1},
    {7341, 29, 5, 2},   {7344, 33, 28, 3},  {7347, 27, 26, 2},
    {7349, 12, 6, 2},   {7352, 20, 5, 3},   {7355, 27, 26, 7},
    {7357, 30, 16, 8},  {7358, 20, 15, 6},  {7360, 23, 18, 2},
    {7363, 34, 24, 19}, {7365, 26, 20, 5},  {7368, 19, 13, 11},
    {7371, 28, 16, 9},  {7373, 11, 8, 6},   {7376, 41, 22, 16},
    {7379, 22, 8, 4},   {7381, 32, 10, 2},  {7384, 31, 26, 9},
    {7387, 23, 20, 10}, {7389, 22, 17, 13}, {7390, 13, 12, 10},
    {7392, 33, 27, 21}, {7395, 8, 4, 2},    {7396, 29, 16, 2},
    {7397, 28, 26, 21}, {7400, 19, 9, 5},   {7403, 30, 20, 14},
    {7405, 17, 14, 6},  {7406, 27, 20, 1},  {7407, 27, 21, 17},
    {7408, 13, 3, 1},   {7411, 24, 18, 16}, {7413, 17, 6, 2},
    {7414, 13, 12, 2},  {7415, 10, 9, 5},   {7416, 35, 23, 9},
    {7419, 32, 29, 2},  {7420, 26, 25, 5},  {7421, 26, 14, 2},
    {7424, 18, 13, 7},  {7427, 30, 17, 1},  {7429, 30, 10, 4},
    {7430, 11, 9, 6},   {7432, 30, 19, 7},  {7435, 13, 5, 2},
    {7437, 31, 20, 19}, {7440, 22, 21, 15}, {7442, 10, 6, 5},
    {7443, 23, 12, 6},  {7445, 12, 5, 2},   {7448, 21, 20, 14},
    {7451, 16, 14, 9},  {7453, 30, 29, 7},  {7454, 24, 13, 5},
    {7456, 27, 8, 1},   {7457, 12, 9, 2},   {7459, 40, 32, 14},
    {7461, 20, 19, 7},  {7464, 21, 18, 14}, {7467, 25, 20, 4},
    {7469, 30, 26, 22}, {7470, 17, 8, 3},   {7472, 23, 13, 7},
    {7474, 23, 17, 15}, {7475, 24, 10, 8},  {7477, 23, 16, 2},
    {7480, 32, 15, 2},  {7482, 18, 4, 3},   {7483, 25, 18, 2},
    {7485, 26, 8, 1},   {7486, 22, 19, 16}, {7488, 21, 16, 6},
    {7491, 10, 9, 7},   {7493, 20, 10, 3},  {7494, 20, 9, 8},
    {7496, 23, 21, 12}, {7498, 33, 15, 12}, {7499, 27, 14, 9},
    {7501, 21, 13, 2},  {7502, 20, 17, 8},  {7504, 45, 17, 2},
    {7507, 34, 14, 9},  {7509, 21, 10, 4},  {7510, 26, 7, 2},
    {7512, 22, 21, 3},  {7514, 38, 21, 5},  {7515, 18, 17, 11},
    {7516, 27, 25, 18}, {7517, 20, 19, 10}, {7520, 19, 9, 7},
    {7523, 35, 22, 1},  {7525, 17, 11, 4},  {7528, 18, 13, 1},
    {7531, 13, 10, 3},  {7533, 11, 8, 7},   {7536, 42, 21, 14},
    {7539, 26, 19, 10}, {7541, 28, 10, 2},  {7544, 23, 13, 7},
    {7546, 15, 8, 5},   {7547, 24, 22, 11}, {7549, 22, 16, 12},
    {7550, 27, 26, 9},  {7552, 17, 8, 3},   {7554, 12, 11, 6},
    {7555, 26, 13, 2},  {7557, 18, 16, 12}, {7560, 41, 21, 3},
    {7563, 29, 8, 7},   {7565, 24, 22, 13}, {7568, 15, 5, 3},
    {7571, 16, 14, 7},  {7573, 12, 7, 6},   {7576, 41, 40, 11},
    {7579, 14, 11, 3},  {7581, 23, 21, 14}, {7582, 13, 6, 2},
    {7584, 51, 46, 10}, {7587, 15, 9, 4},   {7589, 15, 10, 7},
    {7592, 45, 10, 1},  {7595, 34, 32, 13}, {7597, 32, 11, 7},
    {7600, 19, 18, 2},  {7603, 6, 5, 2},    {7605, 28, 15, 3},
    {7606, 24, 9, 7},   {7608, 39, 25, 10}, {7610, 18, 11, 2},
    {7611, 32, 27, 6},  {7613, 25, 8, 6},   {7615, 19, 13, 7},
    {7616, 31, 21, 14}, {7617, 18, 5, 3},   {7618, 22, 17, 6},
    {7619, 29, 18, 6},  {7621, 32, 29, 19}, {7624, 9, 2, 1},
    {7627, 40, 15, 2},  {7629, 35, 10, 7},  {7632, 54, 3, 2},
    {7633, 24, 14, 7},  {7635, 27, 11, 2},  {7637, 13, 10, 8},
    {7640, 29, 3, 2},   {7643, 16, 5, 1},   {7645, 28, 15, 10},
    {7648, 37, 33, 10}, {7651, 24, 16, 13}, {7653, 21, 14, 11},
    {7654, 20, 14, 1},  {7656, 30, 13, 7},  {7659, 17, 13, 10},
    {7661, 30, 20, 4},  {7664, 30, 23, 5},  {7667, 27, 26, 7},
    {7669, 30, 6, 3},   {7670, 13, 11, 4},  {7672, 39, 5, 1},
    {7675, 9, 6, 1},    {7676, 21, 14, 11}, {7677, 38, 25, 11},
    {7680, 27, 9, 3},   {7683, 23, 14, 7},  {7685, 10, 9, 8},
    {7688, 35, 16, 6},  {7690, 23, 19, 11}, {7691, 36, 29, 2},
    {7693, 30, 9, 4},   {7696, 17, 16, 15}, {7699, 10, 6, 3},
    {7701, 20, 14, 8},  {7704, 15, 12, 10}, {7707, 19, 18, 15},
    {7709, 18, 4, 2},   {7712, 27, 22, 5},  {7715, 22, 16, 14},
    {7717, 7, 6, 1},    {7718, 11, 5, 3},   {7720, 27, 25, 4},
    {7723, 30, 21, 9},  {7725, 23, 22, 17}, {7726, 30, 23, 7},
    {7728, 21, 6, 3},   {7731, 11, 10, 1},  {7733, 15, 8, 3},
    {7736, 22, 11, 3},  {7737, 13, 5, 1},   {7739, 48, 33, 29},
    {7741, 32, 22, 11}, {7744, 33, 28, 27}, {7746, 22, 19, 10},
    {7747, 28, 26, 12}, {7749, 28, 25, 2},  {7752, 35, 30, 2},
    {7754, 31, 7, 5},   {7755, 16, 7, 2},   {7756, 29, 6, 4},
    {7757, 17, 10, 7},  {7760, 32, 23, 21}, {7763, 14, 10, 6},
    {7765, 18, 17, 1},  {7768, 27, 18, 15}, {7771, 38, 35, 27},
    {7773, 30, 19, 18}, {7776, 41, 39, 36}, {7778, 24, 23, 3},
    {7779, 40, 22, 14}, {7781, 24, 22, 5},  {7782, 25, 17, 1},
    {7784, 7, 4, 2},    {7786, 16, 15, 9},  {7787, 26, 18, 7},
    {7789, 17, 13, 2},  {7792, 47, 31, 29}, {7795, 38, 33, 8},
    {7797, 14, 10, 1},  {7798, 29, 22, 3},  {7800, 11, 10, 5},
    {7803, 19, 14, 2},  {7804, 9, 8, 2},    {7805, 23, 14, 5},
    {7806, 17, 12, 6},  {7808, 25, 24, 10}, {7811, 34, 23, 18},
    {7813, 38, 30, 28}, {7816, 20, 3, 2},   {7819, 8, 5, 3},
    {7821, 14, 10, 8},  {7824, 39, 36, 14}, {7827, 31, 28, 6},
    {7829, 15, 14, 8},  {7832, 37, 19, 3},  {7834, 16, 15, 9},
    {7835, 28, 20, 10}, {7836, 23, 20, 7},  {7837, 30, 23, 1},
    {7840, 33, 29, 23}, {7842, 29, 10, 4},  {7843, 16, 9, 6},
    {7844, 23, 10, 7},  {7845, 32, 31, 7},  {7846, 23, 22, 2},
    {7848, 34, 31, 21}, {7851, 28, 13, 5},  {7853, 30, 28, 26},
    {7856, 12, 3, 2},   {7859, 22, 3, 2},   {7861, 20, 16, 2},
    {7862, 17, 13, 10}, {7864, 19, 13, 1},  {7867, 40, 35, 21},
    {7869, 17, 6, 2},   {7872, 27, 22, 18}, {7875, 28, 21, 13},
    {7877, 18, 7, 4},   {7879, 24, 19, 8},  {7880, 25, 19, 17},
    {7881, 26, 9, 6},   {7882, 17, 15, 10}, {7883, 17, 16, 6},
    {7885, 35, 30, 13}, {7886, 25, 15, 13}, {7888, 12, 9, 7},
    {7891, 18, 5, 1},   {7893, 20, 5, 2},   {7894, 21, 16, 13},
    {7896, 45, 39, 7},  {7898, 29, 8, 2},   {7899, 11, 10, 2},
    {7901, 17, 13, 10}, {7904, 45, 34, 25}, {7906, 15, 11, 1},
    {7907, 20, 10, 6},  {7909, 20, 2, 1},   {7912, 7, 3, 2},
    {7914, 29, 11, 4},  {7915, 32, 22, 6},  {7916, 25, 12, 4},
    {7917, 27, 26, 23}, {7918, 19, 6, 1},   {7920, 20, 19, 5},
    {7923, 26, 21, 14}, {7925, 10, 3, 2},   {7928, 23, 15, 5},
    {7930, 17, 10, 5},  {7931, 27, 24, 17}, {7933, 15, 11, 10},
    {7936, 40, 23, 21}, {7939, 11, 4, 2},   {7941, 15, 14, 2},
    {7944, 14, 7, 1},   {7947, 18, 13, 10}, {7949, 28, 17, 7},
    {7950, 38, 27, 6},  {7952, 19, 13, 2},  {7954, 23, 21, 20},
    {7955, 16, 11, 1},  {7957, 26, 14, 1},  {7960, 19, 13, 12},
    {7963, 11, 7, 2},   {7965, 36, 31, 15}, {7968, 27, 13, 12},
    {7970, 29, 19, 3},  {7971, 18, 16, 15}, {7973, 21, 14, 3},
    {7974, 13, 12, 10}, {7976, 44, 5, 3},   {7978, 14, 4, 3},
    {7979, 24, 2, 1},   {7981, 20, 15, 14}, {7982, 25, 8, 4},
    {7984, 37, 18, 2},  {7986, 28, 10, 3},  {7987, 7, 6, 1},
    {7989, 24, 22, 12}, {7992, 26, 25, 17}, {7995, 17, 12, 8},
    {7997, 16, 3, 2},   {8000, 16, 3, 1},   {8002, 12, 4, 3},
    {8003, 26, 21, 2},  {8005, 18, 16, 11}, {8008, 23, 22, 17},
    {8011, 20, 16, 5},  {8013, 28, 26, 20}, {8016, 31, 30, 25},
    {8018, 18, 15, 9},  {8019, 39, 33, 16}, {8021, 32, 18, 3},
    {8023, 16, 11, 9},  {8024, 21, 14, 11}, {8027, 11, 10, 9},
    {8029, 33, 19, 16}, {8031, 13, 12, 3},  {8032, 19, 15, 13},
    {8035, 30, 24, 4},  {8037, 34, 33, 30}, {8038, 13, 9, 8},
    {8040, 39, 13, 7},  {8042, 15, 8, 5},   {8043, 25, 17, 16},
    {8045, 18, 8, 3},   {8048, 30, 23, 3},  {8050, 31, 12, 11},
    {8051, 24, 15, 5},  {8052, 11, 9, 2},   {8053, 20, 19, 12},
    {8056, 27, 16, 5},  {8059, 34, 21, 6},  {8061, 10, 7, 2},
    {8064, 27, 23, 9},  {8066, 32, 27, 15}, {8067, 42, 29, 17},
    {8069, 15, 14, 4},  {8072, 23, 11, 6},  {8074, 28, 26, 3},
    {8075, 29, 24, 16}, {8077, 30, 27, 18}, {8080, 39, 14, 11},
    {8083, 31, 17, 6},  {8085, 31, 22, 13}, {8088, 25, 10, 8},
    {8091, 16, 9, 1},   {8093, 20, 5, 3},   {8096, 32, 11, 2},
    {8099, 28, 25, 23}, {8101, 15, 6, 4},   {8102, 31, 2, 1},
    {8104, 30, 23, 17}, {8107, 25, 16, 1},  {8109, 15, 9, 4},
    {8112, 15, 12, 2},  {8115, 20, 17, 5},  {8117, 32, 15, 14},
    {8118, 21, 15, 1},  {8120, 8, 3, 2},    {8122, 19, 12, 4},
    {8123, 47, 22, 18}, {8125, 26, 18, 10}, {8126, 28, 25, 15},
    {8128, 25, 24, 19}, {8131, 25, 7, 6},   {8132, 9, 5, 4},
    {8133, 30, 19, 18}, {8136, 11, 3, 2},   {8139, 24, 12, 10},
    {8141, 32, 8, 6},   {8142, 29, 28, 20}, {8144, 28, 27, 17},
    {8147, 22, 8, 1},   {8149, 18, 8, 4},   {8152, 29, 25, 14},
    {8155, 18, 10, 5},  {8157, 19, 16, 14}, {8158, 23, 17, 7},
    {8160, 52, 45, 10}, {8163, 29, 28, 25}, {8165, 29, 26, 9},
    {8168, 25, 16, 6},  {8170, 17, 6, 5},   {8171, 21, 17, 8},
    {8173, 11, 8, 7},   {8174, 21, 9, 8},   {8176, 43, 32, 21},
    {8179, 24, 21, 13}, {8181, 9, 6, 4},    {8184, 32, 27, 6},
    {8186, 10, 7, 3},   {8187, 29, 22, 16}, {8189, 41, 10, 4},
    {8192, 9, 5, 2},    {8195, 15, 14, 4},  {8197, 21, 8, 2},
    {8200, 18, 15, 3},  {8202, 15, 12, 11}, {8203, 33, 28, 23},
    {8205, 21, 18, 13}, {8206, 16, 14, 9},  {8208, 19, 13, 11},
    {8211, 34, 18, 13}, {8213, 24, 10, 4},  {8214, 21, 16, 3},
    {8216, 27, 24, 17}, {8219, 29, 27, 10}, {8221, 24, 22, 11},
    {8224, 39, 36, 25}, {8227, 25, 18, 12}, {8228, 23, 18, 6},
    {8229, 36, 29, 3},  {8232, 22, 15, 9},  {8235, 38, 36, 21},
    {8237, 23, 18, 11}, {8238, 15, 5, 3},   {8240, 14, 13, 7},
    {8242, 36, 35, 2},  {8243, 35, 14, 2},  {8245, 18, 16, 14},
    {8248, 25, 15, 13}, {8249, 20, 11, 9},  {8251, 13, 10, 3},
    {8253, 9, 5, 2},    {8256, 32, 21, 14}, {8259, 21, 15, 4},
    {8261, 20, 17, 14}, {8264, 15, 6, 3},   {8266, 12, 7, 6},
    {8267, 21, 16, 2},  {8269, 24, 13, 2},  {8270, 21, 17, 4},
    {8272, 32, 25, 23}, {8275, 35, 10, 9},  {8277, 30, 8, 3},
    {8280, 30, 19, 9},  {8283, 25, 23, 4},  {8285, 47, 42, 24},
    {8288, 27, 5, 4},   {8290, 43, 24, 14}, {8291, 28, 26, 19},
    {8293, 14, 12, 8},  {8294, 21, 12, 11}, {8296, 37, 19, 6},
    {8298, 21, 14, 11}, {8299, 29, 28, 8},  {8301, 27, 13, 10},
    {8304, 32, 17, 15}, {8306, 17, 12, 9},  {8307, 19, 17, 10},
    {8308, 16, 9, 8},   {8309, 28, 26, 10}, {8312, 45, 44, 30},
    {8314, 18, 15, 9},  {8315, 20, 15, 2},  {8317, 30, 28, 19},
    {8318, 21, 16, 10}, {8320, 41, 39, 5},  {8322, 7, 5, 2},
    {8323, 35, 12, 10}, {8325, 31, 14, 10}, {8326, 14, 5, 4},
    {8328, 37, 23, 14}, {8331, 37, 34, 10}, {8333, 26, 18, 14},
    {8336, 33, 10, 2},  {8338, 16, 15, 7},  {8339, 13, 12, 7},
    {8341, 50, 48, 19}, {8344, 27, 15, 1},  {8346, 26, 25, 10},
    {8347, 29, 7, 6},   {8349, 29, 22, 7},  {8352, 35, 18, 14},
    {8355, 18, 6, 5},   {8356, 22, 18, 13}, {8357, 22, 12, 5},
    {8360, 15, 14, 5},  {8361, 24, 20, 8},  {8363, 20, 3, 2},
    {8365, 22, 12, 4},  {8368, 35, 17, 4},  {8369, 29, 15, 1},
    {8371, 13, 8, 2},   {8373, 14, 5, 2},   {8376, 45, 39, 17},
    {8379, 26, 15, 8},  {8381, 22, 15, 5},  {8382, 32, 22, 13},
    {8383, 23, 13, 8},  {8384, 19, 4, 2},   {8387, 14, 12, 3},
    {8389, 39, 16, 13}, {8392, 17, 11, 1},  {8394, 30, 10, 3},
    {8395, 18, 12, 4},  {8396, 17, 14, 11}, {8397, 25, 6, 1},
    {8400, 29, 28, 11}, {8403, 22, 6, 1},   {8405, 32, 22, 16},
    {8406, 30, 9, 1},   {8408, 20, 9, 2},   {8410, 33, 17, 5},
    {8411, 30, 27, 5},  {8413, 26, 25, 24}, {8416, 40, 33, 22},
    {8419, 15, 10, 1},  {8421, 27, 26, 20}, {8422, 33, 22, 9},
    {8424, 33, 31, 28}, {8427, 14, 13, 8},  {8429, 35, 14, 6},
    {8432, 17, 15, 13}, {8435, 33, 28, 26}, {8437, 34, 27, 15},
    {8438, 17, 10, 7},  {8440, 18, 9, 6},   {8443, 16, 12, 6},
    {8445, 38, 6, 5},   {8448, 21, 17, 6},  {8450, 38, 36, 7},
    {8451, 13, 5, 2},   {8452, 11, 8, 6},   {8453, 14, 11, 5},
    {8454, 21, 6, 4},   {8456, 15, 13, 8},  {8458, 19, 10, 6},
    {8459, 10, 9, 7},   {8461, 24, 22, 14}, {8462, 33, 32, 31},
    {8464, 34, 15, 5},  {8467, 31, 24, 21}, {8468, 15, 11, 3},
    {8469, 26, 14, 13}, {8472, 30, 21, 5},  {8474, 37, 6, 1},
    {8475, 6, 4, 1},    {8477, 46, 34, 26}, {8480, 21, 6, 3},
    {8483, 26, 19, 10}, {8485, 23, 10, 5},  {8486, 22, 21, 19},
    {8488, 13, 6, 3},   {8490, 26, 16, 13}, {8491, 35, 32, 25},
    {8493, 33, 26, 23}, {8494, 10, 7, 4},   {8496, 25, 19, 3},
    {8498, 15, 14, 11}, {8499, 28, 18, 15}, {8501, 37, 36, 31},
    {8504, 35, 24, 10}, {8507, 31, 6, 5},   {8509, 19, 16, 15},
    {8512, 33, 27, 4},  {8514, 19, 14, 8},  {8515, 27, 10, 7},
    {8517, 14, 8, 5},   {8520, 28, 15, 13}, {8523, 21, 18, 5},
    {8525, 36, 19, 2},  {8528, 15, 4, 2},   {8531, 18, 2, 1},
    {8533, 23, 20, 4},  {8534, 27, 7, 1},   {8536, 19, 10, 3},
    {8538, 20, 11, 4},  {8539, 12, 9, 4},   {8541, 26, 9, 5},
    {8542, 30, 28, 9},  {8544, 31, 5, 2},   {8546, 19, 11, 10},
    {8547, 21, 13, 10}, {8549, 41, 23, 6},  {8552, 29, 10, 1},
    {8554, 33, 17, 7},  {8555, 14, 13, 7},  {8557, 30, 23, 17},
    {8560, 21, 13, 3},  {8561, 15, 5, 1},   {8563, 13, 5, 2},
    {8565, 33, 14, 11}, {8567, 24, 16, 11}, {8568, 28, 25, 10},
    {8570, 37, 10, 3},  {8571, 12, 8, 2},   {8573, 22, 14, 4},
    {8574, 22, 13, 2},  {8576, 19, 9, 1},   {8578, 30, 7, 4},
    {8579, 13, 11, 6},  {8581, 8, 7, 3},    {8584, 32, 15, 13},
    {8587, 34, 8, 2},   {8589, 15, 12, 10}, {8590, 27, 23, 17},
    {8592, 23, 5, 4},   {8594, 21, 20, 5},  {8595, 17, 13, 10},
    {8596, 13, 7, 6},   {8597, 33, 27, 26}, {8598, 23, 14, 1},
    {8600, 21, 14, 3},  {8603, 16, 9, 1},   {8605, 30, 23, 7},
    {8606, 19, 17, 7},  {8608, 47, 44, 33}, {8610, 21, 17, 14},
    {8611, 25, 18, 1},  {8613, 24, 15, 14}, {8616, 22, 19, 11},
    {8619, 29, 28, 5},  {8621, 22, 10, 2},  {8622, 33, 24, 19},
    {8624, 4, 3, 2},    {8627, 38, 15, 11}, {8628, 25, 24, 7},
    {8629, 33, 16, 11}, {8630, 20, 6, 3},   {8632, 41, 32, 3},
    {8634, 19, 14, 8},  {8635, 23, 14, 13}, {8637, 20, 13, 7},
    {8640, 23, 10, 4},  {8642, 15, 11, 6},  {8643, 26, 18, 2},
    {8645, 17, 14, 12}, {8648, 31, 17, 6},  {8651, 26, 11, 3},
    {8653, 26, 18, 10}, {8656, 39, 14, 4},  {8659, 14, 9, 4},
    {8661, 30, 7, 1},   {8662, 31, 22, 18}, {8664, 29, 19, 11},
    {8667, 22, 21, 18}, {8669, 20, 12, 3},  {8672, 29, 23, 12},
    {8675, 18, 17, 10}, {8677, 13, 11, 6},  {8678, 25, 19, 15},
    {8680, 32, 7, 2},   {8683, 19, 18, 11}, {8685, 31, 9, 6},
    {8687, 21, 18, 3},  {8688, 25, 20, 7},  {8690, 24, 14, 11},
    {8691, 40, 17, 5},  {8693, 32, 24, 15}, {8696, 25, 20, 6},
    {8699, 26, 20, 3},  {8701, 33, 5, 2},   {8702, 24, 17, 15},
    {8704, 39, 20, 2},  {8706, 25, 22, 6},  {8707, 8, 6, 1},
    {8709, 19, 16, 10}, {8710, 26, 25, 18}, {8712, 30, 13, 3},
    {8715, 33, 32, 8},  {8716, 36, 22, 21}, {8717, 37, 18, 16},
    {8720, 27, 17, 5},  {8723, 32, 12, 2},  {8725, 25, 9, 6},
    {8726, 21, 19, 8},  {8728, 13, 7, 1},   {8731, 30, 26, 16},
    {8733, 40, 32, 31}, {8736, 35, 33, 24}, {8738, 28, 18, 11},
    {8739, 28, 20, 2},  {8741, 32, 27, 21}, {8744, 45, 12, 11},
    {8747, 18, 14, 7},  {8749, 29, 22, 6},  {8752, 30, 29, 11},
    {8755, 25, 24, 14}, {8757, 35, 26, 12}, {8758, 17, 8, 4},
    {8760, 27, 9, 4},   {8761, 27, 25, 15}, {8763, 18, 17, 5},
    {8764, 10, 3, 1},   {8765, 14, 9, 3},   {8768, 27, 10, 8},
    {8771, 37, 4, 2},   {8773, 20, 5, 3},   {8774, 30, 29, 11},
    {8776, 26, 19, 9},  {8778, 16, 5, 2},   {8779, 45, 5, 2},
    {8781, 34, 10, 1},  {8782, 33, 10, 4},  {8784, 26, 25, 7},
    {8786, 30, 18, 3},  {8787, 49, 45, 22}, {8788, 22, 15, 13},
    {8789, 27, 7, 2},   {8790, 17, 2, 1},   {8792, 24, 9, 2},
    {8794, 14, 7, 2},   {8795, 12, 5, 3},   {8797, 26, 18, 7},
    {8800, 25, 14, 6},  {8803, 31, 21, 4},  {8805, 42, 16, 9},
    {8808, 15, 9, 6},   {8810, 24, 17, 7},  {8811, 22, 8, 2},
    {8813, 37, 4, 2},   {8816, 17, 8, 6},   {8818, 29, 11, 9},
    {8819, 15, 13, 4},  {8821, 28, 24, 23}, {8822, 32, 28, 25},
    {8824, 19, 14, 6},  {8827, 30, 17, 10}, {8829, 15, 9, 8},
    {8832, 15, 9, 6},   {8835, 20, 16, 5},  {8837, 20, 10, 6},
    {8840, 33, 22, 16}, {8843, 33, 20, 17}, {8845, 31, 24, 2},
    {8846, 21, 20, 12}, {8848, 33, 4, 3},   {8851, 33, 14, 4},
    {8853, 23, 17, 16}, {8854, 12, 5, 1},   {8856, 14, 5, 2},
    {8858, 29, 23, 17}, {8859, 35, 30, 27}, {8861, 18, 16, 10},
    {8864, 37, 12, 7},  {8866, 29, 16, 5},  {8867, 32, 9, 5},
    {8869, 34, 19, 7},  {8872, 51, 4, 2},   {8875, 19, 18, 1},
    {8877, 21, 13, 10}, {8880, 36, 9, 6},   {8882, 17, 11, 6},
    {8883, 28, 27, 10}, {8885, 30, 26, 20}, {8888, 43, 13, 9},
    {8891, 10, 9, 4},   {8893, 27, 7, 6},   {8894, 17, 6, 5},
    {8896, 41, 10, 6},  {8899, 29, 2, 1},   {8901, 17, 12, 2},
    {8902, 10, 9, 2},   {8904, 40, 25, 10}, {8906, 35, 16, 13},
    {8907, 22, 21, 16}, {8909, 22, 17, 9},  {8912, 18, 17, 1},
    {8914, 16, 4, 3},   {8915, 11, 6, 5},   {8917, 28, 21, 3},
    {8918, 16, 10, 9},  {8920, 33, 32, 31}, {8923, 11, 8, 6},
    {8925, 22, 11, 7},  {8928, 35, 29, 2},  {8931, 17, 6, 5},
    {8933, 24, 10, 4},  {8936, 17, 3, 1},   {8939, 22, 21, 14},
    {8941, 36, 35, 31}, {8942, 21, 16, 13}, {8944, 39, 34, 1},
    {8947, 28, 20, 6},  {8949, 29, 22, 13}, {8950, 9, 5, 1},
    {8952, 27, 14, 2},  {8954, 25, 23, 1},  {8955, 18, 13, 1},
    {8957, 16, 10, 3},  {8960, 12, 7, 5},   {8963, 30, 26, 10},
    {8965, 30, 7, 5},   {8968, 29, 11, 10}, {8971, 23, 5, 4},
    {8973, 24, 19, 12}, {8976, 23, 8, 2},   {8978, 18, 17, 5},
    {8979, 29, 22, 16}, {8981, 22, 2, 1},   {8984, 9, 8, 6},
    {8987, 16, 12, 10}, {8989, 15, 7, 2},   {8992, 35, 6, 2},
    {8995, 19, 16, 13}, {8996, 25, 16, 12}, {8997, 33, 22, 1},
    {8998, 23, 10, 2},  {9000, 28, 19, 17}, {9003, 37, 20, 10},
    {9005, 19, 14, 4},  {9008, 31, 17, 7},  {9010, 17, 12, 1},
    {9011, 31, 14, 10}, {9013, 32, 9, 7},   {9015, 12, 3, 1},
    {9016, 23, 21, 8},  {9019, 26, 22, 15}, {9021, 28, 22, 10},
    {9024, 32, 29, 11}, {9027, 23, 14, 3},  {9029, 36, 26, 1},
    {9030, 32, 8, 1},   {9032, 45, 19, 1},  {9035, 30, 22, 5},
    {9037, 28, 14, 5},  {9038, 21, 9, 8},   {9040, 9, 6, 4},
    {9043, 27, 26, 23}, {9045, 33, 8, 6},   {9046, 36, 30, 15},
    {9048, 23, 16, 10}, {9051, 17, 10, 7},  {9053, 21, 15, 12},
    {9056, 11, 3, 2},   {9059, 24, 15, 6},  {9061, 26, 18, 7},
    {9064, 33, 4, 2},   {9066, 15, 6, 1},   {9067, 9, 2, 1},
    {9069, 36, 31, 12}, {9070, 23, 13, 3},  {9072, 45, 38, 6},
    {9073, 20, 16, 5},  {9075, 30, 17, 15}, {9077, 16, 6, 4},
    {9080, 17, 10, 7},  {9082, 19, 17, 14}, {9083, 31, 18, 5},
    {9085, 26, 13, 5},  {9087, 23, 19, 7},  {9088, 33, 31, 1},
    {9090, 18, 7, 3},   {9091, 20, 9, 7},   {9093, 11, 8, 4},
    {9096, 27, 26, 11}, {9098, 25, 15, 14}, {9099, 16, 9, 1},
    {9101, 14, 8, 6},   {9102, 15, 14, 2},  {9104, 47, 33, 24},
    {9107, 28, 13, 6},  {9109, 9, 5, 2},    {9110, 32, 17, 8},
    {9112, 9, 7, 5},    {9115, 30, 9, 7},   {9117, 18, 16, 7},
    {9118, 21, 11, 10}, {9120, 35, 25, 5},  {9122, 47, 29, 23},
    {9123, 15, 13, 10}, {9124, 23, 17, 12}, {9125, 26, 12, 10},
    {9128, 39, 25, 22}, {9131, 7, 5, 4},    {9133, 26, 9, 4},
    {9136, 21, 14, 1},  {9139, 10, 3, 1},   {9141, 36, 7, 1},
    {9144, 35, 25, 4},  {9146, 12, 2, 1},   {9147, 31, 16, 2},
    {9149, 30, 28, 7},  {9152, 33, 29, 7},  {9155, 25, 11, 6},
    {9157, 30, 10, 7},  {9160, 22, 15, 3},  {9163, 20, 7, 6},
    {9165, 13, 12, 3},  {9166, 19, 10, 3},  {9167, 12, 3, 2},
    {9168, 15, 10, 4},  {9171, 38, 5, 4},   {9172, 33, 26, 8},
    {9173, 31, 17, 12}, {9174, 29, 23, 8},  {9176, 43, 37, 30},
    {9178, 19, 17, 8},  {9179, 10, 6, 5},   {9181, 23, 13, 6},
    {9183, 30, 13, 1},  {9184, 33, 26, 14}, {9187, 20, 18, 4},
    {9189, 26, 14, 8},  {9192, 21, 18, 13}, {9194, 16, 10, 5},
    {9195, 14, 13, 1},  {9196, 31, 12, 5},  {9197, 14, 12, 5},
    {9200, 31, 24, 2},  {9203, 28, 20, 6},  {9205, 14, 11, 3},
    {9208, 35, 10, 9},  {9211, 36, 19, 17}, {9213, 38, 7, 5},
    {9216, 21, 14, 8},  {9219, 30, 6, 4},   {9221, 33, 26, 5},
    {9222, 33, 23, 11}, {9224, 35, 21, 7},  {9227, 38, 29, 3},
    {9229, 20, 19, 17}, {9230, 17, 12, 8},  {9232, 49, 18, 8},
    {9235, 23, 22, 2},  {9236, 22, 14, 9},  {9237, 18, 16, 12},
    {9240, 17, 11, 7},  {9243, 5, 2, 1},    {9245, 43, 35, 18},
    {9246, 27, 18, 3},  {9248, 37, 18, 13}, {9251, 8, 3, 2},
    {9253, 42, 28, 10}, {9254, 24, 17, 13}, {9256, 19, 6, 4},
    {9259, 22, 16, 6},  {9261, 30, 18, 10}, {9264, 39, 17, 14},
    {9266, 23, 14, 6},  {9267, 23, 8, 5},   {9269, 42, 18, 14},
    {9272, 13, 4, 3},   {9275, 10, 6, 2},   {9277, 11, 3, 2},
    {9280, 23, 7, 2},   {9283, 20, 7, 6},   {9285, 29, 6, 5},
    {9286, 25, 20, 14}, {9288, 49, 43, 18}, {9291, 8, 5, 1},
    {9293, 20, 18, 3},  {9296, 29, 17, 15}, {9299, 22, 21, 1},
    {9301, 35, 25, 18}, {9304, 27, 13, 9},  {9307, 31, 25, 4},
    {9309, 19, 16, 10}, {9312, 21, 19, 16}, {9314, 27, 23, 12},
    {9315, 26, 21, 8},  {9317, 23, 20, 2},  {9318, 18, 6, 3},
    {9320, 55, 22, 8},  {9323, 18, 4, 1},   {9325, 18, 16, 1},
    {9326, 35, 23, 20}, {9328, 21, 14, 8},  {9331, 31, 30, 2},
    {9333, 26, 14, 8},  {9336, 17, 5, 3},   {9338, 23, 8, 5},
    {9339, 23, 14, 1},  {9341, 19, 18, 14}, {9342, 19, 10, 8},
    {9344, 51, 44, 5},  {9346, 18, 10, 3},  {9347, 16, 14, 7},
    {9348, 19, 9, 3},   {9349, 25, 18, 2},  {9352, 31, 26, 15},
    {9354, 16, 7, 4},   {9355, 22, 16, 4},  {9357, 35, 34, 2},
    {9360, 28, 23, 1},  {9362, 23, 13, 1},  {9363, 26, 21, 17},
    {9365, 21, 20, 7},  {9368, 38, 7, 1},   {9371, 28, 21, 17},
    {9373, 28, 10, 4},  {9374, 18, 9, 2},   {9376, 23, 5, 1},
    {9379, 14, 9, 5},   {9381, 19, 9, 6},   {9384, 17, 11, 6},
    {9387, 31, 30, 9},  {9388, 24, 7, 3},   {9389, 20, 7, 3},
    {9392, 29, 24, 11}, {9395, 17, 15, 4},  {9397, 32, 14, 7},
    {9400, 18, 13, 1},  {9403, 30, 8, 1},   {9405, 18, 4, 1},
    {9406, 28, 27, 7},  {9408, 28, 15, 6},  {9410, 32, 11, 8},
    {9411, 24, 22, 1},  {9412, 11, 9, 2},   {9413, 11, 10, 9},
    {9416, 19, 11, 6},  {9419, 30, 19, 13}, {9421, 14, 9, 5},
    {9422, 22, 10, 9},  {9424, 22, 9, 6},   {9426, 21, 6, 5},
    {9427, 23, 18, 8},  {9428, 21, 17, 7},  {9429, 21, 9, 2},
    {9430, 37, 18, 5},  {9432, 52, 43, 25}, {9434, 19, 14, 12},
    {9435, 7, 5, 4},    {9437, 14, 12, 1},  {9438, 15, 13, 6},
    {9440, 25, 22, 14}, {9443, 30, 19, 2},  {9445, 34, 14, 12},
    {9446, 12, 5, 3},   {9448, 23, 21, 20}, {9451, 13, 12, 3},
    {9452, 18, 14, 13}, {9453, 32, 29, 23}, {9456, 35, 19, 5},
    {9458, 25, 16, 6},  {9459, 36, 18, 13}, {9460, 20, 17, 1},
    {9461, 32, 28, 6},  {9462, 34, 14, 5},  {9464, 37, 18, 13},
    {9466, 23, 14, 1},  {9467, 27, 14, 7},  {9469, 28, 22, 18},
    {9472, 45, 40, 15}, {9474, 23, 20, 7},  {9475, 29, 25, 2},
    {9477, 26, 11, 7},  {9480, 18, 9, 2},   {9482, 19, 12, 4},
    {9483, 20, 10, 5},  {9485, 23, 22, 9},  {9488, 25, 12, 7},
    {9491, 19, 6, 5},   {9493, 38, 22, 3},  {9495, 23, 20, 18},
    {9496, 29, 17, 15}, {9498, 41, 17, 10}, {9499, 29, 20, 10},
    {9501, 29, 26, 4},  {9504, 19, 15, 6},  {9505, 15, 10, 1},
    {9506, 12, 7, 5},   {9507, 29, 28, 14}, {9509, 13, 9, 2},
    {9512, 39, 17, 16}, {9514, 28, 21, 9},  {9515, 18, 9, 2},
    {9517, 32, 18, 11}, {9520, 13, 5, 2},   {9523, 8, 4, 2},
    {9525, 30, 27, 2},  {9526, 16, 10, 1},  {9528, 37, 33, 22},
    {9531, 32, 14, 2},  {9533, 36, 31, 5},  {9536, 31, 26, 6},
    {9538, 24, 9, 3},   {9539, 38, 27, 13}, {9541, 31, 21, 14},
    {9544, 43, 32, 9},  {9547, 12, 8, 2},   {9549, 37, 34, 32},
    {9550, 19, 15, 11}, {9552, 11, 4, 1},   {9554, 29, 28, 15},
    {9555, 24, 22, 10}, {9556, 21, 9, 1},   {9557, 27, 20, 1},
    {9560, 27, 13, 2},  {9563, 25, 9, 2},   {9565, 6, 5, 4},
    {9568, 49, 26, 14}, {9571, 24, 16, 5},  {9572, 31, 19, 9},
    {9573, 8, 6, 5},    {9574, 38, 37, 8},  {9576, 29, 25, 7},
    {9579, 34, 33, 6},  {9581, 12, 7, 4},   {9584, 36, 29, 27},
    {9587, 36, 29, 22}, {9589, 28, 22, 15}, {9592, 15, 8, 1},
    {9595, 28, 18, 15}, {9596, 18, 11, 5},  {9597, 28, 25, 22},
    {9600, 19, 6, 4},   {9602, 21, 5, 2},   {9603, 19, 10, 4},
    {9604, 13, 11, 9},  {9605, 37, 26, 12}, {9606, 15, 12, 1},
    {9608, 14, 13, 1},  {9611, 14, 6, 5},   {9613, 32, 16, 3},
    {9616, 25, 8, 2},   {9618, 16, 3, 1},   {9619, 29, 14, 10},
    {9621, 29, 20, 11}, {9624, 31, 17, 7},  {9626, 7, 6, 1},
    {9627, 25, 19, 18}, {9629, 16, 15, 10}, {9632, 30, 23, 5},
    {9635, 26, 21, 10}, {9637, 21, 18, 6},  {9638, 21, 19, 13},
    {9640, 33, 7, 3},   {9643, 18, 14, 11}, {9645, 9, 7, 6},
    {9646, 31, 30, 17}, {9648, 31, 24, 21}, {9650, 32, 18, 11},
    {9651, 22, 14, 4},  {9653, 22, 15, 2},  {9656, 19, 16, 2},
    {9659, 15, 14, 8},  {9661, 22, 14, 6},  {9662, 22, 9, 8},
    {9664, 35, 21, 1},  {9667, 34, 22, 6},  {9669, 29, 27, 24},
    {9670, 35, 34, 3},  {9672, 17, 9, 3},   {9675, 44, 25, 23},
    {9677, 30, 27, 12}, {9680, 25, 6, 2},   {9683, 46, 37, 21},
    {9685, 17, 14, 4},  {9688, 20, 9, 2},   {9691, 17, 14, 1},
    {9693, 10, 6, 3},   {9694, 50, 18, 17}, {9696, 36, 27, 21},
    {9698, 38, 27, 24}, {9699, 30, 6, 1},   {9701, 23, 12, 3},
    {9703, 13, 7, 1},   {9704, 7, 3, 2},    {9706, 22, 15, 11},
    {9707, 26, 15, 1},  {9709, 16, 13, 10}, {9710, 30, 15, 9},
    {9712, 27, 19, 17}, {9715, 27, 10, 6},  {9717, 9, 6, 2},
    {9718, 46, 29, 11}, {9720, 37, 4, 2},   {9722, 12, 11, 8},
    {9723, 32, 29, 25}, {9724, 30, 28, 15}, {9725, 14, 8, 5},
    {9726, 29, 28, 23}, {9728, 30, 5, 2},   {9731, 16, 12, 10},
    {9733, 22, 21, 8},  {9736, 15, 7, 6},   {9739, 15, 14, 2},
    {9741, 11, 8, 1},   {9742, 30, 18, 15}, {9744, 36, 9, 2},
    {9747, 26, 8, 1},   {9749, 20, 7, 6},   {9752, 42, 17, 3},
    {9755, 6, 5, 1},    {9757, 22, 14, 6},  {9760, 56, 49, 47},
    {9763, 30, 18, 1},  {9764, 25, 13, 9},  {9765, 28, 17, 11},
    {9768, 41, 23, 22}, {9771, 29, 8, 1},   {9773, 32, 10, 6},
    {9776, 24, 19, 1},  {9778, 24, 18, 15}, {9779, 40, 21, 20},
    {9781, 40, 28, 15}, {9784, 38, 37, 9},  {9787, 20, 13, 9},
    {9789, 17, 8, 2},   {9792, 27, 26, 21}, {9794, 24, 15, 6},
    {9795, 26, 5, 4},   {9797, 35, 32, 2},  {9800, 42, 41, 10},
    {9802, 12, 10, 3},  {9803, 18, 10, 2},  {9805, 22, 21, 18},
    {9806, 30, 9, 8},   {9808, 43, 32, 21}, {9809, 19, 15, 10},
    {9811, 41, 28, 27}, {9812, 31, 30, 22}, {9813, 26, 17, 12},
    {9815, 27, 26, 8},  {9816, 46, 33, 22}, {9818, 30, 19, 10},
    {9819, 18, 4, 3},   {9821, 34, 23, 18}, {9824, 13, 10, 3},
    {9826, 22, 10, 7},  {9827, 13, 12, 5},  {9829, 4, 3, 1},
    {9832, 39, 37, 25}, {9834, 32, 25, 7},  {9835, 32, 29, 1},
    {9836, 18, 13, 10}, {9837, 35, 16, 2},  {9838, 25, 19, 12},
    {9840, 49, 47, 13}, {9842, 35, 28, 24}, {9843, 18, 12, 2},
    {9845, 38, 18, 11}, {9848, 25, 23, 10}, {9850, 19, 16, 4},
    {9851, 11, 6, 5},   {9853, 18, 11, 10}, {9856, 36, 15, 1},
    {9859, 34, 28, 1},  {9861, 33, 18, 16}, {9862, 13, 10, 3},
    {9864, 35, 32, 21}, {9866, 34, 27, 14}, {9867, 26, 8, 2},
    {9869, 23, 8, 6},   {9872, 13, 11, 3},  {9875, 19, 18, 2},
    {9877, 19, 16, 15}, {9880, 35, 28, 5},  {9881, 27, 7, 5},
    {9883, 35, 34, 11}, {9885, 38, 15, 3},  {9888, 33, 6, 4},
    {9891, 46, 38, 7},  {9892, 31, 4, 1},   {9893, 32, 28, 18},
    {9894, 27, 7, 1},   {9896, 39, 38, 26}, {9898, 23, 13, 5},
    {9899, 34, 21, 13}, {9901, 10, 4, 3},   {9902, 27, 19, 10},
    {9904, 21, 10, 2},  {9907, 10, 7, 1},   {9909, 11, 10, 7},
    {9910, 27, 16, 15}, {9912, 42, 35, 15}, {9915, 29, 17, 8},
    {9917, 32, 9, 6},   {9920, 49, 18, 14}, {9923, 37, 33, 26},
    {9924, 30, 29, 26}, {9925, 12, 9, 7},   {9928, 39, 38, 31},
    {9931, 34, 10, 3},  {9933, 23, 6, 2},   {9934, 34, 7, 3},
    {9936, 22, 21, 1},  {9938, 25, 19, 9},  {9939, 32, 26, 17},
    {9941, 29, 12, 10}, {9944, 15, 14, 6},  {9947, 23, 17, 8},
    {9949, 32, 24, 10}, {9952, 31, 30, 11}, {9955, 9, 8, 5},
    {9957, 25, 11, 4},  {9958, 17, 14, 4},  {9960, 30, 15, 10},
    {9962, 20, 14, 3},  {9963, 34, 29, 20}, {9965, 34, 24, 23},
    {9968, 36, 3, 2},   {9971, 11, 8, 5},   {9973, 27, 24, 12},
    {9976, 21, 19, 5},  {9979, 26, 10, 2},  {9981, 30, 27, 22},
    {9984, 27, 10, 7},  {9987, 14, 11, 10}, {9989, 21, 20, 6},
    {9992, 7, 4, 2},    {9994, 29, 22, 3},  {9995, 41, 40, 31},
    {9997, 26, 10, 6},  {10000, 19, 13, 9}};

// ANSI range [160, 2000]
/**
 * ANSI X9.62 irreducible pentanomials
 *
 * Accredited Standards Committee X9:
 * Harold Deal, Chairman
 * Alice Droogan, Vice Chairman
 * Cynthia Fuller, Secretariat
 * The X9F subcommittee on Data and Information Security:
 * Glenda Barnes, Chairman
 * The X9F1 working group:
 * M. Blake Greenlee, Chairman
 */
static polynomial_t ansi_pentanomials[] = {
    {160, 1, 2, 117},   {163, 1, 2, 8},     {164, 1, 2, 49},
    {165, 1, 2, 25},    {168, 1, 2, 65},    {171, 1, 3, 42},
    {173, 1, 2, 10},    {176, 1, 2, 43},    {179, 1, 2, 4},
    {181, 1, 2, 89},    {184, 1, 2, 81},    {187, 1, 2, 20},
    {188, 1, 2, 60},    {189, 1, 2, 49},    {190, 1, 2, 47},
    {192, 1, 2, 7},     {195, 1, 2, 37},    {197, 1, 2, 21},
    {200, 1, 2, 81},    {203, 1, 2, 45},    {205, 1, 2, 21},
    {206, 1, 2, 63},    {208, 1, 2, 83},    {211, 1, 2, 165},
    {213, 1, 2, 62},    {216, 1, 2, 107},   {219, 1, 2, 65},
    {221, 1, 2, 18},    {222, 1, 2, 73},    {224, 1, 2, 159},
    {226, 1, 2, 30},    {227, 1, 2, 21},    {229, 1, 2, 21},
    {230, 1, 2, 13},    {232, 1, 2, 23},    {235, 1, 2, 45},
    {237, 1, 2, 104},   {240, 1, 3, 49},    {243, 1, 2, 17},
    {245, 1, 2, 37},    {246, 1, 2, 11},    {248, 1, 2, 243},
    {251, 1, 2, 45},    {254, 1, 2, 7},     {256, 1, 2, 155},
    {259, 1, 2, 254},   {261, 1, 2, 74},    {262, 1, 2, 207},
    {264, 1, 2, 169},   {267, 1, 2, 29},    {269, 1, 2, 117},
    {272, 1, 3, 56},    {275, 1, 2, 28},    {277, 1, 2, 33},
    {280, 1, 2, 113},   {283, 1, 2, 200},   {285, 1, 2, 77},
    {288, 1, 2, 191},   {290, 1, 2, 70},    {291, 1, 2, 76},
    {293, 1, 3, 154},   {296, 1, 2, 123},   {298, 1, 2, 78},
    {299, 1, 2, 21},    {301, 1, 2, 26},    {304, 1, 2, 11},
    {306, 1, 2, 106},   {307, 1, 2, 93},    {309, 1, 2, 26},
    {311, 1, 3, 155},   {312, 1, 2, 83},    {315, 1, 2, 142},
    {317, 1, 3, 68},    {320, 1, 2, 7},     {323, 1, 2, 21},
    {325, 1, 2, 53},    {326, 1, 2, 67},    {328, 1, 2, 51},
    {331, 1, 2, 134},   {334, 1, 2, 5},     {335, 1, 2, 250},
    {336, 1, 2, 77},    {338, 1, 2, 112},   {339, 1, 2, 26},
    {341, 1, 2, 57},    {344, 1, 2, 7},     {347, 1, 2, 96},
    {349, 1, 2, 186},   {352, 1, 2, 263},   {355, 1, 2, 138},
    {356, 1, 2, 69},    {357, 1, 2, 28},    {360, 1, 2, 49},
    {361, 1, 2, 44},    {363, 1, 2, 38},    {365, 1, 2, 109},
    {368, 1, 2, 85},    {371, 1, 2, 156},   {373, 1, 3, 172},
    {374, 1, 2, 109},   {376, 1, 2, 77},    {379, 1, 2, 222},
    {381, 1, 2, 5},     {384, 1, 2, 299},   {387, 1, 2, 146},
    {389, 1, 2, 159},   {392, 1, 2, 145},   {395, 1, 2, 333},
    {397, 1, 2, 125},   {398, 1, 3, 23},    {400, 1, 2, 245},
    {403, 1, 2, 80},    {405, 1, 2, 38},    {408, 1, 2, 323},
    {410, 1, 2, 16},    {411, 1, 2, 50},    {413, 1, 2, 33},
    {416, 1, 3, 76},    {419, 1, 2, 129},   {421, 1, 2, 81},
    {424, 1, 2, 177},   {427, 1, 2, 245},   {429, 1, 2, 14},
    {430, 1, 2, 263},   {432, 1, 2, 103},   {434, 1, 2, 64},
    {435, 1, 2, 166},   {437, 1, 2, 6},     {440, 1, 2, 37},
    {442, 1, 2, 32},    {443, 1, 2, 57},    {445, 1, 2, 225},
    {448, 1, 3, 83},    {451, 1, 2, 33},    {452, 1, 2, 10},
    {453, 1, 2, 88},    {454, 1, 2, 195},   {456, 1, 2, 275},
    {459, 1, 2, 332},   {461, 1, 2, 247},   {464, 1, 2, 310},
    {466, 1, 2, 78},    {467, 1, 2, 210},   {469, 1, 2, 149},
    {472, 1, 2, 33},    {475, 1, 2, 68},    {477, 1, 2, 121},
    {480, 1, 2, 149},   {482, 1, 2, 13},    {483, 1, 2, 352},
    {485, 1, 2, 70},    {488, 1, 2, 123},   {491, 1, 2, 270},
    {493, 1, 2, 171},   {496, 1, 3, 52},    {499, 1, 2, 174},
    {501, 1, 2, 332},   {502, 1, 2, 99},    {504, 1, 3, 148},
    {507, 1, 2, 26},    {509, 1, 2, 94},    {512, 1, 2, 51},
    {515, 1, 2, 73},    {517, 1, 2, 333},   {520, 1, 2, 291},
    {523, 1, 2, 66},    {525, 1, 2, 92},    {528, 1, 2, 35},
    {530, 1, 2, 25},    {531, 1, 2, 53},    {533, 1, 2, 37},
    {535, 1, 2, 143},   {536, 1, 2, 165},   {539, 1, 2, 37},
    {541, 1, 2, 36},    {542, 1, 3, 212},   {544, 1, 2, 87},
    {546, 1, 2, 8},     {547, 1, 2, 165},   {548, 1, 2, 385},
    {549, 1, 3, 274},   {552, 1, 2, 41},    {554, 1, 2, 162},
    {555, 1, 2, 326},   {557, 1, 2, 288},   {560, 1, 2, 157},
    {562, 1, 2, 56},    {563, 1, 4, 159},   {565, 1, 2, 66},
    {568, 1, 2, 291},   {571, 1, 2, 408},   {572, 1, 2, 238},
    {573, 1, 2, 220},   {576, 1, 3, 52},    {578, 1, 2, 138},
    {579, 1, 3, 526},   {581, 1, 2, 138},   {584, 1, 2, 361},
    {586, 1, 2, 14},    {587, 1, 2, 130},   {589, 1, 2, 365},
    {591, 1, 2, 38},    {592, 1, 2, 143},   {595, 1, 2, 9},
    {597, 1, 2, 64},    {598, 1, 2, 131},   {600, 1, 2, 239},
    {603, 1, 2, 446},   {605, 1, 2, 312},   {608, 1, 2, 213},
    {611, 1, 2, 13},    {613, 1, 2, 377},   {616, 1, 2, 465},
    {619, 1, 2, 494},   {621, 1, 2, 17},    {624, 1, 2, 71},
    {627, 1, 2, 37},    {629, 1, 2, 121},   {630, 1, 2, 49},
    {632, 1, 2, 9},     {635, 1, 2, 64},    {637, 1, 2, 84},
    {638, 1, 2, 127},   {640, 1, 3, 253},   {643, 1, 2, 153},
    {644, 1, 2, 24},    {645, 1, 2, 473},   {648, 1, 2, 235},
    {653, 1, 2, 37},    {656, 1, 2, 39},    {659, 1, 2, 25},
    {661, 1, 2, 80},    {664, 1, 2, 177},   {666, 1, 2, 100},
    {667, 1, 2, 161},   {669, 1, 2, 314},   {672, 1, 2, 91},
    {674, 1, 2, 22},    {675, 1, 2, 214},   {677, 1, 2, 325},
    {678, 1, 2, 95},    {680, 1, 2, 91},    {681, 1, 2, 83},
    {683, 1, 2, 153},   {685, 1, 3, 4},     {688, 1, 2, 71},
    {691, 1, 2, 242},   {693, 1, 2, 250},   {696, 1, 2, 241},
    {699, 1, 2, 40},    {701, 1, 2, 466},   {703, 1, 2, 123},
    {704, 1, 2, 277},   {706, 1, 2, 27},    {707, 1, 2, 141},
    {709, 1, 2, 9},     {710, 1, 3, 29},    {712, 1, 2, 623},
    {715, 1, 3, 458},   {717, 1, 2, 320},   {720, 1, 2, 625},
    {723, 1, 2, 268},   {725, 1, 2, 331},   {728, 1, 2, 51},
    {731, 1, 2, 69},    {733, 1, 2, 92},    {734, 1, 2, 67},
    {736, 1, 2, 359},   {739, 1, 2, 60},    {741, 1, 2, 34},
    {744, 1, 2, 347},   {747, 1, 2, 158},   {749, 1, 2, 357},
    {752, 1, 2, 129},   {755, 1, 4, 159},   {757, 1, 2, 359},
    {760, 1, 2, 17},    {763, 1, 2, 17},    {764, 1, 2, 12},
    {765, 1, 2, 137},   {766, 1, 3, 280},   {768, 1, 2, 115},
    {770, 1, 2, 453},   {771, 1, 2, 86},    {773, 1, 2, 73},
    {776, 1, 2, 51},    {779, 1, 2, 456},   {781, 1, 2, 209},
    {784, 1, 2, 59},    {786, 1, 2, 118},   {787, 1, 2, 189},
    {788, 1, 2, 375},   {789, 1, 2, 5},     {790, 1, 2, 111},
    {792, 1, 2, 403},   {795, 1, 2, 137},   {796, 1, 2, 36},
    {797, 1, 2, 193},   {800, 1, 2, 463},   {802, 1, 2, 102},
    {803, 1, 2, 208},   {805, 1, 2, 453},   {808, 1, 3, 175},
    {811, 1, 2, 18},    {813, 1, 2, 802},   {816, 1, 3, 51},
    {819, 1, 2, 149},   {821, 1, 2, 177},   {824, 1, 2, 495},
    {827, 1, 2, 189},   {829, 1, 2, 560},   {830, 1, 2, 241},
    {832, 1, 2, 39},    {835, 1, 2, 350},   {836, 1, 2, 606},
    {837, 1, 2, 365},   {840, 1, 2, 341},   {843, 1, 2, 322},
    {848, 1, 2, 225},   {851, 1, 2, 442},   {853, 1, 2, 461},
    {854, 1, 2, 79},    {856, 1, 2, 842},   {859, 1, 2, 594},
    {863, 1, 2, 90},    {864, 1, 2, 607},   {867, 1, 2, 380},
    {869, 1, 2, 82},    {872, 1, 2, 691},   {874, 1, 2, 110},
    {875, 1, 2, 66},    {877, 1, 2, 140},   {878, 1, 2, 343},
    {880, 1, 3, 221},   {883, 1, 2, 488},   {885, 1, 2, 707},
    {886, 1, 2, 227},   {888, 1, 2, 97},    {891, 1, 2, 364},
    {893, 1, 2, 13},    {896, 1, 2, 19},    {899, 1, 3, 898},
    {901, 1, 2, 581},   {904, 1, 3, 60},    {907, 1, 3, 26},
    {909, 1, 3, 168},   {910, 1, 2, 357},   {912, 1, 2, 569},
    {914, 1, 2, 4},     {915, 1, 2, 89},    {917, 1, 2, 22},
    {920, 1, 3, 517},   {922, 1, 2, 24},    {923, 1, 2, 142},
    {925, 1, 2, 308},   {928, 1, 2, 33},    {929, 1, 2, 36},
    {931, 1, 2, 72},    {933, 1, 2, 527},   {934, 1, 3, 800},
    {936, 1, 3, 27},    {939, 1, 2, 142},   {940, 1, 2, 204},
    {941, 1, 2, 573},   {944, 1, 2, 487},   {946, 1, 3, 83},
    {947, 1, 2, 400},   {949, 1, 2, 417},   {950, 1, 2, 859},
    {952, 1, 3, 311},   {955, 1, 2, 606},   {957, 1, 2, 158},
    {958, 1, 2, 191},   {960, 1, 2, 491},   {962, 1, 2, 18},
    {963, 1, 2, 145},   {965, 1, 2, 213},   {968, 1, 2, 21},
    {970, 1, 2, 260},   {971, 1, 2, 6},     {973, 1, 2, 113},
    {974, 1, 2, 211},   {976, 1, 2, 285},   {978, 1, 2, 376},
    {980, 1, 2, 316},   {981, 1, 2, 383},   {984, 1, 2, 349},
    {987, 1, 3, 142},   {989, 1, 2, 105},   {992, 1, 2, 585},
    {995, 1, 3, 242},   {997, 1, 2, 453},   {1000, 1, 3, 68},
    {1002, 1, 2, 266},  {1003, 1, 2, 410},  {1004, 1, 2, 96},
    {1005, 1, 2, 41},   {1006, 1, 2, 63},   {1008, 1, 2, 703},
    {1011, 1, 2, 17},   {1013, 1, 2, 180},  {1016, 1, 2, 49},
    {1017, 1, 2, 746},  {1018, 1, 2, 27},   {1019, 1, 2, 96},
    {1021, 1, 2, 5},    {1024, 1, 2, 515},  {1027, 1, 2, 378},
    {1032, 1, 2, 901},  {1035, 1, 2, 76},   {1037, 1, 2, 981},
    {1038, 1, 2, 41},   {1040, 1, 2, 429},  {1043, 1, 3, 869},
    {1045, 1, 2, 378},  {1046, 1, 2, 39},   {1048, 1, 3, 172},
    {1051, 1, 3, 354},  {1053, 1, 2, 290},  {1056, 1, 2, 11},
    {1059, 1, 3, 6},    {1061, 1, 2, 166},  {1064, 1, 2, 946},
    {1066, 1, 2, 258},  {1067, 1, 2, 69},   {1068, 1, 2, 223},
    {1069, 1, 2, 146},  {1070, 1, 3, 94},   {1072, 1, 2, 443},
    {1073, 1, 3, 235},  {1074, 1, 2, 395},  {1075, 1, 2, 92},
    {1076, 1, 2, 22},   {1077, 1, 2, 521},  {1080, 1, 2, 151},
    {1083, 1, 2, 538},  {1088, 1, 2, 531},  {1091, 1, 2, 82},
    {1093, 1, 2, 173},  {1096, 1, 2, 351},  {1099, 1, 2, 464},
    {1101, 1, 2, 14},   {1104, 1, 2, 259},  {1107, 1, 2, 176},
    {1109, 1, 2, 501},  {1112, 1, 2, 1045}, {1114, 1, 2, 345},
    {1115, 1, 2, 268},  {1117, 1, 2, 149},  {1118, 1, 2, 475},
    {1120, 1, 3, 386},  {1123, 1, 2, 641},  {1124, 1, 2, 156},
    {1125, 1, 2, 206},  {1128, 1, 3, 7},    {1131, 1, 2, 188},
    {1132, 1, 2, 20},   {1133, 1, 2, 667},  {1136, 1, 2, 177},
    {1139, 1, 2, 45},   {1141, 1, 2, 134},  {1143, 1, 2, 7},
    {1144, 1, 2, 431},  {1147, 1, 2, 390},  {1149, 1, 2, 221},
    {1150, 1, 2, 63},   {1152, 1, 2, 971},  {1155, 1, 2, 94},
    {1157, 1, 2, 105},  {1160, 1, 2, 889},  {1162, 1, 2, 288},
    {1163, 1, 2, 33},   {1165, 1, 2, 494},  {1168, 1, 2, 473},
    {1171, 1, 2, 396},  {1172, 1, 2, 426},  {1173, 1, 2, 673},
    {1176, 1, 2, 19},   {1179, 1, 2, 640},  {1181, 1, 2, 82},
    {1184, 1, 2, 1177}, {1187, 1, 2, 438},  {1189, 1, 2, 102},
    {1192, 1, 3, 831},  {1194, 1, 2, 317},  {1195, 1, 2, 293},
    {1197, 1, 2, 269},  {1200, 1, 3, 739},  {1203, 1, 2, 226},
    {1205, 1, 2, 4},    {1208, 1, 2, 915},  {1211, 1, 2, 373},
    {1213, 1, 2, 245},  {1216, 1, 2, 155},  {1219, 1, 2, 225},
    {1221, 1, 2, 101},  {1222, 1, 2, 215},  {1224, 1, 2, 157},
    {1227, 1, 2, 361},  {1229, 1, 2, 627},  {1232, 1, 2, 225},
    {1235, 1, 2, 642},  {1237, 1, 2, 150},  {1240, 1, 2, 567},
    {1243, 1, 2, 758},  {1244, 1, 2, 126},  {1245, 1, 2, 212},
    {1248, 1, 2, 1201}, {1250, 1, 2, 37},   {1251, 1, 2, 1004},
    {1253, 1, 2, 141},  {1254, 1, 2, 697},  {1256, 1, 2, 171},
    {1258, 1, 2, 503},  {1259, 1, 2, 192},  {1261, 1, 2, 14},
    {1262, 1, 2, 793},  {1264, 1, 2, 285},  {1267, 1, 2, 197},
    {1269, 1, 2, 484},  {1272, 1, 2, 223},  {1274, 1, 2, 486},
    {1275, 1, 2, 25},   {1277, 1, 2, 451},  {1280, 1, 2, 843},
    {1283, 1, 2, 70},   {1285, 1, 2, 564},  {1288, 1, 2, 215},
    {1290, 1, 2, 422},  {1291, 1, 2, 245},  {1292, 1, 2, 78},
    {1293, 1, 2, 26},   {1296, 1, 2, 379},  {1299, 1, 2, 172},
    {1301, 1, 2, 297},  {1303, 1, 2, 306},  {1304, 1, 3, 574},
    {1307, 1, 2, 157},  {1309, 1, 2, 789},  {1312, 1, 2, 1265},
    {1315, 1, 2, 270},  {1316, 1, 2, 12},   {1317, 1, 2, 254},
    {1318, 1, 3, 94},   {1320, 1, 2, 835},  {1322, 1, 2, 538},
    {1323, 1, 2, 1198}, {1325, 1, 2, 526},  {1328, 1, 2, 507},
    {1330, 1, 2, 609},  {1331, 1, 2, 289},  {1333, 1, 2, 276},
    {1336, 1, 2, 815},  {1339, 1, 2, 284},  {1341, 1, 2, 53},
    {1342, 1, 2, 477},  {1344, 1, 2, 469},  {1346, 1, 2, 57},
    {1347, 1, 2, 61},   {1349, 1, 2, 40},   {1352, 1, 2, 583},
    {1355, 1, 2, 117},  {1357, 1, 2, 495},  {1360, 1, 2, 393},
    {1363, 1, 2, 852},  {1365, 1, 2, 329},  {1368, 1, 2, 41},
    {1370, 1, 2, 108},  {1371, 1, 2, 145},  {1373, 1, 2, 613},
    {1376, 1, 2, 1201}, {1378, 1, 2, 362},  {1379, 1, 2, 400},
    {1381, 1, 2, 56},   {1382, 1, 3, 58},   {1384, 1, 2, 1131},
    {1387, 1, 2, 33},   {1389, 1, 2, 41},   {1392, 1, 2, 485},
    {1394, 1, 2, 30},   {1395, 1, 2, 233},  {1397, 1, 2, 397},
    {1400, 1, 2, 493},  {1403, 1, 2, 717},  {1405, 1, 2, 558},
    {1406, 1, 2, 13},   {1408, 1, 3, 45},   {1411, 1, 2, 200},
    {1413, 1, 2, 101},  {1416, 1, 3, 231},  {1418, 1, 2, 283},
    {1419, 1, 2, 592},  {1421, 1, 2, 30},   {1424, 1, 2, 507},
    {1427, 1, 2, 900},  {1429, 1, 2, 149},  {1432, 1, 2, 251},
    {1435, 1, 2, 126},  {1437, 1, 2, 545},  {1439, 1, 2, 535},
    {1440, 1, 3, 1023}, {1443, 1, 2, 413},  {1445, 1, 2, 214},
    {1448, 1, 3, 212},  {1450, 1, 2, 155},  {1451, 1, 2, 193},
    {1453, 1, 2, 348},  {1456, 1, 2, 1011}, {1459, 1, 2, 1032},
    {1461, 1, 2, 446},  {1462, 1, 2, 165},  {1464, 1, 2, 275},
    {1467, 1, 2, 113},  {1469, 1, 2, 775},  {1472, 1, 2, 613},
    {1474, 1, 2, 59},   {1475, 1, 2, 208},  {1477, 1, 2, 1325},
    {1480, 1, 2, 285},  {1483, 1, 2, 1077}, {1484, 1, 2, 61},
    {1485, 1, 2, 655},  {1488, 1, 2, 463},  {1491, 1, 2, 544},
    {1493, 1, 2, 378},  {1494, 1, 2, 731},  {1496, 1, 2, 181},
    {1498, 1, 2, 416},  {1499, 1, 2, 477},  {1501, 1, 2, 60},
    {1502, 1, 2, 111},  {1504, 1, 2, 207},  {1506, 1, 2, 533},
    {1507, 1, 2, 900},  {1509, 1, 2, 209},  {1512, 1, 2, 1121},
    {1515, 1, 2, 712},  {1517, 1, 2, 568},  {1520, 1, 2, 81},
    {1522, 1, 2, 47},   {1523, 1, 2, 240},  {1525, 1, 2, 102},
    {1528, 1, 2, 923},  {1531, 1, 2, 1125}, {1532, 1, 2, 466},
    {1533, 1, 2, 763},  {1536, 1, 2, 881},  {1538, 1, 2, 6},
    {1539, 1, 2, 80},   {1541, 1, 2, 4},    {1544, 1, 2, 99},
    {1546, 1, 2, 810},  {1547, 1, 2, 493},  {1549, 1, 2, 426},
    {1552, 1, 2, 83},   {1555, 1, 2, 254},  {1557, 1, 2, 20},
    {1560, 1, 2, 11},   {1563, 1, 2, 41},   {1565, 1, 2, 18},
    {1568, 1, 2, 133},  {1571, 1, 2, 21},   {1573, 1, 2, 461},
    {1574, 1, 2, 331},  {1576, 1, 2, 147},  {1579, 1, 2, 374},
    {1581, 1, 2, 160},  {1584, 1, 2, 895},  {1587, 1, 2, 433},
    {1589, 1, 2, 882},  {1592, 1, 2, 223},  {1594, 1, 2, 971},
    {1595, 1, 2, 18},   {1597, 1, 2, 42},   {1598, 1, 2, 385},
    {1600, 1, 2, 57},   {1603, 1, 2, 917},  {1605, 1, 2, 46},
    {1608, 1, 2, 271},  {1610, 1, 2, 250},  {1611, 1, 2, 58},
    {1613, 1, 2, 48},   {1614, 1, 2, 1489}, {1616, 1, 2, 139},
    {1619, 1, 2, 289},  {1621, 1, 2, 1577}, {1622, 1, 2, 1341},
    {1624, 1, 2, 1095}, {1626, 1, 2, 191},  {1627, 1, 2, 189},
    {1629, 1, 2, 397},  {1632, 1, 2, 211},  {1635, 1, 2, 113},
    {1637, 1, 2, 234},  {1640, 1, 2, 715},  {1643, 1, 2, 760},
    {1644, 1, 2, 236},  {1645, 1, 2, 938},  {1646, 1, 2, 435},
    {1648, 1, 2, 77},   {1651, 1, 2, 873},  {1653, 1, 2, 82},
    {1654, 1, 3, 201},  {1656, 1, 2, 361},  {1658, 1, 2, 552},
    {1659, 1, 2, 374},  {1661, 1, 2, 84},   {1662, 1, 3, 958},
    {1664, 1, 2, 399},  {1667, 1, 2, 1020}, {1669, 1, 2, 425},
    {1670, 1, 2, 19},   {1672, 1, 2, 405},  {1675, 1, 2, 77},
    {1677, 1, 2, 844},  {1680, 1, 2, 1549}, {1682, 1, 2, 354},
    {1683, 1, 2, 1348}, {1684, 1, 2, 474},  {1685, 1, 2, 493},
    {1686, 1, 2, 887},  {1688, 1, 2, 921},  {1690, 1, 2, 200},
    {1691, 1, 2, 556},  {1693, 1, 2, 137},  {1696, 1, 2, 737},
    {1699, 1, 2, 405},  {1701, 1, 2, 568},  {1702, 1, 2, 245},
    {1704, 1, 3, 55},   {1706, 1, 2, 574},  {1707, 1, 2, 221},
    {1709, 1, 2, 201},  {1712, 1, 2, 445},  {1714, 1, 2, 191},
    {1715, 1, 2, 612},  {1717, 1, 2, 881},  {1718, 1, 2, 535},
    {1720, 1, 2, 525},  {1723, 1, 2, 137},  {1725, 1, 2, 623},
    {1727, 1, 2, 22},   {1728, 1, 2, 545},  {1730, 1, 2, 316},
    {1731, 1, 2, 925},  {1732, 1, 2, 75},   {1733, 1, 2, 285},
    {1736, 1, 2, 435},  {1739, 1, 2, 409},  {1741, 1, 3, 226},
    {1744, 1, 2, 35},   {1747, 1, 2, 93},   {1749, 1, 2, 236},
    {1752, 1, 2, 559},  {1754, 1, 2, 75},   {1755, 1, 2, 316},
    {1757, 1, 2, 21},   {1758, 1, 2, 221},  {1760, 1, 3, 1612},
    {1761, 1, 2, 131},  {1762, 1, 2, 318},  {1763, 1, 2, 345},
    {1765, 1, 2, 165},  {1766, 1, 2, 1029}, {1768, 1, 2, 1403},
    {1771, 1, 2, 297},  {1773, 1, 2, 50},   {1776, 1, 2, 17},
    {1779, 1, 3, 1068}, {1781, 1, 2, 18},   {1784, 1, 2, 1489},
    {1786, 1, 2, 614},  {1787, 1, 2, 457},  {1789, 1, 2, 80},
    {1792, 1, 2, 341},  {1794, 1, 2, 95},   {1795, 1, 2, 89},
    {1796, 1, 2, 829},  {1797, 1, 2, 80},   {1800, 1, 2, 1013},
    {1803, 1, 2, 248},  {1805, 1, 2, 82},   {1808, 1, 2, 25},
    {1811, 1, 2, 117},  {1812, 1, 2, 758},  {1813, 1, 3, 884},
    {1816, 1, 2, 887},  {1819, 1, 2, 116},  {1821, 1, 2, 326},
    {1822, 1, 3, 31},   {1824, 1, 2, 821},  {1826, 1, 2, 298},
    {1827, 1, 2, 154},  {1829, 1, 2, 162},  {1832, 1, 3, 1078},
    {1834, 1, 2, 210},  {1835, 1, 2, 288},  {1837, 1, 2, 200},
    {1840, 1, 2, 195},  {1842, 1, 2, 799},  {1843, 1, 2, 872},
    {1845, 1, 2, 526},  {1848, 1, 2, 871},  {1850, 1, 2, 79},
    {1851, 1, 2, 250},  {1852, 1, 2, 339},  {1853, 1, 2, 705},
    {1856, 1, 2, 585},  {1858, 1, 2, 1368}, {1859, 1, 2, 120},
    {1861, 1, 2, 509},  {1864, 1, 2, 1379}, {1867, 1, 2, 117},
    {1868, 1, 2, 250},  {1869, 1, 2, 617},  {1872, 1, 3, 60},
    {1874, 1, 2, 70},   {1875, 1, 2, 412},  {1876, 1, 2, 122},
    {1877, 1, 2, 796},  {1880, 1, 2, 1647}, {1882, 1, 2, 128},
    {1883, 1, 2, 1062}, {1885, 1, 2, 813},  {1888, 1, 2, 923},
    {1891, 1, 2, 1766}, {1892, 1, 3, 497},  {1893, 1, 2, 461},
    {1894, 1, 3, 215},  {1896, 1, 2, 451},  {1897, 1, 2, 324},
    {1898, 1, 2, 613},  {1899, 1, 2, 485},  {1901, 1, 2, 330},
    {1904, 1, 2, 337},  {1907, 1, 2, 45},   {1909, 1, 2, 225},
    {1910, 1, 3, 365},  {1912, 1, 2, 599},  {1914, 1, 2, 544},
    {1915, 1, 2, 473},  {1916, 1, 2, 502},  {1917, 1, 2, 485},
    {1920, 1, 2, 67},   {1922, 1, 2, 36},   {1923, 1, 4, 40},
    {1925, 1, 2, 576},  {1928, 1, 2, 763},  {1930, 1, 2, 155},
    {1931, 1, 2, 648},  {1933, 1, 2, 971},  {1936, 1, 2, 117},
    {1939, 1, 2, 5},    {1941, 1, 2, 1133}, {1942, 1, 2, 147},
    {1944, 1, 2, 617},  {1947, 1, 2, 1162}, {1949, 1, 2, 621},
    {1952, 1, 3, 65},   {1954, 1, 2, 1226}, {1955, 1, 2, 109},
    {1957, 1, 2, 17},   {1960, 1, 2, 939},  {1963, 1, 2, 1137},
    {1965, 1, 2, 364},  {1968, 1, 3, 922},  {1970, 1, 2, 388},
    {1971, 1, 2, 100},  {1972, 1, 2, 474},  {1973, 1, 2, 438},
    {1976, 1, 3, 1160}, {1978, 1, 2, 158},  {1979, 1, 2, 369},
    {1981, 1, 2, 96},   {1982, 1, 2, 1027}, {1984, 1, 2, 129},
    {1987, 1, 2, 80},   {1989, 1, 2, 719},  {1992, 1, 2, 1241},
    {1995, 1, 2, 37},   {1997, 1, 2, 835},  {1998, 1, 3, 1290},
    {2000, 1, 2, 981}};

static int compare_poly(const void *a, const void *b) {
	const polynomial_t *one = a;
	const polynomial_t *other = b;
	return (one->m - other->m);
}

bool poly_exists(unsigned long m) { return m >= 2 && m <= 10000; }

const polynomial_t *poly_find(unsigned long m) {
	if (!poly_exists(m)) {
		return NULL;
	}
	polynomial_t *search_tri = ansi_trinomials;
	polynomial_t *search_penta = ansi_pentanomials;
	size_t len_tri = sizeof(ansi_trinomials) / sizeof(polynomial_t);
	size_t len_penta = sizeof(ansi_pentanomials) / sizeof(polynomial_t);
	if (m < 160 || m > 2000) {
		search_tri = hp_trinomials;
		search_penta = hp_pentanomials;
		len_tri = sizeof(hp_trinomials) / sizeof(polynomial_t);
		len_penta = sizeof(hp_pentanomials) / sizeof(polynomial_t);
	}

	polynomial_t searched = {(unsigned int)m};
	polynomial_t *tri = (polynomial_t *)bsearch(
	    &searched, search_tri, len_tri, sizeof(polynomial_t), &compare_poly);
	if (tri) {
		return tri;
	} else {
		return (polynomial_t *)bsearch(&searched, search_penta, len_penta,
		                               sizeof(polynomial_t), &compare_poly);
	}
}

GEN poly_gen(const polynomial_t *polynomial) {
	pari_sp ltop = avma;

	GEN coeffs = gtovec0(gen_0, polynomial->m + 1);
	gel(coeffs, polynomial->m + 1) = gen_1;
	gel(coeffs, polynomial->e1 + 1) = gen_1;
	gel(coeffs, polynomial->e2 + 1) = gen_1;
	gel(coeffs, polynomial->e3 + 1) = gen_1;
	gel(coeffs, 1) = gen_1;

	GEN poly = gmul(gtopolyrev(coeffs, -1), gmodulss(1, 2));
	return gerepilecopy(ltop, ffgen(poly, -1));
}

GEN poly_find_gen(unsigned long m) { return poly_gen(poly_find(m)); }
